/*	$NetBSD$	*/

/*-
 * Copyright (c) 2023 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Robert Swindells.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD$");

#include <sys/param.h>
#include <sys/types.h>
#include <sys/bus.h>
#include <sys/device.h>

#include <dev/fdt/fdtvar.h>

#include <drm/drm_device.h>

#include <linux/device.h>
#include <linux/clk.h>
#include <linux/reset.h>
#include <linux/interrupt.h>
#include <linux/regulator/consumer.h>
#include <linux/dma-mapping.h>
#include <linux/platform_device.h>
#include <linux/io-pgtable.h>

struct gpu_softc {
	struct device		*sc_dev;
	struct drm_device	*sc_ddev;
	void			*sc_pdev;
	int			sc_phandle;
	void			*sc_cookie;
};

struct reset_control {
	struct fdtbus_reset rc_reset;
};

struct platform_irq_cookie {
	
};

struct clk *devm_clk_get(struct device *dev, const char *id)
{
	struct gpu_softc *const sc = device_private(dev);
	struct clk *fdtclk;

	if (id) {
		fdtclk = fdtbus_clock_get(sc->sc_phandle, id);
	} else {
		fdtclk = fdtbus_clock_get_index(sc->sc_phandle, 0);
	}
	return fdtclk;

}

struct clk *devm_clk_get_optional(struct device *dev, const char *id)
{
	struct clk *fdtclk = devm_clk_get(dev, id);

	if (!fdtclk)
		return NULL;
	else
		return fdtclk;
}

int clk_prepare_enable(struct clk *clk)
{

	if (clk_enable(clk) != 0) {
		printf("couldn't enable clock\n");
		return -1;
	}
	return 0;
}

void clk_disable_unprepare(struct clk *clk)
{

	clk_disable(clk);
}

struct reset_control *
devm_reset_control_array_get(struct device *dev, bool shared, bool optional)
{
	struct gpu_softc *const sc = device_private(dev);

	return (struct reset_control *) fdtbus_reset_get_index(sc->sc_phandle, 0);
}

struct reset_control *
devm_reset_control_array_get_optional_shared(struct device *dev)
{
	return devm_reset_control_array_get(dev, true, true);
}

int reset_control_assert(struct reset_control *reset)
{

	return -fdtbus_reset_assert((struct fdtbus_reset *) reset);;
}

int reset_control_deassert(struct reset_control *reset)
{

	return -fdtbus_reset_deassert((struct fdtbus_reset *) reset);
}

int devm_request_irq(struct device *dev, unsigned int irq,
    irq_handler_t handler, unsigned long irqflags,
    const char *devname, void *dev_id)
{
	struct gpu_softc *const sc = device_private(dev);

	sc->sc_ddev->irq_cookie = fdtbus_intr_establish_byname(sc->sc_phandle, devname, IPL_BIO,
	    0, handler, dev_id, devname);

	if (sc->sc_ddev->irq_cookie == NULL)
		aprint_error_dev(sc->sc_dev, "failed to get irq for %s\n",
		    devname);
	return 0;
}

int devm_request_threaded_irq(struct device *dev, unsigned int irq,
    irq_handler_t handler, void (*thread_fn)(void *), unsigned long irqflags,
    const char *devname, void *dev_id)
{
	struct gpu_softc *const sc = device_private(dev);

	sc->sc_ddev->irq_cookie = fdtbus_intr_establish_byname(sc->sc_phandle,
	    devname, IPL_BIO, 0, handler, dev_id, devname);

	if (sc->sc_ddev->irq_cookie == NULL) {
		aprint_error_dev(sc->sc_dev, "failed to get irq for %s\n",
		    devname);
		return 1;
	}
	sc->sc_cookie = softint_establish(SOFTINT_BIO, thread_fn, dev_id);

	return 0;
}

struct regulator *
devm_regulator_get(struct device *dev, const char *id)
{
	struct gpu_softc *const sc = device_private(dev);
	struct fdtbus_regulator *reg;

	reg = fdtbus_regulator_acquire(sc->sc_phandle, id);
	printf("devm_regulator_get: %s %p\n", id, reg);
	return (struct regulator *) reg;
}

struct regulator *
devm_regulator_get_optional(struct device *dev, const char *id)
{
	struct regulator *reg = devm_regulator_get(dev, id);

	return reg;
}

void *dma_alloc_wc(struct device *dev, size_t size,
				bus_dmamap_t *dma_addr, int gfp)
{
	struct gpu_softc *const sc = device_private(dev);
	void *addr;
	int error;

	error = bus_dmamap_create(sc->sc_ddev->dmat, size, 1, size, 0,
		BUS_DMA_NOWAIT, dma_addr);
	if (error)
		goto fail0;
	error = bus_dmamem_alloc(sc->sc_ddev->dmat, size, PAGE_SIZE,
	    0, (*dma_addr)->dm_segs, 1, &(*dma_addr)->dm_nsegs, BUS_DMA_NOWAIT);
	if (error)
		goto fail1;
	error = bus_dmamem_map(sc->sc_ddev->dmat, (*dma_addr)->dm_segs,
	    (*dma_addr)->dm_nsegs, size, &addr,
	    BUS_DMA_NOWAIT | BUS_DMA_COHERENT | BUS_DMA_NOCACHE);
	if (error)
		goto fail2;
	error = bus_dmamap_load(sc->sc_ddev->dmat, *dma_addr, addr,
	    size, NULL, BUS_DMA_NOWAIT | BUS_DMA_NOCACHE);
	if (error)
		goto fail3;

	return addr;
fail3:
	bus_dmamem_unmap(sc->sc_ddev->dmat, *dma_addr, size);
fail2:
	bus_dmamem_free(sc->sc_ddev->dmat,
	    (*dma_addr)->dm_segs, (*dma_addr)->dm_nsegs);
fail1:
	bus_dmamap_destroy(sc->sc_ddev->dmat, *dma_addr);
fail0:
	return (void *) 0;
}

void dma_mmap_wc(struct device *dev, size_t size,
				off_t off, int prot, bus_dmamap_t *dma_addr)
{
	struct gpu_softc *const sc = device_private(dev);

	bus_dmamem_mmap(sc->sc_ddev->dmat, (*dma_addr)->dm_segs,
		(*dma_addr)->dm_nsegs, off, prot,
		BUS_DMA_PREFETCHABLE | BUS_DMA_NOWAIT);
}

void dma_free_wc(struct device *dev, size_t size,
				void *cpu_addr, bus_dmamap_t dma_addr)
{
	struct gpu_softc *const sc = device_private(dev);

	bus_dmamap_unload(sc->sc_ddev->dmat, dma_addr);
	bus_dmamem_unmap(sc->sc_ddev->dmat, cpu_addr, size);
	bus_dmamap_destroy(sc->sc_ddev->dmat, dma_addr);
}

void *dev_get_drvdata(struct device *dev)
{
	return device_private(dev);
}

#if defined(__aarch64__) || defined(__arm__)

int mali_lpae_map(struct io_pgtable_ops *ops, unsigned long iova,
	    bus_addr_t paddr, size_t size, int prot)
{
	struct io_pgtable *iop = io_pgtable_ops_to_pgtable(ops);

	printf("mali_lpae_map: %lx %lx %zx\n", iova, paddr, size);
	return 0;
}

size_t mali_lpae_unmap(struct io_pgtable_ops *ops, unsigned long iova,
	size_t size, void *gather)
{
	return 0;
}

bus_addr_t mali_lpae_iova_to_phys(struct io_pgtable_ops *ops, uint64_t)
{
}

struct io_pgtable_ops mali_lpae_ops {
	.map = mali_lpae_map;
	.unmap = mali_lpae_unmap;
	.iova_to_phys = mali_lpae_iova_to_phys;
};
#endif

struct io_pgtable_ops *
alloc_io_pgtable_ops(enum io_pgtable_fmt fmt, struct io_pgtable_cfg *cfg,
    void *cookie)
{
	struct io_pgtable *iop;

	iop = kmem_alloc(sizeof(*iop), KM_NOSLEEP);
	if (iop == NULL)
		return NULL;

	iop->fmt = fmt;
	iop->cookie = cookie;
	iop->cfg = *cfg;

	iop->ops = mali_lpae_ops;

	return &iop->ops;
}

void free_io_pgtable_ops(struct io_pgtable_ops *ops)
{
	struct io_pgtable *iop = io_pgtable_ops_to_pgtable(ops);

	/* free pagetable */
	kmem_free(iop, sizeof(*iop));
}
