package org.jboss.cache.commands.write;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.cache.CacheException;
import org.jboss.cache.Fqn;
import org.jboss.cache.InvocationContext;
import org.jboss.cache.NodeSPI;
import org.jboss.cache.commands.VersionedDataCommand;
import org.jboss.cache.config.Option;
import org.jboss.cache.optimistic.DataVersion;
import org.jboss.cache.transaction.GlobalTransaction;

import javax.transaction.Transaction;
import javax.transaction.TransactionManager;
import java.util.Collections;

/**
 * Behaves like {@link org.jboss.cache.commands.write.InvalidateCommand}. Also, potentially throws a cache exception if
 * data versioning is used and the node in memory has a newer data version than what is passed in.
 * <p/>
 * Finally, the data version of the in-memory node is updated to the version being evicted to prevent versions
 * going out of sync.
 * <p/>
 *
 * @author Mircea.Markus@jboss.com
 * @since 2.2
 */
public class OptimisticInvalidateCommand extends InvalidateCommand implements VersionedDataCommand
{
   private static final Log log = LogFactory.getLog(OptimisticInvalidateCommand.class);
   private static boolean trace = log.isTraceEnabled();

   /*
     dependencies
    */
   private TransactionManager transactionManager;

   /**
    * Params.
    */
   protected GlobalTransaction globalTransaction;
   private DataVersion dataVersion;

   public OptimisticInvalidateCommand(Fqn fqn)
   {
      super(fqn);
   }

   public OptimisticInvalidateCommand()
   {
   }

   public void initialize(TransactionManager txManager)
   {
      this.transactionManager = txManager;
   }

   @Override
   public Object perform(InvocationContext ctx)
   {
      NodeSPI node = enforceNodeLoading();
      if (trace) log.trace("Invalidating fqn:" + fqn);
      if (node == null)
      {
         // check if a tombstone already exists
         NodeSPI nodeSPI = dataContainer.peek(fqn, false, true);
         if (nodeSPI == null)
         {
            if (dataVersion == null)
            {
               if (trace)
                  log.trace("Would have created a tombstone since the node doesn't exist, but the version to invalidate is null and hence cannot create a tombstone!");
               return null;
            }
            createTombstone(ctx);
            nodeSPI = (NodeSPI) dataContainer.getRoot().getChild(fqn);
         }
         node = nodeSPI;
      }
      removeData(ctx);
      invalidateNode(node);
      updateDataVersion();
      return null;
   }

   protected void createTombstone(InvocationContext ctx)
   {
      if (trace)
         log.trace("Node doesn't exist; creating a tombstone with data version " + dataVersion);
      // create the node we need.
      Option o = ctx.getOptionOverrides();
      boolean origCacheModeLocal = o.isCacheModeLocal();
      o.setCacheModeLocal(true);
      o.setDataVersion(dataVersion);
      // if we are in a tx this call should happen outside of any tx
      try
      {
         Transaction suspended = null;
         if (transactionManager != null)
         {
            suspended = transactionManager.suspend();
         }
         spi.put(fqn, Collections.emptyMap());
         if (suspended != null) transactionManager.resume(suspended);
         ctx.getOptionOverrides().setCacheModeLocal(origCacheModeLocal);
      }
      catch (Exception e)
      {
         log.error("Unable to create tombstone!", e);
      }
   }

   private void updateDataVersion()
   {
      if (dataVersion != null)
      {
         NodeSPI n = dataContainer.peek(fqn, false, true);
         n.setVersion(dataVersion);
      }
   }

   protected void removeData(InvocationContext ctx) throws CacheException
   {
      NodeSPI n = dataContainer.peekVersioned(fqn, dataVersion);
      if (n == null)
      {
         log.warn("node " + fqn + " not found");
         return;
      }
      notifier.notifyNodeEvicted(fqn, true, ctx);
      n.clearDataDirect();
      n.setDataLoaded(false);
      notifier.notifyNodeEvicted(fqn, false, ctx);
   }

   public DataVersion getDataVersion()
   {
      return dataVersion;
   }

   public void setDataVersion(DataVersion dataVersion)
   {
      this.dataVersion = dataVersion;
   }

   public GlobalTransaction getGlobalTransaction()
   {
      return globalTransaction;
   }

   public void setGlobalTransaction(GlobalTransaction gtx)
   {
      this.globalTransaction = gtx;
   }

   public boolean isVersioned()
   {
      return dataVersion != null;
   }

   @Override
   public String toString()
   {
      return "OptimisticInvalidateCommand{" +
            "dataVersion=" + dataVersion +
            " ,fqn=" + fqn +
            '}';
   }

   @Override
   public Object[] getParameters()
   {
      return new Object[]{fqn, dataVersion};
   }

   @Override
   public void setParameters(int commandId, Object[] args)
   {
      fqn = (Fqn) args[0];
      dataVersion = (DataVersion) args[1];
   }

   public void rollback()
   {
      //no op
   }
}
