/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     refine_2d.c                                                    */
/*                                                                          */
/* description:  recursive refinement of 2 dim. hierarchical meshes;        */
/*               implementation of the newest vertex bisection              */
/*               file contains all routines depending on dim == 2;          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

static const EL_INFO *refine_function_2d(const EL_INFO *);

static const EL_INFO *get_refine_patch_2d(const EL_INFO *el_info, 
					  RC_LIST_EL *ref_list,
					  int *n_neighs)
{
  FUNCNAME("get_refine_patch_2d");
#if ALBERTA_DEBUG
  EL             *el = el_info->el;
#endif
  const EL_INFO  *neigh_info;
  int             opp_vertex;

  if (el_info->neigh[2] && el_info->opp_vertex[2] != 2)
  {
/*--------------------------------------------------------------------------*/
/*  neighbour is not compatible devisible; refine neighbour first, store the*/
/*  opp_vertex to traverse back to el                                       */
/*--------------------------------------------------------------------------*/
    opp_vertex = el_info->opp_vertex[2];

    neigh_info = traverse_neighbour(stack_2d, el_info, 2);
    neigh_info->el->mark = MAX(neigh_info->el->mark, 1);
    neigh_info = refine_function_2d(neigh_info);

/*--------------------------------------------------------------------------*/
/*  now go back to the original element and refine the patch                */
/*--------------------------------------------------------------------------*/
    el_info = traverse_neighbour(stack_2d, neigh_info, opp_vertex);
    DEBUG_TEST_EXIT(el_info->el == el, "invalid traverse_neighbour1");
  }
  
  if ((ref_list[1].el_info.el = el_info->neigh[2])) {
    DEBUG_TEST_EXIT(el_info->opp_vertex[2] == 2,
	   "no compatible ref. edge after recursive refinement of neighbour");

/*--------------------------------------------------------------------------*/
/*  NEW CODE: fill the neighbour el_info structure! DK                      */
/*--------------------------------------------------------------------------*/

    neigh_info = traverse_neighbour(stack_2d, el_info, 2);
    ref_list[1].el_info = *neigh_info;

    traverse_neighbour(stack_2d, neigh_info, 2);

    *n_neighs = 2;
  }

  return(el_info);
}

/*--------------------------------------------------------------------------*/
/* AI_bisect_element_2d: bisects a single element into child[0] and child[1]*/
/*--------------------------------------------------------------------------*/

void AI_bisect_element_2d(MESH *mesh, EL *el, DOF *dof[3])
{
  EL        *child[2];
  int        i_child, node;

  child[0] = get_element(mesh);
  child[1] = get_element(mesh);
  child[0]->mark = child[1]->mark = MAX(0, el->mark-1);
  el->mark = 0;

/*--------------------------------------------------------------------------*/
/*  transfer hidden data from parent to children                            */
/*--------------------------------------------------------------------------*/

  if (el->child[1] && 
      ((MESH_MEM_INFO *)mesh->mem_info)->leaf_data_info->refine_leaf_data)
((MESH_MEM_INFO *)mesh->mem_info)->leaf_data_info->refine_leaf_data(el, child);

  AI_free_leaf_data((void *) el->child[1], mesh);

  el->child[0] = child[0];
  el->child[1] = child[1];
  
  if (child[0]->mark > 0) do_more_refine_2d = true;

  if(mesh->n_dof[VERTEX]) {
/*--------------------------------------------------------------------------*/
/*  vertex 2 is the newest vertex                                           */
/*--------------------------------------------------------------------------*/

    child[0]->dof[2] = child[1]->dof[2] = dof[0];

/*--------------------------------------------------------------------------*/
/*  the other vertices are handed on from the parent                        */
/*--------------------------------------------------------------------------*/

    for (i_child = 0; i_child < 2; i_child++) {
      child[i_child]->dof[i_child] = el->dof[2];
      child[i_child]->dof[1-i_child] = el->dof[i_child];
    }
  }

/*--------------------------------------------------------------------------*/
/*  there is one more leaf element, two hierachical elements and one more   */
/*  edge                                        			    */
/*--------------------------------------------------------------------------*/

  if(mesh->n_edges > -1)
    mesh->n_edges++;
  mesh->n_elements++;
  mesh->n_hier_elements += 2;

  if (mesh->n_dof[EDGE]) {
    node = mesh->node[EDGE];

/*--------------------------------------------------------------------------*/
/*  there are dof's in the midpoint of the edges			    */
/*--------------------------------------------------------------------------*/
    child[0]->dof[node + 1] = child[1]->dof[node] = get_dof(mesh, EDGE);

/*--------------------------------------------------------------------------*/
/*  dofs handed on by the parent                                            */
/*--------------------------------------------------------------------------*/
    child[0]->dof[node + 2] = el->dof[node + 1];
    child[1]->dof[node + 2] = el->dof[node];

/*--------------------------------------------------------------------------*/
/*  dofs in the refinement edge                                             */
/*--------------------------------------------------------------------------*/
    child[0]->dof[node] = dof[1];
    child[1]->dof[node + 1] = dof[2];
  }

  if (mesh->n_dof[CENTER]) {
    int   node = mesh->node[CENTER];

/*--------------------------------------------------------------------------*/
/* there are dofs at the barycenter of the triangles                        */
/*--------------------------------------------------------------------------*/
    child[0]->dof[node] = get_dof(mesh, CENTER);
    child[1]->dof[node] = get_dof(mesh, CENTER);
  }

  return;
}

/*--------------------------------------------------------------------------*/

void AI_bisect_patch_2d(MESH *mesh, RC_LIST_EL ref_list[], int n_neighs)
{
  DOF      *dof[3] = {nil, nil, nil};
  EL       *el = ref_list[0].el_info.el, *neigh = ref_list[1].el_info.el;
  int       node, i;

/*--------------------------------------------------------------------------*/
/*  there is one new vertex in the refinement edge                          */
/*--------------------------------------------------------------------------*/

  if(mesh->n_dof[VERTEX])
    dof[0] = get_dof(mesh, VERTEX);

  if(mesh->n_vertices > -1)
    mesh->n_vertices++;
  if(mesh->n_edges > -1)
    mesh->n_edges++;

  if (mesh->n_dof[EDGE]) {
/*--------------------------------------------------------------------------*/
/*  there are two additional dofs in the refinement edge                    */
/*--------------------------------------------------------------------------*/
    dof[1] = get_dof(mesh, EDGE);
    dof[2] = get_dof(mesh, EDGE);
  }

/*--------------------------------------------------------------------------*/
/*  first refine the element                                                */
/*--------------------------------------------------------------------------*/
  AI_bisect_element_2d(mesh, el, dof);

  if (neigh) {
    DOF    *tmp = dof[1];
/*--------------------------------------------------------------------------*/
/*  there is a neighbour; refine it also, but first exchange dof[1] and     */
/*  dof[2]; thus, dof[1] is always added on child[0]!                       */
/*--------------------------------------------------------------------------*/
    dof[1] = dof[2];
    dof[2] = tmp;

    AI_bisect_element_2d(mesh, neigh, dof);
  }
  
/*--------------------------------------------------------------------------*/
/*  if there are functions to interpolate data to the finer grid, do so     */
/*--------------------------------------------------------------------------*/
  
  if (call_refine_interpol_2d)  refine_interpol(mesh, ref_list, n_neighs);

/*--------------------------------------------------------------------------*/
/*  if there should be no dof information on interior leaf elements remove  */
/*  dofs from edges and the centers of parents                              */
/*--------------------------------------------------------------------------*/
  if (mesh->n_dof[EDGE]) {
/*--------------------------------------------------------------------------*/
/*  the only DOF that can be freed is that in the refinement edge; all other*/
/*  DOFs are handed on the children                                         */
/*--------------------------------------------------------------------------*/
    node = mesh->node[EDGE];

    free_dof(el->dof[node+2], mesh, EDGE, true);
  }

  if (mesh->n_dof[CENTER]) {
    node = mesh->node[CENTER];

    for (i = 0; i < n_neighs; i++)
      free_dof(ref_list[i].el_info.el->dof[node], mesh, CENTER, true);
  }

  return;
}


/*--------------------------------------------------------------------------*/
/*  refine_function_2d: gets the refinement patch via get_refine_patch_2d() */
/*  and checks whether it compatibly divisible (same refinement edge) or    */
/*  not; in the first case the patch is refined by refine_patch_2d() in the */
/*  second the incompatible neighbour(s) is (are) refined first via a       */
/*  recursive call of refine_function_2d() in get_refine_patch_2d().        */
/*--------------------------------------------------------------------------*/

static const EL_INFO *refine_function_2d(const EL_INFO *el_info)
{
  int          n_neighs;
  MESH         *mesh = el_info->mesh;
  RC_LIST_EL   ref_list[2];

  if (el_info->el->mark <= 0)  
    return(el_info);    /*   element may not be refined   */

  ref_list->el_info = *el_info;
  n_neighs = 1;

/*--------------------------------------------------------------------------*/
/*  get the refinement patch                                                */
/*--------------------------------------------------------------------------*/
  el_info = get_refine_patch_2d(el_info, ref_list, &n_neighs);

/*--------------------------------------------------------------------------*/
/*  and now refine the patch                                                */
/*--------------------------------------------------------------------------*/
  AI_bisect_patch_2d(mesh, ref_list, n_neighs);

  return(el_info);
}

/*--------------------------------------------------------------------------*/
/*  AI_post_refine_2d(mesh): Projects all new vertices with user supplied   */
/*  projection routines. Also called from submesh_3d.c                      */
/*--------------------------------------------------------------------------*/

static void new_coords_fct_2d(const EL_INFO *el_info, void *data)
{
  EL      *el = el_info->el;
  int     j;
  static const REAL_B mid_lambda = {0.5, 0.5, 0.0};

  if (el->child[0] && el_info->active_projection
      && el_info->active_projection->func && (el->new_coord == nil))
  {
    el->new_coord = get_real_d(el_info->mesh);

    for (j = 0; j < DIM_OF_WORLD; j++)
      el->new_coord[j] = (el_info->coord[0][j] + el_info->coord[1][j])*0.5;
    el_info->active_projection->func(el->new_coord, el_info, mid_lambda);

/*--------------------------------------------------------------------------*/
/*  new_coord entry must also be set on the neighbor along the refinement   */
/*  edge, if this neighbor exists.                                          */
/*--------------------------------------------------------------------------*/
    if(el_info->neigh[2] && !el_info->neigh[2]->new_coord)
      el_info->neigh[2]->new_coord = el->new_coord;
  }

  return;
}

void AI_post_refine_2d(MESH *mesh)
{
  FLAGS fill_flag = CALL_EVERY_EL_PREORDER|FILL_PROJECTION|FILL_COORDS|FILL_NEIGH;
  mesh_traverse(mesh, -1, fill_flag, new_coords_fct_2d, nil);

  return;
}


static U_CHAR refine_2d(MESH *mesh)
{
  int         n_elements = mesh->n_elements;
  FLAGS       fill_flag = CALL_LEAF_EL|FILL_NEIGH|FILL_BOUND;
  const EL_INFO *el_info;

  if(mesh->parametric)
    fill_flag |= FILL_PROJECTION;

  call_refine_interpol_2d = count_refine_interpol(mesh);

  stack_2d = get_traverse_stack();
  do_more_refine_2d = true;
  while (do_more_refine_2d)
  {
    do_more_refine_2d = false;
    el_info = traverse_first(stack_2d, mesh, -1, fill_flag);

    while (el_info)
    {
      if (el_info->el->mark > 0)
      {
	do_more_refine_2d |= (el_info->el->mark > 1);
	el_info = refine_function_2d(el_info);
      }
      el_info = traverse_next(stack_2d, el_info);
    }
  }

  free_traverse_stack(stack_2d);

  n_elements = mesh->n_elements - n_elements;

  return(n_elements ? MESH_REFINED : 0);
}

