/*
 *      uvcvideo.c  --  USB Video Class driver
 *
 *      Copyright (C) 2005-2006
 *          Laurent Pinchart (laurent.pinchart@skynet.be)
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 2 of the License, or
 *      (at your option) any later version.
 *
 */

#include <linux/kernel.h>
#include <linux/version.h>
#include <linux/list.h>
#include <linux/module.h>
#include <linux/usb.h>
#include <linux/videodev.h>
#include <linux/vmalloc.h>
#include <linux/wait.h>
#include <asm/atomic.h>

#include "uvcvideo.h"
#include "isight.h"

/* ------------------------------------------------------------------------
 * UVC Controls
 */

int uvc_query_ctrl(struct uvc_device *dev, __u8 query, __u8 unit,
			__u8 intfnum, __u8 cs, void *data, __u16 size)
{
	__u8 type = USB_TYPE_CLASS | USB_RECIP_INTERFACE;
	unsigned int pipe;
	int ret;

	pipe = (query & 0x80) ? usb_rcvctrlpipe(dev->udev, 0)
			      : usb_sndctrlpipe(dev->udev, 0);
	type |= (query & 0x80) ? USB_DIR_IN : USB_DIR_OUT;

	ret = usb_control_msg(dev->udev, pipe, query, type, cs << 8,
			unit << 8 | intfnum, data, size, UVC_CTRL_TIMEOUT);

	if (ret != size) {
		uvc_printk(KERN_ERR, "Failed to query (%u) UVC control %u "
			"(unit %u) : %d (exp. %u).\n", query, cs, unit, ret,
			size);
		return -EIO;
	}

	return 0;
}

static int uvc_get_video_ctrl(struct uvc_video_device *video,
	struct uvc_streaming_control *ctrl, int probe, __u8 query)
{
	__u8 data[34];
	__u8 size;
	int ret;

	size = video->dev->uvc_version > 0x0100 ? 34 : 26;
	ret = uvc_query_ctrl(video->dev, query, 0, video->streaming->intfnum,
		probe ? VS_PROBE_CONTROL : VS_COMMIT_CONTROL, &data, size);

	if (ret < 0)
		return ret;

	ctrl->bmHint = le16_to_cpup((__le16*)&data[0]);
	ctrl->bFormatIndex = data[2];
	ctrl->bFrameIndex = data[3];
	ctrl->dwFrameInterval = le32_to_cpup((__le32*)&data[4]);
	ctrl->wKeyFrameRate = le16_to_cpup((__le16*)&data[8]);
	ctrl->wPFrameRate = le16_to_cpup((__le16*)&data[10]);
	ctrl->wCompQuality = le16_to_cpup((__le16*)&data[12]);
	ctrl->wCompWindowSize = le16_to_cpup((__le16*)&data[14]);
	ctrl->wDelay = le16_to_cpup((__le16*)&data[16]);
	ctrl->dwMaxVideoFrameSize = le32_to_cpup((__le32*)&data[18]);
	ctrl->dwMaxPayloadTransferSize = le32_to_cpup((__le32*)&data[22]);

	if (size == 34) {
		ctrl->dwClockFrequency = le32_to_cpup((__le32*)&data[26]);
		ctrl->bmFramingInfo = data[30];
		ctrl->bPreferedVersion = data[31];
		ctrl->bMinVersion = data[32];
		ctrl->bMaxVersion = data[33];
	}
	else {
		ctrl->dwClockFrequency = video->dev->clock_frequency;
		ctrl->bmFramingInfo = 0;
		ctrl->bPreferedVersion = 0;
		ctrl->bMinVersion = 0;
		ctrl->bMaxVersion = 0;
	}

	if (ctrl->dwMaxVideoFrameSize == 0 && video->dev->uvc_version <= 0x0100) {
		/* Some broken UVC 1.0 devices return a null
		 * dwMaxVideoFrameSize. Try to get the value from the format
		 * and frame descriptor.
		 */
		struct uvc_format *format = NULL;
		struct uvc_frame *frame = NULL;

		if (ctrl->bFormatIndex <= video->streaming->nformats)
			format = &video->streaming->format[ctrl->bFormatIndex - 1];
		if (format && ctrl->bFrameIndex <= format->nframes) {
			frame = &format->frame[ctrl->bFrameIndex - 1];
			ctrl->dwMaxVideoFrameSize = frame->dwMaxVideoFrameBufferSize;
		}
	}

	return 0;
}

int uvc_set_video_ctrl(struct uvc_video_device *video,
	struct uvc_streaming_control *ctrl, int probe)
{
	__u8 data[34];
	__u8 size;

	size = video->dev->uvc_version > 0x0100 ? 34 : 26;
	memset(data, 0, sizeof data);

	*(__le16*)&data[0] = cpu_to_le16(ctrl->bmHint);
	data[2] = ctrl->bFormatIndex;
	data[3] = ctrl->bFrameIndex;
	*(__le32*)&data[4] = cpu_to_le32(ctrl->dwFrameInterval);
	*(__le16*)&data[8] = cpu_to_le16(ctrl->wKeyFrameRate);
	*(__le16*)&data[10] = cpu_to_le16(ctrl->wPFrameRate);
	*(__le16*)&data[12] = cpu_to_le16(ctrl->wCompQuality);
	*(__le16*)&data[14] = cpu_to_le16(ctrl->wCompWindowSize);
	*(__le16*)&data[16] = cpu_to_le16(ctrl->wDelay);
	/* Note: Some of the fields below are not required for IN devices (see
	 * UVC spec, 4.3.1.1), but we still copy them in case support for OUT
	 * devices is added in the future. */
	*(__le32*)&data[18] = cpu_to_le32(ctrl->dwMaxVideoFrameSize);
	*(__le32*)&data[22] = cpu_to_le32(ctrl->dwMaxPayloadTransferSize);

	if (size == 34) {
		*(__le32*)&data[26] = cpu_to_le32(ctrl->dwClockFrequency);
		data[30] = ctrl->bmFramingInfo;
		data[31] = ctrl->bPreferedVersion;
		data[32] = ctrl->bMinVersion;
		data[33] = ctrl->bMaxVersion;
	}

	return uvc_query_ctrl(video->dev, SET_CUR, 0,
		video->streaming->intfnum,
		probe ? VS_PROBE_CONTROL : VS_COMMIT_CONTROL, &data, size);
}

int uvc_probe_video(struct uvc_video_device *video, struct uvc_streaming_control *probe)
{
	struct uvc_streaming_control probe_min, probe_max;
	__u16 bandwidth;
	unsigned int i;
	int ret;

	mutex_lock(&video->streaming->mutex);

	/* Perform probing. The device should adjust the requested values
	 * according to its capabilities. However, some devices, namely the
	 * first generation UVC Logitech webcams, don't implement the Video
	 * Probe control properly, and just return the needed bandwidth. For
	 * that reason, if the needed bandwidth exceeds the maximum available
	 * bandwidth, try to lower the quality.
	 */
	if ((ret = uvc_set_video_ctrl(video, probe, 1)) < 0)
		goto done;

	/* Get the minimum and maximum values for compression settings. */
	if (!(video->dev->quirks & UVC_QUIRK_PROBE_MINMAX)) {
		if ((ret = uvc_get_video_ctrl(video, &probe_min, 1, GET_MIN)) < 0 ||
		    (ret = uvc_get_video_ctrl(video, &probe_max, 1, GET_MAX)) < 0)
			goto done;

		probe->wCompQuality = probe_max.wCompQuality;
	}

	for (i = 0; i < 2; ++i) {
		if ((ret = uvc_set_video_ctrl(video, probe, 1)) < 0 ||
		    (ret = uvc_get_video_ctrl(video, probe, 1, GET_CUR)) < 0)
			goto done;

		bandwidth = probe->dwMaxPayloadTransferSize;
		if (bandwidth <= video->streaming->maxpsize)
			break;

		if (video->dev->quirks & UVC_QUIRK_PROBE_MINMAX) {
			ret = -ENOSPC;
			goto done;
		}

		/* TODO: negotiate compression parameters */
		probe->wKeyFrameRate = probe_min.wKeyFrameRate;
		probe->wPFrameRate = probe_min.wPFrameRate;
		probe->wCompQuality = probe_max.wCompQuality;
		probe->wCompWindowSize = probe_min.wCompWindowSize;
	}

done:
	mutex_unlock(&video->streaming->mutex);
	return ret;
}

/* ------------------------------------------------------------------------
 * Video codecs
 */

/* Values for bmHeaderInfo (Video and Still Image Payload Headers, 2.4.3.3) */
#define UVC_STREAM_EOH	(1 << 7)
#define UVC_STREAM_ERR	(1 << 6)
#define UVC_STREAM_STI	(1 << 5)
#define UVC_STREAM_RES	(1 << 4)
#define UVC_STREAM_SCR	(1 << 3)
#define UVC_STREAM_PTS	(1 << 2)
#define UVC_STREAM_EOF	(1 << 1)
#define UVC_STREAM_FID	(1 << 0)

static int uvc_video_decode_uvc(struct uvc_video_queue *queue,
		struct uvc_buffer *buf, const __u8 *data, unsigned int len)
{
	unsigned int maxlen, nbytes;
	void *mem;
	__u8 fid;

	/* Sanity checks:
	 * - packet must be at least 2 bytes long
	 * - bHeaderLength value must be at least 2 bytes (see above)
	 * - bHeaderLength value can't be larger than the packet size.
	 */
	if (len < 2 || data[0] < 2 || data[0] > len)
		return -EINVAL;

	/* Skip payloads marked with the error bit ("error frames"). */
	if (data[1] & UVC_STREAM_ERR) {
		uvc_trace(UVC_TRACE_FRAME, "Dropping packet (error bit set).\n");
		return 0;
	}

	fid = data[1] & UVC_STREAM_FID;

	/* Store the packet FID bit and return immediately when the buffer is
	 * NULL.
	 */
	if (buf == NULL) {
		queue->last_fid = fid;
		return 0;
	}

	/* Synchronize to the input stream by waiting for the FID bit to be
	 * toggled when the the buffer state is not UVC_BUF_STATE_ACTIVE.
	 * queue->last_fid is initialized to -1, so the first isochronous
	 * frame will always be in sync.
	 */
	if (buf->state != UVC_BUF_STATE_ACTIVE) {
		if (fid == queue->last_fid) {
			uvc_trace(UVC_TRACE_FRAME, "Dropping packet (out of "
				"sync).\n");
			return 0;
		}

		/* TODO: Handle PTS and SCR. */
		buf->state = UVC_BUF_STATE_ACTIVE;
	}

	/* Mark the buffer as done if we're at the beginning of a new frame.
	 * End of frame detection is better implemented by checking the EOF
	 * bit (FID bit toggling is delayed by one frame compared to the EOF
	 * bit), but some devices don't set the bit at end of frame (and the
	 * last packet can be lost anyway). We thus must check if the FID has
	 * been toggled.
	 *
	 * queue->last_fid is initialized to -1, so the first isochronous
	 * frame will never trigger an end of frame detection.
	 *
	 * Empty buffers (bytesused == 0) don't trigger end of frame detection
	 * as it doesn't make sense to return an empty buffer. This also
	 * avoids detecting and of frame conditions at FID toggling if the
	 * previous packet had the EOF bit set.
	 */
	if (fid != queue->last_fid && buf->buf.bytesused != 0) {
		uvc_trace(UVC_TRACE_FRAME, "Frame complete (FID bit "
				"toggled).\n");
		buf->state = UVC_BUF_STATE_DONE;
		return -EAGAIN;
	}

	queue->last_fid = fid;

	/* Copy the video data to the buffer. */
	len -= data[0];
	maxlen = buf->buf.length - buf->buf.bytesused;
	mem = queue->mem + buf->buf.m.offset + buf->buf.bytesused;
	nbytes = min(len, maxlen);
	memcpy(mem, data + data[0], nbytes);
	buf->buf.bytesused += nbytes;

	/* Drop the current frame if the buffer size was exceeded. */
	if (len > maxlen) {
		uvc_trace(UVC_TRACE_FRAME, "Frame complete (overflow).\n");
		buf->state = UVC_BUF_STATE_DONE;
	}

	/* Mark the buffer as done if the EOF marker is set. */
	if (data[1] & UVC_STREAM_EOF && buf->buf.bytesused != 0) {
		uvc_trace(UVC_TRACE_FRAME, "Frame complete (EOF found).\n");
		if (data[0] == len)
			uvc_trace(UVC_TRACE_FRAME, "EOF in empty packet.\n");
		buf->state = UVC_BUF_STATE_DONE;
	}

	return 0;
}

static int uvc_video_decode(struct uvc_video_queue *queue,
		struct uvc_buffer *buf, const __u8 *data, unsigned int len)
{
	return queue->decode(queue, buf, data, len);
}

/* ------------------------------------------------------------------------
 * URB handling
 */

/*
 * Completion handler for status URB.
 */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,19)
static void uvc_status_complete(struct urb *urb, struct pt_regs *regs)
#else
static void uvc_status_complete(struct urb *urb)
#endif
{
	struct uvc_device *dev = urb->context;
	int len, ret;

	switch (urb->status) {
	case 0:
		break;

	case -ENOENT:		/* usb_kill_urb() called. */
	case -ECONNRESET:	/* usb_unlink_urb() called. */
	case -ESHUTDOWN:	/* The endpoint is being disabled. */
	case -EPROTO:		/* Device is disconnected (reported by some 
				 * host controller). */
		return;

	default:
		uvc_printk(KERN_WARNING, "Non-zero status (%d) in status "
			"completion handler.\n", urb->status);
		return;
	}

	len = urb->actual_length;
	if (len >= 4 && (dev->status[0] & 0x0f) == 2) {
		if (dev->status[2] == 0)
			uvc_printk(KERN_DEBUG, "Button event (%d).\n",
				dev->status[3]);
	}

	/* Resubmit the URB. */
	urb->interval = dev->int_ep->desc.bInterval;
	if ((ret = usb_submit_urb(urb, GFP_ATOMIC)) < 0) {
		uvc_printk(KERN_ERR, "Failed to resubmit status URB (%d).\n",
			ret);
	}
}

/*
 * Initialise the status endpoint.
 */
int uvc_init_status(struct uvc_device *dev)
{
	struct usb_endpoint_descriptor *desc = &dev->int_ep->desc;
	unsigned int pipe;
	int interval = desc->bInterval;

	dev->int_urb = usb_alloc_urb(0, GFP_KERNEL);
	if (dev->int_urb == NULL)
		return -ENOMEM;

	pipe = usb_rcvintpipe(dev->udev, desc->bEndpointAddress);

	/* For high-speed interrupt endpoints, the bInterval value is used as
	 * an exponent of two. Some developers forgot about it.
	 */
	if (interval > 16 && dev->udev->speed == USB_SPEED_HIGH &&
	    (dev->quirks & UVC_QUIRK_STATUS_INTERVAL))
		interval = fls(interval) - 1;

	usb_fill_int_urb(dev->int_urb, dev->udev, pipe,
		dev->status, sizeof dev->status, uvc_status_complete,
		dev, interval);

	return usb_submit_urb(dev->int_urb, GFP_KERNEL);
}

/*
 * Completion handler for video URBs.
 */
static void uvc_video_complete_isoc(struct urb *urb,
	struct uvc_video_queue *queue, struct uvc_buffer *buf)
{
	int ret, i;

	for (i = 0; i < urb->number_of_packets; ++i) {
		if (urb->iso_frame_desc[i].status < 0) {
			uvc_trace(UVC_TRACE_FRAME, "USB isochronous frame "
				"lost (%d).\n", urb->iso_frame_desc[i].status);
			continue;
		}

		/* Decode the payload packet.
		 * uvc_video_decode is entered twice when a frame transition
		 * has been detected because the end of frame can only be
		 * reliably detected when the first packet of the new frame
		 * is processed. The first pass detects the transition and
		 * closes the previous frame's buffer, the second pass
		 * processes the data of the first payload of the new frame.
		 */
		do {
			ret = uvc_video_decode(queue, buf,
				urb->transfer_buffer + urb->iso_frame_desc[i].offset,
				urb->iso_frame_desc[i].actual_length);

			if (buf == NULL)
				break;

			if (buf->state == UVC_BUF_STATE_DONE ||
			    buf->state == UVC_BUF_STATE_ERROR)
				buf = uvc_queue_next_buffer(queue, buf);
		} while (ret == -EAGAIN);
	}
}

static void uvc_video_complete_bulk(struct urb *urb,
	struct uvc_video_queue *queue, struct uvc_buffer *buf)
{
	int ret;

	/* Decode the payload packet.
	 * uvc_video_decode is entered twice when a frame transition
	 * has been detected because the end of frame can only be
	 * reliably detected when the first packet of the new frame
	 * is processed. The first pass detects the transition and
	 * closes the previous frame's buffer, the second pass
	 * processes the data of the first payload of the new frame.
	 */
	do {
		ret = uvc_video_decode(queue, buf,
			urb->transfer_buffer, urb->actual_length);

		if (buf == NULL)
			break;

		if (buf->state == UVC_BUF_STATE_DONE ||
		    buf->state == UVC_BUF_STATE_ERROR)
			buf = uvc_queue_next_buffer(queue, buf);
	} while (ret == -EAGAIN);
}

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,19)
static void uvc_video_complete(struct urb *urb, struct pt_regs *regs)
#else
static void uvc_video_complete(struct urb *urb)
#endif
{
	struct uvc_video_device *video = urb->context;
	struct uvc_video_queue *queue = &video->queue;
	struct uvc_buffer *buf = NULL;
	unsigned long flags;
	int ret;

	switch (urb->status) {
	case 0:
		break;

	default:
		uvc_printk(KERN_WARNING, "Non-zero status (%d) in video "
			"completion handler.\n", urb->status);

	case -ENOENT:		/* usb_kill_urb() called. */
	case -ECONNRESET:	/* usb_unlink_urb() called. */
	case -ESHUTDOWN:	/* The endpoint is being disabled. */
		uvc_queue_cancel(queue);
		return;
	}

	spin_lock_irqsave(&queue->irqlock, flags);
	if (!list_empty(&queue->irqqueue))
		buf = list_entry(queue->irqqueue.next, struct uvc_buffer, queue);
	spin_unlock_irqrestore(&queue->irqlock, flags);

	if (urb->number_of_packets)
		uvc_video_complete_isoc(urb, queue, buf);
	else
		uvc_video_complete_bulk(urb, queue, buf);

	if ((ret = usb_submit_urb(urb, GFP_ATOMIC)) < 0) {
		uvc_printk(KERN_ERR, "Failed to resubmit video URB (%d).\n",
			ret);
        }
}

/*
 * Uninitialize isochronous/bulk URBs and free transfer buffers.
 */
static void uvc_uninit_video(struct uvc_video_device *video)
{
	struct urb *urb;
	unsigned int i;

	for (i = 0; i < UVC_URBS; ++i) {
		if ((urb = video->urb[i]) == NULL)
			continue;

		usb_kill_urb(urb);
		/* urb->transfer_buffer_length is not touched by USB core, so
		 * we can use it here as the buffer length.
		 */
		if (video->urb_buffer[i]) {
			usb_buffer_free(video->dev->udev,
				urb->transfer_buffer_length,
				video->urb_buffer[i], urb->transfer_dma);
			video->urb_buffer[i] = NULL;
		}

		usb_free_urb(urb);
		video->urb[i] = NULL;
	}
}

/*
 * Initialize isochronous URBs and allocate transfer buffers. The packet size
 * is given by the endpoint.
 */
static int uvc_init_video_isoc(struct uvc_video_device *video, struct usb_host_endpoint *ep)
{
	struct urb *urb;
	unsigned int npackets, i, j;
	__u16 psize;
	__u32 size;
	int ret;

	psize = le16_to_cpu(ep->desc.wMaxPacketSize);
	psize = (psize & 0x07ff) * (1 + ((psize >> 11) & 3));

	size = video->streaming->ctrl.dwMaxVideoFrameSize;
	if (size > UVC_MAX_FRAME_SIZE)
		return -EINVAL;

	npackets = (size + psize - 1) / psize;
	if (npackets > UVC_MAX_ISO_PACKETS)
		npackets = UVC_MAX_ISO_PACKETS;

	size = npackets * psize;

	for (i = 0; i < UVC_URBS; ++i) {
		urb = usb_alloc_urb(npackets, GFP_KERNEL);
		if (urb == NULL) {
			uvc_uninit_video(video);
			return -ENOMEM;
		}

		video->urb_buffer[i] = usb_buffer_alloc(video->dev->udev,
			size, GFP_KERNEL, &urb->transfer_dma);
		if (video->urb_buffer[i] == NULL) {
			usb_free_urb(urb);
			uvc_uninit_video(video);
			return -ENOMEM;
		}

		urb->dev = video->dev->udev;
		urb->context = video;
		urb->pipe = usb_rcvisocpipe(video->dev->udev,
				ep->desc.bEndpointAddress);
		urb->transfer_flags = URB_ISO_ASAP | URB_NO_TRANSFER_DMA_MAP;
		urb->interval = ep->desc.bInterval;
		urb->transfer_buffer = video->urb_buffer[i];
		urb->complete = uvc_video_complete;
		urb->number_of_packets = npackets;
		urb->transfer_buffer_length = size;

		for (j = 0; j < npackets; ++j) {
			urb->iso_frame_desc[j].offset = j * psize;
			urb->iso_frame_desc[j].length = psize;
		}

		video->urb[i] = urb;
	}

	/* Submit the URBs. */
	for (i = 0; i < UVC_URBS; ++i) {
		if ((ret = usb_submit_urb(video->urb[i], GFP_KERNEL)) < 0) {
			uvc_printk(KERN_ERR, "Failed to submit isoc URB %u "
					"(%d).\n", i, ret);
			uvc_uninit_video(video);
			return ret;
		}
	}

	return 0;
}

/*
 * Initialize bulk URBs and allocate transfer buffers. The packet size is
 * given by the endpoint.
 */
static int uvc_init_video_bulk(struct uvc_video_device *video, struct usb_host_endpoint *ep)
{
	struct urb *urb;
	unsigned int pipe, i;
	__u32 size;
	int ret;

	size = video->streaming->ctrl.dwMaxPayloadTransferSize;
	if (size > UVC_MAX_FRAME_SIZE)
		return -EINVAL;

	pipe = usb_rcvbulkpipe(video->dev->udev, ep->desc.bEndpointAddress);

	for (i = 0; i < UVC_URBS; ++i) {
		urb = usb_alloc_urb(0, GFP_KERNEL);
		if (urb == NULL) {
			uvc_uninit_video(video);
			return -ENOMEM;
		}

		video->urb_buffer[i] = usb_buffer_alloc(video->dev->udev,
			size, GFP_KERNEL, &urb->transfer_dma);
		if (video->urb_buffer[i] == NULL) {
			usb_free_urb(urb);
			uvc_uninit_video(video);
			return -ENOMEM;
		}

		usb_fill_bulk_urb(urb, video->dev->udev, pipe,
			video->urb_buffer[i], size, uvc_video_complete,
			video);
		urb->transfer_flags = URB_NO_TRANSFER_DMA_MAP;

		video->urb[i] = urb;
	}

	/* Submit the URBs. */
	for (i = 0; i < UVC_URBS; ++i) {
		if ((ret = usb_submit_urb(video->urb[i], GFP_KERNEL)) < 0) {
			uvc_printk(KERN_ERR, "Failed to submit bulk URB %u "
					"(%d).\n", i, ret);
			uvc_uninit_video(video);
			return ret;
		}
	}

	return 0;
}

/* ------------------------------------------------------------------------
 * Video device
 */

/*
 * Initialize the UVC video device.
 *
 * This function is called before registering the device with V4L.
 */
int uvc_video_init(struct uvc_video_device *video)
{
	struct uvc_streaming_control *probe = &video->streaming->ctrl;
	struct uvc_format *format = NULL;
	struct uvc_frame *frame = NULL;
	int ret;

	/* Retrieve the default format and commit it. Some cameras (namely the
	 * Fuji Finepix) set the format and frame indexes to zero. The UVC
	 * standard doesn't clearly make this a spec violation, so just set
	 * the fields to some sensible value.
	 *
	 * Some webcams don't suport GET_DEF request on the probe control. We
	 * fall back to GET_CUR if GET_DEF fails.
	 */
	if ((ret = uvc_get_video_ctrl(video, probe, 1, GET_DEF)) < 0 &&
	    (ret = uvc_get_video_ctrl(video, probe, 1, GET_CUR)) < 0)
		return ret;

	if (probe->bFormatIndex == 0)
		probe->bFormatIndex = 1;
	if (probe->bFrameIndex == 0)
		probe->bFrameIndex = 1;

	if ((ret = uvc_set_video_ctrl(video, probe, 0)) < 0)
		return ret;

	/* Validate the default format and frame. */
	if (probe->bFormatIndex <= video->streaming->nformats)
		format = &video->streaming->format[probe->bFormatIndex - 1];
	if (format && probe->bFrameIndex <= format->nframes)
		frame = &format->frame[probe->bFrameIndex - 1];

	if (format == NULL || frame == NULL) {
		uvc_printk(KERN_WARNING "Device initialization failed: the "
			"default format and/or frame are not valid.\n");
		return -EINVAL;
	}

	video->streaming->cur_format = format;
	video->streaming->cur_frame = frame;
	atomic_set(&video->active, 1);

	/* Select the video decoder based. This might be better handled
	 * elsewhere, based on the format GUID instead of the device ID.
	 */
	if (is_isight (video->dev->udev) == 0)
		video->queue.decode = isight_decode_video;
	else
		video->queue.decode = uvc_video_decode_uvc;

	return 0;
}

/*
 * Enable or disable the video stream.
 */
int uvc_video_enable(struct uvc_video_device *video, int enable)
{
	struct usb_interface *intf = video->streaming->intf;
	struct usb_host_interface *alts;
	struct usb_host_endpoint *ep;
	int intfnum = video->streaming->intfnum;
	unsigned int bandwidth, psize, i;
	int ret;

	if (!enable) {
		uvc_uninit_video(video);
		usb_set_interface(video->dev->udev, intfnum, 0);
		uvc_queue_enable(&video->queue, 0);
		return 0;
	}

	if ((ret = uvc_queue_enable(&video->queue, 1)) < 0) 
		return ret;

	if (intf->num_altsetting > 1) {
		/* Isochronous endpoint, select the alternate setting. */
		bandwidth = video->streaming->ctrl.dwMaxPayloadTransferSize;

		for (i = 0; i < intf->num_altsetting; ++i) {
			alts = &intf->altsetting[i];
			ep = uvc_find_endpoint(alts,
					video->streaming->input.bEndpointAddress);
			if (ep == NULL)
				continue;

			/* Check if the bandwidth is high enough. */
			psize = le16_to_cpu(ep->desc.wMaxPacketSize);
			psize = (psize & 0x07ff) * (1 + ((psize >> 11) & 3));
			if (psize >= bandwidth)
				break;
		}

		if (i >= intf->num_altsetting)
			return -EIO;

		if ((ret = usb_set_interface(video->dev->udev, intfnum, i)) < 0)
			return ret;

		return uvc_init_video_isoc(video, ep);
	} else {
		/* Bulk endpoint, proceed to URB initialization. */
		ep = uvc_find_endpoint(&intf->altsetting[0],
				video->streaming->input.bEndpointAddress);
		if (ep == NULL)
			return -EIO;

		return uvc_init_video_bulk(video, ep);
	}
}

