/**************************************************************************
 * 
 * Copyright 2006-2007 Tungsten Graphics, Inc., Cedar Park, TX., USA
 * Copyright (c) Intel Corp. 2007.
 * All Rights Reserved.
 * 
 * Intel funded Tungsten Graphics (http://www.tungstengraphics.com) to
 * develop this code.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sub license, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDERS, AUTHORS AND/OR ITS SUPPLIERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR 
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE 
 * USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 * 
 * 
 **************************************************************************/
/*
 * Authors: Thomas Hellstrm <thomas-at-tungstengraphics-dot-com>
 */

#include "drmP.h"
#include "psb_drv.h"

/*
 * Implements an Intel Poulsbo sync flush operation.
 */

static void psb_perform_flush0(drm_device_t * dev)
{
	drm_psb_private_t *dev_priv = (drm_psb_private_t *) dev->dev_private;
	drm_fence_manager_t *fm = &dev->fm;
	drm_fence_class_manager_t *fc = &fm->class[0];
	drm_fence_driver_t *driver = dev->driver->fence_driver;
	uint32_t diff;
	uint32_t sequence;

	if (!dev_priv)
		return;

	if (fc->pending_exe_flush) {
		sequence = dev_priv->comm[0];

		/*
		 * First update fences with the current breadcrumb.
		 */

		diff = sequence - fc->last_exe_flush;
		if (diff < driver->wrap_diff && diff != 0) {
			drm_fence_handler(dev, 0, sequence, DRM_FENCE_TYPE_EXE);
		}

		if (dev_priv->fence0_irq_on && !fc->pending_exe_flush) {
			psb_2D_irq_off(dev_priv);
			dev_priv->fence0_irq_on = 0;
		} else if (!dev_priv->fence0_irq_on && fc->pending_exe_flush) {
			psb_2D_irq_on(dev_priv);
			dev_priv->fence0_irq_on = 1;
		}
	}
}

void psb_poke_flush(drm_device_t * dev, uint32_t class)
{
	drm_fence_manager_t *fm = &dev->fm;
	unsigned long flags;

	BUG_ON(class != 0);
	write_lock_irqsave(&fm->lock, flags);
	psb_perform_flush0(dev);
	write_unlock_irqrestore(&fm->lock, flags);
}

int psb_fence_emit_sequence(drm_device_t * dev, uint32_t class, uint32_t flags,
			    uint32_t * sequence, uint32_t * native_type)
{
	drm_psb_private_t *dev_priv = (drm_psb_private_t *) dev->dev_private;
	int ret;

	if (!dev_priv)
		return -EINVAL;

	if (class != 0)
		return -EINVAL;

	++dev_priv->sequence;
	ret = psb_blit_sequence(dev_priv);
	if (ret)
		return ret;

	*sequence = dev_priv->sequence;
	*native_type = DRM_FENCE_TYPE_EXE;
	return 0;
}

void psb_fence_handler(drm_device_t * dev, uint32_t class)
{
	drm_fence_manager_t *fm = &dev->fm;

	write_lock(&fm->lock);
	psb_perform_flush0(dev);
	write_unlock(&fm->lock);
}

int psb_fence_has_irq(drm_device_t * dev, uint32_t class, uint32_t flags)
{
	/*
	 * We have an irq that tells us when we have a new breadcrumb.
	 */

	if (class == 0 && flags == DRM_FENCE_TYPE_EXE)
		return 1;

	return 0;
}
