/*
 * SCIM Bridge
 *
 * Copyright (c) 2006 Ryo Dairiki <ryo-dairiki@users.sourceforge.net>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.*
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the
 * GNU Lesser General Public License for more details.*
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 */

#include <alloca.h>
#include <pthread.h>
#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <sys/types.h>

#include "scim-bridge-debug.h"
#include "scim-bridge-display.h"
#include "scim-bridge-output.h"
#include "scim-bridge-path.h"
#include "scim-bridge-string.h"

/* Data types */
struct _ScimBridgeDisplay
{
    char *name;
    int display_number;
    int screen_number;
};

/* Private variables */
static const char SOCKET_DIR[] = "/tmp";
static const char SOCKET_NAME[] = "scim-bridge-0_2_3.socket";

static const char LOCKFILE_DIR[] = "/tmp";
static const char LOCKFILE_NAME[] = "scim-bridge-0_2_3.lockfile";

static char *lockfile_path = NULL;
static char *socket_path = NULL;

static int display_number = -1;
static int screen_number = -1;

static char *display_name = NULL;

static pthread_once_t static_initializer = PTHREAD_ONCE_INIT;

/* Private function */
static void static_initialize ()
{
    display_name = getenv ("DISPLAY");
    if (display_name == NULL) {
        scim_bridge_perrorln ("Cannot get the environment variable \"\"DISPLAY");
        abort ();
    }

    if ((display_number < 0 || screen_number < 0) && display_name != NULL) {
        int has_colon = 0;
        int has_dot = 0;
        int error = 0;
        const char *c;
        for (c = display_name; !error && *c != '\0'; ++c) {
            if (*c == ':') {
                if (has_colon) {
                    break;
                } else {
                    has_colon = 1;
                    display_number = 0;
                    screen_number = 0;
                }
            } else if (has_colon) {
                switch (*c) {
                    case '0':
                    case '1':
                    case '2':
                    case '3':
                    case '4':
                    case '5':
                    case '6':
                    case '7':
                    case '8':
                    case '9':
                        if (!has_dot) display_number = display_number * 10 + atoi (c);
                        else screen_number = screen_number * 10 + atoi (c);
                        break;
                    case '.':
                        if (!has_dot) {
                            has_dot = 1;
                            break;
                        }
                        /* else no break */
                    default:
                        display_number = -1;
                        screen_number = -1;
                        error = 1;
                        break;
                }
            }
        }
    }

    if (display_number < 0 || screen_number < 0) scim_bridge_perrorln ("Invalid environmental variable: DISPLAY = %s", display_name == NULL ? "NULL":display_name);

    if (display_name != NULL) {
        const uid_t uid = getuid ();
        const size_t lockfile_path_length = snprintf (NULL, 0, "%s/%s-%d-%d-%d", LOCKFILE_DIR, LOCKFILE_NAME, display_number, screen_number, uid);

        lockfile_path = malloc (sizeof (char) * (lockfile_path_length + 1));
        sprintf (lockfile_path, "%s/%s-%d-%d-%d", LOCKFILE_DIR, LOCKFILE_NAME, display_number,screen_number, uid);

        const size_t socket_path_length = snprintf (NULL, 0, "%s/%s-%d-%d-%d", SOCKET_DIR, SOCKET_NAME, display_number, screen_number, uid);
        socket_path = malloc (sizeof (char) * (socket_path_length + 1));
        sprintf (socket_path, "%s/%s-%d-%d-%d", SOCKET_DIR, SOCKET_NAME, display_number, screen_number, uid);
    } else {
        lockfile_path = NULL;
        socket_path = NULL;
    }
}


/* Implementations */
const char *scim_bridge_path_get_lockfile ()
{
    pthread_once (&static_initializer, &static_initialize);
    return lockfile_path;
}


const char *scim_bridge_path_get_socket ()
{
    pthread_once (&static_initializer, &static_initialize);
    return socket_path;
}


ScimBridgeDisplay *scim_bridge_alloc_display ()
{
    ScimBridgeDisplay *display = malloc (sizeof (ScimBridgeDisplay));
    display->name = malloc (sizeof (char));
    display->name[0] = '\0';
    display->display_number = -1;
    display->screen_number = -1;
    return display;
}


void scim_bridge_free_display (ScimBridgeDisplay *display)
{
    free (display->name);
    free (display);
}


void scim_bridge_copy_display (ScimBridgeDisplay *dst, const ScimBridgeDisplay *src)
{
    const size_t display_name_str_length = strlen (src->name);    
    free (dst->name);
    dst->name = malloc (sizeof (char) * (display_name_str_length + 1));
    strcpy (dst->name, src->name);
    dst->display_number = src->display_number;
    dst->screen_number = src->screen_number;
}


void scim_bridge_display_fetch_current (ScimBridgeDisplay *display)
{
    pthread_once (&static_initializer, &static_initialize);
    
    const size_t display_name_str_length = strlen (display_name);

    free (display->name);
    display->name = malloc (sizeof (char) * (display_name_str_length + 1));
    strcpy (display->name, display_name);
    
    display->display_number = display_number;
    display->screen_number = screen_number;
}


const char *scim_bridge_display_get_name (const ScimBridgeDisplay *display)
{
    return display->name;
}


void scim_bridge_display_set_name (ScimBridgeDisplay *display, const char *display_name)
{
    free (display->name);
    display->name = malloc (sizeof (char) * (strlen (display_name) + 1));
    strcpy (display->name, display_name);
}


int scim_bridge_display_get_display_number (const ScimBridgeDisplay *display)
{
    return display->display_number;
}


void scim_bridge_display_set_display_number (ScimBridgeDisplay *display, int display_number)
{
    display->display_number = display_number;
}


int scim_bridge_display_get_screen_number (const ScimBridgeDisplay *display)
{
    return display->screen_number;
}


void scim_bridge_display_set_screen_number (ScimBridgeDisplay *display, int screen_number)
{
    display->screen_number = screen_number;
}
