/*
 * SCIM Bridge
 *
 * Copyright (c) 2006 Ryo Dairiki <ryo-dairiki@users.sourceforge.net>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.*
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the
 * GNU Lesser General Public License for more details.*
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 */

#include <unistd.h>

#include <sys/select.h>
#include <sys/types.h>

#include <glib.h>

#include <gtk/gtk.h>

#include "scim-bridge-client.h"
#include "scim-bridge-client-gtk.h"
#include "scim-bridge-client-imcontext-gtk.h"
#include "scim-bridge-client-protected.h"
#include "scim-bridge-exception.h"
#include "scim-bridge-imcontext.h"
#include "scim-bridge-output.h"

/* Private Variables */
static GIOChannel *messenger_iochannel = NULL;
static guint messenger_event_source = -1;

/* Private Functions */
static gboolean handle_message (GIOChannel *source, GIOCondition condition, gpointer data)
{
    const int socket_fd = scim_bridge_client_get_socket_fd ();

    fd_set read_set;
    FD_ZERO (&read_set);
    FD_SET (socket_fd, &read_set);

    struct timeval timeout;
    timeout.tv_sec = 0;
    timeout.tv_usec = 0;
    if (select (socket_fd + 1, &read_set, NULL, NULL, &timeout) > 0) {
        if (scim_bridge_client_read_and_dispatch ()) {
            scim_bridge_exception_output ();
            return FALSE;
        }
    }

    return TRUE;
}


/* Public Functions */
void scim_bridge_client_gtk_initialize ()
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT, 5, "scim_bridge_client_gtk_initialize");
    if (scim_bridge_client_is_active ()) return;

    if (!scim_bridge_client_initialize ()) {
        messenger_iochannel = g_io_channel_unix_new (scim_bridge_client_get_socket_fd ());
        messenger_event_source = g_io_add_watch (messenger_iochannel, G_IO_IN | G_IO_ERR | G_IO_HUP | G_IO_NVAL, &handle_message, NULL);
    } else {
        scim_bridge_exception_output ();
    }

    scim_bridge_client_imcontext_static_initialize ();
}


void scim_bridge_client_gtk_finalize ()
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT, 5, "scim_bridge_client_gtk_finalize");

    if (messenger_iochannel != NULL) {
        g_io_channel_close (messenger_iochannel);
        g_io_channel_unref (messenger_iochannel);
        messenger_iochannel = NULL;
        g_source_remove (messenger_event_source);
        messenger_event_source = -1;
    }

    scim_bridge_client_finalize ();

    scim_bridge_client_imcontext_static_finalize ();
}


void scim_bridge_client_connection_lost ()
{
    if (messenger_iochannel != NULL) {
        g_io_channel_close (messenger_iochannel);
        g_io_channel_unref (messenger_iochannel);
        messenger_iochannel = NULL;
        g_source_remove (messenger_event_source);
        messenger_event_source = -1;
    }
}
