/*
 * SCIM Bridge
 *
 * Copyright (c) 2006 Ryo Dairiki <ryo-dairiki@users.sourceforge.net>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.*
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the
 * GNU Lesser General Public License for more details.*
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 */

#include <assert.h>
#include <errno.h>
#include <pthread.h>
#include <semaphore.h>
#include <string.h>

#include <list>
#include <vector>

#define Uses_SCIM_BACKEND
#define Uses_SCIM_CONFIG
#define Uses_SCIM_CONFIG_MODULE
#define Uses_SCIM_CONFIG_PATH
#define Uses_SCIM_EVENT
#define Uses_SCIM_FRONTEND
#define Uses_SCIM_PANEL_CLIENT
#define Uses_SCIM_HOTKEY
#define Uses_SCIM_IMENGINE
#define Uses_SCIM_IMENGINE_MODULE

#include <scim.h>

#include "scim-bridge-agent.h"
#include "scim-bridge-agent-exception.h"
#include "scim-bridge-agent-event-server.h"
#include "scim-bridge-agent-imcontext.h"
#include "scim-bridge-agent-output.h"
#include "scim-bridge-agent-panel-listener.h"
#include "scim-bridge-agent-protected.h"
#include "scim-bridge-agent-signal-listener.h"
#include "scim-bridge-agent-socket-listener.h"
#include "scim-bridge-debug.h"
#include "scim-bridge-display.h"
#include "scim-bridge-output.h"
#include "scim-bridge-path.h"

#include "event/scim-bridge-agent-alloc-imcontext-event.h"
#include "event/scim-bridge-agent-client-closed-event.h"
#include "event/scim-bridge-agent-client-opened-event.h"
#include "event/scim-bridge-agent-cursor-location-changed-event.h"
#include "event/scim-bridge-agent-filter-key-event.h"
#include "event/scim-bridge-agent-focus-changed-event.h"
#include "event/scim-bridge-agent-free-imcontext-event.h"
#include "event/scim-bridge-agent-imcontext-allocated-event.h"
#include "event/scim-bridge-agent-imcontext-freed-event.h"
#include "event/scim-bridge-agent-key-handled-event.h"
#include "event/scim-bridge-agent-panel-pending-event.h"
#include "event/scim-bridge-agent-set-preedit-enabled-event.h"
#include "event/scim-bridge-agent-reset-imcontext-event.h"
#include "event/scim-bridge-agent-quit-event.h"

using std::list;
using std::vector;

using namespace scim;

/* Macros */
#ifndef SCIM_KEYBOARD_ICON_FILE
#define SCIM_KEYBOARD_ICON_FILE (SCIM_ICONDIR "/keyboard.png")
#endif

/* Class definition */
class ScimBridgeAgentImpl: public ScimBridgeAgent, public ScimBridgeAgentEventServer, public ScimBridgeAgentProtected
{

    public:

        ScimBridgeAgentImpl (bool noexit_enabled);
        ~ScimBridgeAgentImpl ();

        void launch () throw (ScimBridgeAgentException);

        void push_event (const ScimBridgeAgentEvent *event);

        /* Semi public funcitons */
        void reload_config ();
        void update_lookup_table_page_size (scim_bridge_imcontext_id_t imcontext_id, int page_size);
        void lookup_table_page_up (scim_bridge_imcontext_id_t imcontext_id);
        void lookup_table_page_down (scim_bridge_imcontext_id_t imcontext_id);
        void trigger_property (scim_bridge_imcontext_id_t imcontext_id, const String &property);
        void process_helper_event (scim_bridge_imcontext_id_t imcontext_id, const String &target_uuid, const String &helper_uuid, const Transaction &trans);
        void panel_move_preedit_caret (scim_bridge_imcontext_id_t imcontext_id, int caret_pos);
        void lookup_table_select_candidate (scim_bridge_imcontext_id_t imcontext_id, int cand_index);
        void panel_process_key_event (scim_bridge_imcontext_id_t imcontext_id, const KeyEvent &key_event);
        void panel_commit (scim_bridge_imcontext_id_t imcontext_id, const WideString &wstr);
        void panel_forward_key_event (scim_bridge_imcontext_id_t imcontext_id, const KeyEvent &key_event);
        void panel_request_help (scim_bridge_imcontext_id_t imcontext_id);
        void panel_request_factory_menu (scim_bridge_imcontext_id_t imcontext_id);
        void panel_change_factory (scim_bridge_imcontext_id_t imcontext_id, const String &uuid);

        /* The following methods are not public method */
        retval_t initialize ();

        bool is_quit_triggered ();

        void read_and_dispatch ();

    private:

        bool quit_triggered;

        bool noexit_enabled;

        pthread_t event_dispatcher_thread;
        sem_t event_dispatcher_sem;

        vector<ScimBridgeAgentEventClient*> event_clients;
        list<scim_bridge_agent_event_client_id_t> free_event_clients;
        size_t event_client_count;
        pthread_mutex_t event_clients_mutex;

        list<const ScimBridgeAgentEvent*> events;

        String scim_language;

        ConfigModule *scim_config_module;
        ConfigPointer scim_config;
        BackEndPointer scim_backend;

        KeyboardLayout scim_keyboard_layout;

        FrontEndHotkeyMatcher scim_frontend_hotkey_matcher;
        IMEngineHotkeyMatcher scim_imengine_hotkey_matcher;

        ScimBridgeAgentSignalListener *signal_listener;
        ScimBridgeAgentSocketListener *socket_listener;
        ScimBridgeAgentPanelListener *panel_listener;

        void add_event_client (ScimBridgeAgentEventClient *client);
        void remove_event_client (ScimBridgeAgentEventClient *client);

        const ScimBridgeAgentEvent *poll_event ();

        void finalize ();

        void slot_reload_config (const ConfigPointer &config);

        bool filter_hotkeys (ScimBridgeAgentIMContext *imcontext, const KeyEvent &key_event);

        /* Event Handlers */
        void panel_event_occured (const ScimBridgeAgentPanelPendingEvent *event);

        void client_opend (const ScimBridgeAgentClientOpenedEvent *event);
        void client_closed (const ScimBridgeAgentClientClosedEvent *event);

        void alloc_imcontext (const ScimBridgeAgentAllocIMContextEvent *event);
        void free_imcontext (const ScimBridgeAgentFreeIMContextEvent *event);

        void focus_changed (const ScimBridgeAgentFocusChangedEvent *event);
        void filter_key_event (const ScimBridgeAgentFilterKeyEvent *event);

        void cursor_location_changed (const ScimBridgeAgentCursorLocationChangedEvent* event);
        void set_preedit_enabled (const ScimBridgeAgentSetPreeditEnabledEvent *event);
        void reset_imcontext (const ScimBridgeAgentResetIMContextEvent *event);

};

/* Helper functions */
static void *run_event_dispatcher (void *arg)
{
    ScimBridgeAgentImpl *agent = static_cast<ScimBridgeAgentImpl*> (arg);

    while (!agent->is_quit_triggered ()) {
        agent->read_and_dispatch ();
    }

    return NULL;
}


static void sig_notify (int signal)
{
}


static bool check_socket_frontend ()
{
    SocketAddress address;
    SocketClient client;

    uint32 magic;

    address.set_address (scim_get_default_socket_frontend_address ());

    if (!client.connect (address))
        return false;

    if (!scim_socket_open_connection (magic, "ConnectionTester", "SocketFrontEnd", client, 1000)) {
        return false;
    } else {
        return true;
    }
}


/* Implimentations */
ScimBridgeAgent *ScimBridgeAgent::alloc (bool noexit_enabled)
{
    ScimBridgeAgentImpl *agent = new ScimBridgeAgentImpl (noexit_enabled);
    if (agent->initialize ()) {
        delete agent;
        return NULL;
    } else {
        return agent;
    }
}


ScimBridgeAgentImpl::ScimBridgeAgentImpl (bool noexit):
quit_triggered (false), noexit_enabled (noexit), event_client_count (0), 
signal_listener (NULL), panel_listener (NULL), socket_listener (NULL)
{
    pthread_mutex_init (&event_clients_mutex, NULL);
    sem_init (&event_dispatcher_sem, 0, 0);

    struct sigaction notify_action;
    notify_action.sa_handler = &sig_notify;
    notify_action.sa_flags = 0;
    sigaction (SIGNOTIFY, &notify_action, NULL);
}


ScimBridgeAgentImpl::~ScimBridgeAgentImpl ()
{
    pthread_mutex_lock (&event_clients_mutex);
    for (list<const ScimBridgeAgentEvent*>::iterator i = events.begin (); i !=  events.end (); ++i) {
        const ScimBridgeAgentEvent *event = *i;
        delete event;
    }
    pthread_mutex_unlock (&event_clients_mutex);

    pthread_mutex_destroy (&event_clients_mutex);
    sem_trywait (&event_dispatcher_sem);
    sem_destroy (&event_dispatcher_sem);

	if (signal_listener != NULL) {
		signal_listener->close_event_client ();
		delete signal_listener;
	}
	if (socket_listener != NULL) {
		socket_listener->close_event_client ();
		delete socket_listener;
	}
	if (panel_listener != NULL) {
		panel_listener->close_event_client ();
		delete panel_listener;
	}
}


bool ScimBridgeAgentImpl::is_quit_triggered ()
{
    return quit_triggered;
}


void ScimBridgeAgentImpl::read_and_dispatch ()
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 8, "read'n dispatch ()");
    const ScimBridgeAgentEvent *event = poll_event ();
    if (event != NULL) {
        switch (event->get_code ()) {
            case CLIENT_OPENED_EVENT:
                client_opend (static_cast<const ScimBridgeAgentClientOpenedEvent*> (event));
                break;
            case CLIENT_CLOSED_EVENT:
                client_closed (static_cast<const ScimBridgeAgentClientClosedEvent*> (event));
                break;
            case RESET_IMCONTEXT_EVENT:
                reset_imcontext (static_cast<const ScimBridgeAgentResetIMContextEvent*> (event));
                break;
            case FOCUS_CHANGED_EVENT:
                focus_changed (static_cast<const ScimBridgeAgentFocusChangedEvent*> (event));
                break;
            case ALLOC_IMCONTEXT_EVENT:
                alloc_imcontext (static_cast<const ScimBridgeAgentAllocIMContextEvent*> (event));
                break;
            case FREE_IMCONTEXT_EVENT:
                free_imcontext (static_cast<const ScimBridgeAgentFreeIMContextEvent*> (event));
                break;
            case CURSOR_LOCATION_CHANGED_EVENT:
                cursor_location_changed (static_cast<const ScimBridgeAgentCursorLocationChangedEvent*> (event));
                break;
            case FILTER_KEY_EVENT:
                filter_key_event (static_cast<const ScimBridgeAgentFilterKeyEvent*> (event));
                break;
            case SET_PREEDIT_ENABLED_EVENT:
                set_preedit_enabled (static_cast<const ScimBridgeAgentSetPreeditEnabledEvent*> (event));
                break;
            case PANEL_PENDING_EVENT:
                panel_event_occured (static_cast<const ScimBridgeAgentPanelPendingEvent*> (event));
                break;
            case QUIT_EVENT:
                finalize ();
                break;
            default:
                // do nothing
                scim_bridge_perrorln ("Unknown event, ignored.(Not yet implemented?)");
                break;
        }
        delete event;
    }
}


void ScimBridgeAgentImpl::push_event (const ScimBridgeAgentEvent *event)
{
    if (quit_triggered) return;

    pthread_mutex_lock (&event_clients_mutex);
    events.push_back (event);
    int sem_value;
    sem_post (&event_dispatcher_sem);
    sem_getvalue (&event_dispatcher_sem, &sem_value);
    if (sem_value == 0) sem_post (&event_dispatcher_sem);
    pthread_mutex_unlock (&event_clients_mutex);
}


const ScimBridgeAgentEvent *ScimBridgeAgentImpl::poll_event ()
{
    pthread_mutex_lock (&event_clients_mutex);
    while (!quit_triggered) {
        if (!events.empty ()) {
            const ScimBridgeAgentEvent *event = events.front ();
            events.pop_front ();
            pthread_mutex_unlock (&event_clients_mutex);
            return event;
        } else {
            int sem_value;
            do {
                sem_trywait (&event_dispatcher_sem);
                sem_getvalue (&event_dispatcher_sem, &sem_value);
            } while (sem_value > 0);
            pthread_mutex_unlock (&event_clients_mutex);
            sem_wait (&event_dispatcher_sem);
            pthread_mutex_lock (&event_clients_mutex);
        }
    }
    pthread_mutex_unlock (&event_clients_mutex);
    return NULL;
}


void ScimBridgeAgentImpl::add_event_client (ScimBridgeAgentEventClient *client)
{
    if (quit_triggered) return;
    pthread_mutex_lock (&event_clients_mutex);
    ++event_client_count;
    if (free_event_clients.empty ()) {
        const scim_bridge_agent_event_client_id_t client_id = event_clients.size ();
        event_clients.push_back (client);
        client->set_event_client_id (client_id);
        client->set_event_server (this);
        pthread_mutex_unlock (&event_clients_mutex);
    } else {
        const scim_bridge_agent_event_client_id_t client_id = free_event_clients.back ();
        free_event_clients.pop_back ();
        event_clients[client_id] = client;
        client->set_event_client_id (client_id);
        client->set_event_server (this);
        pthread_mutex_unlock (&event_clients_mutex);
    }
}


void ScimBridgeAgentImpl::remove_event_client (ScimBridgeAgentEventClient *client)
{
    if (quit_triggered || client->get_event_server () != this) return;

    pthread_mutex_lock (&event_clients_mutex);
    const scim_bridge_agent_event_client_id_t client_id = client->get_event_client_id ();
    event_clients[client_id] = NULL;
    client->close_event_client ();
    free_event_clients.push_back (client_id);
    --event_client_count;
    pthread_mutex_unlock (&event_clients_mutex);
}


void ScimBridgeAgentImpl::launch () throw (ScimBridgeAgentException)
{
	if (signal_listener->launch ()) {
		throw ScimBridgeAgentException (String ("Failed to launch the signal listener"));
	}

	if (panel_listener->launch ()) {
		throw ScimBridgeAgentException (String ("Failed to launch the panel listener"));
	}

	if (socket_listener->launch ()) {
		throw ScimBridgeAgentException (String ("Failed to launch the socket listener"));
	}

    if (pthread_create (&event_dispatcher_thread, NULL, &run_event_dispatcher, this)) throw ScimBridgeAgentException (String ("Cannot invoke the event dispatcher: ") + strerror (errno));
    if (pthread_join (event_dispatcher_thread, NULL)) throw ScimBridgeAgentException (String ("Cannot join the event dispatcher: ") + strerror (errno));
}


retval_t ScimBridgeAgentImpl::initialize ()
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_SCIM, 6, "Initializing scim...");

    // Get system language.
    scim_language = scim_get_locale_language (scim_get_current_locale ());
    ScimBridgeAgentIMContext::set_scim_language (scim_language);

    // Get modules list
    vector<String> imengine_vector;
    vector<String> config_vector;
    scim_get_imengine_module_list (imengine_vector);
    scim_get_config_module_list (config_vector);

    // Use socket if avaiable.
    bool use_socket = true;
    if (find (imengine_vector.begin (), imengine_vector.end (), "socket") == imengine_vector.end () ||
        find (config_vector.begin (), config_vector.end (), "socket") == config_vector.end ())
        use_socket = false;

    // Use simple config module as default if available.
    String config_module_name;
    if (config_vector.size () > 0) {
        config_module_name = scim_global_config_read (SCIM_GLOBAL_CONFIG_DEFAULT_CONFIG_MODULE, String ("simple"));
        if (find (config_vector.begin (), config_vector.end (), config_module_name) == config_vector.end ())
            config_module_name = config_vector [0];
    } else {
        config_module_name = "dummy";
    }

    // Use user defined immodules if available
    bool use_user_imegine = false;
    vector <String> user_imengine_vector;
    const char *imengine_list_str = getenv ("SCIM_BRIDGE_IMENGINE_MODULES");
    if (imengine_list_str != NULL) {
        vector <String> spec_imengine_vector;
        scim_split_string_list (spec_imengine_vector, imengine_list_str, ',');

        user_imengine_vector.clear ();

        for (size_t i = 0; i < spec_imengine_vector.size (); ++i) {
            if (find (imengine_vector.begin (), imengine_vector.end (), imengine_vector [i]) != imengine_vector.end ()) {
                user_imengine_vector.push_back (spec_imengine_vector [i]);
                use_user_imegine = true;
            }
        }
    }

    // If you try to use the socket feature manually,
    // then let you do it by yourself.
    if (config_module_name == "socket" ||
        find (user_imengine_vector.begin (), user_imengine_vector.end (), "socket") != user_imengine_vector.end ())
        use_socket = false;

    // Try to start a SCIM SocketFrontEnd process if necessary.
    if (scim_get_default_socket_frontend_address () != scim_get_default_socket_imengine_address () &&
        scim_get_default_socket_frontend_address () != scim_get_default_socket_config_address ())
        use_socket = false;

    if (use_socket) {
        // If no Socket FrontEnd is running, then launch one.
        // And set manual to false.
        if (!check_socket_frontend ()) {
            scim_bridge_perrorln ("Launching a SCIM daemon with Socket FrontEnd...");
            char *new_argv [] = { "--no-stay", 0 };
            scim_launch (true,
                config_module_name,
                (!user_imengine_vector.empty () ? scim_combine_string_list (user_imengine_vector, ',') : "all"),
                "socket",
                new_argv);
            use_user_imegine = false;
        }

        // If there is one Socket FrontEnd running and it's not manual mode,
        // then just use this Socket Frontend.
        if (!use_user_imegine) {
            for (int i = 0; i < 100; ++i) {
                if (check_socket_frontend ()) {
                    config_module_name = "socket";
                    user_imengine_vector.clear ();
                    user_imengine_vector.push_back ("socket");
                    break;
                }
                scim_usleep (100000);
            }
        }
    }

    if (config_module_name != "dummy") {
        //load config module
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_SCIM, 2, "Loading Config module...: %s", config_module_name.c_str ());
        scim_config_module = new ConfigModule (config_module_name);

        //create config instance
        if (scim_config_module != NULL && scim_config_module->valid ())
            scim_config = scim_config_module->create_config ();
    }

    if (scim_config.null ()) {
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT | SCIM_BRIDGE_DEBUG_SCIM, 2, "Config module cannot be loaded, using dummy Config.");

        if (scim_config_module) delete scim_config_module;
        scim_config_module = NULL;

        scim_config = new DummyConfig ();
        config_module_name = "dummy";
    }

    // create backend
    scim_backend = new CommonBackEnd (scim_config, user_imengine_vector.empty () ? user_imengine_vector : imengine_vector);

    if (scim_backend.null ()) {
        scim_bridge_psyslogln (SYSLOG_CRITICAL, "Cannot create BackEnd Object!");
        return RETVAL_FAILED;
    }

    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 4, "Initialize scim, done!");

    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 4, "Loading configurations...");

    slot_reload_config (scim_config);
    scim_config->signal_connect_reload (slot (this, &ScimBridgeAgentImpl::slot_reload_config));

    ScimBridgeDisplay *display = scim_bridge_alloc_display ();
    scim_bridge_display_fetch_current (display);

    panel_listener = ScimBridgeAgentPanelListener::alloc (scim_config->get_name (), display, this);
    scim_bridge_free_display (display);

    if (panel_listener == NULL) return RETVAL_FAILED;
    add_event_client (panel_listener);

    socket_listener = ScimBridgeAgentSocketListener::alloc ();
	if (socket_listener == NULL) return RETVAL_FAILED;
    add_event_client (socket_listener);

    signal_listener = ScimBridgeAgentSignalListener::alloc ();
	if (signal_listener == NULL) return RETVAL_FAILED;
    add_event_client (signal_listener);

    ScimBridgeAgentIMContext::static_initialize (panel_listener, scim_backend);

    ScimBridgeAgentIMContext::set_on_the_spot_enabled_globally (scim_config->read (String (SCIM_CONFIG_FRONTEND_ON_THE_SPOT), ScimBridgeAgentIMContext::is_on_the_spot_enabled_globally ()));
    ScimBridgeAgentIMContext::set_imengine_shared (scim_config->read (String (SCIM_CONFIG_FRONTEND_SHARED_INPUT_METHOD), ScimBridgeAgentIMContext::is_imengine_shared ()));

    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 4, "ScimBridgeAgent is now ready");
    return RETVAL_SUCCEEDED;
}


void ScimBridgeAgentImpl::finalize ()
{
    quit_triggered = true;
    pthread_mutex_lock (&event_clients_mutex);
    for (vector<ScimBridgeAgentEventClient*>::iterator i = event_clients.begin (); i != event_clients.end (); ++i) {
        ScimBridgeAgentEventClient *client = *i;
        if (client != NULL && client != socket_listener && client != panel_listener && client != signal_listener) {
            client->close_event_client ();
            delete client;
        }
    }

    signal_listener->close_event_client ();
    delete signal_listener;
    signal_listener = NULL;

    socket_listener->close_event_client ();
    delete socket_listener;
    socket_listener = NULL;

    panel_listener->close_event_client ();
    delete panel_listener;
    panel_listener = NULL;

    event_clients.clear ();
    pthread_mutex_unlock (&event_clients_mutex);

    ScimBridgeAgentIMContext::static_finalize ();

    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 4, "Finalize, done");
}


bool ScimBridgeAgentImpl::filter_hotkeys (ScimBridgeAgentIMContext *imcontext, const KeyEvent &key_event)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 8, "filter_hotkeys");

    scim_frontend_hotkey_matcher.push_key_event (key_event);

    const FrontEndHotkeyAction hotkey_action = scim_frontend_hotkey_matcher.get_match_result ();

    if (hotkey_action == SCIM_FRONTEND_HOTKEY_TRIGGER) {
        const bool new_status = !imcontext->is_enabled ();
        imcontext->set_enabled (new_status);
        
        if (ScimBridgeAgentIMContext::is_imengine_shared ()) {
            ScimBridgeAgentIMContext::set_enabled_by_default (new_status);
            scim_config->write (String (SCIM_CONFIG_FRONTEND_IM_OPENED_BY_DEFAULT), new_status);
        }
        return true;
    } else if (hotkey_action == SCIM_FRONTEND_HOTKEY_ON) {
        const bool new_status = true;
        imcontext->set_enabled (new_status);

        if (ScimBridgeAgentIMContext::is_imengine_shared ()) {
            ScimBridgeAgentIMContext::set_enabled_by_default (new_status);
            scim_config->write (String (SCIM_CONFIG_FRONTEND_IM_OPENED_BY_DEFAULT), new_status);
        }
        return true;
    } else if (hotkey_action == SCIM_FRONTEND_HOTKEY_OFF) {
        const bool new_status = false;
        imcontext->set_enabled (new_status);

        if (ScimBridgeAgentIMContext::is_imengine_shared ()) {
            ScimBridgeAgentIMContext::set_enabled_by_default (new_status);
            scim_config->write (String (SCIM_CONFIG_FRONTEND_IM_OPENED_BY_DEFAULT), new_status);
        }
        return true;
    } else if (hotkey_action == SCIM_FRONTEND_HOTKEY_NEXT_FACTORY) {
        imcontext->open_next_imengine ();
        return true;
    } else if (hotkey_action == SCIM_FRONTEND_HOTKEY_PREVIOUS_FACTORY) {
        imcontext->open_previous_imengine ();
        return true;
    } else if (hotkey_action == SCIM_FRONTEND_HOTKEY_SHOW_FACTORY_MENU) {
        vector<IMEngineFactoryPointer> factories;
        vector<PanelFactoryInfo> menu;

        scim_backend->get_factories_for_encoding (factories, "UTF-8");

        for (size_t i = 0; i < factories.size (); ++i) {
            menu.push_back (PanelFactoryInfo (factories [i]->get_uuid (), utf8_wcstombs (factories [i]->get_name ()), factories [i]->get_language (), factories [i]->get_icon_file ()));
        }
        if (!menu.empty ()) panel_listener->show_factory_menu (menu);
        return true;
    } else {
        scim_imengine_hotkey_matcher.push_key_event (key_event);
        if (scim_imengine_hotkey_matcher.is_matched ()) {
            const String factory_uuid = scim_imengine_hotkey_matcher.get_match_result ();
            imcontext->open_imengine_by_uuid (factory_uuid);
            return true;
        }
    }

    return false;
}


/* Event Handlers */
void ScimBridgeAgentImpl::panel_event_occured (const ScimBridgeAgentPanelPendingEvent *event)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 8, "panel_event_occured");
    panel_listener->read_and_dispatch ();
}


void ScimBridgeAgentImpl::client_opend (const ScimBridgeAgentClientOpenedEvent *event)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 8, "client_opened");
    ScimBridgeAgentClientListener *client_listener = ScimBridgeAgentClientListener::alloc (event->get_socket_fd ());
    add_event_client (client_listener);
}


void ScimBridgeAgentImpl::client_closed (const ScimBridgeAgentClientClosedEvent *event)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 8, "client_closed");

    ScimBridgeAgentClientListener *client = event->get_client_listener ();
    remove_event_client (client);
    delete client;

    if (event_client_count <= 3 && !noexit_enabled) finalize ();
}


void ScimBridgeAgentImpl::alloc_imcontext (const ScimBridgeAgentAllocIMContextEvent *event)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 8, "alloc_imcontext");
    ScimBridgeAgentClientListener *client_listener = static_cast<ScimBridgeAgentClientListener*> (event->get_client_listener ());
    ScimBridgeAgentIMContext *imcontext = ScimBridgeAgentIMContext::alloc (client_listener);

    client_listener->push_event (new ScimBridgeAgentIMContextAllocatedEvent (imcontext->get_id ()));
}


void ScimBridgeAgentImpl::free_imcontext (const ScimBridgeAgentFreeIMContextEvent *event)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 8, "free_imcontext");
    ScimBridgeAgentClientListener *client_listener = static_cast<ScimBridgeAgentClientListener*> (event->get_client_listener ());
    ScimBridgeAgentIMContext *imcontext = ScimBridgeAgentIMContext::find (event->get_imcontext_id ());
    if (imcontext == NULL || imcontext->get_client_listener () != event->get_client_listener ()) {
        scim_bridge_perrorln ("Invalid imcontext was requested");
        return;
    }

    panel_listener->prepare (imcontext->get_id ());
    imcontext->free ();
    panel_listener->send ();
    client_listener->push_event (new ScimBridgeAgentIMContextFreedEvent (imcontext->get_id ()));
}


void ScimBridgeAgentImpl::focus_changed (const ScimBridgeAgentFocusChangedEvent *event)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 8, "focus_changed");
    const scim_bridge_imcontext_id_t imcontext_id = event->get_imcontext_id ();

    ScimBridgeAgentIMContext *imcontext = ScimBridgeAgentIMContext::find (imcontext_id);
    if (imcontext == NULL || imcontext->get_client_listener () != event->get_client_listener ()) {
        scim_bridge_perrorln ("Invalid imcontext was requested");
        return;
    }

    panel_listener->prepare (imcontext_id);
    if (event->is_focus_in ()) {
        imcontext->focus_in ();
    } else {
        imcontext->focus_out ();
    }
    panel_listener->send ();
}


void ScimBridgeAgentImpl::filter_key_event (const ScimBridgeAgentFilterKeyEvent *event)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 8, "filter_key_event");
    const scim_bridge_imcontext_id_t imcontext_id = event->get_imcontext_id ();

    ScimBridgeAgentIMContext *imcontext = ScimBridgeAgentIMContext::find (imcontext_id);
    if (imcontext == NULL || imcontext->get_client_listener () != event->get_client_listener ()) {
        scim_bridge_perrorln ("Invalid imcontext was requested");
        return;
    }

    bool consumed = false;

    const KeyEvent &key_event = event->get_key_event ();
    panel_listener->prepare (imcontext_id);
    if (filter_hotkeys (imcontext, key_event)) {
        consumed = true;
    } else if (imcontext->filter_key_event (key_event)) {
        consumed = true;
    } else {
        consumed = false;
    }
    panel_listener->send ();
    imcontext->get_client_listener ()->push_event (new ScimBridgeAgentKeyHandledEvent (imcontext_id, consumed));
}


void ScimBridgeAgentImpl::cursor_location_changed (const ScimBridgeAgentCursorLocationChangedEvent *event)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 8, "cursor_location_changed");
    const scim_bridge_imcontext_id_t imcontext_id = event->get_imcontext_id ();

    ScimBridgeAgentIMContext *imcontext = ScimBridgeAgentIMContext::find (imcontext_id);
    if (imcontext == NULL || imcontext->get_client_listener () != event->get_client_listener ()) {
        scim_bridge_perrorln ("Invalid imcontext was requested");
        return;
    }

    panel_listener->prepare (imcontext_id);
    imcontext->cursor_location_changed (event->get_cursor_x (), event->get_cursor_y ());
    panel_listener->send ();
}


void ScimBridgeAgentImpl::set_preedit_enabled (const ScimBridgeAgentSetPreeditEnabledEvent *event)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 8, "set_preedit_enabled");
    const scim_bridge_imcontext_id_t imcontext_id = event->get_imcontext_id ();

    ScimBridgeAgentIMContext *imcontext = ScimBridgeAgentIMContext::find (imcontext_id);
    if (imcontext == NULL || imcontext->get_client_listener () != event->get_client_listener ()) {
        scim_bridge_perrorln ("Invalid imcontext was requested");
        return;
    }

    panel_listener->prepare (imcontext_id);
    imcontext->set_preedit_enabled (event->is_enabled ());
    panel_listener->send ();
}


void ScimBridgeAgentImpl::reset_imcontext (const ScimBridgeAgentResetIMContextEvent *event)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 8, "reset_imcontext");
    const scim_bridge_imcontext_id_t imcontext_id = event->get_imcontext_id ();

    ScimBridgeAgentIMContext *imcontext = ScimBridgeAgentIMContext::find (imcontext_id);
    if (imcontext == NULL || imcontext->get_client_listener () != event->get_client_listener ()) {
        scim_bridge_perrorln ("Invalid imcontext was requested");
        return;
    }

    panel_listener->prepare (imcontext_id);
    imcontext->reset ();
    panel_listener->send ();
}


void ScimBridgeAgentImpl::slot_reload_config (const ConfigPointer &config)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 8, "slot_reload_config");
    static bool first_run = true;

    scim_frontend_hotkey_matcher.load_hotkeys (scim_config);
    scim_imengine_hotkey_matcher.load_hotkeys (scim_config);

    if (!first_run) {
        ScimBridgeAgentIMContext::set_on_the_spot_enabled_globally (scim_config->read (String (SCIM_CONFIG_FRONTEND_ON_THE_SPOT), ScimBridgeAgentIMContext::is_on_the_spot_enabled_globally ()));
        ScimBridgeAgentIMContext::set_imengine_shared (scim_config->read (String (SCIM_CONFIG_FRONTEND_SHARED_INPUT_METHOD), ScimBridgeAgentIMContext::is_imengine_shared ()));
        ScimBridgeAgentIMContext::set_enabled_by_default (scim_config->read (String (SCIM_CONFIG_FRONTEND_IM_OPENED_BY_DEFAULT), ScimBridgeAgentIMContext::is_enabled_by_default ()));
    }

    // Get keyboard layout setting
    // Flush the global config first, in order to load the new configs from disk.
    scim_global_config_flush ();

    scim_keyboard_layout = scim_get_default_keyboard_layout ();
    ScimBridgeAgentClientListener::set_keyboard_layout (scim_keyboard_layout);

    first_run = false;
}


/* Semi public functions */
void ScimBridgeAgentImpl::reload_config ()
{
    scim_config->reload ();
}


void ScimBridgeAgentImpl::update_lookup_table_page_size (scim_bridge_imcontext_id_t imcontext_id, int page_size)
{
    ScimBridgeAgentIMContext *imcontext = ScimBridgeAgentIMContext::find (imcontext_id);
    if (imcontext != NULL) {
        panel_listener->prepare (imcontext_id);
        imcontext->update_lookup_table_page_size (page_size);
        panel_listener->send ();
    }
}


void ScimBridgeAgentImpl::lookup_table_page_up (scim_bridge_imcontext_id_t imcontext_id)
{
    ScimBridgeAgentIMContext *imcontext = ScimBridgeAgentIMContext::find (imcontext_id);
    if (imcontext != NULL) {
        panel_listener->prepare (imcontext_id);
        imcontext->lookup_table_page_up ();
        panel_listener->send ();
    }
}


void ScimBridgeAgentImpl::lookup_table_page_down (scim_bridge_imcontext_id_t imcontext_id)
{
    ScimBridgeAgentIMContext *imcontext = ScimBridgeAgentIMContext::find (imcontext_id);
    if (imcontext != NULL) {
        panel_listener->prepare (imcontext_id);
        imcontext->lookup_table_page_down ();
        panel_listener->send ();
    }
}


void ScimBridgeAgentImpl::lookup_table_select_candidate (scim_bridge_imcontext_id_t imcontext_id, int cand_index)
{
    ScimBridgeAgentIMContext *imcontext = ScimBridgeAgentIMContext::find (imcontext_id);
    if (imcontext != NULL) {
        panel_listener->prepare (imcontext_id);
        imcontext->lookup_table_select_candidate (cand_index);
        panel_listener->send ();
    }
}


void ScimBridgeAgentImpl::trigger_property (scim_bridge_imcontext_id_t imcontext_id, const String &property)
{
    ScimBridgeAgentIMContext *imcontext = ScimBridgeAgentIMContext::find (imcontext_id);
    if (imcontext != NULL) {
        panel_listener->prepare (imcontext_id);
        imcontext->trigger_property (property);
        panel_listener->send ();
    }
}


void ScimBridgeAgentImpl::process_helper_event (scim_bridge_imcontext_id_t imcontext_id, const String &target_uuid, const String &helper_uuid, const Transaction &trans)
{
    ScimBridgeAgentIMContext *imcontext = ScimBridgeAgentIMContext::find (imcontext_id);
    if (imcontext != NULL) {
        panel_listener->prepare (imcontext_id);
        imcontext->process_helper_event (target_uuid, helper_uuid, trans);
        panel_listener->send ();
    }
}


void ScimBridgeAgentImpl::panel_move_preedit_caret (scim_bridge_imcontext_id_t imcontext_id, int caret_pos)
{
    ScimBridgeAgentIMContext *imcontext = ScimBridgeAgentIMContext::find (imcontext_id);
    if (imcontext != NULL) {
        panel_listener->prepare (imcontext_id);
        imcontext->panel_move_preedit_caret (caret_pos);
        panel_listener->send ();
    }
}


void ScimBridgeAgentImpl::panel_process_key_event (scim_bridge_imcontext_id_t imcontext_id, const KeyEvent &key_event)
{
    ScimBridgeAgentIMContext *imcontext = ScimBridgeAgentIMContext::find (imcontext_id);
    if (imcontext != NULL) {
        panel_listener->prepare (imcontext_id);
        if (!filter_hotkeys (imcontext, key_event) && !imcontext->filter_key_event (key_event)) {
            imcontext->forward_key_event (key_event);
        }
        panel_listener->send ();
    }
}


void ScimBridgeAgentImpl::panel_commit (scim_bridge_imcontext_id_t imcontext_id, const WideString &wstr)
{
    ScimBridgeAgentIMContext *imcontext = ScimBridgeAgentIMContext::find (imcontext_id);
    if (imcontext != NULL) {
        panel_listener->prepare (imcontext_id);
        imcontext->panel_commit (wstr);
        panel_listener->send ();
    }
}


void ScimBridgeAgentImpl::panel_forward_key_event (scim_bridge_imcontext_id_t imcontext_id, const KeyEvent &key_event)
{
    ScimBridgeAgentIMContext *imcontext = ScimBridgeAgentIMContext::find (imcontext_id);
    if (imcontext != NULL) {
        panel_listener->prepare (imcontext_id);
        imcontext->forward_key_event (key_event);
        panel_listener->send ();
    }
}


void ScimBridgeAgentImpl::panel_request_help (scim_bridge_imcontext_id_t imcontext_id)
{
    ScimBridgeAgentIMContext *imcontext = ScimBridgeAgentIMContext::find (imcontext_id);
    if (imcontext != NULL) {
        panel_listener->prepare (imcontext_id);
        imcontext->panel_request_help ();
        panel_listener->send ();
    }
}


void ScimBridgeAgentImpl::panel_request_factory_menu (scim_bridge_imcontext_id_t imcontext_id)
{
    ScimBridgeAgentIMContext *imcontext = ScimBridgeAgentIMContext::find (imcontext_id);
    if (imcontext != NULL) {
        panel_listener->prepare (imcontext_id);

        vector<IMEngineFactoryPointer> factories;
        vector <PanelFactoryInfo> menu;

        scim_backend->get_factories_for_encoding (factories, "UTF-8");

        for (size_t i = 0; i < factories.size (); ++i) {
            menu.push_back (PanelFactoryInfo (factories [i]->get_uuid (), utf8_wcstombs (factories [i]->get_name ()), factories [i]->get_language (), factories [i]->get_icon_file ()));
        }
        if (!menu.empty ()) panel_listener->show_factory_menu (menu);

        panel_listener->send ();
    }
}


void ScimBridgeAgentImpl::panel_change_factory (scim_bridge_imcontext_id_t imcontext_id, const String &uuid)
{
    ScimBridgeAgentIMContext *imcontext = ScimBridgeAgentIMContext::find (imcontext_id);
    if (imcontext != NULL) {
        panel_listener->prepare (imcontext_id);
        imcontext->panel_change_factory (uuid);
        panel_listener->send ();
	
        if (ScimBridgeAgentIMContext::is_imengine_shared ()) {
            scim_config->write (String (SCIM_CONFIG_FRONTEND_IM_OPENED_BY_DEFAULT), ScimBridgeAgentIMContext::is_enabled_by_default ());
        }
    }
}
