/*
 * SCIM Bridge
 *
 * Copyright (c) 2006 Ryo Dairiki <ryo-dairiki@users.sourceforge.net>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.*
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the
 * GNU Lesser General Public License for more details.*
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 */

#include <assert.h>
#include <errno.h>
#include <pthread.h>

#include <sys/socket.h>
#include <sys/types.h>
#include <sys/un.h>

#include "scim-bridge-agent-event-server.h"
#include "scim-bridge-agent-output.h"
#include "scim-bridge-agent-socket-listener.h"
#include "scim-bridge-path.h"

#include "event/scim-bridge-agent-client-opened-event.h"

/* Class definition */
class ScimBridgeAgentSocketListenerImpl: public ScimBridgeAgentSocketListener
{

    public:

        ScimBridgeAgentSocketListenerImpl ();
        ~ScimBridgeAgentSocketListenerImpl ();

		retval_t launch ();

        /* The following methods are semi-public */
		retval_t initialize ();

        int get_socket_fd ();
        void socket_closed ();
        void socket_accepted (int new_fd);

    protected:

        void do_close_event_client ();

    private:

        int socket_fd;
		bool launched;

        pthread_t socket_listener_thread;

};

/* Helper functions */
void *run_socket_listener (void *arg)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 5, "run_socket_listener");

    ScimBridgeAgentSocketListenerImpl *socket_listener = static_cast<ScimBridgeAgentSocketListenerImpl*> (arg);
    while (socket_listener->get_socket_fd () != 0) {
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 3, "Waiting for connection...");

        int fd = -1;
        struct sockaddr_un empty_socket_addr;
        socklen_t emtpy_socket_size = sizeof (empty_socket_addr);
        fd = accept (socket_listener->get_socket_fd (), (sockaddr*)&empty_socket_addr, &emtpy_socket_size);
        if (fd < 0) {
            if (errno == EAGAIN || errno == EWOULDBLOCK || errno == EINTR || errno == 0) {
                continue;
            } else if (errno == EBADF) {
                socket_listener->socket_closed ();
                return NULL;
            } else {
                if (socket_listener->get_socket_fd () >= 0) {
                    scim_bridge_psyslogln (SYSLOG_ERROR, "An exception occured at run_socket_listener: %s", strerror (errno));
                    /* FIXME What if socket is broken by itself? */
                    socket_listener->socket_closed ();
                }
                return NULL;
            }
            usleep (50);
        } else {
            scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 5, "Connection accepted.");
            socket_listener->socket_accepted (fd);
        }
    }

    return NULL;
}


/* Implementations */
ScimBridgeAgentSocketListener::ScimBridgeAgentSocketListener ()
{
}


ScimBridgeAgentSocketListener::~ScimBridgeAgentSocketListener ()
{
}


ScimBridgeAgentSocketListener *ScimBridgeAgentSocketListener::alloc ()
{
    ScimBridgeAgentSocketListenerImpl *socket_listener = new ScimBridgeAgentSocketListenerImpl ();

	if (socket_listener->initialize ()) {
		delete socket_listener;
		return NULL;
	} else {
		return socket_listener;
	}
}


ScimBridgeAgentSocketListenerImpl::ScimBridgeAgentSocketListenerImpl (): launched (false)
{
}

ScimBridgeAgentSocketListenerImpl::~ScimBridgeAgentSocketListenerImpl ()
{
}


retval_t ScimBridgeAgentSocketListenerImpl::initialize ()
{
    const char *socket_path = scim_bridge_path_get_socket ();

    socket_fd = socket (PF_UNIX, SOCK_STREAM, 0);
    if (socket_fd < 0) {
        scim_bridge_psyslogln (SYSLOG_ERROR, "Cannot create an unix domain socket: %s", strerror (errno));
        return RETVAL_FAILED;
    }

    struct sockaddr_un socket_addr;
    memset (&socket_addr, 0, sizeof (struct sockaddr_un));
    socket_addr.sun_family = AF_UNIX;
    strcpy (socket_addr.sun_path, socket_path);

    unlink (socket_path);
    if (bind (socket_fd, (struct sockaddr*)&socket_addr, strlen (socket_addr.sun_path) + sizeof (socket_addr.sun_family)) != 0) {
        scim_bridge_psyslogln (SYSLOG_ERROR, "Cannot bind the socket: %s", strerror (errno));
        return RETVAL_FAILED;
    }

    if (listen (socket_fd, 64)) {
        scim_bridge_psyslogln (SYSLOG_CRITICAL, "Cannot start listening the socket: %s", strerror (errno));
        return RETVAL_FAILED;
    }

	return RETVAL_SUCCEEDED;
}

retval_t ScimBridgeAgentSocketListenerImpl::launch ()
{
	assert (!launched);
	launched = true;

    if (pthread_create (&socket_listener_thread, NULL, &run_socket_listener, this)) {
        scim_bridge_psyslog (SYSLOG_CRITICAL, "Cannot invoke socket listener thread: %s", strerror (errno));
        return RETVAL_FAILED;
	} else {
		return RETVAL_SUCCEEDED;
	}
}


int ScimBridgeAgentSocketListenerImpl::get_socket_fd ()
{
    return socket_fd;
}


void ScimBridgeAgentSocketListenerImpl::socket_accepted (int new_fd)
{
    get_event_server ()->push_event (new ScimBridgeAgentClientOpenedEvent (new_fd));
}


void ScimBridgeAgentSocketListenerImpl::socket_closed ()
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 9, "The socket closed...");
}


void ScimBridgeAgentSocketListenerImpl::do_close_event_client ()
{
    const int tmp_socket_fd = socket_fd;
    socket_fd = -1;
    shutdown (tmp_socket_fd, SHUT_RDWR);
    close (tmp_socket_fd);

    const char *socket_path = scim_bridge_path_get_socket ();
    unlink (socket_path);

    pthread_kill (socket_listener_thread, SIGNOTIFY);
    pthread_join (socket_listener_thread, NULL);
}
