/*
 * SCIM Bridge
 *
 * Copyright (c) 2006 Ryo Dairiki <ryo-dairiki@users.sourceforge.net>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.*
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the
 * GNU Lesser General Public License for more details.*
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 */

#include <assert.h>
#include <errno.h>
#include <pthread.h>
#include <signal.h>

#include <sys/types.h>

#include "scim-bridge-agent-event-server.h"
#include "scim-bridge-agent-output.h"
#include "scim-bridge-agent-signal-listener.h"

#include "event/scim-bridge-agent-quit-event.h"

/* Static variables */
static int signal_status = 0;

static pthread_t signal_listener_thread;

/* Class definition */
class ScimBridgeAgentSignalListenerImpl: public ScimBridgeAgentSignalListener
{

    public:

        ScimBridgeAgentSignalListenerImpl ();
        ~ScimBridgeAgentSignalListenerImpl ();

		retval_t initialize ();
		retval_t launch ();

        void signal_occured ();

    protected:

        void do_close_event_client ();

	private:

		bool launched;

};

/* Helper functions */
void *run_signal_listener (void *arg)
{
    ScimBridgeAgentSignalListenerImpl *signal_listener = static_cast<ScimBridgeAgentSignalListenerImpl*> (arg);

    while (signal_status == 0) {
        pause ();
    }

    if (signal_status == 1) signal_listener->signal_occured ();

    return NULL;
}


static void sig_quit (int sig)
{
    signal_status = 1;
    pthread_kill (signal_listener_thread, SIGNOTIFY);
}


/* Implementations */
ScimBridgeAgentSignalListener::ScimBridgeAgentSignalListener ()
{
}


ScimBridgeAgentSignalListener::~ScimBridgeAgentSignalListener ()
{
}


ScimBridgeAgentSignalListener *ScimBridgeAgentSignalListener::alloc ()
{
    return new ScimBridgeAgentSignalListenerImpl ();
}


ScimBridgeAgentSignalListenerImpl::ScimBridgeAgentSignalListenerImpl (): launched (false)
{
    struct sigaction quit_action;
    quit_action.sa_handler = &sig_quit;
    quit_action.sa_flags = 0;
    sigaction (SIGTERM, &quit_action, NULL);
    sigaction (SIGINT, &quit_action, NULL);
    sigaction (SIGQUIT, &quit_action, NULL);
    sigaction (SIGHUP, &quit_action, NULL);
}

ScimBridgeAgentSignalListenerImpl::~ScimBridgeAgentSignalListenerImpl ()
{
}


retval_t ScimBridgeAgentSignalListenerImpl::launch ()
{
	assert (!launched);
	launched = true;

    if (pthread_create (&signal_listener_thread, NULL, &run_signal_listener, this)) {
        scim_bridge_psyslog (SYSLOG_CRITICAL, "Cannot invoke signal listener thread: %s", strerror (errno));
        return RETVAL_FAILED;
    } else {
		return RETVAL_SUCCEEDED;
	}
}


void ScimBridgeAgentSignalListenerImpl::signal_occured ()
{
    get_event_server ()->push_event (new ScimBridgeAgentQuitEvent ());
}


void ScimBridgeAgentSignalListenerImpl::do_close_event_client ()
{
    struct sigaction quit_action;
    quit_action.sa_handler = SIG_DFL;
    quit_action.sa_flags = 0;
    sigaction (SIGTERM, &quit_action, NULL);
    sigaction (SIGINT, &quit_action, NULL);
    sigaction (SIGQUIT, &quit_action, NULL);
    sigaction (SIGHUP, &quit_action, NULL);

    signal_status = -1;

    pthread_kill (signal_listener_thread, SIGNOTIFY);
    pthread_join (signal_listener_thread, NULL);
}
