#include <alloca.h>
#include <assert.h>
#include <malloc.h>
#include <pthread.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include <syslog.h>

#include "scim-bridge-debug.h"
#include "scim-bridge-agent-output.h"

/* Static variables */
static pthread_once_t static_initializer = PTHREAD_ONCE_INIT;

static const char SCIM_BRIDGE_AGENT_PROCESS_NAME[] = "scim-bridge-agent";

/* Helper functions */
void init_syslog ()
{
    pthread_atfork (NULL, NULL, &init_syslog);

    setlogmask (LOG_UPTO (LOG_DEBUG));
    closelog ();
    openlog (SCIM_BRIDGE_AGENT_PROCESS_NAME, LOG_NDELAY, LOG_DAEMON);
}


static void vxfprintf (FILE *stream, const char *format, va_list ap)
{
    const size_t format_len = strlen (format);
    char *new_format = static_cast <char*> (alloca (sizeof (char) * (format_len + 1)));
    strcpy (new_format, format);
    new_format[format_len] = '\0';
    vfprintf (stream, new_format, ap);
}


static void vxfprintfln (FILE *stream, const char *format, va_list ap)
{
    const size_t format_len = strlen (format);
    char *new_format = static_cast <char*> (alloca (sizeof (char) * (format_len + 2)));
    strcpy (new_format, format);
    new_format[format_len] = '\n';
    new_format[format_len + 1] = '\0';
    vfprintf (stream, new_format, ap);
}


static void vxsyslog (scim_bridge_syslog_priority_t priority, const char *format, va_list ap)
{
    pthread_once (&static_initializer, &init_syslog);

    int syslog_priority;
    switch (priority) {
        case SYSLOG_CRITICAL:
            syslog_priority = LOG_CRIT;
            break;
        case SYSLOG_ERROR:
            syslog_priority = LOG_ERR;
            break;
        case SYSLOG_WARNING:
            syslog_priority = LOG_WARNING;
            break;
        case SYSLOG_INFO:
            syslog_priority = LOG_INFO;
            break;
        case SYSLOG_DEBUG:
        default:
            syslog_priority = LOG_DEBUG;
            break;
    }

    char *expanded_format;
    vasprintf (&expanded_format, format, ap);
    syslog (syslog_priority, "%s", expanded_format);
    free (expanded_format);
}


/* Implementations */
void scim_bridge_pdebugln (scim_bridge_debug_flag_t flag, scim_bridge_debug_level_t level, const char* format,...)
{
    if ((flag & scim_bridge_debug_get_flags ()) && ((10 - level) <= scim_bridge_debug_get_level ())) {
        va_list ap;
        va_start (ap, format);
        vxfprintfln (stdout, format, ap);
        va_end (ap);
    }
}


void scim_bridge_pdebug (scim_bridge_debug_flag_t flag, scim_bridge_debug_level_t level, const char* format,...)
{
    if ((flag & scim_bridge_debug_get_flags ()) && ((10 - level) <= scim_bridge_debug_get_level ())) {
        va_list ap;
        va_start (ap, format);
        vxfprintf (stdout, format, ap);
        va_end (ap);
    }
}


void scim_bridge_perrorln (const char* format,...)
{
    va_list ap;
    va_start (ap, format);
    vxfprintfln (stderr, format, ap);
    va_end (ap);
}


void scim_bridge_perror (const char* format,...)
{
    va_list ap;
    va_start (ap, format);
    vxfprintf (stderr, format, ap);
    va_end (ap);
}


void scim_bridge_println (const char* format,...)
{
    va_list ap;
    va_start (ap, format);
    vxfprintfln (stdout, format, ap);
    va_end (ap);
}


void scim_bridge_print (const char* format,...)
{
    va_list ap;
    va_start (ap, format);
    vxfprintf (stdout, format, ap);
    va_end (ap);
}


void scim_bridge_psyslogln (scim_bridge_syslog_priority_t priority, const char* format,...)
{
    va_list ap;
    va_start (ap, format);
    vxsyslog (priority, format, ap);
    va_end (ap);
    va_start (ap, format);
    vxfprintfln (stdout, format, ap);
    va_end (ap);
}


void scim_bridge_psyslog (scim_bridge_syslog_priority_t priority, const char* format,...)
{
    va_list ap;
    va_start (ap, format);
    vxsyslog (priority, format, ap);
    va_end (ap);
    va_start (ap, format);
    vxfprintf (stdout, format, ap);
    va_end (ap);
}
