/*
 * SCIM Bridge
 *
 * Copyright (c) 2006 Ryo Dairiki <ryo-dairiki@users.sourceforge.net>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.*
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the
 * GNU Lesser General Public License for more details.*
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 */

#include <errno.h>
#include <getopt.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <iostream>

#include "scim-bridge-agent.h"
#include "scim-bridge-debug.h"
#include "scim-bridge-agent-exception.h"
#include "scim-bridge-agent-output.h"
#include "scim-bridge-string.h"

using namespace std;

/* Static variables */
static scim_bridge_debug_level_t debug_level;
static scim_bridge_debug_flag_t debug_flags;

static ScimBridgeAgent *agent;

/* Implementations */
scim_bridge_debug_level_t scim_bridge_debug_get_level ()
{
    return debug_level;
}


scim_bridge_debug_flag_t scim_bridge_debug_get_flags ()
{
    return debug_flags;
}


int main (int argc, char *argv[])
{
    bool standalone_enabled = false;
    bool noexit_enabled = false;

    debug_level = 0;
    debug_flags = SCIM_BRIDGE_DEBUG_NONE;

    const struct option long_options[] = {
        {"help", 0, NULL, 'h'},
        {"verbose", 0, NULL, 'v'},
        {"quiet", 0, NULL, 'q'},
        {"debuglevel", 1, NULL, 'l'},
        {"debug", 1, NULL, 'b'},
        {"noexit", 0, NULL, 'n'},
        {"standalone", 0, NULL, 's'},
        {0, 0, NULL, 0}
    };

    char short_options[] = "vhqdls:b:";

    unsigned int tmp_uint;

    int option = 0;
    while (option != EOF) {
        option = getopt_long (argc, argv, short_options, long_options, NULL);
        switch (option) {
            case 'v':
                debug_level = 9;
                debug_flags = SCIM_BRIDGE_DEBUG_ALL;
                break;
            case 'q':
                debug_level = 0;
                debug_flags = SCIM_BRIDGE_DEBUG_NONE;
                break;
            case 'b':
                if (strcasecmp (optarg, "all") == 0) {
                    debug_flags = SCIM_BRIDGE_DEBUG_ALL;
                } else if (strcasecmp (optarg, "none") == 0) {
                    debug_flags = SCIM_BRIDGE_DEBUG_NONE;
                } else if (strcasecmp (optarg, "agent") == 0) {
                    debug_flags |= SCIM_BRIDGE_DEBUG_AGENT;
                } else if (strcasecmp (optarg, "client") == 0) {
                    debug_flags |= SCIM_BRIDGE_DEBUG_CLIENT;
                } else if (strcasecmp (optarg, "messenger") == 0) {
                    debug_flags |= SCIM_BRIDGE_DEBUG_MESSENGER;
                } else if (strcasecmp (optarg, "scim") == 0) {
                    debug_flags |= SCIM_BRIDGE_DEBUG_SCIM;
                } else if (strcasecmp (optarg, "imcontext") == 0) {
                    debug_flags |= SCIM_BRIDGE_DEBUG_IMCONTEXT;
                } else {
                    cout << "Unknwon debug flag: " << optarg << endl;
                }
                break;
            case 'l':
                if (scim_bridge_string_to_uint (&tmp_uint, optarg)) {
                    cerr << "Invalid debug level: " << optarg << endl;
                    exit (-1);
                } else {
                    debug_level = tmp_uint;
                }
                break;
            case 'n':
                noexit_enabled = true;
                break;
            case 's':
                standalone_enabled = true;
                break;
            case 'h':
                cout << "Usage: scim-bridge-agent [options]" << endl;
                cout << " Options" << endl << endl;
                cout << " -h, --help\tGive this help list" << endl;
                cout << " -v, --verbose\tVerbosely print out the debug message into standard output.This option equals to '--debuglevel=9 --debugflags=all'" << endl;
                cout << " -q, --quiet\tMake it print no debug message at all.This option equals to '--debuglevel=0 --debugflags=none'" << endl;
                cout << " -b, --debug\tSet which category of debug output do you need.Select one or more from 'all', 'none', 'agent', 'messenger', 'imcontext', and 'scim'." << endl;
                cout << " -l, --debuglevel\tSet how verbosely should it print debug output.'--debuglevel=0' equals to '--queit', and '--debuglevel=9' equals to '--verbose'" << endl;
                cout << " -s, --standalone\tGiven this, scim-brige-agent won't daemonize itself." << endl;
                cout << " -n, --noexit\tGiven this, scim-brige-agent won't exit when there is no client." << endl;
                exit (0);
                break;
            case ':':
            case '?':
                exit (-1);
            default:
                break;
        }
    }
    
    agent = ScimBridgeAgent::alloc (noexit_enabled);
    if (agent == NULL) {
        scim_bridge_perrorln ("Failed to allocate the agent, exitting...");
        exit (-1);
    }

    if (!standalone_enabled) {
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 5, "Daemonizing...");
        if (daemon (0, 0)) {
            scim_bridge_perrorln ("Cannot daemonize myself: %s", strerror (errno));
            exit (-1);
        }

        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 5, "Daemonize done");
    }
    
    try {
        agent->launch ();
    } catch (ScimBridgeAgentException ex) {
        scim_bridge_perrorln ("An exception has occured: %s", ex.get_message ().c_str ());
    }
    delete agent;
    agent = NULL;

    scim_bridge_psyslogln (SYSLOG_INFO, "Cleanup, done. Exitting...");
}
