/*
* @license Apache-2.0
*
* Copyright (c) 2021 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

// TypeScript Version: 2.0

/* tslint:disable:max-line-length */
/* tslint:disable:max-file-line-count */

import cdf = require( './../../../../../base/dists/chisquare/cdf' );
import ChiSquare = require( './../../../../../base/dists/chisquare/ctor' );
import entropy = require( './../../../../../base/dists/chisquare/entropy' );
import kurtosis = require( './../../../../../base/dists/chisquare/kurtosis' );
import logpdf = require( './../../../../../base/dists/chisquare/logpdf' );
import mean = require( './../../../../../base/dists/chisquare/mean' );
import median = require( './../../../../../base/dists/chisquare/median' );
import mgf = require( './../../../../../base/dists/chisquare/mgf' );
import mode = require( './../../../../../base/dists/chisquare/mode' );
import pdf = require( './../../../../../base/dists/chisquare/pdf' );
import quantile = require( './../../../../../base/dists/chisquare/quantile' );
import skewness = require( './../../../../../base/dists/chisquare/skewness' );
import stdev = require( './../../../../../base/dists/chisquare/stdev' );
import variance = require( './../../../../../base/dists/chisquare/variance' );

/**
* Interface describing the `chisquare` namespace.
*/
interface Namespace {
	/**
	* Chi-squared distribution cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param k - degrees of freedom
	* @returns evaluated CDF
	*
	* @example
	* var y = ns.cdf( 2.0, 8.0 );
	* // returns ~0.019
	*
	* y = ns.cdf( 0.0, 1.0 );
	* // returns 0.0
	*
	* var mycdf = ns.cdf.factory( 1.0 );
	*
	* var y = mycdf( 2.0 );
	* // returns ~0.843
	*
	* y = mycdf( 1.2 );
	* // returns ~0.727
	*/
	cdf: typeof cdf;

	/**
	* Chi-squared distribution.
	*/
	ChiSquare: typeof ChiSquare;

	/**
	* Returns the differential entropy of a chi-squared distribution (in nats).
	*
	* ## Notes
	*
	* -   If provided `k <= 0`, the function returns `NaN`.
	*
	* @param k - degrees of freedom
	* @returns entropy
	*
	* @example
	* var v = ns.entropy( 9.0 );
	* // returns ~2.786
	*
	* @example
	* var v = ns.entropy( 1.0 );
	* // returns ~0.784
	*
	* @example
	* var v = ns.entropy( -0.2 );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( NaN );
	* // returns NaN
	*/
	entropy: typeof entropy;

	/**
	* Returns the excess kurtosis of a chi-squared distribution.
	*
	* ## Notes
	*
	* -   If provided `k <= 0`, the function returns `NaN`.
	*
	* @param k - degrees of freedom
	* @returns excess kurtosis
	*
	* @example
	* var v = ns.kurtosis( 9.0 );
	* // returns ~1.333
	*
	* @example
	* var v = ns.kurtosis( 1.0 );
	* // returns 12.0
	*
	* @example
	* var v = ns.kurtosis( -0.2 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( NaN );
	* // returns NaN
	*/
	kurtosis: typeof kurtosis;

	/**
	* Chi-squared distribution natural logarithm of probability density function (logPDF).
	*
	* @param x - input value
	* @param k - degrees of freedom
	* @returns evaluated logPDF
	*
	* @example
	* var y = ns.logpdf( 2.0, 1.0 );
	* // returns ~-2.266
	*
	* var mylogpdf = ns.logpdf.factory( 6.0 );
	*
	* y = mylogpdf( 3.0 );
	* // returns ~-2.071
	*/
	logpdf: typeof logpdf;

	/**
	* Returns the expected value of a chi-squared distribution.
	*
	* ## Notes
	*
	* -   If provided `k < 0`, the function returns `NaN`.
	*
	* @param k - degrees of freedom
	* @returns expected value
	*
	* @example
	* var v = ns.mean( 9.0 );
	* // returns 9.0
	*
	* @example
	* var v = ns.mean( 1.0 );
	* // returns 1.0
	*
	* @example
	* var v = ns.mean( -0.2 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( NaN );
	* // returns NaN
	*/
	mean: typeof mean;

	/**
	* Returns the median of a chi-squared distribution.
	*
	* ## Notes
	*
	* -   If provided `k < 0`, the function returns `NaN`.
	*
	* @param k - degrees of freedom
	* @returns median
	*
	* @example
	* var v = ns.median( 9.0 );
	* // returns ~8.343
	*
	* @example
	* var v = ns.median( 2.0 );
	* // returns ~1.386
	*
	* @example
	* var k = ns.median( -0.2 );
	* // returns NaN
	*
	* @example
	* var k = ns.median( NaN );
	* // returns NaN
	*/
	median: typeof median;

	/**
	* Chi-squared distribution moment-generating function (MGF).
	*
	* @param t - input value
	* @param k - degrees of freedom
	* @returns evaluated MGF
	*
	* @example
	* var y = ns.mgf( 0.4, 2 );
	* // returns ~5.0
	*
	* y = ns.mgf( -1.0, 5.0 );
	* // returns ~0.0642
	*
	* y = ns.mgf( 0.0, 10.0 );
	* // returns 1.0
	*
	* var mymgf = ns.mgf.factory( 1.0 );
	*
	* var y = mymgf( 0.2 );
	* // returns ~1.291
	*
	* y = mymgf( 0.4 );
	* // returns ~2.236
	*/
	mgf: typeof mgf;

	/**
	* Returns the mode of a chi-squared distribution.
	*
	* ## Notes
	*
	* -   If provided `k < 0`, the function returns `NaN`.
	*
	* @param k - degrees of freedom
	* @returns mode
	*
	* @example
	* var v = ns.mode( 9.0 );
	* // returns 7.0
	*
	* @example
	* var v = ns.mode( 1.0 );
	* // returns 0.0
	*
	* @example
	* var v = ns.mode( -0.2 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( NaN );
	* // returns NaN
	*/
	mode: typeof mode;

	/**
	* Chi-squared distribution probability density function (PDF).
	*
	* @param x - input value
	* @param k - degrees of freedom
	* @returns evaluated PDF
	*
	* @example
	* var y = ns.pdf( 2.0, 1.0 );
	* // returns ~0.104
	*
	* var mypdf = ns.pdf.factory( 6.0 );
	*
	* y = mypdf( 3.0 );
	* // returns ~0.126
	*/
	pdf: typeof pdf;

	/**
	* Chi-squared distribution quantile function.
	*
	* @param p - input value
	* @param k - degrees of freedom
	* @returns evaluated quantile function
	*
	* @example
	* var y = ns.quantile( 0.8, 3.0 );
	* // returns ~4.642
	*
	* var myquantile = ns.quantile.factory( 2.0 );
	*
	* y = myquantile( 0.3 );
	* // returns ~0.713
	*
	* y = myquantile( 0.7 );
	* // returns ~2.408
	*/
	quantile: typeof quantile;

	/**
	* Returns the skewness of a chi-squared distribution.
	*
	* ## Notes
	*
	* -   If provided `k <= 0`, the function returns `NaN`.
	*
	* @param k - degrees of freedom
	* @returns skewness
	*
	* @example
	* var v = ns.skewness( 9.0 );
	* // returns ~0.943
	*
	* @example
	* var v = ns.skewness( 1.0 );
	* // returns ~2.828
	*
	* @example
	* var v = ns.skewness( -0.2 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( NaN );
	* // returns NaN
	*/
	skewness: typeof skewness;

	/**
	* Returns the standard deviation of a chi-squared distribution.
	*
	* ## Notes
	*
	* -   If provided `k < 0`, the function returns `NaN`.
	*
	* @param k - degrees of freedom
	* @returns standard deviation
	*
	* @example
	* var v = ns.stdev( 9.0 );
	* // returns ~4.243
	*
	* @example
	* var v = ns.stdev( 1.0 );
	* // returns ~1.414
	*
	* @example
	* var v = ns.stdev( -0.2 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( NaN );
	* // returns NaN
	*/
	stdev: typeof stdev;

	/**
	* Returns the variance of a chi-squared distribution.
	*
	* ## Notes
	*
	* -   If provided `k < 0`, the function returns `NaN`.
	*
	* @param k - degrees of freedom
	* @returns variance
	*
	* @example
	* var v = ns.variance( 9.0 );
	* // returns 18.0
	*
	* @example
	* var v = ns.variance( 1.0 );
	* // returns 2.0
	*
	* @example
	* var v = ns.variance( -0.2 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( NaN );
	* // returns NaN
	*/
	variance: typeof variance;
}

/**
* Chi-squared distribution.
*/
declare var ns: Namespace;


// EXPORTS //

export = ns;
