# Copyright (c) DataLab Platform Developers, BSD 3-Clause license, see LICENSE file.

"""
I/O (:mod:`sigima.io`)
-----------------------

This package provides input/output functionality for reading and writing
signals and images in various formats. It includes a registry for managing
the available formats and their associated read/write functions.

General purpose I/O functions
^^^^^^^^^^^^^^^^^^^^^^^^^^^^^

This package provides functions to read and write signals and images, allowing users
to easily handle different file formats without needing to know the specifics
of each format.

It includes the following main functions:

- :py:func:`read_signals`: Read a list of signals from a file.
- :py:func:`read_signal`: Read a single signal from a file.
- :py:func:`write_signal`: Write a single signal to a file.
- :py:func:`read_images`: Read a list of images from a file.
- :py:func:`read_image`: Read a single image from a file.
- :py:func:`write_image`: Write a single image to a file.

Supported formats
^^^^^^^^^^^^^^^^^

.. autodata:: SIGNAL_FORMAT_INFO

.. autodata:: IMAGE_FORMAT_INFO

Adding new formats
^^^^^^^^^^^^^^^^^^

To add new formats, you can create a new class that inherits from
:py:class:`sigima.io.image.base.ImageFormatBase` or
:py:class:`sigima.io.signal.base.SignalFormatBase` and implement the required methods.

.. note::

    Thanks to the plugin system, you can add new formats simply by defining a new class
    in a separate module, and it will be automatically discovered and registered, as
    long as it is imported in your application or library.

Example of a new image format plugin:

.. code-block:: python

    from sigima.io.image.base import ImageFormatBase
    from sigima.io.base import FormatInfo

    class MyImageFormat(ImageFormatBase):
        \"\"\"Object representing MyImageFormat image file type\"\"\"

        FORMAT_INFO = FormatInfo(
            name="MyImageFormat",
            extensions="*.myimg",
            readable=True,
            writeable=False,
        )

        @staticmethod
        def read_data(filename: str) -> np.ndarray:
            \"\"\"Read data and return it

            Args:
                filename (str): path to MyImageFormat file

            Returns:
                np.ndarray: image data
            \"\"\"
            # Implement reading logic here
            pass
"""

from __future__ import annotations

from sigima.io.common.objmeta import (
    read_annotations,
    read_metadata,
    read_roi,
    read_roi_grid,
    write_annotations,
    write_metadata,
    write_roi,
    write_roi_grid,
)
from sigima.io.convenience import (
    read_image,
    read_images,
    read_signal,
    read_signals,
    write_image,
    write_images,
    write_signal,
    write_signals,
)
from sigima.io.image.base import ImageIORegistry
from sigima.io.signal.base import SignalIORegistry

__all__ = [
    "IMAGE_FORMAT_INFO",
    "SIGNAL_FORMAT_INFO",
    "ImageIORegistry",
    "SignalIORegistry",
    "read_annotations",
    "read_image",
    "read_images",
    "read_metadata",
    "read_roi",
    "read_roi_grid",
    "read_signal",
    "read_signals",
    "write_annotations",
    "write_image",
    "write_images",
    "write_metadata",
    "write_roi",
    "write_roi_grid",
    "write_signal",
    "write_signals",
]

SIGNAL_FORMAT_INFO = SignalIORegistry.get_format_info(mode="rst")
IMAGE_FORMAT_INFO = ImageIORegistry.get_format_info(mode="rst")
