/*
 * $Id: ftp-ldap.c,v 1.7 2002/05/02 13:17:12 mt Exp $
 *
 * FTP Proxy LDAP interface handling
 *
 * Author(s): Jens-Gero Boehm <jens-gero.boehm@suse.de>
 *            Pieter Hollants <pieter.hollants@suse.de>
 *            Marius Tomaschewski <mt@suse.de>
 *            Volker Wiegand <volker.wiegand@suse.de>
 *
 * This file is part of the SuSE Proxy Suite
 *            See also  http://proxy-suite.suse.de/
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * A history log can be found at the end of this file.
 */

#ifndef lint
static char rcsid[] = "$Id: ftp-ldap.c,v 1.7 2002/05/02 13:17:12 mt Exp $";
#endif

#include <config.h>

#define _GNU_SOURCE		/* needed for crypt in Linux... */

#if defined(STDC_HEADERS)
#  include <stdio.h>
#  include <string.h>
#  include <stdlib.h>
#  include <stdarg.h>
#  include <errno.h>
#  include <ctype.h>
#endif

#if defined(HAVE_UNISTD_H)
#  include <unistd.h>
#endif

#if defined(TIME_WITH_SYS_TIME)
#  include <sys/time.h>
#  include <time.h>
#else
#  if defined(HAVE_SYS_TIME_H)
#    include <sys/time.h>
#  else
#    include <time.h>
#  endif
#endif

#include <netdb.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <arpa/inet.h>

#if defined(HAVE_LIBLDAP)
#  if defined(HAVE_LDAP_UMICH)
#    include <lber.h>
#  endif
#  include <ldap.h>
#  if !defined(LDAP_PORT)
#    define LDAP_PORT		389
#  endif
#endif

#include "com-config.h"
#include "com-debug.h"
#include "com-misc.h"
#include "com-socket.h"
#include "com-syslog.h"
#include "ftp-client.h"
#include "ftp-cmds.h"
#include "ftp-ldap.h"

/* ------------------------------------------------------------ */

#if defined(HAVE_LIBLDAP)
#  if defined(HAVE_LDAP_UMICH)
#    if defined __sun__
      /*
       * there is only a forward definition of the LDAP
       * connection handle struct in ldap.h on Solaris7,
       * so we have no access to ld_errno.
       */
#      define GET_LDERROR(ld,le)   le = LDAP_OTHER
#    else
#      if defined(LDAP_OPT_ERROR_NUMBER)
         /*
         ** OpenLDAP 2.x
         */
#        define GET_LDERROR(ld,le) \
                ldap_get_option(ld,LDAP_OPT_ERROR_NUMBER,&le)
#      else
         /*
         ** UmichLDAP or OpenLDAP 1.x
         */
#        define GET_LDERROR(ld,le) le = (ld)->ld_errno
#      endif
#    endif
#  else
      /*
      ** Netscape
      */
#    define GET_LDERROR(ld,le)     le = ldap_get_lderrno((ld), NULL, NULL)
#  endif
#endif


/* ------------------------------------------------------------ */

#if defined(HAVE_LIBLDAP)
static int   ldap_fetch(CONTEXT *ctx, char *srv, char *who, char *pwd);
static int   ldap_auth(LDAP *ld, LDAPMessage *e, char *who, char *pwd);
static char *ldap_attrib(LDAP *ld, LDAPMessage *e, char *attr, char *dflt);
static int   ldap_exists(LDAP *ld, LDAPMessage *e, char *attr,
                                                   char *vstr, int cs);
#endif

/* ------------------------------------------------------------ **
**
**	Function......:	ldap_setup_user
**
**	Parameters....:	ctx		Pointer to user context
**			pwd		Pointer to user auth pwd
**
**	Return........:	(none)
**
**	Purpose.......: Read the user specific parameters from
**			an LDAP Server if one is known. Else
**			read the values from the config file.
**
** ------------------------------------------------------------ */

int  ldap_setup_user(CONTEXT *ctx, char *pwd)
{
	char      *who, *p;
	u_int16_t l, u;

	if (ctx == NULL)		/* Basic sanity check	*/
		misc_die(FL, "ldap_setup_user: ?ctx?");

	/*
	** remember the proper name we should use...
	*/
	if(NULL != ctx->userauth) {
		who = ctx->userauth;
	} else {
		who = ctx->username;
	}

	/*
	** dont allow empty or invalid names...
	*/
	pwd=pwd;
	if( !(NULL != who && *who && !strchr(who, '*'))) {
		syslog_error("invalid user name or with asterisk");
		return -1;
	}

#if defined(HAVE_LIBLDAP)
	/*
	** If an LDAP server is configured, insist on using it
	*/
	if ((p = config_str(NULL, "LDAPServer", NULL)) != NULL) {
		int rc = ldap_fetch(ctx, p, who, pwd);
		/*
		** check if we have to read profile from config...
		*/
		syslog_write(T_DBG, "ldap_fetch returned %d", rc);
		if(0 != rc) return rc;
	}
#endif

	/*
	** Inform the auditor that we are using the config file
	*/
	syslog_write(U_INF, "reading data for '%s' from cfg-file", who);

	/*
	** Evaluate the destination FTP server address.
	** This is mandatory! Refuse to run if none given.
	*/
	if (ctx->magic_addr != INADDR_ANY)
		ctx->srv_addr = ctx->magic_addr;
	else if ((ctx->srv_addr = config_addr(who,
			"DestinationAddress",
			INADDR_BROADCAST)) == INADDR_BROADCAST) {
		errno = 0;
		syslog_error("can't eval DestAddr for %s",
		             ctx->cli_ctrl->peer);
		return -1;
	}
#if defined(COMPILE_DEBUG)
	debug(2, "DestAddr for %s: '%s'", ctx->cli_ctrl->peer,
	         socket_addr2str(ctx->srv_addr));
#endif

	/*
	** Evaluate the destination FTP server port
	*/
	if (ctx->magic_port != (u_int16_t)INPORT_ANY)
		ctx->srv_port = ctx->magic_port;
	else if ((ctx->srv_port = config_port(who,
		 "DestinationPort", IPPORT_FTP)) == 0)
	{
		errno = 0;
		syslog_error("can't eval DestPort for %s",
					ctx->cli_ctrl->peer);
		return -1;
	}
#if defined(COMPILE_DEBUG)
	debug(2, "DestPort for %s: %d",
			ctx->cli_ctrl->peer, (int) ctx->srv_port);
#endif

	/*
	** Evaluate the destination transfer mode
	*/
	p = config_str(who, "DestinationTransferMode", "client");
	if (strcasecmp(p, "active") == 0)
		ctx->srv_mode = MOD_ACT_FTP;
	else if (strcasecmp(p, "passive") == 0)
		ctx->srv_mode = MOD_PAS_FTP;
	else if (strcasecmp(p, "client") == 0)
		ctx->srv_mode = MOD_CLI_FTP;
	else {
		syslog_error("can't eval DestMode for %s",
					ctx->cli_ctrl->peer);
		return -1;
	}
#if defined(COMPILE_DEBUG)
	debug(2, "DestMode for %s: %s", ctx->cli_ctrl->peer, p);
#endif

	/*
	** Evaluate the port ranges
	*/
	l = config_port(who, "DestinationMinPort", INPORT_ANY);
	u = config_port(who, "DestinationMaxPort", INPORT_ANY);
	if (l > 0 && u > 0 && u >= l) {
		ctx->srv_lrng = l;
		ctx->srv_urng = u;
	} else {
		ctx->srv_lrng = INPORT_ANY;
		ctx->srv_urng = INPORT_ANY;
	}
#if defined(COMPILE_DEBUG)
	debug(2, "DestRange for %s: %d-%d", ctx->cli_ctrl->peer,
			(int) ctx->srv_lrng, (int) ctx->srv_urng);
#endif

	l = config_port(who, "ActiveMinDataPort", INPORT_ANY);
	u = config_port(who, "ActiveMaxDataPort", INPORT_ANY);
	if (l > 0 && u > 0 && u >= l) {
		ctx->act_lrng = l;
		ctx->act_urng = u;
	} else {
		/* do not try to bind a port < 1024 if running as UID != 0 */
		if(0 == getuid()) {
			ctx->act_lrng = (IPPORT_FTP - 1);
			ctx->act_urng = (IPPORT_FTP - 1);
		} else {
			ctx->act_lrng = INPORT_ANY;
			ctx->act_urng = INPORT_ANY;
		}
	}
#if defined(COMPILE_DEBUG)
	debug(2, "ActiveRange for %s: %d-%d", ctx->cli_ctrl->peer,
			(int) ctx->act_lrng, (int) ctx->act_urng);
#endif

	l = config_port(who, "PassiveMinDataPort", INPORT_ANY);
	u = config_port(who, "PassiveMaxDataPort", INPORT_ANY);
	if (l > 0 && u > 0 && u >= l) {
		ctx->pas_lrng = l;
		ctx->pas_urng = u;
	} else {
		ctx->pas_lrng = INPORT_ANY;
		ctx->pas_urng = INPORT_ANY;
	}
#if defined(COMPILE_DEBUG)
	debug(2, "PassiveRange for %s: %d-%d", ctx->cli_ctrl->peer,
			(int) ctx->pas_lrng, (int) ctx->pas_urng);
#endif

	/*
	** Setup other configuration options
	*/
	ctx->same_adr = config_bool(who, "SameAddress", 1);
	ctx->timeout  = config_int (who, "TimeOut",   900);
#if defined(COMPILE_DEBUG)
	debug(2, "SameAddress for %s: %s", ctx->cli_ctrl->peer,
				ctx->same_adr ? "yes" : "no");
	debug(2, "TimeOut for %s: %d", ctx->cli_ctrl->peer,
						ctx->timeout);
#endif

	/*
	** Adjust the allow/deny flags for the commands
	*/
	p = config_str(who, "ValidCommands", NULL);
	cmds_set_allow(p);

	return 0;
}


#if defined(HAVE_LIBLDAP)
/* ------------------------------------------------------------ **
**
**	Function......:	ldap_fetch
**
**	Parameters....:	ctx		Pointer to user context
**			srv		Pointer ldap server name
**			who		Pointer to user/auth name
**			pwd		Pointer to user/auth pwd
**
**	Return........:	0 on success
**
**	Purpose.......: Read the user specific parameters from
**			an LDAP Server.
**
** ------------------------------------------------------------ */

static int ldap_fetch(CONTEXT *ctx, char *srv, char *who, char *pwd)
{
	char str[MAX_PATH_SIZE], *ptr, *p;
	char *bind_dn, *bind_pw, *base, *idnt, *objc;
	int port, lderr, is_fmt;
	LDAP *ld;
	LDAPMessage *result, *e;
	u_int16_t l, u;

	/* Basic sanity */
	if (ctx == NULL || srv == NULL || who == NULL)
		misc_die(FL, "ldap_fetch: ?ctx? ?srv? ?who?");

	/*
	** prepare bind_dn: LDAPBindDN may contain
	** _one_ %s (printf-fmt); replace it with
	** the name passed via who parmeter...
	*/
	is_fmt  = 0;
	bind_dn = NULL;
	bind_pw = NULL;
	if( (ptr = config_str(NULL, "LDAPBindDN", NULL))) {
		size_t len;

		/*
		** search for exactly one %s; if any other
		** fmt's are present report an parse error...
		*/
		for(p=ptr; p && p[0] && (p = strchr(p, '%')); p++) {
			/* ignore %% pairs...? */
			if('%' == p[1]) continue;
			if(!is_fmt && ('s' == p[1] || 'S' == p[1])) {
				is_fmt = 1;
			} else {
				errno = 0;
				misc_die(FL, "ldap_fetch: ?BindDN-Fmt?");
				return -1;
			}
		}

		if( is_fmt) {
			/*
			** LDAPBindDN is a fmt-string
			*/
			len = strlen(who) + strlen(ptr);
			bind_dn = misc_alloc(FL, len);
#if defined(HAVE_SNPRINTF)
			snprintf(bind_dn, len, ptr, who);
#else
			sprintf(bind_dn, ptr, who);
#endif
		} else {
			bind_dn = misc_strdup(FL, ptr);
		}
	}

	/*
	** Determine LDAP server and port
	*/
	misc_strncpy(str, srv, sizeof(str));
	if ((p = strchr(str, ':')) != NULL) {
		*p++ = '\0';
		port = (int) socket_str2port(p, LDAP_PORT);
	} else {
		port = (int) LDAP_PORT;
	}

#if defined(COMPILE_DEBUG)
	debug(2, "LDAP server: %s:%d", str, port);
#endif

	/*
	** Ready to contact the LDAP server (as anonymous user)
	*/
	if ((ld = ldap_init(str, port)) == NULL) {
		errno = 0;
		syslog_error("can't reach LDAP server %s:%d", str, port);
		if(NULL != bind_dn)
			misc_free(FL, bind_dn);
		return -1;
	} else {
		syslog_write(T_DBG, "LDAP server %s:%d: init succeed",
		                    str, port);
	}

	if((NULL != bind_dn)) {
		if( !is_fmt) {
			/*
			** use LDAPBindPW for bind if not a fmt bind_dn
			*/
			bind_pw = config_str(NULL, "LDAPBindPW", NULL);
		} else	bind_pw = pwd;
	}

	/*
	** now, bind as anonymous or authenticated
	*/
	if(LDAP_SUCCESS != (lderr = ldap_simple_bind_s(ld, bind_dn, bind_pw)))
	{
		ldap_unbind(ld); /* needed? */
		errno = 0;
		if(NULL != bind_dn) {
			syslog_error("can't bind LDAP using dn '%s'", bind_dn);
			misc_free(FL, bind_dn);
		} else {
			syslog_error("can't bind LDAP anonymously");
		}
		return -1;
	} else {
		syslog_write(T_DBG,
		             "LDAP bind dn='%.256s' pw='%.256s': succeed",
		             NIL(bind_dn), NIL(bind_pw));
	}
	if(NULL != bind_dn) {
		misc_free(FL, bind_dn);
		bind_dn = 0;
	}

	/*
	** construct filter for the search
	**	(by LDAPIdentifier, maybe also ObjectClass)
	*/
	if ((objc = config_str(NULL, "LDAPObjectClass", "")) == NULL)
		misc_die(FL, "ldap_fetch: ?ObjectClass?");
	if ((idnt = config_str(NULL, "LDAPIdentifier", "")) == NULL)
		misc_die(FL, "ldap_fetch: ?Identifier?");
	if (*idnt == '\0')
		idnt = "CN";
	if (*objc == '\0') {
#if defined(HAVE_SNPRINTF)
		snprintf(str, sizeof(str), "(%.256s=%.256s)",
			  idnt, who);
#else
		sprintf(str, "(%.256s=%.256s)", idnt, who);
#endif
	} else {
#if defined(HAVE_SNPRINTF)
		snprintf(str, sizeof(str),
			  "(&(ObjectClass=%.256s)(%.256s=%.256s))",
			  objc,  idnt, who);
#else
		sprintf(str,
			  "(&(ObjectClass=%.256s)(%.256s=%.256s))",
			  objc,  idnt, who);
#endif
	}


	syslog_write(U_INF, "reading data for '%s' from LDAP", who);
	if ((base = config_str(NULL, "LDAPBaseDN", NULL)) != NULL)
	{
		syslog_write(T_DBG,
		             "LDAP search: base='%.256s' filter='%.256s'",
		             base, str);

		if(LDAP_SUCCESS != (lderr = ldap_search_s(ld, base,
	           LDAP_SCOPE_SUBTREE, str, NULL, 0, &result)))
		{
			errno = 0;
			syslog_error("can't read LDAP data for %s: %.512s",
			             ctx->cli_ctrl->peer,
			             ldap_err2string(lderr));
			ldap_unbind(ld);
			return -1;
		}

		/*
		** Check if we have a user data
		** (else return 'error' or 'empty')
		*/
		if ((e = ldap_first_entry(ld, result)) == NULL)
		{
			GET_LDERROR(ld,lderr);
			errno = 0;
			syslog_error("can't read LDAP data for %s: %.512s",
			             ctx->cli_ctrl->peer,
			             ldap_err2string(lderr));
			ldap_msgfree(result);
			ldap_unbind(ld);
			return -1;
		}
	} else {
		e = result = 0;
	}

	/*
	** Preform auth on userauth if type is ldap
	*/
	if( (ptr = config_str(NULL, "UserAuthType", NULL))
	    && !strcasecmp(ptr, "ldap"))
	{
		LDAPMessage *res=0, *a=e;
		/*
		** if LDAPAuthDN set, do auth on a different base...
		*/
		if(NULL != (ptr = config_str(NULL, "LDAPAuthDN", NULL)))
		{
			syslog_write(T_DBG,
			       "LDAP auth: base='%.256s' filter='%.256s'",
			       ptr, str);

			if(LDAP_SUCCESS != (lderr = ldap_search_s(ld,
			   ptr, LDAP_SCOPE_SUBTREE, str, NULL, 0, &res)))
			{
				syslog_error("can't read LDAP auth-data"
					     " for %s: %.512s",
					     ctx->cli_ctrl->peer,
					     ldap_err2string(lderr));
				if(result) ldap_msgfree(result);
				ldap_unbind(ld);
				return -1;
			}

			if(NULL == (a = ldap_first_entry(ld, res))) {
				GET_LDERROR(ld,lderr);
				errno = 0;
				syslog_error("can't read LDAP auth-data"
					     " for %s: %.512s",
					     ctx->cli_ctrl->peer,
					     ldap_err2string(lderr));
				if(result) ldap_msgfree(result);
				ldap_msgfree(res);
				ldap_unbind(ld);
				return -1;
			}
		} else if(NULL == base || NULL == e) {
			ldap_unbind(ld);
			misc_die(FL, "ldap_fetch: ?LDAPBaseDN?");
		}

		/*
		** OK, let's check the user auth now
		*/
		if(0 != ldap_auth(ld, a, who, pwd)) {
			errno = 0;
			syslog_write(U_ERR,
			             "LDAP user auth failed for %s from %s",
			             who, ctx->cli_ctrl->peer);
			if(result) ldap_msgfree(result);
			if(res)    ldap_msgfree(res);
			ldap_unbind(ld);
			return -1;
		}
		if(res) ldap_msgfree(res);
	}

	/*
	** read proxy user profile data if we have a base
	*/
	if(NULL == base) {
		ldap_unbind(ld);
		return 0;
	}

	/*
	** Evaluate the destination FTP server address.
	** This is mandatory! Refuse to run if none given.
	*/
	p = ldap_attrib(ld, e, "DestinationAddress", "255.255.255.255");
	if (ctx->magic_addr != INADDR_ANY) {
		ctx->srv_addr = ctx->magic_addr;
	} else
	if ((ctx->srv_addr = socket_str2addr(p,
			INADDR_BROADCAST)) == INADDR_BROADCAST) {
		errno = 0;
		syslog_error("can't eval DestAddr for %s",
					ctx->cli_ctrl->peer);
		ldap_msgfree(result);
		ldap_unbind(ld);
		return -1;
	}
#if defined(COMPILE_DEBUG)
	debug(2, "DestAddr for %s: '%s'", ctx->cli_ctrl->peer,
				socket_addr2str(ctx->srv_addr));
#endif

	/*
	** Evaluate the destination FTP server port
	*/
	p = ldap_attrib(ld, e, "DestinationPort", "ftp");
	if (ctx->magic_port != (u_int16_t)INPORT_ANY) {
		ctx->srv_port = ctx->magic_port;
	} else
	if ((ctx->srv_port = socket_str2port(p,
			IPPORT_FTP)) == (u_int16_t)INPORT_ANY) {
		errno = 0;
		syslog_error("can't eval DestPort for %s",
					ctx->cli_ctrl->peer);
		ldap_msgfree(result);
		ldap_unbind(ld);
		return -1;
	}
#if defined(COMPILE_DEBUG)
	debug(2, "DestPort for %s: %d",
			ctx->cli_ctrl->peer, (int) ctx->srv_port);
#endif

	/*
	** Evaluate the destination transfer mode
	*/
	p = ldap_attrib(ld, e, "DestinationTransferMode", "client");
	if (strcasecmp(p, "active") == 0)
		ctx->srv_mode = MOD_ACT_FTP;
	else if (strcasecmp(p, "passive") == 0)
		ctx->srv_mode = MOD_PAS_FTP;
	else if (strcasecmp(p, "client") == 0)
		ctx->srv_mode = MOD_CLI_FTP;
	else {
		errno = 0;
		syslog_error("can't eval DestMode for %s",
					ctx->cli_ctrl->peer);
		ldap_msgfree(result);
		ldap_unbind(ld);
		return -1;
	}
#if defined(COMPILE_DEBUG)
	debug(2, "DestMode for %s: %s", ctx->cli_ctrl->peer, p);
#endif

	/*
	** Evaluate the port ranges
	*/
	p = ldap_attrib(ld, e, "DestinationMinPort", "0");
	l = socket_str2port(p, INPORT_ANY);
	p = ldap_attrib(ld, e, "DestinationMaxPort", "0");
	u = socket_str2port(p, INPORT_ANY);
	if (l > 0 && u > 0 && u >= l) {
		ctx->srv_lrng = l;
		ctx->srv_urng = u;
	} else {
		ctx->srv_lrng = INPORT_ANY;
		ctx->srv_urng = INPORT_ANY;
	}
#if defined(COMPILE_DEBUG)
	debug(2, "DestRange for %s: %d-%d", ctx->cli_ctrl->peer,
			(int) ctx->srv_lrng, (int) ctx->srv_urng);
#endif

	p = ldap_attrib(ld, e, "ActiveMinDataPort", "0");
	l = socket_str2port(p, INPORT_ANY);
	p = ldap_attrib(ld, e, "ActiveMaxDataPort", "0");
	u = socket_str2port(p, INPORT_ANY);
	if (l > 0 && u > 0 && u >= l) {
		ctx->act_lrng = l;
		ctx->act_urng = u;
	} else {
		/* do not try to bind a port < 1024 if running as UID != 0 */
		if(0 == getuid()) {
			ctx->act_lrng = (IPPORT_FTP - 1);
			ctx->act_urng = (IPPORT_FTP - 1);
		} else {
			ctx->act_lrng = INPORT_ANY;
			ctx->act_urng = INPORT_ANY;
		}
	}
#if defined(COMPILE_DEBUG)
	debug(2, "ActiveRange for %s: %d-%d", ctx->cli_ctrl->peer,
			(int) ctx->act_lrng, (int) ctx->act_urng);
#endif

	p = ldap_attrib(ld, e, "PassiveMinDataPort", "0");
	l = socket_str2port(p, INPORT_ANY);
	p = ldap_attrib(ld, e, "PassiveMaxDataPort", "0");
	u = socket_str2port(p, INPORT_ANY);
	if (l > 0 && u > 0 && u >= l) {
		ctx->pas_lrng = l;
		ctx->pas_urng = u;
	} else {
		ctx->pas_lrng = INPORT_ANY;
		ctx->pas_urng = INPORT_ANY;
	}
#if defined(COMPILE_DEBUG)
	debug(2, "PassiveRange for %s: %d-%d", ctx->cli_ctrl->peer,
			(int) ctx->pas_lrng, (int) ctx->pas_urng);
#endif

	/*
	** Setup other configuration options
	*/
	p = ldap_attrib(ld, e, "SameAddress", "yes");
	if (strcasecmp(p, "y") == 0)
		ctx->same_adr = 1;
	else if (strcasecmp(p, "on") == 0)
		ctx->same_adr = 1;
	else if (strcasecmp(p, "yes") == 0)
		ctx->same_adr = 1;
	else if (strcasecmp(p, "true") == 0)
		ctx->same_adr = 1;
	else if (*p >= '0' && *p <= '9')
		ctx->same_adr = (atoi(p) != 0);
	else
		ctx->same_adr = 0;
	p = ldap_attrib(ld, e, "TimeOut", "900");
	if (*p >= '0' && *p <= '9')
		ctx->timeout = atoi(p);
	else
		ctx->timeout = 900;
#if defined(COMPILE_DEBUG)
	debug(2, "SameAddress for %s: %s", ctx->cli_ctrl->peer,
				ctx->same_adr ? "yes" : "no");
	debug(2, "TimeOut for %s: %d", ctx->cli_ctrl->peer,
						ctx->timeout);
#endif

	/*
	** Adjust the allow/deny flags for the commands
	*/
	p = ldap_attrib(ld, e, "ValidCommands", NULL);
	cmds_set_allow(p);

	/*
	** All relevant attributes have been evaluated
	*/
	ldap_msgfree(result);
	ldap_unbind(ld);

	return 0;
}


/* ------------------------------------------------------------ **
**
**	Function......:	ldap_auth
**
**	Parameters....:	ld		Pointer to LDAP struct
**			e		Pointer to result buffer
**			who		Pointer to user name
**			pwd		Pointer to user pwd
**
**	Return........:	0 on success
**
**	Purpose.......: Preform LDAP userauth
**
** ------------------------------------------------------------ */

static int   ldap_auth(LDAP *ld, LDAPMessage *e, char *who, char *pwd)
{
	char str[MAX_PATH_SIZE];
	char *v, *p, *q;
	size_t len;

	if (ld == NULL || e == NULL)
		misc_die(FL, "ldap_checkauth: ?ld? ?e?");

	/*
	** check "user enabled" flag if present
	*/
	if( (p = config_str(NULL, "LDAPAuthOKFlag", NULL))) {
		misc_strncpy(str, p, sizeof(str));
		if( (v = strchr(str, '=')))
			*v++ = '\0';
		else	v = 0;

		if(v && strlen(v) && strlen(str)) {
			if(0 != ldap_exists(ld, e, str, v, 0)) {
				syslog_write(U_WRN,
				"access denied for %s", NIL(who));
				return -1;
			} else {
				syslog_write(T_DBG,
				"LDAP auth ok-check: '%.256s'='%.256s' passed",
				NIL(str), NIL(v));
			}
		} else {
			errno = 0;
			misc_die(FL, "ldap_auth: ?LDAPAuthOKFlag?");
		}
	} else {
		syslog_write(T_DBG, "LDAP auth ok-check skipped");
	}

	/*
	** check user pass match
	*/
	if( (p = config_str(NULL, "LDAPAuthPWAttr", "")) && strlen(p)) {
		if(NULL == pwd)
			pwd = "";

		v   = config_str(NULL, "LDAPAuthPWType", "plain");
		q   = ldap_attrib(ld, e, p, "");
		p   = 0;
		len = 0;

		if( !strncasecmp(v, "plain", sizeof("plain")-1)) {
			/*
			** plain passwd - no prefix
			*/
			len = sizeof("plain")-1;
			p   = pwd;
#if defined(HAVE_CRYPT)
		} else
		if( !strncasecmp(v, "crypt", sizeof("plain")-1)) {
			/*
			** crypt - no prefix
			*/
			len = sizeof("plain")-1;
			p   = crypt(pwd, q);
		} else
		if( !strncasecmp(v, "{crypt}", sizeof("{crypt}")-1)) {
			/*
			** crypt - {crypt} prefix
			*/
			len = sizeof("{crypt}")-1;
			if(strncasecmp(q, "{crypt}", len)) {
				syslog_error("ldap user auth - prefix missed");
				return -1;
			}
			q+=len;
			p = crypt(pwd, q);
#endif
		} else {
			errno = 0;
			misc_die(FL, "ldap_auth: ?LDAPAuthPWType?");
		}

		/*
		** check if we have different minimal length
		** it is coded in latest "char", i.e. plain9
		*/
		if(0 < len && strlen(v) == len+1 &&
		   '0' <= v[len] && '9' >= v[len])
		{
			len = (size_t)v[len] - '0';
		} else	len = PASS_MIN_LEN;

		syslog_write(T_DBG, "LDAP auth pw-type[%d]='%.256s'", len, v);
#if defined(COMPILE_DEBUG)
		debug(3,            "LDAP auth pw-check: '%.256s' ?= '%.256s'",
		                    NIL(q), NIL(p));
#endif

		/*
		** check (lenght) and compare passwds; the user
		** account is locked if LDAP-PWD is "*" or "!"
		*/
		if(p && strlen(p)>=len && strlen(q) == strlen(p) &&
		   !(1==strlen(q) && ('*' == q[0] || '!' == q[0])))
		{
			if(0 == strcmp(q, p)) {
				syslog_write(T_DBG,
				             "LDAP auth pw-check succeed");
				return 0;
			}
		}
		syslog_write(T_DBG, "LDAP auth pw-check failed");
		return -1;
	} else {
		syslog_write(T_DBG, "LDAP auth pw-check skipped");
	}

	/*
	** OK, all configured LDAPAuth stuff succeed...
	*/
	return 0;
}


/* ------------------------------------------------------------ **
**
**	Function......:	ldap_attrib
**
**	Parameters....:	ld		Pointer to LDAP struct
**			e		Pointer to result buffer
**			attr		Name of desired option
**			dflt		Default value
**
**	Return........:	Value for attr (or dflt if not found)
**
**	Purpose.......: Search the LDAP result message for the
**			desired attribute value and return it.
**			NEVER return a NULL pointer except if
**			the dflt was taken and is NULL itself.
**
** ------------------------------------------------------------ */

static char *ldap_attrib(LDAP *ld, LDAPMessage *e, char *attr, char *dflt)
{
	static char str[MAX_PATH_SIZE];
	char **vals;

	if (ld == NULL || e == NULL || attr == NULL)
		misc_die(FL, "ldap_attrib: ?ld? ?e? ?attr?");

	/*
	** See if this attribute has values available
	*/
	if ((vals = ldap_get_values(ld, e, attr)) == NULL) {
#if defined(COMPILE_DEBUG)
		debug(3, "LDAP result: '%.256s' - '%.1024s'",
						attr, NIL(dflt));
#endif
		return dflt;
	}

	/*
	** Save value (use the first one) and free memory
	*/
	misc_strncpy(str, vals[0], sizeof(str));
	ldap_value_free(vals);

#if defined(COMPILE_DEBUG)
	debug(3, "LDAP result: '%.256s' = '%.1024s'", attr, str);
#endif
	return str;
}


/* ------------------------------------------------------------ **
**
**	Function......:	ldap_exists
**
**	Parameters....:	ld		Pointer to LDAP struct
**			e		Pointer to result buffer
**			attr		Name of desired option
**			vstr		Value of desired option
**			cs		1 for case sensitive check
**
**	Return........:	0 if value found
**
**	Purpose.......: Search the LDAP result message for the
**			desired (multivalue) attribute and check
**			if it contains a value string.
**
** ------------------------------------------------------------ */

static int  ldap_exists(LDAP *ld, LDAPMessage *e, char *attr,
                                                  char *vstr, int cs)
{
	char **vals;
	int    count, at;

	if (ld == NULL || e == NULL || attr == NULL || vstr == NULL)
		misc_die(FL, "ldap_exists: ?ld? ?e? ?attr? ?vstr?");

	/*
	** See if this attribute has values available
	*/
	if ((vals = ldap_get_values(ld, e, attr)) == NULL) {
#if defined(COMPILE_DEBUG)
		debug(3, "LDAP result: no values for '%.256s'", attr);
#endif
		return -1;
	}

	count = ldap_count_values(vals);
	if(cs) {
		for(at=0; at < count; at++) {
#if defined(COMPILE_DEBUG)
		debug(3, "LDAP result: checking[%d:%d] '%.256s'='%.1024s'",
		                              count-1, at, attr, vals[at]);
#endif
			if(misc_strequ(vals[at], vstr)) {
				ldap_value_free(vals);
				return 0;
			}
		}
	} else {
		for(at=0; at < count; at++) {
#if defined(COMPILE_DEBUG)
		debug(3, "LDAP result: checking[%d:%d] '%.256s'='%.1024s'",
		                              count-1, at, attr, vals[at]);
#endif
			if(misc_strcaseequ(vals[at], vstr)) {
				ldap_value_free(vals);
				return 0;
			}
		}
	}
	ldap_value_free(vals);

#if defined(COMPILE_DEBUG)
		debug(3, "LDAP result: '%.256s'='%.1024s' not found",
		                       attr, vstr);
#endif

	return 1;
}
#endif


/* ------------------------------------------------------------
 * $Log: ftp-ldap.c,v $
 * Revision 1.7  2002/05/02 13:17:12  mt
 * implemented simple (ldap based) user auth
 *
 * Revision 1.6  2002/01/14 19:26:38  mt
 * implemented bind_dn and pwd authorized ldap_simple_bind
 * fixed ld_errno fetching macro to work with openldap 2.x
 *
 * Revision 1.5  2001/11/06 23:04:44  mt
 * applied / merged with transparent proxy patches v8
 * see ftp-proxy/NEWS for more detailed release news
 *
 * Revision 1.4  1999/09/24 06:38:52  wiegand
 * added regular expressions for all commands
 * removed character map and length of paths
 * added flag to reset PASV on every PORT
 * added "magic" user with built-in destination
 * added some argument pointer fortification
 *
 * Revision 1.3  1999/09/21 07:14:43  wiegand
 * syslog / abort cleanup and review
 * default PASV port range to 0:0
 *
 * Revision 1.2  1999/09/17 16:32:29  wiegand
 * changes from source code review
 * added POSIX regular expressions
 *
 * Revision 1.1  1999/09/15 14:06:22  wiegand
 * initial checkin
 *
 * ------------------------------------------------------------ */

