/*  mingetty.c
 *
 *  Copyrighr (C) 1996 Florian La Roche
 *  florian@jurix.jura.uni-sb.de florian@suse.de florian@knorke.saar.de
 *
 *  Newer versions should be on susix.jura.uni-sb.de/pub/linux/source/system
 *  or sunsite.unc.edu/pub/Linux/system/Admin/login or /pub/Linux/system/
 *  Daemons/init/.
 *
 *  utmp-handling is from agetty in util-linux 2.5 (probably from
 *  Peter Orbaek <poe@daimi.aau.dk>)
 *
 *  SuSE - GmbH has paid some of the time that was needed to write
 *  this program. Thanks a lot for their support.
 *
 *  This getty can only be used as console getty. It is very small, but
 *  should be very reliable. For a modem getty, I'd also use nothing else
 *  but mgetty.
 *
 *  Usage: mingetty [--noclear] [--noreset] [--login loginprog] 
 *   [--logopts "options for login prog"] tty
 *  Example entry in /etc/inittab: 1:123:respawn:/sbin/mingetty tty1
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version
 *  2 of the License, or (at your option) any later version.
 *
 *  Kurt Garloff <garloff@suse.de>, Feb 98, Jun 99:
 *  Added --login and --logopts options to let the superuser choose
 *    the login program and its options used. 
 *    \\u will be replaced by the user name.
 *  Example (/etc/inittab):
 *  13:23:respawn:/sbin/mingetty --login /usr/bin/ssh --logopts "\\u@target" tty13
 *  mingetty then execs "/usr/bin/ssh logname@target" 
 *  instead of          "/bin/login -- logname"
 * SECURITY notice: mingetty checks the name for a leading - to make sure
 *  the user can not easily pass an option instead of a logname.
 * Better: If the program supports it, pass -- before the username gets
 *  passed, which will prevent the interpretation as options.
 *
 *  Tue Sep  7 21:20:01 CEST 1999 - Werner Fink <werner@suse.de>
 *	Enable and revisit resetting of tty.
 *	Use wtmp interface of glibc
 *	Set tty process group
 *	Restore signal handling in signal handler
 *
 *  Wed May 10 23:25:02 CEST 2000 - Kurt Garloff <garloff@suse.de>
 *	Add support for devfs by delaying the /dev/vcs[a] 
 *	permission changes.
 *
 *  Mon Oct 23 12:40:00 CEST 2000 - Werner Fink <werner@suse.de>
 *	On ro files systems chmod/chown shouldn't force an exit.
 *	Therefore we need a warn message.
 *	Due to the lack of a revoke system call we signal
 *	all file holders including the mmaping processes.
 *
 *  Wed Apr 24 18:54:40 CEST 2002 - Werner Fink <werner@suse.de>
 *	Add support for other TTY devices than virtual console
 *	Add nohost patch from Anders ??
 */

#define VERSION "mingetty-0.9.4b"
#ifndef DEBUG_THIS
# define DEBUG_THIS 0
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/syscall.h>
extern int sigfholder(const char *path, int sig);
#include <string.h>
#include <grp.h>
#include <sys/ioctl.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#if defined __GLIBC__ && __GLIBC__ >= 2
# include <signal.h>
#else
# include <sys/signal.h>
#endif
#include <fcntl.h>
#include <stdarg.h>
#include <ctype.h>
#include <utmp.h>
#include <getopt.h>

#ifdef RESET_TTY
# if defined __GLIBC__ && __GLIBC__ >= 2
#  include <termios.h>
# else
#  include <sys/termios.h>
# endif
# include <sys/ttydefaults.h>
#endif

#if defined(_BSD_SOURCE)
# define strerror(errno)	sys_errlist[(errno)]
#else
# define strerror(errno)	strerror((errno))
#endif

#ifdef linux
#include <sys/param.h>
#define USE_SYSLOG
#endif

 /* If USE_SYSLOG is undefined all diagnostics go directly to /dev/console. */
#ifdef	USE_SYSLOG
#include <syslog.h>
#endif

#define	ISSUE "/etc/issue"	/* displayed before the login prompt */
#include <sys/utsname.h>
#include <time.h>

#define LOGIN "login: "		/* login prompt */

#ifndef _PATH_LOGIN
#define _PATH_LOGIN "/bin/login"
#endif

/* name of this program (argv[0]) */
static char *progname;
/* on which tty line are we sitting? (e.g. tty1) */
static char *tty;
/* some information about this host */
static struct utsname uts;
/* the hostname */
static char hn[MAXHOSTNAMELEN + 1];
/* process ID of this program */
static pid_t pid;
/* session ID of this program */
static pid_t sid;
/* current time */
static time_t cur_time;
/* do not send a reset string to the terminal ? */
static int noclear = 0;
/* Print the whole string of gethostname() instead of just until the next "." */
static int longhostname = 0;
/* Do not print the hostname at all */
static int nohostname = 0;
/* do not reset the terminal ? */
static int noreset = 0;
/* login program */
static char* loginprog;

/*
 * output warn/error messages
 */
static void _vlogger (int priority, const char *fmt, va_list va_alist)
{
	char buf[256], *bp;
#ifndef	USE_SYSLOG
	int fd;
#endif

#ifdef USE_SYSLOG
	buf[0] = '\0';
	bp = buf;
#else
	strncpy (buf, progname, sizeof(buf) - 4);
	buf[sizeof(buf) - 5] = '\0';
	strcat (buf, ": ");
	bp = buf + strlen (buf);
#endif

	vsnprintf (bp, sizeof(buf)-strlen(buf), fmt, va_alist);

#ifdef	USE_SYSLOG
	openlog (progname, LOG_PID, LOG_AUTH);
	syslog (priority, buf);
	closelog ();
#else
	strcat (bp, "\r\n");
	if ((fd = open ("/dev/console", 1)) >= 0) {
		write (fd, buf, strlen (buf));
		close (fd);
	}
#endif
}

/*
 * output error messages
 */
void error (const char *fmt, ...)
{
	va_list va_alist;
	va_start (va_alist, fmt);
	_vlogger (LOG_ERR, fmt, va_alist);
	va_end (va_alist);
	exit (1);
}

/*
 * output warn messages
 */
void warn (const char *fmt, ...)
{
	va_list va_alist;
	va_start (va_alist, fmt);
	_vlogger (LOG_WARNING, fmt, va_alist);
	va_end (va_alist);
	return;
}

static void sigquit_handler (int signum) {
	struct sigaction act;
	sigset_t set;

	/* set default signal action */
  	act.sa_handler = SIG_DFL;
	act.sa_flags = 0;
	if (sigemptyset (&act.sa_mask)	||
	    sigaction (signum, &act, NULL))
		exit (1);

	/* unmask signum */
	if (sigemptyset (&set)		||
	    sigaddset (&set, signum)	||
	    sigprocmask (SIG_UNBLOCK, &set, NULL))
		exit (1);

	kill (getpid(), signum);
	abort();
}

/*
 * update_utmp - update our utmp entry
 *
 * The utmp file holds miscellaneous information about things started by
 * /sbin/init and other system-related events. Our purpose is to update
 * the utmp entry for the current process, in particular the process
 * type and the tty line we are listening to. Return successfully only
 * if the utmp file can be opened for update, and if we are able to find
 * our entry in the utmp file.
 */
static void update_utmp (void)
{
	struct utmp ut;
#if !(defined __GLIBC__ && __GLIBC__ >= 2)
	int ut_fd;
#endif
	struct utmp *utp;

	utmpname (_PATH_UTMP);
	setutent ();
	while ((utp = getutent ()))
		if (utp->ut_type == INIT_PROCESS && utp->ut_pid == pid)
			break;

	if (utp) {
		memcpy (&ut, utp, sizeof (ut));
	} else {
		/* some inits don't initialize utmp... */
		/* XXX we should print out a warning message */
		memset (&ut, 0, sizeof (ut));
		if (strncmp(tty, "tty", 3) == 0 && strspn(tty + 3, "0123456789") == strlen(tty+3))
			/* Standard virtual console devices */
			strncpy (ut.ut_id, tty + 3, sizeof (ut.ut_id));
		else {
			size_t len = strlen(tty);
			char * ptr;
			if (len >= sizeof (ut.ut_id))
				ptr = tty + len - sizeof (ut.ut_id);
			else
				ptr = tty;
			strncpy (ut.ut_id, ptr, sizeof (ut.ut_id));
		}
	}
	endutent ();

	strncpy (ut.ut_user, "LOGIN", sizeof (ut.ut_user));
	strncpy (ut.ut_line, tty, sizeof (ut.ut_line));
	ut.ut_time = cur_time;
	ut.ut_type = LOGIN_PROCESS;
	ut.ut_pid = pid;
	ut.ut_session = sid;

	setutent ();
	pututline (&ut);
	endutent ();

#if defined __GLIBC__ && __GLIBC__ >= 2
	updwtmp(_PATH_WTMP, &ut);
#else
	if ((ut_fd = open (_PATH_WTMP, O_APPEND | O_WRONLY)) >= 0) {
		/* never lock the wtmp file */
		/* flock (ut_fd, LOCK_EX); */
		write (ut_fd, &ut, sizeof (ut));
		/* flock (ut_fd, LOCK_UN); */
		close (ut_fd);
	}
#endif
}

/* open_tty - set up tty as standard { input, output, error } */
static void open_tty (void)
{
	struct sigaction sa;
	sigset_t set;
	/* Was `char buf[20];' but there is more than /dev/tty<number> */
	char buf[PATH_MAX+1];
	int fd;
	gid_t gid = 0;
#ifdef RESET_TTY 
	struct termios termios;
#endif
	struct group *gr = NULL;

	/* Use tty group is available */
	if ((gr = getgrnam("tty")))
		gid = gr->gr_gid;

	/* Set up new standard input. */
	if (snprintf(buf, sizeof(buf), "/dev/%s", tty) < 0) {
		error ("%s: %s", buf, strerror(EOVERFLOW));
	}
	if (chown (buf, 0, gid) || chmod (buf, (gid ? 0660 : 0600))) {
		if (errno == EROFS)
			warn("%s: %s", buf, strerror(errno));
		else
			error ("%s: %s", buf, strerror(errno));
	}

	sa.sa_handler = SIG_IGN;
	sa.sa_flags = 0;
	sigemptyset (&sa.sa_mask);
	sigaction (SIGHUP, &sa, NULL);
	sa.sa_handler = sigquit_handler;
	sigaction (SIGQUIT, &sa, NULL);

#if DEBUG_THIS
	if ((fd = open("/dev/tty", O_RDWR, 0)) >=0) {
		ioctl(fd, TIOCNOTTY, (void *)1);
		close(fd);
		(void)setsid();
	}
#endif
	/* vhangup() will replace all open file descriptors that point to our
	   controlling tty by a dummy that will deny further reading/writing
	   to our device. It will also reset the tty to sane defaults, so we
	   don't have to modify the tty device for sane settings.
	   We also get a SIGHUP/SIGCONT.
	 */
	if ((fd = open (buf, O_RDWR, 0)) < 0)
		error ("%s: cannot open tty: %s", buf, strerror(errno));
	if (ioctl (fd, TIOCSCTTY, (void *)1) == -1)
		error ("%s: cannot get controlling tty: %s", buf, strerror(errno));
	if (!isatty (fd))
		error ("%s: not a tty", buf);

	if (vhangup ())
		error ("vhangup() failed");
	/* Get rid of the present stdout/stderr. */
	close (2);
	close (1);
	close (0);
	close (fd);

	if ((fd = open (buf, O_RDWR, 0)) < 0)
		error ("%s: cannot open tty: %s", buf, strerror(errno));

	if (tcsetpgrp(fd, pid))
		error ("%s: cannot set process group: %s", buf, strerror(errno));

	/* Set up standard output and standard error file descriptors. */
	if (dup2 (fd, 0) != 0)
		error ("%s: dup problem: %s", buf, strerror(errno));
	if (dup2 (fd, 1) != 1)
		error ("%s: dup problem: %s", buf, strerror(errno));
	if (dup2 (fd, 2) != 2)
		error ("%s: dup problem: %s", buf, strerror(errno));
	if (fd > 2)
		close (fd);

	/* Write a reset string to the terminal. This is very linux-specific
	   and should be checked for other systems. */
	if (!noclear)
		write (0, "\033c", 2);

#ifdef linux
	/*
	 * Only for the standard virtual console devices
	 */
	if (strncmp(tty, "tty", 3) == 0 && strspn(tty + 3, "0123456789") == strlen(tty+3)) {
		struct stat st;
		/* Set up new virtual console text capture devices */
		strcpy (buf, "/dev/vcs");
		strcat (buf, &(tty[3]));

		if (stat(buf, &st) < 0) {
			if (errno != ENOENT)
				error ("%s: %s", buf, strerror(errno));
			goto skip;
		}
		if (chown (buf, 0, gid) || chmod (buf, (gid ? 0660 : 0600))) {
			if (errno == EROFS)
				warn("%s: %s", buf, strerror(errno));
			else
				error ("%s: %s", buf, strerror(errno));
		}
#ifdef __NR_revoke
		_syscall(int, revoke, const char*, buf);
#else
		if (sigfholder(buf, SIGHUP)) {		/* Be nice, send SIGHUP */
			sleep (2);
			if (sigfholder(buf, SIGKILL))	/* Hey, go away */
				sigfholder(buf, SIGCONT);
		}
#endif

		/* Set up new virtual console text/attribute capture devices */
		strcpy (buf, "/dev/vcsa");
		strcat (buf, &(tty[3]));

		if (stat(buf, &st) < 0) {
			if (errno != ENOENT)
				error ("%s: %s", buf, strerror(errno));
			goto skip;
		}
		if (chown (buf, 0, gid) || chmod (buf, (gid ? 0660 : 0600))) {
			if (errno == EROFS)
				warn("%s: %s", buf, strerror(errno));
			else
				error ("%s: %s", buf, strerror(errno));
		}
#ifdef __NR_revoke
		_syscall(int, revoke, const char*, buf);
#else
		if (sigfholder(buf, SIGHUP)) {		/* Be nice, send SIGHUP */
			sleep (2);
			if (sigfholder(buf, SIGKILL))	/* Hey, go away */
				sigfholder(buf, SIGCONT);
		}
#endif
	}
skip:
#endif

#ifdef RESET_TTY 
	if (noreset)
		goto out;

	/* The above reset only puts the output things into a sane state.
	 * The input state is not reset.
	 */
	memset (&termios, 0, sizeof termios);
	if (tcgetattr (0, &termios))
		warn ("tcgetattr problem: %s", strerror(errno));

	/* Use defaults of <sys/ttydefaults.h> for base settings */
	termios.c_iflag |= TTYDEF_IFLAG;
	termios.c_oflag |= TTYDEF_OFLAG;
	termios.c_lflag |= TTYDEF_LFLAG;
	termios.c_cflag |= (TTYDEF_SPEED | TTYDEF_CFLAG);

	/* Sane setting, allow eight bit characters, no carriage return delay
	 * the same result as `stty sane cr0 pass8'
	 */
	termios.c_iflag |=  (BRKINT | ICRNL | IMAXBEL);
	termios.c_iflag &= ~(IGNBRK | INLCR | IGNCR | IXOFF | IUCLC | IXANY | ISTRIP);
	termios.c_oflag |=  (OPOST | ONLCR | NL0 | CR0 | TAB0 | BS0 | VT0 | FF0);
	termios.c_oflag &= ~(OLCUC | OCRNL | ONOCR | ONLRET | OFILL | OFDEL |\
			     NLDLY|CRDLY|TABDLY|BSDLY|VTDLY|FFDLY);
	termios.c_lflag |=  (ISIG | ICANON | IEXTEN | ECHO|ECHOE|ECHOK|ECHOCTL|ECHOKE);
	termios.c_lflag &= ~(ECHONL | NOFLSH | XCASE | TOSTOP | ECHOPRT);
	termios.c_cflag |=  (CREAD | CS8 | B38400);
	termios.c_cflag &= ~(PARENB);

	/* VTIME and VMIN can overlap with VEOF and VEOL since they are
	 * only used for non-canonical mode. We just set the at the
	 * beginning, so nothing bad should happen.
	 */
	termios.c_cc[VTIME]    = 0;
	termios.c_cc[VMIN]     = 1;
	termios.c_cc[VINTR]    = CINTR;
	termios.c_cc[VQUIT]    = CQUIT;
	termios.c_cc[VERASE]   = CERASE; /* ASCII DEL (0177) */
	termios.c_cc[VKILL]    = CKILL;
	termios.c_cc[VEOF]     = CEOF;
	termios.c_cc[VSWTC]    = _POSIX_VDISABLE;
	termios.c_cc[VSTART]   = CSTART;
	termios.c_cc[VSTOP]    = CSTOP;
	termios.c_cc[VSUSP]    = CSUSP;
	termios.c_cc[VEOL]     = _POSIX_VDISABLE;
	termios.c_cc[VREPRINT] = CREPRINT;
	termios.c_cc[VDISCARD] = CDISCARD;
	termios.c_cc[VWERASE]  = CWERASE;
	termios.c_cc[VLNEXT]   = CLNEXT;
	termios.c_cc[VEOL2]    = _POSIX_VDISABLE;
	if (tcsetattr (0, TCSADRAIN, &termios))
		warn ("tcsetattr problem: %s", strerror(errno));
#endif

out:
	sa.sa_handler = SIG_DFL;
	sa.sa_flags = 0;
	sigemptyset (&sa.sa_mask);
	sigaction (SIGHUP, &sa, NULL);

	/* Unmask SIGHUP if inherited */
	sigemptyset (&set);
	sigaddset (&set, SIGHUP);
	sigprocmask (SIG_UNBLOCK, &set, NULL);

#if DEBUG_THIS
	printf ("session=%d, pid=%d, pgid=%d\n", getsid (0), getpid (),
			getpgid (0));
#endif
}

static void output_special_char (unsigned char c)
{
	switch (c) {
	case 's':
		printf ("%s", uts.sysname);
		break;
	case 'n':
		printf ("%s", uts.nodename);
		break;
	case 'r':
		printf ("%s", uts.release);
		break;
	case 'v':
		printf ("%s", uts.version);
		break;
	case 'm':
		printf ("%s", uts.machine);
		break;
	case 'o':
#ifdef __USE_GNU
		printf ("%s", uts.domainname);
#else
		printf ("%s", uts.__domainname);
#endif
		break;
#if 0
	case 'd':
	case 't':
		{
			char *weekday[] =
			{"Sun", "Mon", "Tue", "Wed", "Thu",
			 "Fri", "Sat"};
			char *month[] =
			{"Jan", "Feb", "Mar", "Apr", "May",
			 "Jun", "Jul", "Aug", "Sep", "Oct",
			 "Nov", "Dec"};
			time_t now;
			struct tm *tm;

			time (&now);
			tm = localtime (&now);

			if (c == 'd')
				printf ("%s %s %d  %d",
				    weekday[tm->tm_wday], month[tm->tm_mon],
					tm->tm_mday,
				     tm->tm_year < 70 ? tm->tm_year + 2000 :
					tm->tm_year + 1900);
			else
				printf ("%02d:%02d:%02d",
					tm->tm_hour, tm->tm_min, tm->tm_sec);

			break;
		}
/* this is a second method, that is also not used anymore... */
	case 'd':
	case 't':
		{
			char buff[20];
			struct tm *tm = localtime (&cur_time);
			strftime (buff, sizeof (buff),
				c == 'd'? "%a %b %d %Y" : "%X", tm);
			fputs (buff, stdout);
			break;
		}
#else
	case 'd':
	case 't':
		{
			time_t now;
			struct tm *tm;

			time (&now);
			tm = localtime (&now);
			if (c == 'd')
				/* ISO 8601 */
				printf ("%d-%02d-%02d", 1900 + tm->tm_year,
					tm->tm_mon+1, tm->tm_mday);
                        else
                                printf ("%02d:%02d:%02d",
                                        tm->tm_hour, tm->tm_min, tm->tm_sec);
			break;
		}
#endif

	case 'l':
		printf ("%s", tty);
		break;
	case 'u':
	case 'U':
		{
			int users = 0;
			struct utmp *ut;
			setutent ();
			while ((ut = getutent ()))
				if (ut->ut_type == USER_PROCESS)
					users++;
			endutent ();
			printf ("%d", users);
			if (c == 'U')
				printf (" user%s", users == 1 ? "" : "s");
			break;
		}
	default:
		putchar (c);
	}
}

/* do_prompt - show login prompt, optionally preceded by /etc/issue contents */
static void do_prompt (void)
{
#if	! OLD
	FILE *fd;
#else
	int fd;
#endif
	int c;

	write (1, "\n", 1);	/* start a new line */
#if	! OLD
	if ((fd = fopen (ISSUE, "r"))) {
		while ((c = getc (fd)) != EOF) {
			if (c == '\\')
				output_special_char (getc(fd));
			else
				putchar (c);
		}
		fflush (stdout);
		fclose (fd);
	}
#else
	if ((fd = open (ISSUE, O_RDONLY)) >= 0) {
		close (fd);
	}
#endif
	if (!nohostname) {
		write (1, hn, strlen (hn));
		write (1, " ", 1);
	}
	write (1, LOGIN, sizeof (LOGIN) - 1);
}

/* get_logname - get user name, establish speed, erase, kill, eol */
static char *get_logname (void)
{
	static char logname[40];
	char *bp;
	unsigned char c;

	/* flush pending input */
#if defined __GLIBC__ && __GLIBC__ >= 2
	tcflush(0, TCIFLUSH);
#else
	ioctl (0, TCFLSH, 0);
#endif

	for (*logname = 0; *logname == 0;) {
		do_prompt ();
		for (bp = logname;;) {
			if (read (0, &c, 1) < 1) {
				if (errno == EINTR || errno == EIO
							|| errno == ENOENT)
					exit (0);
				error ("%s: read: %s", tty, strerror(errno));
			}
			if (c == '\n' || c == '\r') {
				*bp = 0;
				break;
			} else if (!isprint (c))
				error ("%s: invalid character for login name",
								tty);
			else if (bp - logname >= sizeof (logname) - 1)
				error ("%s: too long login name", tty);
			else
				*bp++ = c;
		}
	}
	return logname;
}

static void usage (void)
{
	error ("%s: usage: '%s [--noclear] [--noreset] [--long-hostname] [--no-hostname] [--login program] [--logopts \"loginprg opts\"] tty' with e.g. tty=tty1", VERSION, progname);
}

static struct option const long_options[] = {
	{ "noclear", no_argument, &noclear, 1},
	{ "long-hostname", no_argument, &longhostname, 1},
	{ "no-hostname", no_argument, &nohostname, 1},
	{ "noreset", no_argument, &noreset, 1},
	{ "login", required_argument, 0, 2},
	{ "logopts", required_argument, 0, 3},
	{ 0, 0, 0, 0 }
};

/* Don't allow the user to pass an option as a user name */
/* To be more safe: Use -- to make sure the rest is interpreted
 * as non-options by the program, if it supports it */
void checkname (char* nm)
{
	char *p = nm;
	if (!nm) abort ();
	if (strlen (nm) > 42) abort ();
	while (isspace (*p)) p++;
	if (*p == '-') abort ();
}

void replacename (char** arr, char* nm)
{
   char *p, *tmp;
   while ((p = *arr)) {
	char *p1 = p;
	while (*p1) {
		if (memcmp (p1, "\\u", 2) == 0)
		{
			tmp = malloc (strlen (p) + strlen (nm));
			if (!tmp) abort ();
			if (p1 != p) memcpy (tmp, p, (p1-p));
			*(tmp + (p1-p)) = 0;
			strcat (tmp, nm);
			strcat (tmp, p1+2);
			*arr = tmp;
		}
		p1++;
	}
	arr++;
   }
}

void mkarray (char** arr, char* str)
{
   char* p = str; char* start = p;
   int i = 0;
   while (*p && i < 14)
   {
	if (isspace (*p))
	{
		*p = 0;
		while (isspace (*++p));
		if (*p) { arr[i++] = start; start = p; };
	}
	else
		p++;
   };
   arr[i++] = start; arr[i++] = 0;
}

/*
 * main program
 */
int main (int argc, char **argv)
{
	char *logname, *s;
	int c;
	char *logoptstr;
	char logcmd[256];
	char *logarr[16];

	loginprog = _PATH_LOGIN;
	logoptstr = "-- \\u";
	
	progname = basename(argv[0]);
	uname (&uts);
	gethostname (hn, MAXHOSTNAMELEN);
	pid = getpid ();
	sid = getsid (0);
	time (&cur_time);
#ifdef linux
	putenv ("TERM=linux");
#endif

	while ((c = getopt_long (argc, argv, "", long_options, (int *) 0))
			!= EOF) {
		switch (c) {
			case 0:
				break;
		        case 2:
		                loginprog = optarg; break;
		        case 3:
		                logoptstr = optarg; break;
			default:
				usage ();
		}
	}
	if (!longhostname && (s = strchr(hn, '.')))
		*s = '\0';
	tty = argv[optind];
	if (!tty)
		usage ();

	/* Skip the "/dev/", we may add it later */
	if (strncmp (tty, "/dev/", 5) == 0)
		tty += 5;

	update_utmp ();
	open_tty ();

	/* flush input and output queues, important for modems */
#if defined __GLIBC__ && __GLIBC__ >= 2
	tcflush(0, TCIOFLUSH);
#else
	ioctl (0, TCFLSH, 2);
#endif

	while ((logname = get_logname ()) == 0);
	strcpy (logcmd, loginprog); strcat (logcmd, " ");
	strcat (logcmd, logoptstr);
	mkarray (logarr, logcmd);
	checkname (logname); replacename (logarr, logname);

#if DEBUG_THIS
	printf ("About to execute %s", logcmd);
	c = 1; while (logarr[c]) printf (" `%s'", logarr[c++]);
	printf ("\n"); sleep (1);
#endif
	execv (loginprog, logarr);
	error ("%s: can't exec %s: %s", tty, loginprog, strerror(errno));
	sleep (1); exit (0);
}

