/*
 * Copyright (c) 2004 Michael Schroeder (mls@suse.de)
 *
 * This program is licensed under the BSD license, read LICENSE.BSD
 * for further information
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <sys/stat.h>

#include <bzlib.h>
#include <zlib.h>

#include "util.h"
#include "md5.h"
#include "rpmhead.h"
#include "rpml.h"
#include "cpio.h"
#include "delta.h"
#include "cfile.h"

/********************************************************************/

void
write32(struct cfile *cf, unsigned int d)
{
  unsigned char dd[4];
  dd[0] = d >> 24;
  dd[1] = d >> 16;
  dd[2] = d >> 8;
  dd[3] = d;
  if (cf->write(cf, dd, 4) != 4)
    {
      perror("write32");
      exit(1);
    }
}

void
writedeltarpm(char *name, unsigned char *rpmlead, struct rpmhead *sigh, struct rpmhead *h, struct instr *instr, int instrlen, unsigned char *addbz2, int addbz2len, unsigned char *new, int oldlen, char *payformat, unsigned char *fullmd5res, unsigned int fullsize, unsigned char *seqmd5res, unsigned char *seq, int seql, char *nevr, int comp, int version)
{
  unsigned char sighdr[16 + 16 * 3 + 4 + 16 + 16 + 4];
  struct cfile *bfd;
  unsigned int *b1;
  int nb1;
  unsigned int *b2;
  int nb2;
  int fd;
  int i, j;
  int datalen, outoff, lastoff;
  unsigned int x1, x2;
  int x3;
  unsigned int written;
  MD5_CTX paymd5;
  unsigned char paymd5res[16];

  if ((fd = open(name, O_RDWR|O_CREAT|O_TRUNC, 0666)) == -1)
    {
      perror(name);
      exit(1);
    }
  b1 = b2 = 0;
  nb1 = nb2 = 0;
  j = 0;
  datalen = 0;
  outoff = lastoff = 0;
  for (i = 0; i < instrlen; i++)
    {
      x1 = instr[i].copyout;
      x2 = instr[i].copyin;
      x3 = instr[i].seekout;
      if (!x1)
	{
	  outoff += x3;
	  x3 = 0;
	  if (!x2)
	    continue;
	}
      if (x1)
	{
	  if ((nb2 & 15) == 0)
	    b2 = xrealloc(b2, (nb2 + 16) * 2 * sizeof(unsigned int));
	  if (lastoff < outoff)
	    b2[2 * nb2] = outoff - lastoff;
	  else
	    b2[2 * nb2] = (lastoff - outoff) | 0x80000000;
	  b2[2 * nb2 + 1] = x1;
	  nb2++;
	  j++;
	  lastoff = outoff + x1;
	  outoff = lastoff + x3;
	  if ((int)outoff < 0)
	    {
	      fprintf(stderr, "ctrlstrm: negative seek\n");
	      exit(1);
	    }
	}
      if (x2)
	{
	  if ((nb1 & 15) == 0)
	    b1 = xrealloc(b1, (nb1 + 16) * 2 * sizeof(unsigned int));
	  b1[2 * nb1] = j;
	  b1[2 * nb1 + 1] = x2;
	  nb1++;
	  j = 0;
	  datalen += x2;
	}
    }
  if (j)
    {
      if ((nb1 & 15) == 0)
	b1 = xrealloc(b1, (nb1 + 16) * 2 * sizeof(unsigned int));
      b1[2 * nb1] = j;
      b1[2 * nb1 + 1] = 0;
      nb1++;
    }

  if (write(fd, rpmlead, 96) != 96)
    {
      perror("rpmlead write");
      exit(1);
    }
  memset(sighdr, 0, sizeof(sighdr));
  sighdr[0] = 0x8e;
  sighdr[1] = 0xad;
  sighdr[2] = 0xe8;
  sighdr[3] = 0x01;
  sighdr[11] = 3;
  sighdr[15] = 4 + 16 + 16;

  sighdr[16*1+3] = 0x3e;        /* HEADER_SIGNATURES */
  sighdr[16*1+7]  = 7;
  sighdr[16*1+11]  = 4 + 16;
  sighdr[16*1+15]  = 16;

  sighdr[16*2+2] = 0x03;        /* size */
  sighdr[16*2+3] = 0xe8;
  sighdr[16*2+7]  = 4;
  sighdr[16*2+15] = 1;

  sighdr[16*3+2] = 0x03;        /* md5 */
  sighdr[16*3+3] = 0xec;
  sighdr[16*3+7]  = 7;
  sighdr[16*3+11] = 4;
  sighdr[16*3+15] = 16;

  sighdr[16 + 16 * 3 + 4 + 16 + 3] = 0x3e;
  sighdr[16 + 16 * 3 + 4 + 16 + 7] = 7;
  sighdr[16 + 16 * 3 + 4 + 16 + 8] = 0xff;
  sighdr[16 + 16 * 3 + 4 + 16 + 9] = 0xff;
  sighdr[16 + 16 * 3 + 4 + 16 + 10] = 0xff;
  sighdr[16 + 16 * 3 + 4 + 16 + 11] = 256 - 3 * 16;
  sighdr[16 + 16 * 3 + 4 + 16 + 15] = 16;

  if (write(fd, sighdr, sizeof(sighdr)) != sizeof(sighdr))
    {
      perror("sig hdr write");
      exit(1);
    }
  rpmMD5Init(&paymd5);
  if (write(fd, h->intro, 16) != 16)
    {
      perror("hdr write");
      exit(1);
    }
  rpmMD5Update(&paymd5, h->intro, 16);
  strncpy(payformat, "drpm", 4);
  if (write(fd, h->data, 16 * h->cnt + h->dcnt) != 16 * h->cnt + h->dcnt)
    {
      perror("hdr write");
      exit(1);
    }
  rpmMD5Update(&paymd5, h->data, 16 * h->cnt + h->dcnt);
  if ((bfd = cfile_open(CFILE_OPEN_WR, fd, 0, comp, CFILE_LEN_UNLIMITED, (cfile_ctxup)rpmMD5Update, &paymd5)) == 0)
    {
      fprintf(stderr, "payload open failed\n");
      exit(1);
    }
  write32(bfd, 0x444c5430 + version);
  write32(bfd, strlen(nevr) + 1);
  if (bfd->write(bfd, nevr, strlen(nevr) + 1) != strlen(nevr) + 1)
    {
      fprintf(stderr, "payload write failed\n");
      exit(1);
    }
  write32(bfd, seql + 16);
  if (bfd->write(bfd, seqmd5res, 16) != 16)
    {
      fprintf(stderr, "payload write failed\n");
      exit(1);
    }
  if (bfd->write(bfd, seq, seql) != seql)
    {
      fprintf(stderr, "payload write failed\n");
      exit(1);
    }
  if (bfd->write(bfd, fullmd5res, 16) != 16)
    {
      fprintf(stderr, "payload write failed\n");
      exit(1);
    }
  if (version > 1)
    {
      write32(bfd, fullsize);
      write32(bfd, comp);
      write32(bfd, 0);
    }
  write32(bfd, 96 + 16 + sigh->cnt * 16 + sigh->dcnt);
  if (bfd->write(bfd, rpmlead, 96) != 96)
    {
      fprintf(stderr, "payload write failed\n");
      exit(1);
    }
  if (bfd->write(bfd, sigh->intro, 16) != 16)
    {
      fprintf(stderr, "payload write failed\n");
      exit(1);
    }
  if (bfd->write(bfd, sigh->data, sigh->cnt * 16 + sigh->dcnt) != sigh->cnt * 16 + sigh->dcnt)
    {
      fprintf(stderr, "payload write failed\n");
      exit(1);
    }
  write32(bfd, payformat - (char *)h->dp);
  write32(bfd, nb1);
  write32(bfd, nb2);
  for (i = 0; i < nb1; i++)
    write32(bfd, b1[2 * i]);
  for (i = 0; i < nb1; i++)
    write32(bfd, b1[2 * i + 1]);
  for (i = 0; i < nb2; i++)
    write32(bfd, b2[2 * i]);
  for (i = 0; i < nb2; i++)
    write32(bfd, b2[2 * i + 1]);
  write32(bfd, oldlen);
  write32(bfd, addbz2len);
  if (addbz2len)
    bfd->write(bfd, addbz2, addbz2len);
  write32(bfd, datalen);
  for (i = 0; i < instrlen; i++)
    bfd->write(bfd, new + instr[i].copyinoff, instr[i].copyin);
  if ((written = bfd->close(bfd)) == -1)
    {
      fprintf(stderr, "payload write failed\n");
      exit(1);
    }
  if (lseek(fd, (off_t)96, SEEK_SET) == (off_t)-1)
    {
      fprintf(stderr, "sig seek failed\n");
      exit(1);
    }
  rpmMD5Final(paymd5res, &paymd5);
  /* add header size */
  written += 16 + h->cnt * 16 + h->dcnt;
  sighdr[16 + 16 * 3 + 0] = written >> 24;
  sighdr[16 + 16 * 3 + 1] = written >> 16;
  sighdr[16 + 16 * 3 + 2] = written >> 8;
  sighdr[16 + 16 * 3 + 3] = written;
  memcpy(&sighdr[16 + 16 * 3 + 4], paymd5res, 16);
  if (write(fd, sighdr, sizeof(sighdr)) != sizeof(sighdr))
    {
      fprintf(stderr, "sig write failed\n");
      exit(1);
    }
  if (close(fd))
    {
      fprintf(stderr, "payload write failed\n");
      exit(1);
    }
  b1 = xfree(b1);
  b2 = xfree(b2);
}

/********************************************************************/



char *
headtofiles(struct rpmhead *h, struct rpmlfile **filesp, int *nfilesp)
{
  char *nevr, *n;
  int cnt, i;
  struct rpmlfile *files;
  unsigned int *fileflags, *filemodes;
  char **filelist, **filemd5s;

  nevr = headtonevr(h);
  if (!nevr)
    return 0;
  filelist = headexpandfilelist(h, &cnt);
  if (!filelist || !cnt)
    {
      *nfilesp = 0;
      *filesp = 0;
      return nevr;
    }
  fileflags = headint32(h, TAG_FILEFLAGS, (int *)0);
  filemd5s = headstringarray(h, TAG_FILEMD5S, (int *)0);
  filemodes = headint16(h, TAG_FILEMODES, (int *)0);
  files = xmalloc(cnt * sizeof(*files));
  for (i = 0; i < cnt; i++)
    {
      n = filelist[i];
      if (*n == '/')
	n++;
      files[i].name = xmalloc(strlen(n) + 1);
      strcpy(files[i].name, n);
      files[i].mode = filemodes[i];
      files[i].fflags = fileflags[i];
      parsemd5(filemd5s[i], files[i].md5);
    }
  *filesp = files;
  *nfilesp = cnt;
  free(fileflags);
  free(filemd5s);
  free(filemodes);
  free(filelist);
  return nevr;
}

unsigned char *seq;
int seqp;
int seql;

int lastseq = -1;
int lastseqstart = 0;

void
addtoseqn(int n)
{
  int l = 1, j = 8;
  while (n >= j)
    {
      j *= 8;
      l++;
    }
  if (seqp + l > seql * 2)
    {
      seq = xrealloc(seq, seql + 32);
      seql += 32;
    }
  while (l-- > 0)
    {
      if (seqp & 1)
	seq[seqp / 2] |= (n & 7) | (l ? 8 : 0);
      else
	seq[seqp / 2] = ((n & 7) | (l ? 8 : 0)) << 4;
      seqp++;
      n >>= 3;
    }
}

void
addtoseq(int i)
{
  int n;
  // fprintf(stderr, "addtoseq %d lastseqstart=%d lastseq=%d\n", i, lastseqstart, lastseq);
  if (i == lastseq + 1)
    {
      lastseq = i;
      return;
    }
  n = lastseq - lastseqstart + 1;
  if (n)
    addtoseqn(n);
  if (i > lastseq + 1 && lastseq >= 0)
    {
      addtoseqn(i - lastseq - 1);
    }
  else if (i != -1)
    {
      addtoseqn(0);
      addtoseqn(i);
    }
  lastseq = lastseqstart = i;
}

struct prune {
  char *name;
};

struct prune *prunes;
int prunen;

void
read_prunelist(char *file)
{
  FILE *fp;
  char *buf, *bp;
  int i, c, bufl;
  struct prune *p;

  if ((fp = fopen(file, "r")) == 0)
    {
      perror(file);
      exit(1);
    }
  bufl = 256;
  buf = xmalloc(256);
  prunes = xmalloc(sizeof(*prunes) * 16);
  for (i = 0;;)
    {
      c = getc(fp);
      if (c && c != EOF && c != '\n')
	{
	  buf[i++] = c;
          if (i == bufl)
	    {
	      bufl += 256;
	      buf = xrealloc(buf, bufl);
	      if (!buf)
		{
		  fprintf(stderr, "out of memory");
		  exit(1);
		}
	    }
	  continue;
	}
      buf[i] = 0;
      bp = buf;
      if (*bp == '/')
	bp++;
      else if (*bp == '.' && bp[1] == '/')
	bp += 2;
      if (!*bp && c == EOF)
	break;
      if (!*bp)
	continue;
      if ((prunen & 15) == 0)
	prunes = xrealloc(prunes, sizeof(*prunes) * (prunen + 16));
      p = prunes + prunen++;
      p->name = malloc(i + 1 - (bp - buf));
      if (!p->name)
	{
	  fprintf(stderr, "out of memory");
	  exit(1);
	}
      memcpy(p->name, bp, i + 1 - (bp - buf));
      i = 0;
    }
  fclose(fp);
}

int
is_pruned(char *n)
{
  int i;
  struct prune *p = prunes;

  for (i = 0; i < prunen; i++, p++)
    if (!strcmp(p->name, n))
      return 1;
  return 0;
}


int
is_unpatched(char *n, struct rpmlfile *files1, int nfiles1, struct rpmlfile *files2, int nfiles2, char *md5s)
{
  int i;
  unsigned char md5[16];

  for (i = 0; i < nfiles2; i++)
    if (!strcmp(n, files2[i].name))
      break;
  if (i == nfiles2)
    return 0;
  if (!(files2[i].fflags & FILE_UNPATCHED))
    return 0;
  for (i = 0; i < nfiles1; i++)
    if (!strcmp(n, files1[i].name))
      break;
  if (i == nfiles1)
    return 1;		/* should not happen... */
  parsemd5(md5s, md5);
  if (memcmp(md5, files1[i].md5, 16))
    return 1;		/* file content may be different */
  return 0;
}

void
put32(FILE *fp, unsigned int i)
{
  fputc(i >> 24, fp);
  fputc(i >> 16, fp);
  fputc(i >>  8, fp);
  fputc(i      , fp);
}

void addtocpio(unsigned char **cpiop, unsigned int *cpiol, unsigned char *d, int l)
{
  int cpl = *cpiol;
  if ((cpl & ~4095) == 0 || (cpl & ~4095) + l > 4096)
    *cpiop = xrealloc(*cpiop, (cpl + l + 4095) & ~4095);
  memcpy(*cpiop + cpl, d, l);
  *cpiol = cpl + l;
}

int
main(int argc, char **argv)
{
  char *rpmname;
  unsigned char rpmlead[96];
  struct rpmhead *h, *sigh;
  char *nevr;
  int filecnt;
  char **filenames, **filemd5s, **filelinktos;
  unsigned int *fileflags, *filemodes, *filerdevs, *filesizes, *fileverify;
  int i, fd, l, l2, l3;
  struct cfile *bfd;
  struct cpiophys cph;
  char *namebuf;
  int namebufl;
  char *np;
  char buf[4096];
  int c, skip;
  char *prunelist = 0;
  int cpiocnt = 0;
  int skipped_notfound = 0;
  int skipped_pruned = 0;
  int skipped_unpatched = 0;
  int skipped_badsize = 0;
  int skipped_fileflags = 0;
  int skipped_verifyflags = 0;
  int skipped_all = 0;
  int pinfo = 0;
  struct rpmlfile *files1 = 0;
  int nfiles1 = 0;
  char *nevr1 = 0;
  struct rpmlfile *files2 = 0;
  int nfiles2 = 0;
  char *nevr2 = 0;
  struct rpmhead *h2 = 0;
  char *seqfile = 0;
  MD5_CTX seqmd5;
  unsigned char seqmd5res[16];

  unsigned char *oldcpio = 0;
  unsigned int oldcpiolen = 0;
  unsigned char *newcpio = 0;
  unsigned int newcpiolen = 0;

  char *payformat;
  MD5_CTX fullmd5;
  unsigned char fullmd5res[16];
  unsigned int fullsize;

  struct cfile *newbz;
  int comp;

  struct instr *instr = 0;
  int instrlen = 0;
  unsigned char *addblk = 0;
  int addblklen = 0;

  int version = 2;

  while ((c = getopt(argc, argv, "V:pl:s:")) != -1)
    {
      switch (c)
	{
	case 'l':
	  prunelist = optarg;
	  break;
	case 's':
	  seqfile = optarg;
	  break;
	case 'V':
	  version = atoi(optarg);
	  break;
	case 'p':
	  pinfo = 1;
	  break;
	default:
	  fprintf(stderr, "usage: makedeltarpm [-l <file>] [-s seq] oldrpm newrpm deltarpm\n");
	  exit(1);
	}
    }
  if (prunelist)
    read_prunelist(prunelist);

  if (argc - optind != (pinfo ? 5 : 3))
    {
      fprintf(stderr, "usage: makedeltarpm [-l <file>] [-s seq] oldrpm newrpm deltarpm\n");
      exit(1);
    }
  if (version != 1 && version != 2)
    {
      fprintf(stderr, "illegal version\n");
      exit(1);
    }
  if (pinfo)
    {
      FILE *pfp;
      int pfd;

      pfp = fopen(argv[argc - 5], "r");
      if (!pfp)
	{
	  perror(argv[argc - 5]);
	  exit(1);
	}
      nevr1 = rpmlread(pfp, argv[argc - 5], 0, &files1, &nfiles1);
      fclose(pfp);
      pfd = open(argv[argc - 4], O_RDONLY);
      if (pfp < 0)
	{
	  perror(argv[argc - 4]);
	  exit(1);
	}
      if (read(pfd, rpmlead, 4) != 4)
	{
	  fprintf(stderr, "%s: not a rpm or rpmlist\n", argv[argc - 4]);
	  exit(1);
	}
      if (rpmlead[0] != 0xed || rpmlead[1] != 0xab || rpmlead[2] != 0xee || rpmlead[3] != 0xdb)
	{
	  pfp = fdopen(pfd, "r");
	  if (!pfp)
	    {
	      perror(argv[argc - 4]);
	      exit(1);
	    }
	  nevr2 = rpmlread(pfp, argv[argc - 4], 1, &files2, &nfiles2);
	  fclose(pfp);
	}
      else
	{
	  if (read(pfd, rpmlead + 4, 92) != 92 || rpmlead[4] != 0x03 || rpmlead[0x4e] != 0 || rpmlead[0x4f] != 5)
	    {
	      fprintf(stderr, "%s: not a v3 rpm or not new header styles\n", argv[argc - 4]);
	      exit(1);
	    }
	  h2 = readhead(pfd, 1);
	  if (!h2)
	    {
	      fprintf(stderr, "could not read signature header\n");
	      exit(1);
	    }
	  free(h2);
	  h2 = readhead(pfd, 0);
	  if (!h2)
	    {
	      fprintf(stderr, "could not read header\n");
	      exit(1);
	    }
	  close(pfd);
	  nevr2 = headtofiles(h2, &files2, &nfiles2);
	}
    }

  rpmMD5Init(&seqmd5);

  rpmname = argv[argc - 3];
  if ((fd = open(rpmname, O_RDONLY)) < 0)
    {
      perror(rpmname);
      exit(1);
    }
  if (read(fd, rpmlead, 96) != 96 || rpmlead[0] != 0xed || rpmlead[1] != 0xab || rpmlead[2] != 0xee || rpmlead[3] != 0xdb)
    {
      fprintf(stderr, "%s: not a rpm\n", rpmname);
      exit(1);
    }
  if (rpmlead[4] != 0x03 || rpmlead[0x4e] != 0 || rpmlead[0x4f] != 5)
    {
      fprintf(stderr, "%s: not a v3 rpm or not new header styles\n", rpmname);
      exit(1);
    }
  sigh = readhead(fd, 1);
  if (!sigh)
    {
      fprintf(stderr, "could not read signature header\n");
      exit(1);
    }
  sigh = xfree(sigh);
  h = readhead(fd, 0);
  if (!h)
    {
      fprintf(stderr, "could not read header\n");
      exit(1);
    }
  nevr = headtonevr(h);
  if (pinfo)
    {
      if (strcmp(nevr, nevr1) != 0)
	{
	  fprintf(stderr, "pinfo rpmlist1 (%s) does not match rpm (%s)\n", nevr1, nevr);
	  exit(1);
	}
      if (strcmp(nevr, nevr2) != 0)
	{
	  fprintf(stderr, "pinfo rpmlist2 (%s) does not match rpm (%s)\n", nevr2, nevr);
	  exit(1);
	}
    }
  filenames = headexpandfilelist(h, &filecnt);
  if (!filenames || filecnt == 0)
    {
      fprintf(stderr, "rpm contains no files\n");
      exit(1);
    }
  fileflags = headint32(h, TAG_FILEFLAGS, (int *)0);
  filemd5s = headstringarray(h, TAG_FILEMD5S, (int *)0);
  filerdevs = headint16(h, TAG_FILERDEVS, (int *)0);
  filesizes = headint32(h, TAG_FILESIZES, (int *)0);
  filemodes = headint16(h, TAG_FILEMODES, (int *)0);
  fileverify = headint32(h, TAG_FILEVERIFY, (int *)0);
  filelinktos = headstringarray(h, TAG_FILELINKTOS, (int *)0);

  bfd = cfile_open(CFILE_OPEN_RD, fd, 0, CFILE_COMP_XX, CFILE_LEN_UNLIMITED, 0, 0);
  if (!bfd)
    {
      fprintf(stderr, "payload open failed\n");
      exit(1);
    }
  namebufl = 1;
  namebuf = xmalloc(namebufl);
  for (;;)
    {
      unsigned int size, nsize, lsize, nlink, rdev;

      if (bfd->read(bfd, &cph, sizeof(cph)) != sizeof(cph))
	{
	  fprintf(stderr, "payload read failed (header)\n");
	  exit(1);
	}
      if (memcmp(cph.magic, "070701", 6))
	{
	  fprintf(stderr, "bad cpio archive\n");
	  exit(1);
	}
      size = cpion(cph.filesize);
      nsize = cpion(cph.namesize);
      nlink = cpion(cph.nlink);
      nsize += (4 - ((nsize + 2) & 3)) & 3;
      if (nsize > namebufl)
	{
	  namebuf = xrealloc(namebuf, nsize);
	  namebufl = nsize;
	}
      if (bfd->read(bfd, namebuf, nsize) != nsize)
	{
	  fprintf(stderr, "payload read failed (name)\n");
	  exit(1);
	}
      namebuf[nsize - 1] = 0;
      if (!strcmp(namebuf, "TRAILER!!!"))
	break;
      cpiocnt++;
      np = namebuf;
      if (*np == '.' && np[1] == '/')
	np += 2;
      skip = 1;
      /* look it up in the header */
      for (i = 0; i < filecnt; i++)
        if (!strcmp(filenames[i] + (filenames[i][0] == '/' ? 1 : 0), np))
	  break;
      rdev = lsize = 0;
      if (i == filecnt)
	{
          fprintf(stderr, "%s not found in rpm header\n", np);
	  skipped_notfound++;
	}
      else if (prunes && is_pruned(np))
	{
	  fprintf(stderr, "skipping %s: pruned\n", np);
	  skipped_pruned++;
	}
      else if (pinfo && S_ISREG(filemodes[i]) && is_unpatched(np, files1, nfiles1, files2, nfiles2, filemd5s[i]))
	{
	  fprintf(stderr, "skipping %s: unpatched but different\n", np);
	  skipped_unpatched++;
	}
      else if (S_ISREG(filemodes[i]))
	{
	  if (size != filesizes[i])
	    {
	      fprintf(stderr, "skipping %s: size missmatch\n", np);
	      skipped_badsize++;
	    }
	  else if ((fileflags[i] & (FILE_CONFIG|FILE_MISSINGOK|FILE_GHOST)) != 0)
	    {
	      fprintf(stderr, "skipping %s: bad file flags\n", np);
	      skipped_fileflags++;
	    }
	  else if ((fileverify[i] & (VERIFY_MD5|VERIFY_FILESIZE)) != (VERIFY_MD5|VERIFY_FILESIZE))
	    {
	      fprintf(stderr, "skipping %s: bad verify flags %x\n", np, fileverify[i]);
	      skipped_verifyflags++;
	    }
	  else
	    {
	      fprintf(stderr, "USING FILE %s\n", np);
	      lsize = size;
	      skip = 0;
	    }
	}
      else if (S_ISLNK(filemodes[i]))
	{
	  fprintf(stderr, "USING SYMLINK %s\n", np);
	  lsize = strlen(filelinktos[i]);
	  skip = 0;
	}
      else
	{
	  fprintf(stderr, "USING ELEMENT %s\n", np);
	  if (S_ISBLK(filemodes[i]) || S_ISCHR(filemodes[i]))
	    rdev = filerdevs[i];
	  skip = 0;
	}
      if (!skip)
	{
	  unsigned char cpiobuf[110 + 3];
	  int ns = strlen(np);
	  sprintf((char *)cpiobuf, "07070100000000%08x00000000000000000000000100000000%08x0000000000000000%08x%08x%08x00000000./", filemodes[i], lsize, devmajor(rdev), devminor(rdev), ns + 3);
	  addtocpio(&oldcpio, &oldcpiolen, cpiobuf, 112);
	  addtocpio(&oldcpio, &oldcpiolen, (unsigned char *)np, ns + 1);
	  ns += 3 - 2;
	  for (; ns & 3 ; ns++)
	    addtocpio(&oldcpio, &oldcpiolen, (unsigned char *)"", 1);
	  rpmMD5Update(&seqmd5, (unsigned char *)np, strlen(np) + 1);
	  rpmMD5Update32(&seqmd5, filemodes[i]);
	  rpmMD5Update32(&seqmd5, lsize);
	  rpmMD5Update32(&seqmd5, rdev);
	  if (S_ISLNK(filemodes[i]))
	    {
	      addtocpio(&oldcpio, &oldcpiolen, (unsigned char *)filelinktos[i], lsize);
	      for (; lsize & 3 ; lsize++)
	        addtocpio(&oldcpio, &oldcpiolen, (unsigned char *)"", 1);
	      skip = 1;
	      rpmMD5Update(&seqmd5, (unsigned char *)filelinktos[i], strlen(filelinktos[i]) + 1);
	    }
	  if (S_ISREG(filemodes[i]) && lsize)
	    {
	      unsigned char fmd5[16];
	      parsemd5(filemd5s[i], fmd5);
	      rpmMD5Update(&seqmd5, fmd5, 16);
	    }
          addtoseq(i);
	}
      else
	skipped_all++;
      l = l2 = size;
      while (l > 0)
	{
	  l3 = l > sizeof(buf) ? sizeof(buf) : l;
	  if (bfd->read(bfd, buf, l3) != l3)
	    {
	      fprintf(stderr, "payload read failed (data)\n");
	      exit(1);
	    }
	  if (!skip)
	    addtocpio(&oldcpio, &oldcpiolen, (unsigned char *)buf, l3);
	  l -= l3;
	}
      if ((l2 & 3) != 0)
	{
	  l2 = 4 - (l2 & 3);
	  if (bfd->read(bfd, buf, l2) != l2)
	    {
	      fprintf(stderr, "payload read failed (pad)\n");
	      exit(1);
	    }
	  if (!skip)
	    addtocpio(&oldcpio, &oldcpiolen, (unsigned char *)"\0\0\0", l2);
	}
    }
  namebuf = xfree(namebuf);
  namebufl = 0;
  bfd->close(bfd);
  addtocpio(&oldcpio, &oldcpiolen, (unsigned char *)"07070100000000000000000000000000000000000000010000000000000000000000000000000000000000000000000000000b00000000TRAILER!!!\0\0\0\0", 124);
  close(fd);

 
  fprintf(stderr, "files used:    %4d/%d = %.1f%%\n", (cpiocnt - skipped_all), cpiocnt, (cpiocnt - skipped_all) * 100. / (cpiocnt ? cpiocnt : 1));
  fprintf(stderr, "files skipped: %4d/%d = %.1f%%\n", skipped_all, cpiocnt, skipped_all * 100. / (cpiocnt ? cpiocnt : 1));
  if (skipped_notfound)
    fprintf(stderr, "  not found:    %4d/%d = %.1f%%\n", skipped_notfound, skipped_all, skipped_notfound * 100. / skipped_all);
  if (skipped_pruned)
    fprintf(stderr, "  pruned:       %4d/%d = %.1f%%\n", skipped_pruned, skipped_all, skipped_pruned * 100. / skipped_all);
  if (skipped_unpatched)
    fprintf(stderr, "  unpatched:    %4d/%d = %.1f%%\n", skipped_unpatched, skipped_all, skipped_unpatched * 100. / skipped_all);
  if (skipped_badsize)
    fprintf(stderr, "  bad size:     %4d/%d = %.1f%%\n", skipped_badsize, skipped_all, skipped_badsize * 100. / skipped_all);
  if (skipped_fileflags)
    fprintf(stderr, "  file flags:   %4d/%d = %.1f%%\n", skipped_fileflags, skipped_all, skipped_fileflags * 100. / skipped_all);
  if (skipped_verifyflags)
    fprintf(stderr, "  verify flags: %4d/%d = %.1f%%\n", skipped_verifyflags, skipped_all, skipped_verifyflags * 100. / skipped_all);
  addtoseq(-1);

  fprintf(stderr, "sequence: ");
  for (i = 0; i < (seqp + 1) / 2; i++)
    fprintf(stderr, "%02x", seq[i]);
  fprintf(stderr, "\n");

  rpmMD5Final(seqmd5res, &seqmd5);

  fileflags = xfree(fileflags);
  filemd5s = xfree(filemd5s);
  filerdevs = xfree(filerdevs);
  filesizes = xfree(filesizes);
  filemodes = xfree(filemodes);
  fileverify = xfree(fileverify);
  filelinktos = xfree(filelinktos);
  filenames = xfree(filenames);
  h = xfree(h);

/****************************************************************/

  /* read in new rpm */
  fprintf(stderr, "reading new rpm...\n");
  rpmname = argv[argc - 2];
  if ((fd = open(rpmname, O_RDONLY)) < 0)
    {
      perror(rpmname);
      exit(1);
    }
  if (read(fd, rpmlead, 96) != 96 || rpmlead[0] != 0xed || rpmlead[1] != 0xab ||
 rpmlead[2] != 0xee || rpmlead[3] != 0xdb)
    {
      fprintf(stderr, "%s: not a rpm\n", rpmname);
      exit(1);
    }
  if (rpmlead[4] != 0x03 || rpmlead[0x4e] != 0 || rpmlead[0x4f] != 5)
    {
      fprintf(stderr, "%s: not a v3 rpm or not new header styles\n", rpmname);
      exit(1);
    }
  sigh = readhead(fd, 1);
  if (!sigh)
    {
      fprintf(stderr, "could not read signature header\n");
      exit(1);
    }
  h = readhead(fd, 0);
  if (!h)
    {
      fprintf(stderr, "could not read header\n");
      exit(1);
    }
  payformat = headstring(h, TAG_PAYLOADFORMAT);
  if (!payformat || strcmp(payformat, "cpio") != 0)
    {
      fprintf(stderr, "payload format is not cpio\n");
      exit(1);
    }
  rpmMD5Init(&fullmd5);
  rpmMD5Update(&fullmd5, rpmlead, 96);
  rpmMD5Update(&fullmd5, sigh->intro, 16);
  rpmMD5Update(&fullmd5, sigh->data, sigh->cnt * 16 + sigh->dcnt);
  rpmMD5Update(&fullmd5, h->intro, 16);
  rpmMD5Update(&fullmd5, h->data, h->cnt * 16 + h->dcnt);
  fullsize = 96 + 16 + sigh->cnt * 16 + sigh->dcnt + 16 + h->cnt * 16 + h->dcnt;
  newbz = cfile_open(CFILE_OPEN_RD, fd, 0, CFILE_COMP_XX, CFILE_LEN_UNLIMITED, (cfile_ctxup)rpmMD5Update, &fullmd5);
  if (!newbz)
    {
      fprintf(stderr, "payload open failed\n");
      exit(1);
    }
  if (cfile_detect_rsync(newbz))
    {
      fprintf(stderr, "detect_rsync failed\n");
      exit(1);
    }
  comp = newbz->comp;
  while ((l = newbz->read(newbz, buf, sizeof(buf))) > 0)
    addtocpio(&newcpio, &newcpiolen, (unsigned char *)buf, l);
  if (l < 0)
    {
      fprintf(stderr, "payload read failed\n");
      exit(1);
    }
  fullsize += newbz->bytes;
  if (newbz->close(newbz))
    {
      fprintf(stderr, "junk at end of payload\n");
      exit(1);
    }
  close(fd);
  rpmMD5Final(fullmd5res, &fullmd5);

  fprintf(stderr, "creating diff...\n");
  mkdiff(DELTAMODE_HASH, oldcpio, oldcpiolen, newcpio, newcpiolen, &instr, &instrlen, (unsigned char **)0, (int *)0, &addblk, &addblklen, (unsigned char **)0, (int *)0);
  fprintf(stderr, "writing delta rpm...\n");

  writedeltarpm(argv[argc - 1], rpmlead, sigh, h, instr, instrlen, addblk, addblklen, newcpio, oldcpiolen, payformat, fullmd5res, fullsize, seqmd5res, seq, (seqp + 1) / 2, nevr, comp, version);

  addblk = xfree(addblk);
  addblklen = 0;
  instr = xfree(instr);
  instrlen = 0;
  oldcpio = xfree(oldcpio);
  newcpio = xfree(newcpio);
  sigh = xfree(sigh);
  h = xfree(h);

  if (seqfile)
    {
      FILE *ifp;
      if ((ifp = fopen(seqfile, "w")) == 0)
	{
	  perror(seqfile);
	  exit(1);
	}
      fprintf(ifp, "%s-", nevr);
      for (i = 0; i < 16; i++)
        fprintf(ifp, "%02x", seqmd5res[i]);
      for (i = 0; i < (seqp + 1) / 2; i++)
        fprintf(ifp, "%02x", seq[i]);
      fprintf(ifp, "\n");
      if (fclose(ifp))
	{
	  perror("fclose seqfile");
	  exit(1);
	}
    }
  nevr = xfree(nevr);
  seq = xfree(seq);
  exit(0);
}
