/*
 * suse-blinux
 * driver for all Papenmeier devices
 * Author: marco Skambraks <marco@suse.de>
 * SuSE Linux AG, Nuernberg
 * http://www.blinux.suse.de
 *
 * special thanks to the brltty team 
 * Nicolas Pitre <nico@cam.org>
 * Stphane Doyon <s.doyon@videotron.ca>
 * Nikhil Nair <nn201@cus.cam.ac.uk>
 *
 * This is free software, placed under the terms of the
 * GNU General Public License, as published by the Free Software
 * Foundation.  Please see the file COPYING for details.
 */

/* 
 * This Driver was written as a project in the
 * (Technical High School, Department for electrical engineering, Vienna, Austria)
 * by:
 *   Tibor Becker
 *   Michael Burger
 *   Herbert Gruber
 *   Heimo Schn
 *   Teacher: August Hrandl <hoerandl@elina.htlw1.ac.at>
 *
 * modified/rebuild by Marco Skambraks <marco@suse.de>
 * SuSE Linux AG, Nuernberg
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <termios.h>
#include <string.h>

#include "brlconf.h"
#include "../brl.h"
#include "../brl_driver.h"

#include "../scr.h"
#include "../misc.h"
#include "../sbllog.h"

/* HACK - send all data twice - HACK */
/* see README for details */
/* #define SEND_TWICE_HACK */

#define CMD_ERR 0
char DefDev[] = BRLDEV;		/* default braille device */
int brl_fd = 0;			/* file descriptor for Braille display */
struct termios oldtio;		/* old terminal settings */

static unsigned int brl_dbg=0;

static char *prevline;
static char *prev;
void allocmem ();
void init_table ();

/* special table for papenmeier */
#define B1 128
#define B2 64
#define B3 32
#define B4 16
#define B5 8
#define B6 4
#define B7 2
#define B8 1

char pm_ones[11] = { B1 + B5 + B4, B2, B2 + B5,
  B2 + B1, B2 + B1 + B4, B2 + B4,
  B2 + B5 + B1, B2 + B5 + B4 + B1, B2 + B5 + B4,
  B5 + B1, B1 + B2 + B4 + B5
};

char pm_tens[11] = { B8 + B6 + B3, B7, B7 + B8,
  B7 + B3, B7 + B3 + B6, B7 + B6,
  B7 + B8 + B3, B7 + B8 + B3 + B6, B7 + B8 + B6,
  B8 + B3, B3 + B6 + B7 + B8
};

/* create bits for number 0..99 - special for papenmeier */
int pm_num (int x)
{
/*  return pm_tens[(x / 10) % 10] | pm_ones[x % 10];  */
if(x) return 255;
  return 255;
}

/* status cell   tens: line number    ones: no/all bits set */
int pm_stat (int line, int on)
{
  if (on)
    return pm_tens[line % 10] | pm_ones[10];
  else
    return pm_tens[line];
}

void brl_debug (unsigned int dbg)
{
brl_dbg=dbg;
}

void initbrlerror (brldim * brl)
{
  printf ("\nInitbrl: failure at open\n");
  if (brl->disp)
    free (brl->disp);
  brl->x = -1;
}

void initbrl (brldim * brl, const char *dev)
{
  brldim res;			/* return result */
int status;
  struct termios tiodata;	/* new terminal settings */

  res.x = BRLCOLS;		/* initialise size of display */
  res.y = BRLROWS;
  res.disp = NULL;		/* clear pointers */

  /* Now open the Braille display device for random access */
  if (!dev)
    dev = DefDev;

  brl_fd = open (dev, O_RDWR | O_NONBLOCK | O_NOCTTY);
  if (brl_fd < 0)
   {
     initbrlerror (brl);
     return;
   }

  tiodata.c_cflag = (CLOCAL | CREAD | CS8);
  /* Ignore bytes with parity errors and make terminal raw and dumb */
  tiodata.c_iflag = IGNPAR;
//  tiodata.c_oflag = 0;

  tiodata.c_lflag = 0;
  tiodata.c_cc[VMIN] = 0;
  tiodata.c_cc[VTIME] = 0;

  if (cfsetispeed (&tiodata, B0) ||
      cfsetospeed (&tiodata, B0) ||
      tcsetattr (brl_fd, TCSANOW, &tiodata) ||
      tcflush (brl_fd, TCIOFLUSH) ||
      cfsetispeed (&tiodata, BAUDRATE) ||
      cfsetospeed (&tiodata, BAUDRATE)
      || tcsetattr (brl_fd, TCSANOW, &tiodata))
   {
     initbrlerror (brl);
     return;
   }
  tcflush (brl_fd, TCIOFLUSH);
/* get status of inteface */
ioctl(brl_fd,TIOCMGET,&status);
/* clear dtr-line */
status &=~TIOCM_DTR;
/* set new status */
ioctl(brl_fd,TIOCMSET,&status);
  /*      Pause to let them take effect */
  sleep (1);
  /* Allocate space for buffers */
  res.disp = (unsigned char *) malloc (res.x * res.y);
  if (!res.disp)
    initbrlerror (&res);

//  init_table ();

  res.brl_fd = brl_fd;
  *brl = res;
  return;
}

void closebrl (brldim * brl)
{
  free (brl->disp);
  tcsetattr (brl_fd, TCSANOW, &oldtio);	/* restore terminal settings */
  close (brl_fd);
  brl->brl_fd = -1;
  
}

struct brlinfo identbrl (const char *name, const char *dev)
{
  struct brlinfo retinfo;
  int devcnt;

  devcnt = sizeof (fhpdevs) / sizeof (struct brlinfo);
  retinfo.cols = 0;
  for (devnr = 0; devnr < devcnt; devnr++)
    if (!strcmp (name, fhpdevs[devnr].name))
     {
       allocmem ();
       printf (" %s on %s\n", FULLNAME, dev);
       retinfo = fhpdevs[devnr];
       break;
     }

  return retinfo;
}

void write_to_braille (int offset, int size, const char *data)
{
  unsigned int i = 0;
  unsigned char BrlHead[] = { cSTX,
    cIdSend,
    0x50, 0x50, 0, 0, 0, 0
  };
  unsigned char brldata[200] = "";
  unsigned char BrlTrail[] = { 0, 0, 0, 0, cETX };

  BrlHead[2] = 0x52;
  BrlHead[3] = 0x5c;

  for (i = 0; i < (unsigned int)(BRLCOLS * 2); i += 2)
   {
     brldata[i] = 0x30 + (data[i / 2] >> 4);
     brldata[i + 1] = 0x30 + (data[i / 2] & 0x0f);
   }
  write (brl_fd, BrlHead, sizeof (BrlHead));
  write (brl_fd, brldata, BRLCOLS * 2);
  write (brl_fd, BrlTrail, sizeof (BrlTrail));
 if(offset || size) return;
 }

void init_table ()
{
  char line[BRLCOLS];
  char spalte[PMSC];
  int i;

  // dont use the internal table for the status column 
  for (i = 0; i < PMSC; i++)
    spalte[i] = 1;		/* 1 = no table */
  write_to_braille (offsetTable + offsetVertical, PMSC, spalte);

  // dont use the internal table for the line
  for (i = 0; i < BRLCOLS; i++)
    line[i] = 1;		// 1 = no table
  write_to_braille (offsetTable + offsetHorizontal, BRLCOLS, line);
}

void setbrlstat (const unsigned char *s)
{
  static char str[30];
  const char highbits[10] = { B2 + B4 + B5, B1, B1 + B2, B1 + B4,
    B1 + B4 + B5, B1 + B5, B1 + B2 + B4, B1 + B2 + B4 + B5, B1 + B2 + B5,
    B2 + B4
  };
  const char lowbits[10] = { B6 + B7 + B8, B3, B3 + B7, B3 + B6, B3 + B6 + B8,
    B3 + B8, B3 + B6 + B7, B3 + B6 + B7 + B8, B3 + B7 + B8, B6 + B7
  };

  if (PMSC == 0)
    return;
  if (memcmp (s, prevline, PMSC) != 0)
   {
     memcpy (prevline, s, PMSC);
     memcpy (str, s, PMSC);
     if (PMSC != 2)
      {
	str[0] = pm_tens[(int) str[0] / 10] | pm_ones[(int) str[0] % 10];
	str[1] = pm_tens[(int) str[1] / 10] | pm_ones[(int) str[1] % 10];
	str[2] = pm_tens[(int) str[2] / 10] | pm_ones[(int) str[2] % 10];
      }
     else
      {
	int left = str[0], right = str[1];

	str[0] = highbits[left / 10] | lowbits[right / 10];
	str[1] = highbits[left % 10] | lowbits[right % 10];
      }

     write_to_braille (offsetVertical, PMSC, str);
   }
}

/* strange mapping for cursor-routing */
unsigned char csr_map[] = {
  10, 13, 12, 15, 14, 17, 16, 19, 18, 21, 20, 23, 22, 25, 24, 27, 26, 29, 28,
    31, 30, 33, 32, 35, 34, 37,
  36, 39, 38, 41, 40, 43, 42, 45, 44, 47, 0
};

/*
 * change bits for the papenmeier terminal
 *                             1 2           1 4
 * dot number -> bit number    3 4   we get  2 5 
 *                             5 6           3 6
 *                             7 8           7 8
 */
unsigned char change_bits[] = {
  0x0, 0x80, 0x10, 0x90, 0x40, 0xc0, 0x50, 0xd0, 0x8, 0x88, 0x18, 0x98, 0x48,
    0xc8, 0x58, 0xd8, 0x20, 0xa0, 0x30, 0xb0, 0x60, 0xe0, 0x70, 0xf0, 0x28,
    0xa8, 0x38, 0xb8, 0x68, 0xe8, 0x78, 0xf8, 0x4, 0x84, 0x14, 0x94, 0x44,
    0xc4, 0x54, 0xd4, 0xc, 0x8c, 0x1c, 0x9c, 0x4c, 0xcc, 0x5c, 0xdc, 0x24,
    0xa4, 0x34, 0xb4, 0x64, 0xe4, 0x74, 0xf4, 0x2c, 0xac, 0x3c, 0xbc, 0x6c,
    0xec, 0x7c, 0xfc, 0x2, 0x82, 0x12, 0x92, 0x42, 0xc2, 0x52, 0xd2, 0xa,
    0x8a, 0x1a, 0x9a, 0x4a, 0xca, 0x5a, 0xda, 0x22, 0xa2, 0x32, 0xb2, 0x62,
    0xe2, 0x72, 0xf2, 0x2a, 0xaa, 0x3a, 0xba, 0x6a, 0xea, 0x7a, 0xfa, 0x6,
    0x86, 0x16, 0x96, 0x46, 0xc6, 0x56, 0xd6, 0xe, 0x8e, 0x1e, 0x9e, 0x4e,
    0xce, 0x5e, 0xde, 0x26, 0xa6, 0x36, 0xb6, 0x66, 0xe6, 0x76, 0xf6, 0x2e,
    0xae, 0x3e, 0xbe, 0x6e, 0xee, 0x7e, 0xfe, 0x1, 0x81, 0x11, 0x91, 0x41,
    0xc1, 0x51, 0xd1, 0x9, 0x89, 0x19, 0x99, 0x49, 0xc9, 0x59, 0xd9, 0x21,
    0xa1, 0x31, 0xb1, 0x61, 0xe1, 0x71, 0xf1, 0x29, 0xa9, 0x39, 0xb9, 0x69,
    0xe9, 0x79, 0xf9, 0x5, 0x85, 0x15, 0x95, 0x45, 0xc5, 0x55, 0xd5, 0xd,
    0x8d, 0x1d, 0x9d, 0x4d, 0xcd, 0x5d, 0xdd, 0x25, 0xa5, 0x35, 0xb5, 0x65,
    0xe5, 0x75, 0xf5, 0x2d, 0xad, 0x3d, 0xbd, 0x6d, 0xed, 0x7d, 0xfd, 0x3,
    0x83, 0x13, 0x93, 0x43, 0xc3, 0x53, 0xd3, 0xb, 0x8b, 0x1b, 0x9b, 0x4b,
    0xcb, 0x5b, 0xdb, 0x23, 0xa3, 0x33, 0xb3, 0x63, 0xe3, 0x73, 0xf3, 0x2b,
    0xab, 0x3b, 0xbb, 0x6b, 0xeb, 0x7b, 0xfb, 0x7, 0x87, 0x17, 0x97, 0x47,
    0xc7, 0x57, 0xd7, 0xf, 0x8f, 0x1f, 0x9f, 0x4f, 0xcf, 0x5f, 0xdf, 0x27,
    0xa7, 0x37, 0xb7, 0x67, 0xe7, 0x77, 0xf7, 0x2f, 0xaf, 0x3f, 0xbf, 0x6f,
    0xef, 0x7f, 0xff,
};

void writebrl (brldim * brl)
{
  int i;

  for (i = 0; i < BRLCOLS; i++)
    brl->disp[i] = change_bits[brl->disp[i]];

  if (memcmp (prev, brl->disp, BRLCOLS) != 0)
   {
     memcpy (prev, brl->disp, BRLCOLS);
     write_to_braille (offsetHorizontal, BRLCOLS, prev);
   }
}

/* ------------------------------------------------------------ */

/* some makros */
/* read byte */
#define READ(OFFS) \
  if (read(brl_fd,buf+OFFS,1) != 1) \
      return EOF;                   \

/* read byte and check value */
#define READ_CHK(OFFS, VAL)  \
    { READ(OFFS);            \
      if (buf[OFFS] != VAL)  \
        return CMD_ERR;      \
    }

int readbrl (int *xx)
{
  static int lastkey = 0;
  unsigned char buf[100];
  unsigned int i, l;
  static unsigned int l_key=0,r_key=0;
  static int eab_l_r=0,eab_up_down=0;
  

  READ_CHK (0, cSTX);		/* STX - Start */
  usleep(5);
  READ_CHK (1, cIdReceive);	/* 'K' */
  usleep(5);
  READ (2);			/* code - 2 bytes */
  usleep(5);
  READ (3);
  usleep(5);
  l = ((buf[2] - 0x50) * 0xf) + (buf[3] - 0x50);	/* Data count */
  l = (l * 2) + 5;		// the display send double-bytes +5 header+etx
  if (l > sizeof (buf))
   {
     sbl_log ("return lenght is biger than buffer-size\n");
     return EOF;
   }
  for (i = 4; i < l; i++)
  {
  usleep(5);
    READ (i);			/* Data */
    }

  if (buf[l - 1] != cETX)	/* ETX - End */
   {
     sbl_log ("no ETX found at end of data-stream %d\n", l - 1);
//    return EOF;
   }






  if (buf[8] != 0x30 && !eab_l_r)
  {
  eab_l_r=1;
    sbl_log ("eab left/right:%d ", buf[8] - 0x30);
  switch (buf[8])
   {
   case '2':
     *xx = 1;
     return lastkey = 1;
     break;
   case '8':
     *xx = 1;
     return lastkey = 2;
     break;
   case '1':
     *xx = 1;
     return lastkey = 5;
     break;
   case '4':
     *xx = 1;
     return lastkey = 6;
     break;
   default:
     *xx = 0;
   }
   
   } /* eab left right */

  if (buf[9] != 0x30 && !eab_up_down)
  {
  eab_up_down=1;
    sbl_log ("eab up/down:%d ", buf[9] - 0x30);
  switch (buf[9])
   {
   case '2':
     *xx = 1;
     return lastkey = 3;
     break;
   case '1':
     *xx = 1;
     return lastkey = 4;
     break;
   case '4':
     *xx = 1;
     return lastkey = 7;
     break;
   case '8':
     *xx = 1;
     return lastkey = 8;
     break;
   default:
     *xx = 0;
   }
   
   } /* up down */

  if (buf[11] != 0x30 && !l_key)
  {
  l_key=1;
    sbl_log ("left-key:%d ", buf[11] - 0x30);
  switch (buf[11])
   {
   case '1':
     *xx = 1;
     return lastkey = 11;
     break;
   case '2':
     *xx = 1;
     return lastkey = 12;
     break;
   default:
     *xx = 0;
   }
   
   } /* left key */

  if (buf[l - 3] != 0x30 && !r_key)
  {
  r_key=1;
  
    sbl_log ("right-key:%d ", buf[l - 3] - 0x30);
  switch (buf[l - 3])
   {
   case '1':
     *xx = 1;
     return lastkey = 13;
     break;
   case '2':
     *xx = 1;
     return lastkey = 14;
     break;
   default:
     *xx = 0;
   }
   
   } /* right key */

/* check routing keys */
  for (i = 0; csr_map[i] && (csr_map[i] < l); i++)
   {
if(csr_map[i]==(l-3)) continue;
     switch (buf[csr_map[i]])
      {
      case '1':
	*xx = 1;
	lastkey = 100 + (i * 2);
	sbl_log ("csrrout=%d ", lastkey);
	return lastkey;
	break;
      case '4':
	*xx = 1;
	lastkey = 101 + (i * 2);
	sbl_log ("csrrout=%d ", lastkey);
	return lastkey;
	break;
      }

   }				// for

  *xx = 0;
  {
    int lasttmp = lastkey;

    lastkey = EOF;
    l_key=r_key=0;
    eab_up_down=eab_l_r=0;
    return lasttmp;
  }

}

void allocmem ()
{
  prevline = (char *) malloc (PMSC);
  prev = (char *) malloc (BRLCOLS + 1);
}
