#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/termios.h>
#include <sys/ioctl.h>
#include <string.h>
#include <errno.h>
#include "../sbllog.h"
#include "brlconf.h"
#include "../brl.h"
/*
#include "../misc.h"
#include "../scr.h"*/
#include "../brl_driver.h"

/* extern int DEBUG; */
struct brlinfo baumdevs[] = {
  {"vario40", "Baum Vario40", 40, 0, 0, 0, B19200},
  {"vario80", "Baum Vario80", 80, 4, 0, 0, B19200},
  {"dm80p", "Baum dm80Plus", 80, 4, 0, 0, B19200}
};

static int devnr;
static unsigned int brl_dbg = 0;
/* Braille display parameters that do not change */
#define DM80P 2
#define BRLROWS 1		/* only one row on braille display */

#define V80BRLCOLS 80		/* The Vario 80 has 80 cells (+4 status) */
#define V80NCELLS 84
#define V80TSPDATACNT 11
#define V40BRLCOLS 40
#define V40NCELLS 40
#define V40TSPDATACNT 5
#define BAUDRATE B19200		/* But both run at 19k2 */

static int brlcols;
static int ncells;

#ifdef USE_PING
/* A query is sent if we don't get any keys in a certain time, to detect
   if the display was turned off. */
/* We record the time at which the last ping reply was received,
   and the time at which the last ping (query) was sent. */
static struct timeval last_ping, last_ping_sent;

/* that many pings are sent, that many chances to reply */
#define PING_MAXNQUERY 2
static int pings;		/* counts number of pings sent since last reply */

/* how long we wait for a reply */
#define PING_REPLY_DELAY 300
#endif

/* for routing keys */

/* Definitions to avoid typematic repetitions of function keys other
   than movement keys */
#define NONREPEAT_TIMEOUT 300
#define READBRL_SKIP_TIME 300
static int lastcmd = EOF;
static char typing_mode;	/* For Vario80 - holds the mode of Command keys */
static char DotsTable[256] = {
  0x00, 0x01, 0x08, 0x09, 0x02, 0x03, 0x0A, 0x0B,
  0x10, 0x11, 0x18, 0x19, 0x12, 0x13, 0x1A, 0x1B,
  0x04, 0x05, 0x0C, 0x0D, 0x06, 0x07, 0x0E, 0x0F,
  0x14, 0x15, 0x1C, 0x1D, 0x16, 0x17, 0x1E, 0x1F,
  0x20, 0x21, 0x28, 0x29, 0x22, 0x23, 0x2A, 0x2B,
  0x30, 0x31, 0x38, 0x39, 0x32, 0x33, 0x3A, 0x3B,
  0x24, 0x25, 0x2C, 0x2D, 0x26, 0x27, 0x2E, 0x2F,
  0x34, 0x35, 0x3C, 0x3D, 0x36, 0x37, 0x3E, 0x3F,
  0x40, 0x41, 0x48, 0x49, 0x42, 0x43, 0x4A, 0x4B,
  0x50, 0x51, 0x58, 0x59, 0x52, 0x53, 0x5A, 0x5B,
  0x44, 0x45, 0x4C, 0x4D, 0x46, 0x47, 0x4E, 0x4F,
  0x54, 0x55, 0x5C, 0x5D, 0x56, 0x57, 0x5E, 0x5F,
  0x60, 0x61, 0x68, 0x69, 0x62, 0x63, 0x6A, 0x6B,
  0x70, 0x71, 0x78, 0x79, 0x72, 0x73, 0x7A, 0x7B,
  0x64, 0x65, 0x6C, 0x6D, 0x66, 0x67, 0x6E, 0x6F,
  0x74, 0x75, 0x7C, 0x7D, 0x76, 0x77, 0x7E, 0x7F,
  0x80, 0x81, 0x88, 0x89, 0x82, 0x83, 0x8A, 0x8B,
  0x90, 0x91, 0x98, 0x99, 0x92, 0x93, 0x9A, 0x9B,
  0x84, 0x85, 0x8C, 0x8D, 0x86, 0x87, 0x8E, 0x8F,
  0x94, 0x95, 0x9C, 0x9D, 0x96, 0x97, 0x9E, 0x9F,
  0xA0, 0xA1, 0xA8, 0xA9, 0xA2, 0xA3, 0xAA, 0xAB,
  0xB0, 0xB1, 0xB8, 0xB9, 0xB2, 0xB3, 0xBA, 0xBB,
  0xA4, 0xA5, 0xAC, 0xAD, 0xA6, 0xA7, 0xAE, 0xAF,
  0xB4, 0xB5, 0xBC, 0xBD, 0xB6, 0xB7, 0xBE, 0xBF,
  0xC0, 0xC1, 0xC8, 0xC9, 0xC2, 0xC3, 0xCA, 0xCB,
  0xD0, 0xD1, 0xD8, 0xD9, 0xD2, 0xD3, 0xDA, 0xDB,
  0xC4, 0xC5, 0xCC, 0xCD, 0xC6, 0xC7, 0xCE, 0xCF,
  0xD4, 0xD5, 0xDC, 0xDD, 0xD6, 0xD7, 0xDE, 0xDF,
  0xE0, 0xE1, 0xE8, 0xE9, 0xE2, 0xE3, 0xEA, 0xEB,
  0xF0, 0xF1, 0xF8, 0xF9, 0xF2, 0xF3, 0xFA, 0xFB,
  0xE4, 0xE5, 0xEC, 0xED, 0xE6, 0xE7, 0xEE, 0xEF,
  0xF4, 0xF5, 0xFC, 0xFD, 0xF6, 0xF7, 0xFE, 0xFF
};

/* delay between auto-detect attemps at initialization */
#define DETECT_DELAY (2500)	/* msec */

/* Communication codes */
#define ESC 0x1B
static char VARIO_DISPLAY[] = { ESC, 0x01 };

#define VARIO_DISPLAY_LEN 2
#define VARIO_DEVICE_ID_LEN 2
#define VARIO_DEVICE_ID_REPLY_LEN 18
#define MAXREAD 18
/* Global variables */

static int brl_fd;		/* file descriptor for comm port */
static struct termios oldtio,	/* old terminal settings for com port */
  curtio;			/* current settings */
static unsigned char *rawdata,	/* translated data to send to display */
 *prevdata,			/* previous data sent */
 *dispbuf;

static struct brlinfo identbrl (const char *name, const char *dev)
{
  int devcnt;
  struct brlinfo retinfo;

  devcnt = sizeof (baumdevs) / sizeof (struct brlinfo);
  retinfo.cols = 0;

  for (devnr = 0; devnr < devcnt; devnr++)
    if (!strcmp (name, baumdevs[devnr].name))
     {
       ncells = baumdevs[devnr].cols + baumdevs[devnr].st_cells;
       brlcols = baumdevs[devnr].cols;
       printf (" %s on %s\n", baumdevs[devnr].fullname, dev);
       retinfo = baumdevs[devnr];
       break;
     }

  return retinfo;

}

static void brl_debug (unsigned int dbg)
{
brl_dbg = dbg;
}

static void initbrl (brldim * brl, const char *tty)
{
  brldim res;			/* return result */
  int i = 0;

  res.disp = rawdata = prevdata = NULL;

  /* Open the Braille display device for random access */
  brl_fd = open (tty, O_RDWR | O_NOCTTY);
  if (brl_fd < 0)
   {
     sbl_log ("Open failed on port %s: %s", tty, strerror (errno));
     brl->brl_fd=-1;
     return;
   }
  tcgetattr (brl_fd, &oldtio);	/* save current settings */
  /* Construct new settings by working from current state */
  memcpy (&curtio, &oldtio, sizeof (struct termios));
  cfmakeraw (&curtio);
  /* should we use cfmakeraw ? */
  /* local */
  curtio.c_lflag &= ~TOSTOP;	/* no SIGTTOU to backgrounded processes */
  /* control */
  curtio.c_cflag |= CLOCAL	/* ignore status lines (carrier detect...) */
    | CREAD;			/* enable reading */
  curtio.c_cflag = BAUDRATE | CS8 | CLOCAL | CREAD;
  /* input */
  curtio.c_iflag &= ~(INPCK	/* no input parity check */
		      | ~IXOFF	/* don't send XON/XOFF */
    );

  /* noncanonical: for first operation */
  curtio.c_cc[VTIME] = 1;	/* 0.1sec timeout between chars on input */
  curtio.c_cc[VMIN] = 0;	/* no minimum input. */

  /* Make the settings active and flush the input/output queues. This time we
     check the return code in case somehow the settings are invalid. */
  if (tcsetattr (brl_fd, TCSAFLUSH, &curtio) == -1)
   {
     sbl_log ("tcsetattr: %s", strerror (errno));
     goto failure;
   }

  lastcmd = EOF;

  res.x = brlcols;		/* initialise size of display */
  res.y = BRLROWS;		/* always 1 (I want a 5 line display!!) */
  res.brl_fd = brl_fd;

  /* Allocate space for buffers */
  dispbuf = res.disp = (unsigned char *) malloc (ncells);
  prevdata = (unsigned char *) malloc (ncells);
  rawdata = (unsigned char *) malloc (2 * ncells + VARIO_DISPLAY_LEN);
  /* 2* to insert ESCs if char is ESC */
  if (!res.disp || !prevdata || !rawdata)
    goto failure;

  /* Initialize rawdata. It will be filled in and used directly to
     write to the display in writebrl(). */
  for (i = 0; i < VARIO_DISPLAY_LEN; i++)
    rawdata[i] = VARIO_DISPLAY[i];
  memset (rawdata + VARIO_DISPLAY_LEN, 0, 2 * ncells * BRLROWS);

  /* Force rewrite of display on first writebrl */
  memset (prevdata, 0x00, ncells);

  typing_mode = 0;		/* sets CK keys to command mode - 1 = typing mode */
  *brl = res;
  return;

failure:;
  sbl_log ("Baum Vario driver giving up");
  closebrl (&res);
  brl->x = -1;
  return;
}

static void closebrl (brldim * brl)
{
  if (brl_fd >= 0)
   {
     tcsetattr (brl_fd, TCSANOW, &oldtio);
     close (brl_fd);
   }
  if (brl->disp)
    free (brl->disp);
  if (rawdata)
    free (rawdata);
  if (prevdata)
    free (prevdata);
    brl->brl_fd = -1;
}

/* Display a NCELLS long buffer on the display. 
   Vario 80 only supports a full update in native Baum mode. */
static void display (const unsigned char *buf)
{
  int i, escs = 0;

  for (i = 0; i < ncells; i++)
   {
     rawdata[VARIO_DISPLAY_LEN + i + escs] = DotsTable[buf[i]];
     if (rawdata[VARIO_DISPLAY_LEN + i + escs] == ESC)
      {
	/* We need to take care of the ESC char, which is used for infotypess. */
	escs++;
	rawdata[VARIO_DISPLAY_LEN + i + escs] = ESC;
      }
   }
  write (brl_fd, rawdata, VARIO_DISPLAY_LEN + ncells + escs);
  tcdrain (brl_fd);		/* Does this help? It seems at it made the scrolling
				   smoother */
}

static void setbrlstat (const unsigned char *s)
/* We have 4 status cells. Unclear how to handle. 
   I choose Tieman style in brlconf.h. Any other suggestions? */
{
  int i;

  if (ncells > brlcols)
   {				/* When we have status cells */
     for (i = 0; i < ncells - brlcols; i++)
      {
	dispbuf[brlcols + i] = 255;
      }
   }

if(s) return;
}

static void writebrl (brldim * brl)
{
  int start, stop;

  if (brl->x != brlcols || brl->y != BRLROWS || brl->disp != dispbuf)
    return;

  /* Determining start and stop for memcpy and prevdata */
  for (start = 0; start < ncells; start++)
    if (brl->disp[start] != prevdata[start])
      break;
  if (start == ncells)
    return;

  for (stop = ncells - 1; stop > start; stop--)
    if (brl->disp[stop] != prevdata[stop])
      break;

  /* Is it better to do it this way, or should we copy the whole display.
     Vario Emulation 1 doesnt support partial updates. So this kinda
     makes no sense */
  memcpy (prevdata + start, brl->disp + start, stop - start + 1);
  display (brl->disp);
}

static int readbrl (int *type)
{
#define ROUTING 0x22
#define BUTTON 0x24
#define FRONTKEY 0x28
  static unsigned int code;	/* 32bits code representing pressed keys once the
				   input bytes are interpreted */

  int res = EOF;		/* command code to return. code is mapped to res. */
  unsigned char buf[MAXREAD];	/* read buffer */

  int readcnt = 0;		/* counter for received bytes */

  /* reset to nonblocking */
  curtio.c_cc[VTIME] = 0;
  curtio.c_cc[VMIN] = 0;
  tcsetattr (brl_fd, TCSANOW, &curtio);
  if (!(readcnt = read (brl_fd, buf, MAXREAD)))
    return EOF;
  curtio.c_cc[VTIME] = 1;
  curtio.c_cc[VMIN] = 0;
  tcsetattr (brl_fd, TCSANOW, &curtio);

  
  if(brl_dbg>1) 
  sbl_log("brlkey: %i %i %i %i %i %i %i %i %i %i %i %i %i %i %i %i \n",
  buf[0],buf[1],buf[2],buf[3],buf[4],buf[5],buf[6],buf[7],buf[8],buf[9],
  buf[10],buf[11],buf[12],buf[13],buf[14],buf[15]);
  
  
  res = EOF;
  *type = 0;
  {
    if (buf[0] == ESC && buf[1] == ROUTING)
     {
       int i;

       for (i = 0; i < readcnt; i++)
	{

	  switch (buf[2 + i])
	   {
	   case 1:
	     *type = 1;
	     return 100 + (8 * i);
	   case 2:
	     *type = 1;
	     return 101 + (8 * i);
	   case 4:
	     *type = 1;
	     return 102 + (8 * i);
	   case 8:
	     *type = 1;
	     return 103 + (8 * i);
	   case 16:
	     *type = 1;
	     return 104 + (8 * i);
	   case 32:
	     *type = 1;
	     return 105 + (8 * i);
	   case 64:
	     *type = 1;
	     return 106 + (8 * i);
	   case 128:
	     *type = 1;
	     return 107 + (8 * i);
	   }

	}			/* for */
     }				/* if */
    if (buf[0] == ESC && buf[1] == BUTTON)
     {
       static int cnt = 0;

       if(devnr==DM80P) buf[2]=~buf[2];
       code = lastcmd ^ buf[2];
       lastcmd = buf[2];
       switch (code)
	{
	case 1:
	  res = 1;
	  *type = 1;
	  cnt++;
	  break;
	case 2:
	  res = 2;
	  *type = 1;
	  cnt++;
	  break;
	case 4:
	  res = 3;
	  *type = 1;
	  cnt++;
	  break;
	case 8:
	  res = 4;
	  *type = 1;
	  cnt++;
	  break;
	case 16:
	  res = 5;
	  *type = 1;
	  cnt++;
	  break;
	case 32:
	  res = 6;
	  *type = 1;
	  cnt++;
	  break;
	default:
	  *type = 0;
	}			/* switch */

       if (buf[2] == 0 || cnt > 2)
	{
	  cnt = 0;
	  *type = 0;
	  lastcmd = 0;
	}

       return res;
     }

  }				/* if */
  return EOF;
}
