/*
 *  suse-blinux - Braille-display support for linux
 *  Author: Marco Skambraks <marco@suse.de>
 *  SuSE GmbH Nuernberg
 *
 *
 * suse-blinux based on some parts brltty
 * special thanks to the Brltty-Team
 * Nicolas Pitre <nico@cam.org>
 * Stphane Doyon <s.doyon@videotron.ca>
 * Nikhil Nair <nn201@cus.cam.ac.uk>
 *
 * This is free software, placed under the terms of the
 * GNU General Public License, as published by the Free Software
 * Foundation.  Please see the file COPYING for details.
*/
#define BRL_C 1

#define __EXTENSIONS__		/* for termios.h */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/termios.h>
#include <string.h>

#define MODEL   Models[devnr].ID
#include "brlconf.h"
#include "../brl.h"
#include "../brl_driver.h"
/*#include "../scr.h"
#include "../misc.h"*/
#include "../config.h"
#include "../sbllog.h"

#if USE_PARALLEL_PORT
#include "alva_api.library/alva_api.h"
#endif
static unsigned int brl_dbg=0;
unsigned char highbits[4] = { 128, 64, 32, 16 };
unsigned char lowbits[4] = { 8, 4, 2, 1 };

/* Braille display parameters */

static int sat540_key (int mode, int key);
typedef struct
{
  char *Name;
  int ID;
  int Cols;
  int NbStCells;
}
BRLPARAMS;

BRLPARAMS Models[] = {
  {
   /* ID == 0 */
   "ABT320",
   ABT320,
   20,
   3}
  ,
  {
   /* ID == 1 */
   "ABT340",
   ABT340,
   40,
   3}
  ,
  {
   /* ID == 2 */
   "ABT340 Desktop",
   ABT34D,
   40,
   5}
  ,
  {
   /* ID == 3 */
   "ABT380",
   ABT380,
   80,
   5}
  ,
  {
   /* ID == 4 */
   "ABT380 Twin Space",
   ABT38D,
   80,
   5}
  ,
  {
   /* ID == 11 */
   "Alva Delphi 40",
   DEL440,
   40,
   3}
  ,
  {
   /* ID == 13 */
   "Alva Delphi 80",
   DEL480,
   80,
   5}
  ,
  {
   /* ID == 14 */
   "Alva Satellite 40",
   SAT540,
   40,
   3}
  ,
  {
   0,0,0,0
   }
};

static int devnr;
struct brlinfo alvadevs[] = {
  {"abt320", "Alva brailleterminal 320", 20, 3, 0, 0, B9600},
  {"abt340", "Alva brailleterminal 340", 40, 3, 0, 0, B9600},
  {"abt34d", "Alva brailleterminal 340 desktop", 40, 5, 0, 0, B9600},
  {"abt380", "Alva brailleterminal 380", 80, 5, 0, 0, B9600},
  {"abt38d", "Alva brailleterminal 380 twin space", 80, 5, 0, 0, B9600},
  {"del440", "Alva Delphi 40", 40, 3, 0, 0, B9600},
  {"del480", "Alva Delphi 80", 80, 5, 0, 0, B9600},
  {"sat540", "Alva Satellite 40", 40, 3, 0, 0, B9600},
  {"sat570", "Alva Satellite 70", 70, 3, 0, 0, B9600}
};

#define BRLROWS		1
#define MAX_STCELLS	5	/* hiest number of status cells */

/* bitpattern for braille-characters */
#define B1 1
#define B2 2
#define B3 4
#define B4 8
#define B5 16
#define B6 32
#define B7 64
#define B8 128

char TransTable[256] = {
  0x00, 0x01, 0x08, 0x09, 0x02, 0x03, 0x0A, 0x0B,
  0x10, 0x11, 0x18, 0x19, 0x12, 0x13, 0x1A, 0x1B,
  0x04, 0x05, 0x0C, 0x0D, 0x06, 0x07, 0x0E, 0x0F,
  0x14, 0x15, 0x1C, 0x1D, 0x16, 0x17, 0x1E, 0x1F,
  0x20, 0x21, 0x28, 0x29, 0x22, 0x23, 0x2A, 0x2B,
  0x30, 0x31, 0x38, 0x39, 0x32, 0x33, 0x3A, 0x3B,
  0x24, 0x25, 0x2C, 0x2D, 0x26, 0x27, 0x2E, 0x2F,
  0x34, 0x35, 0x3C, 0x3D, 0x36, 0x37, 0x3E, 0x3F,
  0x40, 0x41, 0x48, 0x49, 0x42, 0x43, 0x4A, 0x4B,
  0x50, 0x51, 0x58, 0x59, 0x52, 0x53, 0x5A, 0x5B,
  0x44, 0x45, 0x4C, 0x4D, 0x46, 0x47, 0x4E, 0x4F,
  0x54, 0x55, 0x5C, 0x5D, 0x56, 0x57, 0x5E, 0x5F,
  0x60, 0x61, 0x68, 0x69, 0x62, 0x63, 0x6A, 0x6B,
  0x70, 0x71, 0x78, 0x79, 0x72, 0x73, 0x7A, 0x7B,
  0x64, 0x65, 0x6C, 0x6D, 0x66, 0x67, 0x6E, 0x6F,
  0x74, 0x75, 0x7C, 0x7D, 0x76, 0x77, 0x7E, 0x7F,
  0x80, 0x81, 0x88, 0x89, 0x82, 0x83, 0x8A, 0x8B,
  0x90, 0x91, 0x98, 0x99, 0x92, 0x93, 0x9A, 0x9B,
  0x84, 0x85, 0x8C, 0x8D, 0x86, 0x87, 0x8E, 0x8F,
  0x94, 0x95, 0x9C, 0x9D, 0x96, 0x97, 0x9E, 0x9F,
  0xA0, 0xA1, 0xA8, 0xA9, 0xA2, 0xA3, 0xAA, 0xAB,
  0xB0, 0xB1, 0xB8, 0xB9, 0xB2, 0xB3, 0xBA, 0xBB,
  0xA4, 0xA5, 0xAC, 0xAD, 0xA6, 0xA7, 0xAE, 0xAF,
  0xB4, 0xB5, 0xBC, 0xBD, 0xB6, 0xB7, 0xBE, 0xBF,
  0xC0, 0xC1, 0xC8, 0xC9, 0xC2, 0xC3, 0xCA, 0xCB,
  0xD0, 0xD1, 0xD8, 0xD9, 0xD2, 0xD3, 0xDA, 0xDB,
  0xC4, 0xC5, 0xCC, 0xCD, 0xC6, 0xC7, 0xCE, 0xCF,
  0xD4, 0xD5, 0xDC, 0xDD, 0xD6, 0xD7, 0xDE, 0xDF,
  0xE0, 0xE1, 0xE8, 0xE9, 0xE2, 0xE3, 0xEA, 0xEB,
  0xF0, 0xF1, 0xF8, 0xF9, 0xF2, 0xF3, 0xFA, 0xFB,
  0xE4, 0xE5, 0xEC, 0xED, 0xE6, 0xE7, 0xEE, 0xEF,
  0xF4, 0xF5, 0xFC, 0xFD, 0xF6, 0xF7, 0xFE, 0xFF
};

/* Global variables */

char DefDev[] = BRLDEV;		/* default braille device */
int brl_fd;			/* file descriptor for Braille display */
struct termios oldtio;		/* old terminal settings */
unsigned char *rawdata;		/* translated data to send to Braille */
unsigned char *prevdata;	/* previously sent raw data */
unsigned char StatusCells[MAX_STCELLS];	/* to hold status info */
unsigned char PrevStatus[MAX_STCELLS];	/* to hold previous status */
BRLPARAMS *model;		/* points to terminal model config struct */
int ReWrite = 0;		/* 1 if display need to be rewritten */

/* Communication codes */

char BRL_START[] = "\r\033B";	/* escape code to display braille */

#define DIM_BRL_START 3
char BRL_END[] = "\r";		/* to send after the braille sequence */

#define DIM_BRL_END 1
char BRL_ID[] = "\033ID=";

#define DIM_BRL_ID 4

/* Key values */

/* NB: The first 7 key values are the same as those returned by the
 * old firmware, so they can be used directly from the input stream as
 * make and break sequence already combined... not to be changed.
 */
#define KEY_PROG 	0x008	/* the PROG key */
#define KEY_HOME 	0x004	/* the HOME key */
#define KEY_CURSOR 	0x002	/* the CURSOR key */
#define KEY_UP 		0x001	/* the UP key */
#define KEY_LEFT 	0x010	/* the LEFT key */
#define KEY_RIGHT 	0x020	/* the RIGHT key */
#define KEY_DOWN 	0x040	/* the DOWN key */
#define KEY_CURSOR2 	0x080	/* the CURSOR2 key */
#define KEY_HOME2 	0x100	/* the HOME2 key */
#define KEY_PROG2 	0x200	/* the PROG2 key */
#define KEY_ROUTING	0x400	/* cursor routing key set */

/* those keys are not supposed to be combined, so their corresponding 
 * values are not bit exclusive between them.
 */
#define KEY_ROUTING_A	0x1000	/* first cur routing over status display */
#define KEY_ROUTING_B	0x2000	/* second cur routing over status disp. */
#define KEY_ROUTING_C	0x3000	/* ... */
#define KEY_ROUTING_D	0x4000
#define KEY_ROUTING_E	0x5000
#define KEY_ROUTING_F	0x6000

/* first cursor routing offset on main display (old firmware only) */
#define KEY_ROUTING_OFFSET 168

/* Index for new firmware protocol */
int OperatingKeys[10] = { KEY_PROG, KEY_HOME, KEY_CURSOR,
  KEY_UP, KEY_LEFT, KEY_RIGHT, KEY_DOWN,
  KEY_CURSOR2, KEY_HOME2, KEY_PROG2
};
int StatusKeys[6] = { KEY_ROUTING_A, KEY_ROUTING_B, KEY_ROUTING_C,
  KEY_ROUTING_D, KEY_ROUTING_E, KEY_ROUTING_F
};

struct brlinfo identbrl (const char *name, const char *dev)
{
  int devcnt;
  struct brlinfo retinfo;

  retinfo.cols = 0;
  devcnt = sizeof (alvadevs) / sizeof (struct brlinfo);

  for (devnr = 0; devnr < devcnt; devnr++)
    if (!strcmp (alvadevs[devnr].name, name))
     {
       printf (" %s on %s\n", alvadevs[devnr].fullname, dev);
       retinfo = alvadevs[devnr];
       break;
     }

  return retinfo;

}

/* SendToAlva() is shared with speech.c */
  int SendToAlva (unsigned char *data, int len);


int SendToAlva (unsigned char *data, int len)
{
#if USE_PARALLEL_PORT
  BrailleWrite (data, len);
  BrailleProcess ();
  BrailleProcess ();
  BrailleProcess ();
  BrailleProcess ();
  BrailleProcess ();
  return 1;
#else
  if (write (brl_fd, data, len) == len)
    return 1;
  return 0;
#endif
}


void brl_debug(unsigned int dbg)
{
brl_dbg=dbg;
}
void initbrl (brldim * brl, const char *dev)
{
  brldim res;			/* return result */
  struct termios newtio;	/* new terminal settings */
  int ModelID = MODEL;
  unsigned char buffer[DIM_BRL_ID + 1];

  res.disp = rawdata = prevdata = NULL;	/* clear pointers */

#if USE_PARALLEL_PORT

  if (ioperm (LPT_PORT, 3, 1))
   {
     sbl_log ("unable to acquire IO permission at 0x%03X\n",LPT_PORT);
     goto failure;
   }
  BrailleOpen ((unsigned char *) LPT_PORT);
  BrailleProcess ();
  if ((BrailleRead (buffer, DIM_BRL_ID + 1) != DIM_BRL_ID + 1) ||
      (strncmp ((char *) buffer, BRL_ID, DIM_BRL_ID) != 0))
   {
     sbl_log ("can't get braille terminal ID\n");
     goto failure;
   }
  if (ModelID == ABT_AUTO)
    ModelID = buffer[DIM_BRL_ID];

#else

  /* Open the Braille display device for random access */
  if (!dev)
    dev = DefDev;
  brl_fd = open (dev, O_RDWR | O_NOCTTY);
  if (brl_fd < 0)
   {
     sbl_log ("%s: %s\n", dev, strerror (1));
     brl->brl_fd=-1;
   return;
   }

  tcgetattr (brl_fd, &oldtio);	/* save current settings */

  /* Set flow control and 8n1, enable reading */
  newtio.c_cflag = CRTSCTS | CS8 | CLOCAL | CREAD;

  /* Ignore bytes with parity errors and make terminal raw and dumb */
  newtio.c_iflag = IGNPAR;
  newtio.c_oflag = 0;		/* raw output */
  newtio.c_lflag = 0;		/* don't echo or generate signals */
  newtio.c_cc[VMIN] = 0;	/* set nonblocking read */
  newtio.c_cc[VTIME] = 0;

  /* autodetecting ABT model */
  do
   {
     /* to force DTR off */
     cfsetispeed (&newtio, B0);
     cfsetospeed (&newtio, B0);
     tcsetattr (brl_fd, TCSANOW, &newtio);	/* activate new settings */
     usleep (50000);
     tcflush (brl_fd, TCIOFLUSH);	/* clean line */
     usleep (500000);
     /* DTR back on */
     cfsetispeed (&newtio, BAUDRATE);
     cfsetospeed (&newtio, BAUDRATE);
     tcsetattr (brl_fd, TCSANOW, &newtio);	/* activate new settings */
     sleep (1);		/* give time to send ID string */
     /* This "if" statement can be commented out to try autodetect once anyway */
     if (ModelID != ABT_AUTO)
       break;
     if (read (brl_fd, &buffer, DIM_BRL_ID + 1) == DIM_BRL_ID + 1)
      {
	if (!strncmp ((char *) buffer, BRL_ID, DIM_BRL_ID))
	  ModelID = buffer[DIM_BRL_ID];
      }
   }
  while (ModelID == ABT_AUTO);

#endif /* !PARALLEL */

  /* Find out which model we are connected to... */
  for (model = Models; model->Name && model->ID != ModelID; model++);
  if (!model->Name)
   {
     /* Unknown model */
     sbl_log ("*** Detected unknown Alva model which ID is %d.\n",
	       ModelID);
     sbl_log ("*** Please fix Models[] in Alva/brlmain.cc and mail the maintainer\n");
     goto failure;
   }

  /* Set model params... */
  res.x = model->Cols;		/* initialise size of display */
  res.y = BRLROWS;
  res.brl_fd = brl_fd;

  /* Allocate space for buffers */
  res.disp = (unsigned char *) malloc (res.x * res.y);
  rawdata = (unsigned char *) malloc (res.x * res.y);
  prevdata = (unsigned char *) malloc (res.x * res.y);
  if (!res.disp || !rawdata || !prevdata)
   {
     sbl_log ("can't allocate braille buffers\n");
     goto failure;
   }

  ReWrite = 1;			/* To write whole display at first time */

  *brl = res;
  return;

failure:
  if (res.disp)
    free (res.disp);
  if (rawdata)
    free (rawdata);
  if (prevdata)
    free (prevdata);
  brl->x = -1;
  return;
}

void closebrl (brldim * brl)
{
  free (brl->disp);
  free (rawdata);
  free (prevdata);
#if USE_PARALLEL_PORT
  BrailleClose ();
  brl->brl_fd = -1;
#else
  tcsetattr (brl_fd, TCSANOW, &oldtio);	/* restore terminal settings */
  close (brl_fd);
#endif
}

int WriteToBrlDisplay (int Start, int Len, unsigned char *Data)
{
  unsigned char outbuf[DIM_BRL_START + 2 + Len + DIM_BRL_END];
  int outsz = 0;

  memcpy (outbuf, BRL_START, DIM_BRL_START);
  outsz += DIM_BRL_START;
  outbuf[outsz++] = (char) Start;
  outbuf[outsz++] = (char) Len;
  memcpy (outbuf + outsz, Data, Len);
  outsz += Len;
  memcpy (outbuf + outsz, BRL_END, DIM_BRL_END);
  outsz += DIM_BRL_END;
  return SendToAlva (outbuf, outsz);
}

#if USE_PARALLEL_PORT
int ReadCycles = 0;		/* for hack below */
int ShouldRestart = 0;
#endif

void writebrl (brldim * brl)
{
  int i, j, k;
  static int Timeout = 0;

#if USE_PARALLEL_PORT
  /* HACK ALERT!
   * The parallel port library is buggy:  the display just hang after
   * a period of time.  Sources aren't available to let any honest person
   * fix it.  So we close and reopen the library once in a while and hope 
   * for the best.
   */
  if (ReadCycles >= 50)
   {
     int x = brl->x;

     ReadCycles = 0;
     closebrl (brl);
     initbrl (brl, NULL);
     if (brl->x != x)
      {
	ShouldRestart = 1;
	return;
      }
     ReWrite = 1;
   }
#endif

  if (ReWrite || ++Timeout > (REFRESH_RATE / DELAY_TIME))
   {
     ReWrite = Timeout = 0;
     /* We rewrite the whole display */
     i = 0;
     j = model->Cols;
   }
  else
   {
     /* We update only the display part that has been changed */
     i = 0;
     while ((brl->disp[i] == prevdata[i]) && (i < model->Cols))
       i++;
     j = model->Cols - 1;
     while ((brl->disp[j] == prevdata[j]) && (j >= i))
       j--;
     j++;
   }
  if (i < j)			/* there is something different */
   {
     for (k = 0;k < (j - i);k++)
     rawdata[k] = TransTable[(prevdata[i + k] = brl->disp[i + k])];
     WriteToBrlDisplay (model->NbStCells + i, j - i, rawdata);
   }
}

void setbrlstat (const unsigned char *st)
{
  const char highbits[10] = { B2 + B4 + B5, B1, B1 + B2, B1 + B4,
    B1 + B4 + B5, B1 + B5, B1 + B2 + B4, B1 + B2 + B4 + B5, B1 + B2 + B5,
    B2 + B4
  };
  const char lowbits[10] = { B6 + B7 + B8, B3, B3 + B7, B3 + B6, B3 + B6 + B8,
    B3 + B8, B3 + B6 + B7, B3 + B6 + B7 + B8, B3 + B7 + B8, B6 + B7
  };

  /* Update status cells on braille display */
  if (memcmp (st, PrevStatus, model->NbStCells))	/* only if it changed */
   {
     StatusCells[0] = highbits[(int) st[0] / 10] | lowbits[(int) st[1] / 10];
     StatusCells[1] = highbits[(int) st[0] % 10] | lowbits[(int) st[1] % 10];
     StatusCells[2] = highbits[(int) st[2] % 10];

     WriteToBrlDisplay (0, model->NbStCells, StatusCells);
   }
}

int GetABTKey (int *Keys)
{
  unsigned char c;

  read (brl_fd, &c, 1);
  switch (c)
   {
   case 0x71:
     read (brl_fd, &c, 1);

     if (c >= 0x20 && c <= 0x24)
      {
	*Keys = c - 0x20 + 100;
	sbl_log ("key: %d ", *Keys);
	return 1;
      }

     if (c >= 0x30 && c <= 0x34)
      {
	*Keys = c - 0x30 + 200;
	sbl_log ("key: %d ", *Keys);
	return 1;
      }
     if (c < 0x80)
      {
	if (model->ID == SAT540)
	  *Keys = sat540_key (0x71, c);
	else
	  *Keys = c + 1;
	sbl_log ("%d ", *Keys);
	return 1;
      }
     if (c >= 0x80)
      {
	if (model->ID == SAT540)
	  *Keys = sat540_key (0x71, c);
	else
	  *Keys = (c - 0x80) + 1;
	sbl_log ("%d ", *Keys);
	return 0;
      }
     break;
   case 0x72:
     read (brl_fd, &c, 1);
     if (c < 0x80)
      {
	*Keys = c + model->NbStCells + 100;
	return 1;
      }
     break;

   case 0x75:
     read (brl_fd, &c, 1);
     if (c < 0x80)
      {
	*Keys = c + 200 + model->NbStCells;
	sbl_log ("key: %d ", *Keys);
	return 1;
      }

     if (c >= 0x80)
      {
	*Keys = (c - 0x80) + 200 + model->NbStCells;
	sbl_log ("%d ", *Keys);
	return 0;
      }

     break;

   case 0x77:
     read (brl_fd, &c, 1);
     if (c < 0x80)
      {
	if (model->ID == SAT540)
	  *Keys = sat540_key (0x77, c);

	sbl_log ("key: %d ", *Keys);
	return 1;
      }

     if (c >= 0x80)
      {

	if (model->ID == SAT540)
	  *Keys = sat540_key (0x77, c);

	sbl_log ("%d ", *Keys);
	return 0;
      }

     break;

   default:
     *Keys = EOF;
     return 1;
   }

  return 1;			/* makes compiler happy :-) */

}

int readbrl (int *type)
{
  int key = EOF;

  *type = GetABTKey (&key);
  return key;
}
static int sat540_key (int mode, int key)
{

  if (mode == 0x71)
    switch (key)
     {
/* key pressed  released */
     case 0x01:
     case 0x81:
       return 1;
     case 0x04:
     case 0x84:
       return 2;
     case 0x03:
     case 0x83:
       return 3;
     case 0x06:
     case 0x86:
       return 4;
     case 0x05:
     case 0x85:
       return 5;
     case 0x02:
     case 0x82:
       return 6;
     }

  if (mode == 0x77)
    switch (key)
     {
     case 0x00:
     case 0x80:
       return 20;
     case 0x01:
     case 0x81:
       return 21;
     case 0x02:
     case 0x82:
       return 22;
     case 0x03:
     case 0x83:
       return 23;
     case 0x04:
     case 0x84:
       return 24;
     case 0x05:
     case 0x85:
       return 25;
     case 0x20:
     case 0xa0:
       return 30;
     case 0x21:
     case 0xa1:
       return 31;
     case 0x22:
     case 0xa2:
       return 32;
     case 0x23:
     case 0xa3:
       return 33;
     case 0x24:
     case 0xa4:
       return 34;
     case 0x25:
     case 0xa5:
       return 35;
     }

  return 0;

}
