
/*
 * -*- mode: C; tab-width:8;  -*-
 * 
 *	aclTM.c -
 */

/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * See the file aclMesa.c for more informations about authors
 */



#if defined(ACL)

#include "aclDrv.h"



#if 1
#define LOG_DEBUG
#endif

#define LOG_DBGINFO "<aclTM>"
#include "log_debug.h"



static ACLMesaTMNode *
aclTMInsertNode(ACLMesaTMNode *prev_node)
{
    ACLMesaTMNode *node;
    
LOG("aclTMInsertNode(): ######\n");

    if(!(node = malloc(sizeof(ACLMesaTMNode)))) {
	ERR("aclTMAllocNode(): out of memory !\n");
	exit(-1);
    }

    node->prev = prev_node;

    if(prev_node) {
	if((node->next = prev_node->next))
	    node->next->prev = node;
	prev_node->next = node;
    }
    else
	node->next = NULL;

    return node;
}



void
aclTMCreateContext(ACLMesaContext aclMesa)
{
    ACLContext aclctx = aclMesa->aclctx;

    ACLMesaTMNode *newnode;
    ACLMesaTMContext *tmctx = aclMesa->TMCtx;
    Tulong start = 0, end = 0;

LOG("aclTMCreateContext(): ######\n");

    if(tmctx) {
	ERR("aclTMCreateContext(): already inited!\n");
	return;
    }

    if(!(aclMesa->TMCtx = tmctx = malloc(sizeof(ACLMesaTMContext)))) {
	ERR("aclTMCreateContext(): out of memory !\n");
	exit(-1);
    }

#if 0
    if(aclMesa->acltex_mod->TexMemoryInfo(&start, &end) {
	ERR("Tex memory not available in hardware!\n");
	exit(-1);
    }
#endif
    end = 0x100000;

    if(aclMesa->verbose) {
	MSG("Hardware tex RAM info:");
	MSG("    Start texture memory address: 0x%lx\n", start);
	MSG("    End texture memory address:   0x%lx\n", end);
    }

    tmctx->bindNumber = 0;
    tmctx->freeTexMem = end - start + 1;
    tmctx->firstUsedNode = NULL;
    tmctx->insertNode = NULL;
    tmctx->lastUsedNode = NULL;

    newnode = tmctx->firstFreeNode = aclTMInsertNode(NULL);

    newnode->hwleft = NULL;
    newnode->hwright = NULL;
    newnode->start = start;
    newnode->end = end;
    newnode->priority = 0;
    newnode->is_insert = 0;
}



static void
aclTMFindBestMultiblock(ACLMesaTMContext *tmctx,
	ACLMesaTMNode **first, ACLMesaTMNode **last)
{
LOG("aclTMFindBestMultiblock(): ######\n");

}



static void
aclTMReinsertTexmemNode(ACLMesaTMContext *tmctx, ACLMesaTMNode *node)
{
    ACLMesaTMNode *insert;

LOG("aclTMReinsertTexmemNode(): ######\n");

    if(node->is_insert)
	return;

    if(!node->prev) tmctx->firstUsedNode = node->next;
    else node->prev->next = node->next;
    if(!node->next) tmctx->lastUsedNode = node->prev;
    else node->next->prev = node->prev;

    if((insert = tmctx->insertNode)) {
	node->next = insert->next;
	if(insert->next) insert->next->prev = node;
	node->prev = insert;
	insert->next = node;
	insert->is_insert = 0;
    }
    else {
	node->next = tmctx->firstUsedNode;
	node->prev = NULL;
	tmctx->firstUsedNode = node;
    }

    node->priority = 1;
    node->is_insert = 1;
    tmctx->insertNode = node;
}



static void
aclTMReinsertFreeNode(ACLMesaTMContext *tmctx, ACLMesaTMNode *node)
{
    ACLMesaTMNode *insert;
}



static void
aclTMDeleteTexmemNodes(ACLMesaTMContext *tmctx, ACLMesaTMNode *node)
{
    ACLMesaTMNode *last;

    while((last = node)) {
	if(!node->prev) tmctx->firstUsedNode = node->next;
	else node->prev->next = node->next;
	if(!node->next) tmctx->lastUsedNode = node->prev;
	else node->next->prev = node->prev;

	if(node->is_insert)
	    if((tmctx->insertNode = node->prev))
		node->prev->is_insert = 1;

	node = node->hwright;
	free(last);
    }
}



static void
aclTMInsertIntoFreeList(ACLMesaTMContext *tmctx,
	Tuint start, Tuint size, Tbool check)
{
    ACLMesaTMNode *last=NULL, *node=tmctx->firstFreeNodeHw, *insert;
    Tbool last_merged=0, node_merged=0;

    /* first find place to insert (in hw) */
    while(node) {
	if(node->start > start) break;
	last = node;
	node = node->hwright;
    }

    /*
     * INSERT A FREE BLOCK INTO LIST OF FREE NODES. Possibilities:
     *	1 - a block fits exactly between two nodes -> update first node with
     *      new big size and throw out the second node
     *	2 - a block fits to left node
     *	3 - a block fits to right node
     *	4 - a block doesn't fit at all. Allocation of new node required
     */
    if(check) {
        if(last) {
	    if(last->end == start-1) {
		last->end += size;
		last_merged = 1;
	    }
	}
	if(node) {
	    if(node->start == start+size) {
		node_merged = 1;
		if(last_merged) {
		    last->end = node->end;
		    last->hwright = node->hwright;
		    if(node->hwright) node->hwright->hwleft = last;
		}
		else {
		    node->start = start;
		    aclTMReinsertFreeNode(tmctx, node);
		}
	    }
	}
	if(last_merged)
	    aclTMReinsertFreeNode(tmctx, last);
    }

    if(!(last_merged || node_merged)) {
	insert = aclTMInsertNode(NULL);
	insert->start = start;
	insert->end = start+size-1;
	insert->priority = 0;
	insert->is_insert = 0;
	insert->hwleft = last;
	insert->hwright = node;
	if(last) last->hwright = insert;
	else tmctx->firstFreeNodeHw = insert;
	if(node) node->hwleft = insert;
	aclTMReinsertFreeNode(tmctx, insert);
    }
}



/*
 * the folowing strategy hopefuly helps against fragmentation of the tex RAM
 */
static void
aclTMReallocTexmem(ACLMesaTMContext *tmctx, Tuint size, Tuint restsize,
	ACLMesaTMNode *first, ACLMesaTMNode *last, Tbool single)
{
    ACLMesaTMNode *left, *right, *node;
    Tbool free_left, free_right;

    /*
     * FINE, A COMPLETELY FREE CHUNK ARIVED (priority 0)
     * - insert new node into the list of used nodes
     * - update list of free nodes
     */
    if(first->priority == 0) {
	return;
    }

    left = right = NULL;
    free_left = free_right = 0;

    if((left=first->hwleft))
	if(!left->priority) free_left = 1;

    if((right=last->hwright))
	if(!right->priority) free_right = 1;

    /*
     * NO FREE TEX MEM ON THE SIDES OF THE ACQUIRED BLOCK
     * - if single and restsize, alloc new node in TM free nodes list
     * - update 'first' node with size, re-insert it into used TM list
     * - delete all other nodes after 'first', if any.
     */
    if(!(free_left || free_right)) {
LOG("aclTMReallocTexmem(): no free mem on the left/right\n");
	if(single) {
	    if(restsize) {
		first->end = first->start + size - 1;
		aclTMInsertIntoFreeList(tmctx, first->end+1, restsize, 0);
	    }
	}
	else {
	    first->end = first->start + size - 1;
	    first->hwright = last->hwright;
	    if(last->hwright) last->hwright->hwleft = first;
	    last->hwright = NULL;
	    aclTMDeleteTexmemNodes(tmctx, first->hwright);
	    if(restsize)
		aclTMInsertIntoFreeList(tmctx, first->end + 1, restsize, 0);
	}
	aclTMReinsertTexmemNode(tmctx, first);
    }

    /*
     * FREE TEX MEM ON THE LEFT _AND_ ON THE RIGHT OF THE ACQUIRED BLOCK
     * - move one free node out of the free TM list
     * - update it with new size to be used
     * - update other free node with the restsize; update free TM list
     * - delete all 'first' to 'last' nodes
     * - insert new node into the TM list
     */
    else if(free_left && free_right) {
LOG("aclTMReallocTexmem(): free_left && free_right\n");
	if(left->prev) left->prev->next = left->next;
	else tmctx->firstFreeNode = left->next;
	if(left->next) left->next->prev = left->prev;
	left->hwright = right;
	right->hwleft = left;
	left->end = right->start = (left->start + size);
	left->end--;
	right->end = left->end + restsize;
	last->hwright = NULL;
	aclTMDeleteTexmemNodes(tmctx, first);
	aclTMReinsertTexmemNode(tmctx, left);
    }

    /*
     * FREE TEX MEM EITHER ON THE LEFT OR ON THE RIGHT OF THE ACQUIRED BLOCK
     * - update the free node with the rest of size; update free TM list
     * - update 'first' (left) or 'last' (right) node with the new size
     * - if ! single, delete other, not required nodes
     * - re-insert the node into TM list
     */
    else if (free_left && (restsize > 0)) {
LOG("aclTMReallocTexmem(): free_left\n");
	left->end = first->start = last->end - size;
	first->start++;
	first->end = last->end;
	if(!single) {
	    first->hwright = last->hwright;
	    if(last->hwright) {
		last->hwright->hwleft = first;
		last->hwright = NULL;
	    }
	    aclTMDeleteTexmemNodes(tmctx, first->hwright);
	}
	aclTMReinsertTexmemNode(tmctx, first);
    }

    else if (free_right && (restsize > 0)) {
LOG("aclTMReallocTexmem(): free_right\n");
	right->start = last->end = first->start + size;
	last->end--;
	last->start = first->start;
	if(!single) {
	    last->hwleft = first->hwleft;
	    if(first->hwleft) {
		first->hwleft->hwright = last;
	    }
	    last->hwleft->hwright = NULL;
	    aclTMDeleteTexmemNodes(tmctx, first);
	}
	aclTMReinsertTexmemNode(tmctx, last);
    }
}



ACLMesaTMNode *
aclTMAllocTexmem(ACLMesaTMContext *tmctx, Tuint size)
{
    ACLMesaTMNode *newnode, *foundnode, *left, *right, *node;
    Tuint nsize, fsize, tmp;
    Tbool destroy;

    if(!(node = tmctx->firstFreeNode)) {
	ERR("ALLOC texmem: Available texmem not found!\n");
	return NULL;
    }

    fsize = ~0;
    destroy = 0;
    foundnode = NULL;

    /*
     * loop through free nodes; find the best fitting one
     */
LOG("aclTMAllocTexmem(): Searching throgh single FREE blocks\n");
    do {
	nsize = node->end - node->start + 1;
	if((size <= nsize) && (size < fsize)) {
	    foundnode = node;
	    fsize = nsize;
	}
    } while(node = node->next);

    /*
     * loop through single, allocated, nodes; find the best fitting one
     */
    if(!foundnode) {
      if(!(node = tmctx->firstUsedNode)) {
	ERR("ALLOC texmem: Available texmem (size 0x%x) not found!\n", size);
	return NULL;
      }
LOG("aclTMAllocTexmem(): Searching through single LOADED blocks\n");
      destroy = 1;
      do {
	nsize = node->end - node->start + 1;
	if((size <= nsize) && (size < fsize)) {
	    foundnode = node;
	    fsize = nsize;
	}
      } while(node = node->next);
    }

    if(foundnode) {
	/*
	 * OK, we found one node which has size >= required size
	 */

	if(destroy) {

	  if(tmctx->insertNode == foundnode)
		return foundnode;

	  return foundnode;
	}
	else {
	    newnode = tmctx->insertNode =
		aclTMInsertNode(tmctx->insertNode);
	    if(!tmctx->firstUsedNode) {
		tmctx->firstUsedNode = newnode;
		tmctx->lastUsedNode = newnode;
	    }
	}
    }

    /*
     * loop through multiple, allocated, nodes; find the best fitting ones
     */
LOG("aclTMAllocTexmem(): Ooops! Acquire LOADED tex RAM !!!\n");
}



void
aclTMFreeTexmem(ACLMesaTMNode *node)
{
}



#if 0

static struct gl_texture_object *fxTMFindOldestTMBlock(fxMesaContext fxMesa,
						       tfxTMAllocNode *tmalloc,
						       GLuint texbindnumber)
{
  GLuint age,oldestage,lasttimeused;
  struct gl_texture_object *oldesttexobj;

  oldesttexobj=tmalloc->tObj;
  oldestage=0;

  while(tmalloc) {
    lasttimeused=((tfxTexInfo *)(tmalloc->tObj->DriverData))->tmi.lastTimeUsed;

    if(lasttimeused>texbindnumber)
      age=texbindnumber+(UINT_MAX-lasttimeused+1); /* TO DO: check */
    else
      age=texbindnumber-lasttimeused;

    if(age>=oldestage) {
      oldestage=age;
      oldesttexobj=tmalloc->tObj;
    }

    tmalloc=tmalloc->next;
  }

  return oldesttexobj;
}




static GLboolean fxTMFreeOldTMBlock(fxMesaContext fxMesa, GLint tmu)
{
  struct gl_texture_object *oldesttexobj;

  if(!fxMesa->tmAlloc[tmu])
    return GL_FALSE;

  oldesttexobj=fxTMFindOldestTMBlock(fxMesa,fxMesa->tmAlloc[tmu],fxMesa->texBindNumber);

  fxTMMoveOutTM(fxMesa,oldesttexobj);

  return GL_TRUE;
}




static tfxTMFreeNode *fxTMExtractTMFreeBlock(tfxTMFreeNode *tmfree, int texmemsize,
					     GLboolean *success, FxU32 *startadr)
{
  int blocksize;

  /* TO DO: cut recursion */

  if(!tmfree) {
    *success=GL_FALSE;
    return NULL;
  }

  blocksize=tmfree->endAddress-tmfree->startAddress+1;

  if(blocksize==texmemsize) {
    tfxTMFreeNode *nexttmfree;

    *success=GL_TRUE;
    *startadr=tmfree->startAddress;

    nexttmfree=tmfree->next;
    free(tmfree);

    return nexttmfree;
  }

  if(blocksize>texmemsize) {
    *success=GL_TRUE;
    *startadr=tmfree->startAddress;

    tmfree->startAddress+=texmemsize;

    return tmfree;
  }

  tmfree->next=fxTMExtractTMFreeBlock(tmfree->next,texmemsize,success,startadr);

  return tmfree;
}




static tfxTMAllocNode *fxTMGetTMBlock(fxMesaContext fxMesa, struct gl_texture_object *tObj,
				      GLint tmu, int texmemsize)
{
  tfxTMFreeNode *newtmfree;
  tfxTMAllocNode *newtmalloc;
  GLboolean success;
  FxU32 startadr;

  for(;;) { /* TO DO: improve performaces */
    newtmfree=fxTMExtractTMFreeBlock(fxMesa->tmFree[tmu],texmemsize,&success,&startadr);

    if(success) {
      fxMesa->tmFree[tmu]=newtmfree;

      fxMesa->freeTexMem[tmu]-=texmemsize;

      if(!(newtmalloc=malloc(sizeof(tfxTMAllocNode)))) {
	fprintf(stderr,"fx Driver: out of memory !\n");
	fxCloseHardware();
	exit(-1);
      }
      
      newtmalloc->next=fxMesa->tmAlloc[tmu];
      newtmalloc->startAddress=startadr;
      newtmalloc->endAddress=startadr+texmemsize-1;
      newtmalloc->tObj=tObj;

      fxMesa->tmAlloc[tmu]=newtmalloc;

      return newtmalloc;
    }

    if(!fxTMFreeOldTMBlock(fxMesa,tmu)) {
      fprintf(stderr,"fx Driver: internal error in fxTMGetTMBlock()\n");
      fprintf(stderr,"           TMU: %d Size: %d\n",tmu,texmemsize);
    
      fxCloseHardware();
      exit(-1);
    }
  }
}




void fxTMMoveInTM(fxMesaContext fxMesa, struct gl_texture_object *tObj, GLint where)
{
  tfxTexInfo *ti=(tfxTexInfo *)tObj->DriverData;
  int i,l;
  int texmemsize;

#if defined(DEBUG_FXMESA)
  fprintf(stderr,"fxmesa: fxTMMoveInTM(%d)\n",tObj->Name);
#endif

  fxMesa->stats.reqTexUpload++;

  if(!ti->validated) {
    fprintf(stderr,"fx Driver: internal error in fxTMMoveInTM() -> not validated\n");
    fxCloseHardware();
    exit(-1);
  }

  if(ti->tmi.isInTM)
    return;

#if defined(DEBUG_FXMESA)
  fprintf(stderr,"fxmesa: downloading %x (%d) in texture memory in %d\n",tObj,tObj->Name,where);
#endif

  ti->tmi.whichTMU=where;

  switch(where) {
  case FX_TMU0:
  case FX_TMU1:
    texmemsize=grTexTextureMemRequired(GR_MIPMAPLEVELMASK_BOTH,&(ti->info));
    ti->tmi.tm[where]=fxTMGetTMBlock(fxMesa,tObj,where,texmemsize);
    fxMesa->stats.memTexUpload+=texmemsize;

    for(i=ti->info.largeLod,l=ti->minLevel;i<=ti->info.smallLod;i++,l++)
      grTexDownloadMipMapLevel(where,
			       ti->tmi.tm[where]->startAddress,i,
			       ti->info.largeLod,ti->info.aspectRatio,
			       ti->info.format,GR_MIPMAPLEVELMASK_BOTH,
			       ti->tmi.mipmapLevel[l].data);
    break;
  case FX_TMU_SPLIT: /* TO DO: alternate even/odd TMU0/TMU1 */
    texmemsize=grTexTextureMemRequired(GR_MIPMAPLEVELMASK_ODD,&(ti->info));
    ti->tmi.tm[FX_TMU0]=fxTMGetTMBlock(fxMesa,tObj,FX_TMU0,texmemsize);
    fxMesa->stats.memTexUpload+=texmemsize;

    texmemsize=grTexTextureMemRequired(GR_MIPMAPLEVELMASK_EVEN,&(ti->info));
    ti->tmi.tm[FX_TMU1]=fxTMGetTMBlock(fxMesa,tObj,FX_TMU1,texmemsize);
    fxMesa->stats.memTexUpload+=texmemsize;

    for(i=ti->info.largeLod,l=ti->minLevel;i<=ti->info.smallLod;i++,l++) {
      grTexDownloadMipMapLevel(GR_TMU0,ti->tmi.tm[FX_TMU0]->startAddress,i,
			       ti->info.largeLod,ti->info.aspectRatio,
			       ti->info.format,GR_MIPMAPLEVELMASK_ODD,
			       ti->tmi.mipmapLevel[l].data);

      grTexDownloadMipMapLevel(GR_TMU1,ti->tmi.tm[FX_TMU1]->startAddress,i,
			       ti->info.largeLod,ti->info.aspectRatio,
			       ti->info.format,GR_MIPMAPLEVELMASK_EVEN,
			       ti->tmi.mipmapLevel[l].data);
    }
    break;
  default:
    fprintf(stderr,"fx Driver: internal error in fxTMMoveInTM() -> wrong tmu (%d)\n",where);
    fxCloseHardware();
    exit(-1);
  }

  fxMesa->stats.texUpload++;

  ti->tmi.isInTM=GL_TRUE;
}





void fxTMReloadMipMapLevel(fxMesaContext fxMesa, struct gl_texture_object *tObj, GLint level)
{
  tfxTexInfo *ti=(tfxTexInfo *)tObj->DriverData;
  GrLOD_t lodlevel;
  GLint tmu;

  if(!ti->validated) {
    fprintf(stderr,"fx Driver: internal error in fxTMReloadMipMapLevel() -> not validated\n");
    fxCloseHardware();
    exit(-1);
  }

  tmu=ti->tmi.whichTMU;
  fxTMMoveInTM(fxMesa,tObj,tmu);

  fxTexGetInfo(ti->tmi.mipmapLevel[0].width,ti->tmi.mipmapLevel[0].height,
	       &lodlevel,NULL,NULL,NULL,NULL,NULL);

  switch(tmu) {
  case FX_TMU0:
  case FX_TMU1:
    grTexDownloadMipMapLevel(tmu,
			     ti->tmi.tm[tmu]->startAddress,lodlevel+level,
			     ti->info.largeLod,ti->info.aspectRatio,
			     ti->info.format,GR_MIPMAPLEVELMASK_BOTH,
			     ti->tmi.mipmapLevel[level].data);
    break;
  case FX_TMU_SPLIT: /* TO DO: alternate even/odd TMU0/TMU1 */
    grTexDownloadMipMapLevel(GR_TMU0,
			     ti->tmi.tm[GR_TMU0]->startAddress,lodlevel+level,
			     ti->info.largeLod,ti->info.aspectRatio,
			     ti->info.format,GR_MIPMAPLEVELMASK_ODD,
			     ti->tmi.mipmapLevel[level].data);
    
    grTexDownloadMipMapLevel(GR_TMU1,
			     ti->tmi.tm[GR_TMU1]->startAddress,lodlevel+level,
			     ti->info.largeLod,ti->info.aspectRatio,
			     ti->info.format,GR_MIPMAPLEVELMASK_EVEN,
			     ti->tmi.mipmapLevel[level].data);
    break;
  default:
    fprintf(stderr,"fx Driver: internal error in fxTMReloadMipMapLevel() -> wrong tmu (%d)\n",tmu);
    fxCloseHardware();
    exit(-1);
  }
}





void fxTMReloadSubMipMapLevel(fxMesaContext fxMesa, struct gl_texture_object *tObj,
			      GLint level, GLint yoffset, GLint height)
{
  tfxTexInfo *ti=(tfxTexInfo *)tObj->DriverData;
  GrLOD_t lodlevel;
  unsigned short *data;
  GLint tmu;

  if(!ti->validated) {
    fprintf(stderr,"fx Driver: internal error in fxTMReloadSubMipMapLevel() -> not validated\n");
    fxCloseHardware();
    exit(-1);
  }

  tmu=ti->tmi.whichTMU;
  fxTMMoveInTM(fxMesa,tObj,tmu);

  fxTexGetInfo(ti->tmi.mipmapLevel[0].width,ti->tmi.mipmapLevel[0].height,
	       &lodlevel,NULL,NULL,NULL,NULL,NULL);

  if((ti->info.format==GR_TEXFMT_INTENSITY_8) ||
     (ti->info.format==GR_TEXFMT_P_8) ||
     (ti->info.format==GR_TEXFMT_ALPHA_8))
    data=ti->tmi.mipmapLevel[level].data+((yoffset*ti->tmi.mipmapLevel[level].width)>>1);
  else
    data=ti->tmi.mipmapLevel[level].data+yoffset*ti->tmi.mipmapLevel[level].width;

  switch(tmu) {
  case FX_TMU0:
  case FX_TMU1:
    grTexDownloadMipMapLevelPartial(tmu,
				    ti->tmi.tm[tmu]->startAddress,lodlevel+level,
				    ti->info.largeLod,ti->info.aspectRatio,
				    ti->info.format,GR_MIPMAPLEVELMASK_BOTH,
				    data,
				    yoffset,yoffset+height-1);
    break;
  case FX_TMU_SPLIT: /* TO DO: alternate even/odd TMU0/TMU1 */
    grTexDownloadMipMapLevelPartial(GR_TMU0,
				    ti->tmi.tm[FX_TMU0]->startAddress,lodlevel+level,
				    ti->info.largeLod,ti->info.aspectRatio,
				    ti->info.format,GR_MIPMAPLEVELMASK_ODD,
				    data,
				    yoffset,yoffset+height-1);

    grTexDownloadMipMapLevelPartial(GR_TMU1,
				    ti->tmi.tm[FX_TMU1]->startAddress,lodlevel+level,
				    ti->info.largeLod,ti->info.aspectRatio,
				    ti->info.format,GR_MIPMAPLEVELMASK_EVEN,
				    data,
				    yoffset,yoffset+height-1);
    break;
  default:
    fprintf(stderr,"fx Driver: internal error in fxTMReloadSubMipMapLevel() -> wrong tmu (%d)\n",tmu);
    fxCloseHardware();
    exit(-1);
  }
}




static tfxTMAllocNode *fxTMFreeTMAllocBlock(tfxTMAllocNode *tmalloc,
					    tfxTMAllocNode *tmunalloc)
{
  if(!tmalloc)
    return NULL;

  if(tmalloc==tmunalloc) {
    tfxTMAllocNode *newtmalloc;

    newtmalloc=tmalloc->next;
    free(tmalloc);

    return newtmalloc;
  }

  tmalloc->next=fxTMFreeTMAllocBlock(tmalloc->next,tmunalloc);

  return tmalloc;
}




static tfxTMFreeNode *fxTMAddTMFree(tfxTMFreeNode *tmfree, FxU32 startadr, FxU32 endadr)
{
  if(!tmfree)
    return fxTMNewTMFreeNode(startadr,endadr);

  if((endadr+1==tmfree->startAddress) && (tmfree->startAddress & 0x1fffff)) {
    tmfree->startAddress=startadr;

    return tmfree;
  }

  if((startadr-1==tmfree->endAddress) && (startadr & 0x1fffff)) {
    tmfree->endAddress=endadr;

    if((tmfree->next && (endadr+1==tmfree->next->startAddress) &&
        (tmfree->next->startAddress & 0x1fffff))) {
      tfxTMFreeNode *nexttmfree;

      tmfree->endAddress=tmfree->next->endAddress;

      nexttmfree=tmfree->next->next;
      free(tmfree->next);

      tmfree->next=nexttmfree;
    }


    return tmfree;
  }

  if(startadr<tmfree->startAddress) {
    tfxTMFreeNode *newtmfree;

    newtmfree=fxTMNewTMFreeNode(startadr,endadr);
    newtmfree->next=tmfree;

    return newtmfree;
  }

  tmfree->next=fxTMAddTMFree(tmfree->next,startadr,endadr);

  return tmfree;
}




static void fxTMFreeTMBlock(fxMesaContext fxMesa, GLint tmu, tfxTMAllocNode *tmalloc)
{
  FxU32 startadr,endadr;

  startadr=tmalloc->startAddress;
  endadr=tmalloc->endAddress;

  fxMesa->tmAlloc[tmu]=fxTMFreeTMAllocBlock(fxMesa->tmAlloc[tmu],tmalloc);

  fxMesa->tmFree[tmu]=fxTMAddTMFree(fxMesa->tmFree[tmu],startadr,endadr);

  fxMesa->freeTexMem[tmu]+=endadr-startadr+1;
}




void fxTMMoveOutTM(fxMesaContext fxMesa, struct gl_texture_object *tObj)
{
  tfxTexInfo *ti=(tfxTexInfo *)tObj->DriverData;

#if defined(DEBUG_FXMESA)
  fprintf(stderr,"fxmesa: fxTMMoveOutTM(%x (%d))\n",tObj,tObj->Name);
#endif

  if(!ti->tmi.isInTM)
    return;

  switch(ti->tmi.whichTMU) {
  case FX_TMU0:
  case FX_TMU1:
    fxTMFreeTMBlock(fxMesa,ti->tmi.whichTMU,ti->tmi.tm[ti->tmi.whichTMU]);
    break;
  case FX_TMU_SPLIT:
    fxTMFreeTMBlock(fxMesa,FX_TMU0,ti->tmi.tm[FX_TMU0]);
    fxTMFreeTMBlock(fxMesa,FX_TMU1,ti->tmi.tm[FX_TMU1]);
    break;
  default:
    fprintf(stderr,"fx Driver: internal error in fxTMMoveOutTM()\n");
    fxCloseHardware();
    exit(-1);
  }

  ti->tmi.whichTMU=FX_TMU_NONE;
  ti->tmi.isInTM=GL_FALSE;
}




void fxTMFreeTexture(fxMesaContext fxMesa, struct gl_texture_object *tObj)
{
  tfxTexInfo *ti=(tfxTexInfo *)tObj->DriverData;
  int i;

  fxTMMoveOutTM(fxMesa,tObj);

  for(i=0;i<MAX_TEXTURE_LEVELS;i++) {
    if(ti->tmi.mipmapLevel[i].used &&
       ti->tmi.mipmapLevel[i].translated)
      free(ti->tmi.mipmapLevel[i].data);

    ti->tmi.mipmapLevel[i].data;
  }
}




void fxTMFreeAllFreeNode(tfxTMFreeNode *fn)
{
  if(!fn)
    return;

  if(fn->next)
    fxTMFreeAllFreeNode(fn->next);

  free(fn);
}




void fxTMFreeAllAllocNode(tfxTMAllocNode *an)
{
  if(!an)
    return;

  if(an->next)
    fxTMFreeAllAllocNode(an->next);

  free(an);
}




void fxTMClose(fxMesaContext fxMesa)
{
  fxTMFreeAllFreeNode(fxMesa->tmFree[FX_TMU0]);
  fxTMFreeAllAllocNode(fxMesa->tmAlloc[FX_TMU0]);

  if(fxMesa->haveTwoTMUs) {
    fxTMFreeAllFreeNode(fxMesa->tmFree[FX_TMU1]);
    fxTMFreeAllAllocNode(fxMesa->tmAlloc[FX_TMU1]);
  }
}

#endif




#else

int aclTM_dummy(void) { return 0; }

#endif /* ACL */

