
/*
 * -*- mode: C; tab-width:8;  -*-
 * 
 *	aclMesa.c - Mesa interface to hw acceleration library
 */

/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * The ACL Mesa Device Driver was born out of David's FX driver. Without
 * pioneer FX driver, realization of this stuff wouldn't be possible.
 *
 * Authors:
 *	simon pogarcic, <sim@suse.de>
 *	david bucciarelli, <humanware@plus.it>
 */

/******************************************************************************
 *
 * Function names:
 *
 *  Acl.........(ACL API init functions)
 *  aclMesa.....(ACL Mesa Device Driver interface functions, aclMesa.c)
 *  aclDD.......(Internal driver setup, aclDD.c)
 *  aclTM.......(Texture Manager, aclTM.c)
 *  aclSW.......(Software pixel/span functions, aclSW.c)
 *  aclTex......(Texturing setup, aclTex.c)
 *  aclPoints...(Points functions, aclPoints.c)
 *  aclLine.....(Line functions, aclLine.c)
 *  aclTriangle.(Triangle functions, aclTriangle.c)
 *  aclQuad.....(Quad functions, aclQuad.c)
 *  aclRect.....(Rectangle functions, aclRect.c)
 *  aclRender...(RenderVB functions, aclRender.c)
 *  aclSetup....(Hardware/driver setup functions, aclSetup.c)
 *
 * Data type names:
 *
 *  ACL.........(ACL API typedefs)
 *  acl_........(ACL API structs)
 *  ACLMesa.....(ACL Mesa DD typedefs)
 *  aclmesa_....(ACL Mesa DD structs)
 *  aclm_.......(ACL Mesa DD global variables)
 *
 ******************************************************************************/

#if defined(ACL)

#include "aclDrv.h"



#if 0
#define LOG_DEBUG
#endif

#define LOG_DBGINFO "<aclMesa>"
#include "log_debug.h"



ACLMesaContext aclm_current_context = NULL;
ACLMesaSighandler aclm_old_sighandler = NULL;



static void
aclMesaSighandler(int nr)
{
LOG("aclMesaSighandler(): ######\n");

    MSG("signal %d caught\n", nr);
    if (nr == SIGINT) {
	signal(nr, aclm_old_sighandler);
	exit(0);
    }
}



static void
aclMesaOnExit( void )
{
LOG("aclMesaOnExit(): ######\n");

    aclMesaDestroyContext(aclm_current_context);
}



static Tint
aclStrcmp(const void *s1, const void *s2)
{
    return strcasecmp(s1, s2);
}



/*
 * This function gathers configuration strings from environment variables and
 * sets appropriate configuration flags.
 */

static ACLflags
aclMesaGetConfig( Tint *hwidx )
{
    ACLflags xdga, config = 0;
    Tbyte *env;
    Tint SETDEF;

LOG("aclMesaGetConfig: ######\n");

    if((env = getenv("ACLOPT_HWIDX"))) *hwidx=atoi(env);
    else return ACLOPT_HWIDX_LIST;

    if(*hwidx == -1) return ACLOPT_HWIDX_LIST;

/**************************************/
    SETDEF=1;
    xdga = ACLOPT_XDGA_ON;
    if((env = getenv("ACLOPT_RUNMODE"))) {
	if(!aclStrcmp(env,"off")) return ACLOPT_RUNMODE_OFF;
	else if(!aclStrcmp(env,"win")) {
#if 0
	    config |= ACLOPT_RUNMODE_WIN;
	    xdga=ACLOPT_XDGA_OFF;
	    SETDEF=0;
#else
	    MSG("\n");
	    MSG("*****************************************\n");
	    MSG("* Window rendering not implemented yet! *\n");
	    MSG("* Rendering fullscreen (DEFAULT)...     *\n");
	    MSG("*****************************************\n");
	    MSG("\n");
#endif
	}
	else if(!aclStrcmp(env,"full")) ;
	else MSG("ACLOPT_RUNMODE: Bad option '%s'\n", env);
    }
    if(SETDEF) config |= ACLOPT_RUNMODE_FULL;

/**************************************/
    SETDEF=1;
    if((env = getenv("ACLOPT_VIDMODE"))) {
	if(!aclStrcmp(env,"off")) ;
	else if(!aclStrcmp(env,"on")) {
	    config |= ACLOPT_VIDMODE_ON;
	    SETDEF=0;
	}
	else MSG("ACLOPT_VIDMODE: Bad option '%s'\n", env);
    }
    if(SETDEF) config |= ACLOPT_VIDMODE_OFF;

/**************************************/
    SETDEF=1;
    if((env = getenv("ACLOPT_XDGA"))) {
	if(!aclStrcmp(env,"off")) {
	    config |= ACLOPT_XDGA_OFF;
	    SETDEF=0;
	}
	else if(!aclStrcmp(env,"on")) {
	    if(config & ACLOPT_RUNMODE_WIN ) {
		MSG("\n");
		MSG("### XDGA('on') and RUNMODE('win') not possible !!!\n");
		MSG("### Setting XDGA('off')\n");
		MSG("\n");
	    }
	}
	else MSG("ACLOPT_XDGA: Bad option '%s'\n", env);
    }
    if(SETDEF) config |= xdga;

/**************************************/
    SETDEF=1;
    if((env = getenv("ACLOPT_QUAKE"))) {
	if(!aclStrcmp(env,"off")) ;
	else if(!aclStrcmp(env,"on")) {
	    config |= ACLOPT_QUAKE_ON;
	    SETDEF=0;
	}
	else MSG("ACLOPT_QUAKE: Bad option '%s'\n", env);
    }
    if(SETDEF) config |= ACLOPT_QUAKE_OFF;

/**************************************/
    SETDEF=1;
    if((env = getenv("ACLOPT_DBUFF"))) {
	if(!aclStrcmp(env,"bb")) ;
	else if(!aclStrcmp(env,"bp")) {
	    config |= ACLOPT_DBUFF_BITPLANE;
	    SETDEF=0;
	}
	else if(!aclStrcmp(env,"scr")) {
	    config |= ACLOPT_DBUFF_SCREEN;
	    SETDEF=0;
	    if(config & ACLOPT_RUNMODE_WIN ) {
		MSG("\n");
		MSG("### ACLOPT_DBUFF=scr and ACLOPT_RUNMODE=win not possible !!!\n");
		MSG("### Setting default (ACLOPT_DBUFF='bb')\n");
		MSG("\n");
		SETDEF=1;
	    }
	}
	else MSG("ACLOPT_DBUFF: Bad option '%s'\n", env);
    }
    if(SETDEF) config |= ACLOPT_DBUFF_BITBLIT;

LOG("aclMesaGetConfig(): ------> config = 0x%x <------\n", config);
    return config;
}



/*
 * Create a new ACL Mesa Device Driver context and return a handle to it.
 */
ACLMesaContext APIENTRY
aclMesaCreateContext(Display *dpy, Window win, GLint width, GLint height,
	ACLMesaAttribs attr)
{
    ACLMesaContext aclMesa;

    ACLContext aclctx;
    ACLModules aclmod;
    ACLHardware aclhw;
    ACLVisual aclvis;
    ACLScreen aclscr;

    ACLflags config;

    GLboolean is_rgba, is_dbuffer, is_alpha, is_stereo;
    GLint r_bits, g_bits, b_bits, a_bits, i_bits;
    GLint depth_bits, stencil_bits, accum_bits;
    GLvisual *glVis;

    Tint hwidx = 0;

    struct vertex_buffer *VB;

/******************************************************************************
 * before doing anything, gather config flags and setup the signal handling
 ******************************************************************************/
LOG("aclMesaCreateContext(): ######\n");

    config = aclMesaGetConfig( &hwidx );

    if(config & ACLOPT_RUNMODE_OFF) {
	MSG("\n");
	MSG("ACLOPT_RUNMODE=off: ACL driver switched off\n");
	MSG("    - ACL driver won't be used for rendering...\n");
	MSG("\n");
	return NULL;
    }

    if(atexit(aclMesaOnExit)) {
	MSG("\n");
    	MSG("WARNING: Cannot set atexit() routine!\n");
	MSG("\n");
    }

    aclm_old_sighandler =
    	signal( SIGINT,  aclMesaSighandler );

#if 0
	signal( SIGUSR1, aclMesaSighandler );
	signal( SIGHUP,  aclMesaSighandler );
	signal( SIGQUIT, aclMesaSighandler );
	signal( SIGILL,  aclMesaSighandler );
	signal( SIGABRT, aclMesaSighandler );
	signal( SIGFPE,  aclMesaSighandler );
	signal( SIGKILL, aclMesaSighandler );
	signal( SIGSEGV, aclMesaSighandler );
	signal( SIGPIPE, aclMesaSighandler );
	signal( SIGALRM, aclMesaSighandler );
	signal( SIGTERM, aclMesaSighandler );
	signal( SIGUSR2, aclMesaSighandler );
	signal( SIGSTOP, aclMesaSighandler );
	signal( SIGTTIN, aclMesaSighandler );
	signal( SIGTTOU, aclMesaSighandler );
#endif

/******************************************************************************
 * the connection with ACL library and kernel module 
 ******************************************************************************/
    if ( AclEnter() ) {
	MSG("\n");
	MSG("[ ERROR ] Low-level init failed (aclEnter)\n");
	MSG("    - ACL driver won't be used for rendering...\n");
	MSG("\n");
	return NULL;
    }

/******************************************************************************
 * list and check the hardware
 ******************************************************************************/
    if( !(aclhw = AclListHardware(-1))) {
	MSG("\n");
	MSG("*** SUPPORTED HARDWARE NOT FOUND ***\n");
	MSG("    - ACL driver won't be used for rendering...\n");
	MSG("\n");
	return NULL;
    }
    else {
	MSG("\n");
	MSG(" HWIDX CARD                             BUS SLOT RUNSTAT\n");
	MSG("---------------------------------------------------------\n");
	while(aclhw) {
	    const Tbyte *runstat = aclhw->running_fullscr ? "full" : "none";
	    MSG(" %3d.  %-32s  %-2d  %-2d   %-4s\n",
		aclhw->hwidx, aclhw->name, aclhw->bus, aclhw->slot, runstat);
	    aclhw = aclhw->next;
	}
	MSG("\n");
    }

    if(config & ACLOPT_HWIDX_LIST) {
	MSG("\n");
	MSG("=========================================================\n");
	MSG("= ACLOPT_HWIDX not defined or -1. Specify some hw index =\n");
	MSG("= to render. #### EXAMPLE: 'export ACLOPT_HWIDX=0' #### =\n");
	MSG("=========================================================\n");
	MSG("\n");
	exit(0);
    }

    if( !(aclhw = AclListHardware(hwidx))) {
	MSG("\n");
	MSG("ACLOPT_HWIDX=%d: Invalid hardware index\n", hwidx);
	MSG("    - ACL driver won't be used for rendering...\n");
	MSG("\n");
	return NULL;
    }

    if(aclhw->running_fullscr) {
	MSG("\n");
	MSG("*** FULLSCREEN ALREADY RUNNING ON CARD (%d)!!! ***\n", hwidx);
	MSG("    - Rendering not possible, exit\n");
	MSG("\n");
	exit(0);
    }
    
/*TODO: list features of hardware */

/******************************************************************************
 * setup visual attributes
 ******************************************************************************/
LOG("aclMesaCreateContext(): setup visual parameters\n");
    is_rgba = attr->rgba;
    is_dbuffer = attr->dbuff;
    is_stereo = attr->stereo;
    is_alpha = (attr->a > 0);

    r_bits = attr->r;
    g_bits = attr->g;
    b_bits = attr->b;
    a_bits = attr->a;
    i_bits = attr->i;

    depth_bits = attr->depth;
    stencil_bits = attr->stencil;
    accum_bits = attr->accum;
/*
    is_frontalpha = attr->frontalpha;
    is_backalpha = attr->backalpha;
*/
    /* A workaround for Linux GLQuake * /
    if(depth_bits && is_alpha)
	is_alpha = 0;
	    
    */

/******************************************************************************
 * create ACL API screen and visual records required for setting of ACLContext:
 ******************************************************************************/
LOG("aclMesaCreateContext(): ACLVisual (malloc)\n");
    if(!(aclvis = (ACLVisual) malloc(sizeof(struct acl_visual)))) {
	AclExit();
	return NULL;
    }

LOG("aclMesaCreateContext(): ACLScreen (malloc)\n");
    if(!(aclscr = (ACLScreen) malloc(sizeof(struct acl_screen)))) {
	free(aclvis);
	AclExit();
	return NULL;
    }
 
/******************************************************************************
 * alloc and setup required Mesa structures and contexts
 ******************************************************************************/
LOG("aclMesaCreateContext(): ACLMesaContext (malloc)\n");
    if(!(aclMesa = (ACLMesaContext) malloc(sizeof(struct aclmesa_context)))) {
	free(aclscr);
	free(aclvis);
	AclExit();
	return NULL;
    }

LOG("aclMesaCreateContext(): gl_create_visual()\n");
    glVis = aclMesa->glVis = gl_create_visual(
	GL_TRUE,	/* is_rgba */
	is_alpha,
	is_dbuffer,
	GL_FALSE,	/* is_stereo */
	depth_bits,
	stencil_bits,
	accum_bits,
	0,		/* i_bits, */
	r_bits,
	g_bits,
	b_bits,
	a_bits
    );
    
    if( !aclMesa->glVis ) {
	free(aclscr);
	free(aclvis);
	free(aclMesa);
	AclExit();
	return NULL;
    }

LOG("aclMesaCreateContext(): gl_create_context()\n");
    aclMesa->glCtx = gl_create_context(
	aclMesa->glVis,
	NULL,		/* share list context */
	(void *) aclMesa,
	GL_TRUE
    );

    if( !aclMesa->glCtx ) {
	gl_destroy_visual(glVis);
	free(aclscr);
	free(aclvis);
	free(aclMesa);
	AclExit();
	return NULL;
    }

    VB = aclMesa->glCtx->VB;

LOG("aclMesaCreateContext(): gl_create_framebuffer()\n");
    aclMesa->glBuffer = gl_create_framebuffer(glVis);

    if( !aclMesa->glBuffer ) {
	gl_destroy_visual(glVis);
	gl_destroy_context(aclMesa->glCtx);
	free(aclscr);
	free(aclvis);
	free(aclMesa);
	AclExit();
	return NULL;
    }

/******************************************************************************
 * pre-setup of aclMesa context
 ******************************************************************************/
    memcpy(&aclMesa->hwf, aclhw->hwf, sizeof(struct acl_features));
    aclMesa->Scr = aclscr;
    aclMesa->acl_cfg = config;

/******************************************************************************
 * Setup X mode
 ******************************************************************************/
    if(aclXInitVidModes(aclMesa, dpy, width, height )) {
	MSG("\n");
    	MSG("WARNING: Cannot setup X modes!\n");
	MSG("\n");
    }

    aclscr->WindowWidth = width;
    aclscr->WindowHeight = height;

/******************************************************************************
 * Setup visual data
 ******************************************************************************/
    aclvis->RGBAflag = 1;		/* glVis->RGBAflag; */
    aclvis->DBflag = is_dbuffer;	/* glVis->DBflag; */
    aclvis->StereoFlag = 0;		/* glVis->StereoFlag; */

    aclvis->RedBits = glVis->RedBits;
    aclvis->GreenBits = glVis->GreenBits;
    aclvis->BlueBits = glVis->BlueBits;
    aclvis->AlphaBits = glVis->AlphaBits;
    aclvis->IndexBits = glVis->IndexBits;

    aclvis->AccumBits = glVis->AccumBits;
    aclvis->DepthBits = glVis->DepthBits;
    aclvis->StencilBits = glVis->StencilBits;

    aclvis->FrontAlphaFlag = glVis->FrontAlphaEnabled;
    aclvis->BackAlphaFlag = glVis->BackAlphaEnabled;

/******************************************************************************
 * CREATE ACL API LOW-LEVEL CONTEXT !!!
 ******************************************************************************/
LOG("aclMesaCreateContext(): AclCreateContext()\n");
    if( !(aclctx = AclCreateContext(hwidx, aclscr, aclvis, config))) {
	aclXSetDefaultVidModes();
	gl_destroy_visual(glVis);
	gl_destroy_context(aclMesa->glCtx);
	gl_destroy_framebuffer(aclMesa->glBuffer);
	free(aclscr);
	free(aclvis);
	free(aclMesa);
	AclExit();
	return NULL;
    }

/******************************************************************************
 * setup the rest of data in aclMesa context
 ******************************************************************************/
    aclmod = aclctx->Mod;

/* TODO:
    check aclmod->modflags
 */
    aclMesa->vb_pointers.win = VB->Win;
    aclMesa->vb_pointers.clip = VB->Clip;
    aclMesa->vb_pointers.index = VB->Index;
    aclMesa->vb_pointers.tex = VB->TexCoord;

    aclMesa->aclctx = aclctx;
    aclMesa->aclmod = aclmod;
    aclMesa->aclaccel_mod = aclmod->Accel;
    aclMesa->acllb_mod = aclmod->LB;
    aclMesa->acltex_mod = aclmod->Tex;
    aclMesa->aclrgba_mod = aclmod->ColorRGBA;
    aclMesa->aclci_mod = aclmod->ColorCI;

    aclMesa->width = width;
    aclMesa->height = height;
    aclMesa->is_alpha = is_alpha;
    aclMesa->is_dbuffer = aclvis->DBflag;
    aclMesa->is_globtexpal = 0;
    aclMesa->verbose = 1;

    aclMesa->TMCtx = NULL;

/******************************************************************************
 * 3, 2, 1,... blurrrps. START RENDERING !!!
 ******************************************************************************/

    aclmod->StartRender(aclctx);

    aclDDSetupDDPointers(aclMesa->glCtx);
    aclmod->ColorRGBA->ColorMask
	(aclctx, 0xff, 0xff, 0xff, (is_alpha ? 0xff : 0x00));
    if(aclvis->DBflag)
	aclmod->DrawBuffer(aclctx, ACLGL_BACK);
    else
	aclmod->DrawBuffer(aclctx, ACLGL_FRONT);

LOG("aclMesaCreateContext(): ______\n");

    return aclMesa;
}

 

/*
 * Destroy the given Mesa context.
 */
void APIENTRY
aclMesaDestroyContext(ACLMesaContext aclMesa)
{
    ACLModules aclmod;

LOG("aclMesaDestroyContext(): ######\n");

    aclXSetDefaultVidModes();

    if (!aclMesa) {
	AclExit();
	return;
    }

    aclMesa->aclmod->FinishRender(aclMesa->aclctx);
    AclDestroyContext(aclMesa->aclctx);

    if (aclMesa == aclm_current_context)
	aclm_current_context = NULL;

    /*
     * at the end, always call the folowing function:
     */
    gl_destroy_visual(aclMesa->glVis);
    gl_destroy_context(aclMesa->glCtx);
    gl_destroy_framebuffer(aclMesa->glBuffer);

LOG("aclMesaDestroyContext(): ______\n");
    AclExit();
}



/*
 * Swap front/back buffers for current context if double buffered.
 */
void APIENTRY
aclMesaSwapBuffers(ACLMesaContext aclMesa)
{
    ACLMesaContext aclMesaC = aclm_current_context;

LOG("aclMesaSwapBuffers(): ######\n");

    if(aclMesaC)
	if(aclMesaC->is_dbuffer) {
	    aclMesaC->aclmod->SwapBuffer(aclMesaC->aclctx);
	}

LOG("aclMesaSwapBuffers(): ______\n");
}



/*
 * Make the specified aclMesa context the current one.
 */
void APIENTRY
aclMesaMakeCurrent(ACLMesaContext aclMesa)
{
LOG("aclMesaMakeCurrent(): ######\n");

    if (!aclMesa) {
	gl_make_current(NULL, NULL);
	aclm_current_context = NULL;
	return;
    }

    if (aclm_current_context == aclMesa) {
	return;
    }

    aclm_current_context = aclMesa;
    gl_make_current(aclMesa->glCtx, aclMesa->glBuffer);

    aclDDSetupDDPointers(aclMesa->glCtx);

    /*
     * The first time we call MakeCurrent we set the initial viewport size 
     */
    if (aclMesa->glCtx->Viewport.Width == 0)
	gl_Viewport(aclMesa->glCtx, 0, 0, aclMesa->width, aclMesa->height);

LOG("aclMesaMakeCurrent(): ______\n");
}



#else

int aclApi_dummy(void) { return 0; }

#endif /* ACL */

