
/*
 * -*- mode: C; tab-width:8;  -*-
 * 
 *	aclDD.c - Mesa device driver functions
 */

/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * See the file aclMesa.c for more informations about authors
 */

#if defined(ACL)

#include "aclDrv.h"



#if 0
#define LOG_DEBUG
#endif

#define LOG_DBGINFO "<aclDD>"
#include "log_debug.h"



/*****************************************************************************
 aclDDGetBufferSize()
 -----------------------------------------------------------------------------
 	-return buffer size information
 *****************************************************************************/
void
aclDDGetBufferSize(GLcontext *ctx, GLuint *width, GLuint *height)
{
    ACLMesaContext aclMesa = (ACLMesaContext) ctx->DriverCtx;

LOG("aclDDGetBufferSize(): ######\n");

    *width = aclMesa->width;
    *height = aclMesa->height;

LOG("aclDDGetBufferSize(): ______\n");
}



/*****************************************************************************
 aclDDSetColor()
 -----------------------------------------------------------------------------
 	-set current drawing color (RGBA)
 *****************************************************************************/
static void
aclDDSetColor(GLcontext *ctx, GLubyte red, GLubyte green,
	GLubyte blue, GLubyte alpha)
{
    ACLMesaContext aclMesa = (ACLMesaContext) ctx->DriverCtx;

LOG("aclDDSetColor(): ######\n");
LOG("aclDDSetColor(): RGBA=(%d,%d,%d,%d)\n", red, green, blue, alpha);

    aclMesa->aclrgba_mod->Color(aclMesa->aclctx, red, green, blue, alpha);
}



/*****************************************************************************
 aclDDClearColor()
 -----------------------------------------------------------------------------
 	-set current clearing color (RGBA)
 *****************************************************************************/
static void
aclDDClearColor(GLcontext *ctx, GLubyte red, GLubyte green,
	GLubyte blue, GLubyte alpha)
{
    ACLMesaContext aclMesa = (ACLMesaContext) ctx->DriverCtx;

LOG("aclDDClearColor(): ######\n");
LOG("aclDDClearColor(): RGBA=(%d,%d,%d,%d)\n", red, green, blue, alpha);

    aclMesa->aclrgba_mod->ClearColor(aclMesa->aclctx, red, green, blue, alpha);
}



/*****************************************************************************
 aclDDClearDepth()
 -----------------------------------------------------------------------------
 	-set current clear depth value
 *****************************************************************************/
static void
aclDDClearDepth(ACLMesaContext aclMesa, GLfloat depth )
{
LOG("aclDDClearDepth(): ######\n");
LOG("aclDDClearDepth(): depth=(%f)\n", depth);

    aclMesa->acllb_mod->ClearDepth(aclMesa->aclctx, (Tfloat) depth);
}



/*****************************************************************************
 aclDDClear()
 -----------------------------------------------------------------------------
 	-clear color and/or depth buffers
 *****************************************************************************/
static GLbitfield
aclDDClear(GLcontext *ctx, GLbitfield mask, GLboolean all,
	GLint x, GLint y, GLint width, GLint height )
{
    ACLMesaContext aclMesa = (ACLMesaContext) ctx->DriverCtx;
    ACLContext aclctx = aclMesa->aclctx;
    ACLModules aclmod = aclMesa->aclmod;
    GLbitfield newmask;
    Tubyte alpha_mask = aclMesa->is_alpha ? 0xff : 0x00;

LOG("aclDDClear(): ######\n");
LOG("aclDDClear(): x,y,w,h=(%d,%d,%d,%d)\n", x, y, width, height);

    aclMesa->aclrgba_mod->ColorMask(aclctx, 0xff, 0xff, 0xff, alpha_mask);
    aclMesa->acllb_mod->DepthMask(aclctx, ACLGL_TRUE);

    aclDDClearDepth(aclMesa, ctx->Depth.Clear);

    if(ctx->RasterMask & FRONT_AND_BACK_BIT) {
	aclmod->DrawBuffer(aclctx, ACLGL_BACK);
	aclmod->ClearBuffer(aclctx, mask, all, x, y, width, height);
	aclmod->DrawBuffer(aclctx, ACLGL_FRONT);
    }

    newmask = (GLboolean)
	aclmod->ClearBuffer(aclctx, mask, all, x, y, width, height);

    if(!ctx->Depth.Mask)
	aclMesa->acllb_mod->DepthMask(aclctx, ACLGL_FALSE);

    if(!ctx->Color.ColorMask)
	aclMesa->aclrgba_mod->ColorMask(aclctx, 0, 0, 0, 0);

    return newmask;
}



/*****************************************************************************
 aclDDSetBuffer
 -----------------------------------------------------------------------------
 	-set buffer for rendering
 *****************************************************************************/
static GLboolean
aclDDSetBuffer(GLcontext * ctx, GLenum mode)
{
    ACLMesaContext aclMesa = (ACLMesaContext) ctx->DriverCtx;

LOG("aclDDSetBuffer(): ######\n");

    return aclMesa->aclmod->DrawBuffer(aclMesa->aclctx, mode);
}



/*****************************************************************************
 aclDDRendererString
 -----------------------------------------------------------------------------
 	-return string info about renderer
 *****************************************************************************/
static const char *
aclDDRendererString(void)
{
    static char *renderer = 
	"\n*** "
	"\n*** " _ACL_API_VERSION_STRING_
	"\n*** \n";

LOG("aclDDRendererString(): ######\n");

    return renderer;
}



/*****************************************************************************
 aclDDExtensionString
 -----------------------------------------------------------------------------
 	-return string info about supported renderer extensions
 *****************************************************************************/
static const char *
aclDDExtensionString(GLcontext *ctx)
{
    static char *extensions="GL_EXT_blend_color GL_EXT_blend_minmax GL_EXT_blend_logic_op GL_EXT_blend_subtract GL_EXT_paletted_texture GL_EXT_point_parameters GL_EXT_polygon_offset GL_EXT_vertex_array GL_EXT_texture_object GL_EXT_texture3D GL_MESA_window_pos GL_MESA_resize_buffers GL_EXT_shared_texture_palette GL_EXT_rescale_normal GL_EXT_abgr GL_SGIS_texture_edge_clamp GL_SGIS_multitexture GL_EXT_multitexture 3DFX_set_global_palette GL_FXMESA_global_texture_lod_bias";

LOG("aclDDExtensionString(): ######\n");

    return extensions;
}



/*****************************************************************************
 aclDDDither
 -----------------------------------------------------------------------------
 	-setup dither mode
 *****************************************************************************/
void aclDDDither(GLcontext *ctx, GLboolean enable)
{
    ACLMesaContext aclMesa = (ACLMesaContext) ctx->DriverCtx;

LOG("aclDDDither(): ######\n");
}



/*****************************************************************************
 aclDDGetParameteri
 -----------------------------------------------------------------------------
 	-return some informations about supported features
 *****************************************************************************/
static GLint aclDDGetParameteri(const GLcontext *ctx, GLint param)
{
    ACLMesaContext aclMesa = (ACLMesaContext) ctx->DriverCtx;

LOG("aclDDGetParameteri(): ######\n");

    switch(param) {
    case DD_MAX_TEXTURE_SIZE:
	return aclMesa->hwf.maxtexsize;
    case DD_MAX_TEXTURES:
	return aclMesa->hwf.tmunits;
    case DD_HAVE_HARDWARE_FOG:
	return (aclMesa->hwf.fog & ACLHW_FOG);
    case DD_MAX_TEXTURE_COORD_SETS:
	return aclMesa->hwf.maxtexsets;
    default:
	MSG("[FATAL] aclDDGetParameteri(), param=(%d)\n", param)
	exit(-1);
  }
  return 0;
}



/*****************************************************************************
 aclDDNearFar
 -----------------------------------------------------------------------------
 	-set near/far range for fogging
 *****************************************************************************/
void aclDDNearFar(GLcontext *ctx, GLfloat n, GLfloat f)
{
    ACLMesaContext aclMesa = (ACLMesaContext) ctx->DriverCtx;

LOG("aclDDNearFar(): ######\n");
}



/*****************************************************************************
 aclDDAllocDepthBuffer
 -----------------------------------------------------------------------------
 	-allocates depth buffer for the case that hardware doesn't support it.
 *****************************************************************************/
static void
aclDDAllocDepthBuffer(GLcontext *ctx)
{
LOG("aclDDAllocDepthBuffer(): ######\n");
}



/*****************************************************************************
 aclDDFinish
 -----------------------------------------------------------------------------
 	-called whenever glFinish() is called
 *****************************************************************************/
static void
aclDDFinish(GLcontext *ctx)
{
LOG("aclDDFinish(): ######\n");
}



/*****************************************************************************
 aclDDIsInHardware()
 -----------------------------------------------------------------------------
	-called to check if rendering could be done through accelerator unit
 *****************************************************************************

 Tried to write this function as generic as possible. Sure, if we would like to
 check all possible hardware features, we could end up in an enormous 'if'
 monster. Therefore, we simply suppose that the most of reasonable 3d hardware
 covers the most of expected features. Soon or later, when some cooler 3d toys
 appear in the world, this functions could be thrown away. The hw checking
 happens by using the folowing logic: 

    if( some_feature_mesa_requires )
    {
	if( not_aclhw_that_feature )
	    return GL_FALSE;

	if( some_subfeature_mesa_requires_1 && not_aclhw_that_subfeature1 )
	    return GL_FALSE;

	if( some_subfeature_mesa_requires_2 && not_aclhw_that_subfeature2 )
	    return GL_FALSE;

 or even inverted:

	if( not_some_subfeature_mesa_requires && aclhw_that_subfeature_only )
	    return GL_FALSE;
    }
 *****************************************************************************/
Tbool
aclDDIsInHardware(GLcontext *ctx)
{
    ACLMesaContext aclMesa = (ACLMesaContext) ctx->DriverCtx;
    GLuint tflag;
    ACLflags hwf;
    ACLflags modflags = aclMesa->aclmod->modflags;

LOG("aclDDIsInHardware(): ###### Start ######\n")

/*----------------------------------------------------------------------------
 * ACCELERATED ???
 *---------------------------------------------------------------------------*/
    if(!(modflags & ACLMOD_ACCEL))
	return GL_FALSE;

/*----------------------------------------------------------------------------
 * Depth ?
 *---------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------
 * Stencil ?
 *---------------------------------------------------------------------------*/
    if(ctx->RasterMask & STENCIL_BIT)
    {
	hwf = aclMesa->hwf.stencil;
 	if(!(hwf & ACLHW_STENCIL)) {
LOG("Not supported: ===> ACLHW_STENCIL\n");
	    return GL_FALSE;
	}
    }

/*----------------------------------------------------------------------------
 * Blend ?
 *---------------------------------------------------------------------------*/
    if(ctx->Color.BlendEnabled)
    {
	hwf = aclMesa->hwf.blend;
	if(!(hwf & ACLHW_BLEND)) {
LOG("Not supported: ===> ACLHW_BLEND\n");
	    return GL_FALSE;
	}

	if(ctx->Color.BlendEquation != GL_FUNC_ADD_EXT &&
		(hwf & ACLHW_BLEND_ADD_EXT_ONLY)) {
LOG("Not supported: ===> not ACLHW_BLEND_ADD_EXT\n");
	    return GL_FALSE;
	}
    }

/*----------------------------------------------------------------------------
 * Logic Ops ?
 *---------------------------------------------------------------------------*/
    if(ctx->Color.ColorLogicOpEnabled)
    {
	hwf = aclMesa->hwf.logop;
	if(!(hwf & ACLHW_LOGOP)) {
LOG("Not supported: ===> ACLHW_LOGOP\n");
	    return GL_FALSE;
	}

	if(ctx->Color.LogicOp != GL_COPY &&
		(hwf & ACLHW_LOGOP_COPY_ONLY) ) {
LOG("Not supported: ===> not ACLHW_LOGOP_COPY\n");
	    return GL_FALSE;
	}
    }

/*----------------------------------------------------------------------------
 * Light model ?
 *---------------------------------------------------------------------------*/
    if(ctx->Light.Model.ColorControl == GL_SEPARATE_SPECULAR_COLOR)
    {
	hwf = aclMesa->hwf.color;
	if(!(hwf & ACLHW_COLOR_SEPARATE_SPECULAR)) {
LOG("Not supported: ===> ACLHW_COLOR_SEPARATE_SPECULAR\n");
	    return GL_FALSE;
	}
    }

/*----------------------------------------------------------------------------
 * ColorMask ?
 *---------------------------------------------------------------------------*/
    if( !(ctx->Color.ColorMask[0] == ctx->Color.ColorMask[1] &&
	  ctx->Color.ColorMask[1] == ctx->Color.ColorMask[2] &&
	  ctx->Color.ColorMask[2] == ctx->Color.ColorMask[3]) )
    {
	hwf = aclMesa->hwf.color;
	if(!(hwf & ACLHW_COLOR_MASK)) {
LOG("Not supported: ===> ACLHW_COLOR_MASK\n");
	    return GL_FALSE;
	}
    }

/*----------------------------------------------------------------------------
 * Texturing ?
 *---------------------------------------------------------------------------*/
    if((tflag = ctx->Texture.Enabled))
    {
	Tuint tmu, tmus = aclMesa->hwf.tmunits;
	static GLuint texd[ACLDIM_MAX_TMUNITS][3] = {
	    {TEXTURE0_1D, TEXTURE0_2D, TEXTURE0_3D},
	    {TEXTURE1_1D, TEXTURE1_2D, TEXTURE1_3D}
	};

	for(tmu=0; tmu<tmus; tmu++) {
	    hwf = aclMesa->hwf.tex[tmu];

	    /*
	     * Is texturing or even multitexturing required ?  checks common
	     * _1D, _2D, and _3D texture settings for texture set 'tmu'
	     ***************************************************************/

	    if(tflag & (texd[tmu][0] | texd[tmu][1] | texd[tmu][2])) {
		if(!(hwf & ACLHW_TEX)) {
LOG("Not supported: ===> ACLHW_TEX (TMU %d)\n",tmu);
		    return GL_FALSE;
		}

		if(ctx->Texture.Set[tmu].EnvMode != GL_MODULATE &&
			(hwf & ACLHW_TEX_MODULATE_ONLY) ) {
LOG("Not supported: ===> not ACLHW_TEX_MODULATE\n");
		    return GL_FALSE;
		}	
	    }

	    /*
	     * checks _1D texture settings
	     *****************************/

	    if(tflag & texd[tmu][0]) {
		if(!(hwf & ACLHW_TEX_1D)) {
LOG("Not supported: ===> ACLHW_TEX_1D\n");
		    return GL_FALSE;
		}
	    }

	    /*
	     * checks _2D texture settings
	     *****************************/

	    if(tflag & texd[tmu][1]) {
		if(!(hwf & ACLHW_TEX_2D)) {
LOG("Not supported: ===> ACLHW_TEX_2D\n");
		    return GL_FALSE;
		}

		if(ctx->Texture.Set[tmu].Current2D->Complete &&
			ctx->Texture.Set[0].EnvMode == GL_BLEND &&
			!(hwf & ACLHW_TEX_DECAL)) {
LOG("Not supported: ===> ACLHW_TEX_DECAL\n");
		    return GL_FALSE;
		}
	    }

	    /*
	     * checks _3D texture settings
	     *****************************/

	    if(tflag & texd[tmu][2]) {
		if(!(hwf & ACLHW_TEX_3D)) {
LOG("Not supported: ===> ACLHW_TEX_3D\n");
		    return GL_FALSE;
		}
	    }
	}
    }

LOG("aclDDIsInHardware(): ______ End ______\n")

    return GL_TRUE;
}



#if 0

    if (ctx->Color.ColorLogicOpEnabled) {
	accelmod->SetLogOp(aclMesa, GL_TRUE, ctx->Color.LogicOp);
    }

    accelmod->SetAlpha(aclMesa, 0, ctx->Color.AlphaFunc, 
		       ctx->Color.AlphaRef);
    accelmod->SetBlend(aclMesa, 0, ctx->Color.BlendSrc,
		       ctx->Color.BlendDst);
    accelmod->SetFog(aclMesa, 0,
	ctx->FogMode, ctx->Fog.Density,
	ctx->Fog.Color[0], 
	ctx->Fog.Color[1], 
	ctx->Fog.Color[2],
	ctx->Fog.Color[3]);

#endif



/*****************************************************************************
 aclDDUpdateACLPointers()
 -----------------------------------------------------------------------------
 	- called to update accelerated primitives pointers
 *****************************************************************************/
void
aclDDUpdateACLPointers(GLcontext * ctx)
{
    ACLMesaContext aclMesa = (ACLMesaContext) ctx->DriverCtx;
    ACLContext aclctx = aclMesa->aclctx;
    struct acl_render ff;
    ACLflags fflags = 0;

LOG("aclDDUpdateACLPointers(): ######\n");

    if(ctx->Light.ShadeModel == GL_SMOOTH) {
	fflags |= ACLRENDER_SMOOTH;
	if(ctx->Light.Model.TwoSide) {
	    fflags |= ACLRENDER_TWOSIDE;
	}
    }
    else {
	fflags |= ACLRENDER_FLAT;
    }

    if(ctx->Texture.Enabled) {
	fflags |= ACLRENDER_TEX;
    }

    if (ctx->Color.ColorLogicOpEnabled) {
    }

    ff.points = ff.line = ff.triangle = ff.quad = ff.rect = fflags;

/*
 * TODO: cases for vb primitives
 */
    ff.vblines = ff.vbtriangles = ff.vbquads = ff.vbrects = fflags;

/*
 * if there is some non-existing primitive detected, save this information
 * for later usage in functions for choosing primitives (which than returns
 * NULL)
 */
    aclMesa->no_primitive = aclMesa->aclaccel_mod->UpdateFuncPointers(aclctx,
	ACLPRIM_ALL_SINGLE, &ff);

LOG("aclDDUpdateACLPointers(): ______\n");
}



void
aclDDUpdateDDPointers(GLcontext * ctx)
{
    ACLMesaContext aclMesa = (ACLMesaContext) ctx->DriverCtx;

LOG("aclDDUpdateDDPointers(): ###### Start ######\n");
LOG("aclDDUpdateDDPointers(): ctx->NewState: (%d)\n", ctx->NewState); 

    if(aclDDIsInHardware(ctx)) {
LOG("aclDDUpdateDDPointers(): ACCELERATED RENDERING\n"); 

	/* accelerated functions */

	if(ctx->NewState == NEW_DRVSTATE0) {
	    /* Only texture changed */
	    aclSetupTexture(ctx);

	    ctx->Driver.PointsFunc = aclMesa->fpoints;
	    ctx->Driver.LineFunc = aclMesa->fline;
	    ctx->Driver.TriangleFunc = aclMesa->ftriangle;
	    ctx->Driver.QuadFunc = aclMesa->fquad;
	    ctx->Driver.RectFunc = aclMesa->frect;
	    ctx->Driver.RenderVB = aclMesa->frendervb;
	}
	else {
	    aclSetupUnits(ctx);
	    aclDDUpdateACLPointers(ctx);

	    ctx->Driver.PointsFunc = aclMesa->fpoints =
		aclPointsChooseFunction(ctx);
	    ctx->Driver.LineFunc = aclMesa->fline =
		aclLineChooseFunction(ctx);
	    ctx->Driver.TriangleFunc = aclMesa->ftriangle =
		aclTriangleChooseFunction(ctx);
	    ctx->Driver.QuadFunc = aclMesa->fquad =
		aclQuadChooseFunction(ctx);
	    ctx->Driver.RectFunc = aclMesa->frect =
		aclRectChooseFunction(ctx);
	    ctx->Driver.RenderVB = aclMesa->frendervb =
		aclRenderVBChooseFunction(ctx);
	}

	ctx->Driver.RasterSetup = aclMesa->frsetup =
	    aclRSetupChooseFunction(ctx);
    }

    else {
LOG("aclDDUpdateDDPointers(): NOT ACCELERATED RENDERING\n");

	ctx->Driver.PointsFunc = NULL;
	ctx->Driver.LineFunc = NULL;
	ctx->Driver.TriangleFunc = NULL;
	ctx->Driver.QuadFunc = NULL;
	ctx->Driver.RectFunc = NULL;
	ctx->Driver.RenderVB = NULL;

	ctx->Driver.RasterSetup = NULL;
    }

    ctx->Driver.AllocDepthBuffer = aclDDAllocDepthBuffer;

    aclSWSetupDDDepthPointers(ctx);

LOG("aclDDUpdateDDPointers(): ______ End ______\n");
LOG("\n");
LOG("\n");
LOG("\n");
}



void
aclDDSetupDDPointers(GLcontext * ctx)
{
LOG("aclDDSetupDDPointers(): ###### Start ######\n");

    ctx->Driver.UpdateState = aclDDUpdateDDPointers;

    ctx->Driver.RendererString = aclDDRendererString;
    ctx->Driver.ExtensionString = aclDDExtensionString;

    ctx->Driver.Dither = aclDDDither;
    ctx->Driver.NearFar = aclDDNearFar;
    ctx->Driver.GetParameteri = aclDDGetParameteri; 

    ctx->Driver.ClearIndex = NULL;
    ctx->Driver.ClearColor = aclDDClearColor;
    ctx->Driver.Clear = aclDDClear;

    ctx->Driver.Index = NULL;
    ctx->Driver.Color = aclDDSetColor;

    ctx->Driver.SetBuffer = aclDDSetBuffer;
    ctx->Driver.GetBufferSize = aclDDGetBufferSize;

    ctx->Driver.Bitmap = NULL;
    ctx->Driver.DrawPixels = NULL;

    ctx->Driver.Finish = aclDDFinish;
    ctx->Driver.Flush = NULL;

    ctx->Driver.TexEnv = aclTexEnv;
    ctx->Driver.TexImage = aclTexImage;
    ctx->Driver.TexParameter = aclTexParam;
    ctx->Driver.BindTexture = aclTexBind;
    ctx->Driver.DeleteTexture = aclTexDel; 
    ctx->Driver.TexSubImage = aclTexSubImage;
    ctx->Driver.UpdateTexturePalette = aclTexUpdatePalette;
    ctx->Driver.UseGlobalTexturePalette = aclTexUseGlobalPalette;

    aclSWSetupDDPointers(ctx);

    aclDDUpdateDDPointers(ctx);

LOG("aclDDSetupDDPointers(): ______ End ______\n");
}



#else



/*
 * Need this to provide at least one external definition.
 */

int
aclDD_dummy(void)
{
    return 0;
}

#endif /* ACL */
