
/*****************************************************************************
 *                                  G P M o d                                *
 *                    Generic kernel module for PCI hardware                 *
 *                           (C) SuSE GmbH 1997, 1998                        *
 *****************************************************************************/
/* Author: simon pogarcic, sim@suse.de */
/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef _MODULE_GPM_H_
#define _MODULE_GPM_H_

#include "common_gpm.h"
#include "project_gpm.h"

#include <asm/signal.h>



/* Max RAM which could be used for DMA */
#define MAX_DMA_RAM		S8M

/* The bigest chunk size to be used for DMA */
#define MAX_DMAPAGE_ORDER	5

/* The max number of non-contiguous chunks per DMA buffer */
#define MAX_DMA_FRAGMENTS	64



#define LinuxVersionCode(v,p,s)		(((v)<<16)+((p)<<8)+(s))

#define LINUX_2_0 \
    (LINUX_VERSION_CODE < LinuxVersionCode(2,1,0))
    
#define LINUX_2_2 \
    (LINUX_VERSION_CODE >= LinuxVersionCode(2,1,100))



#if LINUX_2_2
  #include <asm/uaccess.h>
  #include <linux/pci.h>
#elif LINUX_2_0
  #include <asm/segment.h>
  #include <linux/bios32.h>
#endif



 
#if LINUX_2_0
  #define VERIFYPARM(addr, type, msg) \
  { \
    int retval; \
    if( (retval = \
	verify_area(VERIFY_READ|VERIFY_WRITE,(void*)(addr),sizeof(type))) )\
    { \
	ERR("\nUSER SEGMENT ACCESS !!! (%s)\n", msg); \
	return retval; \
    } \
  }
#endif



#if LINUX_2_2
  #define REMAP	ioremap
  #define FREE	iounmap
  #define MCOPY_TO_USER(dest, src, type, msg) \
    { if(copy_to_user(dest, src, sizeof(type))) { \
	ERR("\nUSER SEGMENT ACCESS !!! (%s)\n", msg); \
	return -EFAULT; \
    } }
  #define MCOPY_FROM_USER(dest, src, type, msg) \
    { if(copy_from_user(dest, src, sizeof(type))) { \
	ERR("\nUSER SEGMENT ACCESS !!! (%s)\n", msg); \
	return -EFAULT; \
    } }
  #define TEST_AND_SET_BIT test_and_set_bit
  #define TEST_AND_CLEAR_BIT test_and_clear_bit
#elif LINUX_2_0
  #define REMAP	vremap
  #define FREE	vfree
  #define MCOPY_TO_USER(dest, src, type, msg) \
    { VERIFYPARM(dest, type, msg); \
    memcpy_tofs(dest, src, sizeof(type)); }
  #define MCOPY_FROM_USER(dest, src, type, msg) \
    { VERIFYPARM(src, type, msg); \
    memcpy_fromfs(dest, src, sizeof(type)); }
  #define TEST_AND_SET_BIT(nr, addr) set_bit(nr, (void *) addr)
  #define TEST_AND_CLEAR_BIT(nr, addr) clear_bit(nr, (void *) addr)
#endif



#define PCICONF_WRITE_BYTE(dev,reg,val) \
    { unsigned char v = (unsigned char) val; \
      pcibios_write_config_byte(dev->bus, dev->dev_fn, reg, v); \
    }
#define PCICONF_WRITE_WORD(dev,reg,val) \
    { unsigned short v = (unsigned short) val; \
      pcibios_write_config_word(dev->bus, dev->dev_fn, reg, v); \
    }
#define PCICONF_WRITE_DWORD(dev,reg,val) \
    { unsigned int v = (unsigned int) val; \
      pcibios_write_config_dword(dev->bus, dev->dev_fn, reg, v); \
    }

#define PCICONF_READ_BYTE(dev,reg,val) \
    { unsigned char v; \
      pcibios_read_config_byte(dev->bus, dev->dev_fn, reg, &v); \
      val = v; }
#define PCICONF_READ_WORD(dev,reg,val) \
    { unsigned short v; \
      pcibios_read_config_word(dev->bus, dev->dev_fn, reg, &v); \
      val = v; }
#define PCICONF_READ_DWORD(dev,reg,val) \
    { unsigned int v; \
      pcibios_read_config_dword(dev->bus, dev->dev_fn, reg, &v); \
      val = v; }



#define MUTEXBIT_MEMORY_INIT \
    static Tulong			mutexmem = 0; \
    static volatile void *		mutexflags = (void *) &mutexmem; \
    struct wait_queue			*mutex_wait_queue

#define ENTER_MUTEX(bitnr) \
{ \
    struct wait_queue wait = { current, NULL }; \
    add_wait_queue(&mutex_wait_queue, &wait); \
    current->state = TASK_INTERRUPTIBLE; \
    while(TEST_AND_SET_BIT(bitnr, mutexflags)) { \
LOG("RACE CONDITION encountered (%d)\n",bitnr); \
	schedule(); \
    } \
    remove_wait_queue(&mutex_wait_queue, &wait); \
}

#define EXIT_MUTEX(bitnr) \
{ \
    if(!TEST_AND_CLEAR_BIT(bitnr, mutexflags)) { \
	ERR("clear_bit: ALREADY UNLOCKED BIT %d\n", bitnr); \
    } \
}



typedef struct GPMDevInfo_t {
	Tubyte		bus;		/* bus number */
	Tubyte		dev_fn;		/* encoded device & function index */
#if LINUX_2_2
	struct pci_dev	*dev;		/* device data from kernel */
#endif
	Tubyte		slot;		/* slot (device) */
	Tubyte		fn;		/* function */
	Tubyte		irq;		/* generated irq */
	Tushort 	dev_id;		/* dev number */
	Tushort 	ven_id;		/* vendor id */
	Tubyte		rev_id;		/* revision id */
	Tbool		master;		/* master capable? */
	Tuint		basefound;
	void		*PCIKmapBase[6];
	GPMBase_t	PCIBase[6];
	Tuint		idx;		/* which one from SupportedDev[] ?*/
} GPMDevInfo_t;



typedef struct GPMCheckIrq_t {
	struct GPMCheckIrq_t 	*next;
	Tint			(*handleirq)(struct GPMCheckIrq_t *, void *);
	void *			irqconfig_addr;
	void *			irqstatus_addr;
	Tulong			mask;
	Tbool			fast;
	Tuint			cardnr;
	Tuint			devnr;
	Tuint			irq_type;
	Tbool			active;
} GPMCheckIrq_t;



typedef struct GPMIrq_t {
	const Tubyte		name[32];
	const Tuint		chip_type;
	const Tuint		irq_type;
	const Tubyte		base;
	const Tulong		offset_config;
	const Tulong		offset_status;
	Tint			(*handleirq)(struct GPMCheckIrq_t *, void *);
} GPMIrq_t;



typedef struct HighArea_t {
    struct HighArea_t	*next;
    Tulong		addr;
    Tulong		size;
} HighArea_t;



#endif
