
/*****************************************************************************
 *                                  G P M o d                                *
 *                    Generic kernel module for PCI hardware                 *
 *                           (C) SuSE GmbH 1997, 1998                        *
 *****************************************************************************/
/* Author: simon pogarcic, sim@suse.de */
/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define __KERNEL__

#include "project_gpm.def"

#include <linux/pci.h>
#include <linux/fs.h>
#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/version.h>
#include <linux/module.h>
#include <linux/major.h>
#include <linux/mm.h>
#include <linux/shm.h>
#include <linux/malloc.h>
#include <linux/fcntl.h>
#include <linux/wrapper.h>
#include <linux/sched.h>
#include <linux/tqueue.h>
#include <linux/interrupt.h>
/* #include <sys/types.h> */
/* #include <sys/mman.h> */
#include <asm/io.h>
#include <asm/pgtable.h>
#include <asm/signal.h>

char kernel_version[] = UTS_RELEASE;

#if 1
#define LOG_DEBUG
#endif

#define LOG_DBGINFO "<" PROJECT_NAME "> "
#include "log_debug.h"

#include "module_gpm.h"

#define GPM_MODULE_OBJECT
#include "scanpci_gpm.h"
#undef GPM_MODULE_OBJECT



#define	DECODE_BASE_SIZE(s) { (s) = ~(s) + 1; }
#define MEGABYTE_ALIGN(addr) ((addr+S1M-1)&(~(S1M-1)))

/*
 * The driver name as it appears in /proc/devices
 */
static Tbyte *driver = PROJECT_NAME;
static Tint drvmajor = 0;

static GPMDevInfo_t PCIDevice[MAX_DEV_ALOWED];
static Tuint PCIDFound = 0;
static GPMCardInfo_t PCICard[MAX_CARDS_ALOWED];
static Tuint PCICFound = 0;



/*****************************************************************************
 * DMA buffers allocation/deallocation
 *****************************************************************************/

static GPMDmaBuff_t *first_dma_buffer = NULL;
static GPMDmaBuff_t *last_dma_buffer = NULL;
static Tuint dma_buffers_cnt = 0;



#ifdef DMA_DYNAMIC_ALLOC

static Tulong RAMreserved = 0;

static void
dmachunks_free(GPMDmaChunk_t * dc)
{
    Tulong size;

    if (!dc)
	return;

    do {
	Tuint i;
	GPMDmaChunk_t *cc = dc;
	size = (1 << (dc->order + PAGE_SHIFT));
LOG("chf/ a=0x%lx l=0x%lx o=0x%x\n", dc->addr, size, dc->order);
	for (i = MAP_NR(dc->addr); i <= MAP_NR(dc->addr + size - 1); i++)
	    mem_map_unreserve(i);
	free_pages(dc->addr, dc->order);
	RAMreserved -= size;
	dc = dc->next;
	kfree(cc);
    } while (dc);
}



static GPMDmaChunk_t *
dmachunks_alloc(Tulong *sizeptr)
{
    Tulong size;
    GPMDmaChunk_t *fc, *pc, *nc;
    Tint maxord, i, pgcnt, cnt;

    size = *sizeptr;
    if ((size = PAGE_ALIGN(size)) == 0) {
	ERR("cha/ size overflow\n");
	return NULL;
    }
    *sizeptr = size;

LOG("cha/ size = 0x%lx\n", size);

    pgcnt = 0;
    fc = pc = nc = NULL;	/*
				 * first, previous and new chunk 
				 */
    maxord = MAX_DMAPAGE_ORDER;
    do {
	Tulong page;
	Tulong pgsize;

	pgsize = (1 << (maxord + PAGE_SHIFT));
	while (size >= pgsize) {

	    if (pgcnt > MAX_DMA_FRAGMENTS) {
		ERR("cha/ To much fragments (%d)\n", pgcnt);
		dmachunks_free(fc);
		return NULL;
	    }

	    if (!nc)
		if (!(nc =
		  (GPMDmaChunk_t*)kmalloc(sizeof(GPMDmaChunk_t),GFP_KERNEL))){
		    ERR("cha/ kmalloc() GPMDmaChunk_t\n");
		    return NULL;
		}

	    cnt = 3;
	    while(cnt--) {
#if LINUX_2_2
		page = __get_free_pages(GFP_KERNEL, maxord);
#elif LINUX_2_0
		page = __get_free_pages(GFP_KERNEL, maxord, 0);
#endif
		if(page)
		    break;

		MSG("cha/ Order (%d) low. Try %d...\n", maxord, cnt);
		/* wait one second */
		current->timeout = jiffies + 1 * HZ;
		current->state = TASK_INTERRUPTIBLE;
		schedule();
		current->timeout = 0;
	    }

	    if (!page) {
		ERR("cha/ Order (%d) exhausted!\n", maxord);
		if (!maxord) {
		    ERR("cha/ RAM exhausted!\n");
		    dmachunks_free(fc);
		    kfree(nc);
		    return NULL;
		}
		break;
	    }

LOG("a=0x%lx l=0x%lx o=%d\n", page, pgsize, maxord);

	    for (i = MAP_NR(page); i <= MAP_NR(page + pgsize - 1); i++)
		mem_map_reserve(i);

	    nc->addr = page;
	    nc->order = maxord;
	    nc->next = NULL;

	    /*
	     * Register chunks in ascendent order
	     */
	    if (!fc) {
		fc = nc;
	    }
	    else {
		GPMDmaChunk_t *cc = fc;
		pc = NULL;
		do {
		    if(nc->addr < cc->addr) break;
		    pc = cc;
		    cc = cc->next;
		} while(cc);

		if(cc) {
		    if(pc) { pc->next = nc; }
		    else { fc = nc; }
		    nc->next = cc;
		}
		else {
		    pc->next = nc;
		}
	    }

	    nc = NULL;

	    size -= pgsize;
	    RAMreserved += pgsize;
	    pgcnt++;
	}
    } while (size && maxord--);

    return fc;
}



static GPMDmaArea_t *
dmachunks_concat(GPMDmaChunk_t *pc)
{
    GPMDmaArea_t *fa, *ca;
    GPMDmaChunk_t *cc;
    Tulong newaddr, size;

    if(!pc) {
	ERR("chc/ No chunks - nothing to do!\n");
	return NULL;
    }

    if (!(ca = fa =
	    (GPMDmaArea_t *) kmalloc(sizeof(GPMDmaArea_t), GFP_KERNEL))) {
	ERR("chc/ 1. kmalloc() GPMDmaArea_t\n");
	return NULL;
    }

    cc = pc->next;
    size = (1 << (pc->order + PAGE_SHIFT));
    newaddr = pc->addr + size;

    ca->addr = pc->addr;
    ca->size = size;

    while(cc) {
	if( newaddr == cc->addr ) {
	    ca->size += size;
	}
	else {
	    if (!(ca->next =
		    (GPMDmaArea_t*)kmalloc(sizeof(GPMDmaArea_t),GFP_KERNEL))){
	 	ERR("chc/ 2. kmalloc() GPMDmaArea_t\n");
		return NULL;
	    }
	    ca = ca->next;
	    ca->addr = cc->addr;
	    ca->size = (1 << (cc->order + PAGE_SHIFT));;
	}

	pc = cc;
	cc = pc->next;
	size = (1 << (pc->order + PAGE_SHIFT));
	newaddr = pc->addr + size;
    }
    return fa;
}

#else /* ! DMA_DYNAMIC_ALLOC */

/************************************
 STATIC HIGH-MEMORY DMA CACHE MANAGER
 ************************************/

static HighArea_t	*high_free = NULL;
static HighArea_t	*high_used = NULL;

static Tbool		high_memory_reserved = FALSE;



#ifdef LOG_DEBUG
static void
High_memlist( void )
{
    HighArea_t *h;

    LOG("\n");
    LOG("#------ High memory usage ------#\n");
    if(!high_memory_reserved) {
	LOG("   -> HIGH MEMORY NOT INITED YET!\n");
	goto nixda;
    }
    LOG(" ** Free areas:\n");
    if(!(h = high_free))
	LOG("   -> NONE!\n");
    while(h) {
	LOG("   -> Addr 0x%lx, size 0x%lx\n", h->addr, h->size);
	h = h->next;
    }
    LOG(" ** Used areas:\n");
    if(!(h = high_used)) {
	LOG("   -> NONE!\n");
	goto nixda;
    }
    while(h) {
	LOG("   -> Addr 0x%lx, size 0x%lx\n", h->addr, h->size);
	h = h->next;
    }
nixda:
    LOG("#-------------------------------#\n");
    LOG("\n");
}
#endif



static Tulong
High_checkmem( volatile void * virt_addr, Tulong size )
{
    volatile Tubyte *start = virt_addr;
    Tulong ok_size = 0;
    Tushort sched = 1;
    Tubyte oldval;

    while(size--) {
	oldval = *start;
	*start=0xaa;
	if(*start!=0xaa) break;
	*start++=oldval;
	ok_size++;
	if(!(sched++)) schedule();
    }
    return ok_size;
}



static Tint
High_meminit( void )
{
    Tulong addr, real_size, check_size = S1M;
    Tulong himem_addr = 0, himem_size = 0;
    void *r;

#if LINUX_2_2
    himem_addr = virt_to_phys(high_memory);
#elif LINUX_2_0
    himem_addr = virt_to_phys( (volatile void *) high_memory);
#endif

    MSG("\n");
    MSG("Detecting high memory area:\n");

#if 0
    if( !(check_size = (MEGABYTE_ALIGN(himem_addr) - himem_addr)) )
	check_size = S1M;
#endif

LOG_OFF

    addr = himem_addr;
    MSG(" - Mem check: ");
    while((r = REMAP(addr, check_size))) {
	real_size = High_checkmem(r, check_size);
	himem_size += real_size; addr += check_size;
LOG("Check 0x%lx (0x%lx); S=0x%lx\n", addr, (unsigned long) r, real_size );
	FREE(r);
	if(check_size != real_size) break;
	schedule();
	printk(".");
    }
    printk("\n");

LOG_ON

    if(himem_size == 0) {
	MSG(" --- none ---\n");
	return 0;
    }

    MSG(" - Found %ld KB at addr %ld KB\n",himem_size/1024, himem_addr/1024);

    if( !(high_free = (HighArea_t *)kmalloc(sizeof(HighArea_t), GFP_KERNEL))) {
	ERR("_meminit: kmalloc()\n");
	return -EINVAL;
    }

    high_free->addr = himem_addr;
    high_free->size = himem_size;
    high_free->next = NULL;

    high_memory_reserved = TRUE;

    return 0;
}



/*
 * We keep track of high memory usage in two single linked lists:
 *
 *	high_free, which knows about free fragments;
 *	high_used, which records allocated areas;
 *
 * Allocator takes the best matching piecies from the high_free area list and
 * updates the free area size acording to request. If zero, that member of the
 * list gets removed from high_free list.
 *
 * Free routine simply puts freed area into high_free list (which is sorted
 * in ascending order of addresses) and tries to merge neighbour areas. If
 * merging fails, the free high memory gets fragmented. This is not so bad if
 * all clients work with the same sizes of chunks.
 */
static Tulong
High_memalloc( Tulong size )
{
    HighArea_t *hf_ok, *hf, *hu_new, **hptr;

    if(!high_memory_reserved) {
	ERR("_memalloc: HIGH MEMORY NOT INITED YET!\n");
	return 0;
    }

    if(!high_free) {
	ERR("_memalloc: High memory exhausted!\n");
	return 0;
    }

    size = PAGE_ALIGN(size);

    if(!size) {
	ERR("_memalloc: Size overflow or zero!\n");
	return 0;
    }

    hf_ok = NULL;
    hf = high_free;

    do {
	if(size <= hf->size) {
	    if(!hf_ok) hf_ok=hf;
	    if(hf->size < hf_ok->size) hf_ok=hf;
	}
    } while((hf = hf->next));

    if(!hf_ok) {
	ERR("_memalloc: Not enough high memory!\n");
	return 0;
    }

    if( !(hu_new = (HighArea_t *) kmalloc(sizeof(HighArea_t), GFP_KERNEL))) {
	ERR("_memalloc: kmalloc()\n");
	return 0;
    }

    hu_new->addr = hf_ok->addr;
    hu_new->size = size;
    hu_new->next = NULL;

    hf_ok->size -= size;
    hf_ok->addr += size;

    /*
     * add new used area to the list
     */
    hptr = &high_used;
    while(*hptr) hptr = &(*hptr)->next;
    *hptr = hu_new;

    /*
     * delete empty areas
     */
    hptr = &high_free;
    while((hf = *hptr)) {
	if(!(hf->size)) {
	    *hptr = hf->next;
	    kfree(hf);
	} else hptr = &hf->next;
    }

#ifdef LOG_DEBUG
    High_memlist();
#endif

    return hu_new->addr;
}



static void
High_memfree( Tulong addr )
{
    HighArea_t *hf, *hu, *hnext, **hptr;

    if(!high_memory_reserved) {
	ERR("_memfree: HIGH MEMORY NOT INITED YET!\n");
	return;
    }

    if(!high_used) {
	ERR("_memfree: Nothing allocated\n");
	return;
    }

    /*
     * first find out if the area is registered as used
     */
    hptr = &high_used;
    hu = high_used;
    while(addr != hu->addr) {
	hptr = &hu->next;
	if(!(hu = hu->next)) break;
    }
    if(!hu) return;
    *hptr = hu->next;

    /*
     * 'free' that area by simply puting it in the high_free list
     */
     hptr = &high_free;
     hf = high_free;
     if(hf) while(addr > hf->addr) {
	hptr = &hf->next;
	if(!(hf = hf->next)) break;
     }
     hu->next = hf;
     *hptr = hu;

    /*
     * See if we can merge the neighbour areas to biger one
     */
    hf = high_free;
    hnext = hf->next;
    while( hnext ) {
	if( (hf->addr + hf->size) == hnext->addr ) {
	    hf->size += hnext->size;
	    hf->next = hnext->next;
	    kfree(hnext);
	}
	else hf = hnext;
	hnext = hf->next;
    }

#ifdef LOG_DEBUG
    High_memlist();
#endif
}

#endif /* ! DMA_DYNAMIC_ALLOC */



static Tint
dmaareas_remap( Tulong * vmpos, GPMDmaArea_t * area, pgprot_t prot )
{
    Tulong phaddr;
    Tulong phsize;

    while (area) {
	phaddr = area->addr;
	phsize = area->size;

LOG(" -> VM=0x%lx a=0x%lx s=0x%lx\n", *vmpos, phaddr, phsize);

	if(remap_page_range(*vmpos, phaddr, phsize, prot)) {
	    ERR("arr/ remap_page_range()\n");
	    return 1;
	}

	area = area->next;
	(*vmpos) += phsize;
    }
    return 0;
}



#ifdef LOG_DEBUG
static void
dmabuff_query(void)
{
    GPMDmaBuff_t *db = first_dma_buffer;
    Tint cnt = 1;
    Tulong size;

    if (db) {
	LOG("\n");
	LOG("--- Registered RAM buffers ---\n");
    }

    while (db) {
	GPMDmaChunk_t *ch = db->chunks;
	GPMDmaArea_t *ar = db->areas;

	LOG(" ** Chunks (Buffer #%d):\n", db->nr);
	if(!ch)
	    LOG("   -> none\n");
	while (ch) {
	    size = (1 << (ch->order + PAGE_SHIFT));
	    LOG("   -> #%d: a=0x%lx s=0x%lx o=%d\n", cnt,
		ch->addr, size, ch->order);
	    cnt++;
	    ch = ch->next;
	}

	LOG(" ** Areas (Buffer #%d):\n", db->nr);
	while (ar) {
	    LOG("   -> #%d: a=0x%lx s=0x%lx\n",
		cnt++, ar->addr, ar->size);
	    ar = ar->next;
	}

	LOG(" TOTAL SIZE: 0x%lx, MMAPS: %d\n", db->size, db->cnt_mmaps);
	LOG(" FLAGS: 0x%lx\n", db->flags);
	LOG("------------------------------\n");
	db = db->next;
    }
    LOG("\n");
}
#endif



static GPMDmaBuff_t *
dmabuff_alloc(Tulong size, Tuint flags)
{
    GPMDmaBuff_t *db;
    

    if (!(db = (GPMDmaBuff_t *) kmalloc(sizeof(GPMDmaBuff_t), GFP_KERNEL))) {
	ERR("DB alloc: kmalloc() GPMDmaBuff_t\n");
	return NULL;
    }

#ifdef DMA_DYNAMIC_ALLOC

    if (!(db->chunks = dmachunks_alloc(&size))) {
	ERR("DB alloc: dmachunks_alloc()\n");
	return NULL;
    }

    if (!(db->areas = dmachunks_concat(db->chunks))) {
	ERR("DB alloc: dmachunks_concat()\n");
	return NULL;
    }

#else

    db->chunks = NULL;
    if (!(db->areas =
	    (GPMDmaArea_t *) kmalloc(sizeof(GPMDmaArea_t), GFP_KERNEL))) {
	ERR("DB alloc: kmalloc() GPMDmaArea_t\n");
	return NULL;
    }

    if( !(db->areas->addr = High_memalloc(size)) ) {
	return NULL;
    }

    db->areas->size = size;
    db->areas->next = NULL;
    db->cnt_areas = 1;

#endif

    db->cnt_mmaps = 0;
    db->size = size;
    db->flags = DB_NOTUSED;

    if (!first_dma_buffer) first_dma_buffer = db;
    else last_dma_buffer->next = db;

    db->prev = last_dma_buffer;
    last_dma_buffer = db;
    db->next = NULL;
    db->nr = ++dma_buffers_cnt;

LOG("DB alloc: DB #%d reserved\n", db->nr);

    return db;
}



static GPMDmaBuff_t *
dmabuff_free(GPMDmaBuff_t * db)
{
    GPMDmaBuff_t *dbnext, *dbprev;

    if (!db)
	return NULL;

    dbnext = db->next;
    dbprev = db->prev;

LOG("DB free: Releasing DB #%d\n", db->nr);

#ifdef DMA_DYNAMIC_ALLOC
    dmachunks_free(db->chunks);
#else
    High_memfree(db->areas->addr);
    kfree(db->areas);
#endif

    if (dbnext)
	dbnext->prev = dbprev;
    else
	last_dma_buffer = dbprev;

    if (dbprev)
	dbprev->next = dbnext;
    else
	first_dma_buffer = dbnext;

    if (!(dbnext || dbprev))
	dma_buffers_cnt = 0;

    kfree(db);

    return dbnext;
}



static GPMDmaBuff_t *
dmabuff_find(Tint dbnr)
{
    GPMDmaBuff_t *db = first_dma_buffer;

    while(db) {
	if(db->nr == dbnr) break;
	db = db->next;
    }

    return db;
}



/*****************************************************************************
 * Resource usage and validation functions
 *****************************************************************************/

static GPMAddr_t *ValidMmaps = NULL;

/*
 * The valid addresses for mmaping into user space are stored into
 * single linked list. The driver writer can decide which addresses
 * get stored
 */
static Tint
gpmod_pcimmap_validate( Tulong addr )
{
    GPMAddr_t *vmm, **vmmptr = &ValidMmaps;

    while((vmm = *vmmptr)) vmmptr = &vmm->next;
    if( !(vmm = kmalloc(sizeof(GPMAddr_t), GFP_KERNEL)) ) {
	ERR("_pcimmap_validate: kmalloc()\n");
	return 1;
    }
    vmm->addr = addr;
    vmm->next = NULL;
    *vmmptr = vmm;
    return 0;
}

static Tbool
gpmod_pcimmap_isvalid( Tulong addr )
{
    GPMAddr_t *vmm = ValidMmaps;

    while(vmm) {
	if(vmm->addr == addr) return FALSE;
	vmm = vmm->next;
    }
    return TRUE;
}

static void
gpmod_pcimmap_end( void )
{
    GPMAddr_t *vmm, *vmmlast = ValidMmaps;

    if(!vmmlast) return;
    do {
	vmm = vmmlast->next;
	kfree(vmmlast);
    } while ((vmmlast = vmm));
}



/*
 * The virtual memory usage of module is recorded in single linked list
 */
static GPMAddr_t *VirtualAreas = NULL;

static Tint
gpmod_virtual_track( void *addr )
{
    GPMAddr_t *va, **vaptr = &VirtualAreas;

    while((va = *vaptr)) vaptr = &va->next;
    if( !(va = kmalloc(sizeof(GPMAddr_t), GFP_KERNEL)) ) {
	ERR("_virtual_track: kmalloc()\n");
	return 1;
    }
    va->addr = (Tulong) addr;
    va->next = NULL;
    *vaptr = va;
    return 0;
}

static void
gpmod_virtual_end( void )
{
    GPMAddr_t *va, *valast = VirtualAreas;

    if(!valast) return;
    do {
	va = valast->next;
	FREE((void *) valast->addr);
	kfree(valast);
    } while ((valast = va));
}



/*****************************************************************************
 * PCI SCAN functions
 *****************************************************************************/
/*
 * Fixing hardware bug on TX/MX and PM cards, which is making troubles on some
 * PC boards with old PCI BIOS. The code is taken from XFree86 glint driver.
 */
static void
gpmod_checkbug_glint(GPMCardInfo_t *card, Tuint type)
{
    Tuint tmp = 0;
    GPMDevInfo_t *devcopro = PCIDevice + card->cd[0].devidx;
    GPMDevInfo_t *devdelta = PCIDevice + card->cd[1].devidx;
    Tuint basecopro = devcopro->PCIBase[0].base;
    Tuint basedelta = devdelta->PCIBase[0].base;

    /*
     * due to a few bugs in the GLINT Delta we might have to relocate
     * the base address of config region of the Delta, if bit 17 of
     * the base addresses of config region of the Delta and the 500TX,
     * 300SX or Permedia are different. We only handle config type 1 at
     * this point.
     */
    if ((basedelta & 0x20000) ^ (basecopro & 0x20000)) {
	Tuint newbasedelta = 0;
	const Tubyte *infostr = "";

	if( IS_CHIP_CLASS_PM(type) ) {
	    newbasedelta = card->cd[0].PCIBase[4].base;
	    infostr = "Permedia Base4";
	}
	else if( IS_CHIP_FAMILY_TXMX(type) ) {
	    newbasedelta = card->cd[0].PCIBase[3].base;
	    infostr = "500TX/MX Base3";
	}

	MSG(" # FIXING Delta Base 0 bug\n");
	MSG("   %s at 0x%x\n", infostr, newbasedelta);

	/*
	 * if the base addresses are different at bit 17, we have to
	 * remap the base0 for the delta; as wrong as this looks, we
	 * can use the base3 of the 300SX/500TX for this. The delta
	 * is working as a bridge here and gives its own addresses
	 * preference. And we don't need to access base3, as this one
	 * is the bytw swapped local buffer which we don't need.
	 * Using base3 we know that the space is
	 *
	 *              a) large enough
	 *              b) free (well, almost)
	 */
	if ((basecopro & 0x20000) ^ (newbasedelta & 0x20000)) {
	    /*
	     * oops, still different; we know that base3 is at least
	     * 16 MB, so we just take 128k offset into it
	     */
	    newbasedelta += 0x20000;
	    MSG("   Adding 128k offset, now 0x%x\n", newbasedelta);
	}

	/*
	 * read old value
	 * write 0xffffffff
	 * read value
	 * write new value
	 */
	PCICONF_READ_DWORD(devdelta, PCI_BASE_ADDRESS_0, tmp);
	PCICONF_WRITE_DWORD(devdelta, PCI_BASE_ADDRESS_0, 0xffffffff);
	PCICONF_READ_DWORD(devdelta, PCI_BASE_ADDRESS_0, tmp);
	PCICONF_WRITE_DWORD(devdelta, PCI_BASE_ADDRESS_0, newbasedelta);

	/*
	 * additionally, sometimes we see the baserom which might
	 * confuse the chip, so let's make sure that is disabled
	 */
	PCICONF_READ_DWORD(devcopro, PCI_ROM_ADDRESS, tmp);
	PCICONF_WRITE_DWORD(devcopro, PCI_ROM_ADDRESS, 0xffffffff);
	PCICONF_READ_DWORD(devcopro, PCI_ROM_ADDRESS, tmp);
	PCICONF_WRITE_DWORD(devcopro, PCI_ROM_ADDRESS, 0);

	/*
	 * now update our internal structure accordingly
	 */
	PCICONF_READ_DWORD(devdelta, PCI_BASE_ADDRESS_0, tmp);
	card->cd[1].PCIBase[0].base = devdelta->PCIBase[0].base = tmp;
	card->cd[1].PCIBase[0].aoffset = devdelta->PCIBase[0].aoffset =
	    tmp - (tmp & PAGE_MASK);

#if LINUX_2_2
	/*
	 * Update kernel pci_dev structure, if using new kernels.
	 */
	devdelta->dev->base_address[0] = tmp;
	devdelta->dev->rom_address = 0;
#endif

	MSG("   >>> NEW Delta Base0: 0x%x\n\n", tmp);
    }
#if LINUX_2_2
    else {
	/*
	 * The X server might already updated base 0 and rom address, but
	 * what about kernel pci_dev ? Let's inform pci_dev structure about
	 * base 0 and rom address changes, if any.
	 */
	basecopro = devcopro->dev->base_address[0];
	basedelta = devdelta->dev->base_address[0];

	if ((basedelta & 0x20000) ^ (basecopro & 0x20000)) {

	    PCICONF_READ_DWORD(devdelta, PCI_BASE_ADDRESS_0, tmp);
	    
	    MSG(" # FIXING Delta Base 0 bug (KERNEL)\n");
	    MSG("   Was: 0x%x, NEW: 0x%x\n", basedelta, tmp);

	    devdelta->dev->base_address[0] = tmp;
	    devdelta->dev->rom_address = 0;
	}
    }
#endif
}



static Tint
gpmod_find_chip(Tuint chip_type, Tuint sdcnt,
	      Tbyte bus_slot, Tubyte bus, Tubyte slot)
{
    while (sdcnt < PCIDFound) {
	Tuint idx = PCIDevice[sdcnt].idx;
	if (SupportedDevices[idx].chip_type == chip_type && (!bus_slot ||
	    (PCIDevice[sdcnt].bus == bus && PCIDevice[sdcnt].slot == slot)))
	    return sdcnt;
	sdcnt++;
    }
    return -1;
}



static Tint
gpmod_scanbus_checkcard(GPMCardInfo_t *card)
{
    Tuint ct0 = SupportedCards[card->idx].chip_type[0];

    switch(ct0) {
    case CHIP_CLASS_500TX:
    case CHIP_CLASS_MX:
    case CHIP_CLASS_PERMEDIA:
	gpmod_checkbug_glint(card, ct0);
    default:
	return 0;
    }

    return 0;
}



static void
gpmod_load_cdata(void)
{
    Tuint csup_cnt, bcnt;
    GPMCard_t *cscan;

    MSG("\n");
    MSG("Scanning for supported cards...\n");

    /*
     * go through all cards we support 
     */
    csup_cnt = 0;
    while (csup_cnt < NUMBER_OF_SUPPORTED_CARDS) {
	int fdevice, ffirst = 0;

	cscan = SupportedCards + csup_cnt;

	/*
	 * find first chip on the card 
	 */
	while (ffirst < PCIDFound) {
	    Tuint chip0 = cscan->chip_type[0];
	    Tubyte bus, slot;
	    Tuint chipcnt = 1;

	    GPMDevInfo_t *dev;
	    GPMCardInfo_t *card = PCICard + PCICFound;
	    GPMCardDev_t *cd = card->cd + 0;

	    if ((fdevice = gpmod_find_chip(chip0, ffirst, 0, 0, 0)) == -1)
		break;

	    card->error = TRUE;

	    dev = PCIDevice + fdevice;
	    cd->basefound = dev->basefound;
	    cd->devidx = fdevice;
	    for(bcnt=0; bcnt<6; bcnt++) {
		GPMBase_t *dbt = cd->PCIBase + bcnt;
		GPMBase_t *dbs = dev->PCIBase + bcnt;
		if( !(dev->basefound & BASE(bcnt)) )
		    continue;
		dbt->base = dbs->base;
		dbt->aoffset = dbs->aoffset;
		dbt->size = dbs->size;
		dbt->flags = dbs->flags;
	    }

	    bus = dev->bus;
	    slot = dev->slot;
	    ffirst = fdevice + 1;

	    /*
	     * now scan for other required chips on the card 
	     */
	    while (chipcnt < cscan->maxdev) {
		Tuint nchip = cscan->chip_type[chipcnt];

		if ((fdevice = gpmod_find_chip(nchip, 0, 1, bus, slot)) == -1)
		    break;

		dev = PCIDevice + fdevice;
		cd++;

	        cd->basefound = dev->basefound;
		cd->devidx = fdevice;
	        for(bcnt=0; bcnt<6; bcnt++) {
		    GPMBase_t *dbt = cd->PCIBase + bcnt;
		    GPMBase_t *dbs = dev->PCIBase + bcnt;
		    if( !(dev->basefound & BASE(bcnt)) )
			continue;
		    dbt->base = dbs->base;
		    dbt->aoffset = dbs->aoffset;
		    dbt->size = dbs->size;
		    dbt->flags = dbs->flags;
		}

		chipcnt++;
	    }
	    if (fdevice != -1) {
		/*
		 * all req. devices found - now prepare for the next card 
		 */
		card->maxdev = chipcnt;
		card->idx = csup_cnt;
		card->slot = slot;
		card->bus = bus;
		card->error = FALSE;

		MSG("\n");
		MSG("### C(%d): %s\n", PCICFound, cscan->name);
		MSG(" * Slot %d, Bus %d\n", slot, bus);

		gpmod_scanbus_checkcard( card );

		PCICFound++;
	    }
	}
	csup_cnt++;
    }
    if (!PCICFound)
	MSG(" * No supported cards found!\n");
    MSG("\n");
}



static Tint
gpmod_scanbus_checkdev(GPMDevInfo_t *cpcid)
{
    Tuint ct = SupportedDevices[cpcid->idx].chip_type;

    switch(ct) {
    default:
	return 0;
    }

    return 0;
}



static Tint
gpmod_scanbus( void )
{
#if LINUX_2_2
    struct pci_dev *pcidev = NULL;
#endif
    Tubyte bus, dev_fn;
    Tint cnt, sdcnt;

    MSG("\n");
    MSG("Scanning for supported devices...\n");

    for (sdcnt = 0; sdcnt < NUMBER_OF_SUPPORTED_DEVICES; sdcnt++) {
	GPMDev_t *dscan = SupportedDevices + sdcnt;
#if LINUX_2_0
	Tint head = 0;
#endif

	while ( PCIDFound < MAX_DEV_ALOWED &&
#if LINUX_2_2
	    (pcidev = pci_find_device(dscan->ven_id, dscan->dev_id, pcidev)) )
#elif LINUX_2_0
	    (pcibios_find_device(dscan->ven_id, dscan->dev_id,
			head++, &bus, &dev_fn) == PCIBIOS_SUCCESSFUL) )
#endif
	{
	    GPMDevInfo_t *cpcid = PCIDevice + PCIDFound;
	    Tubyte base = PCI_BASE_ADDRESS_0;
#if LINUX_2_0
	    Tubyte cmd, tmpcmd;
#endif

#if LINUX_2_2
	    bus = pcidev->bus->number;
	    dev_fn = pcidev->devfn;
	    cpcid->dev = pcidev;
#endif

	    cpcid->bus = bus;
	    cpcid->dev_fn = dev_fn;
	    cpcid->slot = PCI_SLOT(dev_fn);
	    cpcid->fn = PCI_FUNC(dev_fn);
	    cpcid->basefound = 0;
	    cpcid->idx = sdcnt;

#if LINUX_2_2
	    cpcid->master = (pcidev->master != 0);
	    cpcid->ven_id = pcidev->vendor;
	    cpcid->dev_id = pcidev->device;
	    cpcid->irq = pcidev->irq;
#elif LINUX_2_0
	    PCICONF_READ_BYTE(cpcid, PCI_COMMAND, cmd);
	    PCICONF_WRITE_BYTE(cpcid, PCI_COMMAND, cmd | PCI_COMMAND_MASTER);
	    PCICONF_READ_BYTE(cpcid, PCI_COMMAND, tmpcmd);
	    PCICONF_WRITE_BYTE(cpcid, PCI_COMMAND, cmd);
	    cpcid->master = ((tmpcmd & PCI_COMMAND_MASTER) != 0);
	    PCICONF_READ_WORD(cpcid, PCI_VENDOR_ID, cpcid->ven_id);
	    PCICONF_READ_WORD(cpcid, PCI_DEVICE_ID, cpcid->dev_id);
	    PCICONF_READ_BYTE(cpcid, PCI_INTERRUPT_LINE, cpcid->irq);
#endif

	    PCICONF_READ_BYTE(cpcid, PCI_REVISION_ID, cpcid->rev_id);

	    for(cnt=0; cnt < 6; cnt++) {
		Tuint bs, batmp;
		Tulong ba, *baddr = &(cpcid->PCIBase[cnt].base);
		Tulong *aoffset = &(cpcid->PCIBase[cnt].aoffset);
		Tulong *bsize = &(cpcid->PCIBase[cnt].size);
		Tuint *bflags = &(cpcid->PCIBase[cnt].flags);

#if LINUX_2_2
		ba = pcidev->base_address[cnt];
		PCICONF_READ_DWORD(cpcid, base, batmp);
#elif LINUX_2_0
		PCICONF_READ_DWORD(cpcid, base, ba);
		batmp = ba;
#endif
		*baddr = *aoffset = *bsize = *bflags = 0;

		if( ! ba )
		    continue;
		
		cpcid->basefound |= BASE(cnt);

		PCICONF_WRITE_DWORD(cpcid, base, 0xffffffff);
		PCICONF_READ_DWORD(cpcid, base, bs);
		DECODE_BASE_SIZE( bs );
		PCICONF_WRITE_DWORD(cpcid, base, batmp);
		*bsize = bs;

		*bflags = ba & 0xf;

		if(*bflags & PCI_BASE_ADDRESS_SPACE_IO) {
		    *baddr = (ba & PCI_BASE_ADDRESS_IO_MASK);
		}
		else {
#if LINUX_2_0
		    Tuint ba_high = 0;
		    Tuint bs_high = 0;
#endif
		    *baddr = (ba & PCI_BASE_ADDRESS_MEM_MASK);
		    *aoffset = *baddr - (*baddr & PAGE_MASK);

		    if(*bflags & PCI_BASE_ADDRESS_MEM_TYPE_64) {
			base += 4;
#if LINUX_2_0
			PCICONF_READ_DWORD(cpcid, base, ba_high);
			PCICONF_WRITE_DWORD(cpcid, base, 0xffffffff);
			PCICONF_READ_DWORD(cpcid, base, bs_high);
			DECODE_BASE_SIZE( bs_high );
			PCICONF_WRITE_DWORD(cpcid, base, ba_high);
#endif
		    }
#if LINUX_2_0
		    *baddr |= (((u64) ba_high) << 32);
		    *bsize |= (((u64) bs_high) << 32);
#endif
		}
		base += 4;
	    }

	    MSG("\n");
	    MSG("### Check id: ven 0x%x, dev 0x%x...\n",
		cpcid->ven_id, cpcid->dev_id);

	    if(gpmod_scanbus_checkdev(cpcid)) {
		MSG(" * Driver check failed!\n");
		continue;
	    }

	    /* OK, this is our device; search & destroy next one */

	    MSG("### D(%d) %s\n", PCIDFound, dscan->name);
	    MSG(" * Devfn: %d (Slot %d, Fn %d); Bus: %d\n",
		dev_fn, cpcid->slot, cpcid->fn, bus);
	    MSG(" * Rev: 0x%x, Irq: %d (%saster capable)\n",
		cpcid->rev_id, cpcid->irq, (cpcid->master ? "M" : "NOT m")); 

	    for(cnt=0; cnt < 6; cnt++) {
		Tulong baddr = cpcid->PCIBase[cnt].base;
		Tulong aoffset = cpcid->PCIBase[cnt].aoffset;
		Tuint bflags = cpcid->PCIBase[cnt].flags;
		const char *atype="MEM, ", *mtype="???", *ptype=", Non-Pref";

		if(!(cpcid->basefound & BASE(cnt))) continue;

		gpmod_pcimmap_validate(baddr);

		if(bflags & PCI_BASE_ADDRESS_SPACE_IO) {
		    atype = "I/O"; mtype = ""; ptype = "";
		}
		else {
		    if(bflags & PCI_BASE_ADDRESS_MEM_PREFETCH) {
			ptype = ", Pref";
		    }
		    switch( bflags & PCI_BASE_ADDRESS_MEM_TYPE_MASK ) {
		    case PCI_BASE_ADDRESS_MEM_TYPE_32:
			mtype = "32b";
			break;
		    case PCI_BASE_ADDRESS_MEM_TYPE_1M:
			mtype = "20b";
			break;
		    case PCI_BASE_ADDRESS_MEM_TYPE_64:
			mtype = "64b";
			break;
		    }
		}
		MSG(" * BASE%d = 0x%lx; (%s%s)\n", cnt, baddr, atype, mtype);
		MSG("   - Size 0x%lx %s\n", cpcid->PCIBase[cnt].size, ptype);
		if(aoffset)
		    MSG("     >>> align: -0x%lx)\n", (Tulong) aoffset);
	    }
	    
	    PCIDFound++;
	}
    }
    return 0;
}



static Tint
gpmod_kmmap_devices( void )
{
    Tuint dcnt, cnt;

    MSG("\n");
    for(dcnt = 0; dcnt < PCIDFound; dcnt++ ) {
	GPMDevInfo_t *dev = PCIDevice + dcnt;
	Tuint kmmapbase = SupportedDevices[dev->idx].kmmap_base;

	if(kmmapbase)
	    MSG("### Kernel-mmap Device #%d:\n", dcnt);

	for(cnt = 0; cnt < 6; cnt++ ) {
	    void **kmmap = &(dev->PCIKmapBase[cnt]);

	    if(kmmapbase & BASE(cnt)) {
		Tulong aoffset = dev->PCIBase[cnt].aoffset;
		Tulong bsize = dev->PCIBase[cnt].size;
		Tulong baddr = dev->PCIBase[cnt].base;

		if(!(*kmmap = REMAP((baddr & PAGE_MASK), (bsize + aoffset)))) {
		    ERR("_do_scanbus: remap PCI Base%d\n", cnt);
		}
		else {
		    MSG(" * Base%d to 0x%lx\n", cnt, (Tulong) *kmmap);
		    gpmod_virtual_track(*kmmap);
		    *kmmap += aoffset;
		    if(aoffset)
		        MSG("   >>> starts: 0x%lx\n", (Tulong) *kmmap);
		}
	    }
	    else
		*kmmap = NULL;
	}
    }

    return 0;
}


/*****************************************************************************
 * IOCTL functions
 *****************************************************************************/

/* set 'u' TRUE, if the function is called from user space */
static Tint
gpmod_pciconfig(GPMPciParam_t * u_pciparam, Tbool write, Tbool u)
{
    GPMPciParam_t k_pciparam, *pciparam;
    GPMCardInfo_t *card;
    GPMDevInfo_t *dev;
    Tint cardnr, pid, dcnt;

    if(u) {
	pciparam = &k_pciparam;
	MCOPY_FROM_USER(pciparam, u_pciparam, GPMPciParam_t, "_pciconfig");
    }
    else
	pciparam = u_pciparam;

    cardnr = pciparam->Icardnr;
    pid = current->pid;

    if (cardnr >= PCICFound) {
	ERR("_pciconfig: cardnr (%d)\n", cardnr);
	return -EINVAL;
    }

    card = &PCICard[cardnr];

    if(write) {
LOG("WRITE PCI CONFIG: (card #%d, PID %d):\n", cardnr, pid);
	for (dcnt = 0; dcnt < card->maxdev; dcnt++) {
	    GPMPciConfig_t *cfg = pciparam->cfg + dcnt;
	    dev = PCIDevice + card->cd[dcnt].devidx;
	    switch(cfg->val_size) {
	    case SIZE_BYTE:
		PCICONF_WRITE_BYTE(dev, cfg->reg, cfg->value);
		PCICONF_READ_BYTE(dev, cfg->reg, cfg->real_value);
		break;
	    case SIZE_WORD:
		PCICONF_WRITE_WORD(dev, cfg->reg, cfg->value);
		PCICONF_READ_WORD(dev, cfg->reg, cfg->real_value);
		break;
	    case SIZE_DWORD:
		PCICONF_WRITE_DWORD(dev, cfg->reg, cfg->value);
		PCICONF_READ_DWORD(dev, cfg->reg, cfg->real_value);
		break;
	    }
LOG("     Dev #%d: value 0x%x at offset 0x%x\n",
		dcnt, cfg->real_value, cfg->reg);
	}
    }

    else {
LOG("READ PCI CONFIG: (card #%d, PID %d):\n", cardnr, pid);
	for (dcnt = 0; dcnt < card->maxdev; dcnt++) {
	    GPMPciConfig_t *cfg = pciparam->cfg + dcnt;
	    dev = PCIDevice + card->cd[dcnt].devidx;
	    switch(cfg->val_size) {
	    case SIZE_BYTE:
		PCICONF_READ_BYTE(dev, cfg->reg, cfg->real_value);
		break;
	    case SIZE_WORD:
		PCICONF_READ_WORD(dev, cfg->reg, cfg->real_value);
		break;
	    case SIZE_DWORD:
		PCICONF_READ_DWORD(dev, cfg->reg, cfg->real_value);
		break;
	    }
LOG("     Dev #%d: value 0x%x at offset 0x%x\n",
		dcnt, cfg->real_value, cfg->reg);
	}
    }

    if(u)
	MCOPY_TO_USER(u_pciparam, pciparam, GPMPciParam_t, "_pciconfig");

    return 0;
}



static Tint
gpmod_docardinfo(GPMCardInfo_t * u_arg)
{
    Tint pid;
    Tuint ccnt;

    pid = current->pid;
LOG("SEND cards info: client pid %d\n", pid);

    if (PCICFound == 0) {
	ERR("No supported cards found!\n");
	return -ENODEV;
    }

    for (ccnt = 0; ccnt < PCICFound; ccnt++) {
	GPMCardInfo_t *dest = u_arg + ccnt;
	MCOPY_TO_USER(dest, &PCICard[ccnt], GPMCardInfo_t, "_docardinfo");
    }

    return 0;
}



static Tint
gpmod_rwio(Tulong port,
	Tulong *value, Tubyte size, Tubyte wlop, Tbool w)
{
    switch(size) {
    case SIZE_BYTE:
	{
	  Tubyte bval = (Tubyte) *value;
	  if(w) {
	    switch(wlop) {
	    case LOGOP_WRITE_OVER:
		break;
	    case LOGOP_WRITE_OR:
		bval |= inb(port);
		break;
	    case LOGOP_WRITE_AND:
		bval &= inb(port);
		break;
	    case LOGOP_WRITE_NAND:
		bval = (~bval) & inb(port);
		break;
	    }
	    outb(bval, port);
	    bval = inb(port);
	  }
	  else bval = inb(port);
	  *value = bval;
	}
	break;
    case SIZE_WORD:
	{
	  Tushort sval = (Tushort) *value;
	  if(w) {
	    switch(wlop) {
	    case LOGOP_WRITE_OVER:
		break;
	    case LOGOP_WRITE_OR:
		sval |= inw(port);
		break;
	    case LOGOP_WRITE_AND:
		sval &= inw(port);
		break;
	    case LOGOP_WRITE_NAND:
		sval = (~sval) & inw(port);
		break;
	    }
	    outw(sval, port);
	    sval = inw(port);
	  }
	  else sval = inw(port);
	  *value = sval;
	}
	break;
    case SIZE_DWORD:
	{
	  Tuint ival = (Tuint) *value;
	  if(w) {
	    switch(wlop) {
	    case LOGOP_WRITE_OVER:
		break;
	    case LOGOP_WRITE_OR:
		ival |= inl(port);
		break;
	    case LOGOP_WRITE_AND:
		ival &= inl(port);
		break;
	    case LOGOP_WRITE_NAND:
		ival = (~ival) & inl(port);
		break;
	    }
	    outl(ival, port);
	    ival = inl(port);
	  }
	  else ival = inl(port);
	  *value = ival;
	}
	break;
    default:
	ERR("_rwio: invalid value size (%d)\n", size);
	return 1;
    }
LOG("  - val 0x%lx at I/O 0x%lx\n", *value, port);
    return 0;
}

static Tint
gpmod_rwmmap(volatile void *addr,
	Tulong *value, Tubyte size, Tubyte wlop, Tbool w)
{
    switch(size) {
    case SIZE_BYTE:
	{
	  Tubyte bval = (Tubyte) *value;
	  if(w) {
	    switch(wlop) {
	    case LOGOP_WRITE_OVER:
		break;
	    case LOGOP_WRITE_OR:
		bval |= *((Tubyte *) addr);
		break;
	    case LOGOP_WRITE_AND:
		bval &= *((Tubyte *) addr);
		break;
	    case LOGOP_WRITE_NAND:
		bval = (~bval) & *((Tubyte *) addr);
		break;
	    }
	    *((Tubyte *) addr) = bval;
	    bval = *((Tubyte *) addr);
	  }
	  else bval = *((Tubyte *) addr);
	  *value = bval;
	}
	break;
    case SIZE_WORD:
	{
	  Tushort sval = (Tushort) *value;
	  if(w) {
	    switch(wlop) {
	    case LOGOP_WRITE_OVER:
		break;
	    case LOGOP_WRITE_OR:
		sval |= *((Tushort *) addr);
		break;
	    case LOGOP_WRITE_AND:
		sval &= *((Tushort *) addr);
		break;
	    case LOGOP_WRITE_NAND:
		sval = (~sval) & *((Tushort *) addr);
		break;
	    }
	    *((Tushort *) addr) = sval;
	    sval = *((Tushort *) addr);
	  }
	  else sval = *((Tushort *) addr);
	  *value = sval;
	}
	break;
    case SIZE_DWORD:
	{
	  Tuint ival = (Tuint) *value;
	  if(w) {
	    switch(wlop) {
	    case LOGOP_WRITE_OVER:
		break;
	    case LOGOP_WRITE_OR:
		ival |= *((Tuint *) addr);
		break;
	    case LOGOP_WRITE_AND:
		ival &= *((Tuint *) addr);
		break;
	    case LOGOP_WRITE_NAND:
		ival = (~ival) & *((Tuint *) addr);
		break;
	    }
	    *((Tuint *) addr) = ival;
	    ival = *((Tuint *) addr);
	  }
	  else ival = *((Tuint *) addr);
	  *value = ival;
	}
	break;
    case SIZE_QWORD:
	{
	  Tulong lval = *value;
	  if(w) {
	    switch(wlop) {
	    case LOGOP_WRITE_OVER:
		break;
	    case LOGOP_WRITE_OR:
		lval |= *((Tulong *) addr);
		break;
	    case LOGOP_WRITE_AND:
		lval &= *((Tulong *) addr);
		break;
	    case LOGOP_WRITE_NAND:
		lval = (~lval) & *((Tulong *) addr);
		break;
	    }
	    *((Tulong *) addr) = lval;
	    lval = *((Tulong *) addr);
	  }
	  else lval = *((Tulong *) addr);
	  *value = lval;
	}
	break;
    default:
	ERR("_rwmmap: invalid value size (%d)\n", size);
	return 1;
    }
LOG("  - val 0x%lx at kmmaped 0x%lx\n", *value, (Tulong) addr);
    return 0;
}

/* set 'u' TRUE, if the function is called from user space */
static Tint
gpmod_rwbase(GPMIoctlRw_t *u_par, Tbool write, Tbool u)
{
    GPMIoctlRw_t k_par, *par;
    GPMCardInfo_t *card;
    GPMDevInfo_t *dev;
    Tint ret, pid, cardnr, devnr, basenr;

    if(u) {
	par = &k_par;
	MCOPY_FROM_USER(par, u_par, GPMIoctlRw_t, "_rwbase");
    }
    else
	par = u_par;

    cardnr = par->Icardnr;
    devnr = par->Idevnr;
    basenr = par->Ibasenr;
    pid = current->pid;

#ifdef LOG_DEBUG
    if(write)
	LOG("IOCTL write (C:%d D:%d B:%d, PID %d)\n",
	    cardnr, devnr, basenr, pid);
    else
	LOG("IOCTL read (C:%d D:%d B:%d, PID %d)\n",
	    cardnr, devnr, basenr, pid);
#endif

    if (cardnr >= PCICFound) {
	ERR("_rwbase: Icardnr (%d)\n", cardnr);
	return -EINVAL;
    }

    card = &PCICard[cardnr];

    if (devnr >= card->maxdev) {
	ERR("_rwbase: Idevnr (%d)\n", devnr);
	return -EINVAL;
    }

    dev = PCIDevice + card->cd[devnr].devidx;

    if(!(BASE(basenr) & dev->basefound)) {
	ERR("_rwbase: Ibasenr (%d)\n", basenr);
	return -EINVAL;
    }

    if(par->offset_bytes > dev->PCIBase[basenr].size) {
	ERR("_rwbase: offset out of range (0x%lx)\n", par->offset_bytes);
	return -EINVAL;
    }

    if(dev->PCIBase[basenr].flags & PCI_BASE_ADDRESS_SPACE_IO) {
	Tulong port = dev->PCIBase[basenr].base + par->offset_bytes;
	if((ret = gpmod_rwio(port,
		&par->value, par->value_size, par->write_logop, write)))
	    return ret;
    }
    else {
	volatile void *membase = dev->PCIKmapBase[basenr];
	if(!membase) {
	    ERR("_rwbase: Base%d not kernel-mapped\n", basenr);
	    return -EINVAL;
	}
	membase += par->offset_bytes;
	if((ret = gpmod_rwmmap(membase,
		&par->value, par->value_size, par->write_logop, write)))
	    return ret;
    }

    if(u)
	MCOPY_TO_USER(u_par, par, GPMIoctlRw_t, "_rwbase");

    return 0;
}



/*****************************************************************************
 *                     DEVICE SPECIFIC IRQ MANAGEMENT                        *
 *****************************************************************************/

static Tint
gpmod_isr_setconfig(GPMCheckIrq_t *cirq)
{
    switch(cirq->irq_type) {
    case IRQTYPE_PERMEDIA2:
    case IRQTYPE_DELTA:
	goto USUAL_IRQ_CONFIG;
    default:
	return 1;
    }

USUAL_IRQ_CONFIG:
{
    volatile Tuint *irqconfig = cirq->irqconfig_addr;

    *irqconfig = (Tuint) cirq->mask;
LOG("   # type 0x%x <ON>, Conf: 0x%x\n", cirq->irq_type, *irqconfig);
}
    cirq->active = TRUE;
    return 0;
}



static Tint
gpmod_isr_deactivate(GPMCheckIrq_t *cirq)
{
    switch(cirq->irq_type) {
    case IRQTYPE_PERMEDIA2:
    case IRQTYPE_DELTA:
	goto USUAL_IRQ_DEACTIVATE;
    default:
	return 1;
    }

USUAL_IRQ_DEACTIVATE:
{
    volatile Tuint *irqconfig = cirq->irqconfig_addr;

    *irqconfig = 0;
LOG("   # type 0x%x <OFF>, Conf: 0x%x\n", cirq->irq_type, *irqconfig);
}
    cirq->active = FALSE;
    return 0;
}



/* ------------------------------------------------------------------------- */

static struct tq_struct gpmod_isr_bh = {
	NULL, 0, NULL, NULL
};



static void
gpmod_tq_dmairq(GPMCheckIrq_t *cirq)
{

LOG("@@@ - DMA Interrupt !!!\n");

}

/* ------------------------------------------------------------------------- */



static Tint
gpmod_isr_dmairq(GPMCheckIrq_t *cirq, void *dev_id)
{
    volatile Tuint *irqstatus = cirq->irqstatus_addr;
    Tuint stat = *irqstatus;

    if(!(stat & (Tuint)cirq->mask)) return 0;
LOG("@@@ IRQ: C:%d D:%d T:%d (Stat 0x%x)\n",cirq->cardnr, cirq->devnr,
	cirq->irq_type, stat);

    if(cirq->fast) {
	gpmod_tq_dmairq(cirq);
	goto CLEAR_IRQ;
    }

    gpmod_isr_bh.routine = (void(*)(void *)) gpmod_tq_dmairq;
    gpmod_isr_bh.data = cirq;
    queue_task(&gpmod_isr_bh, &tq_immediate);
    mark_bh(IMMEDIATE_BH);

CLEAR_IRQ:
    stat = 0;
    *irqstatus = stat;

    return 1;
}



/*****************************************************************************
 *****************************************************************************/

/* The fields:
 * 1. IRQ description (max 32 chars),
 * 2. chip class (should be defined in specific project_gpm.h file),
 * 3. irq type (should be defined in specific project_gpm.h file),
 * 4. base of offsets, BASE_NONE for PCI config space or BASE_0/-5,
 * 5. offset of IRQ configuration/mask register
 * 6. offset of IRQ status register
 * 7. the function which handles the particular irq type
 */
static const GPMIrq_t SupportedInterrupts[MAX_IRQS_ALOWED] = {

    {"Permedia 2 IRQ", CHIP_CLASS_PERMEDIA2, IRQTYPE_PERMEDIA2,
	BASE_0, 0x08, 0x10, gpmod_isr_dmairq },

    {"Delta IRQ", CHIP_CLASS_DELTA, IRQTYPE_DELTA,
	BASE_0, 0x808, 0x810, gpmod_isr_dmairq },
};

/* the number of elements in above SupportedInterrupts array */
#define NUMBER_OF_SUPPORTED_IRQS		2


/*****************************************************************************
 *****************************************************************************/



static GPMCheckIrq_t *checkirqnr[MAX_SYSTEM_IRQS];
static Tuint isr_handlers[MAX_SYSTEM_IRQS];
static Tbool isr_cards[MAX_CARDS_ALOWED];

static void *ddid; /* dummy device id */



static void
gpmod_isr( int irq, void *dev_id, struct pt_regs *regs)
{
    GPMCheckIrq_t *cirq = checkirqnr[irq];

    while(cirq) {
	if( cirq->handleirq(cirq, dev_id) ) return;
	cirq = cirq->next;
    }
}



#ifdef LOG_DEBUG
static void
gpmod_isr_query( void )
{
    Tint cnt;
    Tbool none = TRUE;

    LOG("\n");
    LOG("##### Registered IRQ handlers: ######\n");
    for(cnt=0; cnt<MAX_SYSTEM_IRQS; cnt++) {
	GPMCheckIrq_t *cirq = checkirqnr[cnt];
	if(!cirq) continue;
	none = FALSE;
	LOG("=-=-= IRQ nr. %d =-=-=-=-=-=-=-=-=-=-\n", cnt);
	while(cirq) {
	    if(cirq->active)
		LOG(" C:%d D:%d, CFG=0x%lx MASK=0x%lx\n", cirq->cardnr,
		    cirq->devnr, (Tulong) cirq->irqconfig_addr, cirq->mask);
	    else
		LOG(" C:%d D:%d, CFG=0x%lx, -inactive-\n", cirq->cardnr,
		    cirq->devnr, (Tulong) cirq->irqconfig_addr);
	    cirq = cirq->next;
	}
    }
    if(none) LOG(" * nothing registered *\n");
    LOG("#####################################\n");
    LOG("\n");
}
#endif



static void
gpmod_isr_initglob( void )
{
    Tuint cnt;
    for(cnt=0;cnt<MAX_SYSTEM_IRQS;cnt++) {
	checkirqnr[cnt] = NULL;
	isr_handlers[cnt] = 0;
    }
    for(cnt=0;cnt<MAX_CARDS_ALOWED;cnt++)
	isr_cards[cnt] = 0;
}



static Tint
gpmod_isr_addcard(Tuint cardnr)
{
    GPMCardInfo_t *card;
    Tuint dcnt, ireg, irq, newhandlers;

    if( cardnr >= PCICFound ) {
	ERR("_add_irqs: Invalid cardnr (%d)\n", cardnr);
	return -EINVAL;
    }

    card = PCICard + cardnr;
LOG("(ISR) ADD #%d, %s\n", cardnr, SupportedCards[card->idx].name);

    if( isr_cards[cardnr] ) {
LOG(" === Already inited!\n");
	return 0;
    }

    for(dcnt = 0; dcnt < card->maxdev; dcnt++) {
	GPMDevInfo_t *dev = PCIDevice + card->cd[dcnt].devidx;
	GPMDev_t *sdev = SupportedDevices + dev->idx;
	GPMCheckIrq_t **cirqptr, *cirqfirst = NULL, *cirq;

	irq = dev->irq;
LOG(" >>> Dev #%d: IRQ line (%d)\n", dcnt, irq);

	if(irq >= MAX_SYSTEM_IRQS) {
	    ERR("_add_irqs: Invalid IRQ number\n");
	    return -EINVAL;
	}

	/*
	 * first alloc check-irq structures for every irq register
	 * on specific device (one device = one pci irq)
	 */
	cirqptr = &cirqfirst;
	newhandlers = 0;
	for(ireg = 0; ireg < NUMBER_OF_SUPPORTED_IRQS; ireg++) {
	    const GPMIrq_t *sirq = SupportedInterrupts + ireg;
	    void *baddr;

	    if(sirq->chip_type != sdev->chip_type) continue;

	    if(!(cirq = (GPMCheckIrq_t *)
			kmalloc(sizeof(GPMCheckIrq_t),GFP_KERNEL)) ) {
		ERR("_add_irqs: kmalloc()\n");
		return -ENOMEM;
	    }

	    if(sirq->base < BASE_PCI) {
		GPMBase_t *ba = dev->PCIBase + sirq->base;
		if(sirq->offset_config >= ba->size) {
		    ERR("_add_irqs: bad offset_config (0x%lx)\n",
			sirq->offset_config);
		    kfree(cirq);
		    continue;
		}
		if(sirq->offset_status >= ba->size) {
		    ERR("_add_irqs: bad offset_status (0x%lx)\n",
			sirq->offset_status);
		    kfree(cirq);
		    continue;
		}

		if(ba->flags & PCI_BASE_ADDRESS_SPACE_IO) {
		    baddr = (void *) ba->base;
		}
		else if(!(baddr = dev->PCIKmapBase[sirq->base])) {
		    ERR("_add_irqs: Base %d not kernel mapped!\n", sirq->base);
		    kfree(cirq);
		    continue;
		}
	    }
	    else if(sirq->base == BASE_PCI) {
		baddr = (void *) 0;
	    }
	    else {
		ERR("_add_irqs: bad base (%d)\n", sirq->base);
		kfree(cirq);
		continue;
	    }

	    cirq->cardnr = cardnr;
	    cirq->devnr = dcnt;
	    cirq->irq_type = sirq->irq_type;

	    cirq->active = FALSE;
	    cirq->fast = (sdev->irqflags & FAST_IRQ) ? TRUE : FALSE;
	    cirq->mask = 0;

	    cirq->irqconfig_addr = baddr + sirq->offset_config;
	    cirq->irqstatus_addr = baddr + sirq->offset_status;
	    cirq->handleirq = sirq->handleirq;

	    cirq->next = NULL;

	    *cirqptr = cirq;
	    cirqptr = &cirq->next;
	    newhandlers++;

LOG("   + (%d): %s\n",sirq->irq_type, sirq->name);
	}

	if(!newhandlers) {
LOG("   * IRQ HANDLING NOT DEFINED *\n");
	    continue;
	}

	/*
	 * prepare the handler...
	 */
	if( ! isr_handlers[irq] ) {
	    Tint err = 1;
	    if(sdev->irqflags & FAST_IRQ) {
		if((err = request_irq(irq, gpmod_isr, SA_SHIRQ | SA_INTERRUPT,
			"gpmod_isr", ddid))) {
		    ERR("_add_irqs: SA_INTERRUPT failed\n");
		}
#ifdef LOG_DEBUG
		else LOG("   > Attach device and init fast ISR\n");
#endif
	    }

	    if(err) {
		if((err = request_irq(irq, gpmod_isr, SA_SHIRQ,
			"gpmod_isr", ddid))) {
		    ERR("_add_irqs: request_irq() failed\n");
		}
#ifdef LOG_DEBUG
		else LOG("   > Attach device and init ISR\n");
#endif
	    }

	    if(err) {
		GPMCheckIrq_t *next, *prev = cirqfirst;
		do {
		    next = prev->next;
		    kfree(prev);
		} while((prev = next));
		return -EAGAIN;
	    }
	}
#ifdef LOG_DEBUG
	else {
	    LOG("   > Attach device to ISR\n");
	}
#endif

	/*
	 * add new device to ISR
	 */
	cirqptr = &checkirqnr[irq];
	while((cirq = *cirqptr)) cirqptr=&cirq->next;
	*cirqptr = cirqfirst;

	isr_handlers[irq] += newhandlers;
    }

    isr_cards[cardnr] = TRUE;

#ifdef LOG_DEBUG
    gpmod_isr_query();
#endif

    return 0;
}



static Tint
gpmod_isr_remcard(Tuint cardnr)
{
    GPMCardInfo_t *card;
    Tuint dcnt, irq;

    if( cardnr >= PCICFound ) {
	ERR("_add_irqs: Invalid cardnr (%d)\n", cardnr);
	return -EINVAL;
    }

    card = PCICard + cardnr;
LOG("(ISR) REMOVE #%d, %s\n", cardnr, SupportedCards[card->idx].name);

    if( !isr_cards[cardnr] ) {
LOG(" === Not inited yet!\n");
	return 0;
    }

    for(dcnt = 0; dcnt < card->maxdev; dcnt++) {
	GPMDevInfo_t *dev = PCIDevice + card->cd[dcnt].devidx;
	GPMCheckIrq_t **cirqptr, *cirqfirst, *cirqlast, *cirq;

	irq = dev->irq;
LOG(" >>> Dev #%d: IRQ line (%d)\n", dcnt, irq);

	/*
	 * FIRST, find cardnr in array of particular irqnr
	 */
	cirqptr = &checkirqnr[irq];
	while((cirqfirst = *cirqptr)) {
	    if(cirqfirst->cardnr == cardnr) break;
	    cirqptr = &cirqfirst->next;
	}

	if(!cirqfirst) {
LOG("     - already detached -\n");
	    continue;
	}
	    
	/*
	 * NEXT, deactivate all IRQs with that cardnr.
	 * devnr counter is for the very impossible case that different
	 * devices on the same card (slot) have different irqnr's
	 */
	cirq = cirqfirst;
	do {
	    gpmod_isr_deactivate(cirq);
	    isr_handlers[irq]--;
	    cirqlast = cirq;
	    if(!(cirq = cirq->next)) break;
	} while(cirq->cardnr == cardnr);

	/*
	 * NEXT, detach records from ISR and release ISR
	 */
	*cirqptr = cirq;
	cirqlast->next = NULL;


	if( ! isr_handlers[irq] ) {
LOG("   > Detach device and free ISR\n");
	    free_irq(irq, ddid);
	}
#ifdef LOG_DEBUG
	else {
	    LOG("   > Detach device\n");
	}
#endif

	/*
	 * LAST, delete all cardnr && irqnr records
	 */
	cirqlast = cirqfirst;
	do {
	    cirq = cirqlast->next;
LOG("   - free type: 0x%x\n",cirqlast->irq_type);
	    kfree(cirqlast);
	} while((cirqlast = cirq));

    }

    isr_cards[cardnr] = FALSE;

#ifdef LOG_DEBUG
    gpmod_isr_query();
#endif

    return 0;
}



static Tint
gpmod_isr_config(GPMIsrCfg_t * u_par)
{
    GPMCheckIrq_t *cirq;
    GPMCardInfo_t *card;
    GPMDevInfo_t *dev;
    GPMIsrCfg_t par;
    Tubyte cardnr, devnr;

    MCOPY_FROM_USER(&par, u_par, GPMIsrCfg_t, "_isr_config");

    cardnr = par.cardnr; devnr = par.devnr;

    if (cardnr >= PCICFound) {
	ERR("_isr_config: cardnr (%d)\n", cardnr);
	return -EINVAL;
    }

    card = PCICard + cardnr;
LOG("(ISR) CONFIG #%d, %s\n", cardnr, SupportedCards[card->idx].name);

    if( !isr_cards[cardnr] ) {
	ERR("_isr_config: ISR NOT INITED! (C:%d)\n", cardnr);
	return -EINVAL;
    }

    if (devnr >= card->maxdev) {
	ERR("_isr_config: devnr (%d)\n", devnr);
	return -EINVAL;
    }

    dev = PCIDevice + card->cd[devnr].devidx;
LOG(" >>> Dev #%d: IRQ line (%d)\n", devnr, dev->irq);

    cirq = checkirqnr[dev->irq];
    while(cirq) {
	if(cirq->cardnr == cardnr &&
	   cirq->devnr == devnr &&
	   cirq->irq_type == par.irqtype) break;
	cirq = cirq->next;
    }

    if( !cirq ) {
	ERR("_isr_config: IRQ missing (C:%d, D:%d T:0x%x)\n",
	    cardnr, devnr, par.irqtype);
	return -EINVAL;
    }

    if( cirq->mask != par.mask ) {
LOG(" === Set MASK: 0x%lx\n", par.mask);
	cirq->mask = par.mask;
	gpmod_isr_setconfig(cirq);
#ifdef LOG_DEBUG
	gpmod_isr_query();
#endif
    }

    return 0;
}



static Tint
gpmod_getareas(GPMIoctlGetArea_t * u_par)
{
    GPMIoctlGetArea_t par;
    GPMIoctlArea_t area, *u_area;
    GPMDmaArea_t *dbarea;
    GPMDmaBuff_t *db;
    Tint cnt;

    MCOPY_FROM_USER(&par, u_par, GPMIoctlGetArea_t, "_getareas");

    if(!(db = dmabuff_find(par.dbnr)) ) {
	ERR("_getareas: Unknown dbnr (%d)\n", par.dbnr);
	return -EINVAL;
    }

    dbarea = db->areas;
    u_area = par.areas;

    for(cnt=0; cnt<db->cnt_areas; cnt++) {
	area.addr = dbarea->addr;
	area.size = dbarea->size;
	MCOPY_TO_USER(u_area, &area, GPMIoctlArea_t, "_getareas");
	dbarea = dbarea->next;
	u_area++;
    }
    return 0;
}



/*****************************************************************************
 *** IOCTL *******************************************************************
 *****************************************************************************/

static int
gpmod_fops_ioctl(struct inode *inode, struct file *file, Tuint cmd, Tulong arg)
{
    switch (cmd) {
    case IOCTL_PCI_GETCARDDATA:
	return gpmod_docardinfo((GPMCardInfo_t *) arg);
    case IOCTL_PCI_SETCONFIG:
	return gpmod_pciconfig((GPMPciParam_t *) arg, TRUE, TRUE);
    case IOCTL_PCI_GETCONFIG:
	return gpmod_pciconfig((GPMPciParam_t *) arg, FALSE, TRUE);
    case IOCTL_PCI_WRITEBASE:
	return gpmod_rwbase((GPMIoctlRw_t *) arg, TRUE, TRUE);
    case IOCTL_PCI_READBASE:
	return gpmod_rwbase((GPMIoctlRw_t *) arg, FALSE, TRUE);
    case IOCTL_PCI_ISRINSTALL:
	return gpmod_isr_addcard((Tuint) arg);
    case IOCTL_PCI_ISRUNINSTALL:
	return gpmod_isr_remcard((Tuint) arg);
    case IOCTL_PCI_ISRCONFIG:
	return gpmod_isr_config((GPMIsrCfg_t *) arg);
    case IOCTL_PCI_GETAREAS:
	return gpmod_getareas((GPMIoctlGetArea_t *) arg);
    default:
	ERR("IOCTL/ (%d) unknown!\n", cmd);
	return -EINVAL;
    }
    return 0;
}



/*****************************************************************************
 *** MMAP ********************************************************************
 *****************************************************************************/

/*
 * Some code directly stolen from Alessandro Rubini... Thanks Alessandro.
 * Without your heroic act to write your book, I'd still be waiting in
 * darkness for 'blitz' of enlightenment.
 */
#if 0
static Tulong
gpmod_vaddrtopage( void *vaddr )
{
    Tulong page;
    struct task_struct *p;
    struct mm_struct *init_mm_ptr;
    pgd_t *pgd; pmd_t *pmd; pte_t *pte;

    p = current;
    do {
	if (p->pid == 0) break;
    }
    while ((p = p->next_task) != current);
    init_mm_ptr = p->mm;

    page = VMALLOC_VMADDR(vaddr);

    pgd = pgd_offset(init_mm_ptr, page);
    pmd = pmd_offset(pgd, page);
    pte = pte_offset(pmd, page);
    page = pte_page(*pte);        /* this is the physical address */

    return page;
}
#endif



/*
 * MMAP /dev/...pci
 *******************/

#if LINUX_2_2
static int
gpmod_fops_mmap_pci( struct file *file,
	struct vm_area_struct *vma)
#elif LINUX_2_0
static int
gpmod_fops_mmap_pci( struct inode *inode, struct file *file,
	struct vm_area_struct *vma)
#endif
{
    Tulong vmpos = vma->vm_start, vmsize = (vma->vm_end-vma->vm_start);
    Tulong addr = vma->vm_offset;

LOG("MMAP pci/ V: 0x%lx, S: 0x%lx\n", vmpos, vmsize);

    if(gpmod_pcimmap_isvalid(addr)) {
	ERR("_mmap_pci: address denied (0x%lx)\n", addr);
	return -EINVAL;
    }

    /*
     * remap_page_range() is going to align addr, but... why not ?
     */
    addr &= PAGE_MASK;

LOG("MMAP pci/ ===> PCI addr 0x%lx\n", addr);

    if( remap_page_range(vmpos, addr, vmsize, vma->vm_page_prot)) {
	ERR("_mmap_pci: remap_page_range()\n");
	return -EFAULT;
    }

    vma->vm_offset = 0;

#if LINUX_2_2
    vma->vm_file = file;
    file->f_count++;
#elif LINUX_2_0
    vma->vm_inode = inode;
    inode->i_count++;
#endif

#if LINUX_2_0
    MOD_INC_USE_COUNT;
#endif

    return 0;
}



/*
 * /dev/...mem VM operations
 ****************************/

typedef struct MmapBuffData_t {
    GPMDmaBuff_t		*dbuff;
    Tulong			vmaddr;
    struct MmapBuffData_t	*next;
} MmapBuffData_t;

typedef struct MmapPIDData_t {
    Tint			pid;
    MmapBuffData_t		*buffers;
    struct MmapPIDData_t	*next;
} MmapPIDData_t;

static MmapPIDData_t		*mmappings = NULL;



static Tint
gpmod_add_mmapping( Tulong vmaddr, GPMDmaBuff_t *db )
{
    Tint pid = current->pid;
    MmapPIDData_t *mmpid, **mmpidptr;
    MmapBuffData_t *mmbuff, **mmbuffptr;

    /* find PID, if already there */
    mmpidptr = &mmappings;
    while(*mmpidptr) {
	if(pid == (*mmpidptr)->pid) break;
	mmpidptr = &(*mmpidptr)->next;
    }

    if(!*mmpidptr) {
	if(!(mmpid =
		(MmapPIDData_t *)kmalloc(sizeof(MmapPIDData_t), GFP_KERNEL))) {
	    ERR("_add_mmapping: 1. kmalloc()\n");
	    return -ENOMEM;
	}
	*mmpidptr = mmpid;
	mmpid->pid = pid;
	mmpid->buffers = NULL;
	mmpid->next = NULL;
    }

    /* new buffer info record comes at the end */
    mmbuffptr = &(*mmpidptr)->buffers;
    while(*mmbuffptr) mmbuffptr = &(*mmbuffptr)->next;

    if(!(mmbuff =
	    (MmapBuffData_t *)kmalloc(sizeof(MmapBuffData_t), GFP_KERNEL))) {
	ERR("_add_mmapping: 2. kmalloc()\n");
	return -ENOMEM;
    }

    *mmbuffptr = mmbuff;
    mmbuff->dbuff = db;
    mmbuff->vmaddr = vmaddr;
    mmbuff->next = NULL;

    db->cnt_mmaps++;
    return 0;
}



static GPMDmaBuff_t *
gpmod_remove_mmapping( Tulong vmaddr )
{
    GPMDmaBuff_t *db;
    Tint pid = current->pid;
    MmapPIDData_t *mmpid, **mmpidptr;
    MmapBuffData_t *mmbuff, **mmbuffptr;

    /* find PID */
    mmpidptr = &mmappings;
    while(*mmpidptr) {
	if(pid == (*mmpidptr)->pid) break;
	mmpidptr = &(*mmpidptr)->next;
    }
    if(!(mmpid = *mmpidptr)) return NULL;

    /* find buffer */
    mmbuffptr = &mmpid->buffers;
    while(*mmbuffptr) {
	if(vmaddr == (*mmbuffptr)->vmaddr) break;
	mmbuffptr = &(*mmbuffptr)->next;
    }
    if(!(mmbuff = *mmbuffptr)) return NULL;

    /* release the records not needed anymore */

    db = mmbuff->dbuff;
    db->cnt_mmaps--;
    *mmbuffptr = mmbuff->next;
    kfree(mmbuff);

    if(!(mmpid->buffers)) {
	*mmpidptr = mmpid->next;
	kfree(mmpid);
    }
    return db;
}



#ifdef LOG_DEBUG
static void
gpmod_show_mmappings( void )
{
    MmapPIDData_t *mmpid = mmappings;
    MmapBuffData_t *mmbuff;

    LOG("\n");
    LOG("===== Current Memory Mappings =====\n");
    if(!mmpid)
	LOG(" *** NONE ***\n");
    while(mmpid) {
	LOG(" ** PID %d:\n", mmpid->pid);
	mmbuff = mmpid->buffers;
	while(mmbuff) {
	    LOG("  -> Buff #%d to VM 0x%lx\n",
		mmbuff->dbuff->nr, mmbuff->vmaddr);
	    mmbuff = mmbuff->next;
	}
	mmpid = mmpid->next;
    }
    LOG("===================================\n");
    LOG("\n");
}
#endif

#if 0
static void
gpmod_vmops_open(struct vm_area_struct *area)
{
LOG("vm_open/\n");
    MOD_INC_USE_COUNT;
}

static void
gpmod_vmops_close(struct vm_area_struct *area)
{
LOG("vm_close/\n");
    MOD_DEC_USE_COUNT;
}
#endif

static void
gpmod_vmops_unmap(struct vm_area_struct *area,
	       Tulong offset, size_t size)
{
    GPMDmaBuff_t *db;
LOG("\n");
LOG("vm_unmap/ O=0x%lx S=0x%x\n", offset, size);

    if((db = gpmod_remove_mmapping(offset)))
	if(!db->cnt_mmaps) dmabuff_free(db);

#ifdef LOG_DEBUG
    gpmod_show_mmappings();
#endif
}

#if 0
static Tulong
gpmod_vmops_nopage(struct vm_area_struct *vmarea,
		Tulong address, int write_access)
{
    Tulong page = 0;
LOG("vm_nopage/ A=0x%lx V=0x%lx\n", address, vmarea->vm_start);
    return( page );
}
#endif

static struct vm_operations_struct gpmod_vmops =
{
    NULL, /* gpmod_vmops_open, */
    NULL, /* gpmod_vmops_close, */
    gpmod_vmops_unmap,
    NULL,
    NULL,
    NULL,
    NULL, /* gpmod_vmops_nopage, */
    NULL,
    NULL,
    NULL
};



/*
 * MMAP /dev/...mem
 *******************/

#if LINUX_2_2
static int
gpmod_fops_mmap_mem( struct file *file,
	struct vm_area_struct *vma)
#elif LINUX_2_0
static int
gpmod_fops_mmap_mem( struct inode *inode, struct file *file,
	struct vm_area_struct *vma)
#endif
{
    Tint ret;
    GPMDmaBuff_t *db;
    GPMMmapMem_t par, *u_par;
    Tulong vmpos, vmsize;
    Tbool free_if_remap_fails;

    u_par = (GPMMmapMem_t *) (vma->vm_offset);
    vmpos = vma->vm_start;
    vmsize = vma->vm_end - vma->vm_start;
    free_if_remap_fails = TRUE;

LOG("MMAP mem/ VM s=0x%lx e=0x%lx\n", vmpos, vma->vm_end);

    MCOPY_FROM_USER(&par, u_par, GPMMmapMem_t, "_mmap_mem");

    vma->vm_ops = &gpmod_vmops;

    switch(par.action) {
    case MMAP_MEM_ALLOC:
LOG("MMAP mem/ *** MMAP_MEM_ALLOC\n");
	if(!(db = dmabuff_alloc(vmsize, par.flags)) ) {
	    return -ENOMEM;
	}
	break;

    case MMAP_MEM_SHARE:
LOG("MMAP mem/ *** MMAP_MEM_SHARE (Nr. %d)\n", par.buffer_nr);
	if( par.buffer_nr > 0 ) {
	    ERR("MMAP mem/ Buffnr should be <= 0!\n");
	    return -EINVAL;
	}
	free_if_remap_fails = FALSE;
	if(!(db = dmabuff_find(par.buffer_nr)) ) {
	    free_if_remap_fails = TRUE;
	    if(!(db = dmabuff_alloc(vmsize, par.flags)) ) {
		return -ENOMEM;
	    }
	    db->nr = par.buffer_nr;
	}
	break;

    default:
	ERR("MMAP mem/ Unknown action (%d)\n", par.action);
	return -EINVAL;
    }

#ifdef DMA_DYNAMIC_ALLOC
    vma->vm_flags |= VM_LOCKED;
#endif

    if (dmaareas_remap( &vmpos, db->areas, vma->vm_page_prot )) {
	ERR("MMAP mem/ dmaareas_remap()\n");
	if(free_if_remap_fails) dmabuff_free(db);
	return -EFAULT;
    }

    if( (ret = gpmod_add_mmapping(vma->vm_start, db)) ) {
	ERR("MMAP mem/ gpmod_add_mmapping()\n");
	return ret;
    }

#ifdef LOG_DEBUG
    dmabuff_query();
    gpmod_show_mmappings();
#endif

    par.buffer_nr = db->nr;
    par.ret_areas_cnt = db->cnt_areas;

    MCOPY_TO_USER(u_par, &par, GPMMmapMem_t, "_mmap_mem");

    vma->vm_offset = 0;

#if LINUX_2_2
    vma->vm_file = file;
    file->f_count++;
#elif LINUX_2_0
    vma->vm_inode = inode;
    inode->i_count++;
#endif

#if LINUX_2_0
    MOD_INC_USE_COUNT;
#endif

    return 0;
}



/*****************************************************************************
 *** RELEASE *****************************************************************
 *****************************************************************************/

#if LINUX_2_2
static int
gpmod_fops_release(struct inode *inode, struct file *file)
#elif LINUX_2_0
static void
gpmod_fops_release(struct inode *inode, struct file *file)
#endif
{
LOG("RELEASE/ (minor %d)\n", MINOR(inode->i_rdev));

#if LINUX_2_0
    MOD_DEC_USE_COUNT;
#endif

#if LINUX_2_2
    return 0;
#endif
}



static int
gpmod_fops_open(struct inode *, struct file *);



static struct file_operations gpmod_fops_pci = {
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    gpmod_fops_ioctl,
    gpmod_fops_mmap_pci,
    gpmod_fops_open,
#if LINUX_2_2
    NULL,
#endif
    gpmod_fops_release,
    NULL,
    NULL,
    NULL,
    NULL
};

static struct file_operations gpmod_fops_mem = {
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    gpmod_fops_mmap_mem,
    gpmod_fops_open,
#if LINUX_2_2
    NULL,
#endif
    gpmod_fops_release,
    NULL,
    NULL,
    NULL,
    NULL
};

static struct file_operations *gpmod_fops_minors[] = {
    &gpmod_fops_pci,
    &gpmod_fops_mem,
};



/*****************************************************************************
 *** OPEN ********************************************************************
 *****************************************************************************/
static int
gpmod_fops_open(struct inode *inode, struct file *file)
{
    Tint minor = MINOR(inode->i_rdev);
LOG("OPEN/ (minor %d)\n", minor);

    if(minor < 2)
	file->f_op = gpmod_fops_minors[minor];

#if LINUX_2_0
    MOD_INC_USE_COUNT;
#endif

    return 0;
}



/*****************************************************************************
 *** INITIALIZATION **********************************************************
 *****************************************************************************/ 
static void
gpmod_init_func(void)
{
    gpmod_scanbus();
    gpmod_load_cdata();
    gpmod_kmmap_devices();
    High_meminit();
    gpmod_isr_initglob();
}



static void
gpmod_cleanup_func(void)
{
    int cardnr=0;

    /* free all DMA buffers */
    while(dmabuff_free(first_dma_buffer));

    /* free all ISR resources */
    while(cardnr < PCICFound) gpmod_isr_remcard(cardnr++);

    /* free records with valid mmaps */
    gpmod_pcimmap_end();

    /* free virtual areas claimed by module */
    gpmod_virtual_end();
}



int
init_module(void)
{
    Tuint regdev;

    MSG("\n");
    MSG( PROJECT_INFORMATION "\n");
    MSG("+ Generic PCI module + <sim@suse.de>\n");
    MSG("Compiled on <" __TIME__ " " __DATE__ ">\n");
    MSG("Kernel release: " UTS_RELEASE "\n");
    MSG("\n");

    if((regdev = register_chrdev(drvmajor, driver, &gpmod_fops_pci))== -EBUSY){
	MSG("Unable to register device %d\n", drvmajor);
	return -EIO;
    }
    if (drvmajor == 0)
	drvmajor = regdev;

    gpmod_init_func();

    MSG("\n");
    MSG("Device (%d) registered\n", drvmajor);
    MSG("...LOADING FINISHED.\n");
    MSG("\n");

    return 0;
}



void
cleanup_module(void)
{
    MSG("\n");
    MSG("REMOVING MODULE...\n");

    unregister_chrdev(drvmajor, driver);
    MSG("Device (%d) unregistered\n", drvmajor);
    MSG("\n");

    gpmod_cleanup_func();

    MSG("\n");
    MSG("...MODULE REMOVED.\n");
    MSG("\n");
    return;
}
#undef __KERNEL__
