
/******************************************************************************
 *                                   ACL API                                  *
 *                 Rendering Library for Accelerated 3d Hardware              *
 *                         (C) David Bucciarelli, 1998                        *
 *                           (C) SuSE GmbH 1997, 1998                         *
 ******************************************************************************/
/* authors: david bucciarelli, <humanware@plus.it>
            simon pogarcic, <sim@suse.de>
 */

/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */



#ifndef _ACLAPI_H_
#define _ACLAPI_H_

/*-----------------------------------------------------------------------------
 
 Last version changes:
	-
	-
	-

 Contributors:
	alexander stohr <AlexS@freepage.de>

 ******************************************************************************
 ******************************************************************************/

#define _ACL_API_VERSION_STRING_ \
	"ACL API Version 3.1, February 1999"



/*+****************************************************************************
 ******************************************************************************
 
 DESCRIPTION: data types
 ------------------------------------------------------------------------------

 ******************************************************************************
 ******************************************************************************/

#ifndef _LOCAL_TYPES_

/* 8 bit: */
typedef char			Tbyte;
typedef unsigned char		Tubyte;
    
/* 16 bit: */
typedef short			Tshort;
typedef unsigned short		Tushort;

/* 32 bit: */    
typedef int			Tint;
typedef unsigned int		Tuint;

/* 64 or 32 bit */
typedef long			Tlong;
typedef unsigned long		Tulong;

/* single precission: */    
typedef float			Tfloat;

/* double precission: */
typedef double			Tdouble;

/* boolean */
typedef unsigned char		Tbool;

/* pointer to 64 bit value (sometimes) */
typedef unsigned long*		Tptr64;

/* pointer to 32 bit value */
typedef unsigned int*		Tptr32;

/* pointer to 16 bit value */
typedef unsigned short*		Tptr16;

/* pointer to 8 bit value */
typedef unsigned char*		Tptr8;

#else

#include "local_types.h"

#endif



#ifndef DEPTH_BITS
  #define DEPTH_BITS 16
#endif



#ifdef _ACL_OPENGL_

#include <GL/gl.h>

#define ACLGL_FALSE 		GL_FALSE
#define ACLGL_TRUE		GL_TRUE

#define ACLGL_FRONT		GL_FRONT
#define ACLGL_BACK		GL_BACK
#define ACLGL_FRONT_AND_BACK	GL_FRONT_AND_BACK

#define ACLGL_COLOR_BUFFER_BIT	GL_COLOR_BUFFER_BIT
#define ACLGL_DEPTH_BUFFER_BIT	GL_DEPTH_BUFFER_BIT
#define ACLGL_ACCUM_BUFFER_BIT  GL_ACCUM_BUFFER_BIT
#define ACLGL_STENCIL_BUFFER_BIT GL_STENCIL_BUFFER_BIT
#define ACLGL_SCISSOR_BIT	GL_SCISSOR_BIT
#define ACLGL_ALL_ATTRIB_BITS	GL_ALL_ATTRIB_BITS

#define ACLGL_NEVER		GL_NEVER
#define ACLGL_LESS		GL_LESS
#define ACLGL_GEQUAL		GL_GEQUAL
#define ACLGL_LEQUAL		GL_LEQUAL
#define ACLGL_GREATER		GL_GREATER
#define ACLGL_NOTEQUAL		GL_NOTEQUAL
#define ACLGL_EQUAL		GL_EQUAL
#define ACLGL_ALWAYS		GL_ALWAYS

#define ACLGL_CLEAR		GL_CLEAR
#define ACLGL_AND		GL_AND
#define ACLGL_AND_REVERSE	GL_AND_REVERSE
#define ACLGL_COPY		GL_COPY
#define ACLGL_AND_INVERTED	GL_AND_INVERTED
#define ACLGL_NOOP		GL_NOOP
#define ACLGL_XOR		GL_XOR
#define ACLGL_OR		GL_OR
#define ACLGL_NOR		GL_NOR
#define ACLGL_EQUIV		GL_EQUIV
#define ACLGL_INVERT		GL_INVERT
#define ACLGL_OR_REVERSE	GL_OR_REVERSE
#define ACLGL_COPY_INVERTED	GL_COPY_INVERTED
#define ACLGL_OR_INVERTED	GL_OR_INVERTED
#define ACLGL_NAND		GL_NAND
#define ACLGL_SET		GL_SET

#define ACLGL_KEEP		GL_KEEP
#define ACLGL_ZERO		GL_ZERO
#define ACLGL_REPLACE		GL_REPLACE
#define ACLGL_INCR		GL_INCR
#define ACLGL_DECR		GL_DECR

#define ACLglenum		GLenum

#if DEPTH_BITS == 16
  #define ACLgldepth		GLushort
#elif DEPTH_BITS == 32
  #define ACLgldepth		GLint;
#else
    illegal number of depth bits
#endif

#else

typedef enum {
	ACLGL_FALSE = 		0,
	ACLGL_TRUE =		1,

	ACLGL_FRONT =		0x0404,
	ACLGL_BACK =		0x0405,
	ACLGL_FRONT_AND_BACK =	0x0408,

	ACLGL_DEPTH_BUFFER_BIT =   0x00000100,
	ACLGL_ACCUM_BUFFER_BIT =   0x00000200,
	ACLGL_STENCIL_BUFFER_BIT = 0x00000400,
	ACLGL_COLOR_BUFFER_BIT =   0x00004000,
        ACLGL_SCISSOR_BIT =	   0x00080000,
	ACLGL_ALL_ATTRIB_BITS =	   0x000fffff,

        ACLGL_NEVER =		0x0200,
        ACLGL_LESS =		0x0201,
        ACLGL_GEQUAL =		0x0206,
        ACLGL_LEQUAL =		0x0203,
        ACLGL_GREATER =		0x0204,
        ACLGL_NOTEQUAL =	0x0205,
        ACLGL_EQUAL =		0x0202,
        ACLGL_ALWAYS =		0x0207,

	ACLGL_CLEAR =		0x1500,
        ACLGL_AND =		0x1501,
	ACLGL_AND_REVERSE =	0x1502,
	ACLGL_COPY =		0x1503,
	ACLGL_AND_INVERTED =	0x1504,
	ACLGL_NOOP =		0x1505,
	ACLGL_XOR =		0x1506,
	ACLGL_OR =		0x1507,
	ACLGL_NOR =		0x1508,
	ACLGL_EQUIV =		0x1509,
	ACLGL_INVERT =		0x150a,
	ACLGL_OR_REVERSE =	0x150b,
	ACLGL_COPY_INVERTED =	0x150c,
	ACLGL_OR_INVERTED =	0x150d,
	ACLGL_NAND =		0x150e,
	ACLGL_SET =		0x150f,

	ACLGL_ZERO =		0,
	ACLGL_KEEP =		0x1e00,
	ACLGL_REPLACE =		0x1e01,
	ACLGL_INCR =		0x1e02,
	ACLGL_DECR =		0x1e03,
} ACLglenum;

#if DEPTH_BITS == 16
  typedef unsigned short	ACLgldepth;
#elif DEPTH_BITS == 32
  typedef int			ACLgldepth;
#else
    illegal number of depth bits
#endif

#endif



typedef enum {
	ACLRENDER_FLAT =		0x00000001,
	ACLRENDER_SMOOTH =		0x00000002,
	ACLRENDER_TWOSIDE =		0x00000004,
	ACLRENDER_TEX =			0x00000008,
	ACLRENDER_STRIP =		0x00000010,
	ACLRENDER_FAN =			0x00000020,
	ACLRENDER_LOOP =		0x00000040,

 	ACLOPT_DBUFF_BITBLIT =		0x00000001,
	ACLOPT_DBUFF_BITPLANE =		0x00000002,
	ACLOPT_DBUFF_SCREEN =		0x00000004,
	ACLOPT_DBUFF_MASK =		~0x7,
	ACLOPT_QUAKE_OFF =		0x00000008,
	ACLOPT_QUAKE_ON =		0x00000010,
 	ACLOPT_XDGA_ON =		0x00000020,
	ACLOPT_XDGA_OFF =		0x00000040,
 	ACLOPT_VIDMODE_OFF =		0x00000080,
	ACLOPT_VIDMODE_ON =		0x00000100,
	ACLOPT_RUNMODE_FULL =		0x00000200,
	ACLOPT_RUNMODE_OFF =		0x00000400,
	ACLOPT_RUNMODE_WIN =		0x00000800,
	ACLOPT_RUNMODE_MASK =		~0xe00,
	ACLOPT_HWIDX_LIST =		0x00001000,

	ACLMOD_RGBA =			0x00000001,
	ACLMOD_CI =			0x00000002,
	ACLMOD_LB =			0x00000004,
	ACLMOD_ACCEL =			0x00000008,
	ACLMOD_TEX =			0x00000010,
	ACLMOD_ALL_RGBA =		0x1d,
	ACLMOD_ALL_CI =			0x1e,
	ACLMOD_ALL =			0x1f,

	ACLPRIM_NONE =			0x0,
	ACLPRIM_POINTS =		0x00000001,
	ACLPRIM_LINE =			0x00000002,
	ACLPRIM_TRIANGLE =		0x00000004,
	ACLPRIM_QUAD =			0x00000008,
	ACLPRIM_RECT =			0x00000010,
	ACLPRIM_ALL_SINGLE =		0x1f,
	ACLPRIM_VB_LINES =		0x00000020,
	ACLPRIM_VB_TRIANGLES =		0x00000040,
	ACLPRIM_VB_QUADS =		0x00000080,
	ACLPRIM_VB_RECTS =		0x00000100,
	ACLPRIM_ALL_VB =		0x1e0,

	ACLHW_VXFORMAT =		0x00000001,

	ACLHW_REN_VXDELTA =		0x00000001,
	ACLHW_REN_MESH =		0x00000002,
	ACLHW_REN_FAN =			0x00000004,
	ACLHW_REN_BACKFACE_CULL =	0x00000008,
	ACLHW_REN_SUBPIX_CORRECTION =	0x00000010,

	ACLHW_DEPTH =			0x00000001,

	ACLHW_STENCIL =			0x00000001,

	ACLHW_SCISSOR =			0x00000001,

	ACLHW_STIPPLE =			0x00000001,

	ACLHW_ALPHA =			0x00000001,
	
	ACLHW_BLEND =			0x00000001,
	ACLHW_BLEND_ADD_EXT_ONLY = 	0x00000002,

	ACLHW_TEX =			0x00000001,
	ACLHW_TEX_1D =			0x00000002,
	ACLHW_TEX_2D =			0x00000004,
	ACLHW_TEX_3D =			0x00000008,
	ACLHW_TEX_SPECULAR =		0x00000010,
	ACLHW_TEX_DIFFUSE =		0x00000020,
	ACLHW_TEX_MODULATE =		0x00000040,
	ACLHW_TEX_DECAL =		0x00000080,
	ACLHW_TEX_COPY =		0x00000100,
	ACLHW_TEX_HIGHLIGHT =		0x00000200,
	ACLHW_TEX_MODULATE_ONLY =	0x00000400,

	ACLHW_COLOR =			0x00000001,
	ACLHW_COLOR_CI =		0x00000002,
	ACLHW_COLOR_YUV =		0x00000004,
	ACLHW_COLOR_SEPARATE_SPECULAR =	0x00000008,
	ACLHW_COLOR_MASK =		0x00000010,

	ACLHW_FOG =			0x00000001,

	ACLHW_LOGOP =			0x00000001,
	ACLHW_LOGOP_COPY_ONLY =		0x00000002,

	ACLFLAG_ALPHA = 		0x00000001,
	ACLFLAG_SPAN =			0x00000002,
	ACLFLAG_MONO =			0x00000004,
	ACLFLAG_DEPTH =			0x00000010

} ACLflags;



typedef struct acl_color_rgba_mod	*ACLColorRGBAMod;
typedef struct acl_color_ci_mod		*ACLColorCIMod;
typedef struct acl_lb_mod 		*ACLLBMod;
typedef struct acl_accel_mod		*ACLAccelMod;
typedef struct acl_tex_mod 		*ACLTexMod;
typedef struct acl_modules 		*ACLModules;

typedef struct acl_vertices		*ACLVertices;
typedef struct acl_visual 		*ACLVisual;
typedef struct acl_tex_image		*ACLTexImage;
typedef struct acl_screen		*ACLScreen;
typedef struct acl_context 		*ACLContext;
typedef struct acl_hardware		*ACLHardware;
typedef struct acl_features		*ACLFeatures;
typedef struct acl_render		*ACLRender;



#define ACLDIM_MAX_TMUNITS		0x3
#define ACLDIM_MAX_LODS			0x8

/*************************************
 Hardware features
 *************************************/
struct acl_features {
	ACLflags	vxformat;
	ACLflags	render;
	ACLflags	depth;
	ACLflags	stencil;
	ACLflags	scissor;
	ACLflags	stipple;
	ACLflags	alpha;
	ACLflags	blend;
	ACLflags	tex[ACLDIM_MAX_TMUNITS];
	ACLflags	color;
	ACLflags	fog;
	ACLflags	logop;
	ACLflags	misc;
	Tuint		tmunits;
	Tuint		maxtexsize;
	Tuint		maxtexsets;
};



/*************************************
 Hardware information
 *************************************/
struct acl_hardware {
	Tint hwidx;		/*+ The index of the board */
	const Tubyte *name;	/*+ short info about card */
	Tubyte bus;		/*+ PCI bus number */
	Tubyte slot;		/*+ PCI slot number */
	ACLFeatures hwf;	/*+ What is supported ? */
	Tbool running_fullscr;	/*+ Already running fullscreen mode ? */
	ACLHardware next; 	/*+ single linked list */
};



/*************************************
 Pointers to drawing data
 *************************************/
struct acl_vertices {
	Tfloat	(*win)[3];	/*+ Win coords: x,y,z */
	Tfloat	(*clip)[4];	/*+ Clip coords: x,y,z,w */
	Tubyte	(*color)[4];	/*+ Color: r,g,b,a */
	Tuint	*index;		/*+ Color: index */
	Tfloat	(*tex)[4];	/*+ Texture coords: s,t,r,q */
};



/*************************************
 Visual structure for initialization
 *************************************/
struct acl_visual {
	Tbool	RGBAflag;	/*+ If RGBA mode: TRUE, otherwise CI */
	Tbool	DBflag;		/*+ If double buffered: TRUE */
	Tbool	StereoFlag;	/*+ If stereo buffered: TRUE */
	Tbyte	RedBits;	/*+ Bits per color component (R) */
	Tbyte	GreenBits;	/*+ Bits per color component (G) */
	Tbyte	BlueBits;	/*+ Bits per color component (B) */
	Tbyte	AlphaBits;	/*+ Bits per color component (A) */
	Tbyte	IndexBits;	/*+ Bits per pixel if in CI mode */
	Tbyte	AccumBits;	/*+ Bits per color channel, or 0 */
	Tbyte	DepthBits;	/*+ Bits in Z buffer, or 0 */
	Tbyte	StencilBits;	/*+ Bits in stencil buffer, or 0 */
	Tbool	FrontAlphaFlag;	/*+ Software alpha front */
	Tbool	BackAlphaFlag;	/*+ Software alpha back	*/
};



/*************************************
 Texture image
 *************************************/
struct acl_tex_image {
	ACLglenum Format;	/*+ 1 ubyte formats: GL_ALPHA, GL_LUMINANCE,
				 *	GL_INTENSITY, GL_COLOR_INDEX
				 *  2 ubyte formats: GL_LUMINANCE_ALPHA
				 *  3 ubyte formats: GL_RGB
				 *  4 ubyte formats: GL_RGBA
				 */
	ACLglenum IntFormat;	/*+ Internal format as given by the user in the
				 *  glTexImage()
				 */
	Tuint	Border;		/*+ 0 or 1 */
	Tuint	Width;		/*+ = 2^WidthLog2 + 2*Border */
	Tuint	Height;		/*+ = 2^HeightLog2 + 2*Border */
	Tuint	Depth;		/*+ = 2^DepthLog2 + 2*Border */
	Tuint	Width2;		/*+ = Width - 2*Border */
	Tuint	Height2;	/*+ = Height - 2*Border */	
	Tuint	Depth2;		/*+ = Depth - 2*Border */
	Tuint	WidthLog2;	/*+ = log2(Width2) */
	Tuint	HeightLog2;	/*+ = log2(Height2) */
	Tuint	DepthLog2;	/*+ = log2(Depth2) */
	Tuint	MaxLog2;	/*+ = MAX(WidthLog2, HeightLog2) */
	Tubyte	*Data;		/*+ Image data as unsigned bytes */
	void	*DriverData;	/*+ Arbitrary device driver data */
};



/*************************************
 Setup of rendering screen
 *************************************/
struct acl_screen {
	Tuint	VisibleWidth;	/*+ visible screen area */
	Tuint	VisibleHeight;
	Tuint	VirtualWidth;	/*+ virtual screen area */
	Tuint	VirtualHeight;
	Tuint	WindowWidth;	/*+ client window area */
	Tuint	WindowHeight;
	Tubyte	ScreenBpp;	/*+ color depth of visual */
};



/*************************************
 The flags for renderer setup
 *************************************/
struct acl_render {
/*
 * Rendering flags for primitives. Set the appropriate flags if you want that
 * primitive gets rendered with additional option. Primitives are:
 *	ACLPRIM_POINTS, ACLPRIM_LINE, ACLPRIM_TRIANGLE, ...etc.
 */
	ACLflags	antialias;
	ACLflags	stipple;
/*
 * How to set particular primitive pointers ? Through these flags, you can
 * specify rendering functionality for every, currently used primitive:
 *	ACLRENDER_FLAT, ACLRENDER_SMOOTH, ACLRENDER_TWOSIDE,...etc.
 */
	ACLflags	points;	
	ACLflags	line;
	ACLflags	triangle;
	ACLflags	quad;
	ACLflags	rect;
	ACLflags	vblines;
	ACLflags	vbtriangles;
	ACLflags	vbquads;
	ACLflags	vbrects;
};



/*************************************
 Low-level accelerator context
 *************************************/
struct acl_context {
    Tint		index;		/*+ index of the acl render context */
    Tint		hwidx;		/*+ index of the associated hardware */
    struct acl_render	render_cfg; 	/*+ Rendering config */
    void		*CardCTL;	/*+ main control device IO (0,1,2...) */
    void		*CardCTL0;	/*+ device 0 mmaped IO */
    void		*CardCTL1;	/*+ device 1 mmaped IO */
    void		*CardCTL2;	/*+ device 2 mmaped IO (who knows...)*/
    void		*CardFB;	/*+ mmapped FB area */
    void		*CardLB;	/*+ mmapped LB area */
    void		*CardTEX;	/*+ mmapped TEX area */
    void		*HostTEX;	/*+ mmapped TEX area in host */
    void		*RegDMAAddress;	/*+ mmapped DMA start addr. register */
    void		*RegDMACount;	/*+ mmapped DMA count register */
    ACLflags		acl_cfg; 	/*+ client's ACL config */
    ACLModules		Mod;		/*+ modules of the driver context */
    ACLVisual		Vis;		/*+ visual of the driver context */
    ACLScreen		Scr;		/*+ The setup of X screen */
};



/*+****************************************************************************
 ******************************************************************************
 
 DESCRIPTION: RGBA module for pixels/span rendering
 ------------------------------------------------------------------------------
 Mandatory if there is no support for Color Index (CI). All functions within
 the module should be implemented.
 
 ******************************************************************************
 ******************************************************************************/
struct acl_color_rgba_mod {

/*+----------------------------------------------------------------------------



    NAME: ClearColor()
    ====================================
    Sets the color for clearing the color buffer.

    PARAMETERS:
    ====================================
    red, green, blue, alpha:
	- the RGBA color values in range 0-255 (max). The care about current
	  framebuffer format should be taken when implementing this function.
*/
    void (* ClearColor)
	( ACLContext aclctx,
	  Tubyte red, Tubyte green, Tubyte blue, Tubyte alpha );

/*+----------------------------------------------------------------------------



    NAME: Color()
    ====================================
    Sets the current color for flat shaded primitives.

    PARAMETERS:
    ====================================
    red, green, blue, alpha:
	- the RGBA color values in range 0-255. The care about current
	  framebuffer format should be taken.	
*/
    void (* Color)
	( ACLContext aclctx,
	  Tubyte red, Tubyte green, Tubyte blue, Tubyte alpha );

/*+----------------------------------------------------------------------------



    NAME: ColorMask()
    ====================================
    Sets the mask for writting into color buffer

    PARAMETERS:
    ====================================
    red, green, blue, alpha:
	- the RGBA color mask values 0 or 0xff.
*/
    void (* ColorMask)
	( ACLContext aclctx,
	  Tubyte red_mask, Tubyte green_mask, Tubyte blue_mask,
	  Tubyte alpha_mask );

/*+----------------------------------------------------------------------------



    NAME: FBWriteRGBAPixels()
    ====================================
    Draw pixels into frame buffer

    PARAMETERS:
    ====================================
    n:		- number of pixels
    x[], y[]:	- position of each pixel
    **rgba:	- color of each pixel. rgba dimension is either:
			Tubyte (*rgba)[3], if ACLFLAG_ALPHA _not_ set or
			Tubyte (*rgba)[4], if ACLFLAG_ALPHA set
    mask[]:	- 0 = don't draw pixel; NULL, 1 = draw pixel

    flags:	- ACLFLAG_ALPHA, draw alpha component
    		  ACLFLAG_SPAN, draws one horizontal span which starts at
		  x[0], y[0]
		  ACLFLAG_MONO, don't use rgba[][] color, use instead the
		  color set by function Color(). ACLFLAG_ALPHA doesn't have
		  effect then.
*/
    void (* FBWriteRGBAPixels)
	( const ACLContext aclctx,
	  Tuint n, const Tint x[], const Tint y[],
	  const void **rgba, const Tubyte mask[], ACLflags flags );

/*+----------------------------------------------------------------------------



    NAME: FBReadRGBAPixels()
    ====================================
    Read pixels from frame buffer

    PARAMETERS:
    ====================================
    n:		- number of pixels
    x[], y[]:	- position of each pixel
    mask[]:	- 0 = don't update **rgba; NULL, 1 = update **rgba

    flags:	- ACLFLAG_ALPHA, read alpha component
    		  ACLFLAG_SPAN, reads one horizontal span which starts at
		  x[0], y[0]
    RETURNS:
    ====================================
    **rgba:	- color of each pixel read. rgba dimension is either:
			Tubyte (*rgba)[3], if ACLFLAG_ALPHA _not_ set or
			Tubyte (*rgba)[4], if ACLFLAG_ALPHA set
*/
    void (* FBReadRGBAPixels)
	( const ACLContext aclctx,
	  Tuint n, const Tint x[], const Tint y[],
	  void **rgba, const Tubyte mask[], ACLflags flags );
};



/*+****************************************************************************
 ******************************************************************************
 
 DESCRIPTION: Color Index (CI) module for pixels/span rendering
 ------------------------------------------------------------------------------
 Mandatory if there is no support for RGBA. All functions within the module
 should be implemented.

 ******************************************************************************
 ******************************************************************************/
struct acl_color_ci_mod {

/*+----------------------------------------------------------------------------



    NAME: ClearIndex()
    ====================================
    Sets the index for clearing the color buffer.

    PARAMETERS:
    ====================================
    index:
	- the index of color value. The care about current framebuffer
	  format should be taken when implementing this function.
*/
    void (* ClearIndex)
	( ACLContext aclctx,
	  Tuint index );

/*+----------------------------------------------------------------------------



    NAME: Index()
    ====================================
    Sets the current index for flat shaded primitives.

    PARAMETERS:
    ====================================
    index:
	- the index of color value. The care about current framebuffer
	  format should be taken.
*/
    void (* Index)
	( ACLContext aclctx,
	  Tuint index );

/*+----------------------------------------------------------------------------



    NAME: IndexMask()
    ====================================
    Sets the current index mask.

    PARAMETERS:
    ====================================
    index:
	- the index mask of color value.
*/
    void (* IndexMask)
	( ACLContext aclctx,
	  Tuint index_mask );

/*+----------------------------------------------------------------------------



    NAME: FBWriteCIPixels()
    ====================================
    Draw pixels into frame buffer

    PARAMETERS:
    ====================================
    n:		- number of pixels
    x[], y[]:	- position of each pixel
    index[]:	- color index of each pixel
    mask[]:	- 0 = don't draw pixel; NULL, 1 = draw pixel

    flags:	- ACLFLAG_CI8, ACLFLAG_CI32, format of index
    		  ACLFLAG_SPAN, draws one horizontal span which starts at
		  x[0], y[0]
		  ACLFLAG_MONO, don't use index[] color, use instead the
		  color set by function Index(). ACLFLAG_CI8, ACLFLAG_CI32
		  doesn't have effect then.
*/
    void (* FBWriteCIPixels)
	( const ACLContext aclctx,
	  Tuint n, const Tint x[], const Tint y[],
	  const Tuint index[], const Tubyte mask[], Tuint flags );

/*+----------------------------------------------------------------------------



    NAME: FBReadCIPixels()
    ====================================
    Read pixels from frame buffer

    PARAMETERS:
    ====================================
    n:		- number of pixels
    x[], y[]:	- position of each pixel
    mask[]:	- 0 = don't update index[]; NULL, 1 = update index[]

    flags:	- ACLFLAG_CI8, ACLFLAG_CI32, format of index
    		  ACLFLAG_SPAN, reads one horizontal span which starts at
		  x[0], y[0]

    RETURNS:
    ====================================
    index[]:	- color index of each pixel read
*/
    void (* FBReadCIPixels)
	( const ACLContext aclctx,
	  Tuint n, const Tint x[], const Tint y[],
	  const Tuint index[], const Tubyte mask[], Tuint flags );
};



/*+****************************************************************************
 ******************************************************************************
 
 DESCRIPTION: Local buffer module
 ------------------------------------------------------------------------------
 Optional
 
 ******************************************************************************
 ******************************************************************************/
struct acl_lb_mod {

/*+----------------------------------------------------------------------------



    NAME: LBWritePixels()
    ===============================
    Write values into local buffer

    PARAMETERS:
    ====================================
    n:		- number of values
    x[], y[]:	- coordinate of each value, corresponding to FB
    lvalue[]:	- lower 32 bits of value to be written into LB
    hvalue[]:	- higher 32 bits (if any) of value to be written into LB
    mask[]:	- 0 = don't write value[]; NULL, 1 = write value[]

    flags:	- ACLFLAG_DEPTH, draw depth buffer values
    		  ACLFLAG_SPAN, writes one horizontal span which starts at
		  x[0], y[0]
*/
    void (* LBWritePixels)
	( const ACLContext aclctx,
	  Tuint n, const Tint x[], const Tint y[], const Tuint lvalue[],
	  const Tuint hvalue[], const Tubyte mask[], Tubyte flags );

/*+----------------------------------------------------------------------------



    NAME: LBReadPixels()
    ===============================
    Read values from local buffer

    PARAMETERS:
    ====================================
    n:		- number of values
    x[], y[]:	- coordinate of each value to read, corresponding to FB
    mask[]:	- 0 = don't update value[]; NULL, 1 = update value[]

    flags:	- ACLFLAG_DEPTH, read depth buffer values
    		  ACLFLAG_SPAN, reads one horizontal span which starts at
		  x[0], y[0]

    RETURNS:
    ====================================
    lvalue[]:	- array of values from lower 32 bits being read
    hvalue[]:	- array of values from higher 32 bits (if any) being read
*/
    void (* LBReadPixels)
	( const ACLContext aclctx,
	  Tuint n, const Tint x[], const Tint y[], Tuint lvalue[],
	  Tuint hvalue[], const Tubyte mask[], Tubyte flags );

/*+----------------------------------------------------------------------------



    NAME: EnableDepth
    ====================================
    Enable or disable the depth test

    PARAMETERS:
    ====================================
    state:	- depth test enable 1 or disable 0
*/
    void (* EnableDepth)
	( ACLContext aclctx,
	  Tbool state );

/*+----------------------------------------------------------------------------



    NAME: ClearDepth
    ====================================
    Set the current clear depth value

    PARAMETERS:
    ====================================
    depth:	- normalized depth value (0.0 - 1.0)
*/
    void (* ClearDepth)
	( ACLContext aclctx,
	  Tfloat depth );

/*+----------------------------------------------------------------------------



    NAME: DepthFunc
    ====================================
    Set the current depth function

    PARAMETERS:
    ====================================
    func:	- ACLGL_NEVER, ACLGL_LESS, ACLGL_EQUAL, ACLGL_LEQUAL,
		  ACLGL_GREATER, ACLGL_NOTEQUAL, ACLGL_GEQUAL, ACLGL_ALWAYS
*/
    void (* DepthFunc)
	( ACLContext aclctx,
	  ACLglenum func );

/*+----------------------------------------------------------------------------



    NAME: DepthMask
    ====================================
    Enable or disable writting to the depth buffer 

    PARAMETERS:
    ====================================
    mask:	- enable (ACLGL_TRUE) or disable (ACLGL_FALSE) writting
*/
    void (* DepthMask)
	( ACLContext aclctx,
	  ACLglenum mask );

/*+----------------------------------------------------------------------------



    NAME: DepthRange
    ====================================
    Specify the mapping of z values

    PARAMETERS:
    ====================================
    near, far:	- range of depth values. 0.0 <= value <= 1.0
*/
    void (* DepthRange)
	( ACLContext aclctx,
	  Tfloat near, Tfloat far );

/*+----------------------------------------------------------------------------



    NAME: EnableStencil
    ====================================
    Enables or disables stencil test

    PARAMETERS:
    ====================================a
    enable:	- enable (TRUE) or disable (FALSE) stencil test
*/
    void (* EnableStencil)
	( ACLContext aclctx,
	  Tbool enable );

/*+----------------------------------------------------------------------------



    NAME: ClearStencil
    ====================================
    Cleares stencil buffer with specified value

    PARAMETERS:
    ====================================
    stencil:	- the clear stencil value
*/
    void (* ClearStencil)
	( ACLContext aclctx,
	  Tint stencil );

/*+----------------------------------------------------------------------------



    NAME: StencilFunc
    ====================================
    Sets the comparison function which is used to compare values in the
    stencil buffer with the reference value, applying the comparison to the
    buffer value which is result of an 'AND' operation between low order
    bitplanes in stencil buffer with the corresponding low-order bits of the
    mask. The reference value is also ANDed with the mask value before
    comparison. If, for example, the choosen function is ACLGL_GREATER, then
    the fragment passes if the reference value is greater than the value in
    the stencil buffer. 

    PARAMETERS:
    ====================================
    func:	- ACLGL_NEVER, ACLGL_LESS, ACLGL_EQUAL, ACLGL_LEQUAL,
		  ACLGL_GREATER, ACLGL_NOTEQUAL, ACLGL_GEQUAL, ACLGL_ALWAYS
    ref:	- reference value
    mask:	- mask for AND operation
*/
    void (* StencilFunc)
	( ACLContext aclctx,
	  ACLglenum func, Tint ref, Tuint mask );

/*+----------------------------------------------------------------------------



    NAME: StencilOp
    ====================================
    If the fragment passes or fails the stencil test, the operation which
    modify data in stencil buffer can be set with this function.

    PARAMETERS:
    ====================================
    - ACLGL_KEEP, ACLGL_ZERO, ACLGL_REPLACE, ACLGL_INCR, ACLGL_DECR,
      ACLGL_INVERT

    fail:	- if fragment fails the stencil test
    zfail:	- stencil pass, z failed
    zpass:	- stencil pass, z pass or none at all
*/
    void (* StencilOp)
	( ACLContext aclctx,
	  ACLglenum fail, ACLglenum zfail, ACLglenum zpass );

};



/*+****************************************************************************
 ******************************************************************************
 
 DESCRIPTION: Accelerated primitives module
 ------------------------------------------------------------------------------
 Optional

 ******************************************************************************
 ******************************************************************************/
struct acl_accel_mod {

/*+----------------------------------------------------------------------------



    NAME: UpdateFuncPointers
    ====================================
    Update/set the function pointers in the module. Implementation of this
    function is not optional, if implementing the module.

    PARAMETERS:
    ====================================
    fmask:	- the function pointers which should be updated (ACLPRIM_*)
    fflags:	- the ACLRENDER_* flags set for every primitive 

    RETURNS:
    ====================================
    the mask with bits set for functions which should be, but were NOT updated
*/
    ACLflags (* UpdateFuncPointers)
	( ACLContext aclctx,
	  ACLflags fmask, ACLRender fflags );

/*+----------------------------------------------------------------------------



    NAME: EnableDither
    ====================================
    Enables or disables dither mode.

    PARAMETERS:
    ====================================
    enable:	- 0 or 1
*/
    void (* EnableDither)
	( ACLContext aclctx, 
	  Tbool enable);

/*+----------------------------------------------------------------------------



    NAME: DitherMode
    ====================================
    Sets dither mode.

    PARAMETERS:
    ====================================
    mode:	- ???
*/
    void (* DitherMode)
	( ACLContext aclctx, 
	  Tuint mode);

/*+----------------------------------------------------------------------------



    NAME: EnableLogicOp
    ====================================
    Enables or disables logic operations for fragments

    PARAMETERS:
    ====================================
    enable:	- 0 or 1
*/
    void (* EnableLogicOp)
	( ACLContext aclctx, 
          Tbool enable);

/*+----------------------------------------------------------------------------



    NAME: LogicOp
    ====================================
    Sets logic operation for fragments

    PARAMETERS:
    ====================================
    logop:	- ACLGL_CLEAR, ACLGL_AND, ACLGL_AND_REVERSE, ACLGL_COPY,
		  ACLGL_AND_INVERTED, ACLGL_NOOP, ACLGL_XOR, ACLGL_OR,
		  ACLGL_NOR, ACLGL_EQUIV, ACLGL_INVERT, ACLGL_OR_REVERSE,
		  ACLGL_COPY_INVERTED, ACLGL_OR_INVERTED, ACLGL_NAND,
		  ACLGL_SET
*/
    void (* LogicOp)
	( ACLContext aclctx, 
          Tint logop);

/*+----------------------------------------------------------------------------



    NAME: EnableAlpha
    ====================================
    Enables or disables alpha test

    PARAMETERS:
    ====================================
    enable:	- 0 or 1
*/
    void (* EnableAlpha)
	( ACLContext aclctx,
	  Tbool enable);

/*+----------------------------------------------------------------------------



    NAME: AlphaFunc
    ====================================
    Sets parameters for alpha test

    PARAMETERS:
    ====================================
    func:	- ACLGL_NEVER, ACLGL_LESS, ACLGL_EQUAL, ACLGL_LEQUAL,
		  ACLGL_GREATER, ACLGL_NOTEQUAL, ACLGL_GEQUAL, ACLGL_ALWAYS
    ref:	- 8 bit alpha value for comparison, if alpha buffer not used 
*/
    void (* AlphaFunc)
	( ACLContext aclctx,
	  ACLglenum func, Tubyte ref);

/*+----------------------------------------------------------------------------



    NAME: EnableBlend
    ====================================
    Enables or disables blending operation

    PARAMETERS:
    ====================================
    enable:	- 0 or 1
*/
    void (* EnableBlend)
	( ACLContext aclctx, 
	  Tbool enable);

/*+----------------------------------------------------------------------------



    NAME: BlendFunc
    ====================================
    Sets source and destination factor for blending operation

    PARAMETERS:
    ====================================
    src:	- source factor for b. operation
    dst:	- destination factor for b. operation
*/
    void (* BlendFunc)
	( ACLContext aclctx, 
	  ACLglenum src, ACLglenum dst);

/*+----------------------------------------------------------------------------



    NAME: EnableFog
    ====================================
    Enables or disables fog operation

    PARAMETERS:
    ====================================
    enable:	- 0 or 1
*/
    void (* EnableFog)
	( ACLContext aclctx,
	  Tbool enable);

/*+----------------------------------------------------------------------------



    NAME: FogColor
    ====================================
    Sets fogging color

    PARAMETERS:
    ====================================
    r,g,b,a:	- 8 bit color values for fog
*/
    void (* FogColor)
	( ACLContext aclctx,
	  Tubyte r, Tubyte g, Tubyte b, Tubyte a );

/*+----------------------------------------------------------------------------



    NAME: FogMode
    ====================================
    Sets fogging operation mode.

    PARAMETERS:
    ====================================
    mode:	- fog over depth function, can be linear, exponential...
    density:	- describes how fast fog gets opaque
*/
    void (* FogMode)
	( ACLContext aclctx,
	  Tint mode, Tfloat density);

/*+----------------------------------------------------------------------------



    NAME: Points
    ====================================
    Draws the set of points

    PARAMETERS:
    ====================================
    vb:		 - record of pointers to vertices
    first, last: - range of vertex indices for single points
*/
    void (* Points)
	( ACLContext aclctx,
	  ACLVertices vb, Tuint first, Tuint last );

/*+----------------------------------------------------------------------------



    NAME: Line
    ====================================
    Draws a line

    PARAMETERS:
    ====================================
    vb:		- record of pointers to vertices
    v0-1:	- vertex indices
    pv:		- index of flat color for flat shading
*/
    void (* Line)
	( ACLContext aclctx,
	  ACLVertices vb, Tuint v0, Tuint v1, Tuint pv );
  
/*+----------------------------------------------------------------------------



    NAME: Triangle
    ====================================
    Draws a triangle

    PARAMETERS:
    ====================================
    vb:		- record of pointers to vertices
    v0-2:	- vertex indices
    pv:		- index of flat color for flat shading
*/
    void (* Triangle)
	( ACLContext aclctx,
	  ACLVertices vb, Tuint v0, Tuint v1, Tuint v2, Tuint pv );

/*+----------------------------------------------------------------------------



    NAME: Quad
    ====================================
    Draws a quadrilateral

    PARAMETERS:
    ====================================
    vb:		- record of pointers to vertices
    v0-3:	- vertex indices
    pv:		- index of flat color for flat shading
*/
    void (* Quad)
	( ACLContext aclctx,
	  ACLVertices vb, Tuint v0, Tuint v1, Tuint v2, Tuint v3, Tuint pv );

/*+----------------------------------------------------------------------------



    NAME: Rect
    ====================================
    Draws a screen-aligned rectangle

    PARAMETERS:
    ====================================
    vb:		- record of pointers to vertices
    v0-3:	- vertex indices
    pv:		- index of flat color for flat shading
*/
    void (* Rect)
	( ACLContext aclctx,
	  ACLVertices vb, Tuint v0, Tuint v1, Tuint v2, Tuint v3, Tuint pv );

/*+----------------------------------------------------------------------------



    NAME: LinesVB
    ====================================
    Draws a set of lines from vertex buffer

    PARAMETERS:
    ====================================
    vb:		- record of pointers to vertices
*/
    void (* LinesVB)
	( ACLContext aclctx,
	  ACLVertices vb );
  
/*+----------------------------------------------------------------------------



    NAME: TrianglesVB
    ====================================
    Draws a set of triangles from vertex buffer

    PARAMETERS:
    ====================================
    vb:		- record of pointers to vertices
*/
    void (* TrianglesVB)
	( ACLContext aclctx,
	  ACLVertices vb );
  
/*+----------------------------------------------------------------------------



    NAME: QuadsVB
    ====================================
    Draws a set of quads from vertex buffer

    PARAMETERS:
    ====================================
    vb:		- record of pointers to vertices
*/
    void (* QuadsVB)
	( ACLContext aclctx,
	  ACLVertices vb );
  
/*+----------------------------------------------------------------------------



    NAME: RectsVB
    ====================================
    Draws a set of rectangles from vertex buffer

    PARAMETERS:
    ====================================
    vb:		- record of pointers to vertices
*/
    void (* RectsVB)
	( ACLContext aclctx,
	  ACLVertices vb );
  
};



/*+****************************************************************************
 ******************************************************************************
 
 DESCRIPTION: Textures module
 ------------------------------------------------------------------------------
 Optional

 In current implementation:
 	- mipmapping is silently ignored
	- GL_TEXTURE_1D and GL_TEXTURE_3D are not handled here

 ******************************************************************************
 ******************************************************************************/
struct acl_tex_mod {

/*+----------------------------------------------------------------------------



    NAME: EnableTex
    ====================================
    Enable or disable texture mapping

    PARAMETERS:
    ====================================
    state:	- 1 or 0
*/
    void (* EnableTex)
	( ACLContext aclctx,
	  Tbool state );

/*+----------------------------------------------------------------------------



    NAME: GetMinTexAddress
    ====================================
    Query the minimum address in the texture memory of the graphics accelerator
    usable by the Mesa ACL texture manager.

    RETURNS:
    ====================================
    Tulong:	- start/base adress of texture memory
*/
    Tulong (* GetMinTexAddress)
	( ACLContext aclctx );

/*+----------------------------------------------------------------------------



    NAME: GetMaxTexAddress
    ====================================
    Return the maximum address in the texture memory of the graphics
    accelerator usable by the Mesa ACL texture manager.

    RETURNS:
    ====================================
    Tulong:	- end Adress of texture memory.
*/
    Tulong (* GetMaxTexAddress)
	( ACLContext aclctx );

/*+----------------------------------------------------------------------------



    NAME: RequiredTexMem
    ====================================
    Return the amount of texture memory required in order to store the texture
    image described by "tex" in the texture memory of the graphics accelerator.
    If the texture image has to be aligned to a specific amount of bytes (i.e:
    16),the returned number must be a multiple of the specific amount of bytes.

    PARAMETERS:
    ====================================
    tex:	- a pointer to an ACLTexImage structure describing the image

    RETURNS
    ====================================
*/
    Tulong (* RequiredTexMem)
	( ACLContext aclctx,
	  ACLTexImage tex );

/*+----------------------------------------------------------------------------



    NAME: TranslateImage
    ====================================
    Translate from the internal Mesa core representation of a texture map to a
    format suitable for the hardware.  This function can be also a NOP if the
    Mesa core representation can be used directly. The translated image should
    be stored in a CPU memory area pointed by the translatedImage field.
    Returns GL_FALSE if failed (i.e. not supported by the hardware), GL_TRUE
    otherwise.

    PARAMETERS:
    ====================================
	tex = a pointer to an ACLTexImage structure describing the image
	format = ???
	level = mip map level
	name = ???
	dd = pointer to image in main memory

    RETURNS:
    ====================================
	Tbool = GL_TRUE on success, GL_FALSE on error.
*/
    Tbool (* TranslateImage)
	( ACLContext aclctx,
	  ACLTexImage tex, Tint format, Tint level, Tuint name, void *dd );

/*+----------------------------------------------------------------------------



    NAME: SubImage
    ====================================
    cuts of a pice of a texture and stores it elsewhere

    PARAMETERS:
    ====================================
	target = ???
	dd = pointer to target location in main memory
	level = mip map level
	xoffset, yoffset = origin of cutoff into the texture plain
	width, height = dimensions of cutoff
	internal format =
	image = description of source texture 
*/
    void (* SubImage)
       ( ACLContext aclctx, 
	 ACLglenum target, void *dd, Tint level, Tint xoffset,
	 Tint yoffset, Tint width, Tint height, Tint internalFormat,
	 const ACLTexImage image );

/*+----------------------------------------------------------------------------



    NAME: DownloadToTexMem
    ====================================
    Download to the texture memory (lb) the maps described by "tex". The MLX
    Library should have only to copy the already translated image stored in
    ACLTexImage->DriverData into the texture memory starting from address
    "memAddress"

    PARAMETERS:
    ====================================
	tex = a pointer to a ACLTexImage describing an image
	level = mip map level
	dd = texture memory address where to upload the image
*/
    void (* DownloadToTexMem)
	( ACLContext aclctx,
	  Tint level, void * dd);

/*+----------------------------------------------------------------------------



    NAME: Bind
    ====================================
    Select the texture memory address to use as source for texture mapping

    PARAMETERS:
    ====================================
	name = ???
	dd = texture memory address where the image was previously uploaded
*/
    void (* Bind)
	( ACLContext aclctx, 
	  Tuint name, void *dd );

/*+----------------------------------------------------------------------------



    NAME: Env
    ====================================
    Select texture function to use for texture mapping. If the function isn't
    supported by the hardware, it must return GL_FALSE, otherwise GL_TRUE

    PARAMETERS:
    ====================================
	pname = GL_TEXTURE_ENV_MODE or GL_TEXTURE_ENV_COLOR
	param = GL_MODULATE, GL_BLEND, GL_DECAL,
			or GL_REPLACE if pname=GL_TEXTURE_ENV_MODE
			else it is the costant color used for the GL_BLEND mode
*/
    Tbool (* Env)
	( ACLContext aclctx,
	  ACLglenum pname, const Tfloat *param );

/*+----------------------------------------------------------------------------



    NAME: Param
    ====================================
    Handle glTexParameter() settings

    PARAMETERS:
    ====================================
	target = ???
	pname = GL_TEXTURE_MIN_FILTER, GL_TEXTURE_MAG_FILTER,
		GL_TEXTURE_WRAP_S or GL_TEXTURE_WRAP_T
	param = dependant on pname:
		if GL_TEXTURE_MIN_FILTER or GL_TEXTURE_MAG_FILTER
			then GL_NEAREST or GL_LINEAR
		if GL_TEXTURE_WRAP_S or GL_TEXTURE_WRAP_T
			then GL_CLAMP or GL_REPEAT
	dd = pointer to ???
*/
    void (* Param)
	( ACLContext aclctx,
	  ACLglenum target, ACLglenum pname, 
	  const Tfloat *param, void *dd);

/*+----------------------------------------------------------------------------



    NAME: TexDel
    ====================================
    Free of alloced memory

    PARAMETERS:
    ====================================
	dd = pointer to driver memory

    RETURNS:
    ====================================
	void* = ???
*/
    void* (* TexDel)
	( ACLContext aclctx,
	  void * dd );

/*+----------------------------------------------------------------------------



    NAME: AllocTexObjData
    ====================================
    Free of alloced memory

    PARAMETERS:
    ====================================
	dd = pointer to driver memory ???

    RETURNS:
    ====================================
	void* = ???
*/
   void* (* AllocTexObjData)
   	( ACLContext aclctx );

/*+----------------------------------------------------------------------------

    

    NAME: Palette
    ====================================
    applys a palette to a texture

    PARAMETERS:
    ====================================
	size = ???
	data = ???
	dd = pointer to driver memory
*/
    void (* Palette)
        ( ACLContext aclctx, 
	  Tuint size, Tuint *data, void *dd);

/*+----------------------------------------------------------------------------



    NAME: SwitchToGlobalPalette
    ====================================
    selects global palette

    PARAMETERS:
    ====================================
*/
   void (*SwitchToGlobalPalette)
         ( ACLContext aclctx );

/*+----------------------------------------------------------------------------



    NAME: DownloadGlobalPalette
    ====================================
    queries global palette

    PARAMETERS:
    ====================================
	size = ???
	data = pointer to ???
	dd   = pointer to ???
*/
   void (*DownloadGlobalPalette)
         ( ACLContext aclctx, 
	   Tuint size, Tuint *data, void *dd);

/*+----------------------------------------------------------------------------
TODO:
	TranslateSubTexImage
	UploadToSubTexMem
	etc.
*/
};



/*+****************************************************************************
 ******************************************************************************
 
 DESCRIPTION: Main module structure
 ------------------------------------------------------------------------------
 Mandatory, all functions must be implemented

 ******************************************************************************
 ******************************************************************************/
struct acl_modules {

/*+----------------------------------------------------------------------------
 


    NAME: StartRender
    ====================================
    to be called before any rendering, for renderer context setup
*/
    void (*StartRender)
	( ACLContext );

/*+----------------------------------------------------------------------------



    NAME: FinishRender
    ====================================
    after rendering finished, this should put screen into the state before
    rendering started
*/
    void (*FinishRender)
	( ACLContext );

/*+----------------------------------------------------------------------------



    NAME: SwapBuffer
    ====================================
    swaps rendering FRONT and BACK buffer
*/
    void (*SwapBuffer)
	( ACLContext );

/*+----------------------------------------------------------------------------



    NAME: DrawBuffer
    ====================================
    sets the current rendering buffer

    PARAMETERS:
    ====================================
    buffer:	- Where to render ? (ACLGL_FRONT, ACLGL_BACK, ...)

    RETURNS:
    ====================================
    Tuint	- ACLGL_TRUE on success, or ACLGL_FALSE if failed
*/
    Tuint (*DrawBuffer)
	( ACLContext aclctx,
	  ACLglenum buffer );

/*+----------------------------------------------------------------------------



    NAME: ClearBuffer()
    ====================================
    Clear the color/depth/stencil/accum buffer(s)
    
    PARAMETERS:
    ====================================
    mask:	- indicates which buffers need to be cleared
    all:	- if true, then clear the whole buffer, else clear the
		  region defined by x, y, width and height.
    x,y:	- origin of buffer
    width,
    height:	- size of buffer
    
    RETURNS:
    ====================================
    - a bitmask indicating which buffers weren't cleared by the driver
      function
*/
    Tuint (* ClearBuffer)
	( ACLContext aclctx,
	  Tuint mask, Tbool all,
	  Tint x, Tint y, Tint width, Tint height );

/*+----------------------------------------------------------------------------
TODO:
    GetSurface();
    DriverString();
*/

/*+****************************************************************************
 BASIC ACL MODULES
 ------------------------------------------------------------------------------
 at least one module is required, either CI or RGBA.
 ******************************************************************************/
    ACLflags		modflags;	/*+ flags set for existing modules */

    ACLColorRGBAMod    	ColorRGBA;
    ACLColorCIMod      	ColorCI;

/*+****************************************************************************
 ADVANCED ACL MODULES
 ------------------------------------------------------------------------------
 all modules are optional and pointers could be set NULL, if module missing
 ******************************************************************************/
    ACLAccelMod        	Accel;
    ACLLBMod		LB;
    ACLTexMod		Tex;
};



/*+****************************************************************************
 ******************************************************************************

                    ACL Interface Initialization Functions

 ******************************************************************************
 ******************************************************************************/

/*+----------------------------------------------------------------------------



    NAME: AclEnter
    ====================================
    The implementation of this function should prepare the low-level drawing
    library for usage. It checks if kernel module is loaded or even loads
    it, opens devices and loads the hardware information into user space, to
    make access to it possible. In multithreaded application, it should only
    increment the usage count of the library for every new thread started.

    RETURNS:
    ====================================
    0 on success or 1 if failed
*/
Tbool	AclEnter( void );

/*+----------------------------------------------------------------------------



    NAME: AclListHardware
    ====================================
    Scans for available hardware and drivers and returns a info-list, which
    user can use to gather information about rendering hardware. It also
    checks if the index, given as parameter, is valid.

    PARAMETER:
    ====================================
    hwidx:	- hardware index to check or -1 just for list

    RETURNS:
    ====================================
    ACLHardware: - if hwidx was -1, it returns only pointer to the
		   first member of hardware list. Otherwise, it
		   returns pointer to the record of wished hardware, or
		   NULL if such hwidx doesn't exist.
*/
ACLHardware AclListHardware( Tint hwidx );

/*+----------------------------------------------------------------------------



    NAME: AclCreateContext
    ====================================
    Allocates new ACLModules and setups new ACLContext with appropriate
    pointers. The ACLModules gets bound with the specified hardware-driver
    context. The functions from particular driver implementation are connected
    with the ACLContext.

    PARAMETERS:
    ====================================
    hwidx:	- Index of wished hardware, as specified in information
		  returned by AclListHardware()
    screen:	- Pointer to screen data of context
    visual:	- Pointer to visual data of context
    config:	- setup flags ACLOPT_*

    RETURNS:
    ====================================
    ACLContext:	- pointer to the created ACLContext.
*/
ACLContext AclCreateContext( Tint hwidx,
	ACLScreen screen, ACLVisual visual, ACLflags config);

/*+----------------------------------------------------------------------------



    NAME: AclDestroyContext
    ====================================
    Frees resources used by specified ACLContext

    PARAMETER:
    ====================================
    aclctx:	- previously used ACL context
*/
void	AclDestroyContext( ACLContext aclctx );

/*+----------------------------------------------------------------------------



    NAME: AclExit
    ====================================
    Similar like AclEnter, the AclExit implementation should free low-level
    library resources of current process and decrement the usage count of
    library. Therefore, it's important that every new process or thread
    started with AclEnter() also calls AclExit() on finish.
*/
void	AclExit( void );



#endif

/*+****************************************************************************
 ******************************************************************************
                                ACL API END
 ******************************************************************************
 ******************************************************************************/
