
/*****************************************************************************
 *                                    M L X                                  *
 *                 Rendering Library for Accelerated 3d Hardware             *
 *                        (C) SuSE GmbH 1997, 1998, 1999                     *
 *****************************************************************************/
/* author: simon pogarcic, sim@suse.de */

/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>
#include <linux/pci.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/mman.h>

#if 1
#define LOG_DEBUG
#endif

#define LOG_DBGINFO "<agent>"
#include "log_debug.h"

#include "mlx.h"
#include "agent.h"



/*****************************************************************************
 * ALL supported driver info records must be listed here; this is connection
 * point with every driver implemented in MLX. To add a new driver, edit
 * the file agent.h and add new macros here and there.
 *****************************************************************************/
MLXDRIVER_PROTO_TXDELTA
MLXDRIVER_PROTO_MXDELTA
MLXDRIVER_PROTO_PMDELTA
MLXDRIVER_PROTO_PM2
MLXDRIVER_PROTO_PM3

MLXDriverRec	*mlx_drivers[] = {
	MLXDRIVER_TXDELTA
	MLXDRIVER_MXDELTA
	MLXDRIVER_PMDELTA
	MLXDRIVER_PM2
	MLXDRIVER_PM3
	NULL
};



/*****************************************************************************
 * TODO: put this in shared memory and organize data sharing between clients
 *****************************************************************************/
MLXCardCtx	*mlx_cards[MAX_CARDS_ALOWED];
Tint		mlx_aclctx_counter = 0;



ACLHardware	mlxacl_hardware = NULL;
Tbool		mlx_entered = 0;

/*****************************************************************************
                            MLX Internal Functions
 *****************************************************************************/

static void
MlxSetDriverIndices( void )
{
    Tint drvidx = 0;
    MLXDriverRec **dr = mlx_drivers;

    while(*dr) {
	(*dr)->drvidx = drvidx++;
	dr++;
    }
}



static MLXDriverRec *
MlxGetDriver( Tint card_type )
{
    MLXDriverRec **dr = mlx_drivers;

    while(*dr) {
	if((*dr)->card_type == card_type) return *dr;
	dr++;
    }
    return NULL;
}



static Tbool
MlxCheckConfig( ACLflags cfg )
{
    return 0;
}



/*****************************************************************************
                            ACL API Initialization
 *****************************************************************************/ 

Tbool
AclEnter( void )
{
    Tubyte cardnr;
    ACLHardware aclhw, *paclhw = &mlxacl_hardware;

LOG("AclEnter(): ######\n");

    if(mlx_entered) {
	MSG("MLX already initialized. Continuing...\n");
    	return 0;
    }

    if( gpcli_enter_app() )
	return 1;

    mlx_entered = 1;
    MlxSetDriverIndices();

    /*
     * Create ACLHardware list which supply basic informations about
     * supported hardware and drivers
     *************************************************************************/
    for(cardnr=0; cardnr<GpcliCardsFound; cardnr++) {
	GPMCardInfo_t *card = GpcliCardsInfo + cardnr;
	GPMCard_t *scard = GpcliSupportedCards + card->idx;
	MLXDriverRec *dr;
	MLXCardCtx *cctx = mlx_cards[cardnr];
LOG("AclEnter(): (card %d) ACLHardware (malloc)\n", cardnr);
	if(!(aclhw=*paclhw=(ACLHardware)malloc(sizeof(struct acl_hardware))))
	    AclExit();
	dr = MlxGetDriver(scard->card_type);
	aclhw->hwidx = cardnr;
	aclhw->name = scard->name;
	aclhw->bus = card->bus;
	aclhw->slot = card->slot;
	aclhw->hwf = &(dr->features);
	aclhw->next = NULL;
	paclhw = &(aclhw->next);

	if(cctx) {
	    aclhw->running_fullscr = CCTX(is_full_screen) ? 1 : 0;
	}

/* TODO: SHM
	this is only temporary; mlx_cards should be stored in shared memory
	and reset to NULL on shared memory init.
*/
	mlx_cards[cardnr] = NULL;
    }

    MSG("\n");
    MSG("DOOR TO MLX RESOURCES OPENED\n");

    return 0;
}



void
AclExit( void )
{
    if(!mlx_entered) {
	MSG("WARNING - AclExit(): Library not inited yet\n");
    	return;
    }

    gpcli_exit_app(0,0);

    mlx_entered = 0;
    MSG("\n");
    MSG("DOOR TO MLX RESOURCES CLOSED\n");
}



ACLHardware
AclListHardware( Tint hwidx )
{
    ACLHardware aclhw = mlxacl_hardware;

    if(!mlx_entered) {
	MSG("WARNING - AclListHardware(): Library not inited yet\n");
    	return NULL;
    }

    if(hwidx == -1)
	return mlxacl_hardware;

    while(aclhw) {
	if(aclhw->hwidx == hwidx) return aclhw;
	aclhw = aclhw->next;
    }
    return NULL;
}



ACLContext
AclCreateContext (Tint hwidx, ACLScreen scr, ACLVisual vis, ACLflags cfg )
{
    MLXCardCtx *cctx = mlx_cards[hwidx];
    MLXDriverRec *dr = NULL;

    ACLContext aclctx = NULL;
    ACLModules mod = NULL;

    ACLflags modflags;

    if(!mlx_entered) {
	ERR("###### AclEnter() not called yet. ######\n");
    	return NULL;
    }

    if(!vis || !scr) {
	ERR("###### ACLScreen, ACLVisual, where ??? ######\n");
	return NULL;
    }
/*
 * Alloc hardware/card context, if not already allocated by some other client
 *****************************************************************************/
/* TODO: SHM MUTEX */
    if(!cctx) {
LOG("AclCreateContext(): MLXCardCtx (malloc)\n");
	if(!(cctx = mlx_cards[hwidx] = (MLXCardCtx*) malloc
		(sizeof( MLXCardCtx )))) {
	    return NULL;
	}
	dr = MlxGetDriver( GpcliSupportedCards
		[ GpcliCardsInfo[hwidx].idx ].card_type);
	CCTX(drvidx) = dr->drvidx;
	CCTX(card_type) = dr->card_type;
	CCTX(nr_clients) = 0;
	CCTX(lock) = 0;
    }
    else {
	dr = mlx_drivers[cctx->drvidx];
    }

    if(!dr) {
	ERR("Internal MLX: no driver found!\n");
	return NULL;
    }

    if(MlxCheckConfig(cfg)) {
	ERR("Internal ACL DD: bad configuration! (%d)\n", cfg);
	return NULL;
    }

/*
 * Alloc memory for rendering context
 *****************************************************************************/
LOG("AclCreateContext(): ACLContext (malloc)\n");
    if(!(aclctx = (ACLContext) malloc(sizeof(struct acl_context))))
	return NULL;

/*
 * Alloc memory for this context modules
 *****************************************************************************/
LOG("AclCreateContext(): ACLModules (malloc)\n");
    if(!(mod = (ACLModules) malloc(sizeof(struct acl_modules)))) {
	free(aclctx);
	return NULL;
    }
    
    modflags = mod->modflags = dr->modflags;

    if(modflags & ACLMOD_RGBA) {
LOG("AclCreateContext(): ACLColorRGBAMod (malloc)\n");
	if(!(mod->ColorRGBA = (ACLColorRGBAMod) malloc
		(sizeof(struct acl_color_rgba_mod)))) {
	    free(aclctx);
	    free(mod);
	    return NULL;
	}
    }

    if(modflags & ACLMOD_CI) {
LOG("AclCreateContext(): ACLColorCIMod (malloc)\n");
	if(!(mod->ColorCI = (ACLColorCIMod) malloc
		(sizeof(struct acl_color_ci_mod)))) {
	    if(mod->ColorRGBA) free(mod->ColorRGBA);
	    free(aclctx);
	    free(mod);
	    return NULL;
	}
    }

    if(modflags & ACLMOD_ACCEL) {
LOG("AclCreateContext(): ACLAccelMod (malloc)\n");
	if(!(mod->Accel = (ACLAccelMod) malloc
		(sizeof(struct acl_accel_mod)))) {
	    if(mod->ColorRGBA) free(mod->ColorRGBA);
	    if(mod->ColorCI) free(mod->ColorCI);
	    free(aclctx);
	    free(mod);
	    return NULL;
	}
    }

    if(modflags & ACLMOD_LB) {
LOG("AclCreateContext(): ACLLBMod (malloc)\n");
	if(!(mod->LB = (ACLLBMod) malloc
		(sizeof(struct acl_lb_mod)))) {
	    if(mod->ColorRGBA) free(mod->ColorRGBA);
	    if(mod->ColorCI) free(mod->ColorCI);
	    if(mod->Accel) free(mod->Accel);
	    free(aclctx);
	    free(mod);
	    return NULL;
	}
    }

    if(modflags & ACLMOD_TEX) {
LOG("AclCreateContext(): ACLTexMod (malloc)\n");
	if(!(mod->Tex = (ACLTexMod) malloc
		(sizeof(struct acl_tex_mod)))) {
	    if(mod->ColorRGBA) free(mod->ColorRGBA);
	    if(mod->ColorCI) free(mod->ColorCI);
	    if(mod->Accel) free(mod->Accel);
	    if(mod->LB) free(mod->LB);
	    free(aclctx);
	    free(mod);
	    return NULL;
	}
    }

    aclctx->Vis = vis;
    aclctx->Scr = scr;
    aclctx->Mod = mod;

    aclctx->hwidx = hwidx;
    aclctx->acl_cfg = cfg;

/*============================================================================
 * at this place it should be ensured that only one single thread/process is
 * accessing the card data structure. If already accessed by someone else,
 * than wait.
 *===========================================================================*/

LOG("AclCreateContext(): Connecting with driver...\n");
    if(dr->init(aclctx)) {
	ERR("\n")
	ERR("################################\n");
	ERR("###### driver init failed ######\n");
	ERR("################################\n");
	ERR("\n")
	if(mod->ColorRGBA) free(mod->ColorRGBA);
	if(mod->ColorCI) free(mod->ColorCI);
	if(mod->Accel) free(mod->Accel);
	if(mod->LB) free(mod->LB);
	if(mod->Tex) free(mod->Tex);
	free(aclctx);
	free(mod);
	return NULL;
    }

/* everything was OK... now register the new acl context */
    CCTX(aclctx_index) = ++mlx_aclctx_counter;
    CCTX(nr_clients)++;
    CCTX(lock) = 1;

/*============================================================================
 * the card data structure is now written, and other clients should be now
 * alowed to refer it.
 * 'lock', if true, means that the modes arranged on the card must be used by
 * every client. However, for the case that window client is not using back or
 * local buffer, those buffers still could be set by other newcoming clients
 * who require them. One full-screen client can take the card for himself (if
 * some other client is not already full-screen), where other window clients go
 * sleeping.
 * if we set 'lock' false, the card context could be completely new set. This
 * could be useful by VT switching of screens. 
 *===========================================================================*/
 
    return aclctx;
}



void
AclDestroyContext( ACLContext aclctx )
{
    MLXCardCtx *cctx;
    ACLModules mod;

LOG("AclDestroyContext(): ######\n");

    if(!aclctx) return;

    cctx = mlx_cards[aclctx->hwidx];
    mod = aclctx->Mod;

    if(mod->ColorRGBA) free(mod->ColorRGBA);
    if(mod->ColorCI) free(mod->ColorCI);
    if(mod->Accel) free(mod->Accel);
    if(mod->LB) free(mod->LB);
    if(mod->Tex) free(mod->Tex);
    free(mod);

    free(aclctx->Vis);
    free(aclctx->Scr);
    free(aclctx);

/* TODO: SHM MUTEX */
    if( !CCTX(nr_clients) ) {
	ERR("Internal: CCTX(nr_clients) already == 0 !!!\n");
	return;
    }

    CCTX(nr_clients)--;
}



