
/*****************************************************************************
 *                                    M L X                                  *
 *                 Rendering Library for Accelerated 3d Hardware             *
 *                           (C) SuSE GmbH 1997, 1998                        *
 *****************************************************************************/
/* author: simon pogarcic, sim@suse.de */

/*****************************************************************************/
/*****************************************************************************/
/* Permedia2 OpenGL/Mesa driver                                              */
/* (c) 1997, 1998, 1999 SuSE GmbH                                            */
/* ------------------------------------------------------------------------- */
/* Authors:                                                                  */
/*	Simon Pogarcic, sim@suse.de                                          */
/*	Stefan Dirsch, sndirsch@suse.de                                      */
/*	Helmut Fahrion                                                       */
/*****************************************************************************/
/*****************************************************************************/

/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <sys/ioctl.h>
#include <linux/pci.h>

#if 1
  #define LOG_DEBUG
#endif

#define LOG_DBGINFO "<pm2>"
#include "log_debug.h"



#include "pm2.h"



GENPROTO_DRIVER_INIT_MAIN(PM2)



PM2HardwareContext	pm2_hwctx;
GLINTRenderingValues	pm2_values;

static Tubyte 		pm2_msg[] = "PM2";
static Tubyte		pm2_info[] = "Permedia 2 MLX Driver";

/******************************************************************************
 *                          Permedia 2 driver setup                           *
 ******************************************************************************/

MLXDriverRec PM2_driver = {
	-1,
	CARD_PERMEDIA2,
	pm2_msg, pm2_info,
	PM2Init,
	(ACLMOD_RGBA | ACLMOD_LB | ACLMOD_ACCEL),
    {
/* formatting of vertices */
	0,

/* renderer features */
	ACLHW_REN_VXDELTA |
	ACLHW_REN_MESH |
	ACLHW_REN_FAN |
	ACLHW_REN_BACKFACE_CULL |
	ACLHW_REN_SUBPIX_CORRECTION,

/* depth test */
	ACLHW_DEPTH,

/* stencil test */
	ACLHW_STENCIL,

/* scissor test */
	ACLHW_SCISSOR,

/* stippled primitives */
	0,

/* alpha buffer operations */
	0,

/* alpha blending operations */
	0,

/* tex operations */
    {
	0,
    },

/* color operations */
	0,

/* fog operations */
	0,

/* logic operations */
	0,

/* miscelaneous features */
	0,

/* number of independent texture units (for multitexturing) */
	1,

/* max alowed texture size in pixels */
	0,

/* max number of texture sets */
	0
    }
};



static Tint
PM2_InitVidMem(ACLContext aclctx)
{
    MLX_INIT_ACCESS_CARD;

    Tubyte bytes_pp, pciaddrdiv;
    Tuint screen_size, free_pix_offset;
    ACLScreen scr = aclctx->Scr;
    Tint free_vidmem_size = CCTX(VidMemSize);
    Tbool fullscr = (aclctx->acl_cfg & ACLOPT_RUNMODE_FULL) ? 1 : 0;
    Tbool do_config = (!CCTX(lock) || (CCTX(lock) && fullscr)) ? 1 : 0;

    /* this info should be read from aclctx visual info... */
    Tbool do_config_back = 1;
    Tbool do_config_depth = 1;

LOG("PM2_InitVidMem(): ######\n");

    if(fullscr) {
	if(CCTX(lock) && CCTX(is_full_screen)) {
	    ERR("###### Fullscreen already running! ######\n");
	    return 1;
	}
	CCTX(is_full_screen) = 1;
	sctx = CCTX(Screen) = &CCTX(FullScreen);
    }
    else {
	CCTX(is_full_screen) = 0;
	sctx = CCTX(Screen) = &CCTX(WinScreen);
    }

    if(do_config) {
	sctx->VisibleW = scr->VisibleWidth;
	sctx->VisibleH = scr->VisibleHeight;
	sctx->VirtualW = scr->VirtualWidth;
	sctx->VirtualH = scr->VirtualHeight;
	sctx->ScreenBpp = scr->ScreenBpp;

	/*
	 * A HACK: if you set screen_size == visible, all buffers could be
	 * seen, if you config card with virtual screen which covers whole
	 * video memory. The virtual width should be equal to visible width.
	 */
#if 1
	sctx->ScreenSize = sctx->VirtualW * sctx->VirtualH;
#else
	sctx->ScreenSize = sctx->VisibleW * sctx->VisibleH;
#endif

	/*
	 * bytes per pixel
	 */
	switch(sctx->ScreenBpp) {
	case 8:
	    sctx->BytesPP = 1;
	    break;
	case 15:
	case 16:
	    sctx->BytesPP = 2;
	    break;
	case 24:
	case 32:
	    sctx->BytesPP = 4;
	    break;
	default:
	    ERR("###### Unsupported ScreenBpp: %d ######\n", sctx->ScreenBpp);
	    return 1;
	}

	/*
	 * Partial Product values for screen width:
	 */
	sctx->ScreenPProd = GlintGetPProd(CCTX(card_type), sctx->VirtualW);
    }

    bytes_pp = sctx->BytesPP;
    pciaddrdiv = (Tubyte)(8/bytes_pp);
    screen_size = sctx->ScreenSize;

    free_pix_offset = 0;

/*
 * free_vidmem_size counter is always in bytes. We assume that front, back and
 * depth buffer all require the same size which screen covers. The rest of mem
 * is taken for textures.
 */

/*****************************************************************************
 Front Buffer:
 *****************************************************************************/
    if(do_config) {
	if((free_vidmem_size -= (screen_size * bytes_pp)) < 0) {
	    ERR("###### no memory for framebuffer ######\n");
	    return 1;
	}
	/* pixels */
	sctx->FrontBuffOffset = free_pix_offset;
	/* quad words (64 bit) addressing */
	sctx->FrontBuffAddrOffset = free_pix_offset / pciaddrdiv;
    }
    free_pix_offset += screen_size;

/*****************************************************************************
 Back Buffer:
 *****************************************************************************/
    if(do_config || (!CCTX(win_back_set) && do_config_back)) {
	if((free_vidmem_size -= (screen_size * bytes_pp)) < 0) {
	    ERR("###### no memory for backbuffer ######\n");
	    return 1;
	}
	/* pixels */
	sctx->BackBuffOffset = free_pix_offset;
	sctx->FrontBackOffset = sctx->BackBuffOffset - sctx->FrontBuffOffset;
	/* quad words (64 bit) addressing */
	sctx->BackBuffAddrOffset = free_pix_offset / pciaddrdiv;
	CCTX(win_back_set) = 1;
    }
    if(CCTX(win_back_set))
	free_pix_offset += screen_size;

/*****************************************************************************
 Depth Buffer: depth buff always 16 bit (depth addr = 2 * pix address)
 *****************************************************************************/
    if(do_config || (!CCTX(win_depth_set) && do_config_depth)) {
	if((free_vidmem_size -= (screen_size * 2)) < 0) {
	    ERR("###### no memory for depthbuffer ######\n");
	    return 1;
	}
	/* bytes */
	sctx->DepthBuffOffset = free_pix_offset * 2;
	CCTX(win_depth_set) = 1;
    }
    if(CCTX(win_depth_set))
	free_pix_offset += (screen_size / 2);

/*****************************************************************************
 Textures Memory:
 *****************************************************************************/
    if(do_config) {
	if (free_vidmem_size <= 0) {
	    MSG("###### no memory for textures ######\n");
	    return 1;
	}
	/* pixels */
	sctx->TexBuffOffset = free_pix_offset;
	/* double words (32 bit) - the tex pixels are 32 bit addressed (?)*/
	sctx->TexBuffMin = (CCTX(VidMemSize) - free_vidmem_size) / 4;
	sctx->TexBuffMax = CCTX(VidMemSize) / 4;
    }
    
    return 0;
}



Tbool
PM2Init(ACLContext aclctx)
{
    MLX_INIT_ACCESS_CARD;
    MLX_INIT_ACCESS_MMCTL;

    Tint hwidx = aclctx->hwidx;
    ACLScreen scr = aclctx->Scr;
    MLXDriverRec *drv = mlx_drivers[cctx->drvidx];
    GPMCardInfo_t *card = GpcliCardsInfo + hwidx;
    Tint dac, tmp = 0;
    
    MSG("\n");
    MSG("[%s] %s, Card: %d\n", drv->drvmsg, drv->drvinfo, hwidx);
    MSG("\n");

    aclctx->CardCTL = aclctx->CardCTL0 = gpcli_mmap_pcidev(BASE_0, hwidx, 0);

    if ( !(MMIOBASE = aclctx->CardCTL) ) {
	ERR("###### CTL region not mapped! ######\n");
	return 1;
    }

    /*
     * set hardware/card context if not already
     ************************************************************************/
    if(!CCTX(lock)) {
#if 0
	GPMPciParam_t pciparam = {
	    hwidx, {
		{ PCI_COMMAND, SIZE_BYTE, 
		  PCI_COMMAND_IO | PCI_COMMAND_MEMORY | PCI_COMMAND_MASTER,
		  0, FALSE },
		{ PCI_COMMAND, SIZE_BYTE, 
		  PCI_COMMAND_IO | PCI_COMMAND_MEMORY | PCI_COMMAND_MASTER,
		  0, FALSE },
	    }
	};

	/*
	 * Before access, enable IO and memory of the card(s); we expect
	 * maximal two PCI devices on the card
	 ********************************************************************/
	gpcli_set_pciconfig( &pciparam );
#endif

	MSG("* SETTING NEW CARD CONTEXT...\n");

	if(! (dac = GlintProbeRamdac(MMIOBASE))) {
	    ERR("###### RAMDAC not supported ######\n");
	    return 1;
	}

	CCTX(ExtDacType) = -1;
	GlintFindRamdac(&(cctx->ExtDacType),dac);

	tmp = REG_READ(PMMemConfig);
	CCTX(VidMemSize) = PMFBSize(tmp);

	CCTX(MemType) = (REG_READ(PMRomControl) & 0x10) ? RAM_SDRAM: RAM_SGRAM;
	CCTX(is_vga) = (REG_READ(ChipConfig) & 0x2) ? 1 : 0;

	/* take care with this stuff !!! */
#ifdef PCI_DISCONNECT
	REG_WRITE_SLOW(3, FIFODiscon);
#else
	REG_WRITE_SLOW(0, FIFODiscon);
#endif
    }

    if(PM2_InitVidMem(aclctx)) {
	ERR("###### Cannot init VidMem!!! ######\n");
	return 1;
    }

    sctx = CCTX(Screen);

    if(CCTX(is_full_screen))
	VAL(win_x) = VAL(win_y) = 0;
    else
	/* one day... */
	VAL(win_x) = VAL(win_y) = 0;
	
    VAL(win_w) = scr->WindowWidth;
    VAL(win_h) = scr->WindowHeight;

    MSG("* CARD CONFIGURATION:\n");
    MSG(">>> Permedia 2 CTL mmaped to 0x%lx\n", (Tulong) aclctx->CardCTL);
    MSG(">>> FB (PCI Base 1) at 0x%lx; RAM: %dk\n",
	card->cd[0].PCIBase[1].base, CCTX(VidMemSize)/1024 );
    MSG(">>> RAMDAC type: BUILT-IN\n"); 
    MSG(">>> Fitted Memory type: %s\n", GlintRamtypeString(CCTX(MemType)));
    MSG(">>> VGA core: %s\n", cctx->is_vga ? "ENABLED" : "DISABLED");
    MSG("\n");
    MSG("* VIDEO MEMORY CONFIGURATION:\n");
    MSG(">>> Available Video RAM: %d Bytes\n", CCTX(VidMemSize));
    MSG(">>> Virtual Screen size: %d pix\n",sctx->VirtualW * sctx->VirtualH);
    MSG(">>> Visible Screen size: %d pix\n",sctx->VisibleW * sctx->VisibleH);
    MSG(">>> Used screen size:    %d pix\n", sctx->VisibleW * sctx->VisibleH);
    MSG(">>> Screen: pprod=0x%x, bpp=%d, bytespp=%d\n", sctx->ScreenPProd,
	sctx->ScreenBpp, sctx->BytesPP);
    MSG(">>> Front buffer: %d Kb, Offset: %d pixels\n",
	(sctx->ScreenSize * sctx->BytesPP)/1024, sctx->FrontBuffOffset);
    MSG(">>> Back buffer:  %d Kb, Offset %d pixels\n",
	(sctx->ScreenSize * sctx->BytesPP)/1024, sctx->BackBuffOffset);
    MSG(">>> Depth buffer: %d Kb, Offset %d words\n",
	(sctx->ScreenSize * 2)/1024, sctx->BackBuffOffset);
    MSG(">>> Tex memory:   %d Kb, Offset %d pixels\n",
	((sctx->TexBuffMax-sctx->TexBuffMin)*4)/1024, sctx->TexBuffOffset);
    MSG("\n");
    MSG("* SCREEN CONFIGURATION:\n");
    MSG(">>> Rendering into %s\n",CCTX(is_full_screen)?"FULLSCREEN":"WINDOW");
    MSG(">>> Virtual size (WxH): (%dx%d)\n",sctx->VirtualW, sctx->VirtualH);
    MSG(">>> Visible size (WxH): (%dx%d)\n",sctx->VisibleW, sctx->VisibleH);
    MSG(">>> Client X,Y: (%d,%d) size WxH: (%dx%d)\n",
	VAL(win_x), VAL(win_y), VAL(win_w), VAL(win_h));
    MSG("\n");

#if 0
    /* lists partial products */
    for(tmp=0;tmp<=2048;tmp+=32) {
	Tint pprod1 = GlintGetPProd(CARD_PERMEDIA2,tmp);
	Tint pprod2 = GlintCalcPProd(CARD_PERMEDIA2,tmp);
	MSG("W: %d -> Table: %d,%d,%d    Calculated: %d,%d,%d\n",
		tmp,pprod1>>6,(pprod1>>3)&7,pprod1&7,
		pprod2>>6,(pprod2>>3)&7,pprod2&7)
    }
#endif

    return PM2InitACLAPIModules(aclctx);
}

