#!/usr/bin/perl -w
#
##############################################################################
#          Generic parser for easy hardware register specification
# V1.0
##############################################################################
# Author: simon pogarcic, sim@suse.de
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Library General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Library General Public License for more details.
#
# You should have received a copy of the GNU Library General Public
# License along with this library; if not, write to the Free
# Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
##############################################################################
#
#          PUT YOUR DEFINITIONS WHICH DOESN'T FIT FOR PARSER HERE
#   They will appear at the start of automaticly generated register file
#
##############################################################################
$code_start = <<'EOF';

/*****************************************************************************
 *                                    M L X                                  *
 *                 Rendering Library for Accelerated 3d Hardware             *
 *                           (C) SuSE GmbH 1997, 1998                        *
 *****************************************************************************/
/* author: simon pogarcic, sim@suse.de */

/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef _GLINTREGS_H_
#define _GLINTREGS_H_



/*
 * TAG_REG is used for FIFO access
 */
#define TAG_REG(major,offset)	( ((0x1f & (major))<<4) | (0xf & (offset)) )


/*
 * OFF_REG is offset in REGION 0 (MMIO access)
 */
#define OFF_REG(major,offset)	(0x8000 | (((major)<<7) | ((offset)<<3)) )



#define UNIT_MASK		0xfffffffe /* bit 0 */
#define UNIT_DISABLE		0x0
#define UNIT_ENABLE		0x1



/*
 * Tags, used in DMA HOLD and INCREMENT mode
 */
#define VFIX_s(n)		V ##n## Fixed0Tag
#define VFIX_t(n)		V ##n## Fixed1Tag
#define VFIX_q(n)		V ##n## Fixed2Tag
#define VFIX_Ks(n)		V ##n## Fixed3Tag
#define VFIX_Kd(n)		V ##n## Fixed4Tag
#define VFIX_r(n)		V ##n## Fixed5Tag
#define VFIX_g(n)		V ##n## Fixed6Tag
#define VFIX_b(n)		V ##n## Fixed7Tag
#define VFIX_a(n)		V ##n## Fixed8Tag
#define VFIX_f(n)		V ##n## Fixed9Tag
#define VFIX_x(n)		V ##n## Fixed10Tag
#define VFIX_y(n)		V ##n## Fixed11Tag
#define VFIX_z(n)		V ##n## Fixed12Tag
#define VFIX_pc(n)		V ##n## Fixed14Tag

#define VFLO_s(n)		V ##n## Float0Tag
#define VFLO_t(n)		V ##n## Float1Tag
#define VFLO_q(n)		V ##n## Float2Tag
#define VFLO_Ks(n)		V ##n## Float3Tag
#define VFLO_Kd(n)		V ##n## Float4Tag
#define VFLO_r(n)		V ##n## Float5Tag
#define VFLO_g(n)		V ##n## Float6Tag
#define VFLO_b(n)		V ##n## Float7Tag
#define VFLO_a(n)		V ##n## Float8Tag
#define VFLO_f(n)		V ##n## Float9Tag
#define VFLO_x(n)		V ##n## Float10Tag
#define VFLO_y(n)		V ##n## Float11Tag
#define VFLO_z(n)		V ##n## Float12Tag
#define VFLO_pc(n)		V ##n## Float14Tag

/*
 * Simple MMAP FIFO access with absolute offset
 */
#define VFIXoff_s(n)		V ##n## Fixed0Off
#define VFIXoff_t(n)		V ##n## Fixed1Off
#define VFIXoff_q(n)		V ##n## Fixed2Off
#define VFIXoff_Ks(n)		V ##n## Fixed3Off
#define VFIXoff_Kd(n)		V ##n## Fixed4Off
#define VFIXoff_r(n)		V ##n## Fixed5Off
#define VFIXoff_g(n)		V ##n## Fixed6Off
#define VFIXoff_b(n)		V ##n## Fixed7Off
#define VFIXoff_a(n)		V ##n## Fixed8Off
#define VFIXoff_f(n)		V ##n## Fixed9Off
#define VFIXoff_x(n)		V ##n## Fixed10Off
#define VFIXoff_y(n)		V ##n## Fixed11Off
#define VFIXoff_z(n)		V ##n## Fixed12Off
#define VFIXoff_pc(n)		V ##n## Fixed14Off

#define VFLOoff_s(n)		V ##n## Float0Off
#define VFLOoff_t(n)		V ##n## Float1Off
#define VFLOoff_q(n)		V ##n## Float2Off
#define VFLOoff_Ks(n)		V ##n## Float3Off
#define VFLOoff_Kd(n)		V ##n## Float4Off
#define VFLOoff_r(n)		V ##n## Float5Off
#define VFLOoff_g(n)		V ##n## Float6Off
#define VFLOoff_b(n)		V ##n## Float7Off
#define VFLOoff_a(n)		V ##n## Float8Off
#define VFLOoff_f(n)		V ##n## Float9Off
#define VFLOoff_x(n)		V ##n## Float10Off
#define VFLOoff_y(n)		V ##n## Float11Off
#define VFLOoff_z(n)		V ##n## Float12Off
#define VFLOoff_pc(n)		V ##n## Float14Off

/*
 * Bit offsets for DMA INDEXED mode
 */
#define VFIXidx_s(n)		V ##n## Fixed0Idx
#define VFIXidx_t(n)		V ##n## Fixed1Idx
#define VFIXidx_q(n)		V ##n## Fixed2Idx
#define VFIXidx_Ks(n)		V ##n## Fixed3Idx
#define VFIXidx_Kd(n)		V ##n## Fixed4Idx
#define VFIXidx_r(n)		V ##n## Fixed5Idx
#define VFIXidx_g(n)		V ##n## Fixed6Idx
#define VFIXidx_b(n)		V ##n## Fixed7Idx
#define VFIXidx_a(n)		V ##n## Fixed8Idx
#define VFIXidx_f(n)		V ##n## Fixed9Idx
#define VFIXidx_x(n)		V ##n## Fixed10Idx
#define VFIXidx_y(n)		V ##n## Fixed11Idx
#define VFIXidx_z(n)		V ##n## Fixed12Idx
#define VFIXidx_pc(n)		V ##n## Fixed14Idx

#define VFLOidx_s(n)		V ##n## Float0Idx
#define VFLOidx_t(n)		V ##n## Float1Idx
#define VFLOidx_q(n)		V ##n## Float2Idx
#define VFLOidx_Ks(n)		V ##n## Float3Idx
#define VFLOidx_Kd(n)		V ##n## Float4Idx
#define VFLOidx_r(n)		V ##n## Float5Idx
#define VFLOidx_g(n)		V ##n## Float6Idx
#define VFLOidx_b(n)		V ##n## Float7Idx
#define VFLOidx_a(n)		V ##n## Float8Idx
#define VFLOidx_f(n)		V ##n## Float9Idx
#define VFLOidx_x(n)		V ##n## Float10Idx
#define VFLOidx_y(n)		V ##n## Float11Idx
#define VFLOidx_z(n)		V ##n## Float12Idx
#define VFLOidx_pc(n)		V ##n## Float14Idx


/* Values for PM/PM2 MemConfig register */
#define RowCharge8		1 << 10
#define TimeRCD8		1 <<  7
#define TimeRC8			0x6 << 3
#define TimeRP8			1
#define CAS3Latency8		0 << 16
#define BootAdress8		0x10
#define NumberBanks8		0x3 << 29
#define RefreshCount8		0x41 << 21
#define TimeRASMin8		1 << 13
#define DeadCycle8		1 << 17
#define BankDelay8		0 << 18
#define Burst1Cycle8		1 << 31
#define SDRAM8			0 << 4

#define RowCharge6		1 << 10
#define TimeRCD6		1 <<  7
#define TimeRC6			0x6 << 3
#define TimeRP6			0x2
#define CAS3Latency6		1 << 16
#define BootAdress6		0x60
#define NumberBanks6		0x2 << 29
#define RefreshCount6		0x41 << 21
#define TimeRASMin6		1 << 13
#define DeadCycle6		1 << 17
#define BankDelay6		0 << 18
#define Burst1Cycle6		1 << 31
#define SDRAM6			0 << 4

#define RowCharge4		0 << 10
#define TimeRCD4		0 <<  7
#define TimeRC4			0x4 << 3
#define TimeRP4			1 
#define CAS3Latency4		0 << 16
#define BootAdress4		0x10
#define NumberBanks4		1 << 29
#define RefreshCount4		0x30 << 21
#define TimeRASMin4		1 << 13
#define DeadCycle4		0 << 17
#define BankDelay4		0 << 18
#define Burst1Cycle4		1 << 31
#define SDRAM4			0 << 4

EOF



$code_end = <<'EOF';



#endif 	/* _GLINTREGS_H_ */

EOF



$text = <<'EOF';
###############################################################################

# ==========================================================================
# === REGISTER =============== OFFSET ======= ACCESS = CARD ================
# ==========================================================================

@PCI

@BCOMM PCI Region Registers

CFGVendorId			0x00		ro
CFGDeviceId			0x02		ro
CFGRevisionId			0x08		ro
CFGClassCode			0x09		ro
CGFHeaderType			0x0e		ro
CFGCommand			0x04		rw
CFGStatus			0x06		ro
CFGBist				0x0f		ro
CFGLatTimer			0x0d		rw
CFGCacheLine			0x0c		ro
CFGMaxLat			0x3f		ro
CFGMinGrant			0x3e		ro
CFGIntPin			0x3d		ro
CFGIntLine			0x3c		rw
CFGCardBus			0x28		ro
CFGSubsystemVendorId		0x2c		wc
CFGSubsystemId			0x2e		wc
CFGCapPtr			0x34		ro
CFGCapID			0x40		ro
CFGNextPtr			0x41		ro
CFGAGPRev			0x42		ro
CFGAGPStatus			0x44		ro
CFGAGPCommand			0x48		rw
CFGIndirectAddress		0xf8		rw
CFGIndirectData			0xfc		rw
CFGBaseAddr0			0x10		rw
CFGBaseAddr1			0x14		rw
CFGBaseAddr2			0x18		rw
CFGRomAddr			0x30		rw

@END



@REGION0

@BCOMM GLINT Region 0 Registers

ResetStatus			0x0000		rw txmx
IntEnable			0x0008		rw txmx
IntFlags			0x0010		rw txmx
InFIFOSpace			0x0018		ro txmx
OutFIFOWords			0x0020		ro txmx
InDMAAddress			0x0028		rw txmx
InDMACount			0x0030		rw txmx
ErrorFlags			0x0038		rw txmx
VClkCtl				0x0040		rw txmx
TestRegistar			0x0048		rw txmx
ApertureOne			0x0050		rw txmx
ApertureTwo			0x0058		rw txmx
DMAControl			0x0060		rw txmx
FIFODiscon			0x0068		rw txmx



@LCOMM GLINT Permedia Region 0 additional Registers
ChipConfig			0x0070		rw pm2

@VALUES SCLK
	_			[10-11]
		+SEL_MASK	3
		+SEL_MCLK_HALF	3
@ENDVALUES

OutDMAAddress			0x0080		rw
OutDMACount			0x0088		rw
AGPTexBaseAddress		0x0090		rw



ByDMAAddress			0x00a0		rw
ByDMAStride			0x00b8		rw
ByDMAMemAddr			0x00c0		rw
ByDMASize			0x00c8		rw
ByDMAByteMask			0x00d0		rw
ByDMAControl			0x00d8		rw
ByDMAComplete			0x00e8		rw


@LCOMM GLINT Permedia Memory Control Registers
PMReboot			0x1000
PMRomControl			0x1040
PMBootAddress			0x1080
PMMemConfig			0x10C0



@LCOMM Permedia 2 Control
MemControl			0x1040



PMBypassWriteMask		0x1100
PMFramebufferWriteMask	        0x1140
PMCount				0x1180



@LCOMM LocalBuffer Registers
LBMemoryCtl			0x1000
LBMemoryEDO			0x1008



@LCOMM Framebuffer Registers
FBMemoryCtl			0x1800
FBModeSel			0x1808
FBGCWrMask			0x1810
FBGCColorLower			0x1818
FBTXMemCtl			0x1820
FBWrMaskk			0x1830
FBGCColorUpper			0x1838



@LCOMM Core FIFO
InOutFIFO			0x2000



@LCOMM 500TX Internal Video Registers
VTGHLimit			0x3000
VTGHSyncStart			0x3008
VTGHSyncEnd			0x3010
VTGHBlankEnd			0x3018
VTGVLimit			0x3020
VTGVSyncStart			0x3028
VTGVSyncEnd			0x3030
VTGVBlankEnd			0x3038
VTGHGateStart			0x3040
VTGHGateEnd			0x3048
VTGVGateStart			0x3050
VTGVGateEnd			0x3058
VTGPolarity			0x3060
VTGFrameRowAddr			0x3068
VTGVLineNumber			0x3070
VTGSerialClk			0x3078
VTGModeCtl			0x3080



@LCOMM Permedia Video Control Registers
PMScreenBase			0x3000
PMScreenStride			0x3008
PMHTotal			0x3010
PMHgEnd				0x3018
PMHbEnd				0x3020
PMHsStart			0x3028
PMHsEnd				0x3030
PMVTotal			0x3038
PMVbEnd				0x3040
PMVsStart			0x3048
PMVsEnd				0x3050
PMVideoControl			0x3058
PMInterruptLine			0x3060
PMDDCData			0x3068
PMLineCount			0x3070



@BCOMM Permedia 2 RAMDAC Registers

PM2DACWriteAddress		0x4000
PM2DACIndexReg			0x4000
PM2DACData			0x4008
PM2DACReadMask			0x4010
PM2DACReadAddress		0x4018
PM2DACCursorColorAddres	        0x4020
PM2DACCursorColorData	        0x4028
PM2DACIndexData			0x4050
PM2DACCursorData		0x4058
PM2DACCursorXLsb		0x4060
PM2DACCursorXMsb		0x4068
PM2DACCursorYLsb		0x4070
PM2DACCursorYMsb		0x4078

PM2DACCursorControl		0x06
PM2DACIndexCMR			0x18

@VALUES PM2DAC_
	PixFmt			[0-3]
		+CI8		0
		+332		1
		+2320		2
		+2321		3
		+5551		4
		+4444		5
		+565		6
		+8888		8
		+Packed		9
	GUI			[4]{Svga/Graphics}
	ColorOrder		[5]{BGR/RGB}
	TrueColor		[7]
@ENDVALUES
		
PM2DACIndexMDCR			0x19
PM2DACIndexPalettePage	        0x1c
PM2DACIndexMCR			0x1e
PM2DACIndexClockAM		0x20
PM2DACIndexClockAN		0x21
PM2DACIndexClockAP		0x22
PM2DACIndexClockBM		0x23
PM2DACIndexClockBN		0x24
PM2DACIndexClockBP		0x25
PM2DACIndexClockCM		0x26
PM2DACIndexClockCN		0x27
PM2DACIndexClockCP		0x28
PM2DACIndexClockStatus		0x29
PM2DACIndexMemClockM		0x30
PM2DACIndexMemClockN		0x31
PM2DACIndexMemClockP		0x32
PM2DACIndexMemClockStat		0x33
PM2DACIndexColorKeyControl	0x40
PM2DACIndexColorKeyOverlay	0x41
PM2DACIndexColorKeyRed		0x42
PM2DACIndexColorKeyGreen	0x43
PM2DACIndexColorKeyBlue		0x44



@BCOMM GLINT Delta Region 0 Registers 

@LCOMM Control Status Registers

DResetStatus			0x0800
DIntEnable			0x0808
DIntFlags			0x0810
DErrorFlags			0x0838
DTestRegister			0x0848
DFIFODis			0x0868

@END

# ==========================================================================
# === REGISTER =============== MAJOR = OFFSET ======= TYPE == ACCESS = CARD
# ==========================================================================

@BCOMM Graphic Core Registers

@UNIT Delta

V0Fixed<0-14>			0x20	0x0		ctl     rw

V1Fixed<0-14>			0x21	0x0		ctl	rw

V2Fixed<0-14>			0x22	0x0		ctl	rw

V0Float<0-14>			0x23	0x0		ctl	rw

V1Float<0-14>			0x24	0x0		ctl	rw

V2Float<0-14>			0x25	0x0		ctl	rw

DeltaMode			0x26	0x0		ctl	rw

@VALUES DM_
	Chip			[0-1]
		+300SX		0
		+500TX		1
		+Permedia	2
	DepthFormat		[2-3]
		+16		1
		+24		2
		+32		3
	Fog			[4]
	Texture			[5]
	SmoothShading		[6]
	Depth			[7]
	SpecularTex		[8]
	DiffuseTex		[9]
	SubPixelCorr		[10]
	DiamondExit		[11]
	NoDraw			[12]
	Clamp			[13]
	TexturePar		[14-15]
		+AsGiven	0
		+Clamp		1
		+Norm		2
        BackFaceCull            [17]
        ColorOrder              [18]
                +ABGR           0
                +ARGB           1
@ENDVALUES
		
DrawTriangle			0x26	0x1		ctl	wo
RepeatTriangle			0x26	0x2		ctl	wo
DrawLine01			0x26	0x3		ctl	wo
DrawLine10			0x26	0x4		ctl	wo
RepeatLine			0x26	0x5		ctl	wo

@END



@UNIT Rasterizer

StartXDom			0x00	0x0		ctl	rw
dXDom				0x00	0x1		ctl	rw
StartXSub			0x00	0x2		ctl	rw
dXSub				0x00	0x3		ctl	rw
StartY				0x00	0x4		ctl	rw
dY				0x00	0x5		ctl	rw
Count				0x00	0x6		ctl	rw
Render				0x00	0x7		cmd	wo

@VALUES R_
	AreaStipple		[0]
	LineStipple		[1]
	ResetLStiple		[2]
	FastFill		[3]
	Primitive		[6-7]
		+Line		0
		+Trapezoid	1
		+Point		2
                +Rectangle      3
	Antialias		[8]
	AAliasQuality		[9]{4x4/8x8}
	PointTable		[10]
	SyncOnBMask		[11]
	SyncOnHData		[12]
	Texture			[13]
	Fog			[14]
	Coverage		[15]
	SubPixelCorr		[16]
	SpanOperation		[17]{CColor/Hdata}
        RejectNegativeFace      [20]
        XPositive               [21]
        YPositive               [22]
@ENDVALUES

ContinueNewLine			0x00	0x8		cmd	wo
ContinueNewDom			0x00	0x9		cmd	wo
ContinueNewSub			0x00	0xa		cmd	wo
Continue			0x00	0xb		cmd	wo
FlushSpan			0x00	0xc		cmd	wo txmx
BitMaskPattern			0x00	0xd		mix	wo

PointTable<0-3>			0x01	0x0		ctl	rw txmx
RasterizerMode			0x01	0x4		ctl	rw 

@VALUES RSM_
	MirrorBMask		[0]{UNSET/SET}
	InvertBMask		[1]*
	FractAdjust		[2-3]
		+None		0
		+Zero		1
		+Half		2
		+NHalf		3
	BiasCoordAdd		[4-5]
		+Zero		0
		+Half		1
		+NHalf		2
	BMaskSwapMode		[7-8]
		+ABCD		0
		+BADC		1
		+CDAB		2
		+DCBA		3
	BMask			[9]{Packed/Scanline}
	BMaskOffset		[10-14]
	HDataSwapMode		[15-16]
		+ABCD		0
		+BADC		1
		+CDAB		2
		+DCBA		3
	Glint			[17]{Single/Multi}
	YLimits			[18]
@ENDVALUES

Ylimits				0x01	0x5		ctl	rw
ScanLineOwnership		0x01	0x6		ctl	rw txmx
WaitForCompletion		0x01	0x7		cmd	wo
PixelSize			0x01	0x8		ctl	rw txmx
Xlimits				0x01	0x9		ctl	rw pm2
RectangleOrigin			0x01	0xa		ctl	wo pm2
RectangleSize			0x01	0xb		ctl	wo pm2

PackedDataLimits		0x02	0xa		ctl	rw pm2

@END



@UNIT ScissorStipple

ScissorMode			0x03	0x0		ctl	rw

@VALUES SCM_
	UserScissor		[0]	
        ScreenScissor		[1]
@ENDVALUES

ScissorMinXY			0x03	0x1		ctl	rw
ScissorMaxXY			0x03	0x2		ctl	rw
ScreenSize			0x03	0x3		ctl	rw
AreaStippleMode			0x03	0x4		ctl	rw

@VALUES ASM_
	XAddrSel		[1-3]
		+1bit		0
		+2bit		1
		+3bit		2
		+4bit		3
		+5bit		4
	YAddrSel		[4-6]
		+1bit		0
		+2bit		1
		+3bit		2
		+4bit		3
		+5bit		4
	XOffset			[7-11]
	YOffset			[12-16]
	StipplePattern		[17]{NoInv/Inv}
	X			[18]{NoMirror/Mirror}
	Y			[19]{NoMirror/Mirror}
@ENDVALUES

LineStippleMode			0x03	0x5		ctl	rw txmx
LoadLineStippleCounters		0x03	0x6		ctl	rw txmx
UpdateLineStippleCounters	0x03	0x7		cmd	wo txmx
SaveLineStippleState		0x03	0x8		cmd	wo txmx
WindowOrigin			0x03	0x9		ctl	rw

AreaStipplePattern<0-7>		0x04	0x0		ctl	rw
AreaStipplePattern<8-15>	0x04	0x8		ctl	rw txmx

AreaStipplePattern<16-31>	0x05	0x0		ctl	rw txmx


@END



@UNIT TextureAddress

TextureAddressMode		0x07	0x0		ctl	rw !pm
@VALUES TexAddrM_
        PerCorr                 [1]
@ENDVALUES
SStart				0x07	0x1		ctl	rw
dSdx				0x07	0x2		ctl	rw
dSdyDom				0x07	0x3		ctl	rw
TStart				0x07	0x4		ctl	rw
dTdx				0x07	0x5		ctl	rw
dTdyDom				0x07	0x6		ctl	rw
QStart				0x07	0x7		ctl	rw
dQdx				0x07	0x8		ctl	rw
dQdyDom				0x07	0x9		ctl	rw
LOD				0x07	0xa		ctl	rw mx
dSdy				0x07	0xb		ctl	rw mx
dTdy				0x07	0xc		ctl	rw mx
dQdy				0x07	0xd		ctl	rw mx

@END



@UNIT TextureRead

TextureReadMode			0x09	0x0		ctl	rw txmx


TextureFormat			0x09	0x1		ctl	rw !mx

@VALUES TexF_
	Order			[0]{LittleE/BigE}
	Format			[1]{565_16bpp/555_16bpp}
	ColorOrder		[2]{BGR/RGB}
	NrComponents		[3-4]
		+1		0
		+2		1
		+3		2
		+4		3
	OutFormat		[5-6]
		+Texel		0
		+Color		1
		+BMask		2
	BMaskMirror		[7]
	BMaskInvert		[8]
	BMaskByteSw		[9]
	LUTOffset		[10-17]
	OneCompFmt		[18-19]
		+Lum		0
		+Alpha		1
		+Inten		2
@ENDVALUES

TextureCacheControl		0x09	0x2		ctl	rw txmx

@VALUES TexCC_
	Invalidate		[0]{Off/Cache}
	Cache			[1]
@ENDVALUES

BorderColor			0x09	0x5		ctl	rw txmx
TexelLUTIndex			0x09	0x8		ctl	rw mx pm2
TexelLUTData			0x09	0x9		ctl	rw mx pm2
TexelLUTAddress			0x09	0xa		ctl	rw mx pm2
TexelLUTTransfer		0x09	0xb		ctl	rw mx pm2
TextureFilterMode		0x09	0xc		ctl	rw mx
TextureChromaUpper		0x09	0xd		ctl	wo mx
TextureChromaLower		0x09	0xe		ctl	wo mx

TxBaseAddr<0-11>		0x0a	0x0		ctl	rw !mx
TxBaseAddrLR			0x0a	0x1		ctl	rw txmx

TextureBaseAddress		0x0b	0x0		ctl	rw pm2
TextureMapFormat		0x0b	0x1		ctl	rw pm2

@VALUES TexMF_
        WindowBottomLeft        [16]
        SubPatch                [17]
        TexelSize               [19-20]
          +8bits                0
          +16bits               1
          +32bits               2
          +4bits                3
          +24bits               4
@ENDVALUES


TextureDataFormat		0x0b	0x2		ctl	rw pm2
@VALUES TexDF_
        TextureFormat           [0-3]
          +8888                 0
          +5551                 1
          +4444                 2
          +332F                 5
          +332B                 6
          +2321F                9
          +2321B                10
          +232FO                11
          +232BO                12
          +5551B                13
          +CI8                  14
          +CI4                  15
          +565F                 16
          +565B                 17
          +YUV411               18   
          +YUV422               19
        NoAlphaBuffer           [4]
        ColorOrderRGB           [5]
        FormatExtension         [6]
        AlphaMap                [7-8]
          +None                 0
          +Include              1
          +Exclude              2
        SpanFormatFlip          [9]
@ENDVALUES


TexelLUT<0-15>			0x1d	0x0		ctl	rw pm2

AlphaMapUpperBound		0x1e	0x3		ctl	rw pm2
AlphaMapLowerBound		0x1e	0x4		ctl	rw pm2
TextureID			0x1e	0xe		cmd	rw pm2
TexelLUTID			0x1e	0xf		cmd	rw pm2

@END



@UNIT TextureColorFog

Texel0				0x0c	0x0		ctl	rw
Texel1				0x0c	0x1		ctl	rw txmx
Texel2				0x0c	0x2		ctl	rw txmx
Texel3				0x0c	0x3		ctl	rw txmx
Texel4				0x0c	0x4		ctl	rw txmx
Texel5				0x0c	0x5		ctl	rw txmx
Texel6				0x0c	0x6		ctl	rw txmx
Texel7				0x0c	0x7		ctl	rw txmx
Interp0				0x0c	0x8		ctl	rw txmx
Interp1				0x0c	0x9		ctl	rw txmx
Interp2				0x0c	0xa		ctl	rw txmx
Interp3				0x0c	0xb		ctl	rw txmx
Interp4				0x0c	0xc		ctl	rw txmx
TextureFilter			0x0c	0xd		ctl	rw txmx
TextureReadModePM2		0x0c	0xe		ctl	rw pm2

@VALUES TexRMPM2_
        SWarpMode               [1-2]
            +Clamp              0
            +Repeat             1
            +Mirror             2
        TWarpMode               [3-4]
            +Clamp              0
            +Repeat             1
            +Mirror             2
        FilterMode              [17]
        PackedData              [24]
@ENDVALUES

TexelLUTMode			0x0c	0xf		ctl	rw pm2
@VALUES TexLUTM_
        Pixels                  [10-11]
            +1                  0
            +2                  1
            +4                  2
@ENDVALUES

@END



@UNIT TextureFogColor

TextureColorMode		0x0d	0x0		ctl	rw

@VALUES TexCM_
	OGLAppMode		[1-3]
		+Modulate	0
		+Decal		1
		+Blend		2
                +Copy           3
                +ModulateHighl  4
                +DecalHighl     5
                +CopyHighl      7
	QD3DDecal		[1]
	QD3DModulate		[2]
	QD3DHighlight		[3]
	TexType			[4]{OGL/QD3D}
	KdDDA			[5]
	KsDDA			[6]
	BaseFormat		[7-9]
		+Alpha		0
		+Lum		1
		+LumAlpha	2
		+Intensity	3
		+RGB		4
		+RGBA		5
	CLoadMode		[10-11]
		+None		0
		+Ks		1
		+Kd		2
@ENDVALUES

TextureEnvColor			0x0d	0x1		ctl	rw txmx
FogMode				0x0d	0x2		ctl	rw

@VALUES FOG_
	ColorMode		[1]{RGBA/CI}
        FogTest                 [2]
@ENDVALUES

FogColor			0x0d	0x3		ctl	rw
Fstart				0x0d	0x4		ctl	rw
dFdx				0x0d	0x5		ctl	rw
dFdyDom				0x0d	0x6		ctl	rw
KsStart				0x0d	0x9		ctl	rw
dKsdx				0x0d	0xa		ctl	rw
dKsdyDom			0x0d	0xb		ctl	rw
KdStart				0x0d	0xc		ctl	rw
dKddx				0x0d	0xd		ctl	rw
dKddyDom			0x0d	0xe		ctl	rw


@END



@UNIT ColorDDA

RStart				0x0f	0x0		ctl	rw
dRdx				0x0f	0x1		ctl	rw
dRdyDom				0x0f	0x2		ctl	rw
GStart				0x0f	0x3		ctl	rw
dGdx				0x0f	0x4		ctl	rw
dGdyDom				0x0f	0x5		ctl	rw
BStart				0x0f	0x6		ctl	rw
dBdx				0x0f	0x7		ctl	rw
dBdyDom				0x0f	0x8		ctl	rw
AStart				0x0f	0x9		ctl	rw
dAdx				0x0f	0xa		ctl	rw txmx
dAdyDom				0x0f	0xb		ctl	rw txmx
ColorDDAMode			0x0f	0xc		ctl	rw

@VALUES DDA_
	Shading		        [1]
                 +Flat          0
                 +Gouraud       1
@ENDVALUES

ConstantColor			0x0f	0xd		ctl	rw
Color				0x0f	0xe		mix	wo

@END



@UNIT AlphaTest

AlphaTestMode			0x10	0x0		ctl	rw txmx
AntialiasMode			0x10	0x1		ctl	rw txmx

@END



@UNIT AlphaBlend

AlphaBlendMode			0x10	0x2		ctl	rw !mx pm2

@VALUES ABM_TXMX_
	Src			[1-4]
		+ZERO		0
		+ONE		1
		+DESTCOLOR	2
		+OM_DESTCOLOR	3
		+SRCALPHA	4
		+OM_SRCALPHA	5
		+DESTALPHA	6
		+OM_DESTALPHA	7
		+SRCALPHASAT	8
	Dest			[5-7]
		+ZERO		0
		+ONE		1
		+SRCCOLOR	2
		+OM_SRCCOLOR	3
		+SRCALPHA	4
		+OM_SRCALPHA	5
		+DESTALPHA	6
		+OM_DESTALPHA	7
	CFormat			[8-11]
		+8888		0
		+5555		1
		+4444		2
		+4444_Front	3
		+4444_Back	4
		+332_Front	5
		+332_Back	6
		+121_Front	7
		+121_Back	8
		+555_Back	13
		+CI8		14
		+CI4		15
	AlphaBuff		[12]{Present/NotPresent}
	ColorOrder		[13]{BGR/RGB}
	AlphaType		[14]{OGL/QD3D}
	AlphaDest		[15]{FBData/FBSrcData}
	ColorConv		[16]{Scale/Shift}
	AlphaConv		[17]{Scale/Shift}
@ENDVALUES

@VALUES ABM_PM2_
	Operation		[1-7]
		+Format		16
		+Blend		84
		+PreMult	81
	CFormat			[8-11]
		+8888		0
		+5551F		1
		+4444		2
		+332F		5
		+332B		6
		+2321F		9
		+2321B		10
		+232FO		11
		+232BO		12
		+5551B		13
		+CI8		14
		+565F		0
		+565B		1
	AlphaBuff		[12]{Present/NotPresent}
	ColorOrder		[13]{BGR/RGB}
	BlendType		[14]{RGB/Ramp}
	CFormatExt		[16]
	ColorConv		[17]{Scale/Shift}
	AlphaConv		[18]{Scale/Shift}
@ENDVALUES
		
ChromaUpper			0x1e	0x1		ctl	rw mx
ChromaLower			0x1e	0x2		ctl	rw mx
ChromaTestMode			0x1e	0x3		ctl	rw mx

@END



@UNIT ColorFormat

DitherMode			0x10	0x3		ctl	rw

@VALUES DTM_TXMX_
	Dither			[1]
	XOffset			[6-7]
	YOffset			[8-9]
	ColorOrder		[10]{BGR/RGB}
	Alpha			[14]{DefDither/NoDither}
	Mode			[15]{Truncate/Round}
@ENDVALUES	
       
@VALUES DTM_PM2_
	Dither			[1]
	CFormat			[2-5]
	XOffset			[6-7]
	YOffset			[8-9]
	ColorOrder		[10]{BGR/RGB}
	DitherMethod		[11]{Ordered/Line}
	ForceAlpha		[12-13]
		+ToZero		1
		+ToMax		2
@ENDVALUES

@VALUES COLOR_TXMX_
	Format			[2-5]
		+8888		0
		+5555		1
		+4444		2
		+4444F		3
		+4444B		4
		+332F		5
		+332B		6
		+121F		7
		+121B		8
		+555B		13
		+CI8		14
		+CI4		15
@ENDVALUES

@VALUES COLOR_PM2_
	Format			[0-3]
		+8888		0
		+5551F		1
		+4444		2
		+332F		5
		+332B		6
		+2321F		9
		+2321B		10
		+232FO		11
		+232BO		12
		+5551B		13
		+CI8		14
		+565F		0
		+565B		1
	Ext			[16]
@ENDVALUES

@END



@UNIT LogicalOps

FBSoftwareWriteMask		0x10	0x4		ctl	rw
LogicOpMode			0x10	0x5		ctl	rw

@VALUES LOP_
	Op			[1-4]
		+Clear		0
		+And		1
		+AndRev		2
		+Copy		3
		+AndInv		4
		+NoOp		5
		+Xor		6
		+Or		7
		+Nor		8
		+Equ		9
		+Invert		10
		+OrRev		11
		+CopyInv	12
		+OrInv		13
		+Nand		14
		+Set		15
	UseFBWriteData		[5]{VAR/CONST}
@ENDVALUES
	
FBWriteData			0x10	0x6		ctl	rw

@END



@UNIT Router

RouterMode			0x10	0x8		ctl	rw ?

@VALUES RM_
	Order			[0]{TexDepth/DepthTex}
@ENDVALUES

@END



@UNIT LBReadWrite

LBReadMode			0x11	0x0		ctl	rw !mx

@VALUES LBRM_
	PProd			[0-8]
	PProd0			[0-2]
	PProd1			[3-5]
	PProd2			[6-8]
	Src			[9]
	Dest			[10]
	DataLB			[16-17]
		+Default	0
		+Stencil	1
		+Depth		2
	WinOrigin		[18]{Top/Bottom}
	Patch			[19]
	ScanLineInt		[20-21]
		+1		0
		+2		1
		+4		2
		+8		3
	PatchCode		[22-24]
	PProd3			[25-27]
@ENDVALUES

LBReadFormat			0x11	0x1		ctl	rw
@VALUES LBRF_
	DepthWidth		[0-1]
                +16             0
                +15             3
	StencilWidth		[2-3]
		+0		0
		+1              3
@ENDVALUES


LBSourceOffset			0x11	0x2		ctl	rw
LBData				0x11	0x3		ctl	wo pm2
LBStencil			0x11	0x5		out	n
LBDepth				0x11	0x6		out	n
LBWindowBase			0x11	0x7		ctl	rw
LBWriteMode			0x11	0x8		ctl	rw

@VALUES LBWM_
	Write			[0]
	Upload			[1-2]
		+None		0
		+LBDepth	1
		+LBStencil	2
@ENDVALUES

LBWriteFormat			0x11	0x9		ctl	rw

@VALUES LBF_
	DepthWidth		[0-1]
		+16		0
		+24		1
		+32		2
	StencilWidth		[2-3]
		+0		0
		+4		1
		+8		2
	StencilPos		[4-6]
		+16		0
		+20		1
		+24		2
		+28		3
		+32		4
	FCWidth			[7-8]
		+0		0
		+4		1
		+8		2
	FCPos			[9-11]
		+16		0
		+20		1
		+24		2
		+28		3
		+32		4
		+36		5
		+40		6
	GIDWidth		[12]
		+0		0
		+4		1
	GIDPos			[13-16]
		+16		0
		+20		1
		+24		2
		+28		3
		+32		4
		+36		5
		+40		6
		+44		7
		+48		8
	Compact32		[17]
@ENDVALUES
	
TextureData			0x11	0xd		ctl	wo
TextureDownloadOffset		0x11	0xe		ctl	rw
LBWindowOffset			0x11	0xf		ukn	rw txmx

@END



@UNIT GIDStencilDepth

Window				0x13	0x0		ctl	rw

@VALUES WIN_
	CmpPass			[1-2]
		+ALWAYS		0
		+NEVER		1
		+EQ		2
		+NEQ		3
	ForceLBUpdate		[3]
	LBUpdateSrc		[4]{Data/Registers}
	GID			[5-8]
	FC			[9-16]
	StencilFCP		[17]
	DepthFCP		[18]
	NoWriteFiltering	[19]
@ENDVALUES

StencilMode			0x13	0x1		ctl	rw

@VALUES STM_
	UpdateZPass		[1-3]
		+KEEP		0
		+ZERO		1
		+REPLACE	2
		+INC		3
		+DEC		4
		+INV		5
	UpdateZFail		[4-6]
		+KEEP		0
		+ZERO		1
		+REPLACE	2
		+INC		3
		+DEC		4
		+INV		5
	UpdateStFail		[7-9]
		+KEEP		0
		+ZERO		1
		+REPLACE	2
		+INC		3
		+DEC		4
		+INV		5
	CompareFunc		[10-12]
		+NEVER		0
		+LESS		1
		+EQUAL		2
		+LEQUAL		3
		+GREATER	4
		+NEQUAL		5
		+GEQUAL		6
		+ALWAYS		7
	Src			[13-14]
		+TestLogic	0
		+StencilReg	1
		+LBData		2
		+LBSourceData	3
@ENDVALUES

StencilData			0x13	0x2		ctl	rw
Stencil				0x13	0x3		mix	rw
DepthMode			0x13	0x4		ctl	rw

@VALUES DPM_
	Write			[1]
	Src			[2-3]
		+Fragment	0
		+LBData		1
		+Register	2
		+LBSrcData	3
	Func			[4-6]
		+NEVER		0
		+LESS		1
		+EQ		2
		+LEQ		3
		+GREATER	4
		+NEQ		5
		+GEQ		6
		+ALWAYS		7
@ENDVALUES

Depth				0x13	0x5		mix	rw
ZStartU				0x13	0x6		ctl	rw
ZStartL				0x13	0x7		ctl	rw
dZdxU				0x13	0x8		ctl	rw
dZdxL				0x13	0x9		ctl	rw
dZdyDomU			0x13	0xa		ctl	rw
dZdyDomL			0x13	0xb		ctl	rw
FastClearDepth			0x13	0xc		ctl	rw txmx

@END



@UNIT FBReadWrite

FBReadMode			0x15	0x0		ctl	rw

@VALUES FBRM_
	PProd			[0-8]
	Src			[9]
	Dest			[10]
	DataFB			[15]{Default/Color}
	WinOrigin		[16]{Top/Bottom}
	Packed			[19]
	ScanLineInt		[23-24]
		+1		0
		+2		1
		+4		2
		+8		3
@ENDVALUES

FBSourceOffset			0x15	0x1		ctl	rw
FBPixelOffset			0x15	0x2		ctl	rw
FBColor				0x15	0x3		out	n
FBData				0x15	0x4		mix	wo
FBSourceData			0x15	0x5		mix	wo
FBWindowBase			0x15	0x6		ctl	rw
FBWriteMode			0x15	0x7		ctl	rw

@VALUES FBWM_
	Write			[0]
	ColorUpload		[3]
@ENDVALUES

FBHardwareWriteMask		0x15	0x8		ctl	rw
FBBlockColor			0x15	0x9		ctl	rw
FBReadPixel			0x15	0xa		ctl	rw pm2

@VALUES FBRP_
	Bits                    [0-2]
                +8              0
                +16             1
                +32             2
                +24             4
@ENDVALUES


PatternRamMode			0x15	0xf		ctl	rw txmx

PatternRamData<0-15>		0x16	0x0		ctl	rw txmx

PatternRamData<16-31>		0x17	0x0		ctl	rw txmx

FBBlockColorU			0x18	0xd		ctl	rw
FBBlockColorL			0x18	0xe		ctl	rw
SuspendUntilFrameBlank		0x18	0xf		cmd	wo

FBSourceBase			0x1b	0x0		ctl	rw pm2
FBSourceDelta			0x1b	0x1		cmd	rw pm2

@END



@UNIT HostOut

FilterMode			0x18	0x0		ctl	rw

@VALUES FM_
	DepthTags		[4]{Cull/Pass}
	DepthData		[5]*
	StencilTags		[6]*
	StencilData		[7]*
	ColorTag		[8]*
	ColorData		[9]*
	SyncTag			[10]*
	SyncData		[11]*
	StatisticTag		[12]*
	StatisticData		[13]*
@ENDVALUES

StatisticMode			0x18	0x1		ctl	rw
MinRegion			0x18	0x2		ctl	rw
MaxRegion			0x18	0x3		ctl	rw
ResetPickResult			0x18	0x4		cmd	wo
MinHitRegion			0x18	0x5		cmd	wo
MaxHitRegion			0x18	0x6		cmd	wo
PickResult			0x18	0x7		cmd	rw
Sync				0x18	0x8		cmd	wo

@END



@UNIT Multiple

Config				0x1b	0x2		cmd	wo pm2

@END



@UNIT YUV

YUVMode				0x1e	0x0		ctl	rw pm2
ChromaUpperBound		0x1e	0x1		ctl	rw pm2
ChromaLowerBound		0x1e	0x2		ctl	rw pm2

@END



@UNIT KsDDA

KsRStart			0x19	0x0		ctl	rw mx
dKsRdx				0x19	0x1		ctl	rw mx
dKsRdyDom			0x19	0x2		ctl	rw mx
KsGStart			0x19	0x3		ctl	rw mx
dKsGdx				0x19	0x4		ctl	rw mx
dKsGdyDom			0x19	0x5		ctl	rw mx
KsBStart			0x19	0x6		ctl	rw mx
dKsBdx				0x19	0x7		ctl	rw mx
dKsBdyDom			0x19	0x8		ctl	rw mx

@END



@UNIT KdDDA

KdRStart			0x1a	0x0		ctl	rw mx
dKdRdx				0x1a	0x1		ctl	rw mx
dKdRdyDom			0x1a	0x2		ctl	rw mx
KdGStart			0x1a	0x3		ctl	rw mx
dKdGdx				0x1a	0x4		ctl	rw mx
dKdGdyDom			0x1a	0x5		ctl	rw mx
KdBStart			0x1a	0x6		ctl	rw mx
dKdBdx				0x1a	0x7		ctl	rw mx
dKdBdyDom			0x1a	0x8		ctl	rw mx

@END



###############################################################################
EOF

sub myprint;

$curr = `pwd`;
chomp $curr;

open OUT, ">$curr/glintregs.h";

myprint $code_start;
myprint "/*************** (START) AUTOMATICLY GENERATED REGISTER FILE *****************/";
myprint "\n/*\n * Generated on ", `date`, " */\n\n";

for $parse (0,1) {

    $reg_sufix_string = ("Tag", "Off")[$parse];
    $reg_define_macro = ("TAG_REG", "OFF_REG")[$parse];

    if ($parse) {
	myprint "\n\n\n/*","*"x76,
	"\n * REGION 0 REGISTER OFFSETS: \n ","*"x77, "/\n"
    }

    $mode=0;
    $gemetzeltes = $text;

    MAIN_LOOP:
    while ( $gemetzeltes =~ s/^(.*)\n// ) {
	$line = $1;

	$line =~ s/^\s*//;
	$line =~ s/\s*$//;
	$line =~ s/#.*//;
	$line || next MAIN_LOOP;

	if ($line =~ /\@BCOMM\s+(.*)/) {
		$comm = $1;
		$parse || myprint "\n/*\n * ", $comm, "\n */\n\n";
		next;
	}

	if ($line =~ /\@LCOMM\s+(.*)/) {
		$comm = $1;
		$parse || myprint "\n/* ", $comm, " */\n\n";
		next;
	}

	if ($line =~ /\@UNIT\s+(\w+)/) {
		$name = $1;
		$parse || myprint "/*\n * UNIT: ", $name, "\n */\n";
		$mode = 1;
		next;
	}

	if ($line =~ /\@PCI/) {
		$name = "PCI Config Region";
		$parse || do {  myprint "\n"; $mode = 2; };
		next;
	}

	if ($line =~ /\@REGION0/) {
		$name = "Region 0 Registers";
		$parse || do {  myprint "\n"; $mode = 3; };
		next;
	}

	if ($line =~ /\@VALUES\s+(\w+)/) {
		$vpref = $1;
		$lastmode = $mode;
		$mode = 0;
		$parse || do { myprint "\n"; $mode = 4; };
		($strunset, $strset) = ("DIS", "EN");
		$single_bit = 0;
		next;
	}

	if ($line =~ /\@ENDVALUES/) {
		$name = "";
		if( $single_bit ) {
		    val_formats("$vpref$lastval$l_strunset", $bvmask, "");
		    val_formats("$vpref$lastval$l_strset", $bvset, "");
		}
		$parse || myprint "\n";
		$mode = $lastmode;
		next;
	}

	if ($line =~ /\@END/) {
		$name = "";
		$parse || myprint "\n\n\n";
		$mode = 0;
		next;
	}

	UNIT_MODE: { $mode == 1 || last UNIT_MODE;
	    ($reg, $major, $offset, $type, $access, @cards) = split ' ', $line;
	    if ($reg =~ /(\w+)<(\d+)-(\d+)>/ ) {
		($p_reg,$start,$end) = ($1,$2,$3);
		$c_off = $offset;
		$offset =~ /0x/ && ($c_off = hex($offset));
		for $cnt ($start..$end) {
		  $reg = $p_reg.$cnt;
		  $offset = sprintf "0x%x",$c_off;
	    	  reg_formats($reg,$major,$offset,$type,$access,$parse,@cards);
		  $c_off++;
		}
		next MAIN_LOOP;
	    }
	    reg_formats($reg,$major,$offset,$type,$access,$parse,@cards);
	    next MAIN_LOOP;
	}

	PCI_MODE: { $mode == 2 || last PCI_MODE;
	    ($reg, $offset, $access) = split ' ', $line;
	    reg_simplef($reg, $offset);
	    next MAIN_LOOP;
	}

	REGION0_MODE: { $mode == 3 || last REGION0_MODE;
	    ($reg, $offset, $access) = split ' ', $line;
	    reg_simplef($reg, $offset);
	    next MAIN_LOOP;
	}

	VALUES_MODE: { $mode == 4 || last VALUES_MODE;
	    if( $line =~ /(\w+)\s+\[\s*([0-9- ]+)\s*\]/ ) {
	        ($val, $bit) = ($1, $2);
		if( $line !~ /\]\s*\*/ ) {
		    ($strunset, $strset) = ("DIS", "EN");
		}
		if( $line =~ /\]\s*\{\s*(\w+)\s*\/(\w+)\s*\}/ ) {
		    ($strunset, $strset) = ($1, $2);
		}
		if( $single_bit ) {
		    val_formats("$vpref$lastval$l_strunset", "0x0", "");
		    val_formats("$vpref$lastval$l_strset", $bvset, "");
		}

		if( $bit =~ /(\d+)\s*-\s*(\d+)/ ) {
		    ($bfrom, $bto) = ($1, $2);
		    $single_bit = 0;
		}
		elsif ( $bit =~ /(\d+)/ ) {
		    $bfrom = $bto = $1;
		    ($lastval,$l_strunset,$l_strset)=($val,$strunset,$strset);
		    $single_bit = 1;
		}
		else {
		    print("/* BIT RANGE SYNTAX ERROR ENCOUNTERD! */\n");
		    myprint("/* BIT RANGE SYNTAX ERROR ENCOUNTERD! */\n");
		}

		($bvmask, $bvset) = get_bits_value($bfrom,$bto);
		$comment = "/* bits $bfrom-$bto */";
		if($bfrom == $bto) { $comment = "/* bit $bfrom */"; }
		val_formats("$vpref${val}MASK", $bvmask, $comment);
	    }
	    elsif($line =~ /\s*\+(\w+)\s+(\d+)/) {
		($vsuf, $rv) = ($1, $2);
		$single_bit = 0;
		$srv = shift_value_nrbits($rv, $bfrom);
		$comment = "/* val $rv, shift $bfrom */";
		val_formats("$vpref$val$vsuf", $srv, $comment);
	    }
	    else {
		print("/* VALUE LINE SYNTAX ERROR ENCOUNTERD! */\n");
		myprint("/* VALUE LINE SYNTAX ERROR ENCOUNTERD! */\n");
	    }
	    next MAIN_LOOP;
	}
    }
}

myprint "/**************** (END) AUTOMATICLY GENERATED REGISTER FILE ******************/\n";
myprint $code_end;

close OUT;



sub shift_value_nrbits
{
	my ($val, $nrbits) = @_;
	my ($ret);

	$ret = sprintf "0x%x", ($val << $nrbits);

	return $ret;
}



sub get_bits_value
{
	my ($start, $end) = @_;
	my ($cnt, $bmask, $bset);

	$bset = 0;
	for $cnt ($start..$end) {
	    $bset |= (1 << $cnt);
	}
	$bmask = sprintf "0x%x", ~$bset;
	$bset = sprintf "0x%x", $bset;

	return ($bmask,$bset);
}



sub val_formats
{
	my ($reg, $rval, $comm) = @_;
	my ($cs, $len, $rtab, $vtab);
	
	$len = length($reg);
	$rtab =" "."\t"x(5 - int(($len + 13)/8) );

	$len = length($rval);
	$vtab =" "."\t"x(2 - int(($len + 1)/8) );

	myprint "    #define ", $reg, $rtab, $rval, $vtab, "$comm\n";
}



sub reg_simplef
{
	my ($reg, $offset) = @_;
	my ($len, $tab);
	
	$len = length($reg);
	$tab =" "."\t"x(5 - int(($len + 9)/8) );

	myprint "#define ", $reg, $tab, $offset;
	myprint "\n";
}



sub reg_formats
{
	my ($reg,$major,$offset,$type,$access,$second,@cards) = @_;
	my ($cs, $len, $tab, $boff, $shift);
	
	$len = length($reg) + length($reg_sufix_string);
	$tab =" "."\t"x(5 - int(($len + 9)/8) );

	$cs = "";
	foreach $card (@cards) {
		$cs .= " ".$card;
	}

	$cs && ($cs = " /*".$cs." */");

	myprint "#define ", $reg, $reg_sufix_string, $tab,
		$reg_define_macro, "( ", $major, ", ", $offset," )", $cs;
	myprint "\n";

	if($second) {
	    $offset =~ /0x/ && ($shift = hex($offset));
	    $boff = sprintf "0x%x", (1 << ($shift + 16));
	    myprint "#define ", $reg, "Idx", $tab, $boff;
	    myprint "\n\n";
	}
}



sub myprint { foreach $txt (@_) { print OUT $txt; } }



