/*
 * -*- mode: C; tab-width:8;  -*-
 * 
 * aclApi.c - Mesa interface to hw acceleration library
 */

/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ********************************************************************
 *
 * Function names:
 *  aclMesa....     (The driver API)
 *  aclDD....       (Mesa device driver functions)
 *  aclTM....       (Texture manager functions)
 *  aclSetup....    (setup functions)
 *  acl....         (Internal driver functions)
 *
 * Data type names:
 *  aclMesa....     (Public driver data types)
 *  tacl....        (Private driver data types)
 *
 ********************************************************************/

#if defined(ACL)

#include "aclDrv.h"

TINIT(0)



ACLSetup aclMesaHwSetup;
ACLContext aclMesaCurrentCtx = NULL;
acl_sighandler_t aclOldSignalHandler = NULL;

extern aclXVidModeDataPtr aclMesaCurrentVidMode;



static void
aclMesaSighandler (int nr)
{
    MSG("signal %d caught\n", nr);
    if (nr == SIGINT) {
      signal ( nr, aclOldSignalHandler );
      exit(0);
    }
}



static void
aclMesaOnExit( void )
{
    aclMesaDestroyContext(aclMesaCurrentCtx);
}



/*
 * Create a new ACL/Mesa context and return a handle to it.
 */
ACLContext APIENTRY
aclMesaCreateContext(Display *dpy, Window win,
		    GLint width, GLint height, aclMesaVisAtr * const va)
{
    ACLContext aclMesa;
    ACLVisual aclvis;
    ACLModules aclmod;

    ACLint drvidx;

    char *env_hw = getenv("ACL_MESA_HW");
    char *env_full = getenv("ACL_MESA_FULL");
    char *env_quake = getenv("ACL_MESA_QUAKE");

TON

LOG("aclMesaCreateContext() Start\n");

/******************************************************************************
 * before doing anything, catch the signals
 ******************************************************************************/

    if(atexit(aclMesaOnExit)) {
    	MSG("WARNING! Cannot set atexit() routine!\n");
    }

    aclOldSignalHandler =
	signal(SIGINT, aclMesaSighandler);
    signal( SIGUSR1, aclMesaSighandler );

#if 0

    signal( SIGHUP,  aclMesaSighandler );
    signal( SIGQUIT, aclMesaSighandler );
    signal( SIGILL,  aclMesaSighandler );
    signal( SIGABRT, aclMesaSighandler );
    signal( SIGFPE,  aclMesaSighandler );
    signal( SIGKILL, aclMesaSighandler );
    signal( SIGSEGV, aclMesaSighandler );
    signal( SIGPIPE, aclMesaSighandler );
    signal( SIGALRM, aclMesaSighandler );
    signal( SIGTERM, aclMesaSighandler );
    signal( SIGUSR2, aclMesaSighandler );
    signal( SIGSTOP, aclMesaSighandler );
    signal( SIGTTIN, aclMesaSighandler );
    signal( SIGTTOU, aclMesaSighandler );
#endif



/******************************************************************************
 * the connection with ACL library - hardware scan and init 
 ******************************************************************************/
LOG("aclMesaCreateContext() (AclEnter)\n");
    if ( AclEnter() ) {
	MSG("\n");
	MSG("LIBRARY INITIALIZATION FAILED.\n");
	MSG("    Falling back to software rendering...\n\n");
	return NULL;
    }



/******************************************************************************
 * choosing the driver for rendering
 ******************************************************************************/
LOG("aclMesaCreateContext() (AclQueryCard)\n");
    if(env_hw) {
	if( !strcmp(env_hw,"PM2") ) {
	    drvidx = AclQueryCard(-1, ACL_DRIVER_PM2, ACL_GET_FIRST);
	}
	else if( !strcmp(env_hw,"TXMX") ) {
	    drvidx = AclQueryCard(-1, ACL_DRIVER_TXMX, ACL_GET_FIRST);
	}
	else {
	    MSG("\n");
	    MSG("UNKNOWN DRIVER: '%s' ( in env ACL_MESA_HW )\n", env_hw);
	    MSG("    Falling back to software rendering...\n\n");
	    AclExit( NULL );
	    return NULL;
	}
    }
    else {
	MSG("\n");
	MSG("( ACL_MESA_HW ): You have to specify the env var\n");
	MSG("    Supported: 'PM2'\n");
	MSG("    Falling back to software rendering...\n\n");
	AclExit( NULL );
	return NULL;
    }

    if ( drvidx == -1 ) {
	MSG("\n");
	MSG("MISSING DEVICE: no '%s' hardware found!\n", env_hw);
	MSG("    Falling back to software rendering...\n\n");
	AclExit( NULL );
	return NULL;
    }

LOG("aclMesaCreateContext() Start: aclMesaHwSetup (malloc)\n");
    aclMesaHwSetup = (ACLSetup) malloc(sizeof(struct acl_setup));
    if (!aclMesaHwSetup) {
	AclExit( NULL );
	return NULL;
    }

LOG("aclMesaCreateContext() Start: aclvis (malloc)\n");
    aclvis = (ACLVisual) malloc(sizeof(struct gl_visual));
    if (!aclvis) {
	free(aclMesaHwSetup);
	AclExit( NULL );
	return NULL;
    }

    aclvis->RGBAflag = va->is_rgba;
    aclvis->DBflag = va->is_db;
    aclvis->StereoFlag = va->is_stereo;

    aclvis->RedBits = va->r;
    aclvis->GreenBits = va->g;
    aclvis->BlueBits = va->b;
    aclvis->AlphaBits = va->a;
    aclvis->IndexBits = va->i;

    aclvis->AccumBits = va->accum;
    aclvis->DepthBits = va->depth;
    aclvis->StencilBits = va->stencil;

    aclvis->FrontAlphaEnabled = va->is_frontalpha;
    aclvis->BackAlphaEnabled = va->is_backalpha;

LOG("ALPHA bits: %d\n", va->a);

LOG("aclMesaCreateContext(): AclCreateContext\n");
    aclMesa = AclCreateContext( drvidx );
    
    if (!aclMesa) {
	free(aclvis);
	AclExit( NULL );
	return NULL;
    }
    
    aclmod = aclMesa->mod;

    aclMesa->Flags = 0;
    aclMesa->width = width;
    aclMesa->height = height;
    aclMesa->haveDoubleBuffer = aclvis->DBflag;

    aclMesa->vis = gl_create_visual(GL_TRUE,	/* aclvis->DBflag */
				     (aclvis->AlphaBits > 0),
				     aclvis->DBflag, 
				     GL_FALSE,	/* aclvis->StereoFlag, */
				     aclvis->DepthBits,
				     aclvis->StencilBits,
				     aclvis->AccumBits,
				     0,		/* aclvis->IndexBits, */
				     aclvis->RedBits,
				     aclvis->GreenBits,
				     aclvis->BlueBits,
				     aclvis->AlphaBits);

    aclMesa->hctx = gl_create_context(aclMesa->vis,
				     NULL,	/* share list context */
				     (void *) aclMesa,
				     GL_TRUE);

    aclMesa->buff = gl_create_framebuffer(aclMesa->vis);

LOG("aclMesaCreateContext(): aclXInitVidModes()\n");
    if(aclXInitVidModes( dpy, width, height )) {
    	MSG("*** CANNOT SETUP X VIDEO MODES\n");
    }

    /*
     * ACL_MESA_FULL environment variable:
     * ===================================
     * 0 or undefined:	BitBlt 'swap'
     * 1:		Screen swap
     * 2:		BitBlt 'swap'
     * 3:		Pixel Depth swap (not implemented yet)
     */
    if (env_full) {
	if( *env_full == '1')
	    aclMesa->Flags |= ACL_SWAP_SCREEN;
	else if( *env_full == '3')
	    aclMesa->Flags |= ACL_SWAP_PIXDEPTH;
    }

    if(env_quake && *env_quake=='1')
	aclMesa->Flags |= ACL_QUAKE_HACK;

/******************************************************************************
 * ...3,2,1, GO !!! Everything is prepared, now do the final initialization
 ******************************************************************************/
LOG("aclMesaCreateContext() (InitClient)\n");
    if(aclmod->InitClient( aclMesa, aclMesaHwSetup )) {
	MSG("*** CANNOT INITIALIZE RENDERING HARDWARE! ***\n\n");
	free(aclvis);
	AclExit( NULL );
	return NULL;
    }
    
LOG("aclMesaCreateContext() (aclSetupDDPointers)\n");
    aclSetupDDPointers(aclMesa->hctx);

LOG("aclMesaCreateContext() (DrawBuffer)\n");
    if(aclvis->DBflag)
      aclmod->DrawBuffer(aclMesa, GL_BACK);
    else
      aclmod->DrawBuffer(aclMesa, GL_FRONT);
    
LOG("aclMesaCreateContext() End\n");
    return aclMesa;
}

 

/*
 * Destroy the given PM/Mesa context.
 */
void APIENTRY
aclMesaDestroyContext(ACLContext aclMesa)
{
    ACLModules aclmod;

LOG("aclMesaDestroyContext()\n");

    if (!aclMesa) {
	AclExit(NULL);
	return;
    }

    aclmod = aclMesa->mod;

    aclmod->ExitClient( aclMesa );

    if (aclMesa == aclMesaCurrentCtx)
	aclMesaCurrentCtx = NULL;

    aclXSetDefaultVidModes();

    /*
     * at the end, always call the folowing function:
     */
    gl_destroy_visual(aclMesa->vis);
    gl_destroy_context(aclMesa->hctx);
    gl_destroy_framebuffer(aclMesa->buff);

    AclExit( aclMesa );
}



/*
 * Swap front/back buffers for current context if double buffered.
 */
void APIENTRY
aclMesaSwapBuffers(ACLContext aclMesa)
{

TLOG("----- aclMesaSwapBuffers() -----\n");

    if (aclMesaCurrentCtx)
	if (aclMesaCurrentCtx->haveDoubleBuffer) {
	    aclMesa->mod->SwapBuffer( aclMesa );
	}
}



/*
 * Make the specified ACL/Mesa context the current one.
 */
void APIENTRY
aclMesaMakeCurrent(ACLContext aclMesa)
{

TLOG("aclMesaMakeCurrent(...)\n");

    if (!aclMesa) {
	gl_make_current(NULL, NULL);
	aclMesaCurrentCtx = NULL;
	return;
    }

    if (aclMesaCurrentCtx == aclMesa) {
	return;
    }

    aclMesaCurrentCtx = aclMesa;
    gl_make_current(aclMesa->hctx, aclMesa->buff);

    aclSetupDDPointers(aclMesa->hctx);

    /*
     * The first time we call MakeCurrent we set the initial viewport size 
     */
    if (aclMesa->hctx->Viewport.Width == 0)
	gl_Viewport(aclMesa->hctx, 0, 0, aclMesa->width, aclMesa->height);
}



#else

/*
 * Need this to provide at least one external definition.
 */

int
gl_pm_dummy_function_api(void)
{
    return 0;
}

#endif /* ACL */

