/*
 * Compiz cube model plugin
 *
 * cubemodel.h
 *
 * This plugin displays wavefront (.obj) 3D mesh models inside of
 * the transparent cube.
 *
 * Copyright : (C) 2008 by David Mikos
 * E-mail    : infiniteloopcounter@gmail.com
 *
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */
 
#ifndef _CUBEMODEL_INTERNAL_H
#define _CUBEMODEL_INTERNAL_H

#include <math.h>
#include <float.h>

#include <core/core.h>
#include <core/pluginclasshandler.h>
#include <composite/composite.h>
#include <opengl/opengl.h>
#include <cube/cube.h>

#include "cubemodel_options.h"

#define LRAND()   ((long) (random () & 0x7fffffff))
#define NRAND(n)  ((int) (LRAND () % (n)))
#define MAXRAND   (2147483648.0) /* unsigned 1 << 31 as a float */

/* some constants */
#define PI        3.14159265358979323846
#define PIdiv2    1.57079632679489661923
#define toDegrees 57.2957795130823208768
#define toRadians 0.01745329251994329577

/* return random number in range [0,x) */
#define randf(x) ((float) (rand () / (((double) RAND_MAX + 1) / (x))))

typedef struct _vect3d
{
    float r[3];
} vect3d;

typedef struct _vect2d
{
    float r[2];
} vect2d;

typedef struct _groupIndices
{
    int polyCount;
    int complexity;

    int startV;
    int numV;
    int startT;
    int numT;
    int startN;
    int numN;

    int materialIndex; //negative for no new material

    bool texture;
    bool normal;
} groupIndices;

typedef struct _mtlStruct
{
    char *name;

    GLfloat Ka[4];
    GLfloat Kd[4];
    GLfloat Ks[4];

    GLfloat Ns[1]; /* shininess */
    GLfloat Ni[1]; /* optical_density - currently not used*/

    int illum;

    unsigned height, width;

    int map_Ka; /* index to tex, negative for none */
    int map_Kd;
    int map_Ks;
    int map_d;

    int map_params; /* index tex map with height/width */
} mtlStruct;

typedef struct _CubemodelObject
{
    pthread_t thread;
    bool      threadRunning;
    bool      finishedLoading;
    bool      updateAttributes; /* after finished loading in thread, read in
				   new attributes not read in before to avoid
				   race condition*/
    char *filename;
    char *post;

    int size;
    int lenBaseFilename;
    int startFileNum;
    int maxNumZeros;

    GLuint dList;
    bool   compiledDList;

    float  rotate[4], translate[3], scale[3];
    float  rotateSpeed, scaleGlobal;
    float  color[4];

    int   fileCounter;
    bool  animation;
    int   fps;
    float time;

    vect3d **reorderedVertex;
    vect2d **reorderedTexture;
    vect3d **reorderedNormal;

    unsigned int *indices;
    groupIndices *group;

    vect3d *reorderedVertexBuffer;
    vect2d *reorderedTextureBuffer;
    vect3d *reorderedNormalBuffer;

    int nVertex;
    int nTexture;
    int nNormal;
    int nGroups;
    int nIndices;
    int nUniqueIndices;
    int *nMaterial;

    mtlStruct        **material;
    GLTexture::List  *tex; /* stores all the textures */
    char             **texName;
    unsigned int     *texWidth;
    unsigned int     *texHeight;

    int nTex;
} CubemodelObject;

typedef struct _fileParser
{
    FILE *fp;
    char *oldStrline;
    char *buf;
    int  bufferSize;
    int  cp;
    bool lastTokenOnLine;
    int  tokenCount;
} fileParser;

class CubemodelScreen :
    public PluginClassHandler <CubemodelScreen, CompScreen>,
    public CompositeScreenInterface,
    public GLScreenInterface,
    public CubeScreenInterface,
    public CubemodelOptions
{
    public:

	CubemodelScreen (CompScreen *);
	~CubemodelScreen ();
	
	CompositeScreen *cScreen;
	GLScreen	*gScreen;
	CubeScreen	*cubeScreen;
	
    public:

	void
	initWorldVariables ();

	void
	updateModel (int start,
		     int end);
				      
	void
	initCubemodel ();

	void
	freeCubemodel ();

	void
	updateCubemodel ();
	
	bool
	compileDList (CubemodelObject *data);
	
	void
	loadMaterials (CubemodelObject *data,
		       char            *approxPath,
		       char            *filename,
		       mtlStruct       **material,
		       int             *nMat);
		       
	bool
	initLoadModelObject (CubemodelObject *modelData);
	
	bool
	loadModelObject (CubemodelObject *modelData);

	bool
	addModelObject (CubemodelObject *modelData,
			CompString      sFile,
			float           *translate,
			float           *rotate,
			float           rotateSpeed,
			float           *scale,
			float           *color,
			bool            animation,
			float           fps);

	void
	loadingOptionChanged (CompOption	     *opt,
			      CubemodelOptions::Options num);

	void
	optionChanged (CompOption *opt,
		       CubemodelOptions::Options num);

	void
	setLightPosition (GLenum light);

	void
	donePaint ();
	
	void
	preparePaint (int);
	
	void
	cubeClearTargetOutput (float, float);
	
	void
	cubePaintInside (const GLScreenPaintAttrib &,
			 const GLMatrix		   &,
			 CompOutput		   *,
			 int			     );

	bool mDamage;

	int   mHsize;
	float mSideDistance; /* perpendicular distance to side wall from centre */
	float mTopDistance;  /* perpendicular distance to top wall from centre */
	float mRadius;       /* radius on which the hSize points lie */
	float mArcAngle;   	/* 360 degrees / horizontal size */
	float mRatio;        /* screen width to height */

	GLuint mObjDisplayList;

	std::vector <CubemodelObject *> mModels;
	std::vector <CompString >       mModelFilename;

    public:
    
	bool
	drawModelObject (CubemodelObject *data,
			 float           scale);

	bool
	deleteModelObject (CubemodelObject *data);

	bool
	nodifyModelObject (CubemodelObject *data,
			   CompOption      *option,
			   int             nOption);

	bool
	updateModelObject (CubemodelObject *data,
			   float           time);

	bool
	drawVBOModel (CubemodelObject *data,
		      float           *vertex,
		      float           *normal);
};

#define CUBEMODEL_SCREEN(s)						       \
    CubemodelScreen *cms = CubemodelScreen::get (s)

fileParser *
initFileParser (FILE *fp,
                int bufferSize);

void
updateFileParser (fileParser *fParser,
                  FILE *fp);

void
freeFileParser (fileParser *);

char *
getLine (fileParser *);

char *
getLineToken (fileParser *);

char *
getLineToken2 (fileParser *, bool);

void
skipLine (fileParser *);

char *
strsep2 (char **strPtr, const char *delim);

class CubemodelPluginVTable :
    public CompPlugin::VTableForScreen <CubemodelScreen>
{
    public:
	bool init ();
};

#endif
