/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is Skrooge plugin for operation management.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgoperationpluginwidget.h"
#include "skgsortfilterproxymodel.h"
#include "skgmainpanel.h"
#include "skgobjectmodel.h"
#include "skgobjectbase.h"
#include "skgbankincludes.h"
#include "skgtraces.h"
#include "skgservices.h"
#include "skgsplittabledelegate.h"

#include <kstandarddirs.h>
#include <kstandardaction.h>
#include <kmessagebox.h>
#include <kinputdialog.h>
#include <skgcalculatoredit.h>

#include <QtGui/QHeaderView>
#include <QDomDocument>

/**
 * @def NOUPDATE
 * For a not modified field
 */
#define NOUPDATE "-------"

SKGOperationPluginWidget::SKGOperationPluginWidget(SKGMainPanel* iParent, SKGDocumentBank* iDocument, KAction* iFastEditionAction)
                : SKGTabWidget(iParent, iDocument), fastEditionAction(iFastEditionAction), lastFastEditionOperationFound(0), showClosedAccounts(false),
                numberFieldIsNotUptodate(true),modeInfoZone(0)
{
        SKGTRACEIN(1, "SKGOperationPluginWidget::SKGOperationPluginWidget");

        ui.setupUi(this);

        ui.kTitle->hide();
        ui.kresetInternalFiler->hide();
        ui.kReconciliatorFrame2->hide();

        {
                SKGTRACEIN(10, "SKGOperationPluginWidget::SKGOperationPluginWidget-model creation");
                //Bind operation view
                objectModel = new SKGObjectModel((SKGDocumentBank*) getDocument(), "v_operation_display", "1=0", this);
                SKGSortFilterProxyModel* modelproxy = new SKGSortFilterProxyModel(this);
                modelproxy->setSourceModel(objectModel);
                modelproxy->setSortRole(Qt::UserRole);
                modelproxy->setDynamicSortFilter(true);

                ui.kOperationView->setModel(modelproxy);
                ui.kOperationView->setWindowTitle(i18n("Operations"));
                ui.kOperationView->setDefaultSaveParameters(getDocument(), "SKG_DEFAULT_OPERATION");

                //Add registered global action in contextual menu
                ui.kOperationView->insertAction(0, iParent->getGlobalAction("edit_delete"));
                QAction* sep=new QAction(this);
                sep->setSeparator(true);
                ui.kOperationView->insertAction(0, sep);
                ui.kOperationView->insertAction(0, iParent->getGlobalAction("edit_duplicate_operation"));
                ui.kOperationView->insertAction(0, iParent->getGlobalAction("edit_switch_bookmark"));
                ui.kOperationView->insertAction(0, iParent->getGlobalAction("edit_split_operation"));
                ui.kOperationView->insertAction(0, iParent->getGlobalAction("edit_point_selected_operation"));
                ui.kOperationView->insertAction(0, iParent->getGlobalAction("schedule_operation"));
                ui.kOperationView->insertAction(0, sep);
                ui.kOperationView->insertAction(0, iParent->getGlobalAction("open_report"));

                connect(ui.kOperationView->selectionModel(), SIGNAL(selectionChanged(QItemSelection,QItemSelection)),
                        this, SLOT(onSelectionChanged()));

                connect(ui.kSplitOperation,SIGNAL(clicked()),this,SLOT(onSplitOperationClicked()));

                connect(objectModel, SIGNAL(modelAboutToBeReset()), ui.kOperationView, SLOT(saveSelection()));
                connect(objectModel, SIGNAL(modelReset()), ui.kOperationView, SLOT(resetSelection()));
        }

        //Add completions
        KCompletion *comp = ui.kTypeEdit->completionObject();
        connect(ui.kTypeEdit,SIGNAL(returnPressed(const QString&)),comp,SLOT(addItem(const QString&)));

        //Set view parameters
        ui.kOperationView->sortByColumn(0, Qt::AscendingOrder);

        //Add Standard KDE Icons to buttons to Operations
        ui.kModifyOperationBtn->setIcon(KIcon("dialog-ok-apply"));
        ui.kAddOperationBtn->setIcon(KIcon("list-add"));
        ui.kresetInternalFiler->setIcon(KIcon("edit-delete"));
        ui.kSplitOperation->setIcon(KIcon("skrooge_split"));
        ui.kRemoveSubOperation->setIcon(KIcon("edit-delete"));
        ui.kReconciliatorButton->setIcon(KIcon("object-rotate-left"));

        ui.kOperationTabPages->setTabIcon(0,KIcon("dialog-ok-apply"));
        ui.kOperationTabPages->setTabIcon(1,KIcon("skrooge_transfer"));
        ui.kOperationTabPages->setTabIcon(2,KIcon("skrooge_split"));
        ui.kOperationTabPages->setTabIcon(3,KIcon("skrooge_unit"));

        //Refresh
        connect((const QObject*) getDocument(), SIGNAL(transactionSuccessfullyEnded(int)), this, SLOT(refresh()), Qt::QueuedConnection);
        connect((const QObject*) getDocument(), SIGNAL(transactionSuccessfullyEnded(int)), this, SLOT(onFilterChanged()), Qt::QueuedConnection);
        connect((const QObject*) getDocument(), SIGNAL(transactionSuccessfullyEnded(int)), this, SLOT(onRefreshInformationZone()), Qt::QueuedConnection);
        connect((const QObject*) getDocument(), SIGNAL(transactionSuccessfullyEnded(int)), this, SLOT(onSelectionChanged()), Qt::QueuedConnection);
        connect(ui.kHideUselessOperation, SIGNAL(stateChanged(int)), this, SLOT(onFilterChanged()), Qt::QueuedConnection);

        //Fast edition
        connect(qApp, SIGNAL(focusChanged(QWidget*, QWidget*)), this, SLOT(onFocusChanged()));
        connect(fastEditionAction, SIGNAL(triggered(bool)), this, SLOT(onFastEdition()));

        // SubOperations
        connect(ui.kAmountEdit,SIGNAL(textChanged(const QString&)),this,SLOT(onQuantityChanged()));
        connect(ui.kSubOperationsTable,SIGNAL(cellChanged(int,int)),this,SLOT(onSubopCellChanged(int,int)));
        connect(ui.kRemoveSubOperation,SIGNAL(clicked(bool)),this,SLOT(onRemoveSubOperation()));
        ui.kSubOperationsTable->verticalHeader()->setDefaultSectionSize(ui.kSubOperationsTable->verticalHeader()->minimumSectionSize());
        ui.kSubOperationsTable->horizontalHeader()->setResizeMode(QHeaderView::Interactive);
        ui.kSubOperationsTable->setWordWrap(false);
        ui.kSubOperationsTable->setItemDelegate(new SKGSplitTableDelegate(ui.kSubOperationsTable, getDocument()));


        standardPage = ui.kOperationTabPages->widget(0);
        splittedPage = ui.kOperationTabPages->widget(2);
        ui.kOperationTabPages->removeTab(2);

        displayStandard();

        refresh();
        onOperationCreatorModified();
}

SKGOperationPluginWidget::~SKGOperationPluginWidget()
{
        SKGTRACEIN(1, "SKGOperationPluginWidget::~SKGOperationPluginWidget");
        objectModel=NULL;
}

void SKGOperationPluginWidget::onSelectionChanged()
{
        SKGTRACEIN(10, "SKGOperationPluginWidget::onSelectionChanged");

        //Mapping
        QItemSelectionModel *selModel=ui.kOperationView->selectionModel();
        QModelIndexList indexes=selModel->selectedRows();
        int nbSelect=indexes.count();
        bool onConsolidatedTable=false;
        if (nbSelect) {
                QModelIndex idx=indexes[indexes.count()-1];

                QSortFilterProxyModel* proxyModel=(QSortFilterProxyModel*) ui.kOperationView->model();
                QModelIndex idxs=proxyModel->mapToSource(idx);

                SKGOperationObject obj(objectModel->getObject(idxs));
                onConsolidatedTable=(obj.getTable()=="v_operation_consolidated");

                ui.kDateEdit->setDate(SKGServices::stringToTime(obj.getAttribute("d_date")).date());
                ui.kCommentEdit->setText(obj.getAttribute(obj.getTable()=="v_operation_consolidated" ? "t_REALCOMMENT" : "t_comment"));
                QString number=obj.getAttribute("i_number");
                if (number=="0") number="";
                ui.kNumberEdit->setText(number);
                ui.kAccountEdit->setText(obj.getAttribute("t_ACCOUNT"));
                ui.kPayeeEdit->setText(obj.getAttribute("t_payee"));
                ui.kTypeEdit->setText(obj.getAttribute("t_mode"));
                ui.kUnitEdit->setText(obj.getAttribute("t_UNIT"));
                QString cat=obj.getAttribute("t_REALCATEGORY");
                if (cat.isEmpty()) cat=obj.getAttribute("t_CATEGORY");
                ui.kCategoryEdit->setText(cat);
                QString quantity=obj.getAttribute("f_REALQUANTITY");
                if (quantity.isEmpty()) quantity=obj.getAttribute("f_QUANTITY");
                if (SKGServices::stringToDouble(quantity)>0) quantity="+"+quantity;
                ui.kAmountEdit->setText(quantity);

                if (nbSelect>1) {
                        displayStandard();
                        ui.kAccountEdit->setText(NOUPDATE);
                        ui.kTypeEdit->setText(NOUPDATE);
                        ui.kUnitEdit->setText(NOUPDATE);
                        ui.kCategoryEdit->setText(NOUPDATE);
                        ui.kCommentEdit->setText(NOUPDATE);
                        ui.kPayeeEdit->setText(NOUPDATE);
                } else {
                        int nbSubOperations = obj.getNbSubOperations();
                        if (nbSubOperations > 1 && !onConsolidatedTable) {
                                displaySplitted();

                                displaySubOperations();
                        } else {
                                displayStandard();
                        }
                }
        } else {
                displayStandard();
                ui.kDateEdit->setDate(QDate::currentDate());
                ui.kPayeeEdit->setText("");
                ui.kCategoryEdit->setText("");
                ui.kAmountEdit->setText("");
                ui.kTypeEdit->setText("");
        }

        ui.kDateEdit->setEnabled(nbSelect<=1);
        ui.kAmountEdit->setEnabled(nbSelect<=1);
        ui.kNumberEdit->setEnabled(nbSelect<=1);
        ui.kSplitOperation->setEnabled(nbSelect<=1 && !onConsolidatedTable);

        onOperationCreatorModified();

        emit selectionChanged();
}

void SKGOperationPluginWidget::onOperationCreatorModified()
{
        SKGTRACEIN(10, "SKGOperationPluginWidget::onOperationCreatorModified");

        int currentPage=ui.kOperationTabPages->currentIndex();
        int indexSharePage=2;

        //Is it an existing unit ?
        QString unitName=ui.kUnitEdit->currentText();
        SKGUnitObject unit(getDocument());
        unit.setName(unitName);
        unit.setSymbol(unitName);
        if (unit.load().isSucceeded()) {
                //BUG 2299524: ui.kOperationTabPages->setTabEnabled(3, unit.getType()==SKGUnitObject::SHARE);
                ui.kOperationTabPages->setTabEnabled(indexSharePage, true);

                if (currentPage==indexSharePage && unit.getType()==SKGUnitObject::SHARE) {
                        //Update units
                        QString unitOfUnitName=((SKGDocumentBank*) getDocument())->getPrimaryUnit();
                        SKGUnitObject unitOfUnit;
                        unit.getUnit(unitOfUnit);
                        if (unitOfUnit.exist()) unitOfUnitName=unitOfUnit.getSymbol();

                        ui.kUnitShare->setText(unitOfUnitName);
                        ui.kUnitCommission->setText(unitOfUnitName);
                        ui.kUnitTax->setText(unitOfUnitName);

                        //Update total in "purchase / sale share" page
                        double total=ui.kAmountSharesEdit->value()+(ui.kCommissionEdit->value()+ui.kTaxEdit->value())*(ui.kAmountEdit->value()>0 ? 1 : -1);
                        ui.KTotal->setText(KGlobal::locale()->formatMoney (total, unitOfUnitName,2));
                } else {
                        //BUG 2692665
                        ui.kUnitShare->setText(unitName);
                        ui.kUnitCommission->setText(unitName);
                        ui.kUnitTax->setText(unitName);
                }

        } else ui.kOperationTabPages->setTabEnabled(indexSharePage, false);

        bool activated=ui.kAccountEdit->currentText().length()>0 &&
                       ui.kAmountEdit->text().length()>0 &&
                       unitName.length()>0 &&
                       (currentPage!=indexSharePage || ui.kAmountSharesEdit->text().length()>0);

        int nbSelect=getNbSelectedObjects();

        ui.kAddOperationBtn->setEnabled(activated);
        ui.kModifyOperationBtn->setEnabled(activated && nbSelect>0 && currentPage==0);

        QList<QTableWidgetItem*> selectedItems = ui.kSubOperationsTable->selectedItems();
        ui.kRemoveSubOperation->setEnabled(selectedItems.count()>0);
}

void SKGOperationPluginWidget::onUpdateOperationClicked()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGOperationPluginWidget::onUpdateOperationClicked",err);
        bool confirmed=true;
        //Get Selection
        SKGObjectBase::SKGListSKGObjectBase selection=getSelectedObjects();

        int nb=selection.count();
        {
                SKGBEGINPROGRESSTRANSACTION(*getDocument(), i18n("Operation update"), err, nb);

                QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
                err=updateSelection(selection);
                QApplication::restoreOverrideCursor();
        }

        //status bar
        if (err.isSucceeded()) {
                if (confirmed) err=SKGError(0, i18n("Operation updated."));
                else err=SKGError(0, i18n("Operation canceled"));
        } else err.addError(ERR_FAIL, i18n("Operation update failed"));

        //Display error
        getMainPanel()->displayErrorMessage(err);
}

SKGError SKGOperationPluginWidget::updateSelection(const SKGObjectBase::SKGListSKGObjectBase& iSelection)
{
        SKGError err;
        SKGTRACEINRC(10, "SKGOperationPluginWidget::updateSelection",err);

        //Get Selection
        QWidget* currentPageWidget=ui.kOperationTabPages->currentWidget();
        SKGObjectBase::SKGListSKGObjectBase selection=getSelectedObjects();
        int nb=iSelection.count();

        for (int i=0; err.isSucceeded() && i<nb; ++i) {
                SKGObjectBase obj=iSelection.at(i);
                SKGOperationObject operationObj=SKGOperationObject(obj.getDocument(), obj.getID());

                //Update operation if single selection
                if (currentPageWidget==standardPage) {
                        //Get subop
                        SKGSubOperationObject subOp;
                        int nbSubop=0;
                        if (obj.getTable()=="v_operation_consolidated") {
                                //It's a sub operation
                                subOp=SKGSubOperationObject(obj.getDocument(), SKGServices::stringToInt(obj.getAttribute("i_SUBOPID")));
                                nbSubop=1;
                        } else {
                                //It's a real operation, we take the first one
                                SKGObjectBase::SKGListSKGObjectBase subOps;
                                if (err.isSucceeded()) err=operationObj.getSubOperations(subOps);
                                nbSubop=subOps.count();
                                if (nbSubop) subOp=subOps[0];
                        }

                        SKGCategoryObject cat;
                        if (err.isSucceeded() && ui.kCategoryEdit->text()!=NOUPDATE) {
                                if (err.isSucceeded() && nbSubop>1) {
                                        err=SKGError(25, i18n("Cannot update a splitted operation"));
                                } else {
                                        err=SKGCategoryObject::createPathCategory((SKGDocumentBank*) getDocument(), ui.kCategoryEdit->text(), cat, true);
                                        if (err.isSucceeded()) err=subOp.setCategory(cat);
                                }
                        } else {
                                //Get current category to be able to
                                subOp.getCategory(cat);
                        }

                        if (err.isSucceeded() && nb==1) {
                                if (currentPageWidget==standardPage && nbSubop>1) {
                                        err=SKGError(25, i18n("Cannot update a splitted operation"));
                                } else {
                                        double val=ui.kAmountEdit->value();

                                        //Is the sign forced ?
                                        if (ui.kAmountEdit->sign()==0) {
                                                SKGObjectBase cat2(cat.getDocument(), "v_category_display", cat.getID());

                                                //Are we able to find to sign with the category ?
                                                if (cat2.getAttribute("t_TYPEEXPENSE")==tr("Expense")) val=-val;
                                        }
                                        err=subOp.setQuantity(val);
                                }
                                if (err.isSucceeded()) err=operationObj.setNumber(SKGServices::stringToInt(ui.kNumberEdit->text()));
                                if (err.isSucceeded()) err=operationObj.setDate(ui.kDateEdit->date());
                        }

                        if (err.isSucceeded() && ui.kCommentEdit->text()!=NOUPDATE) {
                                if (obj.getTable()=="v_operation_consolidated") err=subOp.setComment(ui.kCommentEdit->text());
                        }
                        if (err.isSucceeded()) err=subOp.save();
                } else {
                        int nbsubop=ui.kSubOperationsTable->rowCount();
                        QList<int> listIdSubOp;
                        for (int j=0; err.isSucceeded() && j<nbsubop; ++j) {
                                //Get values
                                QTableWidgetItem* item=ui.kSubOperationsTable->item(j,0);
                                int id=item->data(Qt::UserRole).toInt();
                                QString catName=item->text();
                                item=ui.kSubOperationsTable->item(j,1);
                                QString comment=item->text();
                                item=ui.kSubOperationsTable->item(j,2);
                                double val=SKGServices::stringToDouble(item->text());

                                SKGSubOperationObject subOperation;
                                if (id) {
                                        //Update existing sub op
                                        subOperation=SKGSubOperationObject((SKGDocumentBank*) getDocument(), id);
                                } else {
                                        //Create new sub op
                                        err=operationObj.addSubOperation(subOperation);
                                }

                                //Create sub operation object
                                SKGCategoryObject cat;
                                if (err.isSucceeded() && !catName.isEmpty()) {
                                        err=SKGCategoryObject::createPathCategory((SKGDocumentBank*) getDocument(), catName, cat, true);
                                        if (err.isSucceeded()) err=subOperation.setCategory(cat);
                                }
                                if (err.isSucceeded()) err=subOperation.setComment(comment);
                                if (err.isSucceeded()) err=subOperation.setQuantity(val);
                                if (err.isSucceeded()) err=subOperation.save();

                                //The sub operation created or updated mustn't be removed
                                listIdSubOp.push_back(subOperation.getID());
                        }

                        //Remove useless subop
                        if (err.isSucceeded()) {
                                SKGObjectBase::SKGListSKGObjectBase subOps;
                                err=operationObj.getSubOperations(subOps);
                                int nbsubop=subOps.count();
                                for (int j=0; err.isSucceeded() && j<nbsubop; ++j) {
                                        SKGObjectBase sop=subOps.at(j);
                                        if (!listIdSubOp.contains(sop.getID())) {
                                                err=sop.remove();
                                        }
                                }
                        }
                }

                if (err.isSucceeded() && ui.kCommentEdit->text()!=NOUPDATE) {
                        if (obj.getTable()!="v_operation_consolidated")  err=operationObj.setComment(ui.kCommentEdit->text());
                }

                if (err.isSucceeded() && ui.kAccountEdit->text()!=NOUPDATE) {
                        SKGAccountObject account(getDocument());
                        err=account.setName(ui.kAccountEdit->text());
                        if (err.isSucceeded()) err=account.load();
                        if (err.isSucceeded()) err=operationObj.setParentAccount(account);
                }
                if (err.isSucceeded() && ui.kTypeEdit->text()!=NOUPDATE) err=operationObj.setMode(ui.kTypeEdit->text());
                if (err.isSucceeded() && ui.kPayeeEdit->text()!=NOUPDATE) err=operationObj.setPayee(ui.kPayeeEdit->text());
                if (err.isSucceeded() && ui.kUnitEdit->text()!=NOUPDATE) {
                        SKGUnitObject unit(getDocument());
                        err=unit.setSymbol(ui.kUnitEdit->text());
                        if (!unit.exist()) {
                                if (err.isSucceeded()) err= unit.setName(ui.kUnitEdit->text());
                                if (err.isSucceeded()) err = unit.save();

                                SKGUnitValueObject unitVal;
                                if (err.isSucceeded()) err=unit.addUnitValue(unitVal);
                                if (err.isSucceeded()) err=unitVal.setDate(QDate::currentDate());
                                if (err.isSucceeded()) err=unitVal.setQuantity(1);
                                if (err.isSucceeded()) err=unitVal.save();

                                if (err.isSucceeded()) getDocument()->sendMessage(tr("Unit [%1] has been created").arg(ui.kUnitEdit->text()), true);
                        } else err=unit.load();

                        if (err.isSucceeded()) err=operationObj.setUnit(unit);
                }

                //Save
                if (err.isSucceeded()) err=operationObj.save();

                if (err.isSucceeded()) err=getDocument()->stepForward(i+1);
        }

        return err;
}

void SKGOperationPluginWidget::onAddOperationClicked()
{
        SKGError err;
        SKGTRACEIN(10, "SKGOperationPluginWidget::onAddOperationClicked");

        //Get parameters
        int currentPage=ui.kOperationTabPages->currentIndex();
        QWidget* currentPageWidget=ui.kOperationTabPages->currentWidget();
        QString accountName=ui.kAccountEdit->currentText();
        QString catName=ui.kCategoryEdit->currentText();

        bool confirmed=true;
        SKGOperationObject operation;
        {
                SKGBEGINTRANSACTION(*getDocument(), i18n("Operation creation"), err);

                QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

                //Get account
                SKGAccountObject accountObj(getDocument());
                if (err.isSucceeded()) err=accountObj.setName(accountName);
                if (err.isSucceeded()) err=accountObj.load();

                //Create operation object
                if (err.isSucceeded()) err=accountObj.addOperation(operation);
                if (err.isSucceeded()) err=operation.setMode(ui.kTypeEdit->currentText());
                if (err.isSucceeded()) err=operation.setPayee(ui.kPayeeEdit->currentText());
                if (err.isSucceeded()) err=operation.setNumber(SKGServices::stringToInt(ui.kNumberEdit->text()));
                if (err.isSucceeded()) err=operation.setComment(ui.kCommentEdit->text());
                if (err.isSucceeded()) err=operation.setDate(ui.kDateEdit->date());

                SKGUnitObject unit(getDocument());
                err=unit.setSymbol(ui.kUnitEdit->text());
                if (!unit.exist()) {
                        if (err.isSucceeded()) err= unit.setName(ui.kUnitEdit->text());
                        if (err.isSucceeded()) err = unit.save();

                        SKGUnitValueObject unitVal;
                        if (err.isSucceeded()) err=unit.addUnitValue(unitVal);
                        if (err.isSucceeded()) err=unitVal.setDate(QDate::currentDate());
                        if (err.isSucceeded()) err=unitVal.setQuantity(1);
                        if (err.isSucceeded()) err=unitVal.save();

                        if (err.isSucceeded()) getDocument()->sendMessage(tr("Unit [%1] has been created").arg(ui.kUnitEdit->text()), true);
                } else err=unit.load();

                if (err.isSucceeded()) err=operation.setUnit(unit);
                if (err.isSucceeded()) err=operation.save();

                if (currentPage==0) {
                        //STD OPERATION (SPLITTED OR NOT)
                        if (currentPageWidget==standardPage) {
                                //Create sub operation object
                                SKGSubOperationObject subOperation;
                                if (err.isSucceeded()) err=operation.addSubOperation(subOperation);
                                SKGCategoryObject cat;
                                if (err.isSucceeded() && !catName.isEmpty() && currentPage==0) {
                                        err=SKGCategoryObject::createPathCategory((SKGDocumentBank*) getDocument(), catName, cat, true);
                                        if (err.isSucceeded()) err=subOperation.setCategory(cat);
                                }
                                double val=ui.kAmountEdit->value();
                                if (err.isSucceeded()) {
                                        if (cat.exist()) {
                                                //Is the sign forced ?
                                                if (ui.kAmountEdit->sign()==0) {
                                                        SKGObjectBase cat2(cat.getDocument(), "v_category_display", cat.getID());

                                                        //Are we able to find to sign with the category ?
                                                        if (cat2.getAttribute("t_TYPEEXPENSE")==tr("Expense")) val=-val;
                                                }
                                        }
                                        err=subOperation.setQuantity(val);
                                }
                                if (err.isSucceeded()) err=subOperation.save();
                        } else {
                                int nbsubop=ui.kSubOperationsTable->rowCount();
                                for (int i=0; err.isSucceeded() && i<nbsubop; ++i) {
                                        //Get values
                                        QTableWidgetItem* item=ui.kSubOperationsTable->item(i,0);
                                        QString catName=item->text();
                                        item=ui.kSubOperationsTable->item(i,1);
                                        QString comment=item->text();
                                        item=ui.kSubOperationsTable->item(i,2);
                                        double val=SKGServices::stringToDouble(item->text());

                                        //Create sub operation object
                                        SKGSubOperationObject subOperation;
                                        if (err.isSucceeded()) err=operation.addSubOperation(subOperation);
                                        SKGCategoryObject cat;
                                        if (err.isSucceeded() && !catName.isEmpty() && currentPage==0) {
                                                err=SKGCategoryObject::createPathCategory((SKGDocumentBank*) getDocument(), catName, cat, true);
                                                if (err.isSucceeded()) err=subOperation.setCategory(cat);
                                        }
                                        if (err.isSucceeded()) err=subOperation.setComment(comment);
                                        if (err.isSucceeded()) err=subOperation.setQuantity(val);
                                        if (err.isSucceeded()) err=subOperation.save();
                                }
                        }
                } else if (currentPage==1) {
                        //TRANSFER
                        //Create sub operation object
                        SKGSubOperationObject subOperation;
                        if (err.isSucceeded()) err=operation.addSubOperation(subOperation);
                        if (err.isSucceeded()) err=subOperation.setQuantity(-ui.kAmountEdit->value());
                        if (err.isSucceeded()) err=subOperation.save();

                        //Get account
                        SKGAccountObject accountObj2(getDocument());
                        if (err.isSucceeded()) err=accountObj2.setName(ui.kTargetAccountEdit->currentText());
                        if (err.isSucceeded()) err=accountObj2.load();

                        //Check unit of target account
                        //Correction bug 2299303 vvv
                        SKGUnitObject unitTargetAccount;
                        double operationQuantity=ui.kAmountEdit->value();
                        if (err.isSucceeded()) err=accountObj2.getUnit(unitTargetAccount);
                        if (err.isSucceeded() && unitTargetAccount.exist() && unit!=unitTargetAccount) {
                                //The unit of the operation is not compliant with the unit of the target account
                                //We ask to the user if he wants to continue or convert into the target account
                                bool ok=false;
                                QApplication::setOverrideCursor(QCursor(Qt::ArrowCursor));
                                double newval=KInputDialog::getDouble  (i18n("Confirmation"),
                                                                        i18n("Operation unit is not compatible with target account.\n"
                                                                             "Click Cancel if you want to continue anyway, "
                                                                             "else enter the value in account target unit(%1) :", unitTargetAccount.getSymbol()),
                                                                        SKGUnitObject::convert(operationQuantity, unit, unitTargetAccount),
                                                                        -DBL_MAX, DBL_MAX, 2, &ok, this);
                                QApplication::restoreOverrideCursor();
                                if (ok) {
                                        operationQuantity=newval;
                                        unit=unitTargetAccount;
                                }

                        }
                        //Correction bug 2299303 ^^^

                        //create transferred operation
                        SKGOperationObject operation2;
                        if (err.isSucceeded()) err=accountObj2.addOperation(operation2);
                        if (err.isSucceeded()) err=operation2.setMode(ui.kTypeEdit->currentText());
                        if (err.isSucceeded()) err=operation2.setPayee(ui.kPayeeEdit->currentText());
                        if (err.isSucceeded()) err=operation2.setNumber(SKGServices::stringToInt(ui.kNumberEdit->text()));
                        if (err.isSucceeded()) err=operation2.setComment(ui.kCommentEdit->text());
                        if (err.isSucceeded()) err=operation2.setDate(ui.kDateEdit->date());
                        if (err.isSucceeded()) err=operation2.setUnit(unit);
                        if (err.isSucceeded()) err=operation2.setGroupOperation(operation);
                        if (err.isSucceeded()) err=operation2.save();

                        //Create sub operation object
                        SKGSubOperationObject subOperation2;
                        if (err.isSucceeded()) err=operation2.addSubOperation(subOperation2);
                        if (err.isSucceeded()) err=subOperation2.setQuantity(operationQuantity);
                        if (err.isSucceeded()) err=subOperation2.save();
                } else if (currentPage==2) {
                        //PURCHASE OR SALE SHARE
                        //Create sub operation object
                        SKGSubOperationObject subOperation;
                        double val=ui.kAmountEdit->value();
                        if (err.isSucceeded()) err=operation.addSubOperation(subOperation);
                        if (err.isSucceeded()) err=subOperation.setQuantity(val);
                        if (err.isSucceeded()) err=subOperation.save();

                        if (err.isSucceeded() && val>0) err=operation.setProperty("SKG_OP_ORIGINAL_AMOUNT", SKGServices::doubleToString(ui.kAmountSharesEdit->value()));
                        if (err.isSucceeded()) err=operation.save();

                        //Get account
                        SKGAccountObject accountObj2(getDocument());
                        if (err.isSucceeded()) err=accountObj2.setName(ui.kPaymentAccountEdit->currentText());
                        if (err.isSucceeded()) err=accountObj2.load();

                        //create paiement operation for shares
                        SKGOperationObject operation2;
                        if (err.isSucceeded()) err=accountObj2.addOperation(operation2);
                        if (err.isSucceeded()) err=operation2.setMode(ui.kTypeEdit->currentText());
                        if (err.isSucceeded()) err=operation2.setPayee(ui.kPayeeEdit->currentText());
                        if (err.isSucceeded()) err=operation2.setNumber(SKGServices::stringToInt(ui.kNumberEdit->text()));
                        if (err.isSucceeded()) err=operation2.setComment(ui.kCommentEdit->text());
                        if (err.isSucceeded()) err=operation2.setDate(ui.kDateEdit->date());
                        if (err.isSucceeded()) {
                                SKGUnitObject unitOfUnit;
                                err=unit.getUnit(unitOfUnit);
                                if (err.isSucceeded()) err=operation2.setUnit(unitOfUnit);
                        }
                        if (err.isSucceeded()) err=operation2.setGroupOperation(operation);
                        if (err.isSucceeded()) err=operation2.save();

                        //Create main sub operation
                        SKGSubOperationObject subOperation2;
                        if (err.isSucceeded()) err=operation2.addSubOperation(subOperation2);
                        if (err.isSucceeded()) err=subOperation2.setComment(i18n("Shares"));
                        if (err.isSucceeded()) err=subOperation2.setQuantity(ui.kAmountSharesEdit->value()*(val>0 ? -1 : 1));
                        if (err.isSucceeded()) err=subOperation2.save();

                        //Create commission sub operation
                        if (ui.kAmountSharesEdit->value()) {
                                SKGSubOperationObject subOperation2;
                                if (err.isSucceeded()) err=operation2.addSubOperation(subOperation2);
                                if (err.isSucceeded()) err=subOperation2.setComment(i18n("Commission"));
                                if (err.isSucceeded()) err=subOperation2.setQuantity(-ui.kCommissionEdit->value());
                                if (err.isSucceeded()) err=subOperation2.save();
                        }

                        //Create tax sub operation
                        if (ui.kTaxEdit->value()) {
                                SKGSubOperationObject subOperation2;
                                if (err.isSucceeded()) err=operation2.addSubOperation(subOperation2);
                                if (err.isSucceeded()) err=subOperation2.setComment(i18n("Tax"));
                                if (err.isSucceeded()) err=subOperation2.setQuantity(-ui.kTaxEdit->value());
                                if (err.isSucceeded()) err=subOperation2.save();
                        }
                }

                QApplication::restoreOverrideCursor();
        }

        //status bar
        if (err.isSucceeded()) {
                if (confirmed) {
                        err=SKGError(0, i18n("Operation created"));
                        ui.kOperationView->selectObject(operation.getUniqueID());
                } else err=SKGError(0, i18n("Operation canceled"));
        } else err.addError(ERR_FAIL, i18n("Operation creation failed"));

        //Display error
        getMainPanel()->displayErrorMessage(err);
}

SKGObjectBase::SKGListSKGObjectBase SKGOperationPluginWidget::getSelectedObjects()
{
        //Get Selection
        SKGObjectBase::SKGListSKGObjectBase selection;
        QItemSelectionModel *selModel=ui.kOperationView->selectionModel();
        QSortFilterProxyModel* proxyModel=(QSortFilterProxyModel*) ui.kOperationView->model();
        SKGObjectModel* model=(SKGObjectModel*) proxyModel->sourceModel();
        if (model) {
                QModelIndexList indexes=selModel->selectedRows();
                foreach(const QModelIndex& index, indexes) {
                        SKGObjectBase obj=model->getObject(proxyModel->mapToSource(index));
                        selection.push_back(obj);
                }
        }
        return selection;
}

int SKGOperationPluginWidget::getNbSelectedObjects()
{
        QItemSelectionModel *selModel=ui.kOperationView->selectionModel();
        return (selModel ? selModel->selectedRows().count() : 0);
}

QString SKGOperationPluginWidget::getState()
{
        SKGTRACEIN(10, "SKGOperationPluginWidget::getState");
        QDomDocument doc("SKGML");
        QDomElement root;
        if (lastState.hasChildNodes()) {
                doc=lastState;
                root = doc.documentElement();
        } else {
                root = doc.createElement("parameters");
                doc.appendChild(root);
        }
        QString account=root.attribute ( "account");
        QString currentPage=root.attribute ( "currentPage");
        QString hideUselessOperation=root.attribute ( "hideUselessOperation");
        QString filter=root.attribute ( "filter");
        QString modeInfoZoneS=root.attribute ( "modeInfoZone");

        if (account.isEmpty()) root.setAttribute("account", ui.kDisplayAccountCombo->currentText());
        if (currentPage.isEmpty()) root.setAttribute("currentPage", ui.kOperationTabPages->currentIndex());
        if (hideUselessOperation.isEmpty()) root.setAttribute("hideUselessOperation", ui.kHideUselessOperation->checkState()==Qt::Checked ? "Y" : "N");
        if (filter.isEmpty()) root.setAttribute("filter", ui.kFilterEdit->text());
        if (modeInfoZoneS.isEmpty()) root.setAttribute("modeInfoZone", SKGServices::intToString(modeInfoZone));

        //Memorize table settings
        root.setAttribute("view", ui.kOperationView->getState());

        return doc.toString();
}

void SKGOperationPluginWidget::setState(const QString& iState )
{
        SKGTRACEIN(10, "SKGOperationPluginWidget::setState");
        QDomDocument doc("SKGML");
        doc.setContent(iState);
        QDomElement root = doc.documentElement();

        QString account=root.attribute ( "account");
        QString currentPage=root.attribute ( "currentPage");
        QString hideUselessOperation=root.attribute ( "hideUselessOperation");
        QString filter=root.attribute ( "filter");
        QString title=root.attribute ( "title");
        QString title_icon=root.attribute ( "title_icon");
        QString modeInfoZoneS=root.attribute ( "modeInfoZone");

        if (!account.isEmpty()) {
                SKGAccountObject acc;
                SKGNamedObject::getObjectByName(getDocument(), "v_account", account, acc);
                if (acc.isClosed() && showClosedAccounts==false) {
                        showClosedAccounts=true;
                        refresh();
                }
                ui.kDisplayAccountCombo->setText(account);
        }
        if (!modeInfoZoneS.isEmpty()) {
                modeInfoZone=SKGServices::stringToInt(modeInfoZoneS)-1;
                onRefreshInformationZone();
        }
        if (!currentPage.isEmpty()) ui.kOperationTabPages->setCurrentIndex(SKGServices::stringToInt(currentPage));
        if (!hideUselessOperation.isEmpty()) ui.kHideUselessOperation->setCheckState (hideUselessOperation=="Y" ? Qt::Checked : Qt::Unchecked);
        if (!filter.isEmpty()) ui.kFilterEdit->setText(filter);
        if (!title.isEmpty()) {
                //ui.kTitle->setText(title);
                /*QLabel* lab=new QLabel(this);
                lab->setWordWrap(true);
                lab->setText(title);
                ui.kTitle->setWidget(lab);*/
                ui.kTitle->setText(title);

                ui.kTitle->show();
        } else {
                ui.kTitle->hide();
        }
        if (!title_icon.isEmpty()) ui.kTitle->setPixmap(KIcon(title_icon).pixmap(22, 22), KTitleWidget::ImageLeft);

        QSortFilterProxyModel* modelproxy = (QSortFilterProxyModel*) ui.kOperationView->model();
        SKGObjectModel* objectModel=(SKGObjectModel*) modelproxy->sourceModel ();

        QString operationTable=root.attribute ( "operationTable");
        operationWhereClause=root.attribute ( "operationWhereClause");
        if (!operationTable.isEmpty())  objectModel->setTable(operationTable);
        if (!operationWhereClause.isEmpty())  objectModel->setFilter(operationWhereClause);
        if (!operationTable.isEmpty() || !operationWhereClause.isEmpty()) {
                //We keep a copy of given state in case of bookmark
                lastState=doc;
        }
        if (operationTable=="v_operation_consolidated") ui.kOperationView->setDefaultSaveParameters(getDocument(), "SKG_DEFAULT_OPERATION_CONSOLIDATED");
        ui.kOperationView->setState(root.attribute("view"));

        onFilterChanged();
        onRefreshInformationZone();
}

void SKGOperationPluginWidget::refresh()

{
        SKGTRACEIN(1, "SKGOperationPluginWidget::refresh");

        //Refresh account summary
        QSqlDatabase* db = getDocument()->getDatabase();
        setEnabled(db!=NULL);
        if (db!=NULL) {

                //Correction bug 2299394 vvv
                if (ui.kOperationView->isAutoResized())ui.kOperationView->resizeColumnsToContents();
                //Correction bug 2299394 ^^^

                //Disconnect combo filter account
                disconnect(ui.kDisplayAccountCombo, SIGNAL(currentIndexChanged(int)), this, SLOT(onFilterChanged()));
                disconnect(ui.kDisplayAccountCombo, SIGNAL(currentIndexChanged(int)), this, SLOT(onRefreshInformationZone()));
                disconnect(ui.kDisplayAccountCombo, SIGNAL(textChanged(QString)), this, SLOT(onOperationCreatorModified()));
                disconnect(ui.kUnitEdit, SIGNAL(textChanged(QString)), this, SLOT(onOperationCreatorModified()));
                disconnect(ui.kAmountEdit, SIGNAL(textChanged(QString)), this, SLOT(onOperationCreatorModified()));
                disconnect(ui.kAmountSharesEdit, SIGNAL(textChanged(QString)), this, SLOT(onOperationCreatorModified()));
                disconnect(ui.kCommissionEdit, SIGNAL(textChanged(QString)), this, SLOT(onOperationCreatorModified()));
                disconnect(ui.kTaxEdit, SIGNAL(textChanged(QString)), this, SLOT(onOperationCreatorModified()));

                //Set account combo
                QString current=ui.kDisplayAccountCombo->currentText();
                ui.kDisplayAccountCombo->clear();
                ui.kDisplayAccountCombo->addItem(i18n("All"));

                QStringList listAccount;
                SKGServices::getDistinctValues(getDocument(), "account", "t_name", showClosedAccounts ? "" : "t_close='N'", listAccount);
                ui.kDisplayAccountCombo->addItems(listAccount);

                if (listAccount.count()==0) {
                        ui.kTitle->setText(i18n("First you have to create an account."));
                        ui.kTitle->setPixmap(KIcon("dialog-information").pixmap(22, 22));
                        ui.kTitle->show();
                } else if (!lastState.hasChildNodes()) ui.kTitle->hide();

                ui.kAccountEdit->clear();
                ui.kAccountEdit->addItems(listAccount);

                ui.kPaymentAccountEdit->clear();
                ui.kPaymentAccountEdit->addItems(listAccount);

                ui.kTargetAccountEdit->clear();
                ui.kTargetAccountEdit->addItems(listAccount);

                int pos=ui.kDisplayAccountCombo->findText(current);
                if (pos==-1) pos=0;
                ui.kDisplayAccountCombo->setCurrentIndex(pos);

                //Set type combo
                SKGMainPanel::fillComboWithDistinctValue(ui.kTypeEdit, getDocument(), "operation", "t_mode", "");

                //Set type unit
                SKGMainPanel::fillComboWithDistinctValue(ui.kUnitEdit, getDocument(), "unit", "ifnull(t_symbol,t_name)", "t_type!='I'");
                QString primary=((SKGDocumentBank*) getDocument())->getPrimaryUnit();
                if (!primary.isEmpty()) ui.kUnitEdit->setText(primary);

                //Set type category
                SKGMainPanel::fillComboWithDistinctValue(ui.kCategoryEdit, getDocument(), "category", "t_fullname", "");

                //Set type payee
                SKGMainPanel::fillComboWithDistinctValue(ui.kPayeeEdit, getDocument(), "operation", "t_payee", "");

                //Set type number
                numberFieldIsNotUptodate=true;

                //Connect combo filter account
                connect(ui.kDisplayAccountCombo, SIGNAL(currentIndexChanged(int)), this, SLOT(onFilterChanged()), Qt::QueuedConnection);
                connect(ui.kDisplayAccountCombo, SIGNAL(currentIndexChanged(int)), this, SLOT(onRefreshInformationZone()), Qt::QueuedConnection);
                connect(ui.kDisplayAccountCombo, SIGNAL(textChanged(QString)), this, SLOT(onOperationCreatorModified()), Qt::QueuedConnection);
                connect(ui.kUnitEdit, SIGNAL(textChanged(QString)), this, SLOT(onOperationCreatorModified()), Qt::QueuedConnection);
                connect(ui.kAmountEdit, SIGNAL(textChanged(QString)), this, SLOT(onOperationCreatorModified()), Qt::QueuedConnection);
                connect(ui.kAmountSharesEdit, SIGNAL(textChanged(QString)), this, SLOT(onOperationCreatorModified()), Qt::QueuedConnection);
                connect(ui.kCommissionEdit, SIGNAL(textChanged(QString)), this, SLOT(onOperationCreatorModified()), Qt::QueuedConnection);
                connect(ui.kTaxEdit, SIGNAL(textChanged(QString)), this, SLOT(onOperationCreatorModified()), Qt::QueuedConnection);
        }
}

void SKGOperationPluginWidget::openOperation(const SKGOperationObject& iOperation, SKGMainPanel* iParent)
{
        _SKGTRACEIN(10, "SKGOperationPluginWidget::openOperation");

        //Build where clause and title
        int opid=iOperation.getID();
        QString wc="id="+SKGServices::intToString(opid);

        opid=SKGServices::stringToInt(iOperation.getAttribute("i_group_id"));
        if (opid!=0) wc+=" or i_group_id="+SKGServices::intToString(opid);
        wc='('+wc+')';
        QString title=i18n("Operations grouped or splitted");

        //Open
        if (QApplication::keyboardModifiers()&Qt::ControlModifier && QApplication::keyboardModifiers()&Qt::ShiftModifier) {
                //Call debug plugin
                QDomDocument doc("SKGML");
                QDomElement root = doc.createElement("parameters");
                doc.appendChild(root);
                root.setAttribute("sqlOrder", "SELECT * from v_operation_consolidated WHERE "+wc);

                iParent->setNewTabContent(iParent->getPluginByName("Skrooge debug plugin"), -1, doc.toString());
        } else {
                //Call operation plugin
                QDomDocument doc("SKGML");
                QDomElement root = doc.createElement("parameters");
                doc.appendChild(root);
                root.setAttribute("operationTable", "v_operation_consolidated");
                root.setAttribute("operationWhereClause", wc);
                root.setAttribute("title", title);
                root.setAttribute("title_icon", "view-statistics");


                iParent->setNewTabContent(iParent->getPluginByName("Skrooge operation plugin"), -1, doc.toString());
        }
}

void SKGOperationPluginWidget::onDoubleClick()
{
        _SKGTRACEIN(10, "SKGOperationPluginWidget::onDoubleClick");

        //Get selection
        SKGObjectBase::SKGListSKGObjectBase selection=getSelectedObjects();
        if (selection.count()==1) {
                SKGOperationObject op(selection.at(0));
                openOperation(op, getMainPanel());
        }
}

void SKGOperationPluginWidget::onResetInternalFilter()
{
        SKGTRACEIN(10, "SKGOperationPluginWidget::onResetInternalFilter");
        QSortFilterProxyModel* modelproxy = (QSortFilterProxyModel*) ui.kOperationView->model();
        SKGObjectModel* objectModel=(SKGObjectModel*) modelproxy->sourceModel ();

        operationWhereClause="";
        objectModel->setTable("v_operation_display");

        ui.kTitle->hide();
        lastState.clear();

        onFilterChanged();
}

void SKGOperationPluginWidget::onFilterRegExpChanged()
{
        SKGTRACEIN(10, "SKGOperationPluginWidget::onFilterRegExpChanged");
        QRegExp regExp(ui.kFilterEdit->text(), Qt::CaseInsensitive);
        ((SKGSortFilterProxyModel*) ui.kOperationView->model())->setFilterRegExp(regExp);
}

void SKGOperationPluginWidget::onRefreshInformationZone()
{
        SKGTRACEIN(1, "SKGOperationPluginWidget::onRefreshInformationZone");
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
        if (modeInfoZone==0) {
                //Refresh info area
                //Compute where clause
                QString filter="1=1";
                if (ui.kDisplayAccountCombo->currentIndex()>0) {
                        filter="t_name='"+SKGServices::stringToSqlString(ui.kDisplayAccountCombo->currentText())+'\'';
                }
                SKGStringListList listTmp;
                SKGServices::executeSelectSqliteOrder(getDocument(),
                                                      "SELECT SUM(f_CURRENTAMOUNT), SUM(f_CHECKED), SUM(f_COMING_SOON) from v_account_display WHERE "+filter,
                                                      listTmp);
                if (listTmp.count()==2) {
                        KLocale* locale=KGlobal::locale();

                        QString unit1=((SKGDocumentBank*) getDocument())->getPrimaryUnit();
                        double unit1Value=1;
                        QString unit2=((SKGDocumentBank*) getDocument())->getSecondaryUnit();
                        double unit2Value=((SKGDocumentBank*) getDocument())->getSecondaryUnitValue();

                        if (ui.kDisplayAccountCombo->currentIndex()>0) {
                                SKGUnitObject unitAccount;
                                SKGAccountObject account(getDocument());
                                if (account.setName(ui.kDisplayAccountCombo->currentText()).isSucceeded()) {
                                        if (account.load().isSucceeded()) {
                                                if (account.getUnit(unitAccount).isSucceeded()) {
                                                        if (!unitAccount.getSymbol().isEmpty()) {
                                                                unit1=unitAccount.getSymbol();
                                                                unit1Value=SKGServices::stringToDouble(unitAccount.getAttribute("f_CURRENTAMOUNT"));

                                                                if (unit1!=((SKGDocumentBank*) getDocument())->getPrimaryUnit()) {
                                                                        unit2=((SKGDocumentBank*) getDocument())->getPrimaryUnit();
                                                                        unit2Value=1;
                                                                }
                                                        }
                                                }
                                        }
                                }
                        }

                        double v1=SKGServices::stringToDouble(listTmp.at(1).at(0));
                        double v2=SKGServices::stringToDouble(listTmp.at(1).at(1));
                        double v3=SKGServices::stringToDouble(listTmp.at(1).at(2));
                        QString s1=QString("<font color=\"")+(v1<0 ? "red" : "black")+"\">"+locale->formatMoney (v1/unit1Value, unit1,2)+"</font>";
                        QString s2=QString("<font color=\"")+(v2<0 ? "red" : "black")+"\">"+locale->formatMoney (v2/unit1Value, unit1,2)+"</font>";
                        QString s3=QString("<font color=\"")+(v3<0 ? "red" : "black")+"\">"+locale->formatMoney (v3/unit1Value, unit1,2)+"</font>";
                        ui.kInfo->setText(i18n("Balance: %1     Checked: %2     Foreseen: %3", s1, s2, s3));
                        if (!unit2.isEmpty() && unit2Value) {
                                s1=QString("<font color=\"")+(v1<0 ? "red" : "black")+"\">"+locale->formatMoney (v1/unit2Value, unit2,2)+"</font>";
                                s2=QString("<font color=\"")+(v2<0 ? "red" : "black")+"\">"+locale->formatMoney (v2/unit2Value, unit2,2)+"</font>";
                                s3=QString("<font color=\"")+(v3<0 ? "red" : "black")+"\">"+locale->formatMoney (v3/unit2Value, unit2,2)+"</font>";
                        }
                        ui.kInfo->setToolTip(i18n("<p>Balance: %1</p><p>Checked: %2</p><p>Foreseen: %3</p>", s1, s2, s3));
                }
        } else {
                //Refresh reconciliation area
                //Compute where clause
                QString filter="'"+SKGServices::stringToSqlString(ui.kDisplayAccountCombo->currentText())+'\'';
                SKGStringListList listTmp;
                double diff=0;
                SKGServices::executeSelectSqliteOrder(getDocument(),
                                                      "SELECT SUM(f_CHECKED) from v_account_display WHERE t_name="+filter,
                                                      listTmp);
                if (listTmp.count()==2) {
                        diff=SKGServices::stringToDouble(listTmp.at(1).at(0))-ui.kReconcilitorAmountEdit->value();
                }
                SKGServices::executeSelectSqliteOrder(getDocument(),
                                                      "SELECT ABS(TOTAL(f_CURRENTAMOUNT_EXPENSE)),TOTAL(f_CURRENTAMOUNT_INCOME) FROM v_operation_display WHERE t_status='P' AND t_ACCOUNT="+filter,
                                                      listTmp);
                if (listTmp.count()==2) {
                        KLocale* locale=KGlobal::locale();

                        QString unit1=((SKGDocumentBank*) getDocument())->getPrimaryUnit();
                        double unit1Value=1;
                        QString unit2=((SKGDocumentBank*) getDocument())->getSecondaryUnit();
                        double unit2Value=((SKGDocumentBank*) getDocument())->getSecondaryUnitValue();

                        if (ui.kDisplayAccountCombo->currentIndex()>0) {
                                SKGUnitObject unitAccount;
                                SKGAccountObject account(getDocument());
                                if (account.setName(ui.kDisplayAccountCombo->currentText()).isSucceeded()) {
                                        if (account.load().isSucceeded()) {
                                                if (account.getUnit(unitAccount).isSucceeded()) {
                                                        if (!unitAccount.getSymbol().isEmpty()) {
                                                                unit1=unitAccount.getSymbol();
                                                                unit1Value=SKGServices::stringToDouble(unitAccount.getAttribute("f_CURRENTAMOUNT"));

                                                                if (unit1!=((SKGDocumentBank*) getDocument())->getPrimaryUnit()) {
                                                                        unit2=((SKGDocumentBank*) getDocument())->getPrimaryUnit();
                                                                        unit2Value=1;
                                                                }
                                                        }
                                                }
                                        }
                                }
                        }

                        double v1=SKGServices::stringToDouble(listTmp.at(1).at(0));
                        double v2=SKGServices::stringToDouble(listTmp.at(1).at(1));
                        QString sdiff=QString("<font color=\"")+(diff<0 ? "red" : "black")+"\">"+locale->formatMoney (diff/unit1Value, unit1,2)+"</font>";
                        QString s1=QString("<font color=\"")+(v1<0 ? "red" : "black")+"\">"+locale->formatMoney (v1/unit1Value, unit1,2)+"</font>";
                        QString s2=QString("<font color=\"")+(v2<0 ? "red" : "black")+"\">"+locale->formatMoney (v2/unit1Value, unit1,2)+"</font>";
                        ui.kReconciliatorInfo->setText(i18n("%1 - Delta: %2     Expense: %3     Income: %4", unit1, sdiff, s1, s2));
                        if (!unit2.isEmpty() && unit2Value) {
                                sdiff=QString("<font color=\"")+(diff<0 ? "red" : "black")+"\">"+locale->formatMoney (diff/unit2Value, unit2,2)+"</font>";
                                s1=QString("<font color=\"")+(v1<0 ? "red" : "black")+"\">"+locale->formatMoney (v1/unit2Value, unit2,2)+"</font>";
                                s2=QString("<font color=\"")+(v2<0 ? "red" : "black")+"\">"+locale->formatMoney (v2/unit2Value, unit2,2)+"</font>";
                        }
                        ui.kReconciliatorInfo->setToolTip(i18n("<p>Delta: %1</p><p>Expense: %2</p><p>Income: %3</p>", s1, s2));
                }

        }
        QApplication::restoreOverrideCursor();
}

void SKGOperationPluginWidget::onFilterChanged()
{
        SKGTRACEIN(1, "SKGOperationPluginWidget::onFilterChanged");
        if (!isEnabled()) return;
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

        //Enable/ disable widgets
        bool onOneAccount=(ui.kDisplayAccountCombo->currentIndex()>0);
        ui.kAccountEdit->setEnabled(!onOneAccount);
        ui.kReconciliatorButton->setEnabled(onOneAccount);
        if (!onOneAccount) {
                ui.kReconciliatorFrame2->hide();
                ui.kInfo->show();
                modeInfoZone=0;
        }

        ui.kAccountLabel->setEnabled(operationWhereClause.isEmpty());
        ui.kDisplayAccountCombo->setEnabled(operationWhereClause.isEmpty());
        ui.kHideUselessOperation->setEnabled(operationWhereClause.isEmpty());
        if (operationWhereClause.isEmpty()) ui.kresetInternalFiler->hide();
        else ui.kresetInternalFiler->show();

        //Compute where clause
        QString filter2=operationWhereClause;
        if (onOneAccount) {
                QString account=ui.kDisplayAccountCombo->currentText();
                if (operationWhereClause.isEmpty()) filter2="t_ACCOUNT='"+SKGServices::stringToSqlString(account)+'\'';

                ui.kAccountEdit->setText(account);
        }

        //Add status filter
        if (operationWhereClause.isEmpty() && ui.kHideUselessOperation->checkState()==Qt::Checked) {
                if (!filter2.isEmpty()) filter2+=" AND ";
                filter2+="t_status!='C'";
        }

        //Update model
        QSortFilterProxyModel* modelproxy = (QSortFilterProxyModel*) ui.kOperationView->model();
        SKGObjectModel* objectModel=(SKGObjectModel*) modelproxy->sourceModel ();
        objectModel->setFilter(filter2);
        objectModel->refresh();

        //Correction bug 2299394 vvv
        if (ui.kOperationView->isAutoResized())ui.kOperationView->resizeColumnsToContents();
        //Correction bug 2299394 ^^^

        QApplication::restoreOverrideCursor();
}

void SKGOperationPluginWidget::onSplitOperationClicked()
{
        displaySplitted();

        ui.kSubOperationsTable->setRowCount(0);
        ui.kSubOperationsTable->clearContents();

        addSubOperationLine(0, ui.kCategoryEdit->text(), ui.kCommentEdit->text(), ui.kAmountEdit->value(), 0);
}

void SKGOperationPluginWidget::onFocusChanged()
{
        _SKGTRACEIN(10, "SKGOperationPluginWidget::onFocusChanged");

        if (numberFieldIsNotUptodate && ui.kNumberEdit->hasFocus()) {
                SKGTRACEIN(10, "SKGOperationPluginWidget::onFocusChanged-fill i_number");
                QStringList list;
                SKGServices::getDistinctValues(getDocument(), "v_operation_next_numbers", "i_number", "", list);

                //Fill completion
                KCompletion *comp = ui.kNumberEdit->completionObject();
                comp->clear ();
                comp->insertItems(list);

                numberFieldIsNotUptodate=false;
        }

        bool test=ui.kTypeEdit->hasFocus() ||
//                  ui.kAmountEdit->hasFocus() ||
//  		    ui.kNumberEdit->hasFocus() ||
                  ui.kUnitEdit->hasFocus() ||
                  ui.kCategoryEdit->hasFocus() ||
                  ui.kCommentEdit->hasFocus() ||
                  ui.kPayeeEdit->hasFocus()
                  ;
        fastEditionAction->setEnabled(test);
}

void SKGOperationPluginWidget::onFastEdition()
{
        SKGTRACEIN(10, "SKGOperationPluginWidget::onFocusChanged");
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
        SKGError err;

        //Build the where clause
        QString wc;
        if (ui.kTypeEdit->hasFocus()) wc="t_mode LIKE '"+SKGServices::stringToSqlString(ui.kTypeEdit->text())+"%'";
        else if (ui.kUnitEdit->hasFocus()) wc="t_UNIT LIKE '"+SKGServices::stringToSqlString(ui.kUnitEdit->text())+"%'";
        else if (ui.kCategoryEdit->hasFocus()) wc="t_CATEGORY LIKE '"+SKGServices::stringToSqlString(ui.kCategoryEdit->text())+"%'";
        else if (ui.kCommentEdit->hasFocus()) wc="t_comment LIKE '"+SKGServices::stringToSqlString(ui.kCommentEdit->text())+"%'";
        else if (ui.kPayeeEdit->hasFocus()) wc="t_payee LIKE '"+SKGServices::stringToSqlString(ui.kPayeeEdit->text())+"%'";

        if (!wc.isEmpty()) {
                if (wc!=lastFastEditionWhereClause) {
                        lastFastEditionWhereClause=wc;
                        lastFastEditionOperationFound=0;
                }

                //Look for last operation
                if (lastFastEditionOperationFound!=0) {
                        wc+=" AND id<"+SKGServices::intToString(lastFastEditionOperationFound);
                }

                wc+=" ORDER BY d_date DESC, id DESC LIMIT 1";

                SKGObjectBase::SKGListSKGObjectBase operations;
                err=SKGObjectBase::getObjects(getDocument(), "v_operation_display", wc, operations);
                if (err.isSucceeded() && operations.count()) {
                        SKGOperationObject op=operations.at(0);

                        lastFastEditionOperationFound=op.getID();
                        ui.kTypeEdit->setText(op.getAttribute("t_mode"));
                        ui.kUnitEdit->setText(op.getAttribute("t_UNIT"));
                        ui.kCategoryEdit->setText(op.getAttribute("t_CATEGORY"));
                        ui.kCommentEdit->setText(op.getAttribute("t_comment"));
                        ui.kPayeeEdit->setText(op.getAttribute("t_payee"));
                        if (ui.kAccountEdit->isEnabled()) ui.kAccountEdit->setText(op.getAttribute("t_ACCOUNT"));
                        ui.kAmountEdit->setValue(SKGServices::stringToDouble(op.getAttribute("f_QUANTITY")));

                        //Get nb operation linked
                        SKGObjectBase::SKGListSKGObjectBase groupedOperations;
                        op.getGroupedOperations(groupedOperations);
                        int nbGroupedOp=groupedOperations.count();

                        //Get nb sub op
                        SKGObjectBase::SKGListSKGObjectBase subOperations;
                        op.getSubOperations(subOperations);
                        int nbSupOp=subOperations.count();

                        if (nbSupOp>1) {
                                //It is a SPLITTED operation
                                displaySplitted();
                                displaySubOperations(op);
                        } else {
                                if (nbGroupedOp>1) {
                                        //It is a TRANSFER
                                        SKGOperationObject op2=groupedOperations.at(0);
                                        if (op2==op) op2=groupedOperations.at(1);

                                        SKGAccountObject targetAccount;
                                        op2.getParentAccount(targetAccount);

                                        ui.kOperationTabPages->setCurrentIndex(1);
                                        ui.kTargetAccountEdit->setText(targetAccount.getName());
                                } else displayStandard();
                        }

                } else {
                        lastFastEditionWhereClause="";
                        lastFastEditionOperationFound=0;
                }
        }
        QApplication::restoreOverrideCursor();

        //Display error
        getMainPanel()->displayErrorMessage(err);

}

void SKGOperationPluginWidget::displaySplitted()
{
        ui.kOperationTabPages->removeTab(0);
        ui.kOperationTabPages->insertTab(0,splittedPage,KIcon("skrooge_split"),i18n("Splitted"));
        ui.kOperationTabPages->setCurrentIndex(0);
}

void SKGOperationPluginWidget::displayStandard()
{
        ui.kOperationTabPages->removeTab(0);
        ui.kOperationTabPages->insertTab(0,standardPage,KIcon("dialog-ok-apply"),i18n("Standard"));
        ui.kOperationTabPages->setCurrentIndex(0);
}

void SKGOperationPluginWidget::displaySubOperations(const SKGOperationObject& iOperation)
{
        ui.kSubOperationsTable->setRowCount(0);
        ui.kSubOperationsTable->clearContents();

        int nbSubOperations=0;

        QList<SKGObjectBase> subOperations;
        SKGError err =  iOperation.getSubOperations(subOperations);
        nbSubOperations = subOperations.count();
        for (int i = 0; i < nbSubOperations; ++i) {
                SKGSubOperationObject subOperation = subOperations.at(i);

                SKGCategoryObject category;
                subOperation.getCategory(category);

                addSubOperationLine(i, category.getFullName(), subOperation.getComment(), subOperation.getQuantity(), subOperation.getID());
        }

        onQuantityChanged();
}

void SKGOperationPluginWidget::displaySubOperations()
{
        SKGOperationObject operation;
        if (getSelectedOperation(operation).isSucceeded()) displaySubOperations(operation);
}

double SKGOperationPluginWidget::getRemainingQuantity()
{
        double sumQuantities = 0;
        int nbSubOperations = ui.kSubOperationsTable->rowCount();

        for (int i = 0; i < nbSubOperations ; ++i) {
                QTableWidgetItem* quantityItem = ui.kSubOperationsTable->item(i,2);
                sumQuantities = sumQuantities + SKGServices::stringToDouble(quantityItem->text());
        }

        return ui.kAmountEdit->value() - sumQuantities;
}

void SKGOperationPluginWidget::onQuantityChanged()
{
        int nbSubOperations = ui.kSubOperationsTable->rowCount();

        QTableWidgetItem* remainingQuantityItem = ui.kSubOperationsTable->item(nbSubOperations-1,2);
        if (remainingQuantityItem) {
                bool previous=ui.kSubOperationsTable->blockSignals(true); // Disable signals so that filling cell doesn't create new lines
                remainingQuantityItem->setText(SKGServices::doubleToString(SKGServices::stringToDouble(remainingQuantityItem->text())+getRemainingQuantity()));
                ui.kSubOperationsTable->blockSignals(previous); // Reenable signals
        }
}

void SKGOperationPluginWidget::onSubopCellChanged(int row, int column)
{
        int nbSubOperations = ui.kSubOperationsTable->rowCount();
        if (row == nbSubOperations-1 && column == 2) {
                // If the quantity in the last line is edited, we add a new
                // line with the new remaining quantity
                addSubOperationLine(nbSubOperations,"","",0);
                onQuantityChanged();
        } else if (column == 2) {
                onQuantityChanged();
        }
}

void SKGOperationPluginWidget::onRemoveSubOperation()
{
        QList<int> rowsToRemove;
        QList<QTableWidgetItem*> selectedItems = ui.kSubOperationsTable->selectedItems();
        int nb=selectedItems.count();
        for (int i = 0; i < nb; ++i) {
                QTableWidgetItem* item = selectedItems.at(i);
                int row = item->row();
                if (!rowsToRemove.contains(row)) {
                        rowsToRemove.append(row);
                }
        }
        for (int j = rowsToRemove.count()-1; j >=0 ; --j) {
                ui.kSubOperationsTable->removeRow(rowsToRemove.at(j));
        }

        // If all rows removed, add an empty line
        if (ui.kSubOperationsTable->rowCount() == 0) {
                addSubOperationLine(0,"","",0);
        }

        onQuantityChanged();
}

void SKGOperationPluginWidget::onRotateAccountTools()
{
        if (modeInfoZone<=0) {
                //Switch to reconciliation mode
                ui.kReconciliatorFrame2->show();
                ui.kInfo->hide();
                modeInfoZone=1;
        } else {
                //Switch to information mode
                ui.kReconciliatorFrame2->hide();
                ui.kInfo->show();
                modeInfoZone=0;
        }
        onRefreshInformationZone();
}

void SKGOperationPluginWidget::addSubOperationLine(int row, QString category, QString comment, double quantity, int id)
{
        bool previous=ui.kSubOperationsTable->blockSignals(true);

        ui.kSubOperationsTable->insertRow(row);

        // category
        QTableWidgetItem* categoryItem = new QTableWidgetItem(category);
        categoryItem->setData(Qt::UserRole, id);
        ui.kSubOperationsTable->setItem(row,0,categoryItem);

        // comment
        ui.kSubOperationsTable->setItem(row,1,new QTableWidgetItem(comment));

        // Quantity
        QTableWidgetItem* quantityItem = new QTableWidgetItem(SKGServices::doubleToString(quantity));
        ui.kSubOperationsTable->setItem(row,2,quantityItem);

        ui.kSubOperationsTable->blockSignals(previous);

        ui.kSubOperationsTable->resizeColumnsToContents();
}

QWidget* SKGOperationPluginWidget::getWidgetForPrint()
{
        return ui.kOperationView;
}

SKGError SKGOperationPluginWidget::getSelectedOperation(SKGOperationObject& operation)
{
        SKGError err;
        SKGObjectBase::SKGListSKGObjectBase selectedOperations = getSelectedObjects();
        if (selectedOperations.count() > 0) {
                operation = selectedOperations.at(0);
                err.setReturnCode(0);
        } else {
                err.setReturnCode(1);
                err.setMessage(i18n("No Operation Selected"));
        }
        return err;
}

#include "skgoperationpluginwidget.moc"
