/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is Skrooge plugin for import and export operation.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgimportexportplugin.h"
#include "skgtraces.h"
#include "skgerror.h"
#include "skgbankincludes.h"
#include "skgmainpanel.h"

#include <kactioncollection.h>
#include <kaction.h>
#include <kactionmenu.h>
#include <kfiledialog.h>
#include <kencodingfiledialog.h>

#include <QTextCodec>

K_PLUGIN_FACTORY(SKGImportExportPluginFactory, registerPlugin<SKGImportExportPlugin>();)
K_EXPORT_PLUGIN(SKGImportExportPluginFactory("skrooge_importexport", "skrooge_importexport"))

SKGImportExportPlugin::SKGImportExportPlugin(QObject* iParent, const QVariantList& /*iArg*/) : SKGInterfacePlugin(iParent)
{
        SKGTRACEIN(10, "SKGImportExportPlugin::SKGImportExportPlugin");
}

SKGImportExportPlugin::~SKGImportExportPlugin()
{
        SKGTRACEIN(10, "SKGImportExportPlugin::~SKGImportExportPlugin");
        parent=NULL;
        currentBankDocument=NULL;

        importQifAction=NULL;
        importOfxAction=NULL;
        importCsvAction=NULL;
        importCsvUnitAction=NULL;
        exportCsvAction=NULL;
        exportQifAction=NULL;
        processingFoundTransfert=NULL;
        processingBankPerfect=NULL;
        processingBank=NULL;
}

void SKGImportExportPlugin::setupActions(SKGMainPanel* iParent, SKGDocument* iDocument, const QStringList& iArgument)
{
        SKGTRACEIN(10, "SKGImportExportPlugin::setupActions");
        Q_UNUSED(iArgument);

        currentBankDocument=iDocument;
        parent=iParent;

        // Tell the host application to load my GUI component
        setComponentData( SKGImportExportPluginFactory::componentData() );
        setXMLFile("skrooge_importexport.rc");

        //Imports
        KActionMenu* imports=new  KActionMenu(KIcon("document-import"),i18n("Import"), this);
        actionCollection()->addAction( QLatin1String("import"), imports );

        //Import OFX
        QStringList overlayofx;
        overlayofx.push_back("skrooge");

        importOfxAction = new KAction(KIcon("document-import", NULL, overlayofx), i18n("Import &OFX..."), this);
        connect(importOfxAction, SIGNAL(triggered(bool)), SLOT(importOfx()));
        actionCollection()->addAction( QLatin1String("import_ofx"), importOfxAction );
        imports->addAction(importOfxAction);

        if (parent) parent->registedGlobalAction("import_ofx", importOfxAction);

        //Import QIF
        QStringList overlaycsv;
        overlaycsv.push_back("text-csv");

        importQifAction = new KAction(KIcon("document-import"), i18n("Import &QIF..."), this);
        connect(importQifAction, SIGNAL(triggered(bool)), SLOT(importQif()));
        actionCollection()->addAction( QLatin1String("import_qif"), importQifAction );
        imports->addAction(importQifAction);

        if (parent) parent->registedGlobalAction("import_qif", importQifAction);

        //Import CSV
        importCsvAction = new KAction(KIcon("document-import", NULL, overlaycsv), i18n("Import &CSV..."), this);
        connect(importCsvAction, SIGNAL(triggered(bool)), SLOT(importCsv()));
        actionCollection()->addAction( QLatin1String("import_csv"), importCsvAction );
        imports->addAction(importCsvAction);

        if (parent) parent->registedGlobalAction("import_csv", importCsvAction);

        //Import CSV Unit
        importCsvUnitAction = new KAction(KIcon("document-import", NULL, overlaycsv), i18n("Import CSV &Unit..."), this);
        connect(importCsvUnitAction, SIGNAL(triggered(bool)), SLOT(importCsvUnit()));
        actionCollection()->addAction( QLatin1String("import_csv_unit"), importCsvUnitAction );
        imports->addAction(importCsvUnitAction);

        if (parent) parent->registedGlobalAction("import_csv_unit", importCsvUnitAction);

        //Exports
        KActionMenu* exports=new  KActionMenu(KIcon("document-export"),i18n("Export"), this);
        actionCollection()->addAction( QLatin1String("export"), exports );

        //Export QIF
        exportQifAction = new KAction(KIcon("document-export"), i18n("Export &QIF..."), this);
        connect(exportQifAction, SIGNAL(triggered(bool)), SLOT(exportQif()));
        actionCollection()->addAction( QLatin1String("export_qif"), exportQifAction );
        exports->addAction(exportQifAction);

        if (parent) parent->registedGlobalAction("export_qif", exportQifAction);

        //Export CSV
        exportCsvAction = new KAction(KIcon("document-export", NULL, overlaycsv), i18n("Export &CSV..."), this);
        connect(exportCsvAction, SIGNAL(triggered(bool)), SLOT(exportCsv()));
        actionCollection()->addAction( QLatin1String("export_csv"), exportCsvAction );
        exports->addAction(exportCsvAction);

        if (parent) parent->registedGlobalAction("export_csv", exportCsvAction);

        //Processing
        KActionMenu* processing=new  KActionMenu(KIcon("tools-wizard"),i18n("Processing"), this);
        actionCollection()->addAction( QLatin1String("processing"), processing );

        //Processing found and group
        QStringList overlaytransfers;
        overlaytransfers.push_back("skrooge_transfer");

        processingFoundTransfert = new KAction(KIcon("tools-wizard", NULL, overlaytransfers), i18n("Find and group transfers"), this);
        connect(processingFoundTransfert, SIGNAL(triggered(bool)), SLOT(foundTransfer()));
        actionCollection()->addAction( QLatin1String("process_foundtransfer"), processingFoundTransfert );
        processing->addAction(processingFoundTransfert);

        if (parent) parent->registedGlobalAction("process_foundtransfer", processingFoundTransfert);

        //Processing bankperfect
        processingBankPerfect = new KAction(KIcon("tools-wizard"), i18n("Clean BankPerfect's imports"), this);
        connect(processingBankPerfect, SIGNAL(triggered(bool)), SLOT(cleanBankPerfect()));
        actionCollection()->addAction( QLatin1String("process_bankperfect"), processingBankPerfect );
        processing->addAction(processingBankPerfect);

        if (parent) parent->registedGlobalAction("process_bankperfect", processingBankPerfect);

        //Processing banks
        processingBank = new KAction(KIcon("tools-wizard"), i18n("Clean bank's imports"), this);
        connect(processingBank, SIGNAL(triggered(bool)), SLOT(cleanBanks()));
        actionCollection()->addAction( QLatin1String("process_banks"), processingBank );
        processing->addAction(processingBank);

        if (parent) parent->registedGlobalAction("process_banks", processingBank);

}

QString SKGImportExportPlugin::title() const
{
        return i18n("Import / Export");
}

QString SKGImportExportPlugin::icon() const
{
        return "utilities-file-archiver";
}

QString SKGImportExportPlugin::statusTip () const
{
        return i18n("Import / Export management (csv, qif ...)");
}

QString SKGImportExportPlugin::toolTip () const
{
        return i18n("Import / Export management");
}

QStringList SKGImportExportPlugin::tips() const
{
        QStringList output;
        output.push_back(i18n("<p>... skrooge is able to detect automatically transfers after an import.</p>"));
        output.push_back(i18n("<p>... you can import many files in one shot.</p>"));
        output.push_back(i18n("<p>... unit amounts can be imported through a CSV file.</p>"));
        return output;
}

int SKGImportExportPlugin::getOrder() const
{
        return 3;
}

void SKGImportExportPlugin::refresh()
{
        SKGTRACEIN(10, "SKGImportExportPlugin::refresh");

        if (currentBankDocument) {
                bool test=(currentBankDocument->getDatabase()!=NULL);
                if (importQifAction) importQifAction->setEnabled(test);
                if (importOfxAction) importOfxAction->setEnabled(test);
                if (importCsvAction) importCsvAction->setEnabled(test);
                if (exportQifAction) exportQifAction->setEnabled(test);
                if (exportCsvAction) exportCsvAction->setEnabled(test);
                if (processingFoundTransfert) processingFoundTransfert->setEnabled(test);
                if (processingBankPerfect) processingBankPerfect->setEnabled(test);
                if (processingBank) processingBank->setEnabled(test);
        }
}

void SKGImportExportPlugin::close()
{
        SKGTRACEIN(10, "SKGImportExportPlugin::close");
}

void SKGImportExportPlugin::importOfx()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGImportExportPlugin::importOfx",err);
        import(SKGImportExportManager::OFX);
}

void SKGImportExportPlugin::importQif()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGImportExportPlugin::importQif",err);
        import(SKGImportExportManager::QIF);
}

void SKGImportExportPlugin::importCsv()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGImportExportPlugin::importCsv",err);
        import(SKGImportExportManager::CSV);
}

void SKGImportExportPlugin::importCsvUnit()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGImportExportPlugin::importCsvUnit",err);
        import(SKGImportExportManager::CSVUNIT);
}

void SKGImportExportPlugin::import(SKGImportExportManager::ImportExportMode iMode)
{
        SKGError err;
        SKGTRACEINRC(10, "SKGImportExportPlugin::import",err);
        if (currentBankDocument) {
                QString lastCodecUsed=currentBankDocument->getParameter("SKG_LAST_CODEC_USED_FOR_IMPORT");
                if (lastCodecUsed.length()==0) lastCodecUsed=QTextCodec::codecForLocale()->name();

                KEncodingFileDialog::Result result=KEncodingFileDialog::getOpenFileNamesAndEncoding(lastCodecUsed,
                                                   "kfiledialog:///IMPEXP",
                                                   iMode==SKGImportExportManager::CSV || iMode==SKGImportExportManager::CSVUNIT ?
                                                   "*.csv|"+i18n("CSV Files") :
                                                   iMode==SKGImportExportManager::OFX ? "*.ofx|"+i18n("OFX Files") : "*.qif|"+i18n("QIF Files") ,
                                                   parent);
                QSet<QString> fileNames=result.fileNames.toSet();
                QString codec=result.encoding;

                int nbFiles=fileNames.count();
                if (nbFiles) {

                        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
                        {
                                SKGBEGINPROGRESSTRANSACTION(*currentBankDocument, i18n("Import %1 with codec %2",
                                                            (iMode==SKGImportExportManager::CSV || iMode==SKGImportExportManager::CSVUNIT ? i18n("CSV") :
                                                             iMode==SKGImportExportManager::OFX ? i18n("OFX") : i18n("QIF")), codec), err, nbFiles);

                                if (err.isSucceeded()) err=currentBankDocument->setParameter("SKG_LAST_CODEC_USED_FOR_IMPORT", codec);

                                QSetIterator<QString> f(fileNames);
                                int i=1;
                                while (err.isSucceeded() && f.hasNext()) {
                                        QString fileName=f.next();
                                        SKGImportExportManager imp1((SKGDocumentBank*) currentBankDocument, iMode, fileName);
                                        imp1.setCodec(codec);
                                        err=imp1.importFile();
                                        if (err.isFailed()) {
                                                err.addError(ERR_FAIL, i18n("Import of [%1] failed", fileName));
                                        } else {
                                                err=SKGError(0, i18n("File [%1] successfully imported.", fileName));
                                        }

                                        if (err.isSucceeded()) err=currentBankDocument->stepForward(i);
                                        i++;
                                }
                        }
                        QApplication::restoreOverrideCursor();
                }

                //status bar
                if (err.isSucceeded() && nbFiles>1) {
                        err=SKGError(0, i18n("%1 files successfully imported.", nbFiles));
                }

                //Display error
                if (parent) parent->displayErrorMessage(err);
        }
}

void SKGImportExportPlugin::exportCsv()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGImportExportPlugin::exportCsv",err);
        QString fileName=SKGMainPanel::getSaveFileName(KUrl("kfiledialog:///IMPEXP"), "*.csv|"+i18n("CSV Files") , parent);
        if (fileName.isEmpty() || !currentBankDocument) return;

        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
        {
                SKGBEGINTRANSACTION(*currentBankDocument, i18n("Export CSV"), err);
                if (err.isSucceeded()) {
                        SKGImportExportManager imp1((SKGDocumentBank*) currentBankDocument, SKGImportExportManager::CSV, fileName);
                        err=imp1.exportFile();
                }
        }
        QApplication::restoreOverrideCursor();

        //status bar
        if (err.isSucceeded()) err=SKGError(0, i18n("File [%1] successfully exported.", fileName));
        else err.addError(ERR_FAIL, i18n("Export of [%1] failed", fileName));

        //Display error
        if (parent) parent->displayErrorMessage(err);
}

void SKGImportExportPlugin::exportQif()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGImportExportPlugin::exportQif",err);
        QString fileName=SKGMainPanel::getSaveFileName(KUrl("kfiledialog:///IMPEXP"), "*.qif|"+i18n("QIF Files") , parent);
        if (fileName.isEmpty() || !currentBankDocument) return;

        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
        {
                SKGBEGINTRANSACTION(*currentBankDocument, i18n("Export QIF"), err);
                if (err.isSucceeded()) {
                        SKGImportExportManager imp1((SKGDocumentBank*) currentBankDocument, SKGImportExportManager::QIF, fileName);
                        err=imp1.exportFile();
                }
        }
        QApplication::restoreOverrideCursor();

        //status bar
        if (err.isSucceeded()) err=SKGError(0, i18n("File [%1] successfully exported.", fileName));
        else err.addError(ERR_FAIL, i18n("Export of [%1] failed", fileName));

        //Display error
        if (parent) parent->displayErrorMessage(err);
}

void SKGImportExportPlugin::foundTransfer()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGImportExportPlugin::foundTransfer",err);
        if (currentBankDocument) {
                QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
                int NbOperationsMerged=0;
                {
                        SKGBEGINTRANSACTION(*currentBankDocument, i18n("Find and group transfers"), err);
                        if (err.isSucceeded()) {
                                SKGImportExportManager imp1((SKGDocumentBank*) currentBankDocument, SKGImportExportManager::QIF, "");
                                err=imp1.findAndGroupTransfers(NbOperationsMerged);
                        }
                }
                QApplication::restoreOverrideCursor();

                //status bar
                if (err.isSucceeded()) err=SKGError(0, i18n("Document successfully processed. %1 operations merged.", NbOperationsMerged));
                else err.addError(ERR_FAIL, i18n("Processing failed."));

                //Display error
                if (parent) parent->displayErrorMessage(err);
        }
}

void SKGImportExportPlugin::cleanBankPerfect()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGImportExportPlugin::cleanBankPerfect",err);
        if (currentBankDocument) {
                QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
                {
                        SKGBEGINTRANSACTION(*currentBankDocument, i18n("Clean BankPerfect's imports"), err);
                        if (err.isSucceeded()) {
                                SKGImportExportManager imp1((SKGDocumentBank*) currentBankDocument, SKGImportExportManager::QIF, "");
                                err=imp1.cleanBankPerfectImport();
                        }
                }
                QApplication::restoreOverrideCursor();

                //status bar
                if (err.isSucceeded()) err=SKGError(0, i18n("Document successfully cleaned."));
                else err.addError(ERR_FAIL, i18n("Clean failed."));

                //Display error
                if (parent) parent->displayErrorMessage(err);
        }
}

void SKGImportExportPlugin::cleanBanks()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGImportExportPlugin::cleanBanks",err);
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
        {
                SKGBEGINTRANSACTION(*currentBankDocument, i18n("Clean bank's imports"), err);
                if (err.isSucceeded()) {
                        SKGImportExportManager imp1((SKGDocumentBank*) currentBankDocument, SKGImportExportManager::QIF, "");
                        err=imp1.cleanBankImport();
                }
        }
        QApplication::restoreOverrideCursor();

        //status bar
        if (err.isSucceeded()) err=SKGError(0, i18n("Document successfully cleaned."));
        else err.addError(ERR_FAIL, i18n("Clean failed."));

        //Display error
        if (parent) parent->displayErrorMessage(err);
}
#include "skgimportexportplugin.moc"
