/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is Skrooge plugin to generate categories.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgcategoriespluginwidget.h"
#include "skgmainpanel.h"
#include "skgtraces.h"
#include "skgobjectmodel.h"
#include "skgsortfilterproxymodel.h"
#include "skgdocumentbank.h"
#include "skgcategoryobject.h"
#include "skgtransactionmng.h"

#include <klocale.h>

#include <QDomDocument>
#include <QtGui/QWidget>
#include <QGraphicsScene>
#include <QHeaderView>

SKGCategoriesPluginWidget::SKGCategoriesPluginWidget(SKGMainPanel* iParent, SKGDocumentBank* iDocument)
                : SKGTabWidget(iParent, iDocument)
{
        SKGTRACEIN(10, "SKGCategoriesPluginWidget::SKGCategoriesPluginWidget");

        ui.setupUi(this);

        ui.kAddCategoryButton->setIcon(KIcon("list-add"));
        ui.kModifyCategoryButton->setIcon(KIcon("dialog-ok-apply"));

        SKGObjectModel* objectModel = new SKGObjectModel((SKGDocumentBank*) getDocument(), "v_category_display",
                        "r_category_id IS NULL OR r_category_id=''",
                        this, "r_category_id=#ID#");
        SKGSortFilterProxyModel* modelproxy = new SKGSortFilterProxyModel(this);
        modelproxy->setSourceModel(objectModel);
        modelproxy->setSortRole(Qt::UserRole);
        modelproxy->setDynamicSortFilter(true);
        ui.kCategoriesView->setModel(modelproxy);
        ui.kCategoriesView->setWindowTitle(i18n("Categories"));
        ui.kCategoriesView->setDefaultSaveParameters(getDocument(), "SKG_DEFAULT_CATEGORY");

        //Set view parameters
        ui.kCategoriesView->sortByColumn(0, Qt::AscendingOrder);

        //Add registered global action in contextual menu
        if (iParent) {
                ui.kCategoriesView->insertAction(0, iParent->getGlobalAction("edit_delete"));
                QAction* sep=new QAction(this);
                sep->setSeparator(true);
                ui.kCategoriesView->insertAction(0, sep);
                ui.kCategoriesView->insertAction(0, iParent->getGlobalAction("open_report"));

                ui.kCategoriesView->resizeColumnToContents(0);
                ui.kCategoriesView->header()->setStretchLastSection(false);
        }

        ui.kTitle->setPixmap(KIcon("dialog-information").pixmap(22, 22));

        refresh();
        connect((const QObject*) getDocument(), SIGNAL(transactionSuccessfullyEnded(int)), this, SLOT(refresh()), Qt::QueuedConnection);

        connect(ui.kCategoriesView->selectionModel(), SIGNAL(selectionChanged(QItemSelection,QItemSelection)),
                this, SLOT(onSelectionChanged()));

        connect(objectModel, SIGNAL(modelAboutToBeReset()), ui.kCategoriesView, SLOT(saveSelection()));
        connect(objectModel, SIGNAL(modelReset()), ui.kCategoriesView, SLOT(resetSelection()));
}

SKGCategoriesPluginWidget::~SKGCategoriesPluginWidget()
{
        SKGTRACEIN(10, "SKGCategoriesPluginWidget::~SKGCategoriesPluginWidget");
}

void SKGCategoriesPluginWidget::refresh()
{
        SKGTRACEIN(10, "SKGCategoriesPluginWidget::refresh");
        //Correction bug 2299394 vvv
        if (ui.kCategoriesView->isAutoResized()) ui.kCategoriesView->resizeColumnsToContents();
        //Correction bug 2299394 ^^^
        onEditorModified();
}

void SKGCategoriesPluginWidget::onSelectionChanged()
{
        SKGTRACEIN(10, "SKGCategoriesPluginWidget::onSelectionChanged");

        SKGObjectBase::SKGListSKGObjectBase objs=getSelectedObjects();
        if (objs.count()) {
                SKGCategoryObject cat=objs.at(0);
                ui.kNameInput->setText(cat.getName());
        } else {
                ui.kNameInput->setText("");
        }
        onEditorModified();
        emit selectionChanged();
}

SKGObjectBase::SKGListSKGObjectBase SKGCategoriesPluginWidget::getSelectedObjects()
{
        //Get Selection
        SKGObjectBase::SKGListSKGObjectBase selection;
        QItemSelectionModel *selModel=ui.kCategoriesView->selectionModel();
        QSortFilterProxyModel* proxyModel=(QSortFilterProxyModel*) ui.kCategoriesView->model();
        SKGObjectModel* model=(SKGObjectModel*) proxyModel->sourceModel();
        if (model) {
                QModelIndexList indexes=selModel->selectedRows();
                foreach(const QModelIndex& index, indexes) {
                        SKGObjectBase obj=model->getObject(proxyModel->mapToSource(index));
                        selection.push_back(obj);
                }
        }
        return selection;
}

int SKGCategoriesPluginWidget::getNbSelectedObjects()
{
        QItemSelectionModel *selModel=ui.kCategoriesView->selectionModel();
        return (selModel ? selModel->selectedRows().count() : 0);
}

QString SKGCategoriesPluginWidget::getState()
{
        SKGTRACEIN(10, "SKGCategoriesPluginWidget::getState");
        QDomDocument doc("SKGML");
        QDomElement root = doc.createElement("parameters");
        doc.appendChild(root);

        //Memorize table settings
        root.setAttribute("view", ui.kCategoriesView->getState());
        return doc.toString();
}

void SKGCategoriesPluginWidget::setState(const QString& iState )
{
        SKGTRACEIN(10, "SKGCategoriesPluginWidget::setState");
        QDomDocument doc("SKGML");
        doc.setContent(iState);
        QDomElement root = doc.documentElement();

        ui.kCategoriesView->setState(root.attribute ( "view"));
}

QWidget* SKGCategoriesPluginWidget::getWidgetForPrint()
{
        return ui.kCategoriesView;
}

void SKGCategoriesPluginWidget::onEditorModified()
{
        _SKGTRACEIN(10, "SKGCategoriesPluginWidget::onEditorModified");
        int nb=getNbSelectedObjects();
        ui.kModifyCategoryButton->setEnabled(!ui.kNameInput->text().isEmpty() && nb==1);
        ui.kAddCategoryButton->setEnabled(!ui.kNameInput->text().isEmpty() && nb<=1);
}

void SKGCategoriesPluginWidget::onAddCategory()
{
        SKGError err;
        _SKGTRACEINRC(10, "SKGCategoriesPluginWidget::onAddCategory", err);
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
        //Get Selection
        SKGObjectBase::SKGListSKGObjectBase selection=getSelectedObjects();

        int nb=selection.count();
        QString name=ui.kNameInput->text();
        SKGBEGINTRANSACTION(*getDocument(), i18n("Category creation [%1]", name)  , err);

        SKGCategoryObject cat;
        err=SKGCategoryObject::createPathCategory((const SKGDocumentBank*) getDocument(), name, cat);
        if (nb==1 && err.isSucceeded()) {
                SKGCategoryObject parent=selection[0];
                SKGCategoryObject rootOfBranch;
                err=cat.getRootCategory(rootOfBranch);
                if (err.isSucceeded()) err=rootOfBranch.setParentCategory(parent);
                if (err.isSucceeded()) err=rootOfBranch.save();
        }

        //status bar
        if (err.isSucceeded()) err=SKGError(0, i18n("Category [%1] created", name));
        else err.addError(ERR_FAIL, i18n("Category creation failed"));
        QApplication::restoreOverrideCursor();

        //Display error
        getMainPanel()->displayErrorMessage(err);
}

void SKGCategoriesPluginWidget::onUpdateCategory()
{
        SKGError err;
        _SKGTRACEINRC(10, "SKGCategoriesPluginWidget::onUpdateCategory", err);
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
        //Get Selection
        SKGObjectBase::SKGListSKGObjectBase selection=getSelectedObjects();

        int nb=selection.count();
        if (nb==1) {
                SKGBEGINTRANSACTION(*getDocument(), i18n("Category [%1] update", ui.kNameInput->text())  , err);

                //Modification of category object
                SKGCategoryObject cat=selection[0];
                err=cat.setName(ui.kNameInput->text());
                if (err.isSucceeded()) err=cat.save();

                //status bar
                if (err.isSucceeded()) err=SKGError(0, i18n("Category [%1] updated", ui.kNameInput->text()));
                else err.addError(ERR_FAIL, i18n("Category update failed"));
        }
        QApplication::restoreOverrideCursor();

        //Display error
        getMainPanel()->displayErrorMessage(err);
}

void SKGCategoriesPluginWidget::onDoubleClick()
{
        _SKGTRACEIN(10, "SKGCategoriesPluginWidget::onDoubleClick");

        //Get selection
        SKGObjectBase::SKGListSKGObjectBase selection=getSelectedObjects();
        if (selection.count()==1) {
                SKGCategoryObject cat(selection.at(0));
                QString name=cat.getFullName();

                //Build where clause and title
                QString wc="t_REALCATEGORY";
                QString title;
                if (name.isEmpty()) {
                        wc+=" IS NULL OR t_REALCATEGORY=''";
                        wc='('+wc+')';
                        title=i18n("Operations with empty category");
                } else {
                        wc+=" = '"+SKGServices::stringToSqlString(name)+"' OR t_REALCATEGORY like '"+SKGServices::stringToSqlString(name)+OBJECTSEPARATOR+"%'";
                        wc='('+wc+')';
                        title=i18n("Operations with Category containing '%1'",  name);
                }

                //Open
                if (QApplication::keyboardModifiers()&Qt::ControlModifier && QApplication::keyboardModifiers()&Qt::ShiftModifier) {
                        //Call debug plugin
                        QDomDocument doc("SKGML");
                        QDomElement root = doc.createElement("parameters");
                        doc.appendChild(root);
                        root.setAttribute("sqlOrder", "SELECT * from v_operation_consolidated WHERE "+wc);

                        getMainPanel()->setNewTabContent(getMainPanel()->getPluginByName("Skrooge debug plugin"), -1, doc.toString());
                } else {
                        //Call operation plugin
                        QDomDocument doc("SKGML");
                        QDomElement root = doc.createElement("parameters");
                        doc.appendChild(root);
                        root.setAttribute("operationTable", "v_operation_consolidated");
                        root.setAttribute("operationWhereClause", wc);
                        root.setAttribute("title", title);
                        root.setAttribute("title_icon", "view-statistics");


                        getMainPanel()->setNewTabContent(getMainPanel()->getPluginByName("Skrooge operation plugin"), -1, doc.toString());
                }
        }
}

#include "skgcategoriespluginwidget.moc"
