/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is Skrooge plugin to generate categories.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgcategoriesplugin.h"
#include "skgcategoriespluginwidget.h"
#include "skgtraces.h"
#include "skgmainpanel.h"
#include "skgtransactionmng.h"
#include "skgcategoryobject.h"

#include <kstandarddirs.h>
#include <kmessagebox.h>
#include <kactioncollection.h>
#include <kaction.h>

#include <QDir>
#include <QFile>

K_PLUGIN_FACTORY(SKGCategoriesPluginFactory, registerPlugin<SKGCategoriesPlugin>(););
K_EXPORT_PLUGIN(SKGCategoriesPluginFactory("skrooge_categories", "skrooge_categories"));

SKGCategoriesPlugin::SKGCategoriesPlugin(QObject* iParent, const QVariantList& /*iArg*/)
                :SKGInterfacePlugin(iParent), parent(NULL), currentBankDocument(NULL)
{
        SKGTRACEIN(10, "SKGCategoriesPlugin::SKGCategoriesPlugin");
}

SKGCategoriesPlugin::~SKGCategoriesPlugin()
{
        SKGTRACEIN(10, "SKGCategoriesPlugin::~SKGCategoriesPlugin");
        parent=NULL;
        currentBankDocument=NULL;
}

void SKGCategoriesPlugin::setupActions(SKGMainPanel* iParent, SKGDocument* iDocument, const QStringList& iArgument)
{
        SKGTRACEIN(10, "SKGCategoriesPlugin::setupActions");
        Q_UNUSED(iArgument);

        currentBankDocument=iDocument;
        parent=iParent;

        // Tell the host application to load my GUI component
        setComponentData( SKGCategoriesPluginFactory::componentData() );
        setXMLFile("skrooge_categories.rc");

        //Import categories
        QStringList overlaycategories;
        overlaycategories.push_back("skrooge_category");

        KAction* importStdCatAction = new KAction(KIcon("document-import", NULL, overlaycategories), i18n("Import standard &categories"), this);
        connect(importStdCatAction, SIGNAL(triggered(bool)), SLOT(importStandardCategories()));
        actionCollection()->addAction( QLatin1String("import_standard_categories"), importStdCatAction );

        if (parent) parent->registedGlobalAction("import_standard_categories", importStdCatAction);
}

void SKGCategoriesPlugin::refresh()
{
        SKGTRACEIN(10, "SKGCategoriesPlugin::refresh");
        //Automatic categories creation
        if (currentBankDocument && currentBankDocument->getDatabase()!=NULL) {
                QString doc_id=currentBankDocument->getParameter("SKG_UNIQUE_ID");
                if (docUniqueIdentifier!=doc_id) {
                        docUniqueIdentifier=doc_id;

                        int nbCategory=0;
                        SKGError err=SKGObjectBase::getNbObjects(currentBankDocument, "category", "", nbCategory);
                        if (err.isSucceeded() && nbCategory==0) {
                                QFile file(KStandardDirs::locate("data", QString::fromLatin1("skrooge/ressources/default_categories.rc")));
                                if (file.open(QIODevice::ReadOnly | QIODevice::Text)) {

                                        QApplication::setOverrideCursor(QCursor(Qt::ArrowCursor));
                                        int rc=KMessageBox::questionYesNo(parent, i18n("Do you want to import standard categories ?"));
                                        QApplication::restoreOverrideCursor();
                                        if (rc==KMessageBox::Yes) {
                                                importStandardCategories();
                                        }
                                        file.close();
                                } else {
                                        SKGTRACE << file.fileName() << " does not exits for language [" << KGlobal::locale()->language() << "]" << endl;
                                }
                        }
                }
        }
}

void SKGCategoriesPlugin::importStandardCategories()
{
        SKGTRACEIN(10, "SKGCategoriesPlugin::importStandardCategories");
        SKGError err;
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
        QFile file(KStandardDirs::locate("data", QString::fromLatin1("skrooge/ressources/default_categories.rc")));
        if (file.open(QIODevice::ReadOnly | QIODevice::Text)) {

                SKGBEGINTRANSACTION(*currentBankDocument, i18n("Import standard categories"), err);
                QString style;
                QTextStream in(&file);
                while (!in.atEnd() && err.isSucceeded()) {
                        QString line=in.readLine().trimmed();
                        if (!line.isEmpty()) {
                                SKGCategoryObject cat;
                                err=SKGCategoryObject::createPathCategory((SKGDocumentBank*) currentBankDocument,
                                                line, cat);
                        }
                }

                file.close();
        } else {
                SKGTRACE << "WARNING: " << file.fileName() << " does not exits for language [" << KGlobal::locale()->language() << "]" << endl;
        }
        QApplication::restoreOverrideCursor();

        //status
        if (err.isSucceeded())  err=SKGError(0, i18n("Standard categories imported."));
        else err.addError(ERR_FAIL, i18n("Import standard categories failed"));

        //Display error
        if (parent) parent->displayErrorMessage(err);
}

void SKGCategoriesPlugin::close()
{
        SKGTRACEIN(10, "SKGCategoriesPlugin::close");
}

SKGTabWidget* SKGCategoriesPlugin::getWidget()
{
        SKGTRACEIN(10, "SKGCategoriesPlugin::getWidget");
        return new SKGCategoriesPluginWidget(parent, (SKGDocumentBank*) currentBankDocument);
}

QString SKGCategoriesPlugin::title() const
{
        return i18n("Categories");
}

QString SKGCategoriesPlugin::icon() const
{
        return "skrooge_category";
}

QString SKGCategoriesPlugin::statusTip () const
{
        return i18n("Manage categories");
}

QString SKGCategoriesPlugin::toolTip () const
{
        return i18n("Manage categories");
}

QStringList SKGCategoriesPlugin::tips() const
{
        QStringList output;
        output.push_back(i18n("<p>... categories can be reorganized by drag & drop.</p>"));
        return output;
}

int SKGCategoriesPlugin::getOrder() const
{
        return 30;
}

bool SKGCategoriesPlugin::isInContext() const
{
        return true;
}

#include "skgcategoriesplugin.moc"
