/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is Skrooge plugin for bookmarks management.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgbookmarkplugindockwidget.h"
#include "skgtraces.h"
#include "skgdocument.h"
#include "skgobjectmodelbase.h"
#include "skgmainpanel.h"
#include "skgsortfilterproxymodel.h"
#include "skgnodeobject.h"
#include "skginterfaceplugin.h"
#include "skgtransactionmng.h"

#include <kmenu.h>

#include <QHeaderView>
#include <QResizeEvent>

SKGBookmarkPluginDockWidget::SKGBookmarkPluginDockWidget(SKGMainPanel* iParent, SKGDocument* iDocument)
                : SKGTabWidget(iParent, iDocument)
{
        SKGTRACEIN(1, "SKGBookmarkPluginDockWidget::SKGBookmarkPluginDockWidget");

        ui.setupUi(this);
        minSize=minimumSizeHint ();
        ui.kActionsFrame->hide();

        ui.kAddBookmarkGroup->setIcon(KIcon("folder-new"));
        ui.kAddBookmark->setIcon(KIcon("list-add"));
        ui.kRemoveBookmark->setIcon(KIcon("list-remove"));
        ui.kRenameBookmark->setIcon(KIcon("edit-rename"));

        if (ui.kBookmarksList->isSmoothScrolling()) ui.kBookmarksList->switchSmoothScrolling();

        //Build contextual menu
        ui.kBookmarksList->setContextMenuPolicy(Qt::CustomContextMenu);
        connect(ui.kBookmarksList, SIGNAL(customContextMenuRequested(const QPoint & ) ),this,SLOT(showMenu( const QPoint& )));

        mainMenu = new KMenu(ui.kBookmarksList);
        actDelete = mainMenu->addAction(ui.kRemoveBookmark->icon(), i18n("Delete"));
        connect(actDelete, SIGNAL(triggered(bool)), this, SLOT(onRemoveBookmark()));

        mainMenu->addSeparator();

        actAddBookmarkGroup = mainMenu->addAction(ui.kAddBookmarkGroup->icon(), ui.kAddBookmarkGroup->toolTip());
        connect(actAddBookmarkGroup, SIGNAL(triggered(bool)), this, SLOT(onAddBookmarkGroup()));

        actAddBookmark = mainMenu->addAction(ui.kAddBookmark->icon(), ui.kAddBookmark->toolTip());
        connect(actAddBookmark, SIGNAL(triggered(bool)), this, SLOT(onAddBookmark()));

        actSetAutostart = mainMenu->addAction(KIcon("user-online"), i18n("Autostart"));
        connect(actSetAutostart, SIGNAL(triggered(bool)), this, SLOT(onSetAutostart()));

        actUnsetAutostart = mainMenu->addAction(KIcon("user-offline"), i18n("Remove Autostart"));
        connect(actUnsetAutostart, SIGNAL(triggered(bool)), this, SLOT(onUnsetAutostart()));


        //Add model
        SKGObjectModelBase* modelview = new SKGObjectModelBase(getDocument(), "v_node", "r_node_id IS NULL OR r_node_id='' ORDER BY f_sortorder, t_name",
                        this, "r_node_id=#ID# ORDER BY f_sortorder, t_name");
        SKGSortFilterProxyModel* modelproxy = new SKGSortFilterProxyModel(this);
        modelproxy->setSourceModel(modelview);
        ui.kBookmarksList->setModel(modelproxy);

        connect(ui.kBookmarksList->selectionModel(), SIGNAL(selectionChanged(QItemSelection,QItemSelection)), this, SLOT(refresh()));
        connect(ui.kBookmarksList, SIGNAL(doubleClicked(QModelIndex)), this, SLOT(onOpenBookmark()));
}

SKGBookmarkPluginDockWidget::~SKGBookmarkPluginDockWidget()
{
        SKGTRACEIN(1, "SKGBookmarkPluginDockWidget::~SKGBookmarkPluginDockWidget");
        mainMenu=NULL;
        actDelete=NULL;
        actAddBookmark=NULL;
        actAddBookmarkGroup=NULL;
        actSetAutostart=NULL;
        actUnsetAutostart=NULL;
}

void SKGBookmarkPluginDockWidget::showMenu(const QPoint& pos)
{
        if (mainMenu) mainMenu->popup(ui.kBookmarksList->mapToGlobal(pos));
}

int SKGBookmarkPluginDockWidget::getNbSelectedBookmarks()
{
        QItemSelectionModel *selModel=ui.kBookmarksList->selectionModel();
        return (selModel ? selModel->selectedRows().count() : 0);
}

SKGObjectBase::SKGListSKGObjectBase SKGBookmarkPluginDockWidget::getBookmarks()
{
        //Get Selection
        SKGObjectBase::SKGListSKGObjectBase selection;
        QItemSelectionModel *selModel=ui.kBookmarksList->selectionModel();
        QSortFilterProxyModel* proxyModel=(QSortFilterProxyModel*) ui.kBookmarksList->model();
        SKGObjectModelBase* model=(SKGObjectModelBase*) proxyModel->sourceModel();
        if (model) {
                QModelIndexList indexes=selModel->selectedRows();
                foreach(const QModelIndex& index, indexes) {
                        SKGObjectBase obj=model->getObject(proxyModel->mapToSource(index));
                        selection.push_back(obj);
                }
        }
        return selection;
}

void SKGBookmarkPluginDockWidget::onBookmarkFilterRegExpChanged()
{
        SKGTRACEIN(10, "SKGBookmarkPluginDockWidget::onBookmarkFilterRegExpChanged");
        QRegExp regExp(ui.kFilterEdit->text(), Qt::CaseInsensitive);
        ((SKGSortFilterProxyModel*) ui.kBookmarksList->model())->setFilterRegExp(regExp);
}

void SKGBookmarkPluginDockWidget::onSetAutostart()
{
        SKGTRACEIN(10, "SKGBookmarkPluginDockWidget::onSetAutoStart");
        setAutostart("Y");
}

void SKGBookmarkPluginDockWidget::onUnsetAutostart()
{
        SKGTRACEIN(10, "SKGBookmarkPluginDockWidget::onSetAutoStart");
        setAutostart("N");
}

void SKGBookmarkPluginDockWidget::setAutostart(const QString& value)
{
        SKGTRACEIN(10, "SKGBookmarkPluginDockWidget::setAutostart");
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

        SKGObjectBase::SKGListSKGObjectBase selectedBookmarks = getBookmarks();

        SKGError err;

        //foreach selected bookmark, set the t_autostart attribute to 'y' or 'n'
        {
                int nb=selectedBookmarks.count();
                SKGBEGINPROGRESSTRANSACTION(*getDocument(), value == "Y" ? i18n("Autostart bookmarks") : i18n("Do not Autostart bookmarks"), err, nb);
                for (int i = 0 ; err.isSucceeded() && i<nb  ; ++i) {
                        SKGNodeObject bookmark = selectedBookmarks.at(i);
                        err = bookmark.setAttribute("t_autostart",value);
                        if (err.isSucceeded())  err=bookmark.save();
                        if (err.isSucceeded())  err=getDocument()->stepForward(i+1);
                }
        }
        QApplication::restoreOverrideCursor();

        //status bar
        if (err.isSucceeded()) err=SKGError(0, value == "Y" ? i18n("Autostart bookmarks") : i18n("Do not Autostart bookmarks"));
        getMainPanel()->displayErrorMessage(err);

}


void SKGBookmarkPluginDockWidget::openBookmark(const SKGNodeObject& iNode, bool iFirstInNewPage)
{
        SKGTRACEIN(1, "SKGBookmarkPluginDockWidget::openBookmark");
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

        QList<SKGNodeObject> nodes;
        nodes.push_back(iNode);

        int setForcusOn=0;
        int nbTab=getMainPanel()->countTab();
        if (nbTab) {
                setForcusOn=nbTab-1;
                if (QApplication::keyboardModifiers()&Qt::ControlModifier) {
                        setForcusOn=nbTab;
                } else {
                        setForcusOn=getMainPanel()->currentTabIndex();
                }
        }

        int tabNumberForNextOpen=(QApplication::keyboardModifiers()&Qt::ControlModifier || iFirstInNewPage ? -1 : getMainPanel()->currentTabIndex());
        for (int i=0; i<nodes.count(); ++i) { //WARNING: This list is modified dynamically
                SKGNodeObject selectedNode=nodes.at(i);
                QStringList data=SKGServices::splitCSVLine(selectedNode.getData());
                if (data.count()>3) {
                        //This bookmark has information ==> we open it
                        SKGTabWidget* w=getMainPanel()->setNewTabContent(getMainPanel()->getPluginByName(data[0]),tabNumberForNextOpen, data[3], selectedNode.getName());
                        selectedNode.opened = true;
                        tabNumberForNextOpen=-1;
                        if (w) {
                                w->setBookmarkID(SKGServices::intToString(selectedNode.getID()));
                        }
                } else {
                        //This bookmark is a folder ==> we open children by recursion
                        SKGObjectBase::SKGListSKGObjectBase children;
                        selectedNode.getNodes(children);
                        foreach(const SKGObjectBase& item, children) {
                                nodes.push_back(item);
                        }
                }
        }

        //Set focus on first page
        getMainPanel()->setCurrentTabIndex(setForcusOn);

        QApplication::restoreOverrideCursor();
}

void SKGBookmarkPluginDockWidget::onOpenBookmark()
{
        SKGTRACEIN(1, "SKGBookmarkPluginDockWidget::onOpenBookmark");
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
        SKGObjectBase::SKGListSKGObjectBase bookmarks=getBookmarks();
        if (bookmarks.count()) openBookmark(bookmarks.at(0));
        QApplication::restoreOverrideCursor();
}

void SKGBookmarkPluginDockWidget::onAddBookmarkGroup()
{
        SKGTRACEIN(1, "SKGBookmarkPluginDockWidget::onAddBookmarkGroup");
        SKGError err;
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
        {
                //Get current selection name
                QString name;
                SKGObjectBase::SKGListSKGObjectBase bookmarks=getBookmarks();
                if (bookmarks.count()) {
                        SKGNodeObject parentNode=bookmarks.at(0);
                        name=parentNode.getFullName();
                }

                //Add current name
                QString namef=ui.kRenameBookmarkEdit->text();
                if (namef.isEmpty()) namef=i18n("new bookmark folder");

                if (!name.isEmpty()) name+=OBJECTSEPARATOR;
                name+=namef;

                //Create bookmark folder
                SKGBEGINTRANSACTION(*getDocument(), i18n("Bookmark folder creation [%1]", name), err);
                SKGNodeObject node;
                err=SKGNodeObject::createPathNode(getDocument(), name, node, true);
        }
        QApplication::restoreOverrideCursor();

        //status bar
        if (err.isSucceeded()) err=SKGError(0, i18n("Bookmark group created"));
        getMainPanel()->displayErrorMessage(err);
}

void SKGBookmarkPluginDockWidget::onAddBookmark()
{
        SKGTRACEIN(1, "SKGBookmarkPluginDockWidget::onAddBookmark");
        SKGError err;
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

        //Get current page
        SKGTabWidget* currentPage=getMainPanel()->currentTab();
        if (currentPage) {
                //Get current selection name
                QString name;
                SKGObjectBase::SKGListSKGObjectBase bookmarks=getBookmarks();
                if (bookmarks.count()) {
                        SKGNodeObject parentNode=bookmarks.at(0);
                        name=parentNode.getFullName();
                }

                //Add current name
                if (!name.isEmpty()) name+=OBJECTSEPARATOR;

                QString defaultName=currentPage->objectName();
                QString defaultIcon;
                SKGInterfacePlugin* plugin=getMainPanel()->getPluginByName(defaultName);
                if (plugin) {
                        defaultName=plugin->title();
                        defaultIcon=plugin->icon();
                }
                name+=defaultName;

                //Create bookmark
                SKGBEGINTRANSACTION(*getDocument(), i18n("Bookmark creation [%1]", name), err);
                SKGNodeObject node;
                err=SKGNodeObject::createPathNode(getDocument(), name, node, true);
                if (err.isSucceeded()) {
                        QString value=SKGServices::stringToCsv(currentPage->objectName())+';'+
                                      SKGServices::stringToCsv(defaultName)+';'+
                                      SKGServices::stringToCsv(defaultIcon)+';'+
                                      SKGServices::stringToCsv(currentPage->getState());

                        err=node.setData(value);
                        if (err.isSucceeded()) err=node.save();
                }
        }

        QApplication::restoreOverrideCursor();

        //status bar
        if (err.isSucceeded()) err=SKGError(0, i18n("Bookmark created"));
        getMainPanel()->displayErrorMessage(err);
}

void SKGBookmarkPluginDockWidget::onRenameBookmark()
{
        SKGTRACEIN(1, "SKGBookmarkPluginDockWidget::onRenameBookmark");
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

        //Get current selection
        SKGError err;
        SKGObjectBase::SKGListSKGObjectBase bookmarks=getBookmarks();
        if (bookmarks.count()) {
                SKGNodeObject parentNode=bookmarks.at(0);
                {
                        QString name=ui.kRenameBookmarkEdit->text();
                        SKGBEGINTRANSACTION(*getDocument(), i18n("Bookmark update [%1]", name), err);
                        err=parentNode.setName(name);
                        if (err.isSucceeded()) err=parentNode.save();
                }
        }
        QApplication::restoreOverrideCursor();

        //status bar
        if (err.isSucceeded()) err=SKGError(0, i18n("Bookmark renamed"));
        getMainPanel()->displayErrorMessage(err);
}



void SKGBookmarkPluginDockWidget::onRemoveBookmark()
{
        SKGTRACEIN(1, "SKGBookmarkPluginDockWidget::onRemoveBookmark");
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

        SKGError err;
        {
                SKGBEGINTRANSACTION(*getDocument(), i18n("Bookmark delete"), err);
                //Get current selection
                SKGObjectBase::SKGListSKGObjectBase bookmarks=getBookmarks();
                int nb=bookmarks.count();
                for (int i=0; i<nb && err.isSucceeded(); ++i) {
                        SKGNodeObject node=bookmarks.at(i);
                        err=node.remove();
                }
        }

        QApplication::restoreOverrideCursor();

        //status bar
        if (err.isSucceeded()) err=SKGError(0, i18n("Bookmark deleted"));
        getMainPanel()->displayErrorMessage(err);
}

void SKGBookmarkPluginDockWidget::onBookmarkEditorChanged()
{
        SKGTRACEIN(10, "SKGBookmarkPluginDockWidget::onBookmarkEditorChanged");

        //Bookmarks
        SKGObjectBase::SKGListSKGObjectBase bookmarks=getBookmarks();
        int nbSelectedObjects=bookmarks.count();
        int isFolder=true;
        if (nbSelectedObjects==1) {
                SKGNodeObject node=bookmarks.at(0);
                isFolder=node.getData().isEmpty();
        }

        //Enable bookmark buttons
        bool testAdd=getDocument()->getDatabase()!=NULL && nbSelectedObjects<=1 && isFolder;
        ui.kAddBookmarkGroup->setEnabled(testAdd);
        if (actAddBookmarkGroup) actAddBookmarkGroup->setEnabled(ui.kAddBookmarkGroup->isEnabled());
        ui.kAddBookmark->setEnabled(testAdd && getMainPanel()->currentTabIndex()>=0);
        if (actAddBookmark) actAddBookmark->setEnabled(ui.kAddBookmark->isEnabled());
        ui.kRenameBookmark->setEnabled(nbSelectedObjects==1 && !ui.kRenameBookmarkEdit->text().isEmpty());
        ui.kRemoveBookmark->setEnabled(nbSelectedObjects>0);
        if (actDelete) actDelete->setEnabled(ui.kRemoveBookmark->isEnabled());
        ui.kRenameBookmarkEdit->setEnabled(getDocument()->getDatabase()!=NULL && nbSelectedObjects<=1);

}
void SKGBookmarkPluginDockWidget::refresh()
{
        SKGTRACEIN(10, "SKGBookmarkPluginDockWidget::refresh");
        //Automatic open of autostart bookmark
        QString doc_id=getDocument()->getParameter("SKG_UNIQUE_ID");

        if (docUniqueIdentifier!=doc_id) {
                docUniqueIdentifier=doc_id;
                SKGObjectBase::SKGListSKGObjectBase oNodeList;
                SKGError err = SKGObjectBase::getObjects(getDocument(), "v_node", "t_autostart='Y' ORDER BY f_sortorder, t_name", oNodeList);
                int nbAutoStartedBookmarks = oNodeList.count();
                for (int i=0;i<nbAutoStartedBookmarks;i++) {
                        SKGNodeObject autostarted_bookmark = oNodeList.at(i);
                        autostarted_bookmark.load();
                        SKGTRACEIN(10, "autostarting bookmark : "+autostarted_bookmark.getName());
                        openBookmark(autostarted_bookmark, i>0);
                }
        }

        //Bookmarks
        SKGObjectBase::SKGListSKGObjectBase bookmarks=getBookmarks();
        int nbSelectedObjects=bookmarks.count();
        if (nbSelectedObjects==1) {
                SKGNodeObject node=bookmarks.at(0);
                ui.kRenameBookmarkEdit->setText(node.getName());

                if (actSetAutostart) actSetAutostart->setEnabled(node.getAttribute("t_autostart")!="Y");
                if (actUnsetAutostart) actUnsetAutostart->setEnabled(node.getAttribute("t_autostart")=="Y");
        }

        onBookmarkEditorChanged();
}

void SKGBookmarkPluginDockWidget::resizeEvent(QResizeEvent *event)
{
        if (event) {
                QSize newSize=event->size ();
                if (newSize.height()<=minSize.height() || newSize.width()<=minSize.width()) {
                        ui.kActionsFrame->hide();
                } else {
                        ui.kActionsFrame->show();
                }

                //Compute icon size
                int s=qMax(qMin(newSize.width()/5,64),16);
                ui.kBookmarksList->setIconSize(QSize(s, s));

                //TODO: must be fixed
                getMainPanel()->contextList()->setIconSize(QSize(s, s));
        }

        QWidget::resizeEvent(event);
}

#include "skgbookmarkplugindockwidget.moc"
