/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A table with graph for skrooge.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgtablewithgraph.h"
#include "skgservices.h"
#include "skgtraces.h"
#include "skgmainpanel.h"
#include "skggraphicsscene.h"

#include <kmenu.h>
#include <kfiledialog.h>

#include <QtGui/QHeaderView>
#include <QtGui/QGraphicsLineItem>
#include <QDomDocument>
#include <QRegExp>
#include <QPrinter>
#include <QPainter>
#include <QTableWidgetItem>
#include <QDesktopServices>

#include <math.h>

Qt::SortOrder SKGTableWithGraph::sortOrder=Qt::AscendingOrder;
int SKGTableWithGraph::sortColumn=0;

SKGTableWithGraph::SKGTableWithGraph(QWidget *parent)
                : QWidget(parent), scene(NULL), nbLevel(0), mainMenu(NULL)
{
        ui.setupUi(this);
        ui.kLineUp->setIcon(KIcon("arrow-right"));
        ui.kLineDown->setIcon(KIcon("arrow-left"));

        //Build contextual menu
        ui.kTable->setContextMenuPolicy(Qt::CustomContextMenu);
        connect(ui.kTable, SIGNAL(customContextMenuRequested(const QPoint & ) ),this,SLOT(showMenu( const QPoint& )));

        mainMenu = new KMenu(ui.kTable);
        mainMenu->addTitle(i18n("Table"));

        QMenu* exp=mainMenu->addMenu(i18n("Export"));

        QAction* actCSV = exp->addAction(KIcon("text-csv"), i18n("Export CSV..."));
        connect(actCSV, SIGNAL(triggered(bool)), this, SLOT(onExportCSV()));

        QAction* actTXT = exp->addAction(KIcon("text-plain"), i18n("Export TXT..."));
        connect(actTXT, SIGNAL(triggered(bool)), this, SLOT(onExportTXT()));

        //Set headers parameters
        QHeaderView* verticalHeader=ui.kTable->verticalHeader();
        if (verticalHeader)  verticalHeader->hide();

        ui.kTable->setSortingEnabled(false); //sort must be enable for refresh method
        QHeaderView* horizontalHeader=ui.kTable->horizontalHeader();
        if (horizontalHeader) {
                horizontalHeader->setResizeMode(QHeaderView::ResizeToContents);
                horizontalHeader->show();
                horizontalHeader->setSortIndicatorShown(true);
                horizontalHeader->setSortIndicator(sortColumn, sortOrder);
                connect(horizontalHeader, SIGNAL(sortIndicatorChanged(int, Qt::SortOrder)), this, SLOT(refresh()));
        }
        ui.kTable->verticalHeader()->setDefaultSectionSize(ui.kTable->verticalHeader()->minimumSectionSize());

        ui.kDisplayMode->addItem(i18n("Stack"));
        ui.kDisplayMode->addItem(i18n("Histogram"));
        ui.kDisplayMode->addItem(i18n("Pie"));
        ui.kDisplayMode->addItem(i18n("Concentric pie"));

        connect(ui.kDisplayMode, SIGNAL(currentIndexChanged(int)), this, SLOT(redrawGraph()), Qt::QueuedConnection);
        connect(ui.kAllPositive, SIGNAL(stateChanged(int)), this, SLOT(redrawGraph()), Qt::QueuedConnection);
}


SKGTableWithGraph::~SKGTableWithGraph()
{
        if (scene) {
                delete scene;
                scene=NULL;
        }

        mainMenu=NULL;
}

KMenu* SKGTableWithGraph::getTableContextualMenu() const
{
        return mainMenu;
}

KMenu* SKGTableWithGraph::getGraphContextualMenu() const
{
        return ui.graphicView->getContextualMenu();
}

void SKGTableWithGraph::showMenu(const QPoint& pos)
{
        if (mainMenu) mainMenu->popup(ui.kTable->mapToGlobal(pos));
}

QTableWidget* SKGTableWithGraph::table() const
{
        return ui.kTable;
}

SKGStringListList SKGTableWithGraph::getTable()
{
        //Build table
        SKGStringListList table;

        //Get header names
        int nb=ui.kTable->columnCount();
        QStringList cols;
        for (int i=0; i<nb; i++) {
                cols.append(ui.kTable->horizontalHeaderItem(i)->text());
        }
        table.append(cols);

        //Get content
        int nb2=ui.kTable->rowCount();
        for (int i=0; i<nb2; i++) {
                QStringList row;
                for (int j=0; j<nb; j++) {
                        row.append(ui.kTable->item(i,j)->text());
                }
                table.append(row);
        }
        return table;
}
QString SKGTableWithGraph::getState()
{
        SKGTRACEIN(10, "SKGTableWithGraph::getState");
        QDomDocument doc("SKGML");
        QDomElement root = doc.createElement("parameters");
        doc.appendChild(root);

        root.setAttribute("splitterState", QString(ui.splitter->saveState().toHex()));
        root.setAttribute("graphMode", SKGServices::intToString(ui.kDisplayMode->currentIndex()));
        root.setAttribute("nbLevel", SKGServices::intToString(nbLevel));
        root.setAttribute("allPositive", ui.kAllPositive->checkState()==Qt::Checked ? "Y" : "N");
        root.setAttribute("filter", ui.kFilterEdit->text());

        QHeaderView* horizontalHeader=ui.kTable->horizontalHeader();
        root.setAttribute("sortOrder", SKGServices::intToString(horizontalHeader->sortIndicatorOrder()));
        root.setAttribute("sortColumn", SKGServices::intToString(horizontalHeader->sortIndicatorSection()));
        root.setAttribute("graphicViewState", ui.graphicView->getState());

        //TODO: save graphic zoom and sroll position
        return doc.toString();
}

void SKGTableWithGraph::setState(const QString& iState )
{
        SKGTRACEIN(10, "SKGTableWithGraph::setState");
        QDomDocument doc("SKGML");
        if (doc.setContent(iState)) {
                QDomElement root = doc.documentElement();

                QString splitterState=root.attribute ( "splitterState");
                if (!splitterState.isEmpty()) ui.splitter->restoreState(QByteArray::fromHex(splitterState.toAscii()));
                QString graphMode=root.attribute ( "graphMode");
                if (!graphMode.isEmpty()) ui.kDisplayMode->setCurrentIndex(SKGServices::stringToInt(graphMode));
                QString nbLevelString=root.attribute ( "nbLevel");
                if (!nbLevelString.isEmpty()) nbLevel=SKGServices::stringToInt(nbLevelString);
                QString allPositive=root.attribute ( "allPositive");
                if (!allPositive.isEmpty()) ui.kAllPositive->setCheckState((allPositive=="Y" ? Qt::Checked : Qt::Unchecked));
                ui.kFilterEdit->setText(root.attribute ( "filter"));

                QString sortOrder=root.attribute ( "sortOrder");
                QString sortColumn=root.attribute ( "sortColumn");
                if (sortOrder.isEmpty()) sortOrder="0";
                if (sortColumn.isEmpty()) sortColumn="0";
                ui.kTable->horizontalHeader()->setSortIndicator(SKGServices::stringToInt(sortColumn),
                                (Qt::SortOrder) SKGServices::stringToInt(sortOrder));

                QString graphicViewState=root.attribute ( "graphicViewState");
                if (!graphicViewState.isEmpty()) ui.graphicView->setState(graphicViewState);
        } else {
                //vvv Correction bug 2278703:Zoom fit when opening report tab
                ui.kDisplayMode->setCurrentIndex(1);
                ui.kDisplayMode->setCurrentIndex(0);
                //^^^ Correction bug 2278703:Zoom fit when opening report tab
        }

        refresh();
}

void SKGTableWithGraph::onOneLevelMore()
{
        _SKGTRACEIN(10, "SKGTableWithGraph::onOneLevelMore");
        ++nbLevel;
        refresh();
}

void SKGTableWithGraph::onOneLevelLess()
{
        _SKGTRACEIN(10, "SKGTableWithGraph::onOneLevelLess");
        --nbLevel;
        refresh();
}

void SKGTableWithGraph::onFilterModified()
{
        refresh();
}

void SKGTableWithGraph::setData(const SKGStringListList& iData, const QString& iPrimaryUnit,
                                const QString& iSecondaryUnit, double iSecondaryUnitValue)
{
        SKGTRACEIN(10, "SKGTableWithGraph::setData");
        data=iData;
        primaryUnit=iPrimaryUnit;
        secondaryUnit=iSecondaryUnit;
        secondaryUnitValue=iSecondaryUnitValue;

        refresh();

}
bool SKGTableWithGraph::listSort(const QStringList &s1, const QStringList &s2)
{
        if (sortColumn>=s1.count()) sortColumn=s1.count()-1;

        QString v1=s1.at(sortColumn);
        QString v2=s2.at(sortColumn);
        if (sortColumn==0) {
                return (sortOrder ? v1.toLower() > v2.toLower() : v1.toLower() < v2.toLower());
        }

        double vd1=SKGServices::stringToDouble(v1);
        double vd2=SKGServices::stringToDouble(v2);
        return (sortOrder ? vd1>vd2 : vd1<vd2);
}

void SKGTableWithGraph::refresh()
{
        SKGTRACEIN(10, "SKGTableWithGraph::refresh");
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

        /*    {
        SKGTRACEIN(10, "test-Filltable");
        ui.kTable->clear();
        ui.kTable->setRowCount(10);
        ui.kTable->setColumnCount(10);
        for (int i=0; i<9; ++i) {
        for (int j=0; j<9; ++j) {
        SKGTRACEIN(10, "test-setItem");
        QTableWidgetItem* item=new QTableWidgetItem("A");
        ui.kTable->setItem(i, j, item);
        }
        }
        }*/

        //Create grouped table
        int nbLevelMax=0;
        int nbCols=-1;
        SKGStringListList groupedTable=data;
        {
                QString previousLineName="###";
                if (groupedTable.count()) {
                        nbCols=groupedTable.at(0).count();
                }
                for (int i=1; nbCols && i<groupedTable.count(); ++i) { //Dynamically modified
                        QStringList line=groupedTable.at(i);
                        QString val=line.at(0);

                        //Rebuild val for the number of level
                        QStringList vals=val.split(OBJECTSEPARATOR);
                        int nbvals=vals.count();
                        if (nbvals>nbLevel+1) {
                                //Rebuild val
                                val="";
                                for (int k=0; k<=nbLevel; ++k) {
                                        val+=vals[k];
                                        if (k!=nbLevel) val+=OBJECTSEPARATOR;
                                }
                        }
                        nbLevelMax=qMax(nbLevelMax, nbvals-1);

                        if (val==previousLineName) {
                                //Current line is merged with previous one
                                QStringList newLine;
                                newLine.push_back(val);
                                for (int k=1; k<nbCols; ++k) {
                                        double sum2= SKGServices::stringToDouble(line.at(k)) + SKGServices::stringToDouble(groupedTable.at(i-1).at(k));
                                        newLine.push_back(SKGServices::doubleToString(sum2));
                                }

                                groupedTable.replace(i-1, newLine);

                                //Remove current line
                                groupedTable.removeAt(i);
                                --i;
                        } else {
                                //Current line is just modified
                                QStringList newLine;
                                newLine.push_back(val);
                                for (int k=1; k<nbCols; ++k) {
                                        newLine.push_back(line.at(k));
                                }
                                groupedTable.replace(i, newLine);

                                previousLineName=val;
                        }
                }
        }

        //Create filtered table
        {
                QRegExp regexp(ui.kFilterEdit->text(),Qt::CaseInsensitive);
                for (int i=groupedTable.count()-1; i>=1; --i) { //The first line is not filtered because it's the title
                        QStringList line=groupedTable.at(i);
                        if (nbCols) {
                                QString val=line.at(0);

                                //Filtered
                                if (regexp.indexIn(val)==-1) {
                                        groupedTable.removeAt(i);
                                }
                        }
                }
        }

        //Compute sums line
        int nblines=groupedTable.count();
        {
                int nblines=groupedTable.count();
                {
                        QStringList sums;
                        sums.push_back(i18n("Sum"));
                        for (int i=1; i<nbCols; ++i) {
                                double sum=0;
                                for (int j=1; j<nblines; ++j) {
                                        sum+=SKGServices::stringToDouble(groupedTable.at(j).at(i));
                                }
                                sums.push_back(SKGServices::doubleToString(sum));
                        }
                        groupedTable.push_back(sums);
                }
        }
        nblines++;

        //Compute sum and average column
        if (nbCols>2) {
                //Add title
                QStringList newLine=groupedTable.at(0);
                newLine.push_back(tr("Sum"));
                newLine.push_back(tr("Average"));
                groupedTable.replace(0, newLine);

                for (int i=1; i<nblines; ++i) {
                        QStringList newLine=groupedTable.at(i);
                        double sum=0;
                        for (int j=1; j<nbCols; ++j) {
                                sum+=SKGServices::stringToDouble(groupedTable.at(i).at(j));
                        }
                        newLine.push_back(SKGServices::doubleToString(sum));
                        newLine.push_back(SKGServices::doubleToString(sum/(nbCols-1)));
                        groupedTable.replace(i, newLine);
                }
                nbCols+=2;
        }

        //Sort lines
        {
                //Set parameters for static method
                QHeaderView* horizontalHeader=ui.kTable->horizontalHeader();
                sortOrder=horizontalHeader->sortIndicatorOrder();
                sortColumn=horizontalHeader->sortIndicatorSection();

                //Extract title and sums
                QStringList fist=groupedTable.takeFirst();
                QStringList last=groupedTable.takeLast();

                //Sort
                qSort(groupedTable.begin(), groupedTable.end(), listSort);

                //Add title and sums
                groupedTable.insert(0, fist);
                groupedTable.push_back(last);
        }

        IFSKGTRACEL(10) {
                QStringList dump=SKGServices::tableToDump(groupedTable, SKGServices::DUMP_TEXT);
                int nbl=dump.count();
                SKGTRACE << "nbLevelMax=" << nbLevelMax << endl;
                for (int i=0; i<nbl; ++i) {
                        SKGTRACE << dump[i] << endl;
                }
        }


        //Fill table
        KLocale* locale=KGlobal::locale();
        ui.kTable->hide();
        ui.kTable->clear();
        ui.kTable->setRowCount(nblines-1);
        ui.kTable->setColumnCount(nbCols);
        for (int i=0; i<nblines; ++i) {
                QStringList line=groupedTable.at(i);
                if (i==0) {
                        //Set header
                        ui.kTable->setHorizontalHeaderLabels(line);
                } else {
                        for (int j=0; j<nbCols; ++j) {
                                QString val=line.at(j);

                                if (j==0) {
                                        ui.kTable->setItem(i-1, 0, new QTableWidgetItem(val));
                                } else {
                                        double vald=SKGServices::stringToDouble(val);

                                        QTableWidgetItem* item=new QTableWidgetItem(locale->formatMoney (vald, primaryUnit, 2));
                                        if (!secondaryUnit.isEmpty() && secondaryUnitValue) {
                                                item->setToolTip(locale->formatMoney (vald/secondaryUnitValue, secondaryUnit, 2));
                                        }

                                        item->setData(12, vald);
                                        item->setTextAlignment(Qt::AlignRight);
                                        if (vald<0) item->setForeground(QBrush(QColor(Qt::red)));
                                        ui.kTable->setItem(i-1, j, item);
                                }
                        }
                }
        }

        //Enable/Disable buttons
        nbLevel=qMin(nbLevelMax, nbLevel);
        ui.kLineDown->setEnabled(nbLevel>0);
        ui.kLineUp->setEnabled(nbLevel<nbLevelMax);

        //Refresh graphic view
        redrawGraph();

        ui.kTable->show();

        QApplication::restoreOverrideCursor();
}

void SKGTableWithGraph::onSelectionChanged(QTableWidgetItem* current, QTableWidgetItem* previous)
{
        _SKGTRACEIN(10, "SKGTableWithGraph::onSelectionChanged");

        //Unset color on previous selection
        if (previous) {
                QAbstractGraphicsShapeItem * graphicItem=(QAbstractGraphicsShapeItem *) previous->data(1).toLongLong();
                if (graphicItem) {
                        graphicItem->setBrush(QBrush(QColor::fromHsv(graphicItem->data(11).toInt(),
                                                     graphicItem->data(12).toInt(),
                                                     graphicItem->data(13).toInt())));
                        graphicItem->setSelected(false);
                }
        }

        //Set highlight color on current selection
        if (current) {
                QAbstractGraphicsShapeItem * graphicItem=(QAbstractGraphicsShapeItem *) current->data(1).toLongLong();
                if (graphicItem) {
                        graphicItem->setBrush(QBrush(QApplication::palette().color(QPalette::Highlight)));
                        graphicItem->setSelected(true);
                }
        }
}

void SKGTableWithGraph::onDoubleClickGraph()
{
        if (scene) {
                //Get selection
                QList<QGraphicsItem *> selectedGraphItems=scene->selectedItems();
                if (selectedGraphItems.count()) {
                        emit cellDoubleClicked(selectedGraphItems[0]->data(1).toInt(), selectedGraphItems[0]->data(2).toInt());
                }
        }
}

void SKGTableWithGraph::onDoubleClick(int row, int column)
{
        emit cellDoubleClicked(row, column);
}

void SKGTableWithGraph::onSelectionChangedInGraph()
{
        _SKGTRACEIN(10, "SKGTableWithGraph::onSelectionChangedInGraph");
        if (scene) {
                //Get selection
                QList<QGraphicsItem *> selectedGraphItems=scene->selectedItems();
                if (selectedGraphItems.count()) {
                        ui.kTable->setCurrentCell(selectedGraphItems[0]->data(1).toInt(), selectedGraphItems[0]->data(2).toInt());
                }
        }
}

void SKGTableWithGraph::redrawGraph()
{
        SKGTRACEIN(10, "SKGTableWithGraph::redrawGraph");
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

        ui.graphicView->hide();
        ui.kTable->hide();

        //Recreate scene
        if (scene) {
                SKGTRACEIN(10, "SKGTableWithGraph::redrawGraph-remove scene");
                scene->clear();
                delete scene;
        }


        scene=new SKGGraphicsScene();
        {
                SKGTRACEIN(10, "SKGTableWithGraph::redrawGraph-create scene");
                //Get current selection
                int crow=ui.kTable->currentRow();
                int ccolumn=ui.kTable->currentColumn();

                int nbRows=ui.kTable->rowCount()-1; //Sum
                int nbColumns=ui.kTable->columnCount();
                int sumColumnIndex=nbColumns-1;
                if (nbColumns>2) {
                        nbColumns=nbColumns-2; //Sum and average
                        --sumColumnIndex;
                }
                QPen dotPen=QPen(Qt::DotLine);

                int mode=ui.kDisplayMode->currentIndex();
                bool inPositive=false;

                if (mode<2) {
                        ui.kAllPositive->show();
                        inPositive=(ui.kAllPositive->checkState()==Qt::Checked);
                } else ui.kAllPositive->hide();

                //Compute y limits
                double minLimit=0;
                double maxLimit=0;
                SKGTRACEL(30) << "mode=" << mode << endl;
                SKGTRACEL(30) << "crow=" << crow << endl;
                SKGTRACEL(30) << "ccolumn=" << ccolumn << endl;
                SKGTRACEL(30) << "nbRows=" << nbRows << endl;
                SKGTRACEL(30) << "nbColumns=" << nbColumns << endl;
                SKGTRACEL(30) << "sumColumnIndex=" << sumColumnIndex << endl;
                if (mode==0) {
                        //STACK
                        for (int x=0; x<nbRows; ++x) {
                                QTableWidgetItem* tableItem=ui.kTable->item(x,sumColumnIndex);
                                if (tableItem) {
                                        double val=tableItem->data(12).toDouble();
                                        if (inPositive) {
                                                maxLimit=qMax(maxLimit, fabs(val));
                                        } else {
                                                minLimit=qMin(minLimit, val);
                                                maxLimit=qMax(maxLimit, val);
                                        }
                                }
                        }
                } else if (mode==1) {
                        //HISTOGRAM
                        for (int x=0; x<nbRows; ++x) {
                                for (int y=0; y<nbColumns; ++y) {
                                        QTableWidgetItem* tableItem=ui.kTable->item(x,y);
                                        if (tableItem) {
                                                double val=tableItem->data(12).toDouble();
                                                if (inPositive) {
                                                        maxLimit=qMax(maxLimit, fabs(val));
                                                } else {
                                                        minLimit=qMin(minLimit, val);
                                                        maxLimit=qMax(maxLimit, val);
                                                }
                                        }
                                }
                        }
                }
                SKGTRACEL(30) << "minLimit=" << minLimit << endl;
                SKGTRACEL(30) << "maxLimit=" << maxLimit << endl;

                //Compute
                double width=10;
                double maxX=0;
                if (nbRows) {
                        if (mode==0) {
                                width=(maxLimit-minLimit)/nbRows;
                                maxX=width*nbRows+width;
                        } else if (mode==1) {
                                width=(maxLimit-minLimit)/(nbRows*(nbColumns-1));
                                maxX=width*(nbColumns-1)*nbRows+width;
                        }
                }
                SKGTRACEL(30) << "width=" << width << endl;
                SKGTRACEL(30) << "maxX=" << maxX << endl;

                double margin=width;
                if (mode!=2 && mode!=3) {//TODO for mode 2 or 3
                        scene->setSceneRect(-margin, -margin-maxLimit, maxX+margin, maxLimit-minLimit+2*margin);
                        scene->setItemIndexMethod(QGraphicsScene::NoIndex);

                        SKGTRACEL(10) << "Scene rect:" <<scene->sceneRect().x() << "," <<   scene->sceneRect().y()<< "," <<  scene->sceneRect().width()<< "," <<  scene->sceneRect().height() << endl;
                        SKGTRACEL(10) << "Items:" <<nbRows << "x" << (nbColumns-1) << "=" << nbRows*(nbColumns-1) << endl;
                }

                //Redraw scene
                int nbColInMode2=(nbColumns>2 ? sqrt(nbColumns-1)+.5 : 1);
                double ymin=0;
                double ymax=0;
                for (int x=0; x<nbRows; ++x) {
                        QString xname=ui.kTable->item(x, 0)->text();
                        double yPlus=0;
                        double yMoins=0;
                        for (int j=1; j<nbColumns; ++j) {
                                //Get column name
                                QString yname=ui.kTable->horizontalHeaderItem(j)->text();

                                //Get cell value
                                QTableWidgetItem* tableItem=ui.kTable->item(x,j);
                                if (tableItem) {
                                        double val=tableItem->data(12).toDouble();
                                        QString valstring=tableItem->text();

                                        int color_h=359*x/nbRows;
                                        int color_s=255-155*j/nbColumns;
                                        int color_v=255-155*j/nbColumns;

                                        QColor color;
                                        if (x==crow && j==ccolumn) color=QApplication::palette().color(QPalette::Highlight);
                                        else color=QColor::fromHsv(color_h,color_s,color_v);

                                        QBrush brush(color);

                                        QAbstractGraphicsShapeItem* graphItem=NULL;
                                        if (mode==0) {
                                                //STACK
                                                if (val>0 || inPositive) {
                                                        graphItem=scene->addRect(width*x, -yPlus, width, -fabs(val), QPen(), brush);
                                                        yPlus+=fabs(val);
                                                        if (yPlus>ymax) ymax=yPlus;
                                                } else {
                                                        graphItem=scene->addRect(width*x, -yMoins, width, -val, QPen(), brush);
                                                        yMoins+=val;
                                                        if (yMoins<ymin) ymin=yMoins;
                                                }
                                        } else if (mode==1) {
                                                //HISTOGRAM
                                                if (inPositive) {
                                                        graphItem=scene->addRect(width*((j-1)*nbRows+x), 0, width, -fabs(val), QPen(), brush);
                                                        if (fabs(val)>ymax) ymax=fabs(val);
                                                } else {
                                                        graphItem=scene->addRect(width*((j-1)*nbRows+x), 0, width, -val, QPen(), brush);
                                                        if (val>ymax) ymax=val;
                                                        if (val<ymin) ymin=val;
                                                }
                                        } else if (mode==2 || mode==3) {
                                                //PIE
                                                //Compute absolute sum of the column
                                                val=abs(val);
                                                double sumColumn=0;
                                                double previousSum=0;
                                                for (int x2=0; x2<nbRows; ++x2) {
                                                        QTableWidgetItem* tableItem=ui.kTable->item(x2,j);
                                                        if (tableItem) {
                                                                double absVal=abs(tableItem->data(12).toDouble());
                                                                sumColumn+=absVal;
                                                                if (x2<x) previousSum+=absVal;
                                                        }
                                                }

                                                if (sumColumn) {
                                                        int nbvals=xname.split(OBJECTSEPARATOR).count();
                                                        double pas=0;
                                                        double p=0;
                                                        if (mode==3) {
                                                                pas=100/(nbLevel+1);
                                                                p=pas*(nbLevel+1-nbvals);
                                                        }

                                                        QPainterPath path;
                                                        path.moveTo(120*((j-1)%nbColInMode2)+60, 120*floor((j-1)/nbColInMode2)+60);
                                                        path.arcTo (120*((j-1)%nbColInMode2)+10+p/2, 120*floor((j-1)/nbColInMode2)+10+p/2, 100-p, 100-p, 360*previousSum/sumColumn, 360*val/sumColumn);
                                                        if (mode==3 && nbvals<=nbLevel+1 && nbvals>1) {
                                                                p=pas*(nbLevel+1-nbvals+1);
                                                                path.moveTo(120*((j-1)%nbColInMode2)+60, 120*floor((j-1)/nbColInMode2)+60);
                                                                path.arcTo (120*((j-1)%nbColInMode2)+10+p/2, 120*floor((j-1)/nbColInMode2)+10+p/2, 100-p, 100-p, 360*previousSum/sumColumn, 360*val/sumColumn);
                                                        }
                                                        path.closeSubpath();
                                                        graphItem=scene->addPath (path, mode==3 ? QPen(Qt::white) : QPen(), brush);
                                                }
                                        }

                                        if (graphItem) {
                                                graphItem->setZValue(5);
                                                graphItem->setToolTip(xname+'\n'+yname+'\n'+valstring+'\n'+tableItem->toolTip());
                                                graphItem->setFlags(QGraphicsItem::ItemIsSelectable);
                                                graphItem->setData(1, x);
                                                graphItem->setData(2, j);
                                                graphItem->setData(11, color_h);
                                                graphItem->setData(12, color_s);
                                                graphItem->setData(13, color_v);
                                                graphItem->setCursor(Qt::PointingHandCursor);

                                                tableItem->setData(1, (qlonglong)  graphItem);
                                        }
                                } else {
                                        SKGTRACE << "WARNING: cell " << x << "," << j << " null" << endl;
                                }
                        }
                }

                //Draw axis
                if (nbRows) {
                        if (mode==2 || mode==3) {
                                int nbRowInMode2=((int) ((nbColumns-1)/nbColInMode2));
                                if (nbRowInMode2*nbColInMode2<nbColumns-1) ++nbRowInMode2;
                                for (int i=0; i<=nbColInMode2; ++i) {
                                        QGraphicsLineItem* item=scene->addLine(120*i, 0, 120*i, 120*nbRowInMode2);
                                        item->setPen(dotPen);
                                        item->setFlag(QGraphicsItem::ItemIsSelectable, false);
                                }
                                for (int i=0; i<=nbRowInMode2; ++i) {
                                        QGraphicsLineItem* item=scene->addLine(0, 120*i, 120*nbColInMode2, 120*i);
                                        item->setPen(dotPen);
                                        item->setFlag(QGraphicsItem::ItemIsSelectable, false);
                                }

                                for (int j=1; j<nbColumns; ++j) {
                                        //Get column name
                                        QString yname=ui.kTable->horizontalHeaderItem(j)->text();

                                        QGraphicsTextItem* textItem=scene->addText(yname);
                                        textItem->setPos(120*((j-1)%nbColInMode2)+2, 120*floor((j-1)/nbColInMode2)+2);
                                        textItem->scale(.5, .5);
                                        textItem->setFlag(QGraphicsItem::ItemIsSelectable, false);

                                }
                        } else {
                                QGraphicsLineItem* item=scene->addLine(0, -ymin+margin, 0, -ymax-margin);
                                item->setFlag(QGraphicsItem::ItemIsSelectable, false);

                                addArrow(QPointF(0, -ymax-margin), margin/2, 45, 90);

                                int ystep=100*((int) ((ymax-ymin)/1000));
                                if (ystep>0) {
                                        for (int y=ystep; y<ymax+margin; y=y+ystep) {
                                                item=scene->addLine(-margin, -y, maxX, -y);
                                                item->setPen(dotPen);
                                                item->setFlag(QGraphicsItem::ItemIsSelectable, false);

                                                QGraphicsTextItem* textItem=scene->addText(SKGServices::intToString(y));
                                                textItem->setPos(-margin, -y);
                                                textItem->scale(ystep/40, ystep/40);
                                                textItem->setFlag(QGraphicsItem::ItemIsSelectable, false);
                                        }
                                        for (int y=-ystep; y>ymin-margin; y=y-ystep) {
                                                item=scene->addLine(-margin, -y, maxX, -y);
                                                item->setPen(dotPen);
                                                item->setFlag(QGraphicsItem::ItemIsSelectable, false);

                                                QGraphicsTextItem* textItem=scene->addText(SKGServices::intToString(y));
                                                textItem->setPos(-margin, -y);
                                                textItem->scale(ystep/40, ystep/40);
                                                textItem->setFlag(QGraphicsItem::ItemIsSelectable, false);
                                        }
                                }
                                item=scene->addLine(-margin, 0, maxX, 0);
                                item->setFlag(QGraphicsItem::ItemIsSelectable, false);

                                addArrow(QPointF(maxX, 0), margin/2, 45, 180);
                        }
                }
        }
        {
                SKGTRACEIN(10, "SKGTableWithGraph::redrawGraph-add scene to view");
                ui.graphicView->setScene(scene);
                ui.graphicView->onZoomOriginal();
                ui.graphicView->show();
                ui.kTable->show();

                //Add selection event on scene
                connect(scene, SIGNAL(selectionChanged()), this, SLOT(onSelectionChangedInGraph()), Qt::QueuedConnection);
                connect(scene, SIGNAL(doubleClicked()), this, SLOT(onDoubleClickGraph()), Qt::QueuedConnection);
        }
        QApplication::restoreOverrideCursor();
}

void SKGTableWithGraph::addArrow(const QPointF& iPeak, double iSize, double iArrowAngle, double iDegree)
{
        if (scene) {
                QPolygonF pol;
                double radian=3.14*iArrowAngle/360;
                pol << QPointF(0, 0) << QPointF(iSize*cos(radian), iSize*sin(radian)) << QPointF(iSize*cos(radian), -iSize*sin(radian)) << QPointF(0, 0);
                QGraphicsPolygonItem * item=scene->addPolygon (pol, QPen(Qt::black), QBrush(Qt::black));
                item->rotate(iDegree);
                item->moveBy(iPeak.x(), iPeak.y());
                item->setFlag(QGraphicsItem::ItemIsSelectable, false);
        }
}

void SKGTableWithGraph::onExportCSV()
{
        _SKGTRACEIN(10, "SKGTableWithGraph::onExportCSV");
        QString fileName=SKGMainPanel::getSaveFileName(KUrl("kfiledialog:///IMPEXP"), "*.csv|"+i18n("CSV Files") , this);
        if (fileName.isEmpty()) return;

        {
                SKGError err;

                //Write file
                QFile file(fileName);
                if (!file.open(QIODevice::WriteOnly | QIODevice::Text)) {
                        err.setReturnCode(ERR_INVALIDARG);
                        err.setMessage(tr("Save file [%1] failed").arg(fileName));
                } else {
                        QTextStream out(&file);
                        QStringList dump=SKGServices::tableToDump(getTable(), SKGServices::DUMP_CSV);
                        int nbl=dump.count();
                        for (int i=0; i<nbl; ++i) {
                                out << dump[i] << endl;
                        }
                }

                //Close file
                file.close();

        }
        QDesktopServices::openUrl(QUrl(fileName));
}

void SKGTableWithGraph::onExportTXT()
{
        _SKGTRACEIN(10, "SKGTableWithGraph::onExportTXT");
        QString fileName=SKGMainPanel::getSaveFileName(KUrl("kfiledialog:///IMPEXP"), "*.txt|"+i18n("Text document") , this);
        if (fileName.isEmpty()) return;

        {
                SKGError err;

                //Write file
                QFile file(fileName);
                if (!file.open(QIODevice::WriteOnly | QIODevice::Text)) {
                        err.setReturnCode(ERR_INVALIDARG);
                        err.setMessage(tr("Save file [%1] failed").arg(fileName));
                } else {
                        QTextStream out(&file);
                        QStringList dump=SKGServices::tableToDump(getTable(), SKGServices::DUMP_TEXT);
                        int nbl=dump.count();
                        for (int i=0; i<nbl; ++i) {
                                out << dump[i] << endl;
                        }
                }

                //Close file
                file.close();

        }
        QDesktopServices::openUrl(QUrl(fileName));
}
#include "skgtablewithgraph.moc"

