/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
* This file is part of Skrooge and defines classes SKGImportExportManager.
*
* @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgimportexportmanager.h"
#include "skgtraces.h"
#include "skgservices.h"
#include "skgbankincludes.h"
#include "skgobjectbase.h"

#include <QRegExp>
#include <QFileInfo>
#include <QTextCodec>
#include <QCryptographicHash>
int SKGImportExportManager::nbOperationsNotImported=0;
int SKGImportExportManager::nbOperationsImported=0;
SKGError SKGImportExportManager::ofxError;

SKGImportExportManager::SKGImportExportManager(SKGDocumentBank* iDocument,
                ImportExportMode iMode,
                const QString& iFileName)
                :QObject(), document(iDocument), mode(iMode), fileName(iFileName),csvHeaderIndex(-1),
                defaultAccount(NULL), defaultUnit(NULL)
{}


SKGImportExportManager::~SKGImportExportManager()
{
        setDefaultAccount(NULL);
        setDefaultUnit(NULL);
        document=NULL;
        defaultAccount=NULL;
        defaultUnit=NULL;
}

SKGDocumentBank* SKGImportExportManager::getDocument()
{
        return document;
}

SKGError SKGImportExportManager::setCodec(const QString& iCodec)
{
        SKGError err;
        SKGTRACEINRC(10, "SKGImportExportManager::setCodec", err);
        codec=iCodec;
        return err;
}
SKGError SKGImportExportManager::setDefaultAccount(SKGAccountObject* iAccount)
{
        SKGError err;
        SKGTRACEINRC(10, "SKGImportExportManager::setDefaultAccount", err);
        if (defaultAccount) {
                delete defaultAccount;
                defaultAccount=NULL;
        }
        if (iAccount) defaultAccount=new SKGAccountObject(*iAccount);
        return err;
}

SKGError SKGImportExportManager::getDefaultAccount(SKGAccountObject& oAccount)
{
        SKGError err;
        SKGTRACEINRC(10, "SKGImportExportManager::getDefaultAccount", err);
        if (defaultAccount==NULL && document) {
                QFileInfo fInfo(fileName);
                QString name=fInfo.baseName();

                SKGAccountObject account;
                SKGBankObject bank(document);
                if (err.isSucceeded()) err=bank.setName(name);
                if (err.isSucceeded() && bank.load().isFailed()) err=bank.save(false); //Save only
                if (err.isSucceeded()) err=bank.addAccount(account);
                if (err.isSucceeded()) err=account.setName(name);
                if (err.isSucceeded() && account.load().isFailed()) err=account.save(false); //Save only
                if (err.isSucceeded()) defaultAccount=new SKGAccountObject(account);
                if (err.isSucceeded()) err=document->sendMessage(tr("Default account [%1] created for import").arg(name));
        }

        if (defaultAccount!=NULL)  oAccount=*defaultAccount;

        return err;
}

SKGError SKGImportExportManager::setDefaultUnit(SKGUnitObject* iUnit)
{
        SKGError err;
        SKGTRACEINRC(10, "SKGImportExportManager::setDefaultUnit", err);
        if (defaultUnit) {
                delete defaultUnit;
                defaultUnit=NULL;
        }
        if (iUnit) defaultUnit=new SKGUnitObject(*iUnit);
        return err;
}

SKGError SKGImportExportManager::getDefaultUnit(SKGUnitObject& oUnit, const QDate* iDate)
{
        SKGError err;
        SKGTRACEINRC(10, "SKGImportExportManager::getDefaultUnit", err);
        if (document && (defaultUnit==NULL || iDate)) {
                if (defaultUnit==NULL) {
                        delete defaultUnit;
                        defaultUnit=NULL;
                }

                //Do we have to found the best unit for a date ?
                QString wc="t_type IN ('1', '2', 'C')";
                if (iDate) {
                        //Yes
                        wc+=" AND d_MINDATE<'"+SKGServices::dateToSqlString(QDateTime(*iDate))+'\'';
                }

                //Check if a unit exist
                SKGObjectBase::SKGListSKGObjectBase listUnits;
                err=SKGObjectBase::getObjects(document, "v_unit",wc+" ORDER BY ABS(f_CURRENTAMOUNT-1) ASC" , listUnits);
                if (err.isSucceeded()) {
                        if (listUnits.count()==0) {
                                //Not found, we have to create one
                                QDateTime now=QDateTime::currentDateTime();
                                QString postFix=SKGServices::dateToSqlString(now);

                                SKGUnitObject unit(document);
                                QString name=tr("Unit for import %1").arg(postFix);
                                err=unit.setName(name);
                                if (unit.load().isFailed()) {
                                        if (err.isSucceeded()) err=unit.setSymbol(name);
                                        if (err.isSucceeded()) err=unit.save(false); //Save only

                                        SKGUnitValueObject unitval;
                                        if (err.isSucceeded()) err=unit.addUnitValue(unitval);
                                        if (err.isSucceeded()) err=unitval.setQuantity(1);
                                        if (err.isSucceeded()) err=unitval.setDate(now.date());
                                        if (err.isSucceeded()) err=unitval.save(false, false); //Save only without reload

                                        if (err.isSucceeded()) err=document->sendMessage(tr("Default unit [%1] created for import").arg(name));
                                }

                                if (err.isSucceeded()) defaultUnit=new SKGUnitObject(unit);
                        } else {
                                //Found, we can use it
                                defaultUnit=new SKGUnitObject((SKGUnitObject) listUnits.at(0));
                        }
                }
        }

        if (defaultUnit!=NULL) {
                oUnit=*defaultUnit;
        }

        return err;
}

SKGError SKGImportExportManager::importFile()
{
        SKGError err;
        SKGTRACEINRC(2, "SKGImportExportManager::importFile", err);
        err=SKGServices::executeSqliteOrder(document, "ANALYZE");
        if (err.isSucceeded()) {
                if (mode==SKGImportExportManager::QIF) err=SKGImportExportManager::importQIF();
                else if (mode==SKGImportExportManager::CSV) err=SKGImportExportManager::importCSV();
                else if (mode==SKGImportExportManager::CSVUNIT) err=SKGImportExportManager::importCSVUnit();
                else if (mode==SKGImportExportManager::OFX) err=SKGImportExportManager::importOFX();
                else {
                        err.setReturnCode(ERR_NOTIMPL);
                        err.setMessage(tr("This import mode is not yet implemented"));
                }
        }

        return err;
}

SKGError SKGImportExportManager::exportFile()
{
        SKGError err;
        SKGTRACEINRC(2, "SKGImportExportManager::exportFile", err);
        err=SKGServices::executeSqliteOrder(document, "ANALYZE");
        if (err.isSucceeded()) {
                if (mode==SKGImportExportManager::QIF) err=SKGImportExportManager::exportQIF();
                else if (mode==SKGImportExportManager::CSV) err=SKGImportExportManager::exportCSV();
                else {
                        err.setReturnCode(ERR_NOTIMPL);
                        err.setMessage(tr("This export mode is not yet implemented"));
                }
        }

        return err;
}


SKGError SKGImportExportManager::exportQIF()
{
        SKGError err;
        SKGTRACEINRC(2, "SKGImportExportManager::exportQIF", err);
        SKGTRACEL(10) << "Input filename=" << fileName << endl;

        if (document) {
                //Open file
                QFile file(fileName);
                if (!file.open(QIODevice::WriteOnly)) {
                        err.setReturnCode(ERR_INVALIDARG);
                        err.setMessage(tr("Save file [%1] failed").arg(fileName));
                } else {
                        QTextStream stream(&file);
                        SKGTRACEL(2) << "Text codec=" << QTextCodec::codecForLocale ()->name() << endl;

                        err=document->beginTransaction("#INTERNAL#", 2);
                        if (err.isSucceeded()) {

                                //Export categories
                                SKGObjectBase::SKGListSKGObjectBase categories;
                                if (err.isSucceeded()) err=SKGObjectBase::getObjects(document, "v_category_display_tmp", "1=1 ORDER BY t_fullname, id", categories);
                                int nbcat=categories.count();
                                if (err.isSucceeded() && nbcat) {
                                        stream << "!Type:Cat\n";
                                        err=document->beginTransaction("#INTERNAL#", nbcat);
                                        for (int i=0; err.isSucceeded() && i<nbcat; ++i) {
                                                SKGCategoryObject cat=categories.at(i);
                                                stream << "N" << cat.getFullName() << endl;
                                                if (SKGServices::stringToDouble(cat.getAttribute("f_REALCURRENTAMOUNT"))<0) {
                                                        stream << "E" << endl;
                                                } else {
                                                        stream << "I" << endl;
                                                }
                                                stream << "^" << endl;
                                                if (err.isSucceeded()) err=document->stepForward(i+1);
                                        }

                                        if (err.isSucceeded()) err=document->endTransaction(true);
                                        else  document->endTransaction(false);
                                }
                                if (err.isSucceeded()) err=document->stepForward(1);

                                //Get operations
                                QString currentAccountName;
                                SKGObjectBase::SKGListSKGObjectBase operations;
                                if (err.isSucceeded()) err=SKGObjectBase::getObjects(document, "v_operation_display", "1=1 ORDER BY t_ACCOUNT, d_date, id", operations);
                                int nb=operations.count();
                                if (err.isSucceeded()) {
                                        err=document->beginTransaction("#INTERNAL#", nb);
                                        for (int i=0; err.isSucceeded() && i<nb; ++i) {
                                                SKGOperationObject operation=operations.at(i);

                                                //Get account name
                                                QString accountName=operation.getAttribute("t_ACCOUNT");

                                                //In the same account ?
                                                if (accountName!=currentAccountName) {
                                                        SKGAccountObject account(document);
                                                        account.setName(accountName);
                                                        account.load();

                                                        SKGBankObject bank;
                                                        account.getBank(bank);

                                                        //Write header
                                                        stream << "!Account\n";
                                                        stream << 'N' << accountName << endl;
                                                        stream << 'T' << account.getType() << endl;
                                                        QString number=bank.getNumber();
                                                        QString bnumber=account.getAgencyNumber();
                                                        QString cnumber=account.getNumber();
                                                        if (!bnumber.isEmpty()) {
                                                                if (!number.isEmpty()) number+='-';
                                                                number+=bnumber;
                                                        }
                                                        if (!cnumber.isEmpty()) {
                                                                if (!number.isEmpty()) number+='-';
                                                                number+=cnumber;
                                                        }
                                                        stream << 'D' << number << endl;
                                                        //stream << "/"      Statement balance date
                                                        //stream << "$"      Statement balance amount
                                                        stream << '^' << endl;
                                                        currentAccountName=accountName;

                                                        stream << "!Type:Bank\n";
                                                }


                                                //Write operation
                                                /*
                                                DONE	D      Date
                                                DONE	T      Amount
                                                N/A	U      Transaction amount (higher possible value than T)
                                                DONE	C      Cleared status
                                                DONE	N      Number (check or reference number)
                                                DONE	P      Payee/description
                                                DONE	M      Memo
                                                N/A	A      Address (up to 5 lines; 6th line is an optional message)
                                                DONE	L      Category (category/class or transfer/class)
                                                DONE	S      Category in split (category/class or transfer/class)
                                                N/A	E      Memo in split
                                                DONE	$      Dollar amount of split
                                                N/A	%      Percentage of split if percentages are used
                                                N/A	F      Reimbursable business expense flag
                                                N/A	X      Small Business extensions
                                                DONE	^      End of entry
                                                */

                                                stream << 'D' << SKGServices::dateToSqlString(QDateTime(operation.getDate())) << endl;
                                                stream << 'T' << SKGServices::doubleToString(operation.getCurrentAmount()) << endl;

                                                int number=operation.getNumber();
                                                if (number) stream << 'N' << operation.getNumber() << endl;

                                                QString payee=operation.getPayee();
                                                if (payee.length()) stream << 'P' << payee << endl;

                                                QString memo=operation.getMode()+"  "+operation.getComment();
                                                memo=memo.trimmed();
                                                if (memo.length()) stream << 'M' << memo << endl;

                                                SKGOperationObject::OperationStatus status=operation.getStatus();
                                                stream << "C" << (status==SKGOperationObject::POINTED ? "C": (status==SKGOperationObject::CHECKED ? "R": "" )) << endl;

                                                //Get sub operations
                                                SKGObjectBase::SKGListSKGObjectBase suboperations;
                                                err=operation.getSubOperations(suboperations);
                                                if (err.isSucceeded()) {
                                                        int nb=suboperations.size();
                                                        if (nb>1) {
                                                                //Splitted operation
                                                                for (int i=0; i<nb; ++i) {
                                                                        SKGSubOperationObject suboperation=suboperations.at(i);
                                                                        SKGCategoryObject cat;
                                                                        suboperation.getCategory(cat);

                                                                        QString category=cat.getFullName();
                                                                        if (category.length()) stream << 'S' << category << endl;
                                                                        stream << '$' << SKGServices::doubleToString(suboperation.getQuantity()) << endl;
                                                                }
                                                        } else if (nb==1) {
                                                                //Simple operation
                                                                SKGSubOperationObject suboperation=suboperations.at(0);
                                                                SKGCategoryObject cat;
                                                                suboperation.getCategory(cat);

                                                                QString category=cat.getFullName();
                                                                if (category.length()) stream << 'L' << category << endl;
                                                        }
                                                }

                                                stream << '^' << endl;
                                                if (err.isSucceeded()) err=document->stepForward(i+1);
                                        }

                                        if (err.isSucceeded()) err=document->endTransaction(true);
                                        else  document->endTransaction(false);
                                }
                                if (err.isSucceeded()) err=document->stepForward(2);
                                if (err.isSucceeded()) err=document->endTransaction(true);
                                else  document->endTransaction(false);
                        }
                }

                //Close file
                file.close();
        }

        if (err.isFailed()) {
                err.addError(ERR_FAIL, tr("%1 failed").arg("SKGImportExportManager::exportQIF"));
        }
        return err;
}

SKGError SKGImportExportManager::exportCSV()
{
        SKGError err;
        SKGTRACEINRC(2, "SKGDocumentBank::exportCSV", err);
        SKGTRACEL(10) << "Input filename=" << fileName << endl;

        if (document) {
                //Open file
                QFile file(fileName);
                if (!file.open(QIODevice::WriteOnly | QIODevice::Text)) {
                        err.setReturnCode(ERR_INVALIDARG);
                        err.setMessage(tr("Save file [%1] failed").arg(fileName));
                } else {
                        QTextStream out(&file);
                        err=SKGServices::dumpSelectSqliteOrder((SKGDocument*) document,
                                                               "SELECT d_date as date, t_ACCOUNT as account, i_number as number, t_mode as mode, "
                                                               "t_payee as payee, t_comment as comment, f_QUANTITY as quantity, "
                                                               "t_UNIT as unit, f_CURRENTAMOUNT as amount, t_TYPEEXPENSE as sign, t_CATEGORY as category, t_status as status, "
                                                               "t_bookmarked as bookmarked "
                                                               "FROM v_operation_display", &out, SKGServices::DUMP_CSV);
                }

                //Close file
                file.close();
        }

        if (err.isFailed()) {
                err.addError(ERR_FAIL, tr("%1 failed").arg("SKGImportExportManager::exportQIF"));
        }
        return err;
}

SKGError SKGImportExportManager::exportOFX()
{
        SKGError err;
        SKGTRACEINRC(2, "SKGDocumentBank::exportOFX", err);
        SKGTRACEL(10) << "Input filename=" << fileName << endl;

        return err;
}

QStringList SKGImportExportManager::getCSVMappingFromLine(const QString& iLine)
{
        QStringList output;
        QString line=iLine.trimmed();

        //Split first line
        QStringList csvAttributes=SKGServices::splitCSVLine(line);
        int nb=csvAttributes.count();
        //TODO: generic & international stuff
        for (int i=0; i<nb; ++i) {
                QString att=csvAttributes[i].toLower();
                if (att.contains("date") && !output.contains("date")) {
                        output.push_back("date");
                } else if (QRegExp("(?:number|num?ro)").indexIn(att)!=-1 && !output.contains("number")) {
                        output.push_back("number");
                } else if ((att=="mode" || att=="type") && !output.contains("mode")) {
                        output.push_back("mode");
                } else if ((att=="payee" || att=="tiers") && !output.contains("payee")) {
                        output.push_back("payee");
                } else if (QRegExp("(comment|libell?|d?tail|info)").indexIn(att)!=-1 && !output.contains("comment")) {
                        output.push_back("comment");
                } else if ((att=="status" || att=="pointage") && !output.contains("status")) {
                        output.push_back("status");
                } else if (att=="bookmarked" && !output.contains("bookmarked")) {
                        output.push_back("bookmarked");
                } else if (att=="account" && !output.contains("account")) {
                        output.push_back("account");
                } else if (QRegExp("cat\\w*gor\\w*").indexIn(att)!=-1 && !output.contains("category")) {
                        output.push_back("category");
                } else if ((att=="valeur" || att=="value" || att=="amount" || att.contains("montant")) && !output.contains("amount")) {
                        output.push_back("amount");
                } else if (att=="quantity" && !output.contains("quantity")) {
                        output.push_back("quantity");
                } else if (att=="unit" && !output.contains("unit")) {
                        output.push_back("unit");
                } else if ((att=="sign" || att=="sens") && !output.contains("sign")) {
                        output.push_back("sign");
                } else {
                        output.push_back(""); //To ignore this column
                }
        }
        return output;
}

SKGError SKGImportExportManager::setCSVMapping(const QStringList* iCSVMapping)
{
        SKGError err;
        SKGTRACEINRC(10, "SKGImportExportManager::setCSVMapping", err);

        csvMapping.clear();

        if (iCSVMapping==NULL) {
                //Automatic build
                //Open file
                QFile file(fileName);
                if (!file.open(QIODevice::ReadOnly)) {
                        err.setReturnCode(ERR_INVALIDARG);
                        err.setMessage(tr("Open file [%1] failed").arg(fileName));
                } else {
                        QTextStream stream(&file);
                        if (!codec.isEmpty()) stream.setCodec(codec.toAscii().constData());

                        //Ignore useless lines
                        int headerIndex=getCSVHeaderIndex();
                        for (int i=0; i<headerIndex; ++i)
                                stream.readLine();

                        //Get mapping
                        if (!stream.atEnd()) {
                                csvMapping=getCSVMappingFromLine(stream.readLine());
                        } else {
                                err.setReturnCode(ERR_INVALIDARG);
                        }

                        //close file
                        file.close();
                }
        } else {
                //Manual build
                csvMapping=*iCSVMapping;
        }

        if (err.isSucceeded()) {
                //Check if mandatory attributes have been found
                if (!csvMapping.contains("date") || !csvMapping.contains("amount"))
                        err=SKGError(ERR_FAIL, tr("CSV mapping must contain columns date and amount at least"));
        }

        return err;
}

QStringList SKGImportExportManager::getCSVMapping() const
{
        SKGTRACEIN(10, "SKGImportExportManager::getCSVMapping");
        return csvMapping;
}

SKGError SKGImportExportManager::setCSVHeaderIndex(int iIndex)
{
        SKGError err;
        SKGTRACEINRC(10, "SKGImportExportManager::setCSVHeaderIndex", err);

        if (iIndex==-1) {
                //Automatic build
                //Open file
                QFile file(fileName);
                if (!file.open(QIODevice::ReadOnly)) {
                        err.setReturnCode(ERR_INVALIDARG);
                        err.setMessage(tr("Open file [%1] failed").arg(fileName));
                } else {
                        QTextStream stream(&file);
                        if (!codec.isEmpty()) stream.setCodec(codec.toAscii().constData());

                        int i=0;
                        csvHeaderIndex=-1;
                        while (!stream.atEnd() && csvHeaderIndex==-1) {
                                //Read line
                                QStringList map=getCSVMappingFromLine(stream.readLine());
                                if (map.contains("date") && map.contains("amount")) csvHeaderIndex=i;

                                ++i;
                        }

                        //close file
                        file.close();
                }
        } else {
                //Manual build
                csvHeaderIndex=iIndex;
        }

        return err;
}

int SKGImportExportManager::getCSVHeaderIndex()
{
        SKGTRACEIN(10, "SKGImportExportManager::getCSVHeaderIndex");
        if (csvHeaderIndex==-1) setCSVHeaderIndex(-1);
        return csvHeaderIndex;
}

SKGError SKGImportExportManager::importCSVUnit()
{
        SKGError err;
        SKGTRACEINRC(2, "SKGImportExportManager::importCSVUnit", err);
        SKGTRACEL(10) << "Input filename=" << fileName << endl;

        if (document) {
                //Begin transaction
                err=document->beginTransaction("#INTERNAL#", 3);
                if (err.isSucceeded()) {
                        //File name is the name of the unit
                        QFileInfo fInfo(fileName);
                        QString unitName=fInfo.baseName();

                        //Default mapping
                        if (getCSVMapping().count()==0) {
                                err= setCSVMapping(NULL);
                                if (err.isSucceeded()) err=document->sendMessage(tr("Use automatic CSV mapping detection"));
                        }

                        //Step 1 done
                        if (err.isSucceeded()) err=document->stepForward(1);

                        //Open file
                        if (err.isSucceeded()) {
                                QFile file(fileName);
                                if (!file.open(QIODevice::ReadOnly | QIODevice::Text)) {
                                        err.setReturnCode(ERR_INVALIDARG);
                                        err.setMessage(tr("Open file [%1] failed").arg(fileName));
                                } else {
                                        QTextStream stream(&file);
                                        if (!codec.isEmpty()) stream.setCodec(codec.toAscii().constData());

                                        //Ignore useless lines
                                        int headerIndex=getCSVHeaderIndex();
                                        for (int i=0; i<=headerIndex; ++i)
                                                stream.readLine();

                                        //Get data column
                                        QStringList dates;
                                        QStringList lines;
                                        int posdate=csvMapping.indexOf("date");
                                        if (posdate!=-1) {
                                                while (!stream.atEnd()) {
                                                        //Read line
                                                        QString line=stream.readLine().trimmed();
                                                        if (!line.isEmpty()) {
                                                                lines.push_back(line);

                                                                //Get date
                                                                QStringList field=SKGServices::splitCSVLine(line);
                                                                if (posdate<field.count()) dates.push_back(field.at(posdate));
                                                        }
                                                }
                                        }

                                        //close file
                                        file.close();

                                        //Select dateformat
                                        QString dateFormat=SKGServices::getDateFormat(dates);
                                        if (dateFormat.isEmpty()) {
                                                err.setReturnCode(ERR_FAIL);
                                                err.setMessage(tr("Date format not supported"));
                                        }
                                        if (err.isSucceeded())
                                                err=document->sendMessage(tr("Import of [%1] with code [%2] and date format [%3]").arg(fileName).arg(codec).arg(dateFormat));

                                        //Step 2 done
                                        if (err.isSucceeded()) err=document->stepForward(2);

                                        //Treat all lines
                                        if (err.isSucceeded()) {
                                                int nb=lines.size();
                                                err=document->beginTransaction("#INTERNAL#", nb);

                                                //Save last mapping used in a settings
                                                QString mappingDesc;
                                                int nbMap=csvMapping.count();
                                                for (int i=0; i<nbMap;++i) {
                                                        if (i) mappingDesc+='|';
                                                        mappingDesc+=csvMapping.at(i);
                                                }
                                                if (err.isSucceeded()) err=document->setParameter("SKG_LAST_CSV_UNIT_MAPPING_USED", mappingDesc);

                                                int posdate=csvMapping.indexOf("date");
                                                int posvalue=csvMapping.indexOf("amount");
                                                if (posdate!=-1 && posvalue!=-1) {
                                                        for (int i = 0; err.isSucceeded() && i < nb; ++i) {
                                                                QStringList atts=SKGServices::splitCSVLine(lines.at(i));
                                                                err=document->addOrModifyUnitValue(unitName,
                                                                                                   SKGServices::stringToTime(SKGServices::dateToSqlString(atts.at(posdate), dateFormat)).date(),
                                                                                                   SKGServices::stringToDouble(atts.at(posvalue)));

                                                                if (err.isSucceeded()) err=document->stepForward(i+1);
                                                        }
                                                }

                                                if (err.isSucceeded()) err=document->endTransaction(true);
                                                else  document->endTransaction(false);

                                                //Lines treated
                                                if (err.isSucceeded()) err=document->stepForward(3);
                                        }
                                }
                        }
                }
                if (err.isSucceeded()) err=document->endTransaction(true);
                else  document->endTransaction(false);
        }

        if (err.isFailed()) {
                err.addError(ERR_FAIL, tr("%1 failed").arg("SKGImportExportManager::importCSVUnit"));
        }

        return err;
}

SKGError SKGImportExportManager::importCSV()
{
        SKGError err;
        SKGTRACEINRC(2, "SKGImportExportManager::importCSV", err);
        SKGTRACEL(10) << "Input filename=" << fileName << endl;

        //Begin transaction
        if (document) {
                err=document->beginTransaction("#INTERNAL#", 3);
                if (err.isSucceeded()) {
                        //Create account if needed
                        QDateTime now=QDateTime::currentDateTime();
                        QString postFix=SKGServices::dateToSqlString(now);

                        //Default mapping
                        if (getCSVMapping().count()==0) {
                                err= setCSVMapping(NULL);
                                if (err.isSucceeded()) err=document->sendMessage(tr("Use automatic CSV mapping detection"));
                        }

                        //Step 1 done
                        if (err.isSucceeded()) err=document->stepForward(1);

                        //Open file
                        if (err.isSucceeded()) {
                                QFile file(fileName);
                                if (!file.open(QIODevice::ReadOnly | QIODevice::Text)) {
                                        err.setReturnCode(ERR_INVALIDARG);
                                        err.setMessage(tr("Open file [%1] failed").arg(fileName));
                                } else {
                                        QTextStream stream(&file);
                                        if (!codec.isEmpty()) stream.setCodec(codec.toAscii().constData());

                                        //Ignore useless lines
                                        int headerIndex=getCSVHeaderIndex();
                                        for (int i=0; i<=headerIndex; ++i)
                                                stream.readLine();

                                        //Get data column
                                        QStringList dates;
                                        QStringList lines;
                                        int posdate=csvMapping.indexOf("date");
                                        if (posdate!=-1) {
                                                while (!stream.atEnd()) {
                                                        //Read line
                                                        QString line=stream.readLine().trimmed();
                                                        if (!line.isEmpty()) {
                                                                lines.push_back(line);

                                                                //Get date
                                                                QStringList field=SKGServices::splitCSVLine(line);
                                                                if (posdate<field.count()) dates.push_back(field.at(posdate));
                                                        }
                                                }
                                        }

                                        //close file
                                        file.close();

                                        //Select dateformat
                                        QString dateFormat=SKGServices::getDateFormat(dates);
                                        if (dateFormat.isEmpty()) {
                                                err.setReturnCode(ERR_FAIL);
                                                err.setMessage(tr("Date format not supported"));
                                        }
                                        if (err.isSucceeded())
                                                err=document->sendMessage(tr("Import of [%1] with code [%2] and date format [%3]").arg(fileName).arg(codec).arg(dateFormat));

                                        //Step 2 done
                                        if (err.isSucceeded()) err=document->stepForward(2);

                                        //Treat all lines
                                        if (err.isSucceeded()) {
                                                int nb=lines.size();
                                                err=document->beginTransaction("#INTERNAL#", nb);

                                                //Save last mapping used in a settings
                                                QString mappingDesc;
                                                int nbMap=csvMapping.count();
                                                for (int i=0; i<nbMap;++i) {
                                                        if (i) mappingDesc+='|';
                                                        mappingDesc+=csvMapping.at(i);
                                                }
                                                if (err.isSucceeded()) err=document->setParameter("SKG_LAST_CSV_MAPPING_USED", mappingDesc);

                                                SKGUnitObject defUnit;
                                                SKGAccountObject defAccount;
                                                int nbOperationsNotImported=0;
                                                for (int i = 0; err.isSucceeded() && i < nb; ++i) {
                                                        SKGOperationObject currentOperation(document);
                                                        SKGSubOperationObject currentSubOperation(document);

                                                        //Valuate mandatory attribute with default value
                                                        if (csvMapping.indexOf("unit")==-1) {
                                                                err=getDefaultUnit(defUnit);
                                                                if (err.isSucceeded()) err=currentOperation.setUnit(defUnit);
                                                        }
                                                        if (err.isSucceeded() && csvMapping.indexOf("account")==-1) {
                                                                err=getDefaultAccount(defAccount);
                                                                if (err.isSucceeded()) err=currentOperation.setParentAccount(defAccount);
                                                        }

                                                        QString line=lines.at(i);
                                                        QByteArray hash = QCryptographicHash::hash(line.toUtf8(), QCryptographicHash::Md5);

                                                        SKGObjectBase opWithThisHash;
                                                        if (SKGObjectBase::getObject(document, "operation", "t_imported='Y' AND t_import_id='"+QString(hash.toHex())+'\'', opWithThisHash).isSucceeded()) nbOperationsNotImported++;
                                                        else {
                                                                QStringList atts=SKGServices::splitCSVLine(line);
                                                                int nbcol=csvMapping.count();
                                                                for (int c=0; err.isSucceeded() && c<nbcol; ++c) {
                                                                        QString col=csvMapping[c];
                                                                        int pos=csvMapping.indexOf(col);
                                                                        QString val;
                                                                        if (pos!=-1) val=atts.at(pos).trimmed();
                                                                        if (col=="date") {
                                                                                err=currentOperation.setDate(SKGServices::stringToTime(SKGServices::dateToSqlString(val, dateFormat)).date());
                                                                        } else if (col=="number") {
                                                                                if (!val.isEmpty()) err=currentOperation.setNumber(SKGServices::stringToInt(val));
                                                                        } else if (col=="mode") {
                                                                                err=currentOperation.setMode(val);
                                                                        } else if (col=="payee") {
                                                                                err=currentOperation.setPayee(val);
                                                                        } else if (col=="comment") {
                                                                                err=currentOperation.setComment(val);
                                                                        } else if (col=="status") {
                                                                                err=currentOperation.setStatus(val=="C" ?  SKGOperationObject::CHECKED : val=="P" ?  SKGOperationObject::POINTED :SKGOperationObject::NONE);
                                                                        } else if (col=="bookmarked") {
                                                                                err=currentOperation.bookmark(val=="Y");
                                                                        }  else if (col=="amount" && !csvMapping.contains("quantity")) {
                                                                                err=currentSubOperation.setQuantity(SKGServices::stringToDouble(val));
                                                                        }  else if (col=="quantity") {
                                                                                err=currentSubOperation.setQuantity(SKGServices::stringToDouble(val));
                                                                        }  else if (col=="sign") {
                                                                                if (val=="DEBIT" || val=="-") { // krazy:exclude=doublequote_chars
                                                                                        double cval=currentSubOperation.getQuantity();
                                                                                        if (cval>0) err=currentSubOperation.setQuantity(-cval);
                                                                                }
                                                                        } else if (col=="unit") {
                                                                                //Looking for unit
                                                                                SKGUnitObject unit(document);
                                                                                if (val!=defUnit.getName()) { //For performance
                                                                                        err=unit.setName(val);
                                                                                        if (err.isSucceeded()) err=unit.setSymbol(val);
                                                                                        if (err.isSucceeded() && unit.load().isFailed())  err=unit.save(false); //Save only

                                                                                        //This unit is now the default one, it's better for performance
                                                                                        defUnit=unit;
                                                                                } else {
                                                                                        unit=defUnit;
                                                                                }

                                                                                SKGUnitValueObject unitval;
                                                                                if (err.isSucceeded()) err=unit.addUnitValue(unitval);
                                                                                if (err.isSucceeded()) {
                                                                                        int posAmount=csvMapping.indexOf("amount");
                                                                                        int posQuantity=csvMapping.indexOf("quantity");
                                                                                        if (posAmount!=-1 && posQuantity!=-1) {
                                                                                                err=unitval.setQuantity(SKGServices::stringToDouble(atts.at(posAmount))/SKGServices::stringToDouble(atts.at(posQuantity)));
                                                                                        } else {
                                                                                                err=unitval.setQuantity(1);
                                                                                        }
                                                                                }
                                                                                if (err.isSucceeded()) err=unitval.setDate(now.date());
                                                                                if (err.isSucceeded()) err=unitval.save();
                                                                                if (err.isSucceeded()) err=currentOperation.setUnit(unit);
                                                                        } else if (col=="account") {
                                                                                //Looking for account
                                                                                if (val!=defAccount.getName()) { //For performance
                                                                                        SKGAccountObject account(document);
                                                                                        account.setName(val);
                                                                                        err=account.load();
                                                                                        if (err.isFailed()) {
                                                                                                //Not found, we have to create one
                                                                                                SKGBankObject bank(document);
                                                                                                QString name=tr("Bank for import %1").arg(postFix);
                                                                                                err=bank.setName(name);
                                                                                                if (err.isSucceeded() && bank.load().isFailed()) {
                                                                                                        err=bank.save(false);  //Save only
                                                                                                        if (err.isSucceeded()) err=document->sendMessage(tr("Default bank [%1] created for import").arg(name));
                                                                                                }
                                                                                                if (err.isSucceeded()) err=bank.addAccount(account);
                                                                                                if (err.isSucceeded()) err=account.setName(val);
                                                                                                if (err.isSucceeded() && account.load().isFailed())  err=account.save(false);  //Save only
                                                                                        }

                                                                                        //This account is now the default one, it's better for performance
                                                                                        defAccount=account;
                                                                                }
                                                                                if (err.isSucceeded()) err=currentOperation.setParentAccount(defAccount);
                                                                        } else if (col=="category") {
                                                                                //Set Category
                                                                                if (!val.isEmpty()) {
                                                                                        SKGCategoryObject Category;
                                                                                        val.replace('/', OBJECTSEPARATOR);
                                                                                        val.replace(':', OBJECTSEPARATOR);
                                                                                        val.replace(',', OBJECTSEPARATOR);
                                                                                        val.replace(';', OBJECTSEPARATOR);
                                                                                        err=SKGCategoryObject::createPathCategory(document,val, Category);
                                                                                        if (err.isSucceeded())  err=currentSubOperation.setCategory(Category);
                                                                                }
                                                                        }

                                                                }

                                                                if (err.isSucceeded()) err=currentOperation.setAttribute("t_imported", "P");
                                                                if (err.isSucceeded()) err=currentOperation.setImportID(hash.toHex());

                                                                if (err.isSucceeded()) err=currentOperation.save(false);  //Save only
                                                                if (err.isSucceeded()) err=currentSubOperation.setParentOperation(currentOperation);
                                                                if (err.isSucceeded()) err=currentSubOperation.save(false, false);  //Save only without reload
                                                        }
                                                        if (err.isSucceeded() && i%20==0) err = SKGServices::executeSqliteOrder(document, "ANALYZE");
                                                        if (err.isSucceeded()) err=document->stepForward(i+1);
                                                }
                                                if (err.isSucceeded()) err = SKGServices::executeSqliteOrder(document, "UPDATE operation SET t_imported='Y' WHERE t_imported='P'");

                                                if (err.isSucceeded()) err=document->endTransaction(true);
                                                else  document->endTransaction(false);

                                                if (err.isSucceeded() && nbOperationsNotImported)
                                                        err=document->sendMessage(tr("%1 operation(s) not imported because already existing").arg(nbOperationsNotImported));

                                                //Lines treated
                                                if (err.isSucceeded()) err=document->stepForward(3);
                                        }
                                }
                        }
                }
                if (err.isSucceeded()) err=document->endTransaction(true);
                else  document->endTransaction(false);
        }

        if (err.isFailed()) {
                err.addError(ERR_FAIL, tr("%1 failed").arg("SKGImportExportManager::importCSV"));
        }

        return err;
}

SKGError SKGImportExportManager::importOFX()
{
        SKGError err;
        SKGTRACEINRC(2, "SKGDocumentBank::importOFX", err);
        SKGTRACEL(10) << "Input filename=" << fileName << endl;

        if (document) {
                err=document->beginTransaction("#INTERNAL#");
                if (err.isSucceeded()) {
                        SKGImportExportManager::nbOperationsNotImported=0;
                        SKGImportExportManager::nbOperationsImported=0;
                        SKGImportExportManager::ofxError=SKGError();

                        LibofxContextPtr ctx = libofx_get_new_context();

                        ofx_set_transaction_cb(ctx, ofxTransactionCallback, this);
                        //ofx_set_statement_cb(ctx, ofxStatementCallback, this);
                        ofx_set_account_cb(ctx, ofxAccountCallback, this);
                        //ofx_set_security_cb(ctx, ofxSecurityCallback, this);
                        int rc=libofx_proc_file(ctx, fileName.toLatin1(), AUTODETECT);
                        if (rc)  err=SKGError(ERR_FAIL, tr("Import OFX file [%1] failed").arg(fileName));
                        if (err.isSucceeded()) err=SKGImportExportManager::ofxError;

                        if (err.isSucceeded()) err=document->endTransaction(true);
                        else  document->endTransaction(false);

                        if (err.isSucceeded() && SKGImportExportManager::nbOperationsNotImported)
                                err=document->sendMessage(tr("%1 operation(s) not imported because already existing").arg(SKGImportExportManager::nbOperationsNotImported));
                }
        }
        return err;
}

int SKGImportExportManager::ofxAccountCallback(struct OfxAccountData data, void * pv)
{
        SKGTRACEINRC(5, "SKGImportExportManager::ofxAccountCallback", SKGImportExportManager::ofxError);
        if (SKGImportExportManager::ofxError.isFailed()) return 0;

        SKGImportExportManager* impotExporter=(SKGImportExportManager*) pv;
        if (!impotExporter) return 0;
        SKGDocumentBank* doc = impotExporter->getDocument();
        if (!doc) return 0;

        SKGObjectBase tmp;
        if (data.account_id_valid==true) {
                QString agencyNumber;
                QString accountNumber=data.account_id;
                QString bankNumber=data.bank_id;
                if (accountNumber.startsWith(bankNumber+' ')) {
                        accountNumber=accountNumber.right(accountNumber.length()-bankNumber.length()-1);
                        QStringList splitNumbers = accountNumber.split(' ');
                        if (splitNumbers.count()==2) {
                                agencyNumber=splitNumbers.at(0);
                                accountNumber=splitNumbers.at(1);
                        }

                }

                //Check if account is already existing
                SKGAccountObject account;
                SKGImportExportManager::ofxError= SKGObjectBase::getObject(doc, "v_account", "t_number='"+accountNumber+'\'', tmp);
                if (SKGImportExportManager::ofxError.isSucceeded()) {
                        //Already existing
                        account=tmp;
                        SKGImportExportManager::ofxError=impotExporter->setDefaultAccount(&account);
                } else {
                        //Not existing
                        QString bankId=(data.bank_id_valid == true ? data.bank_id : (data.broker_id_valid == true ? data.broker_id : ""));
                        if (!bankId.isEmpty()) {
                                //Check if bank is already existing
                                SKGBankObject bank;
                                SKGImportExportManager::ofxError= SKGObjectBase::getObject(doc, "v_bank", "t_bank_number='"+bankId+'\'', tmp);
                                if (SKGImportExportManager::ofxError.isSucceeded()) {
                                        //Already existing
                                        bank=tmp;

                                } else {
                                        //Create new bank
                                        bank=SKGBankObject(doc);
                                        SKGImportExportManager::ofxError=bank.setName(bankId);
                                        if (SKGImportExportManager::ofxError.isSucceeded()) SKGImportExportManager::ofxError=bank.setNumber(data.bank_id);
                                        if (SKGImportExportManager::ofxError.isSucceeded()) SKGImportExportManager::ofxError=bank.save();
                                }

                                //Create new account
                                QString name=data.account_name;
                                if (name.isEmpty()) name=data.account_id;

                                if (SKGImportExportManager::ofxError.isSucceeded()) SKGImportExportManager::ofxError=bank.addAccount(account);
                                if (SKGImportExportManager::ofxError.isSucceeded()) SKGImportExportManager::ofxError=account.setName(name);
                                if (SKGImportExportManager::ofxError.isSucceeded())  SKGImportExportManager::ofxError=account.setNumber(accountNumber);
                                if (SKGImportExportManager::ofxError.isSucceeded()) SKGImportExportManager::ofxError=account.setAgencyNumber(agencyNumber);
                                if (SKGImportExportManager::ofxError.isSucceeded()) SKGImportExportManager::ofxError=account.setComment(data.account_name);
                                SKGAccountObject::AccountType type=SKGAccountObject::CURRENT;
                                if (data.account_type_valid==true) {
                                        switch (data.account_type) {
                                        case OfxAccountData::OFX_CHECKING:
                                        case OfxAccountData::OFX_SAVINGS:
                                        case OfxAccountData::OFX_CREDITLINE:
                                        case OfxAccountData::OFX_CMA:
                                                type = SKGAccountObject::CURRENT;
                                                break;
                                        case OfxAccountData::OFX_MONEYMRKT:
                                        case OfxAccountData::OFX_INVESTMENT:
                                                type = SKGAccountObject::INVESTMENT;
                                                break;
                                        case OfxAccountData::OFX_CREDITCARD:
                                                type = SKGAccountObject::CREDITCARD;
                                                break;
                                        default:
                                                break;
                                        }
                                }

                                if (SKGImportExportManager::ofxError.isSucceeded()) SKGImportExportManager::ofxError=account.setType(type);
                                if (SKGImportExportManager::ofxError.isSucceeded()) SKGImportExportManager::ofxError=account.save();

                                if (SKGImportExportManager::ofxError.isSucceeded()) SKGImportExportManager::ofxError=impotExporter->setDefaultAccount(&account);
                        }
                }
        }
        if (data.currency_valid==true) {
                //Check if unit is already existing
                SKGUnitObject unit(doc);
                SKGImportExportManager::ofxError= SKGObjectBase::getObject(doc, "v_unit", "t_name='"+QString(data.currency)+"' OR t_name like '%("+QString(data.currency)+")%'", tmp);
                if (SKGImportExportManager::ofxError.isSucceeded()) {
                        //Already existing
                        unit=tmp;
                        SKGImportExportManager::ofxError=impotExporter->setDefaultUnit(&unit);
                } else {
                        //Create new account
                        SKGImportExportManager::ofxError=unit.setName(data.currency);
                        if (SKGImportExportManager::ofxError.isSucceeded()) SKGImportExportManager::ofxError=unit.setSymbol(data.currency);
                        if (SKGImportExportManager::ofxError.isSucceeded()) SKGImportExportManager::ofxError=unit.save();

                        if (SKGImportExportManager::ofxError.isSucceeded()) SKGImportExportManager::ofxError=impotExporter->setDefaultUnit(&unit);
                }
        }
        return SKGImportExportManager::ofxError.getReturnCode();
}

int SKGImportExportManager::ofxTransactionCallback(struct OfxTransactionData data, void * pv)
{
        SKGTRACEINRC(5, "SKGImportExportManager::ofxTransactionCallback", SKGImportExportManager::ofxError);
        if (SKGImportExportManager::ofxError.isFailed()) return 0;

        SKGImportExportManager* impotExporter=(SKGImportExportManager*) pv;
        if (!impotExporter) return 0;
        SKGDocumentBank* doc = impotExporter->getDocument();
        if (!doc) return 0;

        //Get account
        SKGAccountObject account;
        SKGImportExportManager::ofxError=impotExporter->getDefaultAccount(account);
        if (SKGImportExportManager::ofxError.isSucceeded()) {
                //Get operation date
                QDate date=QDateTime::fromTime_t(data.date_posted_valid==true ? data.date_posted : data.date_initiated).date();

                //Get unit
                SKGUnitObject unit;
                SKGImportExportManager::ofxError=impotExporter->getDefaultUnit(unit);

                //Create id
                QString ID;
                if (data.fi_id_valid==true) {
                        ID = QString("ID-") + data.fi_id;
                } else if (data.reference_number_valid==true) {
                        ID = QString("REF-") + data.reference_number;
                }

                //Check if already imported
                SKGObjectBase opWithThisHash;
                if (SKGObjectBase::getObject(doc, "operation", "t_imported='Y' AND t_import_id='"+QString(ID)+'\'', opWithThisHash).isSucceeded() ||
                                (data.invtransactiontype_valid==true && data.invtransactiontype==OFX_SPLIT)) {
                        //Yes
                        SKGImportExportManager::nbOperationsNotImported++;
                } else {
                        //No
                        //Create operation
                        SKGOperationObject ope;
                        if (SKGImportExportManager::ofxError.isSucceeded()) SKGImportExportManager::ofxError=account.addOperation(ope);
                        if (SKGImportExportManager::ofxError.isSucceeded()) SKGImportExportManager::ofxError=ope.setDate(date);
                        if (SKGImportExportManager::ofxError.isSucceeded()) SKGImportExportManager::ofxError=ope.setUnit(unit);
                        if (SKGImportExportManager::ofxError.isSucceeded()) SKGImportExportManager::ofxError=ope.setImported(true);
                        if (SKGImportExportManager::ofxError.isSucceeded()) SKGImportExportManager::ofxError=ope.setImportID(ID);
                        if (SKGImportExportManager::ofxError.isSucceeded() && data.payee_id_valid==true) SKGImportExportManager::ofxError=ope.setPayee(data.payee_id);
                        else if (SKGImportExportManager::ofxError.isSucceeded() && data.name_valid==true) SKGImportExportManager::ofxError=ope.setPayee(data.name);
                        if (SKGImportExportManager::ofxError.isSucceeded() && data.memo_valid==true) SKGImportExportManager::ofxError=ope.setComment(data.memo);
                        if (SKGImportExportManager::ofxError.isSucceeded() && data.check_number_valid==true) SKGImportExportManager::ofxError=ope.setNumber(SKGServices::stringToInt(data.check_number));
                        if (SKGImportExportManager::ofxError.isSucceeded() && data.invtransactiontype_valid) SKGImportExportManager::ofxError=ope.setMode(tr("Title"));
                        if (SKGImportExportManager::ofxError.isSucceeded()) SKGImportExportManager::ofxError=ope.save();

                        //Create sub operation
                        SKGSubOperationObject subop;
                        if (SKGImportExportManager::ofxError.isSucceeded()) SKGImportExportManager::ofxError=ope.addSubOperation(subop);
                        if (SKGImportExportManager::ofxError.isSucceeded() && data.amount_valid==true) SKGImportExportManager::ofxError=subop.setQuantity(data.amount+(data.commission_valid==true ? data.commission : 0));
                        if (SKGImportExportManager::ofxError.isSucceeded()) SKGImportExportManager::ofxError=subop.save();

                        //Commission
                        if (SKGImportExportManager::ofxError.isSucceeded() && data.commission_valid==true && data.amount_valid==true && data.commission>0) {
                                //Create splitter operation
                                SKGSubOperationObject subop2;
                                if (SKGImportExportManager::ofxError.isSucceeded()) SKGImportExportManager::ofxError=ope.addSubOperation(subop2);
                                if (SKGImportExportManager::ofxError.isSucceeded()) SKGImportExportManager::ofxError=subop2.setComment(tr("Commission"));
                                if (SKGImportExportManager::ofxError.isSucceeded()) SKGImportExportManager::ofxError=subop2.setQuantity(-data.commission);
                                if (SKGImportExportManager::ofxError.isSucceeded()) SKGImportExportManager::ofxError=subop2.save();
                        }

                        SKGImportExportManager::nbOperationsImported++;
                }
        }
        /* if (data.invtransactiontype_valid==true) {
                 switch (data.invtransactiontype) {
                 case OFX_BUYDEBT:
                 case OFX_BUYMF:
                 case OFX_BUYOPT:
                 case OFX_BUYOTHER:
                 case OFX_BUYSTOCK:
                         t.m_eAction = MyMoneyStatement::Transaction::eaBuy;
                         break;
                 case OFX_REINVEST:
                         t.m_eAction = MyMoneyStatement::Transaction::eaReinvestDividend;
                         break;
                 case OFX_SELLDEBT:
                 case OFX_SELLMF:
                 case OFX_SELLOPT:
                 case OFX_SELLOTHER:
                 case OFX_SELLSTOCK:
                         t.m_eAction = MyMoneyStatement::Transaction::eaSell;
                         break;
                 case OFX_INCOME:
                         t.m_eAction = MyMoneyStatement::Transaction::eaCashDividend;
                         // NOTE: With CashDividend, the amount of the dividend should
                         // be in data.amount.  Since I've never seen an OFX file with
                         // cash dividends, this is an assumption on my part. (acejones)
                         break;

                         //
                         // These types are all not handled.  We will generate a warning for them.
                         //
                 case OFX_CLOSUREOPT:
                         unhandledtype = true;
                         type = "CLOSUREOPT (Close a position for an option)";
                         break;
                 case OFX_INVEXPENSE:
                         unhandledtype = true;
                         type = "INVEXPENSE (Misc investment expense that is associated with a specific security)";
                         break;
                 case OFX_JRNLFUND:
                         unhandledtype = true;
                         type = "JRNLFUND (Journaling cash holdings between subaccounts within the same investment account)";
                         break;
                 case OFX_MARGININTEREST:
                         unhandledtype = true;
                         type = "MARGININTEREST (Margin interest expense)";
                         break;
                 case OFX_RETOFCAP:
                         unhandledtype = true;
                         type = "RETOFCAP (Return of capital)";
                         break;
                 case OFX_SPLIT:
                         unhandledtype = true;
                         type = "SPLIT (Stock or mutial fund split)";
                         break;
                 case OFX_TRANSFER:
                         unhandledtype = true;
                         type = "TRANSFER (Transfer holdings in and out of the investment account)";
                         break;
                 default:
                         unhandledtype = true;
                         type = QString("UNKNOWN %1").arg(data.invtransactiontype);
                         break;
                 }
         } else
                 t.m_eAction = MyMoneyStatement::Transaction::eaNone;*/
        /*

                // Decide whether to import NAME or PAYEEID if both are present in the download



                // If the payee or memo fields are blank, set them to
                // the other one which is NOT blank.
                if ( t.m_strPayee.isEmpty() ) {
                        if ( ! t.m_strMemo.isEmpty() )
                                t.m_strPayee = t.m_strMemo;
                } else {
                        if ( t.m_strMemo.isEmpty() )
                                t.m_strMemo = t.m_strPayee;
                }

                if (data.security_data_valid==true) {
                        struct OfxSecurityData* secdata = data.security_data_ptr;

                        if (secdata->ticker_valid==true) {
                                t.m_strSymbol = secdata->ticker;
                        }

                        if (secdata->secname_valid==true) {
                                t.m_strSecurity = secdata->secname;
                        }
                }

                t.m_shares = MyMoneyMoney();
                if (data.units_valid==true) {
                        t.m_shares = MyMoneyMoney(data.units);
                }

                t.m_price = MyMoneyMoney();
                if (data.unitprice_valid == true) {
                        t.m_price = MyMoneyMoney(data.unitprice);
                }

                t.m_fees = MyMoneyMoney();
                if (data.fees_valid==true) {
                        t.m_fees += MyMoneyMoney(data.fees);
                }

                if (data.commission_valid==true) {
                        t.m_fees += MyMoneyMoney(data.commission);
                }

                bool unhandledtype = false;
                QString type;



        */
        return SKGImportExportManager::ofxError.getReturnCode();
}

SKGError SKGImportExportManager::importQIF()
{
        SKGError err;
        SKGTRACEINRC(2, "SKGImportExportManager::importQIF", err);
        SKGTRACEL(10) << "Input filename=" << fileName << endl;

        //Info for QIF format:
        //http://mb-net.net/Debian/src/gnucash/gnucash-2.2.6/src/import-export/qif-import/file-format.txt
        //http://web.intuit.com/support/quicken/docs/d_qif.html

        //Begin transaction
        if (document) {
                err=document->beginTransaction("#INTERNAL#", 3);
                if (err.isSucceeded()) {
                        //Create account if needed
                        QDateTime now=QDateTime::currentDateTime();
                        QString postFix=SKGServices::dateToSqlString(now);

                        //Step 1 done
                        if (err.isSucceeded()) err=document->stepForward(1);

                        //Open file
                        QFile file(fileName);
                        if (!file.open(QIODevice::ReadOnly | QIODevice::Text)) {
                                err.setReturnCode(ERR_INVALIDARG);
                                err.setMessage(tr("Open file [%1] failed").arg(fileName));
                        } else {
                                QTextStream stream(&file);
                                if (!codec.isEmpty()) stream.setCodec(codec.toAscii().constData());

                                //load file in memory
                                QStringList lines;
                                QStringList dates;
                                bool inAccountSection=false;
                                while (!stream.atEnd()) {
                                        //Read line
                                        //Check line if line is empty or is a commented
                                        QString line=stream.readLine().trimmed();
                                        if (line.length()>0 && line[0]!='#') {
                                                lines.push_back(line);
                                                //Manage !Account section
                                                if (line=="^" && inAccountSection) {
                                                        inAccountSection=false;
                                                } else if (QString::compare(line, "!account", Qt::CaseInsensitive)==0) {
                                                        inAccountSection=true;
                                                }

                                                //We try to find automatically the date format
                                                else if (!inAccountSection &&line[0]=='D') {
                                                        dates.push_back(line.right(line.length()-1));
                                                }
                                        }
                                }

                                //close file
                                file.close();

                                //Select dateformat
                                QString dateFormat=SKGServices::getDateFormat(dates);
                                if (dateFormat.isEmpty()) {
                                        err.setReturnCode(ERR_FAIL);
                                        err.setMessage(tr("Date format not supported"));
                                }
                                if (err.isSucceeded())
                                        err=document->sendMessage(tr("Import of [%1] with code [%2] and date format [%3]").arg(fileName).arg(codec).arg(dateFormat));

                                //Step 2 done
                                if (err.isSucceeded()) err=document->stepForward(2);

                                //Treat all lines
                                if (err.isSucceeded()) {
                                        SKGOperationObject currentOperation;
                                        SKGSubOperationObject currentSubOperation;
                                        QDate currentOperationDate;
                                        bool latestSubCatMustBeRemoved=false;
                                        QChar inSection='B';
                                        bool investmentAccount=false;
                                        int quantityFactor=1;
                                        QString currentUnitForInvestment;
                                        int nbOperationsNotImported=0;
                                        double currentUnitPrice=1;

                                        int nb=lines.size();
                                        err=document->beginTransaction("#INTERNAL#", nb);
                                        SKGAccountObject* account=NULL;
                                        QString stringForHash;
                                        for (int i = 0; err.isSucceeded() && i < nb; ++i) {
                                                QString line=lines.at(i);
                                                QString val;
                                                QChar op=line[0];
                                                if (line.length()>1) {
                                                        val=line.right(line.length()-1).trimmed();
                                                }

                                                //Manage !Account section
                                                if (QString::compare(line, "!type:bank", Qt::CaseInsensitive)==0 ||
                                                                QString::compare(line, "!type:cash", Qt::CaseInsensitive)==0 ||
                                                                QString::compare(line, "!type:ccash", Qt::CaseInsensitive)==0 ||
                                                                QString::compare(line, "!type:ccard", Qt::CaseInsensitive)==0 ||
                                                                QString::compare(line, "!type:oth a", Qt::CaseInsensitive)==0 ||
                                                                QString::compare(line, "!type:invst", Qt::CaseInsensitive)==0) {
                                                        inSection='B';
                                                        investmentAccount=(QString::compare(val, "type:invst", Qt::CaseInsensitive)==0);
                                                } else if (QString::compare(line, "!account", Qt::CaseInsensitive)==0) {
                                                        inSection='A';
                                                } else if (QString::compare(line, "!type:cat", Qt::CaseInsensitive)==0) {
                                                        inSection='C';
                                                        if (err.isSucceeded()) err=document->sendMessage(tr("Categories found and imported"));
                                                } else if (line.at(0)=='!') {
                                                        inSection='?';
                                                } else if (inSection=='C') {
                                                        //Category creation
                                                        if (op=='N') {
                                                                SKGCategoryObject Category;
                                                                val.replace('/', OBJECTSEPARATOR);
                                                                val.replace(':', OBJECTSEPARATOR);
                                                                val.replace(',', OBJECTSEPARATOR);
                                                                val.replace(';', OBJECTSEPARATOR);
                                                                err=SKGCategoryObject::createPathCategory(document,val, Category);
                                                        }
                                                } else if (inSection=='A') {
                                                        //Account creation
                                                        if (op=='N') {
                                                                //Check if the account already exist
                                                                SKGAccountObject account2;
                                                                err=SKGNamedObject::getObjectByName(document, "account", val, account2);
                                                                if (err.isFailed()) {
                                                                        //Create account
                                                                        SKGBankObject bank(document);
                                                                        err=bank.setName(tr("Bank for import %1").arg(postFix));
                                                                        if (err.isSucceeded() && bank.load().isFailed()) err=bank.save(false);
                                                                        if (err.isSucceeded()) err=bank.addAccount(account2);
                                                                        if (err.isSucceeded()) err=account2.setName(val);
                                                                        if (err.isSucceeded() && account2.load().isFailed()) err=account2.save(false);  //Save only
                                                                }

                                                                if (err.isSucceeded()) {
                                                                        if (account) {
                                                                                delete account;
                                                                                account=NULL;
                                                                        }
                                                                        account=new SKGAccountObject(account2);
                                                                }
                                                        } else if (op=='D') {
                                                                if (account) err=account->setNumber(val);
                                                        } else if (op=='^') {
                                                                //^ 	End of entry
                                                                //save
                                                                if (account) err=account->save();
                                                        }
                                                } else if (inSection=='B') {
                                                        //Operation creation
                                                        /*
                                                        >>>> Items for Non-Investment Accounts <<<<
                                                        DONE	D      Date
                                                        DONE	T      Amount
                                                        	U      Transaction amount (higher possible value than T)
                                                        DONE	C      Cleared status
                                                        DONE	N      Number (check or reference number)
                                                        DONE	P      Payee/description
                                                        DONE	M      Memo
                                                        	A      Address (up to 5 lines; 6th line is an optional message)
                                                        DONE	L      Category (category/class or transfer/class)
                                                        DONE	S      Category in split (category/class or transfer/class)
                                                        	E      Memo in split
                                                        DONE	$      Dollar amount of split
                                                        TODO	%      Percentage of split if percentages are used
                                                        	F      Reimbursable business expense flag
                                                        	X      Small Business extensions
                                                        DONE	^      End of entry

                                                        >>>> Items for Investment Accounts <<<<
                                                        DONE	D  	Date
                                                        	N 	Action
                                                        DONE	Y 	Security
                                                        DONE	I 	Price
                                                        DONE	Q 	Quantity (number of shares or split ratio)
                                                        DONE	T 	Transaction amount
                                                        DONE	C 	Cleared status
                                                        	P 	Text in the first line for transfers and reminders
                                                        DONE	M 	Memo
                                                        	O 	Commission
                                                        	L 	Account for the transfer
                                                        	$ 	Amount transferred
                                                        	^ 	End of entry
                                                        */
                                                        stringForHash+=line;
                                                        if (op=='D') {
                                                                //D  	Date
                                                                /*
                                                                Dates in US QIF files are usually in the format MM/DD/YY, although
                                                                four-digit years are not uncommon.  Dates sometimes occur without the
                                                                slash separator, or using other separators in place of the slash,
                                                                commonly '-' and '.'.  US Quicken seems to be using the ' to indicate
                                                                post-2000 two-digit years (such as 01/01'00 for Jan 1 2000).  Some
                                                                banks appear to be using a completely undifferentiated numeric QString
                                                                formateed YYYYMMDD in downloaded QIF files.
                                                                */
                                                                //Operation creation
                                                                if (err.isSucceeded()) {
                                                                        if (account!=NULL) err=account->addOperation(currentOperation);
                                                                        else {
                                                                                SKGAccountObject defAccount;
                                                                                err=getDefaultAccount(defAccount);
                                                                                if (err.isSucceeded()) err=defAccount.addOperation(currentOperation);
                                                                        }
                                                                }

                                                                //Set date
                                                                currentOperationDate=SKGServices::stringToTime(SKGServices::dateToSqlString(val, dateFormat)).date();
                                                                if (err.isSucceeded()) err=currentOperation.setDate(currentOperationDate);

                                                                //Set unit
                                                                if (err.isSucceeded()) {
                                                                        //Create unit if needed
                                                                        SKGUnitObject unit;
                                                                        err=getDefaultUnit(unit, &currentOperationDate);
                                                                        if (err.isSucceeded())  err=currentOperation.setUnit(unit);
                                                                }

                                                                if (err.isSucceeded()) currentOperation.save();

                                                                //Create suboperation
                                                                if (err.isSucceeded()) err=currentOperation.addSubOperation(currentSubOperation);
                                                        } else if (op=='Y') {
                                                                //Y 	Security
                                                                currentUnitForInvestment=val;

                                                                SKGUnitObject unit(document);
                                                                err = unit.setName(currentUnitForInvestment);
                                                                if (unit.load().isFailed()) {
                                                                        if (err.isSucceeded()) err = unit.setSymbol(currentUnitForInvestment);
                                                                        if (err.isSucceeded()) err = unit.setType(SKGUnitObject::SHARE);
                                                                        if (err.isSucceeded()) err = unit.save(false);  //Save only
                                                                }
                                                                if (err.isSucceeded()) err=currentOperation.setUnit(unit);
                                                        } else if (op=='O') {
                                                                //O 	Commission
                                                                SKGOperationObject commission;
                                                                if (account!=NULL) err=account->addOperation(commission);
                                                                else {
                                                                        SKGAccountObject defAccount;
                                                                        err=getDefaultAccount(defAccount);
                                                                        if (err.isSucceeded()) err=defAccount.addOperation(commission);
                                                                }
                                                                if (err.isSucceeded()) err=commission.setDate(currentOperationDate);
                                                                if (err.isSucceeded()) err=commission.setAttribute("t_imported", "P");
                                                                if (err.isSucceeded()) {
                                                                        //Create unit if needed
                                                                        SKGUnitObject unit;
                                                                        err=getDefaultUnit(unit, &currentOperationDate);
                                                                        if (err.isSucceeded())  err=commission.setUnit(unit);
                                                                }
                                                                if (err.isSucceeded()) err=commission.save();  //Save only
                                                                if (err.isSucceeded()) err=currentOperation.setGroupOperation(commission);

                                                                SKGSubOperationObject subcommission;
                                                                if (err.isSucceeded()) err=commission.addSubOperation(subcommission);
                                                                if (err.isSucceeded()) err=subcommission.setQuantity(-SKGServices::stringToDouble(val));
                                                                if (err.isSucceeded()) err=subcommission.save(false, false);  //Save only whitout reload
                                                        } else if (op=='I') {
                                                                //I 	Price
                                                                currentUnitPrice=SKGServices::stringToDouble(val);
                                                                err=document->addOrModifyUnitValue(currentUnitForInvestment,
                                                                                                   currentOperationDate,
                                                                                                   currentUnitPrice);
                                                        } else if (op=='N') {
                                                                if (investmentAccount) {
                                                                        //N 	Action
                                                                        /*
                                                                        QIF N Line    Notes
                                                                        ============  =====
                                                                        Aktab         Same as ShrsOut.
                                                                        AktSplit      Same as StkSplit.
                                                                        Aktzu         Same as ShrsIn.
                                                                        Buy           Buy shares.
                                                                        BuyX          Buy shares. Used with an L line.
                                                                        Cash          Miscellaneous cash transaction. Used with an L line.
                                                                        CGMid         Mid-term capital gains.
                                                                        CGMidX        Mid-term capital gains. For use with an L line.
                                                                        CGLong        Long-term capital gains.
                                                                        CGLongX       Long-term capital gains. For use with an L line.
                                                                        CGShort       Short-term capital gains.
                                                                        CGShortX      Short-term capital gains. For use with an L line.
                                                                        ContribX      Same as XIn. Used for tax-advantaged accounts.
                                                                        CvrShrt       Buy shares to cover a short sale.
                                                                        CvrShrtX      Buy shares to cover a short sale. Used with an L line.
                                                                        Div           Dividend received.
                                                                        DivX          Dividend received. For use with an L line.
                                                                        Errinerg      Same as Reminder.
                                                                        Exercise      Exercise an option.
                                                                        ExercisX      Exercise an option. For use with an L line.
                                                                        Expire        Mark an option as expired. (Uses D, N, Y & M lines)
                                                                        Grant         Receive a grant of stock options.
                                                                        Int           Same as IntInc.
                                                                        IntX          Same as IntIncX.
                                                                        IntInc        Interest received.
                                                                        IntIncX       Interest received. For use with an L line.
                                                                        K.gewsp       Same as CGShort. (German)
                                                                        K.gewspX      Same as CGShortX. (German)2307068
                                                                        Kapgew        Same as CGLong. Kapitalgewinnsteuer.(German)
                                                                        KapgewX       Same as CGLongX. Kapitalgewinnsteuer. (German)
                                                                        Kauf          Same as Buy. (German)
                                                                        KaufX         Same as BuyX. (German)
                                                                        MargInt       Margin interest paid.
                                                                        MargIntX      Margin interest paid. For use with an L line.
                                                                        MiscExp       Miscellaneous expense.
                                                                        MiscExpX      Miscellaneous expense. For use with an L line.
                                                                        MiscInc       Miscellaneous income.
                                                                        MiscIncX      Miscellaneous income. For use with an L line.
                                                                        ReinvDiv      Reinvested dividend.
                                                                        ReinvInt      Reinvested interest.
                                                                        ReinvLG       Reinvested long-term capital gains.
                                                                        Reinvkur      Same as ReinvLG.
                                                                        Reinvksp      Same as ReinvSh.
                                                                        ReinvMd       Reinvested mid-term capital gains.
                                                                        ReinvSG       Same as ReinvSh.
                                                                        ReinvSh       Reinvested short-term capital gains.
                                                                        Reinvzin      Same as ReinvDiv.
                                                                        Reminder      Reminder. (Uses D, N, C & M lines)
                                                                        RtrnCap       Return of capital.
                                                                        RtrnCapX      Return of capital. For use with an L line.
                                                                        Sell          Sell shares.
                                                                        SellX         Sell shares. For use with an L line.
                                                                        ShtSell       Short sale.
                                                                        ShrsIn        Deposit shares.
                                                                        ShrsOut       Withdraw shares.
                                                                        StkSplit      Share split.
                                                                        Verkauf       Same as Sell. (German)
                                                                        VerkaufX      Same as SellX. (German)
                                                                        Vest          Mark options as vested. (Uses N, Y, Q, C & M lines)
                                                                        WithDrwX      Same as XOut. Used for tax-advantaged accounts.
                                                                        XIn           Transfer cash from another account.
                                                                        XOut          Transfer cash to another account.
                                                                        */
                                                                        val=val.toLower();
                                                                        if (val.contains("sell") || val.contains("verkauf") || val.contains("miscexp")) quantityFactor=-1;
                                                                        err=currentOperation.setComment(val);
                                                                        if (err.isSucceeded()) err=currentOperation.setMode(tr("Title"));
                                                                } else {
                                                                        //N 	Num (check or reference number)
                                                                        //Set number
                                                                        bool ok;
                                                                        int number=val.toInt(&ok);
                                                                        if (ok && number!=0) {
                                                                                err=currentOperation.setNumber(number);
                                                                        } else {
                                                                                err=currentOperation.setMode(val);
                                                                        }
                                                                }
                                                        } else if (op=='Q') {
                                                                //Q 	Quantity (number of shares or split ratio)
                                                                //Set value
                                                                err=currentSubOperation.setQuantity(quantityFactor*SKGServices::stringToDouble(val));
                                                        } else if (op=='T') {
                                                                //T 	Amount
                                                                //Set value
                                                                err=currentSubOperation.setQuantity(SKGServices::stringToDouble(val)/currentUnitPrice);
                                                                if (err.isSucceeded() && investmentAccount) {
                                                                        err=currentOperation.setProperty("SKG_OP_ORIGINAL_AMOUNT", val);
                                                                }
                                                        } else if (op=='$') {
                                                                //Dollar amount of split
                                                                //Set value
                                                                if (!investmentAccount) {
                                                                        err=currentSubOperation.setQuantity(SKGServices::stringToDouble(val));

                                                                        //save
                                                                        if (err.isSucceeded()) err=currentSubOperation.save();

                                                                        //Create suboperation
                                                                        if (err.isSucceeded()) err=currentOperation.addSubOperation(currentSubOperation);

                                                                        latestSubCatMustBeRemoved=true;
                                                                }
                                                        } else if (op=='P') {
                                                                //P Payee
                                                                //Set Payee
                                                                //Clean QIF coming from bankperfect
                                                                val.remove("[auto]");

                                                                err=currentOperation.setPayee(val);
                                                        } else if (op=='M') {
                                                                //M 	Memo
                                                                //Set Memo
                                                                err=currentOperation.setComment(val);
                                                        } else if (op=='S' || op=='L') {
                                                                //S 	Category in split (Category/Transfer/Class)//L 	Category (Category/Subcategory/Transfer/Class)
                                                                //TODO LCategory of transaction
                                                                //L[Transfer account]
                                                                //LCategory of transaction/Class of transaction
                                                                //L[Transfer account]/Class of transaction//Set Category
                                                                SKGCategoryObject Category;
                                                                val.replace('/', OBJECTSEPARATOR);
                                                                val.replace(':', OBJECTSEPARATOR);
                                                                val.replace(',', OBJECTSEPARATOR);
                                                                val.replace(';', OBJECTSEPARATOR);
                                                                err=SKGCategoryObject::createPathCategory(document,val, Category);
                                                                if (err.isSucceeded())  err=currentSubOperation.setCategory(Category);
                                                        } else if (op=='C') {
                                                                //C 	Cleared status
                                                                //Set status
                                                                err=currentOperation.setStatus((val=="C" ? SKGOperationObject::POINTED : (val=="R" ? SKGOperationObject::CHECKED : SKGOperationObject::NONE )));
                                                        } else if (op=='^') {
                                                                //^ 	End of entry
                                                                //save

                                                                QByteArray hash = QCryptographicHash::hash(stringForHash.toUtf8(), QCryptographicHash::Md5);
                                                                SKGObjectBase opWithThisHash;
                                                                if (SKGObjectBase::getObject(document, "operation", "t_imported='Y' AND t_import_id='"+QString(hash.toHex())+'\'', opWithThisHash).isSucceeded()) {
                                                                        err=currentOperation.remove();
                                                                        nbOperationsNotImported++;
                                                                } else {
                                                                        if (err.isSucceeded()) err=currentOperation.setAttribute("t_imported", "P");
                                                                        if (err.isSucceeded()) err=currentOperation.setImportID(hash.toHex());
                                                                        if (err.isSucceeded()) err=currentOperation.save();
                                                                        if (!latestSubCatMustBeRemoved && err.isSucceeded()) err=currentSubOperation.save();
                                                                }
                                                                latestSubCatMustBeRemoved=false;
                                                                currentUnitForInvestment="";
                                                                quantityFactor=1;
                                                                currentUnitPrice=1;
                                                                stringForHash="";
                                                        } else {
                                                                // A 	Address (up to five lines; the sixth line is an optional message)
                                                                // E 	Memo in split
                                                        }
                                                }

                                                if (err.isSucceeded() && i%100==0) err = SKGServices::executeSqliteOrder(document, "ANALYZE");
                                                if (err.isSucceeded()) err=document->stepForward(i+1);
                                        }

                                        if (err.isSucceeded()) err = SKGServices::executeSqliteOrder(document, "UPDATE operation SET t_imported='Y' WHERE t_imported='P'");

                                        if (account) {
                                                delete account;
                                                account=NULL;
                                        }
                                        if (err.isSucceeded()) err=document->endTransaction(true);
                                        else  document->endTransaction(false);

                                        if (err.isSucceeded() && nbOperationsNotImported)
                                                err=document->sendMessage(tr("%1 operation(s) not imported because already existing").arg(nbOperationsNotImported));

                                        //Lines treated
                                        if (err.isSucceeded()) err=document->stepForward(3);
                                }
                        }
                }
                if (err.isSucceeded()) err=document->endTransaction(true);
                else  document->endTransaction(false);
        }

        if (err.isFailed()) {
                err.addError(ERR_FAIL, tr("%1 failed").arg("SKGImportExportManager::importQIF"));
        }

        return err;
}

SKGError SKGImportExportManager::cleanBankImport()
{
        SKGError err;
        SKGTRACEINRC(2, "SKGImportExportManager::cleanBankImport", err);

        //Begin transaction
        if (document) {
                err=document->beginTransaction("#INTERNAL#", 3);
                if (err.isSucceeded()) {
                        //Step 1 Clean operations without mode and with comment with double space
                        SKGObjectBase::SKGListSKGObjectBase operations;
                        if (err.isSucceeded()) err=SKGObjectBase::getObjects(document, "operation", "t_mode='' and t_comment like '%  %'", operations);

                        int nb=operations.count();
                        for (int i=0; err.isSucceeded() && i<nb ;++i) {
                                SKGOperationObject op=operations[i];

                                //Comment is like this: <TYPE>  <INFO>
                                //Example: RETRAIT DAB             20/01/08 11H44 013330 LCL GAILLAC 000497
                                QRegExp rx("(.+) {2,}(.+)");
                                QString comment=op.getComment();
                                if (rx.indexIn(comment)!=-1) {
                                        //Get parameters
                                        QString mode =rx.cap(1);
                                        QString info =rx.cap(2);

                                        //Modify
                                        err=op.setComment(info.trimmed());
                                        if (err.isSucceeded())  err=op.setMode(mode.trimmed());
                                        if (err.isSucceeded())  err=op.save(true, false); //No reload
                                }
                        }

                        //Step 1 done
                        if (err.isSucceeded()) err=document->stepForward(1);

                        //Step 2 Clean operations without mode and with comment
                        if (err.isSucceeded()) err=SKGObjectBase::getObjects(document, "operation", "t_mode='' and t_comment!=''", operations);

                        nb=operations.count();
                        for (int i=0; err.isSucceeded() && i<nb ;++i) {
                                SKGOperationObject op=operations[i];

                                //Comment is like this: <TYPE> <INFO>
                                //Example: RETRAIT DAB 14-05-16607-482390
                                QRegExp rx("(\\S+) +(.+)");
                                QString comment=op.getComment();
                                if (rx.indexIn(comment)!=-1) {
                                        //Get parameters
                                        QString mode =rx.cap(1);
                                        QString info =rx.cap(2);

                                        //Modify
                                        err=op.setComment(info.trimmed());
                                        if (err.isSucceeded())  err=op.setMode(mode.trimmed());
                                        if (err.isSucceeded())  err=op.save(true, false); //No reload
                                }
                        }

                        //Step 2 done
                        if (err.isSucceeded()) err=document->stepForward(2);

                        //Step 3 Clean cheque without number
                        if (err.isSucceeded()) err=SKGObjectBase::getObjects(document, "operation", "i_number=0 and lower(t_mode)='cheque'", operations);

                        nb=operations.count();
                        for (int i=0; err.isSucceeded() && i<nb ;++i) {
                                SKGOperationObject op=operations[i];

                                //Comment is like this: <TYPE>  <INFO>
                                //Example: RETRAIT DAB             20/01/08 11H44 013330 LCL GAILLAC 000497
                                QRegExp rx("(\\d+)");
                                QString comment=op.getComment();
                                if (rx.indexIn(comment)!=-1) {
                                        //Get parameters
                                        int number =SKGServices::stringToInt(rx.cap(1));

                                        //Modify
                                        err=op.setNumber(number);
                                        if (err.isSucceeded())  err=op.save(true, false); //No reload
                                }
                        }

                        //Step 3 done
                        if (err.isSucceeded()) err=document->stepForward(3);
                }

                if (err.isSucceeded()) err=document->endTransaction(true);
                else  document->endTransaction(false);
        }

        if (err.isFailed()) {
                err.addError(ERR_FAIL, tr("%1 failed").arg("SKGImportExportManager::cleanBankImport"));
        }

        return err;
}

SKGError SKGImportExportManager::cleanBankPerfectImport()
{
        SKGError err;
        SKGTRACEINRC(2, "SKGImportExportManager::cleanBankPerfectImport", err);

        //Begin transaction
        if (document) {
                err=document->beginTransaction("#INTERNAL#", 5);
                if (err.isSucceeded()) {
                        //Step 1 Clean cheque
                        err=SKGServices::executeSqliteOrder(document, "UPDATE operation set t_mode='Cheque', t_comment='', i_number=substr(t_mode,4,1000)+0 where t_mode like 'Chq %'");
                        if (err.isSucceeded()) err=SKGServices::executeSqliteOrder(document, "UPDATE operation set t_mode='Cheque', t_comment='', i_number=substr(t_comment,3,1000)+0 where t_mode like 'Ch%' AND t_comment like 'n%'");

                        //Step 1 done
                        if (err.isSucceeded()) err=document->stepForward(1);

                        //TODO Step 2 [auto]

                        //Step 2 done
                        if (err.isSucceeded()) err=document->stepForward(2);

                        //Step 3 Splitted operations
                        SKGStringListList listTmp;
                        if (err.isSucceeded()) err=SKGServices::executeSelectSqliteOrder(document,
                                                           "SELECT id, d_date, t_comment, t_payee FROM operation WHERE i_group_id=0 AND t_payee like '[1/%'",
                                                           listTmp);
                        int nb=listTmp.count();
                        for (int i=1; err.isSucceeded() && i<nb ;++i) { //First line ignored because of it's header
                                QString id=listTmp.at(i).at(0);
                                QString date=listTmp.at(i).at(1);
                                QString comment=listTmp.at(i).at(2);
                                QString payee=listTmp.at(i).at(3);
                                QRegExp rx("(\\S+) +(.+)");
                                if (rx.indexIn(payee)!=-1) {
                                        payee=SKGServices::stringToSqlString(rx.cap(2));
                                }

                                QString wcop="d_date='"+date+"' AND t_comment='"+SKGServices::stringToSqlString(comment)+'\'';
                                err=SKGServices::executeSqliteOrder(document,
                                                                    "UPDATE suboperation SET rd_operation_id="+id+" WHERE "
                                                                    "rd_operation_id!="+id+" AND rd_operation_id in (SELECT id FROM operation WHERE "+wcop+')');

                                if (err.isSucceeded()) err=SKGServices::executeSqliteOrder(document,
                                                                   "DELETE FROM operation WHERE id!="+id+" AND "+wcop);

                                if (err.isSucceeded()) err=SKGServices::executeSqliteOrder(document,
                                                                   "UPDATE operation SET t_comment='', t_payee='"+SKGServices::stringToSqlString(payee)+"' WHERE id="+id);
                        }

                        //Step 3 done
                        if (err.isSucceeded()) err=document->stepForward(3);

                        //Step 4 Titre
                        SKGObjectBase::SKGListSKGObjectBase opTitreList;
                        if (err.isSucceeded()) err=SKGObjectBase::getObjects(document, "operation", "t_mode='Titre'", opTitreList);

                        int nb2=opTitreList.count();
                        for (int i=0; err.isSucceeded() && i<nb2 ;++i) {
                                SKGOperationObject op=opTitreList[i];

                                //Payee is like this: <NB> <NAME> achetÃ¯Â¿Â½ <VALUE>
                                //Example: 57.496 FCP EADS EXPANSION achetÃ¯Â¿Â½ 13,421
                                QRegExp rx("(\\S+) (.+) \\S+ (\\S+)");
                                QString payee=op.getPayee();
                                if (rx.indexIn(payee)!=-1) {
                                        //Get parameters
                                        double nb=SKGServices::stringToDouble(rx.cap(1));
                                        QString unitName =rx.cap(2);
                                        double value=SKGServices::stringToDouble(rx.cap(3));

                                        SKGObjectBase::SKGListSKGObjectBase subOps;
                                        err=op.getSubOperations(subOps);
                                        if (err.isSucceeded() && subOps.count()==1 && nb!=0) {

                                                //Create or Update unit
                                                SKGSubOperationObject subOp=subOps.at(0);
                                                SKGUnitObject unit(document);
                                                if (err.isSucceeded())  err=document->addOrModifyUnitValue(unitName, op.getDate(), value);
                                                if (err.isSucceeded())  err=document->addOrModifyUnitValue(unitName, QDate::currentDate(), subOp.getQuantity()/nb);
                                                if (err.isSucceeded()) {
                                                        err=unit.setName(unitName);
                                                        if (err.isSucceeded())  err=unit.load();
                                                }
                                                if (err.isSucceeded())  err=unit.setType(SKGUnitObject::SHARE);
                                                if (err.isSucceeded())  err=unit.setInternetCode(op.getComment());
                                                if (err.isSucceeded())  err=unit.save();
                                                if (err.isSucceeded())  err=op.setUnit(unit);
                                                if (err.isSucceeded())  err=op.setPayee("");
                                                if (err.isSucceeded())  err=subOp.setQuantity(nb);
                                                if (err.isSucceeded())  err=subOp.save(true, false); //No reload
                                                if (err.isSucceeded())  err=op.save(true, false); //No reload
                                        }
                                }
                        }

                        //Step 4 done
                        if (err.isSucceeded()) err=document->stepForward(4);

                        //Step 5 Update payee with "Transfer" already defined as transfer
                        if (err.isSucceeded()) err=SKGServices::executeSqliteOrder(document,
                                                           "UPDATE operation SET t_payee='' WHERE t_payee like 'Transfert %' AND i_group_id!=0");

                        //Step 5 done
                        if (err.isSucceeded()) err=document->stepForward(5);
                }
                if (err.isSucceeded()) err=document->endTransaction(true);
                else  document->endTransaction(false);
        }
        if (err.isFailed()) {
                err.addError(ERR_FAIL, tr("%1 failed").arg("SKGImportExportManager::cleanBankPerfectImport"));
        }

        return err;
}

SKGError SKGImportExportManager::findAndGroupTransfers(int& oNbOperationsMerged)
{
        SKGError err;
        SKGTRACEINRC(2, "SKGImportExportManager::findAndGroupTransfers", err);

        oNbOperationsMerged=0;

        //Begin transaction
        if (document) {
                err=document->beginTransaction("#INTERNAL#", 2);
                if (err.isSucceeded()) {
                        //Look for operations with
                        //  Same units
                        //  Same dates
                        //  Null i_group_id
                        //  Different accounts
                        //  Oposite amounts
                        SKGStringListList listTmp;
                        //+A.i_group_id=0 AND +B.i_group_id=0 is for avoiding to use bad index
                        err=SKGServices::executeSelectSqliteOrder(document,
                                        "SELECT A.id, B.id FROM v_operation_tmp1 A, v_operation_tmp1 B WHERE A.id<=B.id AND A.rc_unit_id=B.rc_unit_id AND A.d_date=B.d_date AND A.rd_account_id!=B.rd_account_id AND A.f_QUANTITY=-B.f_QUANTITY AND +A.i_group_id=0 AND +B.i_group_id=0 AND A.f_QUANTITY!=0",
                                        listTmp);
                        //Step 1 done
                        if (err.isSucceeded()) err=document->stepForward(1);

                        //Group
                        {
                                oNbOperationsMerged=listTmp.count();
                                if (err.isSucceeded()) err=document->beginTransaction("#INTERNAL#", oNbOperationsMerged-1);
                                for (int i=1; err.isSucceeded() && i<oNbOperationsMerged ;++i) { //First line ignored because of it's header
                                        SKGOperationObject op1(document, SKGServices::stringToInt(listTmp.at(i).at(0)));
                                        SKGOperationObject op2(document, SKGServices::stringToInt(listTmp.at(i).at(1)));

                                        err=op2.setGroupOperation(op1);
                                        if (err.isSucceeded()) err=op2.save(true, false); //No reload
                                        if (err.isSucceeded()) err=document->stepForward(i);
                                }
                                if (err.isSucceeded()) err=document->endTransaction(true);
                                else  document->endTransaction(false);
                        }
                        oNbOperationsMerged=(oNbOperationsMerged-1)*2;

                        //Step 2 done
                        if (err.isSucceeded()) err=document->stepForward(2);

                }
                if (err.isSucceeded()) err=document->endTransaction(true);
                else  document->endTransaction(false);
        }

        if (err.isFailed()) {
                err.addError(ERR_FAIL, tr("%1 failed").arg("SKGImportExportManager::findAndGroupTransfers"));
        }

        return err;
}

#include "skgimportexportmanager.moc"
