/*
 * util.c :  utility functions for the libsvn_client library
 *
 * ====================================================================
 * Copyright (c) 2005 CollabNet.  All rights reserved.
 *
 * This software is licensed as described in the file COPYING, which
 * you should have received as part of this distribution.  The terms
 * are also available at http://subversion.tigris.org/license-1.html.
 * If newer versions of this license are posted there, you may use a
 * newer version instead, at your option.
 *
 * This software consists of voluntary contributions made by many
 * individuals.  For exact contribution history, see the revision
 * history and logs, available at http://subversion.tigris.org/.
 * ====================================================================
 */

#include <apr_pools.h>
#include <apr_strings.h>

#include "svn_string.h"
#include "svn_props.h"
#include "svn_client.h"

/**
 * Duplicate a HASH containing (char * -> svn_string_t *) key/value
 * pairs using POOL.
 */
static apr_hash_t *
svn_client__string_hash_dup (apr_hash_t *hash, apr_pool_t *pool)
{
  apr_hash_index_t *hi;
  const void *key;
  apr_ssize_t klen;
  void *val;
  apr_hash_t *new_hash = apr_hash_make (pool);
  for (hi = apr_hash_first (pool, hash); hi; hi = apr_hash_next (hi))
    {
      apr_hash_this (hi, &key, &klen, &val);
      key = apr_pstrdup (pool, key);
      val = svn_string_dup (val, pool);
      apr_hash_set (new_hash, key, klen, val);
    }
  return new_hash;
}

svn_client_commit_item2_t *
svn_client_commit_item2_dup (const svn_client_commit_item2_t *item,
                             apr_pool_t *pool)
{
  svn_client_commit_item2_t *new_item = apr_palloc (pool, sizeof (*new_item));

  *new_item = *item;

  if (new_item->path)
    new_item->path = apr_pstrdup (pool, new_item->path);

  if (new_item->url)
    new_item->url = apr_pstrdup (pool, new_item->url);

  if (new_item->copyfrom_url)
    new_item->copyfrom_url = apr_pstrdup (pool, new_item->copyfrom_url);

  if (new_item->wcprop_changes)
    new_item->wcprop_changes = svn_prop_array_dup (new_item->wcprop_changes,
                                                   pool);

  return new_item;
}

svn_client_proplist_item_t *
svn_client_proplist_item_dup (const svn_client_proplist_item_t *item,
                              apr_pool_t * pool)
{
  svn_client_proplist_item_t *new_item
    = apr_pcalloc (pool, sizeof (*new_item));

  if (item->node_name)
    new_item->node_name = svn_stringbuf_dup (item->node_name, pool);

  if (item->prop_hash)
    new_item->prop_hash = svn_client__string_hash_dup (item->prop_hash, pool);

  return new_item;
}

