/*
 * newfs.c - NILFS tool, build a NILFS file system
 *
 * Copyright (C) 2005, 2006 Nippon Telegraph and Telephone Corporation.
 *
 * This file is part of NILFS.
 *
 * NILFS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * NILFS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NILFS; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * newfs.c,v 1.13 2006/03/23 06:04:05 ryusuke Exp
 *
 * Written by Hisashi Hifumi
 *            Amagai Yoshiji
 */

#define _FILE_OFFSET_BITS 64
//#define _LARGEFILE_SOURCE
#define _LARGEFILE64_SOURCE
#define __USE_FILE_OFFSET64
#define _XOPEN_SOURCE 600

#undef	ATIME_DIR_ENTRY

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <strings.h>
#include <sys/ioctl.h>
#include <uuid/uuid.h>
#include <string.h>
#include <asm/page.h>
#include "newfs.h"

#define NILFS_DISKHDR_SIZE 	4096		/* HDD header (MBR+superblock) */
#define DEFAULT_BLOCKSIZE_LOG	2		/* blocksize = 1024 * 2^2 bytes */
#define DEFAULT_BLOCKSIZE_BYTE	(1<<(DEFAULT_BLOCKSIZE_LOG+10))
#define DEF_BLKS_PER_SEG	1024		/* default blocks per segment */
#define DEFAULT_CHKINT		(60*60*24*180)	/* default check interval 180 days */

#define usage_size	sizeof(struct segusage)
#define usageh_size	sizeof(struct segusage_hdr)

/*
 * linux/fs.h
 * File types
 *
 * NOTE! These match bits 12..15 of stat.st_mode
 * (ie "(i_mode >> 12) & 15").
 */
#define DT_UNKNOWN	0
#define DT_FIFO		1
#define DT_CHR		2
#define DT_DIR		4
#define DT_BLK		6
#define DT_REG		8
#define DT_LNK		10
#define DT_SOCK		12
#define DT_WHT		14

#define MOUNTS		"/etc/mtab"
#define BUFSIZE		128

struct nilfs_super_block super_block_skeleton = {
	.s_rev_level		= NILFS_CURRENT_REV,
	.s_minor_rev_level	= NILFS_MINOR_REV,
	.s_magic		= NILFS_SUPER_MAGIC,
	.s_bytes		= NILFS_SB_BYTES,
	.s_crc_seed		= 0,
	.s_sum			= 0,
	.s_nsegment		= 0,
	.s_dev_size		= 0,
	.s_first_data_block	= 0,
	.s_log_block_size	= 0,
	.s_blocks_per_segment	= 0,
	.s_r_segments_count	= 1,
	.s_last_segment		= 0,
	.s_last_seq		= 0,
	.s_free_blocks_count	= 0,
	.s_ctime		= 0,
	.s_mtime 		= 0,
	.s_wtime		= 0,
	.s_mnt_count		= 0,
	.s_max_mnt_count	= NILFS_DFL_MAX_MNT_COUNT,
	.s_state		= NILFS_VALID_FS,
	.s_errors		= 1,
	.s_lastcheck		= 0,
	.s_checkinterval	= DEFAULT_CHKINT,
	.s_creator_os		= 0,
	.s_def_resuid		= 0,
	.s_def_resgid		= 0,
	.s_first_ino		= NILFS_USER_INO,
	.s_inode_size		= sizeof(struct nilfs_inode),
	.s_uuid			= "\0",
	.s_volume_name		= "\0",
	.s_last_mounted		= "\0",
	.s_c_interval		= 0,
	.s_c_block_max		= 0,
};

struct nilfs_inode_hdr inode_hdr_skeleton = {
	.ih_ino			= 0,
	.ih_flags		= 0,
	.ih_nfree		= 0,
	.ih_free		= NILFS_USER_INO,
};

/* NILFS initial disk inode */
struct nilfs_inode fs_inodes[NILFS_USER_INO];

#define BT_INODE 	0
#define BT_REGFILE	1

struct nilfs_btree_node btree_hdr_sleleton = {
	.bn_level		= 0,
	.bn_nkeys		= 1,
	.bn_reserved		= {0, 0, 0},
};

struct nilfs_seg_summary seg0_summary_sekeleton = {
	.ss_datasum		= 0,
	.ss_sumsum		= 0,
	.ss_seq			= 0,
	.ss_magic		= NILFS_SEGSUM_MAGIC,
	.ss_nblocks		= 0,
	.ss_nfinfo		= 0,
	.ss_nfbinfo		= 0,
	.ss_nfblk		= 0,
	.ss_nfbblk		= 0,
	.ss_niblk		= 0,
	.ss_nibblk		= 0,
	.ss_create		= 0,
	.ss_prev		= 0,
	.ss_flags		= SS_LOGBGN | SS_LOGEND | SS_FJCP,
};

struct nilfs_checkpoint seg0_checkpoint_sekeleton = {
	.cp_sum			= 0,
	.cp_flags		= NILFS_CP_MAJOR,
	.cp_bytes		= NILFS_CP_BYTES,
	.cp_inode_root		= 0,
	.cp_nblk_lseg		= 0,
	.cp_inodes_count	= 0,
	.cp_blocks_count	= 0,
	.cp_sketch_size		= 0,
};

void sb_to_le(struct nilfs_super_block *);
void sum_to_le(struct nilfs_seg_summary *);
void dentry_to_le(struct nilfs_dir_entry *);
void ihdr_to_le(struct nilfs_inode_hdr *);
void inode_to_le(struct nilfs_inode *);
void btree_to_le(struct nilfs_btree_node *);
void checkpoint_to_le(struct nilfs_checkpoint *);
void segusage_hdr_to_le(struct segusage_hdr *);
void segusage_to_le(struct segusage *);
void mygetopt(int argc, char *argv[]);
void usage();
void show_version();
void merr(char *);
void seek_err(__u32 blocknum);
void too_small_dev(__u64 dev_size, __u64 required_size, int fd);
void cannot_open_device(char *);
void cannot_get_devsize(char *);
void cannot_write_device(int);
void disk_scan(char *, int, int);
void check_mount(int, char *);

uchar *copy_to_buffer(uchar *, uchar *, int);
void write_disk(int);
void write_superblock_replica(char *, int);
void write_segusage_table(int);
uchar *make_segusage_table(__u32, unsigned long, int, struct segusage_hdr *, __u32);
unsigned long nilfs_segchunk_seg_len(long, unsigned long);
unsigned long nilfs_segchunk_seg_num(long, unsigned long);
unsigned long nilfs_susage_blk_len(long, unsigned long);
unsigned long long nilfs_susage_blk_num(int, long, unsigned long, unsigned long);
unsigned long nilfs_seg_segchunk_num(long, unsigned long);
unsigned int  nilfs_seg_blk_len(unsigned long, unsigned long, unsigned long);
void init_segusage_hdr(struct segusage_hdr *, struct nilfs_super_block *);
extern char *optarg;
extern int optind;

int blocksize, blksz_log;
int segsize = 0, block_per_seg;
int show_version_only = 0, quiet = 0, cflag = 0, nflag = 0;
char volume_label[16];
char *device;
__u64 dev_size = 0;
time_t nilfs_ctime;

struct nilfs_super_block *sb;
struct nilfs_btree_node *btree_hdr;
struct nilfs_checkpoint *seg0_checkpoint;
struct nilfs_seg_summary *seg0_summary;

char *progname = "mkfs.nilfs";

#define MOVE_TO_NEXT_BLOCK(ptr) \
	(typeof(ptr))((((long)(ptr))+(blocksize)-1) & ~((blocksize)-1))

/* initial disk layout */

/*
 blk 0      1     2     3      4     5      6     7     8
  +-----+-------+----+------+-----+------+-----+-----+-----+
  +super|segment|root|.atime|dir  |.atime|disk |inode|check|
  +block|summary|dir |      |btree|btree |inode|btree|point|
  +-----+-------+----+------+-----+------+-----+-----+-----+
           ^
           sb->s_first_data_block

  nilfs special file:
  .sketch has disk inode and is registered to root directory,
  but has no file data nor file block btree.
 */
/* contents     	block_number */

#define B_SS		(sb->s_first_data_block + 0)
/* regular files */
#define B_REG		(sb->s_first_data_block + 1)

/* valid after regist_dinode(), before inode_blocks() */
#define B_DIR		fs_inodes[NILFS_ROOT_INO].i_block_root
#define	B_ATIME 	fs_inodes[NILFS_ATIME_INO].i_block_root

#define B_INODE		(B_REG + nfiles*2)
#define B_INODE_BT	(B_INODE + 1)
#define B_CP		(B_INODE + 2)
#define B_MAX		(B_INODE + 3)

/* number of pre-created files (include directories) */
int nfiles;

#define NBLOCK	32
uchar *diskblock[NBLOCK];
struct nilfs_finfo_ex {
	struct nilfs_finfo finfo;
	__le64 blocks[1];
};
struct nilfs_finfo_ex seg0_finfo[NBLOCK];
struct nilfs_fbinfo   seg0_fbinfo[NBLOCK];

void
init_buffers()
{
	int i;
	for (i = 0; i < NBLOCK; i++) {
		if (posix_memalign((void **)&diskblock[i], blocksize, blocksize) != 0)
			merr("posix_memalign failed");			
		memset(diskblock[i], 0, blocksize);
	}
}

void
init_sb(int fd)
{
	/* BLKGETSIZE64 return device size in byte */
	if (ioctl(fd, BLKGETSIZE64, &dev_size) != 0)
		cannot_get_devsize(device);

	sb = &super_block_skeleton;
	sb->s_dev_size = dev_size;		/* device size in bytes */
	sb->s_first_data_block = (NILFS_DISKHDR_SIZE <= blocksize) ? 1 : (NILFS_DISKHDR_SIZE / blocksize);
	sb->s_last_segment = sb->s_first_data_block;
	sb->s_nsegment = dev_size / segsize;
	if (sb->s_nsegment < 2)
		too_small_dev(dev_size, 2 * segsize, fd);
	sb->s_blocks_per_segment = block_per_seg;
	sb->s_log_block_size = blksz_log;
	srand48(nilfs_ctime);
	sb->s_crc_seed = (__u32)mrand48();
	sb->s_wtime = nilfs_ctime;                /* Write time */
	sb->s_ctime = nilfs_ctime;
	sb->s_lastcheck = nilfs_ctime;
	uuid_generate(sb->s_uuid);	/* set uuid using libuuid */
	memcpy(sb->s_volume_name, volume_label, sizeof(volume_label));
}

void
superblock()
{
	sb->s_free_blocks_count = (dev_size >> (blksz_log + 10)) - B_MAX;
}

void
init_inode()
{
	struct nilfs_inode_hdr *root_ino_hdr;
	int i;

	root_ino_hdr = &inode_hdr_skeleton;
	root_ino_hdr->ih_nfree = (blocksize - sizeof(struct nilfs_inode_hdr))
		                 / sizeof(struct nilfs_inode) - NILFS_USER_INO;
	for (i = 0; i < NILFS_USER_INO; i++)
		fs_inodes[i].i_flags = NILFS_INODE_NEW;
}

void
regist_dinode(int ino, unsigned int type, int mode, unsigned int size)
{
	struct nilfs_inode *in = &fs_inodes[ino];

	in->i_mode = (type << 12) | mode;
	in->i_version = 1;
	in->i_flags = 0;
	in->i_blocks = (size+blocksize-1)/blocksize;
	in->i_size = size;
	in->i_block_root = 0;
	if (0 < size) {
		/*
		 * i_block_root:
		 * stored data block number temporary.
		 * B-tree block number will be stored in inode_blocks().
		 */
		in->i_block_root = B_REG + nfiles;
		seg0_finfo[nfiles].finfo.fi_ino = __cpu_to_le64(ino);
		nfiles++;
	}
}

void
set_atime(int ino, time_t t)
{
	__le64 p = __cpu_to_le64(t);

	if (blocksize / sizeof(__le64) <= ino)
		merr("Internal error: set_atime illegal ino");
	memcpy(diskblock[B_ATIME] + ino * sizeof(__le64), (uchar *)&p, sizeof(__le64));
}

void
inc_link_count(int ino)
{
	struct nilfs_inode *in = &fs_inodes[ino];

	in->i_ctime = nilfs_ctime;
	in->i_mtime = nilfs_ctime;
	set_atime(ino, nilfs_ctime);

	in->i_links_count++;
}

uchar *
add_dir_entry(uchar *segptr, unsigned long ino, char *name)
{
	unsigned int name_len = strlen(name);
	unsigned int rec_len = NILFS_DIR_REC_LEN(name_len);
	struct nilfs_dir_entry de;
	uchar *nb = diskblock[B_DIR] + blocksize;
	struct nilfs_inode *in = &fs_inodes[ino];

	if (NILFS_NAME_LEN < name_len) {
		show_version();
		fprintf(stderr, "Internal error: directory entry name too long %s\n", name);
		exit(1);
	}
	memset((void *)&de, 0, sizeof(de));
	de.inode = __cpu_to_le64(ino);
	de.name_len = name_len & 0xff;
	
	switch ((in->i_mode >> 12) & 0xf) {
	case DT_DIR:
		de.file_type = NILFS_FT_DIR;
		break;
	case DT_REG:
		de.file_type = NILFS_FT_REG_FILE;
		break;
	default:
		merr("Internal error: file type not supported\n");
	}
	strncpy(de.name, name, NILFS_NAME_LEN);
	if (nb - segptr < rec_len)
		merr("Internal error: too many directory entry");
	if (ino == NILFS_SKETCH_INO)	/* last entry of root dir */
		rec_len = nb - segptr;
	de.rec_len = __cpu_to_le16(rec_len);
	segptr = copy_to_buffer(segptr, (uchar *)&de, min(rec_len, sizeof(de)));
	inc_link_count(ino);
	return segptr;
}

void
init_btree_hdr()
{
	btree_hdr = &btree_hdr_sleleton;
	btree_to_le(btree_hdr);
}

void
btree_block(int blk, nilfs_btree_key_t key, nilfs_btree_ptr_t ptr, int ino, int type)
{
	uchar *segptr;

	if (blk < 0 || B_MAX <= blk)
		merr("Internal error: btree_entry blk out of bound");
	segptr = diskblock[blk];
	CPU_TO_LE64(key);
	CPU_TO_LE64(ptr);

	segptr = copy_to_buffer(segptr, (uchar *)btree_hdr, sizeof(struct nilfs_btree_node));
	memcpy(segptr, (char *)&key, sizeof(nilfs_btree_key_t));
	segptr += (blocksize - sizeof(struct nilfs_btree_node)) / 2;
	segptr = copy_to_buffer(segptr, (uchar *)&ptr, sizeof(nilfs_btree_ptr_t));

	if (type == BT_REGFILE) {
		seg0_fbinfo[seg0_summary->ss_nfbinfo].fbi_ino = __cpu_to_le64(ino);
		seg0_summary->ss_nfbinfo++;	/* number of btree */
		seg0_summary->ss_nfbblk++;
	} else { /* inode btree */
		seg0_summary->ss_nibblk++;
	}
}

void
inode_blocks()
{
	struct nilfs_inode_hdr *root_ino_hdr = &inode_hdr_skeleton;
	uchar *segptr = diskblock[B_INODE];
	int i, ino, nfree;

	for (i = 0; i < nfiles; i++) {
		ino = __le64_to_cpu(seg0_finfo[i].finfo.fi_ino);
		fs_inodes[ino].i_block_root = B_REG + nfiles + i;
	}

	nfree = root_ino_hdr->ih_nfree;
	ihdr_to_le(root_ino_hdr);
	segptr = copy_to_buffer(segptr, (uchar *)root_ino_hdr, sizeof(struct nilfs_inode_hdr));
	for (i = 0; i < NILFS_USER_INO; i++) {
		inode_to_le(&fs_inodes[i]);
		segptr = copy_to_buffer(segptr, (uchar *)&fs_inodes[i], sizeof(struct nilfs_inode));
	}
	for (; 0 < nfree; nfree--)
		segptr = copy_to_buffer(segptr, (uchar *)&fs_inodes[0], sizeof(struct nilfs_inode));

	seg0_summary->ss_niblk++;

	btree_block(B_INODE_BT,
		    NILFS_ROOT_INO / ((blocksize - sizeof(struct nilfs_inode_hdr)) / sizeof(struct nilfs_inode)),
		    B_INODE, 0, BT_INODE);
}

void
init_checkpoint()
{
	seg0_checkpoint = &seg0_checkpoint_sekeleton;
	seg0_checkpoint->cp_blocks_count = dev_size >> (blksz_log + 10);
}

void
checkpoint()
{
	seg0_checkpoint->cp_inode_root = B_INODE_BT;
	seg0_checkpoint->cp_nblk_lseg = __cpu_to_le64(B_MAX-B_SS);
	seg0_checkpoint->cp_inodes_count = nfiles;
	checkpoint_to_le(seg0_checkpoint);
	copy_to_buffer(diskblock[B_CP], (uchar *)seg0_checkpoint, sizeof(struct nilfs_checkpoint));
}

void
init_segment_summary()
{
	int i;

	seg0_summary = &seg0_summary_sekeleton;
	seg0_summary->ss_create = nilfs_ctime;
	for (i = 0; i < NBLOCK; i++) {
		seg0_finfo[i].finfo.fi_nblocks = __cpu_to_le32(1);
		seg0_finfo[i].finfo.fi_version = __cpu_to_le32(1);
		seg0_finfo[i].blocks[0] = __cpu_to_le64(0);

		seg0_fbinfo[i].fbi_nblocks = __cpu_to_le32(1);
	}
}

void
segment_summary()
{
	struct nilfs_iinfo  seg0_iinfo;
	uchar *segptr = diskblock[B_SS];
	int nfinfo = nfiles;
	int nfbinfo = seg0_summary->ss_nfbinfo;
	int i;

	seg0_summary->ss_nblocks = B_MAX-B_SS;
	seg0_summary->ss_nfinfo = nfiles;
	seg0_summary->ss_nfblk = nfiles;

	seg0_iinfo.ii_ino = __cpu_to_le64(NILFS_ROOT_INO);

	sum_to_le(seg0_summary);

	segptr = copy_to_buffer(segptr, (uchar *)seg0_summary, sizeof(struct nilfs_seg_summary));

	/* file block info */
	for (i = 0; i < nfinfo; i++)
		segptr = copy_to_buffer(segptr, (uchar *)&seg0_finfo[i], sizeof(struct nilfs_finfo_ex));

	/* file btree info */
	for (i = 0; i < nfbinfo; i++)
		segptr = copy_to_buffer(segptr, (uchar *)&seg0_fbinfo[i], sizeof(struct nilfs_fbinfo));

	/* inode block info */
	segptr = copy_to_buffer(segptr, (uchar *)&seg0_iinfo, sizeof(struct nilfs_iinfo));
}

void
checksum()
{
	int i;
	int nfinfo = nfiles;
	int nfbinfo = __le16_to_cpu(seg0_summary->ss_nfbinfo);

	/*segment summary checksum */
	seg0_summary->ss_sumsum
			= crc32c_le(sb->s_crc_seed,
				    diskblock[B_SS] + sizeof(__le32) * 2,
				    sizeof(struct nilfs_seg_summary)
				    + (sizeof(struct nilfs_finfo) + sizeof(__le64)) * nfinfo
				    + sizeof(struct nilfs_fbinfo) * nfbinfo
				    + sizeof(struct nilfs_iinfo)
				    - sizeof(__le32) * 2);
	
	CPU_TO_LE32(seg0_summary->ss_sumsum);
	memcpy(diskblock[B_SS] + sizeof(__le32), (char *)&seg0_summary->ss_sumsum, sizeof(__le32));

	/* checkpoint checksum */
	seg0_checkpoint->cp_sum
		= crc32c_le(sb->s_crc_seed, diskblock[B_CP] + sizeof(__le32), NILFS_CP_BYTES - sizeof(__le32));
	CPU_TO_LE32(seg0_checkpoint->cp_sum);
	memcpy(diskblock[B_CP], (char *)&seg0_checkpoint->cp_sum, sizeof(__le32));

	/* segment0 checksum */
	seg0_summary->ss_datasum = crc32c_le(sb->s_crc_seed,
					     diskblock[B_SS] + sizeof(__le32),
					     blocksize - sizeof(__le32));
	for (i = B_SS + 1; i < B_MAX; i++)
		seg0_summary->ss_datasum
			= crc32c_le(seg0_summary->ss_datasum, diskblock[i], blocksize);

	CPU_TO_LE32(seg0_summary->ss_datasum);
	memcpy(diskblock[B_SS], (char *)&seg0_summary->ss_datasum, sizeof(__le32));
}

int
main(int argc, char *argv[])
{
	uchar *segptr;
	int i, fd, ino;

	mygetopt(argc, argv);

	if ((fd = open(device, O_RDWR)) < 0)
		cannot_open_device(device);

	check_mount(fd, device);

	init_buffers();
	init_sb(fd);
	init_inode();
	init_btree_hdr();
	init_checkpoint();
	init_segment_summary();

	/* directories and files */
	/* regist root dir first!! */
	regist_dinode(NILFS_ROOT_INO, DT_DIR, 0755, blocksize);
	/* pre-created files with file data blocks */
	regist_dinode(NILFS_ATIME_INO, DT_REG, 0444, blocksize);
	/* add new inode here */
	/* pre-created directory and inode entries. no file data blocks */
	regist_dinode(NILFS_SKETCH_INO, DT_REG, 0600, 0);

	/* root directory entries */
	segptr = diskblock[B_DIR];
	segptr = add_dir_entry(segptr, NILFS_ROOT_INO, ".");
	segptr = add_dir_entry(segptr, NILFS_ROOT_INO, "..");
#ifdef  ATIME_DIR_ENTRY
	segptr = add_dir_entry(segptr, NILFS_ATIME_INO, ".atime");
#else
	/* protect from deleting disk inode */
	inc_link_count(NILFS_ATIME_INO);
#endif
	/* add new root directory entry here */
	segptr = add_dir_entry(segptr, NILFS_SKETCH_INO, ".sketch");

	/* btree for files */
	for (i = 0; i < nfiles; i++) {
		ino = __le64_to_cpu(seg0_finfo[i].finfo.fi_ino);
		btree_block(B_REG + nfiles + i, 0, B_REG + i, ino, BT_REGFILE);
	}

	/* disk inode and inode btree */
	inode_blocks();

	superblock();
	checkpoint();
	segment_summary();
	checksum();
	write_disk(fd);

	exit(0);
}

void sb_to_le (struct nilfs_super_block *sb)
{
	CPU_TO_LE32(sb->s_rev_level);
	CPU_TO_LE16(sb->s_minor_rev_level);
	CPU_TO_LE16(sb->s_magic);
	CPU_TO_LE16(sb->s_bytes);
	CPU_TO_LE32(sb->s_crc_seed);
	CPU_TO_LE32(sb->s_nsegment);
	CPU_TO_LE64(sb->s_dev_size);
	CPU_TO_LE64(sb->s_first_data_block);
	CPU_TO_LE32(sb->s_log_block_size);
	CPU_TO_LE32(sb->s_blocks_per_segment);
	CPU_TO_LE32(sb->s_r_segments_count);
	CPU_TO_LE64(sb->s_last_segment);
	CPU_TO_LE64(sb->s_last_seq);
	CPU_TO_LE64(sb->s_free_blocks_count);
	CPU_TO_LE64(sb->s_ctime);
	CPU_TO_LE64(sb->s_mtime);
	CPU_TO_LE64(sb->s_wtime);
	CPU_TO_LE16(sb->s_mnt_count);
	CPU_TO_LE16(sb->s_max_mnt_count);
	CPU_TO_LE16(sb->s_state);
	CPU_TO_LE16(sb->s_errors);
	CPU_TO_LE64(sb->s_lastcheck);
	CPU_TO_LE32(sb->s_checkinterval);
	CPU_TO_LE32(sb->s_creator_os);
	CPU_TO_LE16(sb->s_def_resuid);
	CPU_TO_LE16(sb->s_def_resgid);
	CPU_TO_LE32(sb->s_first_ino);
	CPU_TO_LE16(sb->s_inode_size);
	CPU_TO_LE32(sb->s_c_interval);
	CPU_TO_LE32(sb->s_c_block_max);
}

void sum_to_le(struct nilfs_seg_summary *seg0_summary)
{
	CPU_TO_LE64(seg0_summary->ss_seq);
	CPU_TO_LE16(seg0_summary->ss_magic);
	CPU_TO_LE16(seg0_summary->ss_nblocks);
	CPU_TO_LE16(seg0_summary->ss_nfinfo);
	CPU_TO_LE16(seg0_summary->ss_nfbinfo);
	CPU_TO_LE16(seg0_summary->ss_nfblk);
	CPU_TO_LE16(seg0_summary->ss_nfbblk);
	CPU_TO_LE16(seg0_summary->ss_niblk);
	CPU_TO_LE16(seg0_summary->ss_nibblk);
	CPU_TO_LE64(seg0_summary->ss_create);
	CPU_TO_LE16(seg0_summary->ss_prev);
	CPU_TO_LE16(seg0_summary->ss_flags);
}

void ihdr_to_le(struct nilfs_inode_hdr *root_ino_hdr)
{
	CPU_TO_LE64(root_ino_hdr->ih_ino);
	CPU_TO_LE16(root_ino_hdr->ih_flags);
	CPU_TO_LE16(root_ino_hdr->ih_nfree);
	CPU_TO_LE32(root_ino_hdr->ih_free);
}

void
inode_to_le(struct nilfs_inode *root_i)
{
	CPU_TO_LE64(root_i->i_blocks);
	CPU_TO_LE64(root_i->i_size);
	CPU_TO_LE64(root_i->i_ctime);
	CPU_TO_LE64(root_i->i_mtime);
	CPU_TO_LE64(root_i->i_dtime);
	CPU_TO_LE32(root_i->i_uid);
	CPU_TO_LE32(root_i->i_gid);
	CPU_TO_LE16(root_i->i_mode);
	CPU_TO_LE16(root_i->i_links_count);
	CPU_TO_LE32(root_i->i_flags);
	CPU_TO_LE32(root_i->i_version);
	CPU_TO_LE64(root_i->i_block_root);
	CPU_TO_LE32(root_i->i_generation);
	CPU_TO_LE32(root_i->i_file_acl);
	CPU_TO_LE32(root_i->i_dir_acl);
}

void
btree_to_le(struct nilfs_btree_node *btree_hdr)
{
	CPU_TO_LE16(btree_hdr->bn_level);
	CPU_TO_LE16(btree_hdr->bn_nkeys);
}

void
checkpoint_to_le(struct nilfs_checkpoint *seg0_checkpoint)
{
	CPU_TO_LE16(seg0_checkpoint->cp_flags);
	CPU_TO_LE16(seg0_checkpoint->cp_bytes);
	CPU_TO_LE64(seg0_checkpoint->cp_inode_root);
	CPU_TO_LE64(seg0_checkpoint->cp_nblk_lseg);
	CPU_TO_LE64(seg0_checkpoint->cp_inodes_count);
	CPU_TO_LE64(seg0_checkpoint->cp_blocks_count);
}

void
segusage_hdr_to_le(struct segusage_hdr *segusage_hdr_tmp)
{
	CPU_TO_LE64(segusage_hdr_tmp->suh_free_blocks_count);
	CPU_TO_LE64(segusage_hdr_tmp->suh_last_segment);
	CPU_TO_LE64(segusage_hdr_tmp->suh_wtime);
	CPU_TO_LE64(segusage_hdr_tmp->suh_tm);
	CPU_TO_LE32(segusage_hdr_tmp->suh_aux);
	CPU_TO_LE32(segusage_hdr_tmp->suh_sum);
}

void
segusage_to_le(struct segusage *segusage_tmp)
{
	CPU_TO_LE32(segusage_tmp->su_prev_full_seg);
	CPU_TO_LE32(segusage_tmp->su_next_full_seg);
	CPU_TO_LE64(segusage_tmp->su_lastmod);
	CPU_TO_LE16(segusage_tmp->su_aux);
	CPU_TO_LE16(segusage_tmp->su_flags);
}

void
mygetopt(int argc, char *argv[])
{
	int c, n;

	blocksize = DEFAULT_BLOCKSIZE_BYTE;
	blksz_log = DEFAULT_BLOCKSIZE_LOG;
	block_per_seg = DEF_BLKS_PER_SEG;
	nilfs_ctime = time(NULL);

	while ((c = getopt(argc, argv, "b:B:cL:nq:VP:")) != EOF) {
		switch (c) {
		case 'b':
			blocksize = atoi(optarg);
			if (blocksize <= PAGE_SIZE && blocksize >= 1024
			    && ((blocksize - 1) & blocksize) == 0) {
				blksz_log = blocksize >> 10;
				for (n = 0; blksz_log > 1; blksz_log = blksz_log >> 1)
					n++;
				blksz_log = n;
			} else
				merr("Error: Bad blocksize.");
			break;
		case 'B':
			block_per_seg = atoi(optarg);
			break;
		case 'c':
			cflag++;
			break;
		case 'L':
			strncpy(volume_label, optarg, sizeof(volume_label));
			break;
		case 'n':
			nflag++;
			break;
		case 'q':
			quiet++;
			break;
		case 'V':
			show_version_only++;
			break;
		case 'P': /* Passive mode */
			nilfs_ctime = atol(optarg);
			if ((long)time(NULL) - (long)nilfs_ctime < 0) {
				char cbuf[26], *cbufp;

				ctime_r(&nilfs_ctime, cbuf);
                                if ((cbufp = rindex(cbuf, '\n')) != NULL)
                                        *cbufp = '\0';
				fprintf(stderr, "Warning: Future time: %s (%ld)\n", cbuf, (long)nilfs_ctime);
			}
			break;
		default:
			usage();
		}
	}
	if ((optind == argc) && !show_version_only)
		usage();

	if (show_version_only) {
		show_version();
		exit(0);
	}

	if (block_per_seg >= NILFS_SEG_MIN_BLOCKS && ((block_per_seg-1) & block_per_seg) == 0)
		segsize = block_per_seg * blocksize;
	else
		merr("Error: Bad number of blocks per segment.");

        if (argc > 0) {
                char *cp = strrchr(argv[0], '/');

                progname = (cp ? cp + 1 : argv[0]);
        }
	device = argv[optind];
	if (cflag)
		disk_scan(device, quiet, cflag);
}

void usage()
{
	fprintf(stderr, "usage:%s [-b block-size] [-B blocks per segment] [-c] \n[-L volume-label] [-q] [-r revision level] [-V] device \n", progname);
	exit(1);
}

void show_version()
{
	fprintf(stderr, "%s ver %d.%d\n", progname, NILFS_CURRENT_REV, NILFS_MINOR_REV);
}

void merr(char *m)
{
	show_version();
	fprintf(stderr, "%s\n", m);
	exit(1);
}

void
too_small_dev(__u64 dev_size, __u64 required_size, int fd)
{
	show_version();
	fprintf(stderr,	"Error: Too small device.\n       "
		"device size=%llu bytes, "
		"required size=%llu bytes.\n       ",
		dev_size, required_size);
	fprintf(stderr, "Please enlarge the device, "
		"or shorten segments with -B option.\n");
	close(fd);
	exit(1);
}

void seek_err(__u32 blocknum)
{
	show_version();
	fprintf(stderr, "Error: can not seek to %u blknum %u!\n",
		blocknum * blocksize,
		blocknum);
	perror("");
	exit(1);
}

void cannot_open_device(char *device)
{
	show_version();
	fprintf(stderr, "can not open device! (%s)\n", device);
	exit(1);
}

void cannot_get_devsize(char *device)
{
	show_version();
	fprintf(stderr, "can not get device size! (%s)\n", device);
	exit(1);
}

void cannot_write_device(int fd)
{
	show_version();
	fprintf(stderr, "can not write device!\n");
	close(fd);
	exit(1);
}

void disk_scan(char *device, int quiet, int cflag)
{
	char buf[1024];
	sprintf(buf, "badblocks -b %d %s %s %s\n", blocksize,
		quiet ? "" : "-s", (cflag > 1) ? "-w" : "", device);

	if (!quiet)
		printf("check blocks\n");

	system(buf);
}

void check_mount(int fd, char *device)
{
	FILE *fp;
	char line[BUFSIZE];

	fp = fopen(MOUNTS, "r");
	if (fp == NULL){
		show_version();
		fprintf(stderr, "Cannot open /etc/mtab!!\n");
		close(fd);
		exit(1);
	}

	while (fgets(line, BUFSIZE, fp) != NULL) {
		if (strncmp(strtok(line, " "), device, strlen(device)) == 0) {
			show_version();
			fprintf(stderr, "%s is currently mounted. You cannot make a filesystem on this device.\n", device);
			fclose(fp);
			close(fd);
			exit(1);
		}
	}
	fclose(fp);
}

uchar *copy_to_buffer(uchar *ptr1, uchar *ptr2, int bytes)
{
	memcpy(ptr1, ptr2, bytes);
	ptr1 += bytes;

	return ptr1;
}

void write_disk(int fd)
{
	char *fs_hdr;
	int hdr_size, len, i;
	__u32 crc_seed = sb->s_crc_seed;

	/* build fs header image including superblock */

	hdr_size = (blocksize >= NILFS_DISKHDR_SIZE) ? blocksize : NILFS_DISKHDR_SIZE;
	fs_hdr = (char *)calloc(hdr_size, sizeof(char));
	if (fs_hdr == NULL)
		merr("Error: Memory Allocation");
	len = read(fd, fs_hdr, NILFS_SB_OFFSET_BYTES);
	if (len < 0) {
		close(fd);
		merr("can not read device!");
	}

	sb_to_le(sb);
	/* superblock checksum */
	sb->s_sum = __cpu_to_le32(crc32c_le(crc_seed, (uchar *)sb, NILFS_SB_BYTES));

	memcpy(fs_hdr + NILFS_SB_OFFSET_BYTES, (char *)sb, sizeof(struct nilfs_super_block));

	/* start writing to HDD */
	/* write superblock */
	if (!quiet) {
		show_version();
		fprintf(stderr, "Start writing file system initial data to the device \n");
		fprintf(stderr, "Blocksize:%d  Device:%s\n", blocksize, device);
	}
	if (!nflag) {
		lseek64(fd, 0, SEEK_SET);
		if (write(fd, fs_hdr, hdr_size) < 0)
			cannot_write_device(fd);
		for (i = B_SS; i < B_MAX; i++)
			if (write(fd, diskblock[i], blocksize) < 0)
				cannot_write_device(fd);
		write_segusage_table(fd);
		write_superblock_replica(fs_hdr, fd);
		if (fsync(fd) < 0)
			cannot_write_device(fd);
	}
	if (!quiet)
		fprintf(stderr, "File system initialization succeeded !! \n");

	close(fd);
}

void
write_superblock_replica(char *fs_hdr, int fd)
{
	__u32 blocknum = sb->s_blocks_per_segment;

	if (lseek64(fd, (__u64)blocknum * blocksize, SEEK_SET) < 0)
		seek_err(blocknum);

	if (write(fd, fs_hdr, blocksize) < 0)
		cannot_write_device(fd);
}

void
write_segusage_table(int fd)
{
	__u32 usage_blocks, blocknum;
	__u32 nsegment = sb->s_nsegment;
	__u32 blocks_per_segment = sb->s_blocks_per_segment;
	struct segusage_hdr segusage_hdr_skeleton;
	uchar *segusage_table;
	unsigned long n_usage_b = (blocksize - usageh_size) / usage_size;
	long onchunk = (nsegment + n_usage_b - 1) / n_usage_b;
	unsigned long segnum = 0;
	int secondary, chunk, seglen;
#if 0
	{ /* for debug */
		printf("su hdr size %d su size %d\n",
		       usageh_size, usage_size);
		printf("dev_size %lld nsegment %d blocks_per_segment %d n_usage_b %lu\n",
		       sb->s_dev_size, nsegment, blocks_per_segment, n_usage_b);
	}
#endif
	onchunk = onchunk < NILFS_MAX_SU_NCHUNK ? onchunk : NILFS_MAX_SU_NCHUNK;
	init_segusage_hdr(&segusage_hdr_skeleton, sb);
	for (chunk = 0; chunk < onchunk; chunk++) {
		usage_blocks = nilfs_susage_blk_len(chunk, nsegment);
		seglen = nilfs_segchunk_seg_len(chunk, nsegment);
		segusage_table = make_segusage_table(chunk, segnum, seglen,
						     &segusage_hdr_skeleton, usage_blocks);
		for (secondary = 0; secondary < 2; secondary++) {
			blocknum = nilfs_susage_blk_num(secondary, chunk, nsegment, blocks_per_segment);
#if 0
			printf("chunk %2d-%d seg %lu blk %d seg %d/%d len %d\n",
			       chunk, secondary, segnum-1, blocknum,
			       blocknum/blocks_per_segment, blocknum%blocks_per_segment,
			       usage_blocks);
#endif
			if (lseek64(fd, (__u64)blocknum * blocksize, SEEK_SET) < 0)
				seek_err(blocknum);
			if (write(fd, segusage_table, blocksize * usage_blocks) < 0)
				cannot_write_device(fd);
		}
		segnum += seglen;
		free(segusage_table);
	}
}

uchar *
make_segusage_table(__u32 segchunknum,
		    unsigned long segnum, int seglen,
		    struct segusage_hdr *segusage_hdr_skeleton,
		    __u32 usage_blocks)
{
	__u32 nsegment = sb->s_nsegment;
	__u32 blocks_per_segment = sb->s_blocks_per_segment;
	unsigned long n_usage_b = (blocksize - usageh_size) / usage_size;
	__u32 checksum, block;
	uchar *ptr, *segusage_table;
	struct segusage_hdr segusage_hdr_tmp;
	struct segusage segusage_tmp;
	unsigned long sseg;
	unsigned int ns;
	int nseg;

	memcpy(&segusage_hdr_tmp, segusage_hdr_skeleton, usageh_size);

	if (posix_memalign((void **)&segusage_table, blocksize, blocksize*usage_blocks) != 0)
		merr("Error: Memory Allocation");
	memset(segusage_table, 0, blocksize * usage_blocks);

	ptr = segusage_table;
	/* build structure of segusage */
	segusage_hdr_to_le(&segusage_hdr_tmp);

	sseg = segnum;
	nseg = nilfs_segchunk_seg_len(segchunknum, nsegment);

	block = ns = 0;
	for (; segnum < sseg + seglen; segnum++) {
		if (ns == 0)
			ptr = copy_to_buffer(ptr, (uchar *)&segusage_hdr_tmp, usageh_size);

		memset(&segusage_tmp, 0, usage_size);
		if (segchunknum == 0 && segnum == 0) {
			segusage_tmp.su_prev_full_seg = SU_NOSEG;
			segusage_tmp.su_flags = SEGUSE_USED|SEGUSE_ACTIVE;
		} else
			segusage_tmp.su_prev_full_seg = segnum - 1;
		if (segnum == nsegment - 1)
			segusage_tmp.su_next_full_seg = SU_NOSEG;
		else
			segusage_tmp.su_next_full_seg = segnum + 1;
		if (nilfs_seg_blk_len(segnum, nsegment, blocks_per_segment) < blocks_per_segment)
			segusage_tmp.su_flags |= SEGUSE_SHORTLEN;
		segusage_to_le(&segusage_tmp);
		ptr = copy_to_buffer(ptr, (uchar *)&segusage_tmp, usage_size);

		if (++ns == n_usage_b) {
			ptr = MOVE_TO_NEXT_BLOCK(ptr);
			block++;
			ns = 0;
		}
	}
	for (block = 0; block < usage_blocks; block++) {
		checksum = crc32c_le(sb->s_crc_seed, segusage_table + blocksize * block + sizeof(__le32),
				     blocksize - sizeof(__le32));
		CPU_TO_LE32(checksum);
		memcpy(segusage_table + blocksize * block, (char *)&checksum, sizeof(__le32));
	}
	return segusage_table;
}

/* segment chunk number -> number of segments of the segment chunk */
unsigned long
nilfs_segchunk_seg_len(long segchunknum, unsigned long nsegment)
{
	unsigned long n_usage_b = (blocksize - usageh_size) / usage_size;
	long onchunk = (nsegment + n_usage_b - 1) / n_usage_b;
	long nb, nbm, s;

	if (segchunknum < 0 || NILFS_MAX_SU_NCHUNK <= segchunknum) {
		show_version();
		fprintf(stderr, "segchunknum range error\n");
	}
	if (onchunk <= NILFS_MAX_SU_NCHUNK) {
		if (onchunk <= segchunknum) {
			show_version();
			fprintf(stderr, "segchunknum range error\n");
		}
		if (segchunknum == onchunk - 1) { /* last segment chunk */
			if (nsegment % n_usage_b == 1) /* at least 2 segments */ /* type I */
				return 2;
			else
				return nsegment % n_usage_b;	/* type G */
		} else if (segchunknum == onchunk - 2 && nsegment % n_usage_b == 1)
			return n_usage_b - 1;	/* type H */
		else	/* type F */
			return n_usage_b;
	}

	nb = onchunk / NILFS_MAX_SU_NCHUNK;
	nbm = onchunk % NILFS_MAX_SU_NCHUNK;
	if (segchunknum < nbm - 1) { 	/* type A */
		return (nb + 1) * n_usage_b;
	} else if (segchunknum == nbm - 1) {	/* type B */
		if ((s = nsegment % n_usage_b))
			return nb * n_usage_b + s;
		else
			return (nb + 1) * n_usage_b;
	} else if (segchunknum == NILFS_MAX_SU_NCHUNK - 1 && nbm == 0 && (s = nsegment % n_usage_b)) { /* type E */
		return (nb - 1) * n_usage_b + s;
	} else {	/* type C, D */
		return nb * n_usage_b;
	}
}

/* segment chunk number -> segment number of beginning of the segment chunk */
unsigned long
nilfs_segchunk_seg_num(long segchunknum, unsigned long nsegment)
{
	unsigned long n_usage_b = (blocksize - usageh_size) / usage_size;
	long onchunk = (nsegment + n_usage_b - 1) / n_usage_b;
	long nb, nbm;

	if (segchunknum < 0 || NILFS_MAX_SU_NCHUNK <= segchunknum) {
		show_version();
		fprintf(stderr, "segchunknum range error\n");
	}
	if (onchunk <= NILFS_MAX_SU_NCHUNK) {
		if (onchunk <= segchunknum) {
			show_version();
			fprintf(stderr, "segchunknum range error\n");
		}
		if (segchunknum == onchunk - 1 && nsegment % n_usage_b == 1)
			return segchunknum * n_usage_b - 1;	/* type I */
		else	/* type F, G, H */
			return segchunknum * n_usage_b;
	}

	nb = onchunk / NILFS_MAX_SU_NCHUNK;
	nbm = onchunk % NILFS_MAX_SU_NCHUNK;
	if (segchunknum < nbm) {	/* type A, B */
		return segchunknum * (nb + 1) * n_usage_b;
	} else if (nbm == 0) {	/* type D, E */
		return segchunknum * nb * n_usage_b;
	} else {		/* type C */
		return nsegment - nb * n_usage_b * (NILFS_MAX_SU_NCHUNK - segchunknum);
	}
}

/* segment chunk number -> number of blocks for segment usage table */
unsigned long
nilfs_susage_blk_len(long segchunknum, unsigned long nsegment)
{
	unsigned long n_usage_b = (blocksize - usageh_size) / usage_size;
	long onchunk = (nsegment + n_usage_b - 1) / n_usage_b;

	if (segchunknum < 0 || NILFS_MAX_SU_NCHUNK <= segchunknum) {
		show_version();
		fprintf(stderr, "segchunknum range error\n");
	}
	if (onchunk <= NILFS_MAX_SU_NCHUNK) /* type F, G, H, I */
		return 1;
	else if (segchunknum < onchunk % NILFS_MAX_SU_NCHUNK) /* type A, B */
		return onchunk / NILFS_MAX_SU_NCHUNK + 1;
	else			/* type C, D, E */
		return onchunk / NILFS_MAX_SU_NCHUNK;
}

/* segment number -> segment chunk number */
unsigned long
nilfs_seg_segchunk_num(long segnum, unsigned long nsegment)
{
	unsigned long n_usage_b = (blocksize - usageh_size) / usage_size;
	long onchunk = (nsegment + n_usage_b - 1) / n_usage_b;
	long nb, nbm, segchunknum;

	if (segnum < 0 || nsegment <= segnum) {
		show_version();
		fprintf(stderr, "segnum range error\n");
	}
	if (onchunk <= NILFS_MAX_SU_NCHUNK) {
		if (nsegment % n_usage_b == 1 && segnum == nsegment - 2)
			segchunknum = segnum / n_usage_b + 1;	/* type I */
		else
			segchunknum = segnum / n_usage_b; 	/* type F, G, H */
	} else {
		nb = onchunk / NILFS_MAX_SU_NCHUNK;
		nbm = onchunk % NILFS_MAX_SU_NCHUNK;

		if (0 < nbm) { /* type A, B, C */
			if (nsegment - (NILFS_MAX_SU_NCHUNK - nbm) * nb * n_usage_b <= segnum) { /* type C */
				segchunknum = NILFS_MAX_SU_NCHUNK - (nsegment - segnum - 1) / (nb * n_usage_b) - 1;
			} else if (segnum < (nb + 1) * n_usage_b * (nbm - 1)) { /* type A */
				segchunknum = segnum / ((nb + 1) * n_usage_b);
			} else	{ /* type B */
				segchunknum = nbm - 1;
			}
		} else {	/* type D, E */
			segchunknum = segnum / (nb * n_usage_b);
		}
	}
	return segchunknum;
}

/* segment chunk number -> block number of segment usage table header */
unsigned long long
nilfs_susage_blk_num(int secondary, long segchunknum,
		     unsigned long nsegment, unsigned long blocks_per_segment)
{
	unsigned long n_usage_b = (blocksize - usageh_size) / usage_size;
	long onchunk = (nsegment + n_usage_b - 1) / n_usage_b;
	long nb, nbm;

	if (segchunknum < 0 || NILFS_MAX_SU_NCHUNK <= segchunknum) {
		show_version();
		fprintf(stderr, "segchunknum range error\n");
	}
	if (secondary) {
		long sseg = segchunknum + 1;
		long o = onchunk < NILFS_MAX_SU_NCHUNK ? onchunk : NILFS_MAX_SU_NCHUNK;

		if (sseg == o)
			sseg = 0;
		return (nilfs_segchunk_seg_num(sseg, nsegment)
			+ nilfs_segchunk_seg_len(sseg, nsegment) / 2)
			* blocks_per_segment - nilfs_susage_blk_len(segchunknum, nsegment);
	}
	if (onchunk <= NILFS_MAX_SU_NCHUNK) {
		if (onchunk <= segchunknum) {
			show_version();
			fprintf(stderr, "segchunknum range error\n");
		}
		if (segchunknum == onchunk - 1)	/* type G, I */
			return blocks_per_segment * nsegment - 1;
		else if (segchunknum == onchunk - 2 && nsegment % n_usage_b == 1) /* last segment chunk has 2 segments */ /* type H */
			return (segchunknum + 1 ) * n_usage_b * blocks_per_segment - blocks_per_segment - 1;
		else	/* type F */
			return (segchunknum + 1) * n_usage_b * blocks_per_segment - 1;
	}
	nb = onchunk / NILFS_MAX_SU_NCHUNK;
	nbm = onchunk % NILFS_MAX_SU_NCHUNK;
	if (segchunknum < nbm - 1) {	/* type A */
		return n_usage_b * (nb + 1) * blocks_per_segment * (segchunknum + 1) - (nb + 1);
	} else if (segchunknum == nbm - 1) {	/* type B */
		return blocks_per_segment * nsegment
			- (n_usage_b * nb * blocks_per_segment * (NILFS_MAX_SU_NCHUNK - segchunknum - 1))
			- (nb + 1);
	} else if (nbm == 0) {
		if (segchunknum == NILFS_MAX_SU_NCHUNK - 1)	/* type E */
			return blocks_per_segment * nsegment - nb;
		else	/* type D */
			return (segchunknum + 1) * n_usage_b * nb * blocks_per_segment - nb;
	} else {	/* type C */
		return blocks_per_segment * nsegment
			- (n_usage_b * nb * blocks_per_segment * (NILFS_MAX_SU_NCHUNK - segchunknum - 1))
			- nb;
	}
}

/* segnum segment length in block */
unsigned int
nilfs_seg_blk_len(unsigned long segnum, unsigned long nsegment, unsigned long blocks_per_segment)
{
	unsigned long segchunknum = nilfs_seg_segchunk_num(segnum, nsegment);
	unsigned long chunk_start = nilfs_segchunk_seg_num(segchunknum, nsegment);
	unsigned long chunk_len = nilfs_segchunk_seg_len(segchunknum, nsegment);
	unsigned long blklen;

	if (segnum == chunk_start + chunk_len - 1) {	/* last segment in the chunk */
		blklen = blocks_per_segment - nilfs_susage_blk_len(segchunknum, nsegment);
	} else {
		unsigned long n_usage_b = (blocksize - usageh_size) / usage_size;
		long onchunk = (nsegment + n_usage_b - 1) / n_usage_b;
		long o = onchunk < NILFS_MAX_SU_NCHUNK ? onchunk : NILFS_MAX_SU_NCHUNK;
		long sseg = segchunknum - 1;

		if (sseg < 0)
			sseg = o - 1;
		if (segnum == chunk_start + chunk_len / 2 - 1) {
			/* secondary usage table */
			blklen = blocks_per_segment - nilfs_susage_blk_len(sseg, nsegment);
		} else {
			blklen = blocks_per_segment;
		}
	}
	if (segnum == 0)	/* super block */
		blklen--;
	else if (segnum == 1)
		blklen--;	/* super block replica */
	return blklen;
}

void
init_segusage_hdr(struct segusage_hdr *segusage_hdr_skeleton, struct nilfs_super_block *sb)
{
	segusage_hdr_skeleton->suh_sum = 0;
	segusage_hdr_skeleton->suh_aux = 0;
	segusage_hdr_skeleton->suh_free_blocks_count = sb->s_free_blocks_count;
	segusage_hdr_skeleton->suh_last_segment = sb->s_last_segment;
	segusage_hdr_skeleton->suh_wtime = sb->s_wtime;
	segusage_hdr_skeleton->suh_tm = sb->s_wtime;
}

/* Local Variables:	*/
/* eval: (c-set-style "linux")	*/
/* End:			*/
