/*
 * inspect.c - NILFS tool, inspect nilfs format raw disk
 *
 * Copyright (C) 2005, 2006 Nippon Telegraph and Telephone Corporation.
 *
 * This file is part of NILFS.
 *
 * NILFS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * NILFS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NILFS; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * inspect.c,v 1.18 2006/06/05 17:01:14 ryusuke Exp
 *
 * Written by Amagai Yoshiji <amagai@osrg.net>
 */

#include "inspect.h"

int blksperseg;  /* blocks per segment */
off_t blocksize; /* block size in [byte] */
dbn_t c_blk_n;	 /* current_block_number */

/* */
char *devname;
char *progname = "inspect";
int devfd = 1;

/* command line options */
int sketch_blk = 0;	/* -s blknumber: get sketch data */
char *sketch_out = NULL; /* output filename for sketch data */
			/* use stdout if NULL */

int date_in_epoch = 0;	/* -e: output date in second from epoch */


struct nilfs_seg_info seg_info;
jmp_buf	main_env;
struct sigaction int_act;

#define usage_size	sizeof(struct segusage)
#define usageh_size	sizeof(struct segusage_hdr)

void printhelp();
char *magic_str(unsigned int);

/* {comnand name, comman number, nubmer of args} */
struct ins inscom[] = {
	{"dump",	1, 2},	/* dump block */
	{"d",		1, 2},	/* dump block */
	{"super",	2, 0},	/* dump main superblock */
	{"set",		3, 1},	/* set current block # */
	{"segment",	4, 1},	/* print segment summary */
	{"ss",		4, 1},
	{"inodehdr",	5, 1},	/* print inode header */
	{"ih",		5, 1},
	{"inode",	6, 3},	/* print disk inode */
	{"btree",	7, 1},	/* print btree node */
	{"bentry",	8, 3},	/* print btree entry */
	{"be",		8, 3},
	{"lookup",	9, 2},  /* btree lookup (root, key) */
	{"listcp",	10, 1}, /* list check point from beginning */
	{"listpseg",	11, 1}, /* list partial segment from beginning */
	{"su_hdr",	12, 2}, /* print segment usage headers */
	{"sh",		12, 2}, /* print segment usage headers */

	{"p",		127, 0}, /* inspector status */
	{"print",	127, 0},
	{"number",	128, 1}, /* provided a number only */
	{"bye",		510, 0},
	{"exit",	510, 0},
	{"quit",	510, 0},
	{"q",		510, 0},
	{"help",	511, 0},
	{0, 0, 0} /* end mark */
};

/* SIGINT goto top level */

static void
int_sig(int x)
{
	putchar('\n');
	fflush(stdout);
	siglongjmp(main_env, 1);
}

static void
init_sig()
{
	int_act.sa_handler = int_sig;
	int_act.sa_flags = 0;
	sigemptyset(&int_act.sa_mask);
	sigaddset(&int_act.sa_mask, SIGINT);
	sigaction(SIGINT, &int_act, NULL);
}

/* inspect command */

void
dump_block(dbn_t blk, int off)
{
	__le32 *buffer;
	int i, j;
	int n = 1024; /* bytes per command */
	int c = 8;    /* 32bit per line */
	int bn = n / 4;

	c_blk_n = blk;
	off /= n;

	buffer = (__le32 *)malloc(blocksize);
	if (buffer == NULL) {
		fprintf(stderr, "cannot allocate memory\n");
		return;
	}
	if (readstruct(buffer, blocksize, blk * blocksize) <= 0) {
		free(buffer);
		return;
	}
	printf("block# %ld, segment# %ld\n", c_blk_n, c_blk_n / blksperseg);
	for (i = 0; i < bn / c ; i++) {
		printf("0x%04x  ", i * c * 4 + off * n);
		for (j = 0; j < c; j++) {
			printf("0x%08x  ", __le32_to_cpu(buffer[i * c + j + off * bn]));
		}
		printf("\n");
	}
	free(buffer);
	c_blk_n = blk;
}

void
print_super()
{
	struct nilfs_sb_info *sbi = &sb_info;
	unsigned long n_usage_b;
	long onchunk;
	int nc, m;
	time_t tc;

	if (nilfs_fill_super() < 0)
		return;
	n_usage_b = (blocksize - usageh_size) / usage_size;
	onchunk = (sbi->s_nsegment + n_usage_b - 1) / n_usage_b;
	nc = onchunk < NILFS_MAX_SU_NCHUNK ? onchunk : NILFS_MAX_SU_NCHUNK;

	m = __le16_to_cpu(superblock.s_magic);
	printf("magic\t0x%04x rev %d.%d\n",
	       m, sbi->s_rev_level, sbi->s_minor_rev_level);
	if (m != NILFS_SUPER_MAGIC) {
		printf(" !! 0x%04x is not magic of superblock\n", m);
		printf(" !! may be %s\n", magic_str(m));
	} else if (sbi->s_rev_level != NILFS_CURRENT_REV ||
		   sbi->s_minor_rev_level != NILFS_MINOR_REV) {
		printf(" !! revision mismatch "
		       "(superblock rev.=%d.%d, current rev.=%d.%d). \n",
		       sbi->s_rev_level,
		       sbi->s_minor_rev_level,
		       NILFS_CURRENT_REV, NILFS_MINOR_REV);
	}
	printf("block_size\t%ju\n", blocksize);
	printf("s_nsegment\t%lu\tsegment chunk %d\n", sbi->s_nsegment, nc);
	printf("s_blocks_per_segment\t%lu\n", sbi->s_blocks_per_segment);
	printf("s_crc_seed\t0x%x\n", sbi->s_crc_seed);
	printf("s_last_segment [blk#]\t%lu\n", sbi->s_last_pseg);
	printf("s_last_seq [seq#]\t%llu\n", __le64_to_cpu(superblock.s_last_seq));

	tc = __le64_to_cpu(superblock.s_ctime);
	if (date_in_epoch)
		printf("s_ctime\t%ld\n", tc);
	else
		printf("s_ctime\t%s", ctime(&tc));

	if (superblock.s_volume_name[0] != '\0') {
		char vlabel[16 + 1];

		memcpy(vlabel, superblock.s_volume_name, sizeof(superblock.s_volume_name));
		vlabel[16] = 0;
		printf("s_volume_name\t%s\n", vlabel);
	}
	{
		char uuid_str[36 + 1];

		uuid_unparse(superblock.s_uuid, uuid_str);
		printf("s_uuid\t%s\n", uuid_str);
	}
	c_blk_n = sbi->s_last_pseg;
}

int
load_checkpoint(dbn_t *root, dbn_t cp_start, dbn_t cp_end,
		struct nilfs_cp_info *cpi)
{
	struct nilfs_checkpoint cp;
	int cp_bytes;
	unsigned long cp_sum;
	__u32 crc = sb_info.s_crc_seed;

	if (readstruct(&cp, sizeof(cp), cp_start * blocksize) <= 0) return -1;

	*root = __le64_to_cpu(cp.cp_inode_root);
	cp_sum = __le32_to_cpu(cp.cp_sum);
	cp_bytes = __le16_to_cpu(cp.cp_bytes);
	/* sketch data is not included in crc calculation */
	crc = crc32c_le(crc, (unsigned char *)&cp + sizeof(cp.cp_sum), cp_bytes - sizeof(cp.cp_sum));
	if (crc != cp_sum)
		return -2;
	if (cpi) {
		cpi->flags = __le16_to_cpu(cp.cp_flags);
		cpi->nblk_lseg = (unsigned long)__le64_to_cpu(cp.cp_nblk_lseg);
		cpi->cp_sum = cp_sum;
		if (40 < cp_bytes) {	/* checkpoint with sketch size */
			cpi->sketch_size = __le32_to_cpu(cp.cp_sketch_size);
			if (cpi->sketch)
				readstruct((void *)cpi->sketch, cpi->sketch_size, cp_start * blocksize + cp_bytes);
		}
	}
	return 0;
}

__u32
ss_sum(dbn_t blk, unsigned long sum_bytes)
{
	int nblk, off;
	unsigned int b;
	__u32 crc = sb_info.s_crc_seed;
	__le32 *buffer = (__le32 *)malloc(blocksize);
	struct nilfs_seg_summary ss;

	if (buffer == NULL) {
		return 0;
	}
	nblk = (sum_bytes - 1) / blocksize + 1;
	off = sizeof(ss.ss_datasum) + sizeof(ss.ss_sumsum);

	for(b = 0; b < nblk; b++) {
		unsigned long size = min(sum_bytes, blocksize);
		if (readstruct(buffer, size, (blk + b) * blocksize) <= 0) {
			free(buffer);
			return 0;
		}
		crc = crc32c_le(crc, (unsigned char *)buffer + off, size - off);
		sum_bytes -= size;
		off = 0;
	}
	free(buffer);
	return crc;
}

void
print_ss_flags(int flags)
{
	int n = 0;
	if (flags == 0) {putchar('-'); return;}
#define nbar(n) ((n)++?"|":"")
	if (flags&SS_FJCP) {printf("MajorCP"); n++;}
	if (flags&SS_FNCP) {printf("%sMinorCP",nbar(n));}	
	if (flags&SS_FSBLK) {printf("%sSuperBlock",nbar(n));}	
	if (flags&SS_DDIR) {printf("%sDirtyDir",nbar(n));}	
	if (flags&SS_LOGBGN) {printf("%sLogiBegin",nbar(n));}	
	if (flags&SS_LOGEND) {printf("%sLogiEnd",nbar(n));}	
	if (flags&SS_SYNDT) {printf("%sSyncData",nbar(n));}
#undef nbar
}

void
set_seg_info(struct nilfs_seg_info *ssi, struct nilfs_seg_summary *ss)
{
	ssi->nfinfo = __le16_to_cpu(ss->ss_nfinfo);	/* number of file info structures */
	ssi->nblk_file = __le16_to_cpu(ss->ss_nfblk);	/* number of file blocks in seg */
	ssi->nfbinfo = __le16_to_cpu(ss->ss_nfbinfo);	/* number of fbinfo structures */
	ssi->nblk_inode = __le16_to_cpu(ss->ss_niblk);	/* number of inode block in seg */
							/* = number of inode block info structures */
	ssi->nblocks = __le16_to_cpu(ss->ss_nblocks);	/* number of blocks in seg */
	ssi->nblk_fbt = __le16_to_cpu(ss->ss_nfbblk);	/* number of file B-tree blocks in seg */
							/* = number of file block info structures */
	ssi->nblk_ibt = __le16_to_cpu(ss->ss_nibblk);	/* number of inode B-tree block in seg */
	ssi->flags = __le16_to_cpu(ss->ss_flags);
	ssi->ctime.tv_sec = __le64_to_cpu(ss->ss_create);
	ssi->ctime.tv_usec = 0;
}

int
set_seg_info_crc(dbn_t blk, struct nilfs_seg_info *ssi, struct nilfs_seg_summary *ss)
{
	unsigned long sum_bytes;
	__u32 sumsum, crc;

	set_seg_info(ssi, ss);
	sumsum = __le32_to_cpu(ss->ss_sumsum);
	sum_bytes = nilfs_recalc_segsum_size(ssi, blocksize);
	crc = ss_sum(blk, sum_bytes);
	return crc == sumsum;
}

void
print_ss(dbn_t blk)	/* segment ss */
{
	struct nilfs_seg_summary ss;
	struct nilfs_seg_info *ssi = &seg_info;
	dbn_t iroot, cp_start, cp_end;
	unsigned long sum_bytes;
	__u32 sumsum, crc;
	int cp_ret = 0;

	if (readstruct(&ss, sizeof(ss), blk * blocksize) <= 0) return;

	set_seg_info(ssi, &ss);
	sumsum = __le32_to_cpu(ss.ss_sumsum);
	sum_bytes = nilfs_recalc_segsum_size(ssi, blocksize);

	cp_start = blk + nilfs_seg_blocks(ssi, 0);
	cp_end = blk + ssi->nblocks - 1;

	if (ssi->flags & (SS_FJCP|SS_FNCP)) {
		cp_ret = load_checkpoint(&iroot, cp_start, cp_end, NULL);
	}

	printf("block#\t\t%lu\t", blk);
	printf("full segment#\t%llu\n", __le64_to_cpu(ss.ss_seq));
	printf("sumsum\t\t0x%x\t", sumsum);
	crc = ss_sum(blk, sum_bytes);
	if (sumsum == crc)
		printf("checksum ok\n");
	else
		printf("checksum not match 0x%x\n", crc);
	printf("ss_nblocks\t%u\t", ssi->nblocks);
	printf("(summary size\t%u blk, %lu bytes)\n", ssi->nblk_sum, sum_bytes);
	printf("segment size (w/o checkpoint)\t\t%lu blk\n", nilfs_seg_blocks(ssi, 0));
	printf("ss_nfinfo\t%u\t", ssi->nfinfo);
	printf("ss_nfbinfo\t%u\n", ssi->nfbinfo);
	printf("ss_nfblk\t%u\t", ssi->nblk_file);
	printf("ss_nfbblk\t%u\n", ssi->nblk_fbt);
	printf("ss_niblk\t%u\t", ssi->nblk_inode);
	printf("ss_nibblk\t%u\n", ssi->nblk_ibt);

	/* N.I.Y
	   printf("ss_prev\t\t%u (rel) ==\t%lu (blk#)\n",
	          __le16_to_cpu(ss.ss_prev), blk - __le16_to_cpu(ss.ss_prev));
	 */

	printf("flags\t\t0x%04x ", ssi->flags);
	print_ss_flags(ssi->flags);
	if (ssi->flags & (SS_FJCP|SS_FNCP)) {
		if (cp_ret == -2)
			printf("\ncheckpoint checksum mismatch!");
		else
			printf("\ncheckpoint checksum ok");
	}
	printf("\n");
	if (ssi->flags & (SS_FJCP|SS_FNCP)) {
		printf("inode_root\t%lu\n", iroot);
	}
	if (date_in_epoch)
		printf("ss_create\t%ld\n", ssi->ctime.tv_sec);
	else
		printf("ss_create\t%s", ctime(&ssi->ctime.tv_sec));
}

void
print_inode_hdr(dbn_t blk)
{
	struct nilfs_inode_hdr hdr;

	if (readstruct(&hdr, sizeof(hdr), blk * blocksize) <= 0) return;
	printf("ih_ino\t\t%llu\n", __le64_to_cpu(hdr.ih_ino));
	printf("ih_flags\t0x%x\n", __le16_to_cpu(hdr.ih_flags));
	printf("ih_nfree\t%u\n", __le16_to_cpu(hdr.ih_nfree));
	printf("ih_free\t\t%u\n", __le32_to_cpu(hdr.ih_free));
	c_blk_n = blk;
}

#define DT_UNKNOWN      0
#define DT_FIFO         1
#define DT_CHR          2
#define DT_DIR          4
#define DT_BLK          6
#define DT_REG          8
#define DT_LNK          10
#define DT_SOCK         12
#define DT_WHT          14

void
inode_mode(int mode)
{
	mode >>= 12;
	mode &= 15;

	if (mode == DT_FIFO) {printf("fifo");}
	if (mode == DT_CHR) {printf("chr");}
	if (mode == DT_DIR) {printf("dir");}
	if (mode == DT_BLK) {printf("blk");}
	if (mode == DT_REG) {printf("reg");}
	if (mode == DT_LNK) {printf("lnk");}
	if (mode == DT_SOCK) {printf("sock");}
	if (mode == DT_WHT) {printf("wht");}
}

void
print_inode(off_t blk, int nth, int count)
{
	struct nilfs_inode *vi;
	struct nilfs_inode_hdr *h;
	char *buffer;
	int i, ino;

	if ((buffer = (char *)malloc(blocksize)) == NULL) {
		fprintf(stderr, "cannot allocate memory\n");
		return;
	}
	if (readstruct(buffer, blocksize, blk * blocksize) <= 0) {
		free(buffer);		
		return;
	}
	h = (struct nilfs_inode_hdr *)buffer;
	vi = (struct nilfs_inode *)(buffer + sizeof(struct nilfs_inode_hdr));
	ino = __le64_to_cpu(h->ih_ino);
	vi += nth;
	ino += nth;

	for (i = 0; i < count; i++, vi++) {
		printf("%4d  ", ino+i);
		printf("blks %3llu  ", __le64_to_cpu(vi->i_blocks));
		printf("size %5llu  ", __le64_to_cpu(vi->i_size));
		printf("links %u  ", __le16_to_cpu(vi->i_links_count));
		printf("mode 0%o ", __le16_to_cpu(vi->i_mode));
		inode_mode(__le16_to_cpu(vi->i_mode));
		printf("  blk_root [blk#] %llu  ", __le64_to_cpu(vi->i_block_root));
		printf("flags 0x%08x ", __le32_to_cpu(vi->i_flags));
		printf("ver %u\n", __le32_to_cpu(vi->i_version));
	}

	free(buffer);
}

void
print_btree_hdr(off_t blk)
{
	struct nilfs_btree_node bn;

	if (readstruct(&bn, sizeof(bn), blk * blocksize) <= 0) return;

	printf("level %u\tnkey %u\n",
	       __le16_to_cpu(bn.bn_level),
	       __le16_to_cpu(bn.bn_nkeys));
	c_blk_n = blk;
}

void
print_btree_key(off_t blk, int nth, int count)
{
	__le64 *buffer, key, ptr;
	int p = (blocksize / 8 - 2) / 2 + 2;
	int i;

	if (blocksize / 16 - 3 < nth + count - 1) {
		fprintf(stderr, "range over %d max %ju\n", nth, blocksize / 16 - 3);
		return;
	}
	buffer = (__le64 *)malloc(blocksize);
	if (buffer == NULL) {
		fprintf(stderr, "cannot allocate memory\n");
		return;
	}
	if (readstruct(buffer, blocksize, blk * blocksize) <= 0) {
		free(buffer);
		return;
	}
	for (i = 0; i < count; i++) {
		key = __le64_to_cpu(buffer[nth+i+2]);
		ptr = __le64_to_cpu(buffer[nth+i+p]);
	       printf("%2d key %llu (0x%llx) ptr %llu (0x%llx)\n",
		      nth+i, key, key, ptr, ptr);
	}
	free(buffer);
}

void
btree_lookup(long int root, long int key)
{
}

/* print segment usage headers */

/* segment chunk number -> number of segments of the segment chunk */
unsigned long
nilfs_segchunk_seg_len(struct nilfs_sb_info *sbi, long segchunknum)
{
	unsigned long n_usage_b = (blocksize - usageh_size) / usage_size;
	long onchunk = (sbi->s_nsegment + n_usage_b - 1) / n_usage_b;
	long nb, nbm, s;

	if (onchunk <= NILFS_MAX_SU_NCHUNK) {
		if (segchunknum == onchunk - 1) { /* last segment chunk */
			if (sbi->s_nsegment % n_usage_b == 1) /* at least 2 segments */ /* type I */
				return 2;
			else
				return sbi->s_nsegment % n_usage_b;	/* type G */
		} else if (segchunknum == onchunk - 2 && sbi->s_nsegment % n_usage_b == 1)
			return n_usage_b - 1;	/* type H */
		else	/* type F */
			return n_usage_b;
	}

	nb = onchunk / NILFS_MAX_SU_NCHUNK;
	nbm = onchunk % NILFS_MAX_SU_NCHUNK;
	if (segchunknum < nbm - 1) { 	/* type A */
		return (nb + 1) * n_usage_b;
	} else if (segchunknum == nbm - 1) {	/* type B */
		if ((s = sbi->s_nsegment % n_usage_b))
			return nb * n_usage_b + s;
		else
			return (nb + 1) * n_usage_b;
	} else if (segchunknum == NILFS_MAX_SU_NCHUNK - 1 && nbm == 0 && (s = sbi->s_nsegment % n_usage_b)) { /* type E */
		return (nb - 1) * n_usage_b + s;
	} else {	/* type C, D */
		return nb * n_usage_b;
	}
}

/* segment chunk number -> segment number of beginning of the segment chunk */
unsigned long
nilfs_segchunk_seg_num(struct nilfs_sb_info *sbi, long segchunknum)
{
	unsigned long n_usage_b = (blocksize - usageh_size) / usage_size;
	long onchunk = (sbi->s_nsegment + n_usage_b - 1) / n_usage_b;
	long nb, nbm;

	if (onchunk <= NILFS_MAX_SU_NCHUNK) {
		if (segchunknum == onchunk - 1 && sbi->s_nsegment % n_usage_b == 1)
			return segchunknum * n_usage_b - 1;	/* type I */
		else	/* type F, G, H */
			return segchunknum * n_usage_b;
	}

	nb = onchunk / NILFS_MAX_SU_NCHUNK;
	nbm = onchunk % NILFS_MAX_SU_NCHUNK;
	if (segchunknum < nbm) {	/* type A, B */
		return segchunknum * (nb + 1) * n_usage_b;
	} else if (nbm == 0) {	/* type D, E */
		return segchunknum * nb * n_usage_b;
	} else {		/* type C */
		return sbi->s_nsegment - nb * n_usage_b * (NILFS_MAX_SU_NCHUNK - segchunknum);
	}
}

/* segment chunk number -> number of blocks for segment usage table */
unsigned long
nilfs_susage_blk_len(struct nilfs_sb_info *sbi, long segchunknum)
{
	unsigned long n_usage_b = (blocksize - usageh_size) / usage_size;
	long onchunk = (sbi->s_nsegment + n_usage_b - 1) / n_usage_b;

	if (onchunk <= NILFS_MAX_SU_NCHUNK) /* type F, G, H, I */
		return 1;
	else if (segchunknum < onchunk % NILFS_MAX_SU_NCHUNK) /* type A, B */
		return onchunk / NILFS_MAX_SU_NCHUNK + 1;
	else			/* type C, D, E */
		return onchunk / NILFS_MAX_SU_NCHUNK;
}

dbn_t
nilfs_susage_blk_num(struct nilfs_sb_info *sbi, int secondary, int segchunknum)
{
	unsigned long n_usage_b = (blocksize - usageh_size) / usage_size;
	long onchunk = (sbi->s_nsegment + n_usage_b - 1) / n_usage_b;
	long nb, nbm;

	if (secondary) {
		long sseg = segchunknum + 1;
		long o = onchunk < NILFS_MAX_SU_NCHUNK ? onchunk : NILFS_MAX_SU_NCHUNK;

		if (sseg == o)
			sseg = 0;
		return (nilfs_segchunk_seg_num(sbi, sseg) + nilfs_segchunk_seg_len(sbi, sseg) / 2)
			* sbi->s_blocks_per_segment - nilfs_susage_blk_len(sbi, segchunknum);
	}
	if (onchunk <= NILFS_MAX_SU_NCHUNK) {
		if (segchunknum == onchunk - 1)	/* type G, I */
			return sbi->s_blocks_per_segment * sbi->s_nsegment - 1;
		else if (segchunknum == onchunk - 2 && sbi->s_nsegment % n_usage_b == 1) /* last segment chunk has 2 segments */ /* type H */
			return (segchunknum + 1 ) * n_usage_b * sbi->s_blocks_per_segment - sbi->s_blocks_per_segment - 1;
		else	/* type F */
			return (segchunknum + 1) * n_usage_b * sbi->s_blocks_per_segment - 1;
	}
	nb = onchunk / NILFS_MAX_SU_NCHUNK;
	nbm = onchunk % NILFS_MAX_SU_NCHUNK;
	if (segchunknum < nbm - 1) {	/* type A */
		return n_usage_b * (nb + 1) * sbi->s_blocks_per_segment * (segchunknum + 1) - (nb + 1);
	} else if (segchunknum == nbm - 1) {	/* type B */
		return sbi->s_blocks_per_segment * sbi->s_nsegment
			- (n_usage_b * nb * sbi->s_blocks_per_segment * (NILFS_MAX_SU_NCHUNK - segchunknum - 1))
			- (nb + 1);
	} else if (nbm == 0) {
		if (segchunknum == NILFS_MAX_SU_NCHUNK - 1)	/* type E */
			return sbi->s_blocks_per_segment * sbi->s_nsegment - nb;
		else	/* type D */
			return (segchunknum + 1) * n_usage_b * nb * sbi->s_blocks_per_segment - nb;
	} else {	/* type C */
		return sbi->s_blocks_per_segment * sbi->s_nsegment
			- (n_usage_b * nb * sbi->s_blocks_per_segment * (NILFS_MAX_SU_NCHUNK - segchunknum - 1))
			- nb;
	}
}

/* segment number -> segment chunk number */
unsigned long
nilfs_seg_segchunk_num(struct nilfs_sb_info *sbi, long segnum)
{
	unsigned long n_usage_b = (blocksize - usageh_size) / usage_size;
	long onchunk = (sbi->s_nsegment + n_usage_b - 1) / n_usage_b;
	long nb, nbm, segchunknum;

	if (onchunk <= NILFS_MAX_SU_NCHUNK) {
		if (sbi->s_nsegment % n_usage_b == 1 && segnum == sbi->s_nsegment - 2)
			segchunknum = segnum / n_usage_b + 1;	/* type I */
		else
			segchunknum = segnum / n_usage_b; 	/* type F, G, H */
	} else {
		nb = onchunk / NILFS_MAX_SU_NCHUNK;
		nbm = onchunk % NILFS_MAX_SU_NCHUNK;

		if (0 < nbm) {	/* type A, B, C */
			if (sbi->s_nsegment - (NILFS_MAX_SU_NCHUNK - nbm) * nb * n_usage_b <= segnum) {/* type C */
				segchunknum = NILFS_MAX_SU_NCHUNK - (sbi->s_nsegment - segnum - 1) / (nb * n_usage_b) - 1;
			} else if (segnum < (nb + 1) * n_usage_b * (nbm - 1)) {/*  type A */
				segchunknum = segnum / ((nb + 1) * n_usage_b);
			} else	{/* type B */
				segchunknum = nbm - 1;
			}
		} else {	/* type D, E */
			segchunknum = segnum / (nb * n_usage_b);
		}
	}
	return segchunknum;
}

/* start block number of the usable data block in segment */
dbn_t
nilfs_seg_start_blk(struct nilfs_sb_info *sbi, long segnum)
{
        if (segnum == 0)
                return sbi->s_first_data_block;
        else if (segnum == 1)	/* super block replica */
                return sbi->s_blocks_per_segment + sbi->s_first_data_block;
        else
                return (dbn_t)segnum * sbi->s_blocks_per_segment;
}

/* segnum segment length in block */
unsigned int
nilfs_seg_blklen(struct nilfs_sb_info *sbi, unsigned long segnum)
{
	unsigned long segchunknum = nilfs_seg_segchunk_num(sbi, segnum);
	unsigned long chunk_start = nilfs_segchunk_seg_num(sbi, segchunknum);
	unsigned long chunk_len = nilfs_segchunk_seg_len(sbi, segchunknum);
	unsigned long blklen;

	if (segnum == chunk_start + chunk_len - 1) {	/* last segment in the chunk */
		blklen = sbi->s_blocks_per_segment - nilfs_susage_blk_len(sbi, segchunknum);
	} else {
		unsigned long n_usage_b = (blocksize - usageh_size) / usage_size;
		long onchunk = (sbi->s_nsegment + n_usage_b - 1) / n_usage_b;
		long o = onchunk < NILFS_MAX_SU_NCHUNK ? onchunk : NILFS_MAX_SU_NCHUNK;
		long sseg = segchunknum - 1;

		if (sseg < 0)
			sseg = o - 1;
		if (segnum == chunk_start + chunk_len / 2 - 1) {
			/* secondary usage table */
			blklen = sbi->s_blocks_per_segment - nilfs_susage_blk_len(sbi, sseg);
		} else {
			blklen = sbi->s_blocks_per_segment;
		}
	}
	if (segnum == 0)	/* super block */
		blklen--;
	else if (segnum == 1)
		blklen--;	/* super block replica */
	return blklen;
}

/* listcp, listpseg command */
/* 
 * start: full segment number
 * len: full segment lenght for search 
 * nprint: print checkpoint up to nprint: 0 for all
 * info: print statistics
 * pseg: print partial segment
 */

void
list_check_point(long int start, long int len, int nprint, int info, int pseg) /* listcp command */
{
	struct nilfs_sb_info *sbi = &sb_info;
	off_t blk = start * blksperseg;	/* first segment */
	off_t pblk = 0;
	struct nilfs_seg_info *ssi = &seg_info;
	unsigned int nblocks, flags;
	struct nilfs_seg_summary ss;
	struct timeval ct;
	int nf, nfb, ni, nib;
	int ncp, nss;
	int minlogs, maxlogs;
	char cbuf[26], *cbufp;
	long segnum, psegnum;
	dbn_t segstart = 0;
	unsigned int seglen = 0;

	ct.tv_usec = 0;
	ct.tv_sec = 0;
	nf = nfb = ni = nib = 0;
	nss = ncp = 0;
	psegnum = blk / blksperseg - 1;
	maxlogs = 0;
	minlogs = INT32_MAX;
	for (;;) {
		segnum = blk / blksperseg;
		if (psegnum < segnum) {
			if (--len < 0)
				break;
			psegnum = segnum;
			segstart = nilfs_seg_start_blk(sbi, segnum);
			seglen = nilfs_seg_blklen(sbi, segnum);
			if (blk < segstart)
				blk = segstart;
		}

		if (readstruct(&ss, sizeof(ss), blk * blocksize) <= 0) break;
		if (__le16_to_cpu(ss.ss_magic) != NILFS_SEGSUM_MAGIC) break;
		if (!set_seg_info_crc(blk, ssi, &ss)) break;

		nf += ssi->nblk_file;
		nfb += ssi->nblk_fbt;
		ni +=  ssi->nblk_inode;
		nib +=  ssi->nblk_ibt;

		nss++;
		nblocks = __le16_to_cpu(ss.ss_nblocks);
		flags = __le16_to_cpu(ss.ss_flags);
		if (pseg || flags & (SS_FJCP|SS_FNCP)) {
			dbn_t iroot, cp_start, cp_end;
			struct nilfs_cp_info cpi;
			unsigned long x;
			int cp_ret;

			cpi.sketch = NULL;
			cpi.sketch_size = 0;
			if (flags & (SS_FJCP|SS_FNCP)) {
				cp_start = blk + nilfs_seg_blocks(ssi, 0);
				cp_end = blk + nblocks - 1;
				cp_ret = load_checkpoint(&iroot, cp_start, cp_end, &cpi);
				switch(cp_ret) {
				case -2:
					fprintf(stderr, "checkpoint checksum mismatch!\n");
				case -1:
					return;
				}
				if (cpi.nblk_lseg < minlogs &&
				    NILFS_PSEG_MIN_BLOCKS <= cpi.nblk_lseg)
					minlogs = cpi.nblk_lseg;
				if (maxlogs < cpi.nblk_lseg)
					maxlogs = cpi.nblk_lseg;
			}
			x = (pseg ? nblocks : cpi.nblk_lseg);
			ncp++;
			printf("%10ju %5lu ", blk, x);

			ct.tv_sec = __le64_to_cpu(ss.ss_create);
			if (date_in_epoch) {
				printf("%ld ", ct.tv_sec);
			} else {
				ctime_r(&ct.tv_sec, cbuf);
				if ((cbufp = rindex(cbuf, '\n')) != NULL)
					*cbufp = '\0';
				printf("%s ", cbuf);
			}

			print_ss_flags(flags);
			printf("%5u", cpi.sketch_size);
			if (pseg) {
				printf("%5u %5u %5u %5u",
				       ssi->nblk_file, ssi->nblk_fbt,
				       ssi->nblk_inode, ssi->nblk_ibt);
			}
			printf("\n");

			pblk = blk;
			if (0 < nprint && --nprint == 0)
				break;
		}

		blk += nblocks;
		if (segstart + seglen - blk < NILFS_PSEG_MIN_BLOCKS) {
			blk = (segnum + 1) * blksperseg;
		}
	}
	if (info) {
		printf("file blk/btree %d %d inode blk/btree %d %d\n",
		       nf, nfb, ni, nib);
		printf("segment summary %d checkpoint %d\n",
		       nss, ncp);
		printf("logical segment max len %d min len %d [block]\n",
		       maxlogs, minlogs);
	}
}

void
list_a_segusage_hdr(int nsu, int secondary, int nc)
{
	struct nilfs_sb_info *sbi = &sb_info;
	dbn_t blk;
	int bl, i;
	unsigned int seglen;
	struct segusage_hdr sh;
	__le32 *buffer = (__le32 *)malloc(blocksize);
	__u32 crc;

	if (nsu < 0 || nc <= nsu) {
		printf("segusage out of range %d\n", nsu);
		return;
	}
	if (buffer == NULL) {
		return;
	}
	blk = nilfs_susage_blk_num(sbi, secondary, nsu);
	bl = nilfs_susage_blk_len(sbi, nsu);
	seglen = nilfs_segchunk_seg_len(sbi, nsu);
	for (i = 0; i < bl; i++) {
		if (readstruct(&sh, sizeof(sh), (blk + i) * blocksize) <= 0) break;
		if (readstruct(buffer, blocksize, (blk + i) * blocksize) <= 0) break;
		crc = crc32c_le(sbi->s_crc_seed, (unsigned char *)buffer + sizeof(__le32), blocksize - sizeof(__le32));

		printf("chunk %2d blk# %8ld %3u segment free_blocks_count %llu last_segment %llu checksum ",
		       nsu, blk, seglen,
		       __le64_to_cpu(sh.suh_free_blocks_count),
		       __le64_to_cpu(sh.suh_last_segment));
		if (__le32_to_cpu(sh.suh_sum) == crc)
			printf("ok\n");
		else
			printf("NG %08x %08x\n", __le32_to_cpu(sh.suh_sum), crc);
	}
	free(buffer);
}

void
list_segusage_hdr(int nsu, int secondary)
{
	int i;
	struct nilfs_sb_info *sbi = &sb_info;
	unsigned long n_usage_b = (blocksize - usageh_size) / usage_size;
	long onchunk = (sbi->s_nsegment + n_usage_b - 1) / n_usage_b;
	int nc = onchunk < NILFS_MAX_SU_NCHUNK ? onchunk : NILFS_MAX_SU_NCHUNK;

	if (nsu == -1) /* list all segment usage headers */
		for (i = 0; i < nc; i++)
			list_a_segusage_hdr(i, secondary, nc);
	else
		list_a_segusage_hdr(nsu, secondary, nc);
}

void
get_sketch_data()
{
	dbn_t blk = sketch_blk;
	struct nilfs_seg_summary ss;
	struct nilfs_seg_info *ssi = &seg_info;
	struct nilfs_cp_info cpi;
	dbn_t iroot, cp_start, cp_end;
	int cp_ret;

	if (readstruct(&ss, sizeof(ss), blk * blocksize) <= 0) return;

	if (!set_seg_info_crc(blk, ssi, &ss)) {
		fprintf(stderr, "segment summary checksum not match\n");
		return;
	}
	if (!(ssi->flags & (SS_FJCP|SS_FNCP))) {
		fprintf(stderr, "block not major check point\n");
		return;
	}
	cpi.sketch = (unsigned char *)malloc(blocksize);
	if (cpi.sketch == NULL) {
		fprintf(stderr, "cannot allocate memory\n");
		return;
	}
	cp_start = blk + nilfs_seg_blocks(ssi, 0);
	cp_end = blk + ssi->nblocks - 1;
	cpi.sketch_size = 0;
	cp_ret = load_checkpoint(&iroot, cp_start, cp_end, &cpi);
	if (cp_ret == -2) {
		fprintf(stderr, "checkpoint checksum not match\n");
		goto err;
	}
	if (cp_ret < 0)
		goto err;
	if (cpi.sketch_size == 0) {
		fprintf(stderr, "no sketch\n");
		goto err;
	}
	if (blocksize < cpi.sketch_size) {
		fprintf(stderr, "too big sketch\n");
		goto err;
	}
	if (sketch_out) {
		FILE *fp = fopen(sketch_out, "w");
		if (fp == NULL) {
			fprintf(stderr, "cannot open %s\n", sketch_out);
			goto err;
		}
		fwrite(cpi.sketch, 1, cpi.sketch_size, fp);
		fclose(fp);
	} else
		fwrite(cpi.sketch, 1, cpi.sketch_size, stdout);
 err:
	free(cpi.sketch);
}

int
inspect()
{
	struct nilfs_sb_info *sbi = &sb_info;
	int com;
	char *prompt = "nilfs> ";

	for (;;) {
		com = getcom(stdin, prompt);
		switch(com) {
		case 1: /* print block */
			dump_block(gnum(vcom[1], c_blk_n), gnum(vcom[2], 0));
			break;
		case 2: /* superblock */
			print_super();
			break;
		case 4: /* print segment summary */
			print_ss(gnum(vcom[1], c_blk_n));
			break;
		case 5: /* print inode header */
			print_inode_hdr(gnum(vcom[1], c_blk_n));
			break;
		case 6: /* print disk inode, block number, relative inode number */
			print_inode(gnum(vcom[1], c_blk_n), gnum(vcom[2], 0), gnum(vcom[3], 1));
			break;
		case 7:
			print_btree_hdr(gnum(vcom[1], c_blk_n));
			break;
		case 8:
			print_btree_key(gnum(vcom[1], c_blk_n), gnum(vcom[2], 0), gnum(vcom[3], 1));
			break;
		case 9:	/* btree lookup (root dbn, key) */
			btree_lookup(gnum(vcom[1], c_blk_n), gnum(vcom[2], 1));
			break;
		case 10: /* list check point from beginning */
		case 11: /* list partial segment from beginning */
		{
			int start, len, nprint, stat;
			start = gnum(vcom[1], 0);
			len = gnum(vcom[2], sbi->s_nsegment - start);
			nprint = gnum(vcom[3], 0);
			if (len <= 0 || nprint < 0)
				break;
			/* for compatibility */
			stat = vcom[1][0] == 'a' || vcom[2][0] == 'a'
				|| vcom[3][0] == 'a'|| vcom[4][0] == 'a';
			list_check_point(start, len, nprint, stat, com == 11);
			break;
		}
		case 12: /* print segment usage headers */
			/* segment chunk number, primary/secondary */
			list_segusage_hdr(gnum(vcom[1], -1), gnum(vcom[2], 0));
			break;
		case 3: /* set, number only */
		case 128:
			c_blk_n = gnum(vcom[1], c_blk_n);
		case 127:
			printf("block# %ld, segment# %ld\n", c_blk_n, c_blk_n / blksperseg);
			break;
		case 510: /* bye */
			return -1;
		case 511: /* help */
			printhelp();
			break;
		case -1: /* unknown */
			printf("unknown command %s\n", vcom[0]);
			break;
		case -2: /* syntax error */
			break;
		case -3: /* EOF */
			break;
		default:
			printf("something wrong %s, com=%d\n", vcom[0], com);
		}
	}
	return 1;
}

void
printhelp()
{
}

struct t_m_str {
	unsigned int number;
	char *type;
} m_str[] = {
	{NILFS_SUPER_MAGIC, "super block"},
	{NILFS_SEGSUM_MAGIC, "segment summary"},
	{0, NULL},
};

char *
magic_str(unsigned int magic)
{
	struct t_m_str *m = m_str;

	while (m->number && m->number != magic)
		m++;

	if (m->number)
		return m->type;
	return "unknown magic";
}

int
readstruct(void *b, int size, off_t pos)
{
	int sizeb;
	off_t posb, poso;
	void *buf = NULL;
	size_t alignment = sysconf(_SC_PAGESIZE);
	int r;
	off_t p;
	int  err;

	posb = pos / alignment * alignment;
	poso = pos % alignment;

	/* p = llseek(devfd, (unsigned long)(posb >> 32), (unsigned long)(posb & 0xffffffff), &rs, SEEK_SET); */
	p = lseek(devfd, posb, SEEK_SET);
	if (p < 0) {
		perror("seek");
		fprintf(stderr, "Cannot seek to 0x%jx %ju\n", posb, posb);

	}
	sizeb = ((size - 1) / alignment + 1) * alignment;
	err = posix_memalign(&buf, alignment, sizeb);
	if (err) {
		fprintf(stderr, "posix_memalign failed\n");
		return -1;
	}
	r = read(devfd, buf, sizeb);
	if (r < 0 || r != sizeb) {
		perror("readstruct");
		free(buf);
		return -1;
	}
	memcpy(b, buf+poso, size);
	free(buf);
	return r;
}

int
main(int argc, char *argv[])
{
	mygetopt(argc, argv);
	if (0 < argc) {
		char *cp = strrchr(argv[0], '/');

		progname = (cp ? cp + 1 : argv[0]);
	}
	if (devname == NULL) {
		usage();
		exit(1);
	}
	if ((devfd = open(devname, O_RDONLY|O_DIRECT)) < 0) {
		fprintf(stderr, "cannot open %s\n", devname);
		exit(1);
	}
	if (sketch_blk) {
		if (nilfs_fill_super() < 0)
			exit(1);
		get_sketch_data();
	} else {
		init_sig();
		print_super();
		sigsetjmp(main_env, 1);
		while(0 < inspect());
	}
	if (0 <= devfd)
		close(devfd);
	exit(0);
}

/* Local Variables:	*/
/* eval: (c-set-style "linux")	*/
/* End:			*/
