/*
 * radix-tree-64.c - 64bits version of radix-tree.c
 *
 * Copyright (C) 2005 Nippon Telegraph and Telephone Corporation.
 *
 * This file is part of NILFS.
 *
 * radix-tree-64.c,v 1.20 2006/01/17 04:24:26 kihara Exp
 *
 * Modified for NILFS by Seiji Kihara <kihara@osrg.net>
 *
 * Modifications:
 *  (a) change the key types from unsigned long to __u64,
 *	to handle 64bit keys on 32(<64)bits-architecure machines
 *  (b) change the extern function names from radix_tree_xxx to
 *      radix_tree_64_xxx
 *  (c) add new functions for NILFS
 *      - radix_tree_64_destroy()
 *      - radix_tree_64_force_empty() and __find_first_index()
 */
/*
 * Copyright (C) 2001 Momchil Velikov
 * Portions Copyright (C) 2001 Christoph Hellwig
 * Copyright (C) 2005 SGI, Christoph Lameter <clameter@sgi.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <linux/errno.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include "radix-tree-64.h"
#include <linux/percpu.h>
#include <linux/slab.h>
#include <linux/notifier.h>
#include <linux/cpu.h>
#include <linux/gfp.h>
#include <linux/string.h>
#include <linux/bitops.h>

/* #define RADIX_TREE_64_DEBUG */

#ifdef __KERNEL__
#define RADIX_TREE_64_MAP_SHIFT	6
#else
#define RADIX_TREE_64_MAP_SHIFT	3	/* For more stressful testing */
#endif
#define RADIX_TREE_64_TAGS		2

#define RADIX_TREE_64_MAP_SIZE	(1UL << RADIX_TREE_64_MAP_SHIFT)
#define RADIX_TREE_64_MAP_MASK	(RADIX_TREE_64_MAP_SIZE-1)

#define RADIX_TREE_64_TAG_LONGS	\
	((RADIX_TREE_64_MAP_SIZE + BITS_PER_LONG - 1) / BITS_PER_LONG)

struct radix_tree_64_node {
	unsigned int	count;
	void		*slots[RADIX_TREE_64_MAP_SIZE];
	unsigned long	tags[RADIX_TREE_64_TAGS][RADIX_TREE_64_TAG_LONGS];
};

struct radix_tree_64_path {
	struct radix_tree_64_node *node;
	int offset;
};

#define RADIX_TREE_64_INDEX_BITS  (8 /* CHAR_BIT */ * sizeof(__u64))
#define RADIX_TREE_64_MAX_PATH	\
	(RADIX_TREE_64_INDEX_BITS/RADIX_TREE_64_MAP_SHIFT + 2)

#ifndef __read_mostly
# if defined(CONFIG_X86) || defined(CONFIG_SPARC64) || defined(CONFIG_IA64)
#  define __read_mostly __attribute__((__section__(".data.read_mostly")))
# else
#  define __read_mostly
# endif
#endif
static __u64 height_to_maxindex[RADIX_TREE_64_MAX_PATH] __read_mostly;

/*
 * Radix tree node cache.
 */
static kmem_cache_t *radix_tree_64_node_cachep;

/*
 * Per-cpu pool of preloaded nodes
 */
struct radix_tree_64_preload {
	int nr;
	struct radix_tree_64_node *nodes[RADIX_TREE_64_MAX_PATH];
};
DEFINE_PER_CPU(struct radix_tree_64_preload, radix_tree_64_preloads) = { 0, };

/*
 * This assumes that the caller has performed appropriate preallocation, and
 * that the caller has pinned this thread of control to the current CPU.
 */
static struct radix_tree_64_node *
radix_tree_64_node_alloc(struct radix_tree_64_root *root)
{
	struct radix_tree_64_node *ret;

	ret = kmem_cache_alloc(radix_tree_64_node_cachep, root->gfp_mask);
#ifdef RADIX_TREE_64_DEBUG
	if (ret)
		printk("NILFS: radix-tree_64_node_alloc: node allocated.\n");
#endif
	if (ret == NULL && !(root->gfp_mask & __GFP_WAIT)) {
		struct radix_tree_64_preload *rtp;

		rtp = &__get_cpu_var(radix_tree_64_preloads);
		if (rtp->nr) {
			ret = rtp->nodes[rtp->nr - 1];
			rtp->nodes[rtp->nr - 1] = NULL;
			rtp->nr--;
		}
	}
	return ret;
}

static inline void
radix_tree_64_node_free(struct radix_tree_64_node *node)
{
#ifdef RADIX_TREE_64_DEBUG
	printk("NILFS: freeing radix_tree_64_node.\n");
#endif
	kmem_cache_free(radix_tree_64_node_cachep, node);
}

/*
 * Load up this CPU's radix_tree_64_node buffer with sufficient objects to
 * ensure that the addition of a single element in the tree cannot fail.  On
 * success, return zero, with preemption disabled.  On error, return -ENOMEM
 * with preemption not disabled.
 */
int radix_tree_64_preload(gfp_t gfp_mask)
{
#ifndef RADIX_TREE_64_PRELOAD_DISABLE
	struct radix_tree_64_preload *rtp;
	struct radix_tree_64_node *node;
	int ret = -ENOMEM;

	preempt_disable();
	rtp = &__get_cpu_var(radix_tree_64_preloads);
	while (rtp->nr < ARRAY_SIZE(rtp->nodes)) {
		preempt_enable();
		node = kmem_cache_alloc(radix_tree_64_node_cachep, gfp_mask);
		if (node == NULL)
			goto out;
#ifdef RADIX_TREE_64_DEBUG
		printk("NILFS: radix_tree_64_preload: node allocated.\n");
#endif
		preempt_disable();
		rtp = &__get_cpu_var(radix_tree_64_preloads);
		if (rtp->nr < ARRAY_SIZE(rtp->nodes))
			rtp->nodes[rtp->nr++] = node;
		else {
			kmem_cache_free(radix_tree_64_node_cachep, node);
#ifdef RADIX_TREE_64_DEBUG
			printk("NILFS: radix_tree_64_preload: node freed.\n");
#endif
		}
	}
	ret = 0;
out:
	return ret;
#else /* RADIX_TREE_64_PRELOAD_DISABLE */
	preempt_disable();
	return 0;
#endif
}

static inline void tag_set(struct radix_tree_64_node *node, int tag,
			   int offset)
{
	if (!test_bit(offset, &node->tags[tag][0]))
		__set_bit(offset, &node->tags[tag][0]);
}

static inline void tag_clear(struct radix_tree_64_node *node, int tag,
			     int offset)
{
	__clear_bit(offset, &node->tags[tag][0]);
}

static inline int tag_get(struct radix_tree_64_node *node, int tag, int offset)
{
	return test_bit(offset, &node->tags[tag][0]);
}

/*
 *	Return the maximum key which can be store into a
 *	radix tree with height HEIGHT.
 */
static inline __u64 radix_tree_64_maxindex(unsigned int height)
{
	return height_to_maxindex[height];
}

/*
 *	Extend a radix tree so it can store key @index.
 */
static int radix_tree_64_extend(struct radix_tree_64_root *root, __u64 index)
{
	struct radix_tree_64_node *node;
	unsigned int height;
	char tags[RADIX_TREE_64_TAGS];
	int tag;

	/* Figure out what the height should be.  */
	height = root->height + 1;
	while (index > radix_tree_64_maxindex(height))
		height++;

	if (root->rnode == NULL) {
		root->height = height;
		goto out;
	}

	/*
	 * Prepare the tag status of the top-level node for propagation
	 * into the newly-pushed top-level node(s)
	 */
	for (tag = 0; tag < RADIX_TREE_64_TAGS; tag++) {
		int idx;

		tags[tag] = 0;
		for (idx = 0; idx < RADIX_TREE_64_TAG_LONGS; idx++) {
			if (root->rnode->tags[tag][idx]) {
				tags[tag] = 1;
				break;
			}
		}
	}

	do {
		if (!(node = radix_tree_64_node_alloc(root)))
			return -ENOMEM;

		/* Increase the height.  */
		node->slots[0] = root->rnode;

		/* Propagate the aggregated tag info into the new root */
		for (tag = 0; tag < RADIX_TREE_64_TAGS; tag++) {
			if (tags[tag])
				tag_set(node, tag, 0);
		}

		node->count = 1;
		root->rnode = node;
		root->height++;
	} while (height > root->height);
out:
	return 0;
}

/**
 *	radix_tree_64_insert    -    insert into a radix tree
 *	@root:		radix tree root
 *	@index:		index key
 *	@item:		item to insert
 *
 *	Insert an item into the radix tree at position @index.
 */
int radix_tree_64_insert(struct radix_tree_64_root *root,
			__u64 index, void *item)
{
	struct radix_tree_64_node *node = NULL, *slot;
	unsigned int height, shift;
	int offset;
	int error;

#ifdef RADIX_TREE_64_DEBUG
	printk("NILFS: radix_tree_64_insert: called with index %Lu.\n",
	       index);
#endif
	/* Make sure the tree is high enough.  */
	if ((!index && !root->rnode) ||
			index > radix_tree_64_maxindex(root->height)) {
		error = radix_tree_64_extend(root, index);
		if (error)
			return error;
	}

	slot = root->rnode;
	height = root->height;
	shift = (height-1) * RADIX_TREE_64_MAP_SHIFT;

	offset = 0;			/* uninitialised var warning */
	while (height > 0) {
		if (slot == NULL) {
			/* Have to add a child node.  */
			if (!(slot = radix_tree_64_node_alloc(root)))
				return -ENOMEM;
			if (node) {
				node->slots[offset] = slot;
				node->count++;
			} else
				root->rnode = slot;
		}

		/* Go a level down */
		offset = (index >> shift) & RADIX_TREE_64_MAP_MASK;
		node = slot;
		slot = node->slots[offset];
		shift -= RADIX_TREE_64_MAP_SHIFT;
		height--;
	}

	if (slot != NULL)
		return -EEXIST;

	if (node) {
		node->count++;
		node->slots[offset] = item;
		BUG_ON(tag_get(node, 0, offset));
		BUG_ON(tag_get(node, 1, offset));
	} else
		root->rnode = item;

	return 0;
}
EXPORT_SYMBOL(radix_tree_64_insert);

static inline void **__lookup_slot(struct radix_tree_64_root *root,
				   __u64 index)
{
	unsigned int height, shift;
	struct radix_tree_64_node **slot;

	height = root->height;
	if (index > radix_tree_64_maxindex(height))
		return NULL;

	shift = (height-1) * RADIX_TREE_64_MAP_SHIFT;
	slot = &root->rnode;

	while (height > 0) {
		if (*slot == NULL)
			return NULL;

		slot = (struct radix_tree_64_node **)
			((*slot)->slots +
				((index >> shift) & RADIX_TREE_64_MAP_MASK));
		shift -= RADIX_TREE_64_MAP_SHIFT;
		height--;
	}

	return (void **)slot;
}

/**
 *	radix_tree_64_lookup_slot    -    lookup a slot in a radix tree
 *	@root:		radix tree root
 *	@index:		index key
 *
 *	Lookup the slot corresponding to the position @index in the radix tree
 *	@root. This is useful for update-if-exists operations.
 */
void **radix_tree_64_lookup_slot(struct radix_tree_64_root *root,
				 __u64 index)
{
	return __lookup_slot(root, index);
}
EXPORT_SYMBOL(radix_tree_64_lookup_slot);

/**
 *	radix_tree_64_lookup    -    perform lookup operation on a radix tree
 *	@root:		radix tree root
 *	@index:		index key
 *
 *	Lookup the item at the position @index in the radix tree @root.
 */
void *radix_tree_64_lookup(struct radix_tree_64_root *root,
			   __u64 index)
{
	void **slot;

	slot = __lookup_slot(root, index);
	return slot != NULL ? *slot : NULL;
}
EXPORT_SYMBOL(radix_tree_64_lookup);

/**
 *	radix_tree_64_tag_set - set a tag on a radix tree node
 *	@root:		radix tree root
 *	@index:		index key
 *	@tag: 		tag index
 *
 *	Set the search tag corresponging to @index in the radix tree.  From
 *	the root all the way down to the leaf node.
 *
 *	Returns the address of the tagged item.   Setting a tag on a
 *	not-present item is a bug.
 */
void *radix_tree_64_tag_set(struct radix_tree_64_root *root,
			__u64 index, int tag)
{
	unsigned int height, shift;
	struct radix_tree_64_node *slot;

#ifdef RADIX_TREE_64_DEBUG
	printk("NILFS: radix_tree_64_tag_set: "
	       "called with index %Lu, tag %d.\n",
	       index, tag);
#endif
	height = root->height;
	if (index > radix_tree_64_maxindex(height))
		return NULL;

	shift = (height - 1) * RADIX_TREE_64_MAP_SHIFT;
	slot = root->rnode;

	while (height > 0) {
		int offset;

		offset = (index >> shift) & RADIX_TREE_64_MAP_MASK;
		tag_set(slot, tag, offset);
		slot = slot->slots[offset];
		BUG_ON(slot == NULL);
		shift -= RADIX_TREE_64_MAP_SHIFT;
		height--;
	}

	return slot;
}
EXPORT_SYMBOL(radix_tree_64_tag_set);

/**
 *	radix_tree_64_tag_clear - clear a tag on a radix tree node
 *	@root:		radix tree root
 *	@index:		index key
 *	@tag: 		tag index
 *
 *	Clear the search tag corresponging to @index in the radix tree.  If
 *	this causes the leaf node to have no tags set then clear the tag in the
 *	next-to-leaf node, etc.
 *
 *	Returns the address of the tagged item on success, else NULL.  ie:
 *	has the same return value and semantics as radix_tree_64_lookup().
 */
void *radix_tree_64_tag_clear(struct radix_tree_64_root *root,
			__u64 index, int tag)
{
	struct radix_tree_64_path path[RADIX_TREE_64_MAX_PATH], *pathp = path;
	struct radix_tree_64_node *slot;
	unsigned int height, shift;
	void *ret = NULL;

	height = root->height;
	if (index > radix_tree_64_maxindex(height))
		goto out;

	shift = (height - 1) * RADIX_TREE_64_MAP_SHIFT;
	pathp->node = NULL;
	slot = root->rnode;

	while (height > 0) {
		int offset;

		if (slot == NULL)
			goto out;

		offset = (index >> shift) & RADIX_TREE_64_MAP_MASK;
		pathp[1].offset = offset;
		pathp[1].node = slot;
		slot = slot->slots[offset];
		pathp++;
		shift -= RADIX_TREE_64_MAP_SHIFT;
		height--;
	}

	ret = slot;
	if (ret == NULL)
		goto out;

	do {
		int idx;

		tag_clear(pathp->node, tag, pathp->offset);
		for (idx = 0; idx < RADIX_TREE_64_TAG_LONGS; idx++) {
			if (pathp->node->tags[tag][idx])
				goto out;
		}
		pathp--;
	} while (pathp->node);
out:
	return ret;
}
EXPORT_SYMBOL(radix_tree_64_tag_clear);

#ifndef __KERNEL__	/* Only the test harness uses this at present */
/**
 *	radix_tree_64_tag_get - get a tag on a radix tree node
 *	@root:		radix tree root
 *	@index:		index key
 *	@tag: 		tag index
 *
 *	Return the search tag corresponging to @index in the radix tree.
 *
 *	Returns zero if the tag is unset, or if there is no corresponding item
 *	in the tree.
 */
int radix_tree_64_tag_get(struct radix_tree_64_root *root,
			__u64 index, int tag)
{
	unsigned int height, shift;
	struct radix_tree_64_node *slot;
	int saw_unset_tag = 0;

	height = root->height;
	if (index > radix_tree_64_maxindex(height))
		return 0;

	shift = (height - 1) * RADIX_TREE_64_MAP_SHIFT;
	slot = root->rnode;

	for ( ; ; ) {
		int offset;

		if (slot == NULL)
			return 0;

		offset = (index >> shift) & RADIX_TREE_64_MAP_MASK;

		/*
		 * This is just a debug check.  Later, we can bale as soon as
		 * we see an unset tag.
		 */
		if (!tag_get(slot, tag, offset))
			saw_unset_tag = 1;
		if (height == 1) {
			int ret = tag_get(slot, tag, offset);

			BUG_ON(ret && saw_unset_tag);
			return ret ? 1 : -1;
		}
		slot = slot->slots[offset];
		shift -= RADIX_TREE_64_MAP_SHIFT;
		height--;
	}
}
EXPORT_SYMBOL(radix_tree_64_tag_get);
#endif

static unsigned int
__lookup(struct radix_tree_64_root *root, void **results, __u64 index,
	unsigned int max_items, __u64 *next_index)
{
	unsigned int nr_found = 0;
	unsigned int shift;
	unsigned int height = root->height;
	struct radix_tree_64_node *slot;
	unsigned long i;

	height = root->height;
	if (height == 0)
		goto out;

	shift = (height-1) * RADIX_TREE_64_MAP_SHIFT;
	slot = root->rnode;

	for ( ; height > 1; height--) {

		for (i = (index >> shift) & RADIX_TREE_64_MAP_MASK ;
				i < RADIX_TREE_64_MAP_SIZE; i++) {
			if (slot->slots[i] != NULL)
				break;
			index &= ~((((__u64) 1) << shift) - 1);
			index += ((__u64) 1) << shift;
			if (index == 0)
				goto out;	/* 64-bit wraparound */
		}
		if (i == RADIX_TREE_64_MAP_SIZE)
			goto out;
		shift -= RADIX_TREE_64_MAP_SHIFT;
		slot = slot->slots[i];
	}

	/* Bottom level: grab some items */
	for (i = index & RADIX_TREE_64_MAP_MASK;
		i < RADIX_TREE_64_MAP_SIZE; i++) {
		index++;
		if (slot->slots[i]) {
			results[nr_found++] = slot->slots[i];
			if (nr_found == max_items)
				goto out;
		}
	}
out:
	*next_index = index;
	return nr_found;
}

/**
 *	radix_tree_64_gang_lookup - perform multiple lookup on a radix tree
 *	@root:		radix tree root
 *	@results:	where the results of the lookup are placed
 *	@first_index:	start the lookup from this key
 *	@max_items:	place up to this many items at *results
 *
 *	Performs an index-ascending scan of the tree for present items.  Places
 *	them at *@results and returns the number of items which were placed at
 *	*@results.
 *
 *	The implementation is naive.
 */
unsigned int
radix_tree_64_gang_lookup(struct radix_tree_64_root *root, void **results,
			__u64 first_index, unsigned int max_items)
{
	const __u64 max_index = radix_tree_64_maxindex(root->height);
	__u64 cur_index = first_index;
	unsigned int ret = 0;

	while (ret < max_items) {
		unsigned int nr_found;
		__u64 next_index;	/* Index of next search */

		if (cur_index > max_index)
			break;
		nr_found = __lookup(root, results + ret, cur_index,
					max_items - ret, &next_index);
		ret += nr_found;
		if (next_index == 0)
			break;
		cur_index = next_index;
	}
	return ret;
}
EXPORT_SYMBOL(radix_tree_64_gang_lookup);

/*
 * FIXME: the two tag_get()s here should use find_next_bit() instead of
 * open-coding the search.
 */
static unsigned int
__lookup_tag(struct radix_tree_64_root *root, void **results, __u64 index,
	unsigned int max_items, __u64 *next_index, int tag)
{
	unsigned int nr_found = 0;
	unsigned int shift;
	unsigned int height = root->height;
	struct radix_tree_64_node *slot;

	shift = (height - 1) * RADIX_TREE_64_MAP_SHIFT;
	slot = root->rnode;

	while (height > 0) {
		unsigned long i = (index >> shift) & RADIX_TREE_64_MAP_MASK;

		for ( ; i < RADIX_TREE_64_MAP_SIZE; i++) {
			if (tag_get(slot, tag, i)) {
				BUG_ON(slot->slots[i] == NULL);
				break;
			}
			index &= ~((((__u64) 1) << shift) - 1);
			index += ((__u64) 1) << shift;
			if (index == 0)
				goto out;	/* 64-bit wraparound */
		}
		if (i == RADIX_TREE_64_MAP_SIZE)
			goto out;
		height--;
		if (height == 0) {	/* Bottom level: grab some items */
			unsigned long j = index & RADIX_TREE_64_MAP_MASK;

			for ( ; j < RADIX_TREE_64_MAP_SIZE; j++) {
				index++;
				if (tag_get(slot, tag, j)) {
					BUG_ON(slot->slots[j] == NULL);
					results[nr_found++] = slot->slots[j];
					if (nr_found == max_items)
						goto out;
				}
			}
		}
		shift -= RADIX_TREE_64_MAP_SHIFT;
		slot = slot->slots[i];
	}
out:
	*next_index = index;

	return nr_found;
}

/**
 *	radix_tree_64_gang_lookup_tag - perform multiple lookup on a radix tree
 *	                             based on a tag
 *	@root:		radix tree root
 *	@results:	where the results of the lookup are placed
 *	@first_index:	start the lookup from this key
 *	@max_items:	place up to this many items at *results
 *	@tag:		the tag index
 *
 *	Performs an index-ascending scan of the tree for present items which
 *	have the tag indexed by @tag set.  Places the items at *@results and
 *	returns the number of items which were placed at *@results.
 */
unsigned int
radix_tree_64_gang_lookup_tag(struct radix_tree_64_root *root, void **results,
		__u64 first_index, unsigned int max_items, int tag)
{
	const __u64 max_index = radix_tree_64_maxindex(root->height);
	__u64 cur_index = first_index;
	unsigned int ret = 0;

	while (ret < max_items) {
		unsigned int nr_found;
		__u64 next_index;	/* Index of next search */

		if (cur_index > max_index)
			break;
		nr_found = __lookup_tag(root, results + ret, cur_index,
					max_items - ret, &next_index, tag);
		ret += nr_found;
		if (next_index == 0)
			break;
		cur_index = next_index;
	}
	return ret;
}
EXPORT_SYMBOL(radix_tree_64_gang_lookup_tag);

/**
 *	radix_tree_64_delete    -    delete an item from a radix tree
 *	@root:		radix tree root
 *	@index:		index key
 *
 *	Remove the item at @index from the radix tree rooted at @root.
 *
 *	Returns the address of the deleted item, or NULL if it was not present.
 */
void *radix_tree_64_delete(struct radix_tree_64_root *root, __u64 index)
{
	struct radix_tree_64_path path[RADIX_TREE_64_MAX_PATH], *pathp = path;
	struct radix_tree_64_path *orig_pathp;
	struct radix_tree_64_node *slot;
	unsigned int height, shift;
	void *ret = NULL;
	char tags[RADIX_TREE_64_TAGS];
	int nr_cleared_tags;

#ifdef RADIX_TREE_64_DEBUG
	printk("NILFS: radix_tree_64_delete: called with index %Lu.\n",
	       index);
#endif
	height = root->height;
	if (index > radix_tree_64_maxindex(height))
		goto out;

	shift = (height - 1) * RADIX_TREE_64_MAP_SHIFT;
	pathp->node = NULL;
	slot = root->rnode;

	for ( ; height > 0; height--) {
		int offset;

		if (slot == NULL)
			goto out;

		offset = (index >> shift) & RADIX_TREE_64_MAP_MASK;
		pathp[1].offset = offset;
		pathp[1].node = slot;
		slot = slot->slots[offset];
		pathp++;
		shift -= RADIX_TREE_64_MAP_SHIFT;
	}

	ret = slot;
	if (ret == NULL)
		goto out;

	orig_pathp = pathp;

	/*
	 * Clear all tags associated with the just-deleted item
	 */
	memset(tags, 0, sizeof(tags));
	do {
		int tag;

		nr_cleared_tags = RADIX_TREE_64_TAGS;
		for (tag = 0; tag < RADIX_TREE_64_TAGS; tag++) {
			int idx;

			if (!tags[tag])
				tag_clear(pathp->node, tag, pathp->offset);

			for (idx = 0; idx < RADIX_TREE_64_TAG_LONGS; idx++) {
				if (pathp->node->tags[tag][idx]) {
					tags[tag] = 1;
					nr_cleared_tags--;
					break;
				}
			}
		}
		pathp--;
	} while (pathp->node && nr_cleared_tags);

	/* Now free the nodes we do not need anymore */
	for (pathp = orig_pathp; pathp->node; pathp--) {
		pathp->node->slots[pathp->offset] = NULL;
		if (--pathp->node->count)
			goto out;

		/* Node with zero slots in use so free it */
#ifdef RADIX_TREE_64_DEBUG
		printk("NILFS: radix_tree_64_delete: call node_free.\n");
#endif
		radix_tree_64_node_free(pathp->node);
	}
	root->rnode = NULL;
	root->height = 0;
out:
	return ret;
}
EXPORT_SYMBOL(radix_tree_64_delete);

/**
 *	radix_tree_64_tagged - test whether any items in the tree are tagged
 *	@root:		radix tree root
 *	@tag:		tag to test
 */
int radix_tree_64_tagged(struct radix_tree_64_root *root, int tag)
{
	int idx;

	if (!root->rnode)
		return 0;
	for (idx = 0; idx < RADIX_TREE_64_TAG_LONGS; idx++) {
		if (root->rnode->tags[tag][idx])
			return 1;
	}
	return 0;
}
EXPORT_SYMBOL(radix_tree_64_tagged);

static void
radix_tree_64_node_ctor(void *node, kmem_cache_t *cachep, unsigned long flags)
{
	memset(node, 0, sizeof(struct radix_tree_64_node));
}

static __init __u64 __maxindex(unsigned int height)
{
	unsigned int tmp = height * RADIX_TREE_64_MAP_SHIFT;
	__u64 index = (~((__u64) 0) >> (RADIX_TREE_64_INDEX_BITS - tmp - 1))
				    >> 1;

	if (tmp >= RADIX_TREE_64_INDEX_BITS)
		index = ~((__u64) 0);
	return index;
}

static __init void radix_tree_64_init_maxindex(void)
{
	unsigned int i;

	for (i = 0; i < ARRAY_SIZE(height_to_maxindex); i++)
		height_to_maxindex[i] = __maxindex(i);
}

#ifdef CONFIG_HOTPLUG_CPU
static int radix_tree_64_callback(struct notifier_block *nfb,
                            unsigned long action,
                            void *hcpu)
{
       int cpu = (long)hcpu;
       struct radix_tree_64_preload *rtp;

       /* Free per-cpu pool of perloaded nodes */
       if (action == CPU_DEAD) {
               rtp = &per_cpu(radix_tree_64_preloads, cpu);
               while (rtp->nr) {
                       kmem_cache_free(radix_tree_64_node_cachep,
                                       rtp->nodes[rtp->nr-1]);
#ifdef RADIX_TREE_64_DEBUG
		       printk("NILFS: radix_tree_64_callback: node freed.\n");
#endif
                       rtp->nodes[rtp->nr-1] = NULL;
                       rtp->nr--;
               }
       }
       return NOTIFY_OK;
}

static struct notifier_block radix_tree_64_callback_nb =
{
	.notifier_call = radix_tree_64_callback,
	.priority = 0
};
#endif /* CONFIG_HOTPLUG_CPU */

void __init radix_tree_64_init(void)
{
	radix_tree_64_node_cachep = kmem_cache_create("radix_tree_64_node",
			sizeof(struct radix_tree_64_node), 0,
			SLAB_PANIC, radix_tree_64_node_ctor, NULL);
	radix_tree_64_init_maxindex();
#ifdef CONFIG_HOTPLUG_CPU
	register_cpu_notifier(&radix_tree_64_callback_nb);
#endif
}

/**
 *	__find_first_index: find first index of the present items
 *	@root:		radix tree root
 *	@findex:	the index which the search begin from,
 *			and where the results of the lookup are placed
 *			(in/out)
 *
 *	Performs an index-ascending scan of the tree for a present item
 *	from the @findex.
 *	Places the index of item at *@findex and return true if found.
 *
 *	This is based on __lookup(), and was added for NILFS.
 */
static int
__find_first_index(struct radix_tree_64_root *root, __u64 *findex)
{
	unsigned int shift;
	unsigned int height = root->height;
	struct radix_tree_64_node *slot;
	__u64 index = *findex;

	shift = (height-1) * RADIX_TREE_64_MAP_SHIFT;
	slot = root->rnode;

	while (height > 0) {
		unsigned long i = (index >> shift) & RADIX_TREE_64_MAP_MASK;

		for ( ; i < RADIX_TREE_64_MAP_SIZE; i++) {
			if (slot->slots[i] != NULL) {
				/* found */
				break;
			}
			index &= ~((((__u64) 1) << shift) - 1);
			index += ((__u64) 1) << shift;
			if (index == 0)
				goto not_found;	/* 64-bit wraparound */
		}
		if (i == RADIX_TREE_64_MAP_SIZE)
			goto not_found;
		height--;
		if (height == 0) {	/* Bottom level: grab some items */
			unsigned long j = index & RADIX_TREE_64_MAP_MASK;

			for ( ; j < RADIX_TREE_64_MAP_SIZE; j++, index++) {
				if (slot->slots[j]) {
#ifdef RADIX_TREE_64_DEBUG
					printk("radix_tree_64: "
					       "__find_first_index:");
					printk("slot(%p) index(%Lu)\n",
					       slot->slots[j], index);
#endif
					*findex = index;
					return 1;	/* found */
				}
			}
		}
		shift -= RADIX_TREE_64_MAP_SHIFT;
		slot = slot->slots[i];
	}
not_found:
	return 0;
}

/**
 *	radix_tree_64_force_empty: remove all the items in the tree
 *	@root:		radix tree root
 *
 *	The implementation is naive.
 *
 *	This functions was added for NILFS.
 */
void radix_tree_64_force_empty(struct radix_tree_64_root *root)
{
  	int ret;
	__u64 index = 0;

#ifdef RADIX_TREE_64_DEBUG
	printk("radix_tree_64_force_empty: called for tree %p.\n", root);
#endif
 loop:
	ret = __find_first_index(root, &index);
	if (ret) {
		printk("radix_tree_64_force_empty: root %p, index %Lu.\n",
		       root, index);
		radix_tree_64_delete(root, index);
		index++;
		goto loop;
	}
	if (index) {
		index++;
		goto loop;
	}
#ifdef RADIX_TREE_64_DEBUG
	printk("radix_tree_64_force_empty: return.\n");
#endif
}

/**
 *	radix_tree_64_destroy: destroy pre-allocated data
 *
 *	This function should be called when unloading if this file is
 *	incorporated in any loadable kernel modules.
 *
 *	This functions was added for NILFS.
 */
void radix_tree_64_destroy(void)
{
#ifndef RADIX_TREE_64_PRELOAD_DISABLE
	struct radix_tree_64_preload *rtp;
	struct radix_tree_64_node *node;
	__u32 cpu;

#ifdef CONFIG_HOTPLUG_CPU
	unregister_cpu_notifier(&radix_tree_64_callback_nb);
#endif

#ifdef RADIX_TREE_64_DEBUG
	printk("radix_tree_64_destroy: make preempt disable\n");
#endif
	/* free pre-loaded nodes */
	preempt_disable();
	for (cpu = 0; cpu < NR_CPUS; cpu++) {
		if (!cpu_possible(cpu))
			continue;
#ifdef RADIX_TREE_64_DEBUG
		printk("radix_tree_64_destroy: cpu %d\n", cpu);
#endif
		rtp = &per_cpu(radix_tree_64_preloads, cpu);
		while (rtp->nr) {
			node = rtp->nodes[rtp->nr - 1];
			if (node) {
#ifdef RADIX_TREE_64_DEBUG
				printk("radix_tree_64_destroy: "
				       "freeing node %p\n", node);
#endif
				kmem_cache_free(radix_tree_64_node_cachep,
						node);
			}
			rtp->nodes[rtp->nr - 1] = NULL;
			rtp->nr--;
		}
	}
#ifdef RADIX_TREE_64_DEBUG
	printk("radix_tree_64_destroy: make preempt enable\n");
#endif
	preempt_enable();
#ifdef RADIX_TREE_64_DEBUG
	printk("radix_tree_64_destroy: reach end of func\n");
#endif
#endif

	/* destroy kmem cache */
	if (kmem_cache_destroy(radix_tree_64_node_cachep)) {
		printk(KERN_INFO
		       "NILFS: radix_tree_64: "
		       "some radix tree node were not freed.\n");
	}
}
/* Local Variables:	*/
/* eval: (c-set-style "linux")	*/
/* End:			*/
