/*
 * inode.c - NILFS inode operations.
 *
 * Copyright (C) 2005 Nippon Telegraph and Telephone Corporation.
 *
 * This file is part of NILFS.
 *
 * NILFS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * NILFS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NILFS; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * inode.c,v 1.155 2006/07/18 08:19:20 ryusuke Exp
 *
 * Written by Ryusuke Konishi <ryusuke@osrg.net>
 *
 */

#include <linux/buffer_head.h>
#include <linux/mpage.h>
#include <linux/writeback.h>
#include "nilfs.h"

#ifdef NILFS_SHRINKER_DISABLE
#define nilfs_invalidatepage   NULL
#endif

int nilfs_handle_fbt_error(int err, const char *fname, struct inode *inode)
{
	err = nilfs_convert_btree_error(err);
	if (err == -EINVAL)
		nilfs_error(inode->i_sb, fname,
			    "broken file btree (inode=%lu)\n", inode->i_ino);
	return err;
}

/*
 * nilfs_get_block() - get a file block on the filesystem (callback function)
 * 
 * @inode - inode struct of the target file
 * @iblock - file block number
 * @bh_result - buffer head to be mapped on
 * @create - indicate whether allocating the block or not when it has not
 *      been allocated yet.
 *
 * This function does not issue actual read request of the specified data
 * block. It is done by VFS.
 * Bulk read for direct-io is not supported yet. (should be supported)
 */
int nilfs_get_block(struct inode *inode, sector_t iblock, 
		    struct buffer_head *bh_result, int create)
{
	struct nilfs_inode_info *nilfs_inode = NILFS_I(inode);
	dbn_t dbn = 0;
	int err = 0, ret;

	ret = nilfs_btree_lookup(&nilfs_inode->i_block_root,
				 (fbn_t)iblock, &dbn);
	if (ret == 0) {	/* found */
		map_bh(bh_result, inode->i_sb, dbn);
		goto out;
	}
	if (unlikely(ret == 1)) {
		printk(KERN_ERR "nilfs_get_block: btree_lookup returns buffer_head pointer\niblock=%llu, dbn=%lu\n",
			   iblock, dbn);
		BUG();
	}
	/* data block was not found */
	if (ret == -NILFS_BTREE_ENOKEY && create) {
		bh_result->b_blocknr = 0;
		ret = nilfs_btree_insert(&nilfs_inode->i_block_root,
					 (fbn_t)iblock, bh_result);
		/* How can we recover dirtied btree, if inserted block is 
		   abandoned without being dirtied ?? */
		if (unlikely(ret != 0)) {
			if (ret == -NILFS_BTREE_EEXIST) {
				/*
				 * The get_block() function could be called from multiple
				 * callers for an inode.  However, the page having this block
				 * must be locked in this case.
				 */
				printk(KERN_ERR
				       "nilfs_get_block: a race condition while inserting a data block. "
				       "(inode number=%lu, file block offset=%lu)\n",
				       inode->i_ino, (fbn_t)iblock);
				BUG();
			}
			err = nilfs_handle_fbt_error(ret, __FUNCTION__, inode);
			goto out;
		}
		/* Error handling should be detailed */
		set_buffer_new(bh_result);
		map_bh(bh_result, inode->i_sb, 0); /* dbn must be changed
						      to proper value */
	} else if (ret == -NILFS_BTREE_ENOKEY) {
                /* not found is not error (e.g. hole); must return without 
                   the mapped state flag. */
		;
	} else {
		err = nilfs_convert_btree_error(ret);
	}

 out:
	return err;
}

/*
 * nilfs_readpage() - implement readpage() method of v0_aops {}
 * address_space_operations.
 * @file - file struct of the file to be read
 * @page - the page to be read
 */
static int nilfs_readpage(struct file *file, struct page *page)
{
	return mpage_readpage(page, nilfs_get_block);
}

/*
 * nilfs_readpages() - implement readpages() method of v0_aops {}
 * address_space_operations.
 * @file - file struct of the file to be read
 * @mapping - address_space struct used for reading multiple pages
 * @pages - the pages to be read
 * @nr_pages - number of pages to be read
 */
static int nilfs_readpages(struct file *file, struct address_space *mapping,
			   struct list_head *pages, unsigned nr_pages)
{
	return mpage_readpages(mapping, pages, nr_pages, nilfs_get_block);
}

static int nilfs_writepages(struct address_space *mapping,
			    struct writeback_control *wbc)
{
	page_debug(2, "called but ignored (mapping=%p)\n", mapping);
	return 0;
}

static int nilfs_writepage(struct page *page, struct writeback_control *wbc)
{
	int ret;
	struct super_block *sb;

	page_debug(2, "called (page=%p, wbc nonblocking %d, wbc for_reclaim %d)\n",
		    page, wbc->nonblocking, wbc->for_reclaim);
	BUG_ON(!page->mapping || !page->mapping->host);
	sb = page->mapping->host->i_sb;
	unlock_page(page);
	if (wbc->sync_mode == WB_SYNC_ALL) {
		ret = nilfs_construct_segment(sb);
		if (NILFS_SEG_ERR(ret))
			return ret;
	} else if (wbc->for_reclaim)
		nilfs_flush_segment(NILFS_SB(sb), NILFS_SC_FLUSH_DATA);

	return 0;
}

#if NEED_SYNC_PAGE_RETVAL
static int
#else
static void
#endif
nilfs_sync_page(struct page *page)
{
	page_debug(2, "called (page=%p)\n", page);
#if NEED_SYNC_PAGE_RETVAL
	return 0;
#endif
}

static int nilfs_set_page_dirty(struct page *page)
{
	struct inode *inode = page->mapping->host;
	struct nilfs_sb_info *sbi = NILFS_SB(inode->i_sb);
	struct nilfs_sc_info *sci = NILFS_SC(sbi);
	struct buffer_head *bh, *head;
	sector_t iblock;
	unsigned nr_dirty = 0;
	int err;

	page_debug(3, "called (page=%p)\n", page);
	if (!sci)
		return -EROFS;

	err = nilfs_prepare_file_dirty(inode);
	if (unlikely(err))
		return err;

	lock_page(page);
	if (!page_has_buffers(page)) {
		page_debug(3, "page has no buffer heads. allocating.. (page=%p)\n", page);
		create_empty_buffers(page, 1 << inode->i_blkbits, 0);
	}
	iblock = (sector_t)page->index << (PAGE_CACHE_SHIFT - inode->i_blkbits);
	bh = head = page_buffers(page);
	do {
		if (!buffer_dirty(bh))
			nr_dirty++;
		if (!buffer_mapped(bh)) {
			err = nilfs_get_block(inode, iblock, bh, 1);
			if (unlikely(err)) {
				page_debug(3, "nilfs_get_block() failed (err=%d)\n", err);
				break;
			}
		}
		if (PageUptodate(page) && !buffer_uptodate(bh))
			set_buffer_uptodate(bh);
	} while (iblock++, (bh = bh->b_this_page) != head);
	unlock_page(page);
	if (unlikely(err))
		goto failed;

	err = __set_page_dirty_buffers(page);
	if (unlikely(err))
		goto failed;

	nilfs_set_page_to_be_frozen(page);

	err = nilfs_commit_dirty_file(inode);
	if (nr_dirty)
		nilfs_segctor_add_dirty(sci, nr_dirty);
	return err;
 failed:
	nilfs_cancel_file_dirty(inode);
	return err;
}

static int nilfs_prepare_write(struct file *file, struct page *page,
			       unsigned from, unsigned to)
{
	struct inode *inode = page->mapping->host;
	int err;

	unlock_page(page);
	err = nilfs_prepare_file_dirty(inode);
	lock_page(page);
	if (unlikely(err))
		return err;

 	err = block_prepare_write(page, from, to, nilfs_get_block);
	if (unlikely(err))
		nilfs_cancel_file_dirty(inode);
	return err;
}

static int nilfs_commit_write(struct file *file, struct page *page,
			      unsigned from, unsigned to)
{
	struct inode *inode = page->mapping->host;
	struct nilfs_sb_info *sbi = NILFS_SB(inode->i_sb);
	struct nilfs_sc_info *sci = NILFS_SC(sbi);
	unsigned nr_dirty = 0;
	int err;

	if (sci) {
		unsigned blocksize = 1 << inode->i_blkbits;
		unsigned block_start, block_end;
		struct buffer_head *bh, *head;

		for(bh = head = page_buffers(page), block_start = 0;
		    bh != head || !block_start;
		    block_start = block_end, bh = bh->b_this_page) {
			block_end = block_start + blocksize;
			if (block_end > from && block_start < to &&
			    !buffer_dirty(bh))
				nr_dirty++;
		}
	}
	generic_commit_write(file, page, from, to);
	err = nilfs_commit_dirty_file(inode);
	if (nr_dirty)
		nilfs_segctor_add_dirty(sci, nr_dirty);
	return err;
}

#ifndef NILFS_SHRINKER_DISABLE
#if NEED_INVALIDATEPAGE_RETVAL
#define __ESC(f)   return (f)
static int
#else
#define __ESC(f)   do { (f); return; } while(1)
static void
#endif /* NEED_INVALIDATEPAGE_RETVAL */
nilfs_invalidatepage(struct page *page, unsigned long offset)
{
	struct buffer_head *bh = NULL;

	if (PagePrivate(page)) {
		bh = page_buffers(page);
		if (buffer_nilfs_allocated(bh))
			__ESC(nilfs_invalidate_allocated_page(page, offset));
		if (buffer_nilfs_node(bh))
			__ESC(nilfs_invalidate_node_page(page, offset));
	}
	__ESC(block_invalidatepage(page, offset));
}
#undef __ESC
#endif /* !NILFS_SHRINKER_DISABLE */

#if NEED_GET_BLOCKS_T
static int
nilfs_get_blocks(struct inode *inode, sector_t iblock, unsigned long max_blocks,
		 struct buffer_head *bh_result, int create)
{
	int ret;

	ret = nilfs_get_block(inode, iblock, bh_result, create);
	if (likely(ret == 0))
		bh_result->b_size = (1 << inode->i_blkbits);
	return ret;
}
#endif

static ssize_t
nilfs_direct_IO(int rw, struct kiocb *iocb, const struct iovec *iov,
		loff_t offset, unsigned long nr_segs)
{
	struct file *file = iocb->ki_filp;
	struct inode *inode = file->f_mapping->host;
	ssize_t size;
	int err;

	err = nilfs_construct_fdata_segment(inode->i_sb, inode);
	if (NILFS_SEG_ERR(err))
		return err;

	if (rw == WRITE) {
		inode_debug(1, "falling back to buffered write."
			    "(ino=%lu, offset=%llu, nseg=%lu)\n",
			    inode->i_ino, offset, nr_segs);
		return 0;
	}
	/* Needs synchronization with the cleaner */
	size = blockdev_direct_IO(rw, iocb, inode, inode->i_sb->s_bdev, iov,
				  offset, nr_segs,
#if NEED_GET_BLOCKS_T
				  nilfs_get_blocks,
#else
				  nilfs_get_block,
#endif
				  NULL);
	inode_debug(2, "called blockdev_direct_IO() for a read request."
		    "(ino=%lu, offset=%llu, nr_segs=%lu, result=%Zd)\n",
		    inode->i_ino, offset, nr_segs, size);
	return size;
}

struct address_space_operations nilfs_aops = {
	.writepage		= nilfs_writepage,
	.readpage		= nilfs_readpage,
	.sync_page		= nilfs_sync_page,
	.writepages		= nilfs_writepages,
	.set_page_dirty         = nilfs_set_page_dirty,
	.readpages		= nilfs_readpages,
	.prepare_write		= nilfs_prepare_write,
	.commit_write		= nilfs_commit_write,
	.releasepage		= nilfs_releasepage,
	.invalidatepage		= nilfs_invalidatepage,
        .direct_IO		= nilfs_direct_IO,
};

struct nilfs_inode *nilfs_get_inode(struct super_block *sb, ino_t ino,
				    struct buffer_head **pbh)
{
	int res = -EINVAL;
	struct buffer_head *bh;
	struct nilfs_inode_hdr *ih;
	struct nilfs_inode *vi;
	long ino_offset;

	BUG_ON(pbh == NULL);

	*pbh = NULL;
	if (unlikely(!NILFS_VALID_INODE(sb, ino))) {
		nilfs_error(sb, __FUNCTION__, "bad inode number: %lu",
			    (unsigned long) ino);
		goto failed;
	}

	bh = nilfs_inode_bread(sb, ino);
	if (IS_ERR(bh)) {
		nilfs_warning(sb, __FUNCTION__, "unable to read inode: %lu",
			      (unsigned long) ino);
		return (void *)bh;
	}
	*pbh = bh;

	ih = (struct nilfs_inode_hdr *)bh->b_data;
	ino_offset = ino - le64_to_cpu(ih->ih_ino);
	if (unlikely(ino_offset < 0 || ino_offset >= NILFS_SB(sb)->s_inodes_per_block)) {
		nilfs_error(sb, __FUNCTION__,
			    "inode number mismatch: lookup ino = %lu, "
			    "first ino in the inode block = %lu",
			    (unsigned long) ino,
			    (unsigned long) le64_to_cpu(ih->ih_ino));
		brelse(bh);
		goto failed;
	}
	/*
	 * Code for picking up free nilfs_inode(s) and registering them to
	 * s_free_inodes should be inserted here.
	 */
	vi = (struct nilfs_inode *)(bh->b_data + sizeof(struct nilfs_inode_hdr))
		+ ino_offset;
	if (unlikely(vi->i_flags & cpu_to_le32(NILFS_INODE_NEW | NILFS_INODE_UNUSED))) {
		nilfs_error(sb, __FUNCTION__, "read request for unused inode: %lu",
			    (unsigned long) ino);
		brelse(bh);
		goto failed;
	}
	return vi;
  
 failed:
	return ERR_PTR(res);
}

void nilfs_set_inode_flags(struct inode *inode)
{
	unsigned int flags = NILFS_I(inode)->i_flags;

	inode->i_flags &= ~(S_SYNC | S_APPEND | S_IMMUTABLE | S_NOATIME | S_DIRSYNC);
	if (flags & NILFS_SYNC_FL)
		inode->i_flags |= S_SYNC;
	if (flags & NILFS_APPEND_FL)
		inode->i_flags |= S_APPEND;
	if (flags & NILFS_IMMUTABLE_FL)
		inode->i_flags |= S_IMMUTABLE;
#ifndef NILFS_ATIME_DISABLE
	if (flags & NILFS_NOATIME_FL)
#endif
		inode->i_flags |= S_NOATIME;
	if (flags & NILFS_DIRSYNC_FL)
		inode->i_flags |= S_DIRSYNC;
	mapping_set_gfp_mask(inode->i_mapping,
			     mapping_gfp_mask(inode->i_mapping) & ~__GFP_FS);
}

void nilfs_read_inode(struct inode *inode)
{
	struct nilfs_inode_info *ii = NILFS_I(inode);
	ino_t ino = inode->i_ino;
	struct buffer_head *bh;
	struct nilfs_inode *raw_inode = nilfs_get_inode(inode->i_sb, ino, &bh);
	int ret;

#ifdef CONFIG_NILFS_FS_POSIX_ACL
	ii->i_acl = NILFS_ACL_NOT_CACHED;
	ii->i_default_acl = NILFS_ACL_NOT_CACHED;
#endif
	if (IS_ERR(raw_inode))
		goto bad_inode;

	inode->i_mode = le16_to_cpu(raw_inode->i_mode);
	inode->i_uid = (uid_t)le32_to_cpu(raw_inode->i_uid);
	inode->i_gid = (gid_t)le32_to_cpu(raw_inode->i_gid);
	inode->i_nlink = le16_to_cpu(raw_inode->i_links_count);
	inode->i_size = le64_to_cpu(raw_inode->i_size);
	inode->i_atime.tv_sec = le64_to_cpu(raw_inode->i_mtime);
	inode->i_ctime.tv_sec = le64_to_cpu(raw_inode->i_ctime);
	inode->i_mtime.tv_sec = le64_to_cpu(raw_inode->i_mtime);
	inode->i_mtime.tv_nsec = inode->i_mtime.tv_nsec = inode->i_ctime.tv_nsec = 0;
	ii->i_dtime = le64_to_cpu(raw_inode->i_dtime);
	if (inode->i_nlink == 0 && (inode->i_mode == 0 || ii->i_dtime)) {
		/* this inode is deleted */
		brelse(bh);
		goto bad_inode;
	}
	inode->i_blksize = PAGE_SIZE; /* optimal IO size (for stat), not the fs block size */
	inode->i_blocks = le64_to_cpu(raw_inode->i_blocks);
	ii->i_flags = le32_to_cpu(raw_inode->i_flags);
	ii->i_file_acl = le32_to_cpu(raw_inode->i_file_acl);
	ii->i_dir_acl = S_ISREG(inode->i_mode) ? 0 : le32_to_cpu(raw_inode->i_dir_acl);
	inode->i_generation = le32_to_cpu(raw_inode->i_generation);
	ii->i_dtime = 0;
	ii->i_state = 0;

	ret = nilfs_btree_init(&ii->i_block_root, NILFS_BTREE_BLOCK,
			    le64_to_cpu(raw_inode->i_block_root), inode);

	if (ret < 0) {
		inode_debug(0, "failed nilfs_btree_init, ret=%d, ino=%lu, root=%llu\n",
			    ret, ino, le64_to_cpu(raw_inode->i_block_root));
		brelse(bh);
		goto bad_inode;
	}

	ii->i_bh = 0;  /* This field is valid only for dirty ones */

	if (S_ISREG(inode->i_mode)) {
		if (unlikely(ino == NILFS_SKETCH_INO)) {
			ret = nilfs_read_sketch_inode(ii);
			if (ret < 0) {
				brelse(bh);
				goto bad_inode;
			}
		} else {
			inode->i_op = &nilfs_file_inode_operations;
			inode->i_fop = &nilfs_file_operations;
			inode->i_mapping->a_ops = &nilfs_aops;
		}
	} else if (S_ISDIR(inode->i_mode)) {
		inode->i_op = &nilfs_dir_inode_operations;
		inode->i_fop = &nilfs_dir_operations;
		inode->i_mapping->a_ops = &nilfs_aops;
	} else if (S_ISLNK(inode->i_mode)) {
		inode->i_op = &nilfs_symlink_inode_operations;
		inode->i_mapping->a_ops = &nilfs_aops;
	} else {
		inode->i_op = &nilfs_special_inode_operations;
		init_special_inode(inode, inode->i_mode, 
				   new_decode_dev(le64_to_cpu(raw_inode->i_block_root)));
	}
	brelse(bh);
	nilfs_set_inode_flags(inode);
#ifdef CONFIG_NILFS_DEBUG
	atomic_inc(&nilfs_allocated_inode_n);
#endif
	return;
  
 bad_inode:
	make_bad_inode(inode);
}

void nilfs_update_inode(struct inode *inode, struct nilfs_inode *raw_inode,
			struct buffer_head *bh)
{
	struct nilfs_inode_info *ii = NILFS_I(inode);
	struct super_block *sb = inode->i_sb;
	struct nilfs_sb_info *sbi = NILFS_SB(sb);

	spin_lock_irq(&sbi->s_segctor.dirty_files_lock);
	if (ii->i_state & NILFS_STATE_NEW) {
		memset(raw_inode, 0, NILFS_SB(sb)->s_inode_size);
		ii->i_state &= ~NILFS_STATE_NEW;
	}
	ii->i_state |= NILFS_STATE_INODE_DIRTY;
	spin_unlock_irq(&sbi->s_segctor.dirty_files_lock);

	raw_inode->i_mode = cpu_to_le16(inode->i_mode);
	raw_inode->i_uid = cpu_to_le32(inode->i_uid);
	raw_inode->i_gid = cpu_to_le32(inode->i_gid);
	raw_inode->i_links_count = cpu_to_le16(inode->i_nlink);
	raw_inode->i_size = cpu_to_le64(inode->i_size);
	raw_inode->i_ctime = cpu_to_le64(inode->i_ctime.tv_sec);
	raw_inode->i_mtime = cpu_to_le64(inode->i_mtime.tv_sec);
	raw_inode->i_blocks = cpu_to_le64(inode->i_blocks);

	raw_inode->i_dtime = cpu_to_le64(ii->i_dtime);
	raw_inode->i_flags = cpu_to_le32(ii->i_flags);
	raw_inode->i_file_acl = cpu_to_le32(ii->i_file_acl);
	if (!S_ISREG(inode->i_mode))
		raw_inode->i_dir_acl = cpu_to_le32(ii->i_dir_acl);

	raw_inode->i_generation = cpu_to_le32(inode->i_generation);
	if (unlikely(S_ISCHR(inode->i_mode) || S_ISBLK(inode->i_mode)))
		raw_inode->i_block_root =
			cpu_to_le64(new_encode_dev(inode->i_rdev));
	else
		raw_inode->i_block_root =
			cpu_to_le64(NILFS_BTREE_ROOT_DBN(&ii->i_block_root));
}

void nilfs_truncate(struct inode *inode)
{
	fbn_t iblock;
	unsigned int blocksize;
	struct nilfs_inode_info *ii = NILFS_I(inode);
	int ret;

	inode_debug(1, "called. (ino=%lu)\n", inode->i_ino);
	if (!(S_ISREG(inode->i_mode) || S_ISDIR(inode->i_mode) ||
	      S_ISLNK(inode->i_mode)))
		return;
	if (IS_APPEND(inode) || IS_IMMUTABLE(inode))
		return;

	blocksize = inode->i_sb->s_blocksize;
	iblock = (inode->i_size + blocksize - 1) >> NILFS_BLOCK_SIZE_BITS(inode->i_sb);
	ret = nilfs_prepare_file_dirty(inode);
	if (unlikely(ret))
		return;

	block_truncate_page(inode->i_mapping, inode->i_size, nilfs_get_block);

	ret = nilfs_btree_truncate(&ii->i_block_root, iblock);

	inode->i_mtime = inode->i_ctime = CURRENT_TIME;
	if (IS_SYNC(inode))
		nilfs_set_transaction_flag(NILFS_TI_SYNC);
		
	nilfs_commit_dirty_file(inode);
        /* May construct a logical segment and may fail in sync mode.
	   But truncate has no return value. */
}

void nilfs_delete_inode(struct inode *inode)
{
	struct nilfs_transaction_info ti;
	struct super_block *sb = inode->i_sb;
	int err;

	if (unlikely(is_bad_inode(inode))) {
#if NEED_TRUNCATE_INODE_PAGES
		if (inode->i_data.nrpages)
			truncate_inode_pages(&inode->i_data, 0);
#endif
		clear_inode(inode);
		return;
	}
	err = nilfs_transaction_begin(sb, &ti);
	if (unlikely(err))
		BUG();
#if NEED_TRUNCATE_INODE_PAGES
	if (inode->i_data.nrpages)
		truncate_inode_pages(&inode->i_data, 0);
#endif
	nilfs_free_inode(inode);
	/* nilfs_free_inode() marks inode buffer dirty */
	if (IS_SYNC(inode))
		nilfs_set_transaction_flag(NILFS_TI_SYNC);
	nilfs_transaction_end(sb);
        /* May construct a logical segment and may fail in sync mode.
	   But delete_inode has no return value. */
}

int nilfs_setattr(struct dentry *dentry, struct iattr *iattr)
{
	struct nilfs_transaction_info ti;
	struct inode *inode = dentry->d_inode;
	struct super_block *sb = inode->i_sb;
	int err, err2;

	err = inode_change_ok(inode, iattr);
	if (err)
		return err;

	err = nilfs_transaction_begin(sb, &ti);
	if (unlikely(err))
		BUG();
	err = inode_setattr(inode, iattr);
	if (!err && (iattr->ia_valid & ATTR_MODE))
		err = nilfs_acl_chmod(inode);
	err2 = nilfs_transaction_end(sb);
	return (err ? : err2);
}

struct nilfs_inode *
__nilfs_load_inode_block_nolock(struct nilfs_sb_info *sbi,
				struct inode *inode,
				struct buffer_head **pbh)
{
	struct nilfs_inode_info *ii = NILFS_I(inode);
	struct nilfs_inode *raw_inode;

	/* Caller of this function MUST lock s_segctor.dirty_files_lock */
	if (ii->i_bh == NULL) {
		spin_unlock_irq(&sbi->s_segctor.dirty_files_lock);
		raw_inode = nilfs_get_inode(sbi->s_super, inode->i_ino, pbh);
		spin_lock_irq(&sbi->s_segctor.dirty_files_lock);
		if (!IS_ERR(raw_inode) && ii->i_bh == NULL) {
			ii->i_bh = *pbh;
			get_bh(*pbh);
		}
	} else {
		raw_inode = nilfs_raw_inode(ii);

		*pbh = ii->i_bh;
		get_bh(*pbh);
	}
	return raw_inode;
}

void
nilfs_put_inode(struct inode *inode)
{
	VINODE_DEBUG(inode, "");
}

/* Local Variables:	*/
/* eval: (c-set-style "linux")	*/
/* End:			*/
