/*
 * btree.h - NILFS B-tree operations.
 *
 * Copyright (C) 2005 Nippon Telegraph and Telephone Corporation.
 *
 * This file is part of NILFS.
 *
 * NILFS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * NILFS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NILFS; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * btree.h,v 1.2 2006/02/21 02:32:38 kihara Exp
 *
 * Written by Koji Sato <koji@osrg.net>
 */

#ifndef _NILFS_BTREE_H
#define _NILFS_BTREE_H

#include <linux/types.h>
#include <linux/buffer_head.h>
#include <linux/fs.h>
#include "nilfs_types.h"
#include "radix-tree-64.h"

/**
 * nilfs_btree_type - type of B-tree
 * @NILFS_BTREE_BLOCK: block mapping B-tree
 * @NILFS_BTREE_INODE: inode B-tree
 * @NILFS_BTREE_DIR: director B-tree (not implemented)
 */
enum nilfs_btree_type {
	NILFS_BTREE_BLOCK,
	NILFS_BTREE_INODE,
	NILFS_BTREE_DIR,
};

/* B-tree key and pointer on memory */
typedef __u64 nilfs_btree_key_t;
typedef __u64 nilfs_btree_ptr_t;
typedef __s64 nilfs_btree_keydiff_t;

#define NILFS_BTREE_NODE_NEXT_BH(node)	\
	(*(struct buffer_head **)(&(node)->bn_next))

#define NILFS_BTREE_BH_TO_NODE(bh)	\
	((struct nilfs_btree_node *)((bh)->b_data))


/* Maximum number of levels */
#define NILFS_BTREE_MAX_LEVELS	14

/**
 * nilfs_btree_path -
 * @bp_nlevels: number of levels
 * @bp_index: index at each level
 * @bp_bh: pointer to node block at each level
 */
struct nilfs_btree_path {
	int bp_nlevels;
	int bp_index[NILFS_BTREE_MAX_LEVELS];
	struct buffer_head *bp_bh[NILFS_BTREE_MAX_LEVELS];
};

struct nilfs_btree_operations;

/**
 * nilfs_btree - B-tree structure on memory
 * @bt_state: status
 * @bt_sem: semaphore for B-tree
 * @bt_root: B-tree root
 * @bt_owner: owner of tree
 * @bt_dirty_buffers: list of dirty buffers
 * @bt_ops: B-tree operations table
 * @bt_bt_blksize: block size in bytes
 * @bt_blkbits: block size in bits
 * @bt_rtree: radix tree root
 * @bt_rtree_lock: spinlock for radix tree
 * @bt_inactive_list_lock: spinlock for inactive list
 * @bt_info: B-tree-type-specific data
 */
struct nilfs_btree {
	int bt_state;
	struct rw_semaphore bt_sem;
	nilfs_btree_ptr_t bt_root;
	void *bt_owner;
	struct buffer_head *bt_dirty_buffers;
	struct nilfs_btree_operations *bt_ops;
	unsigned long bt_blksize;
	unsigned int bt_blkbits;
	struct radix_tree_64_root *bt_rtree;
	spinlock_t *bt_rtree_lock;
	spinlock_t *bt_inactive_list_lock;
	void *bt_info;
};

enum nilfs_btree_state {
	NILFS_BTREE_DIRTY,
};

#define NILFS_BTREE_ROOT_DBN(btree)	\
	(NILFS_BTREE_PTR_IS_DBN((btree)->bt_root) ? \
	NILFS_BTREE_PTR_TO_DBN((btree)->bt_root) : \
	0)

#define NILFS_BTREE_MAX_KEY		(~(nilfs_btree_key_t)0)

#define NILFS_BTREE_CHAR_BIT		8
#define NILFS_BTREE_PTR_MASK		((nilfs_btree_ptr_t)1 << (sizeof(nilfs_btree_ptr_t) * NILFS_BTREE_CHAR_BIT - 1))
#define NILFS_BTREE_INVALID_PTR	0
#define NILFS_BTREE_PTR_IS_BH(ptr)	(((ptr) != NILFS_BTREE_INVALID_PTR) && \
					 ((ptr) & NILFS_BTREE_PTR_MASK))
#define NILFS_BTREE_PTR_IS_DBN(ptr)	(((ptr) != NILFS_BTREE_INVALID_PTR) && \
					 !((ptr) & NILFS_BTREE_PTR_MASK))
#define NILFS_BTREE_BH_TO_PTR(bh)	((nilfs_btree_ptr_t)((unsigned long)(bh) >> 1 | NILFS_BTREE_PTR_MASK))
#define NILFS_BTREE_PTR_TO_BH(ptr)	((struct buffer_head *)((unsigned long)(((ptr) & ~NILFS_BTREE_PTR_MASK) << 1)))
#define NILFS_BTREE_DBN_TO_PTR(dbn)	((nilfs_btree_ptr_t)(dbn))
#define NILFS_BTREE_PTR_TO_DBN(ptr)	((dbn_t)ptr)

/**
 * nilfs_btree_operations - B-tree operations table
 * @bop_get_node:
 * @bop_get_new_node:
 * @bop_put_node:
 * @bop_delete_node:
 * @bop_delete_all_node:
 * @bop_find_unused_key:
 * @bop_mark_node_dirty:
 * @bop_mark_node_prepare_dirty:
 */
struct nilfs_btree_operations {
	struct buffer_head *(*bop_get_node)(struct nilfs_btree *, dbn_t);
	struct buffer_head *(*bop_get_new_node)(struct nilfs_btree *);
	void (*bop_put_node)(struct nilfs_btree *, struct buffer_head *);
	void (*bop_delete_node)(struct nilfs_btree *, struct buffer_head *);
	void (*bop_delete_all_node)(struct nilfs_btree *);
	int (*bop_find_unused_key)(struct nilfs_btree *, nilfs_btree_key_t *);
	void (*bop_mark_node_dirty)(struct nilfs_btree *, struct buffer_head *);
	void (*bop_mark_node_prepare_dirty)(struct nilfs_btree *, struct buffer_head *);

};

/* B-tree operations invocation */
#define NILFS_BTREE_BOP(btree, op)	\
	((btree)->bt_ops->bop_##op)
#define NILFS_BTREE_BOP_GET_NODE(btree, dbn)	\
	NILFS_BTREE_BOP(btree, get_node)(btree, dbn)
#define NILFS_BTREE_BOP_GET_NEW_NODE(btree)	\
	NILFS_BTREE_BOP(btree, get_new_node)(btree)
#define NILFS_BTREE_BOP_PUT_NODE(btree, bh)	\
	NILFS_BTREE_BOP(btree, put_node)(btree, bh)
#define NILFS_BTREE_BOP_DELETE_NODE(btree, bh)	\
	NILFS_BTREE_BOP(btree, delete_node)(btree, bh)
#define NILFS_BTREE_BOP_DELETE_ALL_NODE(btree)	\
	NILFS_BTREE_BOP(btree, delete_all_node)(btree)
#define NILFS_BTREE_BOP_MARK_NODE_DIRTY(btree, bh)	\
	NILFS_BTREE_BOP(btree, mark_node_dirty)(btree, bh)
#define NILFS_BTREE_BOP_MARK_NODE_PREPARE_DIRTY(btree, bh)	\
	NILFS_BTREE_BOP(btree, mark_node_prepare_dirty)(btree, bh)
#define NILFS_BTREE_BOP_FIND_UNUSED_KEY(btree, key)	\
	NILFS_BTREE_BOP(btree, find_unused_key)(btree, key)

/* return code in case of error */
#define NILFS_BTREE_ENOMEM	1	/* Out of memory */
#define NILFS_BTREE_EIO		2	/* I/O Error */
#define NILFS_BTREE_ENOKEY	3	/* No such key */
#define NILFS_BTREE_EEXIST	4	/* Key exist */
#define NILFS_BTREE_ENOSPC	5	/* No space left on node */
#define NILFS_BTREE_ENOSIB	6	/* No siblings */
#define NILFS_BTREE_EINVAL	7	/* Invalid argument */
#define NILFS_BTREE_ENOOP	8	/* Undefined operation */

#define nilfs_btree_max_bits()	\
	(sizeof(nilfs_btree_key_t) * NILFS_BTREE_CHAR_BIT - 1)

int nilfs_btree_path_cache_init(void);
void nilfs_btree_path_cache_destroy(void);
int nilfs_btree_test_and_clear_dirty(struct nilfs_btree *);
int nilfs_btree_init(struct nilfs_btree *, int, dbn_t, void *);
int nilfs_btree_lookup(struct nilfs_btree *, unsigned long, unsigned long *);
int nilfs_btree_insert(struct nilfs_btree *, unsigned long, struct buffer_head *);
int nilfs_btree_delete(struct nilfs_btree *, unsigned long);
void nilfs_btree_clear(struct nilfs_btree *);
int nilfs_btree_truncate(struct nilfs_btree *, nilfs_btree_key_t);
int nilfs_btree_find_unused_key(struct nilfs_btree *, unsigned long *);
int nilfs_btree_find_unused_key_in_range(struct nilfs_btree *, nilfs_btree_key_t, nilfs_btree_key_t, nilfs_btree_key_t, nilfs_btree_key_t *);
int nilfs_btree_mark_from_data(struct nilfs_btree *, struct buffer_head *);
int nilfs_btree_mark_from_node(struct nilfs_btree *, struct buffer_head *);
int nilfs_btree_update_data_dbn(struct nilfs_btree *, struct buffer_head *, dbn_t);
int nilfs_btree_update_node_dbn(struct nilfs_btree *, struct buffer_head *, dbn_t);
void nilfs_btree_lookup_dirty_buffers_begin(struct nilfs_btree *);
unsigned int nilfs_btree_lookup_dirty_buffers(struct nilfs_btree *, struct buffer_head **, unsigned int);
void nilfs_btree_lookup_dirty_buffers_end(struct nilfs_btree *);
int nilfs_btree_part_of(struct nilfs_btree *, struct buffer_head *);

#endif	/* _NILFS_BTREE_H */

/* Local Variables:	*/
/* eval: (c-set-style "linux")	*/
/* End:			*/
