/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 *  Copyright (C) 2006 Takuro Ashie
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */ 

#ifdef HAVE_CONFIG_H
#   include <config.h>
#endif

#include <glib/gi18n-lib.h>
#include "tomoe-gucharmap.h"

enum {
    SELECTED_SIGNAL,
    LAST_SIGNAL
};

typedef struct _TomoeGucharmapPrivate TomoeGucharmapPrivate;
struct _TomoeGucharmapPrivate
{
    GtkWidget         *chapters_type_combobox;
    GtkWidget         *chapters_combobox;
    GucharmapChapters *chapters;
    GucharmapCharmap  *charmap;
    gchar             *selected;
};

#define TOMOE_GUCHARMAP_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), TOMOE_TYPE_GUCHARMAP, TomoeGucharmapPrivate))

G_DEFINE_TYPE (TomoeGucharmap, tomoe_gucharmap, GTK_TYPE_TABLE)

static void dispose                        (GObject          *object);
static void setup_chapters_combo_box       (TomoeGucharmap   *page);
static void on_chapters_type_combo_changed (GtkComboBox      *combobox,
                                            gpointer          data);
static void on_chapters_combo_changed      (GtkComboBox      *combobox,
                                            gpointer          data);
static void on_gucharmap_activate          (GucharmapCharmap *charmap,
                                            gunichar          ch,
                                            gpointer          data);
static void on_gucharmap_button_release    (GtkWidget        *widget,
                                            GdkEventButton   *event,
                                            gpointer          data);

static guint gucharmap_signals[LAST_SIGNAL] = { 0 };

GtkWidget *
tomoe_gucharmap_new (void)
{
    return GTK_WIDGET(g_object_new (TOMOE_TYPE_GUCHARMAP, NULL));
}

static void
tomoe_gucharmap_class_init (TomoeGucharmapClass *klass)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

    gobject_class->dispose = dispose;

    gucharmap_signals[SELECTED_SIGNAL] =
      g_signal_new ("selected",
  		  G_TYPE_FROM_CLASS (klass),
  		  G_SIGNAL_RUN_LAST,
  		  G_STRUCT_OFFSET (TomoeGucharmapClass, selected),
  		  NULL, NULL,
  		  g_cclosure_marshal_VOID__VOID,
  		  G_TYPE_NONE, 0);

    klass->selected = NULL;

    g_type_class_add_private (gobject_class, sizeof (TomoeGucharmapPrivate));
}

static void
tomoe_gucharmap_init (TomoeGucharmap *page)
{
    TomoeGucharmapPrivate *priv = TOMOE_GUCHARMAP_GET_PRIVATE (page);
    GtkWidget *vbox, *hbox, *combobox, *widget;

    priv->selected = NULL;

    gtk_table_resize (GTK_TABLE (page), 1, 1);
    gtk_table_set_homogeneous (GTK_TABLE (page), FALSE);

    vbox = gtk_vbox_new (FALSE, 0);
    gtk_container_set_border_width (GTK_CONTAINER (vbox), 8);
    gtk_table_attach_defaults (GTK_TABLE (page), vbox, 0, 1, 0, 1);
    gtk_widget_show (vbox);

    /* opiton menu for selecting unicode blocks */
    hbox = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 2);
    gtk_widget_show (hbox);

    combobox = gtk_combo_box_new_text ();
    priv->chapters_type_combobox = combobox;
    gtk_combo_box_append_text (GTK_COMBO_BOX (combobox), _("Unicode block"));
    gtk_combo_box_append_text (GTK_COMBO_BOX (combobox), _("Unicode script"));
    gtk_combo_box_set_active (GTK_COMBO_BOX (combobox), 0);
    gtk_box_pack_start (GTK_BOX (hbox), combobox, FALSE, FALSE, 2);
    gtk_widget_show (combobox);

    combobox = gtk_combo_box_new_text ();
    priv->chapters_combobox = combobox;
    gtk_box_pack_start (GTK_BOX (hbox), combobox, FALSE, FALSE, 2);
    gtk_widget_show (combobox);

    /* gucharmap */
    priv->chapters  = GUCHARMAP_CHAPTERS (gucharmap_block_chapters_new ());
    priv->charmap = GUCHARMAP_CHARMAP (gucharmap_charmap_new (priv->chapters));
    widget = GTK_WIDGET (priv->charmap);
    gtk_box_pack_start (GTK_BOX (vbox), widget, TRUE, TRUE, 2);
    gtk_widget_show (widget);

    /* Hide chapters page by default. Usually use combo box instead. */
    gtk_widget_hide (GTK_WIDGET (priv->chapters));

#if 0
    /* CharDetails view will be here */
    widget = gtk_paned_get_child2 (GTK_PANED (widget));
    vbox = gtk_vbox_new (FALSE, 0);
    gtk_notebook_append_page (GTK_NOTEBOOK (widget),
                              vbox, gtk_label_new (_("Writing")));
    gtk_widget_show (vbox);
#endif

    /* setup */
    setup_chapters_combo_box (page);
    g_signal_connect (G_OBJECT (priv->chapters_type_combobox), "changed",
                      G_CALLBACK (on_chapters_type_combo_changed),
                      (gpointer) page);
    g_signal_connect (G_OBJECT (priv->chapters_combobox), "changed",
                      G_CALLBACK (on_chapters_combo_changed),
                      (gpointer) page);
    g_signal_connect (G_OBJECT (priv->charmap->chartable), "activate",
                      G_CALLBACK (on_gucharmap_activate),
                      (gpointer) page);
    g_signal_connect (G_OBJECT (priv->charmap->chartable),
                      "button-release-event",
                      G_CALLBACK (on_gucharmap_button_release),
                      (gpointer) page);
}

static void
dispose (GObject *object)
{
    TomoeGucharmapPrivate *priv = TOMOE_GUCHARMAP_GET_PRIVATE (object);

    priv->chapters_type_combobox = NULL;
    priv->chapters_combobox      = NULL;
    priv->chapters               = NULL;
    priv->charmap                = NULL;
    if (priv->selected) {
        g_free (priv->selected);
        priv->selected = NULL;
    }

    if (G_OBJECT_CLASS(tomoe_gucharmap_parent_class)->dispose)
        G_OBJECT_CLASS(tomoe_gucharmap_parent_class)->dispose(object);
}

GucharmapCharmap *
tomoe_gucharmap_get_charmap (TomoeGucharmap *page)
{
    TomoeGucharmapPrivate *priv;

    g_return_val_if_fail (TOMOE_IS_GUCHARMAP (page), NULL);

    priv = TOMOE_GUCHARMAP_GET_PRIVATE (page);
    return priv->charmap;
}

const gchar *
tomoe_gucharmap_get_selected_char (TomoeGucharmap *page)
{
    TomoeGucharmapPrivate *priv;
    GucharmapTable *table;
    gunichar c;
    gchar buf[7];
    gint bytes;

    g_return_val_if_fail (TOMOE_IS_GUCHARMAP (page), NULL);

    priv = TOMOE_GUCHARMAP_GET_PRIVATE (page);
    table = GUCHARMAP_TABLE (priv->charmap->chartable);

    c = gucharmap_table_get_active_character (table);
    bytes = g_unichar_to_utf8(c, buf);

    if (bytes > 0 && bytes < 7) {
        buf[bytes] = '\0';
        g_free (priv->selected);
        priv->selected = g_strdup (buf);
        return priv->selected;
    }

    return NULL;
}

TomoeChar *
tomoe_gucharmap_get_selected_tomoe_char (TomoeGucharmap *page)
{
    /* not implemented yet */
    return NULL;
}

static void
setup_chapters_combo_box (TomoeGucharmap *page)
{
    TomoeGucharmapPrivate *priv = TOMOE_GUCHARMAP_GET_PRIVATE (page);
    GtkTreeModel *model = priv->chapters->tree_model;

    gtk_combo_box_set_model (GTK_COMBO_BOX (priv->chapters_combobox), model);
    gtk_combo_box_set_active (GTK_COMBO_BOX (priv->chapters_combobox), 0);
}

static void
on_chapters_type_combo_changed (GtkComboBox *combobox, gpointer data)
{
    TomoeGucharmap *page = data;
    TomoeGucharmapPrivate *priv = TOMOE_GUCHARMAP_GET_PRIVATE (page);
    gint idx = gtk_combo_box_get_active (combobox);

    if (idx == 1)
        priv->chapters = GUCHARMAP_CHAPTERS (gucharmap_script_chapters_new ());
    else
        priv->chapters = GUCHARMAP_CHAPTERS (gucharmap_block_chapters_new ());
    gucharmap_charmap_set_chapters (priv->charmap, priv->chapters);
    gtk_widget_hide (GTK_WIDGET (priv->chapters));
    setup_chapters_combo_box (page);
}

static void
on_chapters_combo_changed (GtkComboBox *combobox, gpointer data)
{
    TomoeGucharmap *page = data;
    TomoeGucharmapPrivate *priv = TOMOE_GUCHARMAP_GET_PRIVATE (page);
    GtkTreePath *path;
    gchar buf[8];
    gint idx;

    idx = gtk_combo_box_get_active (combobox);
    g_snprintf (buf, 8 , "%d", idx);
    path = gtk_tree_path_new_from_string (buf);
    gtk_tree_view_set_cursor (GTK_TREE_VIEW (priv->chapters->tree_view),
                              path, NULL, FALSE);
    gtk_tree_path_free (path);
}

static void
on_gucharmap_activate (GucharmapCharmap *charmap, gunichar ch,
                       gpointer data)
{
    TomoeGucharmap *gucharmap = TOMOE_GUCHARMAP (data);

    g_signal_emit (G_OBJECT (gucharmap),
                   gucharmap_signals[SELECTED_SIGNAL], 0);
}

static void
on_gucharmap_button_release (GtkWidget *widget,
                             GdkEventButton *event,
                             gpointer data)
{
#if 0
    TomoeGucharmap *gucharmap = TOMOE_GUCHARMAP (data);

    g_signal_emit (G_OBJECT (gucharmap),
                   gucharmap_signals[SELECTED_SIGNAL], 0);
#endif
}

/*
 * vi:ts=4:nowrap:ai:expandtab
 */
