/*
 * control.c
 * Thomas Nemeth, le 15.10.2003
 *
 *   Copyright (C) 1999  Thomas Nemeth
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <strings.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <signal.h>
#include <pthread.h>
#include "defines.h"
#include <ncurses.h>
#include "tmms.h"
#include "xmem.h"
#include "terminal.h"
#include "playlist.h"
#include "playerwin.h"
#include "playlistwin.h"
#include "modules.h"
#include "control.h"
#include "printlog.h"


#define BUFLEN 1024


static int frame, input_end;
static int minutes, seconds;
static int rmins, rsecs;
static int bitrate, length_set;
static pid_t p_pid;
static int fildes[2];
static Status status = STOPPED, request = NOTHING;


/* User Request */

void set_command(Status s)
{
        struct timespec ts = {0, 10};

        request = s;
        pthread_mutex_unlock(&mutex_request);
        nanosleep(&ts, NULL);
        pthread_mutex_lock(&mutex_request);
}


static Status request_changed()
{
        Status requested = request;
        request = NOTHING;
        return requested;
}


static Status wait_request()
{
        Status s = NOTHING;

        pthread_mutex_lock(&mutex_request);
        s = request_changed();
        pthread_mutex_unlock(&mutex_request);

        return s;
}


/* Internal status */

Status get_status()
{
        Status s;
        pthread_mutex_lock(&mutex_status);
        s = status;
        pthread_mutex_unlock(&mutex_status);
        return s;
}


static void set_status(Status s)
{
        pthread_mutex_lock(&mutex_status);
        status = s;
        pthread_mutex_unlock(&mutex_status);
        display_status();
}


/* Real control operations (launch, parse, stop)... */

static void player_cmd()
{
        char *cmd = get_command();
        char *opt = get_option();
        char *pos = get_position(frame, seconds);
        char *args[8];
        int i, arg = 0;

        if ((! cmd) || (! opt))
        {
                return;
        }
        /* Arguments */
        args[arg++] = cmd;
        args[arg++] = opt;
        if (output_device)
        {
                args[arg++] = get_device();
                args[arg++] = output_device;
        }
        if (pos)
        {
                args[arg++] = get_posopt();
                args[arg++] = pos;
        }
        args[arg++] = playlist_current();
#if DEBUG_LEVEL!=0
        printlog(2, "MODULE COMMAND : [");
        for (i = 0 ; i < arg ; i++) printlog(2, "%s ", args[i]);
        printlog(2, "]\n");
#endif
        for (i = arg ; i < 8 ; i++)
        {
                args[i] = NULL;
        }
        /* file descriptors */
        close(fildes[0]);
        dup2(fildes[1], STDOUT_FILENO);
        dup2(fildes[1], STDERR_FILENO);
        /* new session */
        setsid();
        /* Env */
        setenv("LC_ALL", "C", 1);
        /* execution */
        execvp(cmd, args);
        close(fildes[1]);
        free(pos);
}


static void launch_player()
{
        if (! has_module(playlist_current()))
        {
                return;
        }
        printlog(5, "Start position => %02d:%02d [%02d:%02d] (fr %d/br %d)\n",
                        minutes, seconds, rmins, rsecs, frame, bitrate);
        pipe(fildes);
        p_pid = fork();
        switch (p_pid)
        {
                case 0:
                        player_cmd();
                        _exit(0);
                case -1:
                        perror("fork");
                        quit();
                default:
                        close(fildes[1]);
                        fildes[1] = -1;
                        break;
        }
        printlog(5, "Player launched : PID = %d\n", p_pid);
        length_set = FALSE;
        input_end = FALSE;
}


static void grab_output(const char *input, char *datas, int *dpos)
{
        int i;
                       
        for (i = 0 ; input[i] ; i++)
        {
                if ((input[i] != '\n') &&
                    (input[i] != '\r') &&
                    (input[i] != 7))
                {
                        datas[(*dpos)++] = input[i];
                        if (*dpos == BUFLEN)
                        {
                                memset(datas, 0, BUFLEN);
                                *dpos = 0;
                                break;
                        }
                }
                else
                {
                        datas[*dpos] = 0;
                        printlog(2, "MODULE OUTPUT : [%s]\n", datas);
                        get_module()->parse(datas, &frame,
                                        &minutes, &seconds, &rmins, &rsecs,
                                        &bitrate);
                        printlog(5, "> %02d:%02d [%02d:%02d] (fr %d/br %d)\n",
                                        minutes, seconds, rmins, rsecs,
                                        frame, bitrate);
                        if (!length_set)
                        {
                                printlog(5, "Trying length : %02d:%02d\n",
                                                rmins, rsecs);
                                length_set = set_current_length(rmins, rsecs);
                                if (length_set)
                                {
                                        printlog(5, "length OK\n");
                                        display_list();
                                }
                        }
                        *dpos = 0;
                        memset(datas, 0, BUFLEN);
                }
        }
}


static int get_datas(int cleanup)
{
        char input[BUFLEN];
        int  ret = 1;
        static int dpos = 0;
        static char datas[BUFLEN];
        Module *module = get_module();

        if (cleanup)
        {
                memset(datas, 0, BUFLEN);
                dpos = 0;
                unload_module();
                return -1;
        }
        memset(input, 0, BUFLEN);
        if (read(fildes[0], input, BUFLEN) > 0)
        {
                if (module)
                {
                        grab_output(input, datas, &dpos);
                }
        }
        else
        {
                ret = 0;
                input_end = TRUE;
                unload_module();
                memset(datas, 0, BUFLEN);
                dpos = 0;
                printlog(2, "MODULE PLAYING ENDED.\n");
        }

        return ret;
}


void stop_player()
{
        if (p_pid != 0)
        {
                pid_t t;
                close(fildes[0]);
                fildes[0] = -1;
                if (kill( -p_pid, SIGINT) == -1)
                {
                        struct timespec ts = {1, 0};
                        nanosleep(&ts, NULL);
                        if (kill( -p_pid, SIGINT) == -1)
                        {
                                fprintf(stderr, "Failed to kill player...\n");
                        }
                }
                t = waitpid(p_pid, NULL, WNOHANG);
                if ((t != p_pid) && (t != -1))
                {
                        waitpid(p_pid, NULL, 0);
                }
                p_pid = 0;
        }
}


/* Actions */

static void reset_internals()
{
        printlog(5, "Resetting internals variables...\n");
        frame = 0;
        minutes = 0;
        seconds = 0;
        rmins = 0;
        rsecs = 0;
        bitrate = 0;
}


static int do_stop()
{
        int cont = FALSE;

        reset_internals();
        if (input_end == TRUE)
        {
                printlog(5, "Input end... ");
                if (repeat_mode == REPEAT_ONE)
                {
                        printlog(5, "Repeating that title.\n");
                        cont = TRUE;
                }
                else if (repeat_mode == REPEAT_ALL)
                {
                        printlog(5, "Repeating all [selected] titles.\n");
                        cont = playlist_next(TRUE);
                }
                else
                {
                        printlog(5, "No repeat, listening next title if it exists.\n");
                        cont = playlist_next(FALSE);
                        if (! cont)
                        {
                                set_status(STOPPED);
                        }
                }
        }
        else
        {
                printlog(5, "Stop was requested !\n");
                set_status(STOPPED);
        }
        printlog(5, "Continuing : %s\n", cont ? "YES" : "NO");
        display_time(minutes, seconds, rmins, rsecs, bitrate);
        return cont;
}


static void do_ffw()
{
        frame += 150;
        seconds += 10;
}


static void do_rwd()
{
        frame -= 150;
        if (frame < 0)
        {
                frame = 0;
        }
        seconds -= 10;
        if (seconds < 0)
        {
                seconds = 0;
        }
}


static void do_next()
{
        reset_internals();
        playlist_next(TRUE);
        display_list();
        display_song();
        display_time(minutes, seconds, rmins, rsecs, bitrate);
}


static void do_prev()
{
        if ((minutes == 0) && (seconds < 3))
        {
                playlist_prev(TRUE);
        }
        reset_internals();
        display_list();
        display_song();
        display_time(minutes, seconds, rmins, rsecs, bitrate);
}


/* returns TRUE => relaunch player */
static int parse_request(Status s)
{
        int cont;

        printlog(2, "REQUEST : ");
        seconds = 60 * minutes + seconds;
        switch (s)
        {
                case PAUSED:
                        printlog(2, "PAUSED\n");
                        cont = FALSE;
                        set_status(PAUSED);
                        break;
                case STOPPED:
                        printlog(2, "STOPPED\n");
                        cont = do_stop();
                        break;
                case FFW:
                        printlog(2, "FFW\n");
                        do_ffw();
                        cont = TRUE;
                        break;
                case RWD:
                        printlog(2, "RWD\n");
                        do_rwd();
                        cont = TRUE;
                        break;
                case NEXT:
                        printlog(2, "NEXT\n");
                        do_next();
                        cont = TRUE;
                        break;
                case PREV:
                        printlog(2, "PREV\n");
                        do_prev();
                        cont = TRUE;
                        break;
                default:  /* ERROR */
                        printlog(2, "%d\n", s);
                        cont = FALSE;
                        set_status(STOPPED);
        }
        if (cont)
        {
                set_status(PLAYING);
        }
        return cont;
}


/* get datas from player, displays them and wait for a user command */

static void control_player()
{
        int cont = TRUE;

        while (cont)
        {
                Status s;

                if (is_playable(playlist_current()))
                {
                        int got;
                        set_status(PLAYING);
                        display_song();
                        display_list();
                        launch_player();
                        do
                        {
                                s   = request_changed();
                                got = get_datas(FALSE);
                                display_time(minutes, seconds, rmins, rsecs,
                                             bitrate);
                        }
                        while ((s == NOTHING) && (got == 1));
                        if (got == 0)
                        {
                                s = STOPPED;
                        }
                        stop_player();
                        (void) get_datas(TRUE);
                }
                else
                {
                        s = NEXT;
                }
                cont = parse_request(s);
        }
}


/* Main control */

void thread_control()
{
        pthread_t tid = pthread_self();

        pthread_detach(tid);
        status = STOPPED;
        repeat_mode = REPEAT_NOT;
        while (1)
        {
                Status s = wait_request();
                switch (s)
                {
                        case PLAYING:
                                control_player();
                                break;
                        case PAUSED:
                                set_status(PAUSED);
                                break;
                        case STOPPED:
                                do_stop();
                                set_status(STOPPED);
                                break;
                        case NEXT:
                                do_next();
                                set_status(STOPPED);
                                break;
                        case PREV:
                                do_prev();
                                set_status(STOPPED);
                                break;
                        default:
                                break;
                }
        }
}

