/****************************************************************************
** Gestion simplifié des listes de transferts ftp
** Créé par Anacr0x : anacr0x@free.fr
**
** Sous licence GPL v2 ou plus
 ****************************************************************************/

#include "myftptransfercontrol.h"

// ////////////////////////////////////////////////////////////////
//
// MyFtpTransferControl
//
// ////////////////////////////////////////////////////////////////

MyFtpTransferControl::MyFtpTransferControl (QObject *parent) : QFtp(parent)
{
	koVitesseDone = 0;
	autoLaunch = true;

	connect(this, SIGNAL(commandFinished(int, bool)),
	        SLOT(slotCommandFinished(int, bool)));

	connect(this, SIGNAL(dataTransferProgress(qint64, qint64)),
	        SLOT(slotDataTransferProgress(qint64, qint64)));

	connect(&tmpVitTimer, SIGNAL(timeout()), SLOT(slotTmpVit()));
}

bool MyFtpTransferControl::startTransfers()
{
	if (!autoLaunch || state() != QFtp::LoggedIn || transferInProgress() || transferList.isEmpty())
		return false;

	MyTransfer &t = transferList[0];
	if (t.isInProgress())
		return false;

	if (t.cmd == MyTransfer::Download)
	{
		// Ouvre le fichier en écriture et lance le téléchargement
		file.setFileName(t.destFilePath);
		if (file.open(QIODevice::WriteOnly))
		{
			t.setId(get
			        (t.origFilePath, &file));
			return true;
		}
	}
	else if (t.cmd == MyTransfer::Upload)
	{
		// Ouvre le fichier en lecture et lance l'envoi
		file.setFileName(t.origFilePath);
		if (file.open(QIODevice::ReadOnly))
		{
			t.setId(put(&file, t.destFilePath));
			return true;
		}
	}

	return false;
}

bool MyFtpTransferControl::startTransfer(const MyTransfer &transfer)
{
	if (state() != QFtp::LoggedIn)
		return false;

	addTransfer(transfer);

	if (!transferInProgress())
		startTransfers();

	return true;
}

bool MyFtpTransferControl::transferInProgress()
{
	for (int i = 0; i < transferList.size(); ++i)
		if (transferList[i].isInProgress())
			return true;
	return false;
}

void MyFtpTransferControl::addTransfer(const MyTransfer &transfer)
{
	if (transferList.indexOf(transfer) == -1)
	{
		transferList << transfer;
		emit transferListModified (transferList);
	}
}

bool MyFtpTransferControl::removeTransfer(int index)
{
	if (index > 0 && index < transferList.size())
	{
		transferList.removeAt(index);
		emit transferListModified (transferList);
		return true;
	}

	return false;
}

bool MyFtpTransferControl::removeTransfer(const MyTransfer &transfer)
{
	int index = transferList.indexOf(transfer);
	return removeTransfer(index);
}

void MyFtpTransferControl::removeAllTransfers()
{
	if (transferList.isEmpty())
		return;

	if (!transferInProgress())
		transferList.clear();
	else
	{
		MyTransfer t = transferList[0];
		transferList.clear();
		transferList << t;
	}

	emit transferListModified (transferList);
}

bool MyFtpTransferControl::moveTransfer(int from, int to)
{
	if (from <= 0 || to <= 0)
		return false;

	transferList.move(from, to);
	emit transferListModified (transferList);

	return true;
}

void MyFtpTransferControl::setAutoLaunch(bool AutoLaunch)
{
	autoLaunch = AutoLaunch;

	if (autoLaunch)
		startTransfers();
}

bool MyFtpTransferControl::upTransfer(const MyTransfer &transfer)
{
	int found = transferList.indexOf(transfer);
	if (found != -1)
		return moveTransfer(found, found-1);

	return false;
}

bool MyFtpTransferControl::downTransfer(const MyTransfer &transfer)
{
	int found = transferList.indexOf(transfer);
	if (found != transferList.count()-1)
		return moveTransfer(found, found+1);

	return false;
}

void MyFtpTransferControl::slotCommandFinished(int id, bool error)
{
	if (!transferList.isEmpty() && id == transferList[0].getId())
	{
		QString destPath = QFileInfo(transferList.first().destFilePath).path();

		if (transferList[0].cmd == MyTransfer::Download)
			emit downloadFinished(destPath, id, error);
		else
			emit uploadFinished(destPath, id, error);

		slotTimerDel();

		file.close();
		transferList.removeAt(0);
		if (autoLaunch)
			startTransfers();

		emit transferListModified (transferList);
	}
	else
		emit otherCommandFinished(id, error);
}

void MyFtpTransferControl::slotDataTransferProgress(qint64 done, qint64 total)
{
	// S'il s'agit d'un nouveau transfert, on lance le timer d'affichage
	if (doneOctet == 0)
		slotTimerInit();
	doneOctet = done;
	totalOctet = total;
}

void MyFtpTransferControl::slotTmpVit()
{
	progress = doneOctet / 1024;	// Met le nombre d'octets envoyés en Ko
	totalSize = totalOctet / 1024;	// met le nombre total d'octets en Ko
	vitesse = progress - koVitesseDone;	// Donne la vitesse
	if (vitesse != 0)
	{
		temps = (totalSize - progress) / vitesse;
		sec = temps % 60;
		min = temps / 60;
		emit transferRapidityAndRemainTime(vitesse, min, sec, ((float)doneOctet/totalOctet)*100);
	}

	koVitesseDone = progress;
}

void MyFtpTransferControl::slotTimerInit()
{
	doneOctet = 0;
	totalOctet = 0;
	koVitesseDone = 0;
	tmpVitTimer.start(1000);
}

void MyFtpTransferControl::slotTimerDel()
{
	doneOctet = 0;
	totalOctet = 0;
	koVitesseDone = 0;
	tmpVitTimer.stop();
}

