/****************************************************************************
** Gestion simplifié des listes de transferts ftp
** Créé par Anacr0x : anacr0x@free.fr
**
** Sous licence GPL v2 ou plus
 ****************************************************************************/

#ifndef MYFTP_TRANSFER_CONTROL_H
#define MYFTP_TRANSFER_CONTROL_H

#include <QtCore>
#include <QtGui>
#include <QtNetwork>

/*************************************************************************/
//! Sauvegarde les données d'un transfert ftp

/*! MyTransfer permet d'associer à chaque transfert Ftp un id, le nom de
la commande ainsi que les chemins des 2 fichiers.\n
\n
De plus, elle possède quelques fonctions qui facilite son utilisation. */
/*************************************************************************/

class MyTransfer
{
	/*!Stock l'identifiant qftp du transfert*/
	int id;

public:
	/*!Les types de transferts possibles : Download ou Upload */
	enum TransferCmd {Download, Upload};

	/*!Stock le type de transfert*/
	TransferCmd cmd;

	/*!Stock le nom du fichier à télécharger et le nom du fichier destinataire */
	QString origFilePath, destFilePath;

	/*!Enregistre les informations concernant un transfert\n
	OrigFilePath = chemin du fichier d'origine\n
	DestFilePath = chemin du fichier destinataire\n
	Cmd = précise s'l s'agit d'un download ou d'un upload*/
	MyTransfer(const QString &OrigFilePath, const QString &DestFilePath, TransferCmd Cmd)
	{
		origFilePath = OrigFilePath;
		destFilePath = DestFilePath;
		cmd = Cmd;
		id = -1;
	}

	/*!Permet de changer l'id du transfert */
	inline void setId(int Id)
	{
		id = Id;
	}

	/*!Permet de récupérer l'id du transfert */
	inline int getId() const
	{
		return id;
	}

	/*!Permet de savoir si le transfert est en cours */
	inline bool isInProgress() const
	{
		return (id != -1);
	}

	/*!Permet de comparer 2 transfert */
	bool operator== (const MyTransfer &t) const
	{
		return (cmd==t.cmd && id==t.id && origFilePath==t.origFilePath && destFilePath==t.destFilePath);
	}

	/*!Permet de récupérer le nom du fichier original */
	inline QString fileName() const
	{
		return QFileInfo(origFilePath).fileName();
	}

	/*!Permet de récupérer le type de transfert en toutes lettres : "Download" ou "Upload" */
	inline QString transferType() const
	{
		if (cmd == Download)
			return "Download";
		else
			return "Upload";
	}
};


/*************************************************************************/
//! Gère la liste des transferts ftp

/*! MyFtpTransferControl permet de gérer plusieurs transferts Ftp
facilement et proprement.\n
\n
Elle possèdent donc des fonctions d'ajout, de suppression et de
modification de transfert ainsi que différents signaux facilitant son
utilisation. */
/*************************************************************************/

class MyFtpTransferControl : public QFtp
{
	Q_OBJECT

	/*!La liste des transferts */
	QList<MyTransfer> transferList;

	/*!Utilisé par Transfer pour écrire ou lire dans les fichiers */
	QFile file;

	/*!Utilisé pour savoir si l'on doit lancer automatiquement le prochaine transfert */
	bool autoLaunch;

	/*!Stock le nombre d'octet déjà reçu lors d'un transfert. */
	int doneOctet;

	/*!Stock le nombre d'octet total à reçevoir lors d'un transfert. */
	int totalOctet;

	/*!progress == doneOctet / 1024. */
	int progress;

	/*!totalSize == totalOctet / 1024. */
	int totalSize;

	/*!Stock le nombre de Ko transférés en une seconde. */
	int vitesse;

	/*!Stock le temps restant pour finir le transfert en cours (en secondes). */
	int temps;

	/*!Stock uniquement les secondes sans les minutes (cad de 0 à 60). */
	int sec;

	/*!Stock le temps restant pour finir le transfert en cours (en minutes). */
	int min;

	/*!Timer utilisé pour afficher le temps restant et la vitesse du transfert en cours. */
	QTimer tmpVitTimer;

	/*!Permet de faire une moyenne de vitesse */
	int koVitesseDone;

public:
	/*!le constructeur : connecte les signaux utilisés */
	MyFtpTransferControl ( QObject * parent = 0 );

	/*!Lance le prochain transfert en attente */
	bool startTransfers();

	/*!Lance un transfert spécifique */
	bool startTransfer(const MyTransfer &transfer);

	/*!Retourne vrai s'il y a un transfert en cour */
	bool transferInProgress();

	/*!Permet de rajouter un transfert à la liste */
	void addTransfer(const MyTransfer &transfer);

	/*!Permet de suprimer un transfert d'index index */
	bool removeTransfer(int index);

	/*!Permet de suprimer un transfert */
	bool removeTransfer(const MyTransfer &transfer);

	/*!Permet de supprimer tous les transferts */
	void removeAllTransfers();

	/*!Permet de modifier l'ordre de lancement d'un transfert */
	bool moveTransfer(int from, int to);

	/*!Permet d'activer ou non le lancement automatique du prochain transfert */
	void setAutoLaunch(bool AutoLaunch);

	/*!Permet d'avancer un transfert */
	bool upTransfer(const MyTransfer &transfer);

	/*!Permet de reculer un transfert */
	bool downTransfer(const MyTransfer &transfer);

private slots:
	/*!Récupère les id des commandes qui se sont finis */
	void slotCommandFinished(int id, bool error);

	/*!Récupère la quantité d'octet déjà transféré */
	void slotDataTransferProgress(qint64 done, qint64 total);

	/*!Cette fonction affiche toutes les 100 millisecondes le temps restant et la vitesse de transfert d'un
	fichier/dossier */
	void slotTmpVit();

	/*!Cette fonction initialise le timer qui sert à slotTmpVit() */
	void slotTimerInit();

	/*!Cette fonction détruit le timer qui sert à slotTmpVit() */
	void slotTimerDel();

signals:
	/*!Signal envoyé lorsque un commande s'est terminée (autres que les transferts) */
	void otherCommandFinished(int id, bool error);

	/*!Signal envoyé lorsqu'un téléchargement se termine */
	void downloadFinished(const QString &destPath, int id, bool error);

	/*!Signal envoyé lorsqu'un envoi de fichier se termine */
	void uploadFinished(const QString &destPath, int id, bool error);

	/*!Signal envoyé lorsque la liste des envois/téléchargements de fichiers (en attente ou non) à été modifiée */
	void transferListModified (const QList<MyTransfer> &transferList);

	/*!Signal envoyé toutes les secondes pour permettre de connaitre la vitesse, le temps restant et le pourcentage
	du téléchargement */
	void transferRapidityAndRemainTime(int vitesseInKoSec, int remainMin, int remainSec, float pourcentage);
};

#endif
