#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>

#include "mgwtypes.h"
#include "textize.h"
#include "image.h"
#include "filters.h"
#include "interface.h"
#include "utils.h"

#define PS_VECTOR_PITCH 10

static void
ps_terminal_process (gint suspend, GdkPoint o, GdkPoint a, GdkPoint b, GdkPoint c, MgwSizeInfo si, FILE *out)
{
  if (suspend > PS_VECTOR_PITCH * 2) {
    fprintf (out, "%d %d M %d %d %d %d %d %d curveto S\n",
	     o.x, si.h - o.y - 1,
	     a.x, si.h - a.y - 1,
	     b.x, si.h - b.y - 1,
	     c.x, si.h - c.y - 1);
  } else if (suspend > PS_VECTOR_PITCH) {
    fprintf (out, "%d %d M %d %d %d %d %d %d curveto S\n",
	     o.x, si.h - o.y - 1,
	     a.x, si.h - a.y - 1,
	     a.x, si.h - a.y - 1,
	     c.x, si.h - c.y - 1);
  } else {
    fprintf (out, "%d %d M %d %d D S\n",
	     o.x, si.h - o.y - 1,
	     c.x, si.h - c.y - 1);
  }
}

static gboolean
ps_search_path (guchar *p0, gint i, gint j, MgwSizeInfo si, gboolean *checked, FILE *out)
{
  static gint suspend = 0;
  static GdkPoint o, a, b, c;

  if (i < 0 || si.h - 1 < i || j < 0 || si.w - 1 < j ||
      checked[i * si.w + j] || p0[i * si.rs + j * si.nc])
    return FALSE;
    
  checked[i * si.w + j] = TRUE; 

  if (suspend == 0) {
    o.y = i;
    o.x = j;
  } else if (suspend == PS_VECTOR_PITCH) {
    a.y = i;
    a.x = j;
  } else if (suspend == PS_VECTOR_PITCH * 2) {
    b.y = i;
    b.x = j;
  } else if (suspend == PS_VECTOR_PITCH * 3) {
    fprintf (out, "%d %d M %d %d %d %d %d %d curveto S\n",
	     o.x, si.h - o.y - 1,
	     a.x, si.h - a.y - 1,
	     b.x, si.h - b.y - 1,
	       j, si.h - i   - 1);
    o.y = i;
    o.x = j;
    suspend = 0;
  }

  suspend++;

  if (ps_search_path (p0, i - 1, j - 1, si, checked, out) ||
      ps_search_path (p0, i - 1, j    , si, checked, out) ||
      ps_search_path (p0, i - 1, j + 1, si, checked, out) ||
      ps_search_path (p0,     i, j - 1, si, checked, out) ||
      ps_search_path (p0,     i, j + 1, si, checked, out) ||
      ps_search_path (p0, i + 1, j - 1, si, checked, out) ||
      ps_search_path (p0, i + 1, j    , si, checked, out) ||
      ps_search_path (p0, i + 1, j + 1, si, checked, out)) {
    /* recursive call is suspendig...*/

  } else {
    c.y = i;
    c.x = j;
    ps_terminal_process (suspend, o, a, b, c, si, out);
    suspend = 0;
  }

  return TRUE;
}

static void
ps_out_vector (FILE *out, guchar *p0, MgwSizeInfo si)
{
  gint i, j;
  gint iii, jjj;
  gint *rseq = NULL;
  gboolean *checked = NULL;

  if (!(checked = (gboolean *) g_malloc0 (sizeof (gboolean) * si.h * si.w))) {
    g_warning ("ps_out_vector(textize.c): memory allocation failed(a)");
    goto FUNC_END;
  }

  if (!(rseq = (gint *) g_malloc (sizeof (gint) * si.h * si.w))) {
    g_warning ("ps_out_vector(textize.c): memory allocation failed(b)");
    goto FUNC_END;
  }

  for (i = 0; i < si.h * si.w; i++)
    rseq[i] = i;
  for (i = 0; i < si.h * si.w; i++) {
    j = si.h * si.w * ((gdouble) random () / RAND_MAX);
    swap (rseq + i, rseq + j);
  }

  fprintf (out, "0 G\n");
  fprintf (out, "2 W\n");

  for (i = 0; i < si.h; i++) {
    for (j = 0; j < si.w; j++) {
	iii = rseq[i * si.w + j] / si.w;
	jjj = rseq[i * si.w + j] % si.w;

	ps_search_path (p0, iii, jjj, si, checked, out);
    }
  }

 FUNC_END:
  g_free (rseq);
  g_free (checked);
}

#define PS_POLYLENG 20
static void
ps_out_polygon (FILE *out, guchar *p0, MgwSizeInfo si)
{
  gint i, j, iii, jjj;
  gint x1, x2, x3, x4, y1, y2, y3, y4;

  for (i = PS_POLYLENG / 2; i < si.h; i += PS_POLYLENG) {
    for (j = PS_POLYLENG / 2; j < si.w; j += PS_POLYLENG) {
      fprintf (out, "%.3f %.3f %.3f C",
	       3 * p0[i * si.rs + j * si.nc    ] / 255.,
	       3 * p0[i * si.rs + j * si.nc + 1] / 255.,
	       3 * p0[i * si.rs + j * si.nc + 2] / 255.);
      x1 = random () / (RAND_MAX / 5);
      x2 = random () / (RAND_MAX / 5);
      x3 = random () / (RAND_MAX / 5);
      x4 = random () / (RAND_MAX / 5);
      y1 = random () / (RAND_MAX / 5);
      y2 = random () / (RAND_MAX / 5);
      y3 = random () / (RAND_MAX / 5);
      y4 = random () / (RAND_MAX / 5);
      fprintf (out, " %d %d M %d %d D %d %d D %d %d D fill S\n",
	       j - PS_POLYLENG / 2 - y1, si.h - 1 - i - PS_POLYLENG / 2 - x1,
	       j + PS_POLYLENG / 2 + y2, si.h - 1 - i - PS_POLYLENG / 2 - x2,
	       j + PS_POLYLENG / 2 + y3, si.h - 1 - i + PS_POLYLENG / 2 + x3,
	       j - PS_POLYLENG / 2 - y4, si.h - 1 - i + PS_POLYLENG / 2 + x4);
    }
  }
}

static void
image_to_ps (gchar *fname)
{
  MgwSizeInfo si;
  MgwClipRect clip;
  GdkPixbuf *pbuf, *org, *backup;
  FILE *out;
  guchar *p0, *q0;
  gboolean has_clip;

  org = get_pbuf ();
  backup = gdk_pixbuf_copy (org);
  q0 = gdk_pixbuf_get_pixels (backup);

  filter_exec (NULL, "Outline(for PostScriptize)");

  if (has_clip = get_clip_rect_area (&clip)) {
    pbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, FALSE, 8, clip.x1, clip.y1);
    gdk_pixbuf_copy_area (org, clip.x0, clip.y0, clip.x1, clip.y1, pbuf, 0, 0);
  } else {
    pbuf = org;
  }
  p0 = gdk_pixbuf_get_pixels (pbuf);
  get_size_info (&si, pbuf);

  /* ---[ PS START ]---------------------------------------------------------*/
  out = fopen (fname, "w");

  /* HEADER */
  fprintf (out, "%!PS-Adobe-\n%%%%Title: %s\n%%%%Creator: Id: MGW\n%%%%CreationDate:\n%%%%BoundingBox: 0 0 %d %d\n%%%%Orientation: Portrait\n%%%%EndComments\n\n/G {setgray} def\n/C {setrgbcolor} def\n/D {lineto} def\n/R {rlineto} def\n/M {moveto} def\n/S {stroke} def\n/W {setlinewidth} def\n/Z {stroke newpath} def\n\n", get_current_filename (), si.w, si.h);

  /* BODY */
  //ps_out_polygon (out, q0, si);
  ps_out_vector (out, p0, si);

  /* FOOTER */
  fprintf (out, "showpage\n");

  fclose (out);
  /* -----------------------------------------------------------[ PS END ]---*/

  put_string_to_appbar ("Vectorize: DONE");
  if (has_clip)
    g_object_unref (pbuf);
  update_pbuf (backup);
}

static void
image_to_html (gchar *fname)
{
  MgwSizeInfo si;
  MgwClipRect clip;
  GdkPixbuf *pbuf, *org;
  FILE *out;
  gdouble f;
  gint i, j, shift, pixel, new_width, new_height;
  gchar symbol[] = {'@', '*', '%', 'O', '+', '-', '.', ' '};
  guchar *p, *p0;

  put_string_to_appbar ("HTMLize: EXECUTING...");

  if (!(org = get_pbuf ()))
    return;
  if (get_clip_rect_area (&clip)) {
    pbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, FALSE, 8, clip.x1, clip.y1);
    gdk_pixbuf_copy_area (org, clip.x0, clip.y0, clip.x1, clip.y1, pbuf, 0, 0);
  } else
    pbuf = org;

  p0 = gdk_pixbuf_get_pixels (pbuf);
  get_size_info (&si, pbuf);

  out = fopen (fname, "w");

  new_width = TEXTIZE_WIDTH;
  new_height = si.h * ((gdouble) new_width / si.w);

  f = (gdouble) si.h / new_height;

  /* HEADER */
  fprintf (out, "<html><head><title>%s(html)</title></head><BODY BGCOLOR=\"#FFFFFF\"><body><pre>\n", get_current_filename ());

  /* BODY */
  for (i = 0; i < new_height; i+= 2) {
    for (j = 0; j < new_width; j++) {
      p = p0 + (gint) (f * i) * si.rs + (gint) (f * j) * si.nc;
      pixel = (p[0] + p[1] + p[2]) / 3;
      fprintf (out, "<font color=\"#%02x%02x%02x\">%c</font>",
	       p[0], p[1], p[2], symbol[pixel / 32]);
    }
    fprintf (out, "\n");
  }

  /* FOOTER */
  fprintf (out, "</pre></body></html>\n");

  fclose (out);

  put_string_to_appbar ("HTMLize: DONE");
}

static void
save_ps_file (GtkWidget *widget, gpointer data)
{
  gchar *filename = NULL;
  GtkWidget *fselect = GTK_WIDGET (data);

  if (widget == (GTK_FILE_SELECTION (fselect)->ok_button)) {
    filename = (gchar *) gtk_file_selection_get_filename (GTK_FILE_SELECTION (fselect));
  }
  gtk_widget_destroy (fselect);

  if (filename)
    image_to_ps (filename);
}

static void
save_html_file (GtkWidget *widget, gpointer data)
{
  gchar *filename;
  GtkWidget *fselect = GTK_WIDGET (data);
  GdkCursor *watch;

  if (widget == (GTK_FILE_SELECTION (fselect)->ok_button)) {
    filename = (gchar *) gtk_file_selection_get_filename (GTK_FILE_SELECTION (fselect));
    watch = gdk_cursor_new (GDK_WATCH);
    gdk_window_set_cursor (fselect->window, watch);
    gdk_flush ();
    image_to_html (filename);
    gdk_cursor_destroy (watch);
  }
  gtk_widget_destroy (fselect);
}

static void
textize_image (GtkTextBuffer *text_buffer, GdkPixbuf *pbuf, MgwSizeInfo si)
{
  gdouble f;
  gint i, j, count = 0;
  gint pixel, width, height, new_width, new_height;
  gchar buffer[TEXTIZE_WIDTH * TEXTIZE_WIDTH * 10];
  gchar symbol[] = {'@', '*', '%', 'O', '+', '-', '.', ' '};
  guchar *p, *p0 = gdk_pixbuf_get_pixels (pbuf);

  new_width = TEXTIZE_WIDTH;
  new_height = si.h * ((gdouble) new_width / si.w);

  f = (gdouble) si.h / new_height;

  for (i = 0; i < new_height; i+= 2) {
    for (j = 0; j < new_width; j++) {
      p = p0 + (gint) (f * i) * si.rs + (gint) (f * j) * si.nc;
      pixel = (p[0] + p[1] + p[2]) / 3;
      buffer[count++] = symbol[pixel / 32];
    }
    buffer[count++] = '\n';
  }

  gtk_text_buffer_set_text (text_buffer, buffer, count);
}

/* Entry point */
void
save_another_type (GtkWidget *w, gchar *type)
{
  MgwSizeInfo si;
  MgwClipRect clip;
  gchar tmp[1024];

  sprintf (tmp, "SAVING %s FILE", type);

  GtkWidget *fselect = gtk_file_selection_new (tmp);

  if (!strcmp (type, "html")) {
    gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (fselect)->ok_button),
			"clicked",
			GTK_SIGNAL_FUNC (save_html_file), fselect);
    gtk_signal_connect (GTK_OBJECT
			(GTK_FILE_SELECTION (fselect)->cancel_button),
			"clicked",
			GTK_SIGNAL_FUNC (save_html_file), fselect);
  } else if (!strcmp (type, "ps")) {
    gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (fselect)->ok_button),
			"clicked",
			GTK_SIGNAL_FUNC (save_ps_file), fselect);
    gtk_signal_connect (GTK_OBJECT
			(GTK_FILE_SELECTION (fselect)->cancel_button),
			"clicked",
			GTK_SIGNAL_FUNC (save_ps_file), fselect);
  } else
    g_warning (_("save_another_type(textize.c): internal error(type=%s)"),
	       type);
  gtk_signal_connect (GTK_OBJECT (fselect), "destroy",
		      GTK_SIGNAL_FUNC (gtk_grab_remove), fselect);
  gtk_file_selection_hide_fileop_buttons (GTK_FILE_SELECTION (fselect));

  sprintf (tmp, "%s.%s", get_current_filename (), type);
  gtk_file_selection_set_filename (GTK_FILE_SELECTION (fselect), tmp);

  gtk_window_set_position (GTK_WINDOW (fselect), GTK_WIN_POS_CENTER);
  gtk_widget_show (fselect);
  gtk_grab_add (fselect);
}

void
on_textize_activate (void)
{
  MgwSizeInfo si;
  MgwClipRect clip;
  GdkPixbuf *pbuf, *org;
  GtkWidget *twin, *scrolledwindow, *view;
  GtkTextBuffer *buffer;
  PangoFontDescription *font_desc;
  gint width, height;

  if (!(org = get_pbuf ()))
    return;

  set_cursor (GDK_WATCH);

  if (get_clip_rect_area (&clip)) {
    pbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, FALSE, 8, clip.x1, clip.y1);
    gdk_pixbuf_copy_area (org, clip.x0, clip.y0, clip.x1, clip.y1, pbuf, 0, 0);
  } else
    pbuf = org;

  get_size_info (&si, pbuf);

  twin = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (twin), get_current_filename ());
  scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
  gtk_container_add (GTK_CONTAINER (twin), scrolledwindow);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow),
				  GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
  view = gtk_text_view_new ();
  gtk_container_add (GTK_CONTAINER (scrolledwindow), view);

  buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (view));
  font_desc = pango_font_description_from_string ("Courier 10");
  gtk_widget_modify_font (view, font_desc);
  pango_font_description_free (font_desc);

  width = 8 * (TEXTIZE_WIDTH + 1);
  height = width * si.h / si.w;
  if (900 < height)
    height = 900;
  gtk_widget_set_usize (view, width, height);

  textize_image (buffer, pbuf, si);

  gtk_widget_show_all (twin);
  set_cursor (GDK_TOP_LEFT_ARROW);
}
