
#line 1 "./scanner.rl"
/*  Copyright (C) 2011 CZ.NIC, z.s.p.o. <knot-dns@labs.nic.cz>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <config.h>
#include <stdint.h>			// uint32_t
#include <stdlib.h>			// calloc
#include <stdio.h>			// sprintf
#include <libgen.h>			// dirname
#include <stdbool.h>			// bool
#include <math.h>			// pow
#include <string.h>			// strdup
#include <sys/types.h>			// (OpenBSD)
#include <sys/socket.h>			// AF_INET (BSD)
#include <netinet/in.h>			// in_addr (BSD)
#include <arpa/inet.h>			// inet_pton

#include "zscanner/scanner.h"
#include "zscanner/error.h"		// error codes
#include "zscanner/file_loader.h"	// file_loader
#include "zscanner/scanner_functions.h"	// Base64
#include "zscanner/descriptor.h"	// KNOT_RRTYPE_A

/*! \brief Shorthand for setting warning data. */
#define WARN(code) { s->error_code = code; }
/*! \brief Shorthand for setting error data. */
#define ERR(code)   { s->error_code = code; s->stop = true; }

/*!
 * \brief Empty function which is called if no callback function is specified.
 */
static inline void noop(const scanner_t *s)
{
	(void)s;
}

/*!
 * \brief Writes record type number to r_data.
 *
 * \param type		Type number.
 * \param rdata_tail	Position where to write type number to.
 */
static inline void type_num(const uint16_t type, uint8_t **rdata_tail)
{
	*((uint16_t *)*rdata_tail) = htons(type);
	*rdata_tail += 2;
}

/*!
 * \brief Sets bit to bitmap window.
 *
 * \param type		Type number.
 * \param s		Scanner context.
 */
static inline void window_add_bit(const uint16_t type, scanner_t *s) {
	uint8_t win      = type / 256;
	uint8_t bit_pos  = type % 256;
	uint8_t byte_pos = bit_pos / 8;

	((s->windows[win]).bitmap)[byte_pos] |= 128 >> (bit_pos % 8);

	if ((s->windows[win]).length < byte_pos + 1) {
		(s->windows[win]).length = byte_pos + 1;
	}

	if (s->last_window < win) {
		s->last_window = win;
	}
}

// Include scanner file (in Ragel).

#line 88 "scanner.c"
static const int zone_scanner_start = 1056;
static const int zone_scanner_first_final = 1056;
static const int zone_scanner_error = 0;

static const int zone_scanner_en_err_line = 246;
static const int zone_scanner_en_dname_ = 248;
static const int zone_scanner_en_text_ = 257;
static const int zone_scanner_en_default_ttl_ = 269;
static const int zone_scanner_en_zone_origin_ = 278;
static const int zone_scanner_en_include_file_ = 290;
static const int zone_scanner_en_base64_ = 307;
static const int zone_scanner_en_bitmap_ = 314;
static const int zone_scanner_en_nonempty_hex_r_data = 434;
static const int zone_scanner_en_hex_r_data = 443;
static const int zone_scanner_en_dns_alg_ = 453;
static const int zone_scanner_en_cert_type_ = 548;
static const int zone_scanner_en_r_data_a = 589;
static const int zone_scanner_en_r_data_ns = 591;
static const int zone_scanner_en_r_data_soa = 593;
static const int zone_scanner_en_r_data_hinfo = 625;
static const int zone_scanner_en_r_data_minfo = 630;
static const int zone_scanner_en_r_data_mx = 635;
static const int zone_scanner_en_r_data_txt = 640;
static const int zone_scanner_en_r_data_aaaa = 644;
static const int zone_scanner_en_r_data_loc = 646;
static const int zone_scanner_en_r_data_srv = 701;
static const int zone_scanner_en_r_data_naptr = 712;
static const int zone_scanner_en_r_data_cert = 729;
static const int zone_scanner_en_r_data_apl = 740;
static const int zone_scanner_en_r_data_ds = 751;
static const int zone_scanner_en_r_data_sshfp = 764;
static const int zone_scanner_en_r_data_ipseckey = 774;
static const int zone_scanner_en_r_data_rrsig = 813;
static const int zone_scanner_en_r_data_nsec = 955;
static const int zone_scanner_en_r_data_dnskey = 958;
static const int zone_scanner_en_r_data_dhcid = 969;
static const int zone_scanner_en_r_data_nsec3 = 971;
static const int zone_scanner_en_r_data_nsec3param = 1000;
static const int zone_scanner_en_r_data_tlsa = 1013;
static const int zone_scanner_en_r_data_l32 = 1026;
static const int zone_scanner_en_r_data_l64 = 1031;
static const int zone_scanner_en_r_data_eui48 = 1044;
static const int zone_scanner_en_r_data_eui64 = 1050;
static const int zone_scanner_en_main = 1056;


#line 90 "./scanner.rl"


scanner_t* scanner_create(const char     *file_name,
                          const char     *origin,
                          const uint16_t rclass,
                          const uint32_t ttl,
                          void (*process_record)(const scanner_t *),
                          void (*process_error)(const scanner_t *),
                          void *data)
{
	char settings[1024];

	scanner_t *s = calloc(1, sizeof(scanner_t));
	if (s == NULL) {
		return NULL;
	}

	if (file_name != NULL) {
		// Get absolute path of the zone file.
		if (realpath(file_name, (char*)(s->buffer)) != NULL) {
			char *full_name = strdup((char*)(s->buffer));
			s->path = strdup(dirname(full_name));
			free(full_name);
		} else {
			free(s);
			return NULL;
		}

		s->file_name = strdup(file_name);
	} else {
		s->path = strdup(".");
		s->file_name = strdup("<NULL>");
	}

	// Nonzero initial scanner state.
	s->cs = zone_scanner_start;

	// Disable processing during parsing of settings.
	s->process_record = &noop;
	s->process_error = &noop;

	// Create ORIGIN directive and parse it using scanner to set up origin.
	const char *format;
	size_t origin_len = strlen(origin);
	if (origin_len > 0 && origin[origin_len - 1] != '.') {
		format = "$ORIGIN %s.\n";
	} else {
		format = "$ORIGIN %s\n";
	}
	int ret = snprintf(settings, sizeof(settings), format, origin);
	if (ret <= 0 || (size_t)ret >= sizeof(settings) ||
	    scanner_process(settings, settings + ret, true, s) != 0) {
		scanner_free(s);
		return NULL;
	}

	// Set scanner defaults.
	s->default_class = rclass;
	s->default_ttl = ttl;
	s->process_record = process_record ? process_record : &noop;
	s->process_error = process_error ? process_error : &noop;
	s->data = data;
	s->line_counter = 1;

	return s;
}

void scanner_free(scanner_t *s)
{
	if (s != NULL) {
		free(s->file_name);
		free(s->path);
		free(s);
	}
}

int scanner_process(const char *start,
                    const char *end,
                    const bool is_complete,
                    scanner_t  *s)
{
	// Necessary scanner variables.
	const char *p = start;
	const char *pe = end;
	char       *eof = NULL;
	int        stack[RAGEL_STACK_SIZE];

	// Auxiliary variables which are used in scanner body.
	struct in_addr  addr4;
	struct in6_addr addr6;
	uint32_t timestamp;
	int16_t  window;
	int      ret;

	// Next 2 variables are for better performance.
	// Restoring r_data pointer to next free space.
	uint8_t *rdata_tail = s->r_data + s->r_data_tail;
	// Initialization of the last r_data byte.
	uint8_t *rdata_stop = s->r_data + MAX_RDATA_LENGTH - 1;

	// Restoring scanner states.
	int cs  = s->cs;
	int top = s->top;
	memcpy(stack, s->stack, sizeof(stack));

	// End of file check.
	if (is_complete == true) {
		eof = (char *)pe;
	}

	// Writing scanner body (in C).
	
#line 248 "scanner.c"
	{
	short _widec;
	if ( p == pe )
		goto _test_eof;
	goto _resume;

_again:
	switch ( cs ) {
		case 1056: goto st1056;
		case 0: goto st0;
		case 1: goto st1;
		case 2: goto st2;
		case 3: goto st3;
		case 4: goto st4;
		case 5: goto st5;
		case 6: goto st6;
		case 7: goto st7;
		case 8: goto st8;
		case 9: goto st9;
		case 10: goto st10;
		case 11: goto st11;
		case 12: goto st12;
		case 1057: goto st1057;
		case 13: goto st13;
		case 14: goto st14;
		case 15: goto st15;
		case 16: goto st16;
		case 17: goto st17;
		case 18: goto st18;
		case 19: goto st19;
		case 20: goto st20;
		case 21: goto st21;
		case 22: goto st22;
		case 23: goto st23;
		case 24: goto st24;
		case 25: goto st25;
		case 26: goto st26;
		case 27: goto st27;
		case 28: goto st28;
		case 29: goto st29;
		case 30: goto st30;
		case 31: goto st31;
		case 32: goto st32;
		case 33: goto st33;
		case 34: goto st34;
		case 35: goto st35;
		case 36: goto st36;
		case 37: goto st37;
		case 38: goto st38;
		case 39: goto st39;
		case 40: goto st40;
		case 41: goto st41;
		case 42: goto st42;
		case 43: goto st43;
		case 44: goto st44;
		case 45: goto st45;
		case 46: goto st46;
		case 47: goto st47;
		case 48: goto st48;
		case 49: goto st49;
		case 50: goto st50;
		case 51: goto st51;
		case 52: goto st52;
		case 53: goto st53;
		case 54: goto st54;
		case 55: goto st55;
		case 56: goto st56;
		case 57: goto st57;
		case 58: goto st58;
		case 59: goto st59;
		case 60: goto st60;
		case 61: goto st61;
		case 62: goto st62;
		case 63: goto st63;
		case 64: goto st64;
		case 65: goto st65;
		case 66: goto st66;
		case 67: goto st67;
		case 68: goto st68;
		case 69: goto st69;
		case 70: goto st70;
		case 71: goto st71;
		case 72: goto st72;
		case 73: goto st73;
		case 74: goto st74;
		case 75: goto st75;
		case 76: goto st76;
		case 77: goto st77;
		case 78: goto st78;
		case 79: goto st79;
		case 80: goto st80;
		case 81: goto st81;
		case 82: goto st82;
		case 83: goto st83;
		case 84: goto st84;
		case 85: goto st85;
		case 86: goto st86;
		case 87: goto st87;
		case 88: goto st88;
		case 89: goto st89;
		case 90: goto st90;
		case 91: goto st91;
		case 92: goto st92;
		case 93: goto st93;
		case 94: goto st94;
		case 95: goto st95;
		case 96: goto st96;
		case 97: goto st97;
		case 98: goto st98;
		case 99: goto st99;
		case 100: goto st100;
		case 101: goto st101;
		case 102: goto st102;
		case 103: goto st103;
		case 104: goto st104;
		case 105: goto st105;
		case 106: goto st106;
		case 107: goto st107;
		case 108: goto st108;
		case 109: goto st109;
		case 110: goto st110;
		case 111: goto st111;
		case 112: goto st112;
		case 113: goto st113;
		case 114: goto st114;
		case 115: goto st115;
		case 116: goto st116;
		case 117: goto st117;
		case 118: goto st118;
		case 119: goto st119;
		case 120: goto st120;
		case 121: goto st121;
		case 122: goto st122;
		case 123: goto st123;
		case 124: goto st124;
		case 125: goto st125;
		case 126: goto st126;
		case 127: goto st127;
		case 1058: goto st1058;
		case 128: goto st128;
		case 129: goto st129;
		case 130: goto st130;
		case 131: goto st131;
		case 132: goto st132;
		case 133: goto st133;
		case 134: goto st134;
		case 135: goto st135;
		case 136: goto st136;
		case 137: goto st137;
		case 1059: goto st1059;
		case 138: goto st138;
		case 139: goto st139;
		case 140: goto st140;
		case 141: goto st141;
		case 142: goto st142;
		case 143: goto st143;
		case 144: goto st144;
		case 145: goto st145;
		case 1060: goto st1060;
		case 146: goto st146;
		case 147: goto st147;
		case 148: goto st148;
		case 1061: goto st1061;
		case 149: goto st149;
		case 150: goto st150;
		case 151: goto st151;
		case 152: goto st152;
		case 153: goto st153;
		case 154: goto st154;
		case 155: goto st155;
		case 156: goto st156;
		case 157: goto st157;
		case 158: goto st158;
		case 159: goto st159;
		case 1062: goto st1062;
		case 160: goto st160;
		case 161: goto st161;
		case 162: goto st162;
		case 163: goto st163;
		case 1063: goto st1063;
		case 164: goto st164;
		case 165: goto st165;
		case 166: goto st166;
		case 167: goto st167;
		case 168: goto st168;
		case 169: goto st169;
		case 170: goto st170;
		case 171: goto st171;
		case 172: goto st172;
		case 173: goto st173;
		case 174: goto st174;
		case 175: goto st175;
		case 176: goto st176;
		case 177: goto st177;
		case 178: goto st178;
		case 1064: goto st1064;
		case 179: goto st179;
		case 180: goto st180;
		case 181: goto st181;
		case 182: goto st182;
		case 183: goto st183;
		case 184: goto st184;
		case 185: goto st185;
		case 186: goto st186;
		case 187: goto st187;
		case 188: goto st188;
		case 189: goto st189;
		case 190: goto st190;
		case 191: goto st191;
		case 192: goto st192;
		case 193: goto st193;
		case 194: goto st194;
		case 1065: goto st1065;
		case 195: goto st195;
		case 196: goto st196;
		case 197: goto st197;
		case 198: goto st198;
		case 199: goto st199;
		case 200: goto st200;
		case 201: goto st201;
		case 202: goto st202;
		case 203: goto st203;
		case 204: goto st204;
		case 205: goto st205;
		case 206: goto st206;
		case 207: goto st207;
		case 208: goto st208;
		case 209: goto st209;
		case 210: goto st210;
		case 211: goto st211;
		case 212: goto st212;
		case 213: goto st213;
		case 214: goto st214;
		case 215: goto st215;
		case 216: goto st216;
		case 217: goto st217;
		case 218: goto st218;
		case 219: goto st219;
		case 220: goto st220;
		case 221: goto st221;
		case 222: goto st222;
		case 223: goto st223;
		case 224: goto st224;
		case 225: goto st225;
		case 226: goto st226;
		case 227: goto st227;
		case 228: goto st228;
		case 229: goto st229;
		case 230: goto st230;
		case 231: goto st231;
		case 232: goto st232;
		case 233: goto st233;
		case 234: goto st234;
		case 235: goto st235;
		case 236: goto st236;
		case 237: goto st237;
		case 238: goto st238;
		case 239: goto st239;
		case 240: goto st240;
		case 241: goto st241;
		case 242: goto st242;
		case 243: goto st243;
		case 244: goto st244;
		case 245: goto st245;
		case 246: goto st246;
		case 247: goto st247;
		case 1066: goto st1066;
		case 248: goto st248;
		case 249: goto st249;
		case 1067: goto st1067;
		case 250: goto st250;
		case 251: goto st251;
		case 252: goto st252;
		case 253: goto st253;
		case 254: goto st254;
		case 255: goto st255;
		case 256: goto st256;
		case 257: goto st257;
		case 258: goto st258;
		case 1068: goto st1068;
		case 1069: goto st1069;
		case 259: goto st259;
		case 260: goto st260;
		case 261: goto st261;
		case 262: goto st262;
		case 263: goto st263;
		case 264: goto st264;
		case 265: goto st265;
		case 266: goto st266;
		case 267: goto st267;
		case 268: goto st268;
		case 269: goto st269;
		case 270: goto st270;
		case 271: goto st271;
		case 272: goto st272;
		case 1070: goto st1070;
		case 273: goto st273;
		case 274: goto st274;
		case 275: goto st275;
		case 276: goto st276;
		case 277: goto st277;
		case 278: goto st278;
		case 279: goto st279;
		case 280: goto st280;
		case 281: goto st281;
		case 282: goto st282;
		case 1071: goto st1071;
		case 283: goto st283;
		case 284: goto st284;
		case 285: goto st285;
		case 286: goto st286;
		case 287: goto st287;
		case 288: goto st288;
		case 289: goto st289;
		case 290: goto st290;
		case 291: goto st291;
		case 292: goto st292;
		case 293: goto st293;
		case 294: goto st294;
		case 295: goto st295;
		case 296: goto st296;
		case 1072: goto st1072;
		case 297: goto st297;
		case 298: goto st298;
		case 299: goto st299;
		case 300: goto st300;
		case 301: goto st301;
		case 302: goto st302;
		case 303: goto st303;
		case 1073: goto st1073;
		case 304: goto st304;
		case 305: goto st305;
		case 306: goto st306;
		case 307: goto st307;
		case 308: goto st308;
		case 309: goto st309;
		case 310: goto st310;
		case 311: goto st311;
		case 312: goto st312;
		case 1074: goto st1074;
		case 1075: goto st1075;
		case 1076: goto st1076;
		case 313: goto st313;
		case 314: goto st314;
		case 315: goto st315;
		case 316: goto st316;
		case 317: goto st317;
		case 318: goto st318;
		case 319: goto st319;
		case 320: goto st320;
		case 1077: goto st1077;
		case 1078: goto st1078;
		case 321: goto st321;
		case 322: goto st322;
		case 323: goto st323;
		case 324: goto st324;
		case 1079: goto st1079;
		case 325: goto st325;
		case 326: goto st326;
		case 327: goto st327;
		case 328: goto st328;
		case 329: goto st329;
		case 330: goto st330;
		case 331: goto st331;
		case 332: goto st332;
		case 333: goto st333;
		case 334: goto st334;
		case 335: goto st335;
		case 336: goto st336;
		case 337: goto st337;
		case 338: goto st338;
		case 339: goto st339;
		case 340: goto st340;
		case 341: goto st341;
		case 342: goto st342;
		case 343: goto st343;
		case 344: goto st344;
		case 345: goto st345;
		case 346: goto st346;
		case 347: goto st347;
		case 348: goto st348;
		case 349: goto st349;
		case 350: goto st350;
		case 351: goto st351;
		case 352: goto st352;
		case 353: goto st353;
		case 354: goto st354;
		case 355: goto st355;
		case 356: goto st356;
		case 357: goto st357;
		case 358: goto st358;
		case 359: goto st359;
		case 360: goto st360;
		case 361: goto st361;
		case 362: goto st362;
		case 363: goto st363;
		case 364: goto st364;
		case 365: goto st365;
		case 366: goto st366;
		case 367: goto st367;
		case 368: goto st368;
		case 369: goto st369;
		case 370: goto st370;
		case 371: goto st371;
		case 372: goto st372;
		case 373: goto st373;
		case 374: goto st374;
		case 375: goto st375;
		case 376: goto st376;
		case 377: goto st377;
		case 378: goto st378;
		case 379: goto st379;
		case 380: goto st380;
		case 381: goto st381;
		case 382: goto st382;
		case 383: goto st383;
		case 384: goto st384;
		case 385: goto st385;
		case 386: goto st386;
		case 387: goto st387;
		case 388: goto st388;
		case 389: goto st389;
		case 390: goto st390;
		case 391: goto st391;
		case 392: goto st392;
		case 393: goto st393;
		case 394: goto st394;
		case 395: goto st395;
		case 396: goto st396;
		case 397: goto st397;
		case 398: goto st398;
		case 399: goto st399;
		case 400: goto st400;
		case 401: goto st401;
		case 402: goto st402;
		case 403: goto st403;
		case 404: goto st404;
		case 405: goto st405;
		case 406: goto st406;
		case 407: goto st407;
		case 408: goto st408;
		case 409: goto st409;
		case 410: goto st410;
		case 411: goto st411;
		case 412: goto st412;
		case 413: goto st413;
		case 414: goto st414;
		case 415: goto st415;
		case 416: goto st416;
		case 417: goto st417;
		case 418: goto st418;
		case 419: goto st419;
		case 420: goto st420;
		case 421: goto st421;
		case 422: goto st422;
		case 423: goto st423;
		case 424: goto st424;
		case 425: goto st425;
		case 426: goto st426;
		case 427: goto st427;
		case 428: goto st428;
		case 429: goto st429;
		case 430: goto st430;
		case 431: goto st431;
		case 432: goto st432;
		case 433: goto st433;
		case 434: goto st434;
		case 435: goto st435;
		case 436: goto st436;
		case 437: goto st437;
		case 438: goto st438;
		case 439: goto st439;
		case 440: goto st440;
		case 1080: goto st1080;
		case 1081: goto st1081;
		case 1082: goto st1082;
		case 441: goto st441;
		case 442: goto st442;
		case 443: goto st443;
		case 444: goto st444;
		case 445: goto st445;
		case 1083: goto st1083;
		case 446: goto st446;
		case 447: goto st447;
		case 448: goto st448;
		case 449: goto st449;
		case 1084: goto st1084;
		case 1085: goto st1085;
		case 1086: goto st1086;
		case 450: goto st450;
		case 451: goto st451;
		case 1087: goto st1087;
		case 452: goto st452;
		case 453: goto st453;
		case 454: goto st454;
		case 1088: goto st1088;
		case 455: goto st455;
		case 456: goto st456;
		case 457: goto st457;
		case 458: goto st458;
		case 459: goto st459;
		case 460: goto st460;
		case 461: goto st461;
		case 462: goto st462;
		case 463: goto st463;
		case 464: goto st464;
		case 465: goto st465;
		case 466: goto st466;
		case 467: goto st467;
		case 468: goto st468;
		case 469: goto st469;
		case 470: goto st470;
		case 471: goto st471;
		case 472: goto st472;
		case 473: goto st473;
		case 474: goto st474;
		case 475: goto st475;
		case 476: goto st476;
		case 477: goto st477;
		case 478: goto st478;
		case 479: goto st479;
		case 480: goto st480;
		case 481: goto st481;
		case 482: goto st482;
		case 483: goto st483;
		case 484: goto st484;
		case 485: goto st485;
		case 486: goto st486;
		case 487: goto st487;
		case 488: goto st488;
		case 489: goto st489;
		case 490: goto st490;
		case 491: goto st491;
		case 492: goto st492;
		case 493: goto st493;
		case 494: goto st494;
		case 495: goto st495;
		case 496: goto st496;
		case 497: goto st497;
		case 498: goto st498;
		case 499: goto st499;
		case 500: goto st500;
		case 501: goto st501;
		case 502: goto st502;
		case 503: goto st503;
		case 504: goto st504;
		case 505: goto st505;
		case 506: goto st506;
		case 507: goto st507;
		case 508: goto st508;
		case 509: goto st509;
		case 510: goto st510;
		case 511: goto st511;
		case 512: goto st512;
		case 513: goto st513;
		case 514: goto st514;
		case 515: goto st515;
		case 516: goto st516;
		case 517: goto st517;
		case 518: goto st518;
		case 519: goto st519;
		case 520: goto st520;
		case 521: goto st521;
		case 522: goto st522;
		case 523: goto st523;
		case 524: goto st524;
		case 525: goto st525;
		case 526: goto st526;
		case 527: goto st527;
		case 528: goto st528;
		case 529: goto st529;
		case 530: goto st530;
		case 531: goto st531;
		case 532: goto st532;
		case 533: goto st533;
		case 534: goto st534;
		case 535: goto st535;
		case 536: goto st536;
		case 537: goto st537;
		case 538: goto st538;
		case 539: goto st539;
		case 540: goto st540;
		case 541: goto st541;
		case 542: goto st542;
		case 543: goto st543;
		case 544: goto st544;
		case 545: goto st545;
		case 546: goto st546;
		case 547: goto st547;
		case 548: goto st548;
		case 549: goto st549;
		case 1089: goto st1089;
		case 550: goto st550;
		case 551: goto st551;
		case 552: goto st552;
		case 553: goto st553;
		case 554: goto st554;
		case 555: goto st555;
		case 556: goto st556;
		case 557: goto st557;
		case 558: goto st558;
		case 559: goto st559;
		case 560: goto st560;
		case 561: goto st561;
		case 562: goto st562;
		case 563: goto st563;
		case 564: goto st564;
		case 565: goto st565;
		case 566: goto st566;
		case 567: goto st567;
		case 568: goto st568;
		case 569: goto st569;
		case 570: goto st570;
		case 571: goto st571;
		case 572: goto st572;
		case 573: goto st573;
		case 574: goto st574;
		case 575: goto st575;
		case 576: goto st576;
		case 577: goto st577;
		case 578: goto st578;
		case 579: goto st579;
		case 580: goto st580;
		case 581: goto st581;
		case 582: goto st582;
		case 583: goto st583;
		case 584: goto st584;
		case 585: goto st585;
		case 586: goto st586;
		case 587: goto st587;
		case 588: goto st588;
		case 589: goto st589;
		case 590: goto st590;
		case 1090: goto st1090;
		case 591: goto st591;
		case 592: goto st592;
		case 1091: goto st1091;
		case 593: goto st593;
		case 594: goto st594;
		case 595: goto st595;
		case 596: goto st596;
		case 597: goto st597;
		case 598: goto st598;
		case 599: goto st599;
		case 600: goto st600;
		case 601: goto st601;
		case 602: goto st602;
		case 603: goto st603;
		case 604: goto st604;
		case 605: goto st605;
		case 606: goto st606;
		case 1092: goto st1092;
		case 607: goto st607;
		case 608: goto st608;
		case 609: goto st609;
		case 610: goto st610;
		case 611: goto st611;
		case 612: goto st612;
		case 613: goto st613;
		case 614: goto st614;
		case 615: goto st615;
		case 616: goto st616;
		case 617: goto st617;
		case 618: goto st618;
		case 619: goto st619;
		case 620: goto st620;
		case 621: goto st621;
		case 622: goto st622;
		case 623: goto st623;
		case 624: goto st624;
		case 625: goto st625;
		case 626: goto st626;
		case 627: goto st627;
		case 628: goto st628;
		case 1093: goto st1093;
		case 629: goto st629;
		case 630: goto st630;
		case 631: goto st631;
		case 632: goto st632;
		case 633: goto st633;
		case 1094: goto st1094;
		case 634: goto st634;
		case 635: goto st635;
		case 636: goto st636;
		case 637: goto st637;
		case 638: goto st638;
		case 1095: goto st1095;
		case 639: goto st639;
		case 640: goto st640;
		case 641: goto st641;
		case 642: goto st642;
		case 643: goto st643;
		case 1096: goto st1096;
		case 1097: goto st1097;
		case 1098: goto st1098;
		case 644: goto st644;
		case 645: goto st645;
		case 1099: goto st1099;
		case 646: goto st646;
		case 647: goto st647;
		case 648: goto st648;
		case 649: goto st649;
		case 650: goto st650;
		case 651: goto st651;
		case 652: goto st652;
		case 653: goto st653;
		case 654: goto st654;
		case 655: goto st655;
		case 656: goto st656;
		case 657: goto st657;
		case 658: goto st658;
		case 659: goto st659;
		case 660: goto st660;
		case 661: goto st661;
		case 662: goto st662;
		case 663: goto st663;
		case 664: goto st664;
		case 665: goto st665;
		case 666: goto st666;
		case 667: goto st667;
		case 668: goto st668;
		case 669: goto st669;
		case 670: goto st670;
		case 671: goto st671;
		case 672: goto st672;
		case 1100: goto st1100;
		case 1101: goto st1101;
		case 1102: goto st1102;
		case 673: goto st673;
		case 674: goto st674;
		case 675: goto st675;
		case 1103: goto st1103;
		case 1104: goto st1104;
		case 676: goto st676;
		case 677: goto st677;
		case 678: goto st678;
		case 679: goto st679;
		case 1105: goto st1105;
		case 1106: goto st1106;
		case 680: goto st680;
		case 681: goto st681;
		case 682: goto st682;
		case 683: goto st683;
		case 1107: goto st1107;
		case 1108: goto st1108;
		case 684: goto st684;
		case 685: goto st685;
		case 686: goto st686;
		case 687: goto st687;
		case 688: goto st688;
		case 689: goto st689;
		case 690: goto st690;
		case 691: goto st691;
		case 692: goto st692;
		case 693: goto st693;
		case 694: goto st694;
		case 695: goto st695;
		case 696: goto st696;
		case 697: goto st697;
		case 698: goto st698;
		case 699: goto st699;
		case 700: goto st700;
		case 701: goto st701;
		case 702: goto st702;
		case 703: goto st703;
		case 704: goto st704;
		case 705: goto st705;
		case 706: goto st706;
		case 707: goto st707;
		case 708: goto st708;
		case 1109: goto st1109;
		case 709: goto st709;
		case 710: goto st710;
		case 711: goto st711;
		case 712: goto st712;
		case 713: goto st713;
		case 714: goto st714;
		case 715: goto st715;
		case 716: goto st716;
		case 717: goto st717;
		case 718: goto st718;
		case 719: goto st719;
		case 720: goto st720;
		case 721: goto st721;
		case 722: goto st722;
		case 723: goto st723;
		case 1110: goto st1110;
		case 724: goto st724;
		case 725: goto st725;
		case 726: goto st726;
		case 727: goto st727;
		case 728: goto st728;
		case 729: goto st729;
		case 730: goto st730;
		case 731: goto st731;
		case 732: goto st732;
		case 733: goto st733;
		case 734: goto st734;
		case 735: goto st735;
		case 736: goto st736;
		case 1111: goto st1111;
		case 737: goto st737;
		case 738: goto st738;
		case 739: goto st739;
		case 740: goto st740;
		case 741: goto st741;
		case 742: goto st742;
		case 743: goto st743;
		case 744: goto st744;
		case 745: goto st745;
		case 746: goto st746;
		case 747: goto st747;
		case 1112: goto st1112;
		case 1113: goto st1113;
		case 748: goto st748;
		case 749: goto st749;
		case 750: goto st750;
		case 1114: goto st1114;
		case 751: goto st751;
		case 752: goto st752;
		case 753: goto st753;
		case 754: goto st754;
		case 755: goto st755;
		case 756: goto st756;
		case 757: goto st757;
		case 758: goto st758;
		case 759: goto st759;
		case 760: goto st760;
		case 1115: goto st1115;
		case 1116: goto st1116;
		case 1117: goto st1117;
		case 761: goto st761;
		case 762: goto st762;
		case 763: goto st763;
		case 764: goto st764;
		case 765: goto st765;
		case 766: goto st766;
		case 767: goto st767;
		case 768: goto st768;
		case 769: goto st769;
		case 770: goto st770;
		case 771: goto st771;
		case 1118: goto st1118;
		case 1119: goto st1119;
		case 1120: goto st1120;
		case 772: goto st772;
		case 773: goto st773;
		case 774: goto st774;
		case 775: goto st775;
		case 776: goto st776;
		case 777: goto st777;
		case 778: goto st778;
		case 779: goto st779;
		case 780: goto st780;
		case 781: goto st781;
		case 782: goto st782;
		case 783: goto st783;
		case 1121: goto st1121;
		case 784: goto st784;
		case 785: goto st785;
		case 786: goto st786;
		case 1122: goto st1122;
		case 1123: goto st1123;
		case 787: goto st787;
		case 1124: goto st1124;
		case 1125: goto st1125;
		case 788: goto st788;
		case 1126: goto st1126;
		case 1127: goto st1127;
		case 789: goto st789;
		case 790: goto st790;
		case 791: goto st791;
		case 792: goto st792;
		case 793: goto st793;
		case 794: goto st794;
		case 795: goto st795;
		case 796: goto st796;
		case 797: goto st797;
		case 798: goto st798;
		case 799: goto st799;
		case 800: goto st800;
		case 801: goto st801;
		case 802: goto st802;
		case 803: goto st803;
		case 804: goto st804;
		case 805: goto st805;
		case 806: goto st806;
		case 807: goto st807;
		case 808: goto st808;
		case 809: goto st809;
		case 810: goto st810;
		case 811: goto st811;
		case 812: goto st812;
		case 813: goto st813;
		case 814: goto st814;
		case 815: goto st815;
		case 816: goto st816;
		case 817: goto st817;
		case 818: goto st818;
		case 819: goto st819;
		case 820: goto st820;
		case 821: goto st821;
		case 822: goto st822;
		case 823: goto st823;
		case 824: goto st824;
		case 825: goto st825;
		case 826: goto st826;
		case 827: goto st827;
		case 828: goto st828;
		case 829: goto st829;
		case 830: goto st830;
		case 1128: goto st1128;
		case 831: goto st831;
		case 832: goto st832;
		case 833: goto st833;
		case 834: goto st834;
		case 835: goto st835;
		case 836: goto st836;
		case 837: goto st837;
		case 838: goto st838;
		case 839: goto st839;
		case 840: goto st840;
		case 841: goto st841;
		case 842: goto st842;
		case 843: goto st843;
		case 844: goto st844;
		case 845: goto st845;
		case 846: goto st846;
		case 847: goto st847;
		case 848: goto st848;
		case 849: goto st849;
		case 850: goto st850;
		case 851: goto st851;
		case 852: goto st852;
		case 853: goto st853;
		case 854: goto st854;
		case 855: goto st855;
		case 856: goto st856;
		case 857: goto st857;
		case 858: goto st858;
		case 859: goto st859;
		case 860: goto st860;
		case 861: goto st861;
		case 862: goto st862;
		case 863: goto st863;
		case 864: goto st864;
		case 865: goto st865;
		case 866: goto st866;
		case 867: goto st867;
		case 868: goto st868;
		case 869: goto st869;
		case 870: goto st870;
		case 871: goto st871;
		case 872: goto st872;
		case 873: goto st873;
		case 874: goto st874;
		case 875: goto st875;
		case 876: goto st876;
		case 877: goto st877;
		case 878: goto st878;
		case 879: goto st879;
		case 880: goto st880;
		case 881: goto st881;
		case 882: goto st882;
		case 883: goto st883;
		case 884: goto st884;
		case 885: goto st885;
		case 886: goto st886;
		case 887: goto st887;
		case 888: goto st888;
		case 889: goto st889;
		case 890: goto st890;
		case 891: goto st891;
		case 892: goto st892;
		case 893: goto st893;
		case 894: goto st894;
		case 895: goto st895;
		case 896: goto st896;
		case 897: goto st897;
		case 898: goto st898;
		case 899: goto st899;
		case 900: goto st900;
		case 901: goto st901;
		case 902: goto st902;
		case 903: goto st903;
		case 904: goto st904;
		case 905: goto st905;
		case 906: goto st906;
		case 907: goto st907;
		case 908: goto st908;
		case 909: goto st909;
		case 910: goto st910;
		case 911: goto st911;
		case 912: goto st912;
		case 913: goto st913;
		case 914: goto st914;
		case 915: goto st915;
		case 916: goto st916;
		case 917: goto st917;
		case 918: goto st918;
		case 919: goto st919;
		case 920: goto st920;
		case 921: goto st921;
		case 922: goto st922;
		case 923: goto st923;
		case 924: goto st924;
		case 925: goto st925;
		case 926: goto st926;
		case 927: goto st927;
		case 928: goto st928;
		case 929: goto st929;
		case 930: goto st930;
		case 931: goto st931;
		case 932: goto st932;
		case 933: goto st933;
		case 934: goto st934;
		case 935: goto st935;
		case 936: goto st936;
		case 937: goto st937;
		case 938: goto st938;
		case 939: goto st939;
		case 940: goto st940;
		case 941: goto st941;
		case 942: goto st942;
		case 943: goto st943;
		case 944: goto st944;
		case 945: goto st945;
		case 946: goto st946;
		case 947: goto st947;
		case 948: goto st948;
		case 949: goto st949;
		case 950: goto st950;
		case 951: goto st951;
		case 952: goto st952;
		case 953: goto st953;
		case 954: goto st954;
		case 955: goto st955;
		case 956: goto st956;
		case 957: goto st957;
		case 1129: goto st1129;
		case 958: goto st958;
		case 959: goto st959;
		case 960: goto st960;
		case 961: goto st961;
		case 962: goto st962;
		case 963: goto st963;
		case 964: goto st964;
		case 965: goto st965;
		case 1130: goto st1130;
		case 966: goto st966;
		case 967: goto st967;
		case 968: goto st968;
		case 969: goto st969;
		case 970: goto st970;
		case 1131: goto st1131;
		case 971: goto st971;
		case 972: goto st972;
		case 973: goto st973;
		case 974: goto st974;
		case 975: goto st975;
		case 976: goto st976;
		case 977: goto st977;
		case 978: goto st978;
		case 979: goto st979;
		case 980: goto st980;
		case 981: goto st981;
		case 982: goto st982;
		case 983: goto st983;
		case 984: goto st984;
		case 985: goto st985;
		case 986: goto st986;
		case 987: goto st987;
		case 988: goto st988;
		case 1132: goto st1132;
		case 989: goto st989;
		case 990: goto st990;
		case 991: goto st991;
		case 992: goto st992;
		case 993: goto st993;
		case 994: goto st994;
		case 995: goto st995;
		case 996: goto st996;
		case 997: goto st997;
		case 998: goto st998;
		case 999: goto st999;
		case 1000: goto st1000;
		case 1001: goto st1001;
		case 1002: goto st1002;
		case 1003: goto st1003;
		case 1004: goto st1004;
		case 1005: goto st1005;
		case 1006: goto st1006;
		case 1007: goto st1007;
		case 1133: goto st1133;
		case 1008: goto st1008;
		case 1009: goto st1009;
		case 1010: goto st1010;
		case 1011: goto st1011;
		case 1012: goto st1012;
		case 1013: goto st1013;
		case 1014: goto st1014;
		case 1015: goto st1015;
		case 1016: goto st1016;
		case 1017: goto st1017;
		case 1018: goto st1018;
		case 1019: goto st1019;
		case 1020: goto st1020;
		case 1021: goto st1021;
		case 1022: goto st1022;
		case 1134: goto st1134;
		case 1135: goto st1135;
		case 1136: goto st1136;
		case 1023: goto st1023;
		case 1024: goto st1024;
		case 1025: goto st1025;
		case 1026: goto st1026;
		case 1027: goto st1027;
		case 1028: goto st1028;
		case 1029: goto st1029;
		case 1137: goto st1137;
		case 1030: goto st1030;
		case 1031: goto st1031;
		case 1032: goto st1032;
		case 1033: goto st1033;
		case 1034: goto st1034;
		case 1035: goto st1035;
		case 1036: goto st1036;
		case 1037: goto st1037;
		case 1038: goto st1038;
		case 1039: goto st1039;
		case 1040: goto st1040;
		case 1041: goto st1041;
		case 1042: goto st1042;
		case 1138: goto st1138;
		case 1043: goto st1043;
		case 1044: goto st1044;
		case 1045: goto st1045;
		case 1046: goto st1046;
		case 1047: goto st1047;
		case 1048: goto st1048;
		case 1049: goto st1049;
		case 1139: goto st1139;
		case 1050: goto st1050;
		case 1051: goto st1051;
		case 1052: goto st1052;
		case 1053: goto st1053;
		case 1054: goto st1054;
		case 1055: goto st1055;
		case 1140: goto st1140;
	default: break;
	}

	if ( ++p == pe )
		goto _test_eof;
_resume:
	switch ( cs )
	{
tr19:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st1056;
tr81:
#line 1899 "./scanner_body.rl"
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZSCANNER_ERDATA_OVERFLOW);
			p--; {goto st246;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->process_record(s);
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st1056;
tr3194:
#line 704 "./scanner_body.rl"
	{
		s->stop = false;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st1056;
st1056:
	if ( ++p == pe )
		goto _test_eof1056;
case 1056:
#line 1442 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr3135;
		case 32: goto tr3135;
		case 36: goto st138;
		case 40: goto tr3137;
		case 41: goto tr3138;
		case 42: goto tr3139;
		case 92: goto tr3139;
		case 95: goto tr3139;
		case 778: goto tr19;
		case 827: goto st137;
		case 1034: goto tr3140;
		case 1083: goto tr3141;
	}
	if ( _widec < 64 ) {
		if ( 45 <= _widec && _widec <= 57 )
			goto tr3139;
	} else if ( _widec > 90 ) {
		if ( 97 <= _widec && _widec <= 122 )
			goto tr3139;
	} else
		goto tr3139;
	goto tr3134;
tr0:
#line 281 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_NUMBER);
		p--; {goto st246;}
	}
#line 1850 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EUNSUPPORTED_TYPE);
		p--; {goto st246;}
	}
#line 44 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_REST);
		p--; {goto st246;}
	}
	goto st0;
tr23:
#line 382 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_TIME_UNIT);
		p--; {goto st246;}
	}
#line 281 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_NUMBER);
		p--; {goto st246;}
	}
#line 1850 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EUNSUPPORTED_TYPE);
		p--; {goto st246;}
	}
	goto st0;
tr35:
#line 1850 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EUNSUPPORTED_TYPE);
		p--; {goto st246;}
	}
	goto st0;
tr55:
#line 1850 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EUNSUPPORTED_TYPE);
		p--; {goto st246;}
	}
#line 1601 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_RDATA);
		p--; {goto st246;}
	}
	goto st0;
tr69:
#line 1601 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_RDATA);
		p--; {goto st246;}
	}
	goto st0;
tr77:
#line 1601 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_RDATA);
		p--; {goto st246;}
	}
#line 44 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_REST);
		p--; {goto st246;}
	}
	goto st0;
tr83:
#line 44 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_REST);
		p--; {goto st246;}
	}
	goto st0;
tr95:
#line 281 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_NUMBER);
		p--; {goto st246;}
	}
#line 1850 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EUNSUPPORTED_TYPE);
		p--; {goto st246;}
	}
#line 1601 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_RDATA);
		p--; {goto st246;}
	}
#line 44 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_REST);
		p--; {goto st246;}
	}
	goto st0;
tr119:
#line 382 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_TIME_UNIT);
		p--; {goto st246;}
	}
#line 281 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_NUMBER);
		p--; {goto st246;}
	}
#line 1850 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EUNSUPPORTED_TYPE);
		p--; {goto st246;}
	}
#line 1601 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_RDATA);
		p--; {goto st246;}
	}
#line 44 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_REST);
		p--; {goto st246;}
	}
	goto st0;
tr125:
#line 1850 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EUNSUPPORTED_TYPE);
		p--; {goto st246;}
	}
#line 44 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_REST);
		p--; {goto st246;}
	}
	goto st0;
tr524:
#line 281 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_NUMBER);
		p--; {goto st246;}
	}
#line 1850 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EUNSUPPORTED_TYPE);
		p--; {goto st246;}
	}
	goto st0;
tr526:
#line 281 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_NUMBER);
		p--; {goto st246;}
	}
#line 1850 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EUNSUPPORTED_TYPE);
		p--; {goto st246;}
	}
#line 1601 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_RDATA);
		p--; {goto st246;}
	}
	goto st0;
tr591:
#line 707 "./scanner_body.rl"
	{
		ERR(ZSCANNER_EBAD_DIRECTIVE);
		p--; {goto st246;}
	}
	goto st0;
tr602:
#line 238 "./scanner_body.rl"
	{
		s->r_owner_length = 0;
		WARN(ZSCANNER_EBAD_OWNER);
		p--; {goto st246;}
	}
	goto st0;
tr614:
#line 238 "./scanner_body.rl"
	{
		s->r_owner_length = 0;
		WARN(ZSCANNER_EBAD_OWNER);
		p--; {goto st246;}
	}
#line 382 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_TIME_UNIT);
		p--; {goto st246;}
	}
#line 281 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_NUMBER);
		p--; {goto st246;}
	}
#line 1850 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EUNSUPPORTED_TYPE);
		p--; {goto st246;}
	}
	goto st0;
tr638:
#line 238 "./scanner_body.rl"
	{
		s->r_owner_length = 0;
		WARN(ZSCANNER_EBAD_OWNER);
		p--; {goto st246;}
	}
#line 1850 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EUNSUPPORTED_TYPE);
		p--; {goto st246;}
	}
#line 1601 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_RDATA);
		p--; {goto st246;}
	}
	goto st0;
tr653:
#line 1850 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EUNSUPPORTED_TYPE);
		p--; {goto st246;}
	}
#line 1601 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_RDATA);
		p--; {goto st246;}
	}
#line 44 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_REST);
		p--; {goto st246;}
	}
	goto st0;
tr681:
#line 1601 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_RDATA);
		p--; {goto st246;}
	}
#line 44 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_REST);
		p--; {goto st246;}
	}
#line 707 "./scanner_body.rl"
	{
		ERR(ZSCANNER_EBAD_DIRECTIVE);
		p--; {goto st246;}
	}
	goto st0;
tr691:
#line 238 "./scanner_body.rl"
	{
		s->r_owner_length = 0;
		WARN(ZSCANNER_EBAD_OWNER);
		p--; {goto st246;}
	}
#line 1601 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_RDATA);
		p--; {goto st246;}
	}
#line 44 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_REST);
		p--; {goto st246;}
	}
	goto st0;
tr697:
#line 238 "./scanner_body.rl"
	{
		s->r_owner_length = 0;
		WARN(ZSCANNER_EBAD_OWNER);
		p--; {goto st246;}
	}
#line 382 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_TIME_UNIT);
		p--; {goto st246;}
	}
#line 281 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_NUMBER);
		p--; {goto st246;}
	}
#line 1850 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EUNSUPPORTED_TYPE);
		p--; {goto st246;}
	}
#line 1601 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_RDATA);
		p--; {goto st246;}
	}
#line 44 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_REST);
		p--; {goto st246;}
	}
	goto st0;
tr708:
#line 238 "./scanner_body.rl"
	{
		s->r_owner_length = 0;
		WARN(ZSCANNER_EBAD_OWNER);
		p--; {goto st246;}
	}
#line 1850 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EUNSUPPORTED_TYPE);
		p--; {goto st246;}
	}
#line 1601 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_RDATA);
		p--; {goto st246;}
	}
#line 44 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_REST);
		p--; {goto st246;}
	}
	goto st0;
tr721:
#line 238 "./scanner_body.rl"
	{
		s->r_owner_length = 0;
		WARN(ZSCANNER_EBAD_OWNER);
		p--; {goto st246;}
	}
#line 1601 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_RDATA);
		p--; {goto st246;}
	}
	goto st0;
tr774:
#line 238 "./scanner_body.rl"
	{
		s->r_owner_length = 0;
		WARN(ZSCANNER_EBAD_OWNER);
		p--; {goto st246;}
	}
#line 1850 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EUNSUPPORTED_TYPE);
		p--; {goto st246;}
	}
	goto st0;
tr779:
#line 193 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_DNAME_CHAR);
		p--; {goto st246;}
	}
	goto st0;
tr792:
#line 147 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_NUMBER);
		p--; {goto st246;}
	}
#line 193 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_DNAME_CHAR);
		p--; {goto st246;}
	}
	goto st0;
tr800:
#line 505 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_TEXT_CHAR);
		p--; {goto st246;}
	}
#line 509 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_TEXT);
		p--; {goto st246;}
	}
	goto st0;
tr807:
#line 539 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_NUMBER);
		p--; {goto st246;}
	}
#line 505 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_TEXT_CHAR);
		p--; {goto st246;}
	}
#line 509 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_TEXT);
		p--; {goto st246;}
	}
	goto st0;
tr817:
#line 509 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_TEXT);
		p--; {goto st246;}
	}
	goto st0;
tr824:
#line 579 "./scanner_body.rl"
	{
		ERR(ZSCANNER_EBAD_TTL);
		p--; {goto st246;}
	}
	goto st0;
tr830:
#line 281 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_NUMBER);
		p--; {goto st246;}
	}
#line 579 "./scanner_body.rl"
	{
		ERR(ZSCANNER_EBAD_TTL);
		p--; {goto st246;}
	}
	goto st0;
tr832:
#line 382 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_TIME_UNIT);
		p--; {goto st246;}
	}
#line 281 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_NUMBER);
		p--; {goto st246;}
	}
#line 44 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_REST);
		p--; {goto st246;}
	}
#line 579 "./scanner_body.rl"
	{
		ERR(ZSCANNER_EBAD_TTL);
		p--; {goto st246;}
	}
	goto st0;
tr844:
#line 44 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_REST);
		p--; {goto st246;}
	}
#line 579 "./scanner_body.rl"
	{
		ERR(ZSCANNER_EBAD_TTL);
		p--; {goto st246;}
	}
	goto st0;
tr846:
#line 382 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_TIME_UNIT);
		p--; {goto st246;}
	}
#line 281 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_NUMBER);
		p--; {goto st246;}
	}
#line 579 "./scanner_body.rl"
	{
		ERR(ZSCANNER_EBAD_TTL);
		p--; {goto st246;}
	}
	goto st0;
tr859:
#line 596 "./scanner_body.rl"
	{
		ERR(ZSCANNER_EBAD_ORIGIN);
		p--; {goto st246;}
	}
	goto st0;
tr871:
#line 44 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_REST);
		p--; {goto st246;}
	}
#line 596 "./scanner_body.rl"
	{
		ERR(ZSCANNER_EBAD_ORIGIN);
		p--; {goto st246;}
	}
	goto st0;
tr885:
#line 147 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_NUMBER);
		p--; {goto st246;}
	}
#line 596 "./scanner_body.rl"
	{
		ERR(ZSCANNER_EBAD_ORIGIN);
		p--; {goto st246;}
	}
	goto st0;
tr897:
#line 625 "./scanner_body.rl"
	{
		ERR(ZSCANNER_EBAD_INCLUDE_FILENAME);
		p--; {goto st246;}
	}
	goto st0;
tr898:
#line 625 "./scanner_body.rl"
	{
		ERR(ZSCANNER_EBAD_INCLUDE_FILENAME);
		p--; {goto st246;}
	}
#line 44 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_REST);
		p--; {goto st246;}
	}
	goto st0;
tr905:
#line 636 "./scanner_body.rl"
	{
		ERR(ZSCANNER_EBAD_INCLUDE_ORIGIN);
		p--; {goto st246;}
	}
#line 44 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_REST);
		p--; {goto st246;}
	}
	goto st0;
tr915:
#line 636 "./scanner_body.rl"
	{
		ERR(ZSCANNER_EBAD_INCLUDE_ORIGIN);
		p--; {goto st246;}
	}
	goto st0;
tr930:
#line 147 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_NUMBER);
		p--; {goto st246;}
	}
#line 636 "./scanner_body.rl"
	{
		ERR(ZSCANNER_EBAD_INCLUDE_ORIGIN);
		p--; {goto st246;}
	}
	goto st0;
tr942:
#line 951 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_BASE64_CHAR);
		p--; {goto st246;}
	}
	goto st0;
tr956:
#line 1311 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_BITMAP);
		p--; {goto st246;}
	}
	goto st0;
tr1403:
#line 1477 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_HEX_RDATA);
		p--; {goto st246;}
	}
	goto st0;
tr1409:
#line 281 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_NUMBER);
		p--; {goto st246;}
	}
#line 1477 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_HEX_RDATA);
		p--; {goto st246;}
	}
	goto st0;
tr1417:
#line 888 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_HEX_CHAR);
		p--; {goto st246;}
	}
#line 909 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_HEX_RDATA);
		p--; {goto st246;}
	}
#line 1477 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_HEX_RDATA);
		p--; {goto st246;}
	}
	goto st0;
tr1467:
#line 1553 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_ALGORITHM);
		p--; {goto st246;}
	}
	goto st0;
tr1578:
#line 1557 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_CERT_TYPE);
		p--; {goto st246;}
	}
	goto st0;
tr1631:
#line 758 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_ADDRESS_CHAR);
		p--; {goto st246;}
	}
#line 1601 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_RDATA);
		p--; {goto st246;}
	}
	goto st0;
tr1654:
#line 281 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_NUMBER);
		p--; {goto st246;}
	}
#line 1601 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_RDATA);
		p--; {goto st246;}
	}
	goto st0;
tr1673:
#line 382 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_TIME_UNIT);
		p--; {goto st246;}
	}
#line 281 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_NUMBER);
		p--; {goto st246;}
	}
#line 1601 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_RDATA);
		p--; {goto st246;}
	}
	goto st0;
tr1840:
#line 1464 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_LOC_DATA);
		p--; {goto st246;}
	}
#line 1601 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_RDATA);
		p--; {goto st246;}
	}
	goto st0;
tr2197:
#line 858 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_APL);
		p--; {goto st246;}
	}
#line 1601 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_RDATA);
		p--; {goto st246;}
	}
	goto st0;
tr2212:
#line 758 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_ADDRESS_CHAR);
		p--; {goto st246;}
	}
#line 858 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_APL);
		p--; {goto st246;}
	}
#line 1601 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_RDATA);
		p--; {goto st246;}
	}
	goto st0;
tr2255:
#line 888 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_HEX_CHAR);
		p--; {goto st246;}
	}
#line 1601 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_RDATA);
		p--; {goto st246;}
	}
	goto st0;
tr2312:
#line 1159 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_GATEWAY);
		p--; {goto st246;}
	}
#line 1601 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_RDATA);
		p--; {goto st246;}
	}
	goto st0;
tr2327:
#line 281 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_NUMBER);
		p--; {goto st246;}
	}
#line 1159 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_GATEWAY);
		p--; {goto st246;}
	}
#line 1601 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_RDATA);
		p--; {goto st246;}
	}
	goto st0;
tr2341:
#line 1159 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_GATEWAY);
		p--; {goto st246;}
	}
#line 1163 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_GATEWAY_KEY);
		p--; {goto st246;}
	}
#line 1601 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_RDATA);
		p--; {goto st246;}
	}
	goto st0;
tr2364:
#line 1163 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_GATEWAY_KEY);
		p--; {goto st246;}
	}
#line 1601 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_RDATA);
		p--; {goto st246;}
	}
	goto st0;
tr2378:
#line 758 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_ADDRESS_CHAR);
		p--; {goto st246;}
	}
#line 1159 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_GATEWAY);
		p--; {goto st246;}
	}
#line 1601 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_RDATA);
		p--; {goto st246;}
	}
	goto st0;
tr2385:
#line 758 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_ADDRESS_CHAR);
		p--; {goto st246;}
	}
#line 1159 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_GATEWAY);
		p--; {goto st246;}
	}
#line 1163 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_GATEWAY_KEY);
		p--; {goto st246;}
	}
#line 1601 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_RDATA);
		p--; {goto st246;}
	}
	goto st0;
tr2490:
#line 1181 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EUNSUPPORTED_TYPE);
		p--; {goto st246;}
	}
#line 1601 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_RDATA);
		p--; {goto st246;}
	}
	goto st0;
tr2543:
#line 487 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_TIMESTAMP_CHAR);
		p--; {goto st246;}
	}
#line 1601 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_RDATA);
		p--; {goto st246;}
	}
	goto st0;
tr2882:
#line 281 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_NUMBER);
		p--; {goto st246;}
	}
#line 1181 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EUNSUPPORTED_TYPE);
		p--; {goto st246;}
	}
#line 1601 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_RDATA);
		p--; {goto st246;}
	}
	goto st0;
tr2970:
#line 1035 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_BASE32HEX_CHAR);
		p--; {goto st246;}
	}
#line 1601 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_RDATA);
		p--; {goto st246;}
	}
	goto st0;
tr3113:
#line 888 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_HEX_CHAR);
		p--; {goto st246;}
	}
#line 1540 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_CHAR_COLON);
		p--; {goto st246;}
	}
#line 1601 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_RDATA);
		p--; {goto st246;}
	}
	goto st0;
tr3119:
#line 1540 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_CHAR_COLON);
		p--; {goto st246;}
	}
#line 888 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_HEX_CHAR);
		p--; {goto st246;}
	}
#line 1601 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_RDATA);
		p--; {goto st246;}
	}
	goto st0;
tr3123:
#line 1513 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_CHAR_DASH);
		p--; {goto st246;}
	}
#line 888 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_HEX_CHAR);
		p--; {goto st246;}
	}
#line 1601 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_RDATA);
		p--; {goto st246;}
	}
	goto st0;
tr3134:
#line 238 "./scanner_body.rl"
	{
		s->r_owner_length = 0;
		WARN(ZSCANNER_EBAD_OWNER);
		p--; {goto st246;}
	}
#line 44 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_REST);
		p--; {goto st246;}
	}
	goto st0;
tr3152:
#line 238 "./scanner_body.rl"
	{
		s->r_owner_length = 0;
		WARN(ZSCANNER_EBAD_OWNER);
		p--; {goto st246;}
	}
#line 1850 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EUNSUPPORTED_TYPE);
		p--; {goto st246;}
	}
#line 44 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_REST);
		p--; {goto st246;}
	}
	goto st0;
tr3172:
#line 238 "./scanner_body.rl"
	{
		s->r_owner_length = 0;
		WARN(ZSCANNER_EBAD_OWNER);
		p--; {goto st246;}
	}
#line 281 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_NUMBER);
		p--; {goto st246;}
	}
#line 1850 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EUNSUPPORTED_TYPE);
		p--; {goto st246;}
	}
#line 44 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_REST);
		p--; {goto st246;}
	}
	goto st0;
tr3188:
#line 238 "./scanner_body.rl"
	{
		s->r_owner_length = 0;
		WARN(ZSCANNER_EBAD_OWNER);
		p--; {goto st246;}
	}
#line 707 "./scanner_body.rl"
	{
		ERR(ZSCANNER_EBAD_DIRECTIVE);
		p--; {goto st246;}
	}
#line 44 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_REST);
		p--; {goto st246;}
	}
	goto st0;
tr3211:
#line 238 "./scanner_body.rl"
	{
		s->r_owner_length = 0;
		WARN(ZSCANNER_EBAD_OWNER);
		p--; {goto st246;}
	}
#line 281 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_NUMBER);
		p--; {goto st246;}
	}
#line 1850 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EUNSUPPORTED_TYPE);
		p--; {goto st246;}
	}
#line 1601 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_RDATA);
		p--; {goto st246;}
	}
#line 44 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_REST);
		p--; {goto st246;}
	}
	goto st0;
#line 2504 "scanner.c"
st0:
cs = 0;
	goto _out;
tr2:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st1;
tr3:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st1;
tr3135:
#line 232 "./scanner_body.rl"
	{
		if (s->r_owner_length == 0) {
			WARN(ZSCANNER_EBAD_PREVIOUS_OWNER);
			p--; {goto st246;}
		}
	}
	goto st1;
tr3137:
#line 232 "./scanner_body.rl"
	{
		if (s->r_owner_length == 0) {
			WARN(ZSCANNER_EBAD_PREVIOUS_OWNER);
			p--; {goto st246;}
		}
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st1;
tr3138:
#line 232 "./scanner_body.rl"
	{
		if (s->r_owner_length == 0) {
			WARN(ZSCANNER_EBAD_PREVIOUS_OWNER);
			p--; {goto st246;}
		}
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st1;
tr3198:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
#line 232 "./scanner_body.rl"
	{
		if (s->r_owner_length == 0) {
			WARN(ZSCANNER_EBAD_PREVIOUS_OWNER);
			p--; {goto st246;}
		}
	}
	goto st1;
tr3199:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
#line 232 "./scanner_body.rl"
	{
		if (s->r_owner_length == 0) {
			WARN(ZSCANNER_EBAD_PREVIOUS_OWNER);
			p--; {goto st246;}
		}
	}
	goto st1;
tr3189:
#line 704 "./scanner_body.rl"
	{
		s->stop = false;
	}
#line 232 "./scanner_body.rl"
	{
		if (s->r_owner_length == 0) {
			WARN(ZSCANNER_EBAD_PREVIOUS_OWNER);
			p--; {goto st246;}
		}
	}
	goto st1;
tr3191:
#line 704 "./scanner_body.rl"
	{
		s->stop = false;
	}
#line 232 "./scanner_body.rl"
	{
		if (s->r_owner_length == 0) {
			WARN(ZSCANNER_EBAD_PREVIOUS_OWNER);
			p--; {goto st246;}
		}
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st1;
tr3192:
#line 704 "./scanner_body.rl"
	{
		s->stop = false;
	}
#line 232 "./scanner_body.rl"
	{
		if (s->r_owner_length == 0) {
			WARN(ZSCANNER_EBAD_PREVIOUS_OWNER);
			p--; {goto st246;}
		}
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st1;
st1:
	if ( ++p == pe )
		goto _test_eof1;
case 1:
#line 2664 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st1;
		case 32: goto st1;
		case 40: goto tr2;
		case 41: goto tr3;
		case 65: goto tr5;
		case 67: goto tr6;
		case 68: goto tr7;
		case 69: goto tr8;
		case 72: goto tr9;
		case 73: goto tr10;
		case 75: goto tr11;
		case 76: goto tr12;
		case 77: goto tr13;
		case 78: goto tr14;
		case 80: goto tr15;
		case 82: goto tr16;
		case 83: goto tr17;
		case 84: goto tr18;
		case 97: goto tr5;
		case 99: goto tr6;
		case 100: goto tr7;
		case 101: goto tr8;
		case 104: goto tr9;
		case 105: goto tr10;
		case 107: goto tr11;
		case 108: goto tr12;
		case 109: goto tr13;
		case 110: goto tr14;
		case 112: goto tr15;
		case 114: goto tr16;
		case 115: goto tr17;
		case 116: goto tr18;
		case 778: goto tr19;
		case 827: goto st137;
		case 1034: goto tr21;
		case 1083: goto st242;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr4;
	goto tr0;
tr4:
#line 278 "./scanner_body.rl"
	{
		s->number64 = 0;
	}
#line 261 "./scanner_body.rl"
	{
		// Overflow check: 10*(s->number64) + fc - ASCII_0 <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + ASCII_0)
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ENUMBER64_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st2;
tr27:
#line 261 "./scanner_body.rl"
	{
		// Overflow check: 10*(s->number64) + fc - ASCII_0 <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + ASCII_0)
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ENUMBER64_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st2;
st2:
	if ( ++p == pe )
		goto _test_eof2;
case 2:
#line 2762 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr24;
		case 32: goto tr24;
		case 40: goto tr25;
		case 41: goto tr26;
		case 68: goto tr28;
		case 72: goto tr29;
		case 77: goto tr30;
		case 83: goto st152;
		case 87: goto tr32;
		case 100: goto tr28;
		case 104: goto tr29;
		case 109: goto tr30;
		case 115: goto st152;
		case 119: goto tr32;
		case 1034: goto tr33;
		case 1083: goto tr34;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr27;
	goto tr23;
tr37:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st3;
tr38:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st3;
tr53:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st3;
tr24:
#line 731 "./scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st3;
tr25:
#line 731 "./scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st3;
tr26:
#line 731 "./scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st3;
tr33:
#line 731 "./scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st3;
tr632:
#line 423 "./scanner_body.rl"
	{
		if (s->number64 + s->number64_tmp < UINT32_MAX) {
			s->number64 += s->number64_tmp;
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 731 "./scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st3;
tr633:
#line 423 "./scanner_body.rl"
	{
		if (s->number64 + s->number64_tmp < UINT32_MAX) {
			s->number64 += s->number64_tmp;
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 731 "./scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st3;
tr634:
#line 423 "./scanner_body.rl"
	{
		if (s->number64 + s->number64_tmp < UINT32_MAX) {
			s->number64 += s->number64_tmp;
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 731 "./scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st3;
tr636:
#line 423 "./scanner_body.rl"
	{
		if (s->number64 + s->number64_tmp < UINT32_MAX) {
			s->number64 += s->number64_tmp;
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 731 "./scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st3;
st3:
	if ( ++p == pe )
		goto _test_eof3;
case 3:
#line 2992 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st3;
		case 32: goto st3;
		case 40: goto tr37;
		case 41: goto tr38;
		case 65: goto tr39;
		case 67: goto tr40;
		case 68: goto tr41;
		case 69: goto tr42;
		case 72: goto tr43;
		case 73: goto tr44;
		case 75: goto tr45;
		case 76: goto tr46;
		case 77: goto tr47;
		case 78: goto tr48;
		case 80: goto tr49;
		case 82: goto tr50;
		case 83: goto tr51;
		case 84: goto tr52;
		case 97: goto tr39;
		case 99: goto tr40;
		case 100: goto tr41;
		case 101: goto tr42;
		case 104: goto tr43;
		case 105: goto tr44;
		case 107: goto tr45;
		case 108: goto tr46;
		case 109: goto tr47;
		case 110: goto tr48;
		case 112: goto tr49;
		case 114: goto tr50;
		case 115: goto tr51;
		case 116: goto tr52;
		case 1034: goto tr53;
		case 1083: goto st155;
	}
	goto tr35;
tr5:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
	goto st4;
tr39:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
	goto st4;
tr551:
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
	goto st4;
st4:
	if ( ++p == pe )
		goto _test_eof4;
case 4:
#line 3070 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr56;
		case 32: goto tr56;
		case 40: goto tr57;
		case 41: goto tr58;
		case 65: goto st213;
		case 70: goto st216;
		case 80: goto st220;
		case 97: goto st213;
		case 102: goto st216;
		case 112: goto st220;
		case 2058: goto tr62;
		case 2107: goto tr63;
		case 2314: goto tr64;
		case 2363: goto tr64;
		case 2570: goto tr65;
		case 2619: goto tr66;
	}
	goto tr55;
tr70:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st5;
tr71:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st5;
tr73:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st5;
tr56:
#line 1856 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_A; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr57:
#line 1856 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_A; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st5;
tr58:
#line 1856 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_A; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st5;
tr62:
#line 1856 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_A; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st5;
tr135:
#line 1874 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_CERT; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr136:
#line 1874 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_CERT; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st5;
tr137:
#line 1874 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_CERT; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st5;
tr138:
#line 1874 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_CERT; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st5;
tr149:
#line 1858 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_CNAME; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr150:
#line 1858 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_CNAME; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st5;
tr151:
#line 1858 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_CNAME; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st5;
tr152:
#line 1858 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_CNAME; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st5;
tr163:
#line 1883 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_DHCID; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr164:
#line 1883 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_DHCID; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st5;
tr165:
#line 1883 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_DHCID; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st5;
tr166:
#line 1883 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_DHCID; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st5;
tr175:
#line 1875 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_DNAME; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr176:
#line 1875 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_DNAME; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st5;
tr177:
#line 1875 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_DNAME; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st5;
tr178:
#line 1875 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_DNAME; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st5;
tr186:
#line 1882 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_DNSKEY; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr187:
#line 1882 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_DNSKEY; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st5;
tr188:
#line 1882 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_DNSKEY; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st5;
tr189:
#line 1882 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_DNSKEY; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st5;
tr194:
#line 1877 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_DS; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr195:
#line 1877 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_DS; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st5;
tr196:
#line 1877 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_DS; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st5;
tr197:
#line 1877 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_DS; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st5;
tr207:
#line 1892 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_EUI48; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr208:
#line 1892 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_EUI48; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st5;
tr209:
#line 1892 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_EUI48; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st5;
tr210:
#line 1892 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_EUI48; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st5;
tr216:
#line 1893 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_EUI64; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr217:
#line 1893 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_EUI64; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st5;
tr218:
#line 1893 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_EUI64; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st5;
tr219:
#line 1893 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_EUI64; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st5;
tr228:
#line 1861 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_HINFO; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr229:
#line 1861 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_HINFO; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st5;
tr230:
#line 1861 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_HINFO; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st5;
tr231:
#line 1861 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_HINFO; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st5;
tr268:
#line 1879 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_IPSECKEY; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr269:
#line 1879 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_IPSECKEY; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st5;
tr270:
#line 1879 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_IPSECKEY; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st5;
tr271:
#line 1879 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_IPSECKEY; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st5;
tr279:
#line 1868 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_KEY; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr280:
#line 1868 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_KEY; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st5;
tr281:
#line 1868 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_KEY; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st5;
tr282:
#line 1868 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_KEY; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st5;
tr287:
#line 1873 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_KX; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr288:
#line 1873 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_KX; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st5;
tr289:
#line 1873 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_KX; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st5;
tr290:
#line 1873 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_KX; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st5;
tr300:
#line 1889 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_L32; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr301:
#line 1889 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_L32; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st5;
tr302:
#line 1889 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_L32; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st5;
tr303:
#line 1889 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_L32; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st5;
tr309:
#line 1890 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_L64; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr310:
#line 1890 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_L64; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st5;
tr311:
#line 1890 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_L64; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st5;
tr312:
#line 1890 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_L64; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st5;
tr318:
#line 1870 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_LOC; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr319:
#line 1870 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_LOC; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st5;
tr320:
#line 1870 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_LOC; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st5;
tr321:
#line 1870 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_LOC; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st5;
tr326:
#line 1891 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_LP; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr327:
#line 1891 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_LP; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st5;
tr328:
#line 1891 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_LP; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st5;
tr329:
#line 1891 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_LP; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st5;
tr339:
#line 1862 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_MINFO; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr340:
#line 1862 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_MINFO; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st5;
tr341:
#line 1862 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_MINFO; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st5;
tr342:
#line 1862 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_MINFO; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st5;
tr347:
#line 1863 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_MX; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr348:
#line 1863 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_MX; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st5;
tr349:
#line 1863 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_MX; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st5;
tr350:
#line 1863 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_MX; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st5;
tr361:
#line 1872 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NAPTR; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr362:
#line 1872 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NAPTR; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st5;
tr363:
#line 1872 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NAPTR; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st5;
tr364:
#line 1872 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NAPTR; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st5;
tr370:
#line 1888 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NID; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr371:
#line 1888 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NID; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st5;
tr372:
#line 1888 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NID; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st5;
tr373:
#line 1888 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NID; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st5;
tr378:
#line 1857 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NS; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr379:
#line 1857 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NS; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st5;
tr380:
#line 1857 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NS; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st5;
tr382:
#line 1857 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NS; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st5;
tr388:
#line 1881 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NSEC; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr389:
#line 1881 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NSEC; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st5;
tr390:
#line 1881 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NSEC; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st5;
tr392:
#line 1881 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NSEC; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st5;
tr397:
#line 1884 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NSEC3; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr398:
#line 1884 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NSEC3; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st5;
tr399:
#line 1884 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NSEC3; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st5;
tr401:
#line 1884 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NSEC3; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st5;
tr410:
#line 1885 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NSEC3PARAM; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr411:
#line 1885 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NSEC3PARAM; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st5;
tr412:
#line 1885 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NSEC3PARAM; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st5;
tr413:
#line 1885 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NSEC3PARAM; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st5;
tr420:
#line 1860 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_PTR; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr421:
#line 1860 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_PTR; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st5;
tr422:
#line 1860 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_PTR; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st5;
tr423:
#line 1860 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_PTR; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st5;
tr431:
#line 1865 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_RP; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr432:
#line 1865 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_RP; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st5;
tr433:
#line 1865 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_RP; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st5;
tr434:
#line 1865 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_RP; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st5;
tr442:
#line 1880 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_RRSIG; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr443:
#line 1880 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_RRSIG; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st5;
tr444:
#line 1880 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_RRSIG; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st5;
tr445:
#line 1880 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_RRSIG; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st5;
tr450:
#line 1867 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_RT; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr451:
#line 1867 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_RT; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st5;
tr452:
#line 1867 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_RT; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st5;
tr453:
#line 1867 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_RT; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st5;
tr463:
#line 1859 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_SOA; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr464:
#line 1859 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_SOA; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st5;
tr465:
#line 1859 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_SOA; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st5;
tr466:
#line 1859 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_SOA; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st5;
tr472:
#line 1887 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_SPF; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr473:
#line 1887 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_SPF; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st5;
tr474:
#line 1887 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_SPF; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st5;
tr475:
#line 1887 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_SPF; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st5;
tr481:
#line 1871 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_SRV; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr482:
#line 1871 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_SRV; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st5;
tr483:
#line 1871 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_SRV; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st5;
tr484:
#line 1871 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_SRV; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st5;
tr492:
#line 1878 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_SSHFP; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr493:
#line 1878 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_SSHFP; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st5;
tr494:
#line 1878 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_SSHFP; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st5;
tr495:
#line 1878 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_SSHFP; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st5;
tr505:
#line 1886 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_TLSA; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr506:
#line 1886 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_TLSA; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st5;
tr507:
#line 1886 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_TLSA; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st5;
tr508:
#line 1886 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_TLSA; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st5;
tr514:
#line 1864 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_TXT; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr515:
#line 1864 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_TXT; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st5;
tr516:
#line 1864 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_TXT; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st5;
tr517:
#line 1864 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_TXT; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st5;
tr527:
#line 356 "./scanner_body.rl"
	{
		if (s->number64 <= UINT16_MAX) {
			s->r_type = (uint16_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER16_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr528:
#line 356 "./scanner_body.rl"
	{
		if (s->number64 <= UINT16_MAX) {
			s->r_type = (uint16_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER16_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st5;
tr529:
#line 356 "./scanner_body.rl"
	{
		if (s->number64 <= UINT16_MAX) {
			s->r_type = (uint16_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER16_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st5;
tr531:
#line 356 "./scanner_body.rl"
	{
		if (s->number64 <= UINT16_MAX) {
			s->r_type = (uint16_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER16_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st5;
tr734:
#line 1869 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_AAAA; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr735:
#line 1869 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_AAAA; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st5;
tr736:
#line 1869 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_AAAA; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st5;
tr737:
#line 1869 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_AAAA; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st5;
tr745:
#line 1866 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_AFSDB; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr746:
#line 1866 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_AFSDB; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st5;
tr747:
#line 1866 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_AFSDB; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st5;
tr748:
#line 1866 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_AFSDB; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st5;
tr754:
#line 1876 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_APL; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st5;
tr755:
#line 1876 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_APL; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st5;
tr756:
#line 1876 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_APL; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st5;
tr757:
#line 1876 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_APL; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st5;
st5:
	if ( ++p == pe )
		goto _test_eof5;
case 5:
#line 5196 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto st5;
		case 32: goto st5;
		case 40: goto tr70;
		case 41: goto tr71;
		case 92: goto st9;
		case 2058: goto tr73;
		case 2107: goto st10;
		case 2314: goto tr67;
		case 2363: goto tr67;
		case 2570: goto tr75;
		case 2619: goto tr76;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr67;
	} else if ( _widec > 58 ) {
		if ( 60 <= _widec )
			goto tr67;
	} else
		goto tr67;
	goto tr69;
tr64:
#line 1856 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_A; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
tr67:
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
tr84:
#line 1843 "./scanner_body.rl"
	{ p--; }
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
tr85:
#line 1792 "./scanner_body.rl"
	{
		switch (s->r_type) {
		// Next types must not have empty rdata.
		case KNOT_RRTYPE_A:
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
		case KNOT_RRTYPE_SOA:
		case KNOT_RRTYPE_HINFO:
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
		case KNOT_RRTYPE_RP:
		case KNOT_RRTYPE_AAAA:
		case KNOT_RRTYPE_LOC:
		case KNOT_RRTYPE_SRV:
		case KNOT_RRTYPE_NAPTR:
		case KNOT_RRTYPE_CERT:
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_SSHFP:
		case KNOT_RRTYPE_IPSECKEY:
		case KNOT_RRTYPE_RRSIG:
		case KNOT_RRTYPE_NSEC:
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_NSEC3:
		case KNOT_RRTYPE_NSEC3PARAM:
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L32:
		case KNOT_RRTYPE_L64:
		case KNOT_RRTYPE_LP:
		case KNOT_RRTYPE_EUI48:
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st434;}
		// Next types can have empty rdata.
		case KNOT_RRTYPE_APL:
		default:
			{stack[top++] = 6; goto st443;}
		}
	}
	goto st6;
tr140:
#line 1874 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_CERT; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
tr154:
#line 1858 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_CNAME; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
tr168:
#line 1883 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_DHCID; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
tr180:
#line 1875 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_DNAME; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
tr191:
#line 1882 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_DNSKEY; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
tr199:
#line 1877 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_DS; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
tr212:
#line 1892 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_EUI48; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
tr221:
#line 1893 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_EUI64; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
tr233:
#line 1861 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_HINFO; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
tr273:
#line 1879 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_IPSECKEY; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
tr284:
#line 1868 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_KEY; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
tr292:
#line 1873 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_KX; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
tr305:
#line 1889 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_L32; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
tr314:
#line 1890 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_L64; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
tr323:
#line 1870 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_LOC; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
tr331:
#line 1891 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_LP; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
tr344:
#line 1862 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_MINFO; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
tr352:
#line 1863 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_MX; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
tr366:
#line 1872 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NAPTR; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
tr375:
#line 1888 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NID; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
tr384:
#line 1857 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NS; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
tr394:
#line 1881 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NSEC; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
tr403:
#line 1884 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NSEC3; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
tr415:
#line 1885 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NSEC3PARAM; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
tr425:
#line 1860 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_PTR; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
tr436:
#line 1865 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_RP; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
tr447:
#line 1880 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_RRSIG; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
tr455:
#line 1867 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_RT; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
tr468:
#line 1859 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_SOA; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
tr477:
#line 1887 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_SPF; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
tr486:
#line 1871 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_SRV; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
tr497:
#line 1878 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_SSHFP; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
tr510:
#line 1886 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_TLSA; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
tr519:
#line 1864 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_TXT; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
tr533:
#line 356 "./scanner_body.rl"
	{
		if (s->number64 <= UINT16_MAX) {
			s->r_type = (uint16_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER16_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
tr739:
#line 1869 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_AAAA; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
tr750:
#line 1866 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_AFSDB; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
tr759:
#line 1876 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_APL; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 6; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 6; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 6; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 6; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 6; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 6; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 6; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 6; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 6; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 6; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 6; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 6; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 6; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 6; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 6; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 6; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 6; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 6; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 6; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 6; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 6; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 6; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 6; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 6; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 6; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 6; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 6; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st6;
st6:
	if ( ++p == pe )
		goto _test_eof6;
case 6:
#line 8652 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st7;
		case 32: goto st7;
		case 40: goto tr79;
		case 41: goto tr80;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr81;
		case 1083: goto st8;
	}
	goto tr77;
tr79:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st7;
tr80:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st7;
st7:
	if ( ++p == pe )
		goto _test_eof7;
case 7:
#line 8702 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st7;
		case 32: goto st7;
		case 40: goto tr79;
		case 41: goto tr80;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr81;
		case 1083: goto st8;
	}
	goto tr83;
st8:
	if ( ++p == pe )
		goto _test_eof8;
case 8:
	if ( (*p) == 10 )
		goto tr81;
	goto st8;
st9:
	if ( ++p == pe )
		goto _test_eof9;
case 9:
	if ( (*p) == 35 )
		goto tr85;
	goto tr84;
tr63:
#line 1856 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_A; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st10;
tr139:
#line 1874 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_CERT; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st10;
tr153:
#line 1858 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_CNAME; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st10;
tr167:
#line 1883 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_DHCID; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st10;
tr179:
#line 1875 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_DNAME; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st10;
tr190:
#line 1882 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_DNSKEY; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st10;
tr198:
#line 1877 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_DS; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st10;
tr211:
#line 1892 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_EUI48; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st10;
tr220:
#line 1893 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_EUI64; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st10;
tr232:
#line 1861 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_HINFO; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st10;
tr272:
#line 1879 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_IPSECKEY; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st10;
tr283:
#line 1868 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_KEY; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st10;
tr291:
#line 1873 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_KX; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st10;
tr304:
#line 1889 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_L32; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st10;
tr313:
#line 1890 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_L64; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st10;
tr322:
#line 1870 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_LOC; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st10;
tr330:
#line 1891 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_LP; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st10;
tr343:
#line 1862 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_MINFO; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st10;
tr351:
#line 1863 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_MX; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st10;
tr365:
#line 1872 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NAPTR; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st10;
tr374:
#line 1888 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NID; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st10;
tr383:
#line 1857 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NS; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st10;
tr393:
#line 1881 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NSEC; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st10;
tr402:
#line 1884 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NSEC3; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st10;
tr414:
#line 1885 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NSEC3PARAM; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st10;
tr424:
#line 1860 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_PTR; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st10;
tr435:
#line 1865 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_RP; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st10;
tr446:
#line 1880 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_RRSIG; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st10;
tr454:
#line 1867 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_RT; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st10;
tr467:
#line 1859 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_SOA; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st10;
tr476:
#line 1887 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_SPF; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st10;
tr485:
#line 1871 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_SRV; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st10;
tr496:
#line 1878 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_SSHFP; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st10;
tr509:
#line 1886 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_TLSA; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st10;
tr518:
#line 1864 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_TXT; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st10;
tr532:
#line 356 "./scanner_body.rl"
	{
		if (s->number64 <= UINT16_MAX) {
			s->r_type = (uint16_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER16_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st10;
tr738:
#line 1869 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_AAAA; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st10;
tr749:
#line 1866 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_AFSDB; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st10;
tr758:
#line 1876 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_APL; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st10;
st10:
	if ( ++p == pe )
		goto _test_eof10;
case 10:
#line 9065 "scanner.c"
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	if ( _widec == 1034 )
		goto tr73;
	if ( 896 <= _widec && _widec <= 1151 )
		goto st10;
	goto tr69;
tr87:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st11;
tr88:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st11;
tr65:
#line 1856 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_A; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 11; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 11; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 11; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 11; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st11;
tr75:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 11; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 11; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 11; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 11; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st11;
tr141:
#line 1874 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_CERT; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 11; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 11; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 11; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 11; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st11;
tr155:
#line 1858 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_CNAME; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 11; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 11; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 11; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 11; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st11;
tr169:
#line 1883 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_DHCID; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 11; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 11; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 11; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 11; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st11;
tr181:
#line 1875 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_DNAME; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 11; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 11; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 11; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 11; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st11;
tr192:
#line 1882 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_DNSKEY; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 11; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 11; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 11; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 11; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st11;
tr200:
#line 1877 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_DS; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 11; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 11; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 11; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 11; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st11;
tr213:
#line 1892 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_EUI48; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 11; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 11; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 11; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 11; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st11;
tr222:
#line 1893 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_EUI64; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 11; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 11; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 11; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 11; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st11;
tr234:
#line 1861 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_HINFO; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 11; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 11; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 11; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 11; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st11;
tr274:
#line 1879 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_IPSECKEY; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 11; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 11; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 11; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 11; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st11;
tr285:
#line 1868 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_KEY; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 11; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 11; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 11; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 11; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st11;
tr293:
#line 1873 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_KX; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 11; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 11; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 11; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 11; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st11;
tr306:
#line 1889 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_L32; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 11; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 11; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 11; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 11; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st11;
tr315:
#line 1890 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_L64; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 11; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 11; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 11; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 11; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st11;
tr324:
#line 1870 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_LOC; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 11; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 11; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 11; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 11; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st11;
tr332:
#line 1891 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_LP; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 11; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 11; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 11; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 11; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st11;
tr345:
#line 1862 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_MINFO; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 11; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 11; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 11; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 11; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st11;
tr353:
#line 1863 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_MX; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 11; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 11; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 11; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 11; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st11;
tr367:
#line 1872 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NAPTR; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 11; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 11; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 11; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 11; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st11;
tr376:
#line 1888 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NID; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 11; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 11; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 11; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 11; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st11;
tr385:
#line 1857 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NS; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 11; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 11; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 11; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 11; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st11;
tr395:
#line 1881 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NSEC; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 11; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 11; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 11; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 11; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st11;
tr404:
#line 1884 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NSEC3; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 11; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 11; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 11; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 11; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st11;
tr416:
#line 1885 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NSEC3PARAM; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 11; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 11; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 11; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 11; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st11;
tr426:
#line 1860 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_PTR; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 11; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 11; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 11; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 11; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st11;
tr437:
#line 1865 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_RP; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 11; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 11; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 11; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 11; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st11;
tr448:
#line 1880 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_RRSIG; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 11; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 11; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 11; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 11; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st11;
tr456:
#line 1867 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_RT; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 11; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 11; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 11; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 11; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st11;
tr469:
#line 1859 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_SOA; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 11; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 11; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 11; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 11; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st11;
tr478:
#line 1887 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_SPF; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 11; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 11; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 11; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 11; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st11;
tr487:
#line 1871 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_SRV; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 11; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 11; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 11; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 11; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st11;
tr498:
#line 1878 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_SSHFP; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 11; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 11; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 11; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 11; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st11;
tr511:
#line 1886 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_TLSA; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 11; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 11; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 11; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 11; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st11;
tr520:
#line 1864 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_TXT; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 11; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 11; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 11; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 11; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st11;
tr534:
#line 356 "./scanner_body.rl"
	{
		if (s->number64 <= UINT16_MAX) {
			s->r_type = (uint16_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER16_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 11; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 11; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 11; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 11; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st11;
tr740:
#line 1869 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_AAAA; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 11; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 11; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 11; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 11; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st11;
tr751:
#line 1866 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_AFSDB; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 11; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 11; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 11; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 11; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st11;
tr760:
#line 1876 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_APL; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 11; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 11; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 11; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 11; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 11; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 11; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 11; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 11; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 11; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 11; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 11; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 11; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 11; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 11; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 11; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 11; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 11; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 11; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 11; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 11; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 11; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 11; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 11; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 11; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 11; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 11; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 11; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st11;
st11:
	if ( ++p == pe )
		goto _test_eof11;
case 11:
#line 12557 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto st11;
		case 32: goto st11;
		case 40: goto tr87;
		case 41: goto tr88;
		case 92: goto st9;
		case 1802: goto tr81;
		case 1851: goto st8;
		case 2058: goto tr81;
		case 2107: goto st12;
		case 2314: goto tr90;
		case 2363: goto tr91;
		case 2570: goto tr81;
		case 2619: goto tr92;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr67;
	} else if ( _widec > 58 ) {
		if ( 60 <= _widec )
			goto tr67;
	} else
		goto tr67;
	goto tr77;
tr658:
#line 1856 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_A; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st12;
st12:
	if ( ++p == pe )
		goto _test_eof12;
case 12:
#line 12614 "scanner.c"
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 778: goto tr81;
		case 1034: goto tr93;
	}
	if ( _widec > 895 ) {
		if ( 896 <= _widec && _widec <= 1151 )
			goto st12;
	} else if ( _widec >= 640 )
		goto st8;
	goto tr77;
tr93:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1899 "./scanner_body.rl"
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZSCANNER_ERDATA_OVERFLOW);
			p--; {goto st246;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->process_record(s);
	}
	goto st1057;
tr657:
#line 1856 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_A; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1899 "./scanner_body.rl"
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZSCANNER_ERDATA_OVERFLOW);
			p--; {goto st246;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->process_record(s);
	}
	goto st1057;
st1057:
	if ( ++p == pe )
		goto _test_eof1057;
case 1057:
#line 12688 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr3142;
		case 32: goto tr3142;
		case 36: goto tr3143;
		case 40: goto tr3144;
		case 41: goto tr3145;
		case 42: goto tr3146;
		case 58: goto tr67;
		case 92: goto tr3147;
		case 95: goto tr3146;
		case 1802: goto tr19;
		case 1851: goto st137;
		case 2058: goto tr3148;
		case 2107: goto tr3149;
		case 2314: goto tr115;
		case 2363: goto tr116;
		case 2570: goto tr3150;
		case 2619: goto tr3151;
	}
	if ( _widec < 60 ) {
		if ( _widec < 11 ) {
			if ( _widec <= 8 )
				goto tr67;
		} else if ( _widec > 44 ) {
			if ( 45 <= _widec && _widec <= 57 )
				goto tr3146;
		} else
			goto tr67;
	} else if ( _widec > 63 ) {
		if ( _widec < 91 ) {
			if ( 64 <= _widec && _widec <= 90 )
				goto tr3146;
		} else if ( _widec > 96 ) {
			if ( _widec > 122 ) {
				if ( 123 <= _widec )
					goto tr67;
			} else if ( _widec >= 97 )
				goto tr3146;
		} else
			goto tr67;
	} else
		goto tr67;
	goto tr691;
tr96:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st13;
tr97:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st13;
tr3142:
#line 232 "./scanner_body.rl"
	{
		if (s->r_owner_length == 0) {
			WARN(ZSCANNER_EBAD_PREVIOUS_OWNER);
			p--; {goto st246;}
		}
	}
	goto st13;
tr3144:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
#line 232 "./scanner_body.rl"
	{
		if (s->r_owner_length == 0) {
			WARN(ZSCANNER_EBAD_PREVIOUS_OWNER);
			p--; {goto st246;}
		}
	}
	goto st13;
tr3145:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
#line 232 "./scanner_body.rl"
	{
		if (s->r_owner_length == 0) {
			WARN(ZSCANNER_EBAD_PREVIOUS_OWNER);
			p--; {goto st246;}
		}
	}
	goto st13;
st13:
	if ( ++p == pe )
		goto _test_eof13;
case 13:
#line 12819 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto st13;
		case 32: goto st13;
		case 40: goto tr96;
		case 41: goto tr97;
		case 58: goto tr67;
		case 65: goto tr99;
		case 67: goto tr100;
		case 68: goto tr101;
		case 69: goto tr102;
		case 72: goto tr103;
		case 73: goto tr104;
		case 75: goto tr105;
		case 76: goto tr106;
		case 77: goto tr107;
		case 78: goto tr108;
		case 80: goto tr109;
		case 82: goto tr110;
		case 83: goto tr111;
		case 84: goto tr112;
		case 92: goto st9;
		case 97: goto tr99;
		case 99: goto tr100;
		case 100: goto tr101;
		case 101: goto tr102;
		case 104: goto tr103;
		case 105: goto tr104;
		case 107: goto tr105;
		case 108: goto tr106;
		case 109: goto tr107;
		case 110: goto tr108;
		case 112: goto tr109;
		case 114: goto tr110;
		case 115: goto tr111;
		case 116: goto tr112;
		case 1802: goto tr19;
		case 1851: goto st137;
		case 2058: goto tr113;
		case 2107: goto st222;
		case 2314: goto tr115;
		case 2363: goto tr116;
		case 2570: goto tr117;
		case 2619: goto tr118;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr67;
	} else if ( _widec > 47 ) {
		if ( _widec > 57 ) {
			if ( 60 <= _widec )
				goto tr67;
		} else if ( _widec >= 48 )
			goto tr98;
	} else
		goto tr67;
	goto tr95;
tr98:
#line 278 "./scanner_body.rl"
	{
		s->number64 = 0;
	}
#line 261 "./scanner_body.rl"
	{
		// Overflow check: 10*(s->number64) + fc - ASCII_0 <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + ASCII_0)
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ENUMBER64_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 14; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 14; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 14; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 14; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 14; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 14; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 14; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 14; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 14; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 14; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 14; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 14; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 14; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 14; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 14; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 14; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 14; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 14; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 14; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 14; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 14; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 14; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 14; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 14; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 14; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 14; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 14; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st14;
st14:
	if ( ++p == pe )
		goto _test_eof14;
case 14:
#line 12995 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr120;
		case 32: goto tr120;
		case 40: goto tr121;
		case 41: goto tr122;
		case 68: goto tr28;
		case 72: goto tr29;
		case 77: goto tr30;
		case 83: goto st152;
		case 87: goto tr32;
		case 100: goto tr28;
		case 104: goto tr29;
		case 109: goto tr30;
		case 115: goto st152;
		case 119: goto tr32;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr123;
		case 1083: goto tr124;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr27;
	goto tr119;
tr127:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st15;
tr128:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st15;
tr120:
#line 731 "./scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st15;
tr121:
#line 731 "./scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st15;
tr122:
#line 731 "./scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st15;
st15:
	if ( ++p == pe )
		goto _test_eof15;
case 15:
#line 13106 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st15;
		case 32: goto st15;
		case 40: goto tr127;
		case 41: goto tr128;
		case 65: goto tr39;
		case 67: goto tr40;
		case 68: goto tr41;
		case 69: goto tr42;
		case 72: goto tr43;
		case 73: goto tr44;
		case 75: goto tr45;
		case 76: goto tr46;
		case 77: goto tr47;
		case 78: goto tr48;
		case 80: goto tr49;
		case 82: goto tr50;
		case 83: goto tr51;
		case 84: goto tr52;
		case 97: goto tr39;
		case 99: goto tr40;
		case 100: goto tr41;
		case 101: goto tr42;
		case 104: goto tr43;
		case 105: goto tr44;
		case 107: goto tr45;
		case 108: goto tr46;
		case 109: goto tr47;
		case 110: goto tr48;
		case 112: goto tr49;
		case 114: goto tr50;
		case 115: goto tr51;
		case 116: goto tr52;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr129;
		case 1083: goto st245;
	}
	goto tr125;
tr6:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
	goto st16;
tr40:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
	goto st16;
tr552:
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
	goto st16;
st16:
	if ( ++p == pe )
		goto _test_eof16;
case 16:
#line 13186 "scanner.c"
	switch( (*p) ) {
		case 69: goto st17;
		case 78: goto st21;
		case 101: goto st17;
		case 110: goto st21;
	}
	goto tr35;
st17:
	if ( ++p == pe )
		goto _test_eof17;
case 17:
	switch( (*p) ) {
		case 82: goto st18;
		case 114: goto st18;
	}
	goto tr35;
st18:
	if ( ++p == pe )
		goto _test_eof18;
case 18:
	switch( (*p) ) {
		case 84: goto st19;
		case 116: goto st19;
	}
	goto tr35;
st19:
	if ( ++p == pe )
		goto _test_eof19;
case 19:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr135;
		case 32: goto tr135;
		case 40: goto tr136;
		case 41: goto tr137;
		case 2058: goto tr138;
		case 2107: goto tr139;
		case 2314: goto tr140;
		case 2363: goto tr140;
		case 2570: goto tr141;
		case 2619: goto tr142;
	}
	goto tr55;
tr144:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st20;
tr145:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st20;
tr66:
#line 1856 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_A; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 20; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 20; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 20; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 20; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st20;
tr76:
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 20; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 20; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 20; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 20; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st20;
tr142:
#line 1874 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_CERT; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 20; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 20; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 20; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 20; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st20;
tr156:
#line 1858 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_CNAME; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 20; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 20; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 20; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 20; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st20;
tr170:
#line 1883 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_DHCID; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 20; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 20; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 20; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 20; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st20;
tr182:
#line 1875 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_DNAME; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 20; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 20; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 20; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 20; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st20;
tr193:
#line 1882 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_DNSKEY; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 20; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 20; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 20; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 20; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st20;
tr201:
#line 1877 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_DS; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 20; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 20; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 20; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 20; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st20;
tr214:
#line 1892 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_EUI48; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 20; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 20; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 20; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 20; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st20;
tr223:
#line 1893 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_EUI64; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 20; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 20; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 20; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 20; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st20;
tr235:
#line 1861 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_HINFO; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 20; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 20; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 20; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 20; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st20;
tr275:
#line 1879 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_IPSECKEY; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 20; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 20; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 20; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 20; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st20;
tr286:
#line 1868 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_KEY; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 20; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 20; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 20; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 20; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st20;
tr294:
#line 1873 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_KX; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 20; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 20; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 20; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 20; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st20;
tr307:
#line 1889 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_L32; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 20; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 20; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 20; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 20; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st20;
tr316:
#line 1890 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_L64; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 20; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 20; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 20; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 20; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st20;
tr325:
#line 1870 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_LOC; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 20; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 20; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 20; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 20; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st20;
tr333:
#line 1891 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_LP; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 20; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 20; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 20; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 20; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st20;
tr346:
#line 1862 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_MINFO; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 20; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 20; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 20; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 20; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st20;
tr354:
#line 1863 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_MX; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 20; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 20; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 20; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 20; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st20;
tr368:
#line 1872 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NAPTR; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 20; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 20; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 20; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 20; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st20;
tr377:
#line 1888 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NID; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 20; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 20; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 20; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 20; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st20;
tr386:
#line 1857 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NS; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 20; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 20; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 20; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 20; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st20;
tr396:
#line 1881 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NSEC; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 20; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 20; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 20; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 20; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st20;
tr405:
#line 1884 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NSEC3; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 20; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 20; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 20; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 20; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st20;
tr417:
#line 1885 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NSEC3PARAM; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 20; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 20; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 20; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 20; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st20;
tr427:
#line 1860 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_PTR; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 20; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 20; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 20; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 20; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st20;
tr438:
#line 1865 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_RP; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 20; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 20; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 20; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 20; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st20;
tr449:
#line 1880 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_RRSIG; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 20; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 20; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 20; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 20; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st20;
tr457:
#line 1867 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_RT; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 20; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 20; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 20; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 20; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st20;
tr470:
#line 1859 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_SOA; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 20; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 20; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 20; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 20; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st20;
tr479:
#line 1887 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_SPF; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 20; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 20; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 20; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 20; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st20;
tr488:
#line 1871 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_SRV; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 20; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 20; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 20; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 20; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st20;
tr499:
#line 1878 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_SSHFP; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 20; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 20; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 20; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 20; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st20;
tr512:
#line 1886 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_TLSA; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 20; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 20; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 20; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 20; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st20;
tr521:
#line 1864 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_TXT; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 20; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 20; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 20; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 20; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st20;
tr535:
#line 356 "./scanner_body.rl"
	{
		if (s->number64 <= UINT16_MAX) {
			s->r_type = (uint16_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER16_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 20; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 20; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 20; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 20; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st20;
tr741:
#line 1869 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_AAAA; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 20; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 20; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 20; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 20; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st20;
tr752:
#line 1866 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_AFSDB; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 20; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 20; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 20; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 20; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st20;
tr761:
#line 1876 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_APL; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 20; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 20; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 20; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 20; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 20; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 20; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 20; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 20; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 20; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 20; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 20; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 20; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 20; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 20; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 20; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 20; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 20; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 20; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 20; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 20; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 20; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 20; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 20; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 20; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 20; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 20; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 20; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st20;
st20:
	if ( ++p == pe )
		goto _test_eof20;
case 20:
#line 16554 "scanner.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(640 + ((*p) - -128));
				if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			}
		} else {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 58 ) {
		if ( (*p) > 59 ) {
			if ( 60 <= (*p) )
 {				_widec = (short)(640 + ((*p) - -128));
				if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			}
		} else if ( (*p) >= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 777: goto st7;
		case 778: goto tr81;
		case 800: goto st7;
		case 808: goto tr79;
		case 809: goto tr80;
		case 827: goto st8;
		case 1033: goto st20;
		case 1034: goto tr81;
		case 1056: goto st20;
		case 1064: goto tr144;
		case 1065: goto tr145;
		case 1083: goto st12;
	}
	if ( 896 <= _widec && _widec <= 1151 )
		goto st10;
	goto tr77;
st21:
	if ( ++p == pe )
		goto _test_eof21;
case 21:
	switch( (*p) ) {
		case 65: goto st22;
		case 97: goto st22;
	}
	goto tr35;
st22:
	if ( ++p == pe )
		goto _test_eof22;
case 22:
	switch( (*p) ) {
		case 77: goto st23;
		case 109: goto st23;
	}
	goto tr35;
st23:
	if ( ++p == pe )
		goto _test_eof23;
case 23:
	switch( (*p) ) {
		case 69: goto st24;
		case 101: goto st24;
	}
	goto tr35;
st24:
	if ( ++p == pe )
		goto _test_eof24;
case 24:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr149;
		case 32: goto tr149;
		case 40: goto tr150;
		case 41: goto tr151;
		case 2058: goto tr152;
		case 2107: goto tr153;
		case 2314: goto tr154;
		case 2363: goto tr154;
		case 2570: goto tr155;
		case 2619: goto tr156;
	}
	goto tr55;
tr7:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
	goto st25;
tr41:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
	goto st25;
tr553:
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
	goto st25;
st25:
	if ( ++p == pe )
		goto _test_eof25;
case 25:
#line 16697 "scanner.c"
	switch( (*p) ) {
		case 72: goto st26;
		case 78: goto st30;
		case 83: goto st38;
		case 104: goto st26;
		case 110: goto st30;
		case 115: goto st38;
	}
	goto tr35;
st26:
	if ( ++p == pe )
		goto _test_eof26;
case 26:
	switch( (*p) ) {
		case 67: goto st27;
		case 99: goto st27;
	}
	goto tr35;
st27:
	if ( ++p == pe )
		goto _test_eof27;
case 27:
	switch( (*p) ) {
		case 73: goto st28;
		case 105: goto st28;
	}
	goto tr35;
st28:
	if ( ++p == pe )
		goto _test_eof28;
case 28:
	switch( (*p) ) {
		case 68: goto st29;
		case 100: goto st29;
	}
	goto tr35;
st29:
	if ( ++p == pe )
		goto _test_eof29;
case 29:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr163;
		case 32: goto tr163;
		case 40: goto tr164;
		case 41: goto tr165;
		case 2058: goto tr166;
		case 2107: goto tr167;
		case 2314: goto tr168;
		case 2363: goto tr168;
		case 2570: goto tr169;
		case 2619: goto tr170;
	}
	goto tr55;
st30:
	if ( ++p == pe )
		goto _test_eof30;
case 30:
	switch( (*p) ) {
		case 65: goto st31;
		case 83: goto st34;
		case 97: goto st31;
		case 115: goto st34;
	}
	goto tr35;
st31:
	if ( ++p == pe )
		goto _test_eof31;
case 31:
	switch( (*p) ) {
		case 77: goto st32;
		case 109: goto st32;
	}
	goto tr35;
st32:
	if ( ++p == pe )
		goto _test_eof32;
case 32:
	switch( (*p) ) {
		case 69: goto st33;
		case 101: goto st33;
	}
	goto tr35;
st33:
	if ( ++p == pe )
		goto _test_eof33;
case 33:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr175;
		case 32: goto tr175;
		case 40: goto tr176;
		case 41: goto tr177;
		case 2058: goto tr178;
		case 2107: goto tr179;
		case 2314: goto tr180;
		case 2363: goto tr180;
		case 2570: goto tr181;
		case 2619: goto tr182;
	}
	goto tr55;
st34:
	if ( ++p == pe )
		goto _test_eof34;
case 34:
	switch( (*p) ) {
		case 75: goto st35;
		case 107: goto st35;
	}
	goto tr35;
st35:
	if ( ++p == pe )
		goto _test_eof35;
case 35:
	switch( (*p) ) {
		case 69: goto st36;
		case 101: goto st36;
	}
	goto tr35;
st36:
	if ( ++p == pe )
		goto _test_eof36;
case 36:
	switch( (*p) ) {
		case 89: goto st37;
		case 121: goto st37;
	}
	goto tr35;
st37:
	if ( ++p == pe )
		goto _test_eof37;
case 37:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr186;
		case 32: goto tr186;
		case 40: goto tr187;
		case 41: goto tr188;
		case 2058: goto tr189;
		case 2107: goto tr190;
		case 2314: goto tr191;
		case 2363: goto tr191;
		case 2570: goto tr192;
		case 2619: goto tr193;
	}
	goto tr55;
st38:
	if ( ++p == pe )
		goto _test_eof38;
case 38:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr194;
		case 32: goto tr194;
		case 40: goto tr195;
		case 41: goto tr196;
		case 2058: goto tr197;
		case 2107: goto tr198;
		case 2314: goto tr199;
		case 2363: goto tr199;
		case 2570: goto tr200;
		case 2619: goto tr201;
	}
	goto tr55;
tr8:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
	goto st39;
tr42:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
	goto st39;
tr554:
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
	goto st39;
st39:
	if ( ++p == pe )
		goto _test_eof39;
case 39:
#line 16964 "scanner.c"
	switch( (*p) ) {
		case 85: goto st40;
		case 117: goto st40;
	}
	goto tr35;
st40:
	if ( ++p == pe )
		goto _test_eof40;
case 40:
	switch( (*p) ) {
		case 73: goto st41;
		case 105: goto st41;
	}
	goto tr35;
st41:
	if ( ++p == pe )
		goto _test_eof41;
case 41:
	switch( (*p) ) {
		case 52: goto st42;
		case 54: goto st44;
	}
	goto tr35;
st42:
	if ( ++p == pe )
		goto _test_eof42;
case 42:
	if ( (*p) == 56 )
		goto st43;
	goto tr35;
st43:
	if ( ++p == pe )
		goto _test_eof43;
case 43:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr207;
		case 32: goto tr207;
		case 40: goto tr208;
		case 41: goto tr209;
		case 2058: goto tr210;
		case 2107: goto tr211;
		case 2314: goto tr212;
		case 2363: goto tr212;
		case 2570: goto tr213;
		case 2619: goto tr214;
	}
	goto tr55;
st44:
	if ( ++p == pe )
		goto _test_eof44;
case 44:
	if ( (*p) == 52 )
		goto st45;
	goto tr35;
st45:
	if ( ++p == pe )
		goto _test_eof45;
case 45:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr216;
		case 32: goto tr216;
		case 40: goto tr217;
		case 41: goto tr218;
		case 2058: goto tr219;
		case 2107: goto tr220;
		case 2314: goto tr221;
		case 2363: goto tr221;
		case 2570: goto tr222;
		case 2619: goto tr223;
	}
	goto tr55;
tr9:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
	goto st46;
tr43:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
	goto st46;
tr555:
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
	goto st46;
st46:
	if ( ++p == pe )
		goto _test_eof46;
case 46:
#line 17102 "scanner.c"
	switch( (*p) ) {
		case 73: goto st47;
		case 105: goto st47;
	}
	goto tr35;
st47:
	if ( ++p == pe )
		goto _test_eof47;
case 47:
	switch( (*p) ) {
		case 78: goto st48;
		case 110: goto st48;
	}
	goto tr35;
st48:
	if ( ++p == pe )
		goto _test_eof48;
case 48:
	switch( (*p) ) {
		case 70: goto st49;
		case 102: goto st49;
	}
	goto tr35;
st49:
	if ( ++p == pe )
		goto _test_eof49;
case 49:
	switch( (*p) ) {
		case 79: goto st50;
		case 111: goto st50;
	}
	goto tr35;
st50:
	if ( ++p == pe )
		goto _test_eof50;
case 50:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr228;
		case 32: goto tr228;
		case 40: goto tr229;
		case 41: goto tr230;
		case 2058: goto tr231;
		case 2107: goto tr232;
		case 2314: goto tr233;
		case 2363: goto tr233;
		case 2570: goto tr234;
		case 2619: goto tr235;
	}
	goto tr55;
tr44:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
	goto st51;
st51:
	if ( ++p == pe )
		goto _test_eof51;
case 51:
#line 17182 "scanner.c"
	switch( (*p) ) {
		case 78: goto st52;
		case 80: goto st55;
		case 110: goto st52;
		case 112: goto st55;
	}
	goto tr35;
st52:
	if ( ++p == pe )
		goto _test_eof52;
case 52:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr238;
		case 32: goto tr238;
		case 40: goto tr239;
		case 41: goto tr240;
		case 1034: goto tr241;
		case 1083: goto tr242;
	}
	goto tr35;
tr244:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st53;
tr245:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st53;
tr260:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st53;
tr567:
#line 731 "./scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st53;
tr568:
#line 731 "./scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st53;
tr569:
#line 731 "./scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st53;
tr576:
#line 731 "./scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st53;
tr238:
#line 727 "./scanner_body.rl"
	{
		s->r_class = KNOT_CLASS_IN;
	}
	goto st53;
tr239:
#line 727 "./scanner_body.rl"
	{
		s->r_class = KNOT_CLASS_IN;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st53;
tr240:
#line 727 "./scanner_body.rl"
	{
		s->r_class = KNOT_CLASS_IN;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st53;
tr241:
#line 727 "./scanner_body.rl"
	{
		s->r_class = KNOT_CLASS_IN;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st53;
tr585:
#line 423 "./scanner_body.rl"
	{
		if (s->number64 + s->number64_tmp < UINT32_MAX) {
			s->number64 += s->number64_tmp;
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 731 "./scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st53;
tr586:
#line 423 "./scanner_body.rl"
	{
		if (s->number64 + s->number64_tmp < UINT32_MAX) {
			s->number64 += s->number64_tmp;
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 731 "./scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st53;
tr587:
#line 423 "./scanner_body.rl"
	{
		if (s->number64 + s->number64_tmp < UINT32_MAX) {
			s->number64 += s->number64_tmp;
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 731 "./scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st53;
tr589:
#line 423 "./scanner_body.rl"
	{
		if (s->number64 + s->number64_tmp < UINT32_MAX) {
			s->number64 += s->number64_tmp;
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 731 "./scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st53;
st53:
	if ( ++p == pe )
		goto _test_eof53;
case 53:
#line 17455 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st53;
		case 32: goto st53;
		case 40: goto tr244;
		case 41: goto tr245;
		case 65: goto st4;
		case 67: goto st16;
		case 68: goto st25;
		case 69: goto st39;
		case 72: goto st46;
		case 73: goto st54;
		case 75: goto st62;
		case 76: goto st66;
		case 77: goto st74;
		case 78: goto st80;
		case 80: goto st96;
		case 82: goto st99;
		case 83: goto st106;
		case 84: goto st117;
		case 97: goto st4;
		case 99: goto st16;
		case 100: goto st25;
		case 101: goto st39;
		case 104: goto st46;
		case 105: goto st54;
		case 107: goto st62;
		case 108: goto st66;
		case 109: goto st74;
		case 110: goto st80;
		case 112: goto st96;
		case 114: goto st99;
		case 115: goto st106;
		case 116: goto st117;
		case 1034: goto tr260;
		case 1083: goto st127;
	}
	goto tr35;
tr556:
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
	goto st54;
st54:
	if ( ++p == pe )
		goto _test_eof54;
case 54:
#line 17517 "scanner.c"
	switch( (*p) ) {
		case 80: goto st55;
		case 112: goto st55;
	}
	goto tr35;
st55:
	if ( ++p == pe )
		goto _test_eof55;
case 55:
	switch( (*p) ) {
		case 83: goto st56;
		case 115: goto st56;
	}
	goto tr35;
st56:
	if ( ++p == pe )
		goto _test_eof56;
case 56:
	switch( (*p) ) {
		case 69: goto st57;
		case 101: goto st57;
	}
	goto tr35;
st57:
	if ( ++p == pe )
		goto _test_eof57;
case 57:
	switch( (*p) ) {
		case 67: goto st58;
		case 99: goto st58;
	}
	goto tr35;
st58:
	if ( ++p == pe )
		goto _test_eof58;
case 58:
	switch( (*p) ) {
		case 75: goto st59;
		case 107: goto st59;
	}
	goto tr35;
st59:
	if ( ++p == pe )
		goto _test_eof59;
case 59:
	switch( (*p) ) {
		case 69: goto st60;
		case 101: goto st60;
	}
	goto tr35;
st60:
	if ( ++p == pe )
		goto _test_eof60;
case 60:
	switch( (*p) ) {
		case 89: goto st61;
		case 121: goto st61;
	}
	goto tr35;
st61:
	if ( ++p == pe )
		goto _test_eof61;
case 61:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr268;
		case 32: goto tr268;
		case 40: goto tr269;
		case 41: goto tr270;
		case 2058: goto tr271;
		case 2107: goto tr272;
		case 2314: goto tr273;
		case 2363: goto tr273;
		case 2570: goto tr274;
		case 2619: goto tr275;
	}
	goto tr55;
tr11:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
	goto st62;
tr45:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
	goto st62;
tr557:
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
	goto st62;
st62:
	if ( ++p == pe )
		goto _test_eof62;
case 62:
#line 17640 "scanner.c"
	switch( (*p) ) {
		case 69: goto st63;
		case 88: goto st65;
		case 101: goto st63;
		case 120: goto st65;
	}
	goto tr35;
st63:
	if ( ++p == pe )
		goto _test_eof63;
case 63:
	switch( (*p) ) {
		case 89: goto st64;
		case 121: goto st64;
	}
	goto tr35;
st64:
	if ( ++p == pe )
		goto _test_eof64;
case 64:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr279;
		case 32: goto tr279;
		case 40: goto tr280;
		case 41: goto tr281;
		case 2058: goto tr282;
		case 2107: goto tr283;
		case 2314: goto tr284;
		case 2363: goto tr284;
		case 2570: goto tr285;
		case 2619: goto tr286;
	}
	goto tr55;
st65:
	if ( ++p == pe )
		goto _test_eof65;
case 65:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr287;
		case 32: goto tr287;
		case 40: goto tr288;
		case 41: goto tr289;
		case 2058: goto tr290;
		case 2107: goto tr291;
		case 2314: goto tr292;
		case 2363: goto tr292;
		case 2570: goto tr293;
		case 2619: goto tr294;
	}
	goto tr55;
tr12:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
	goto st66;
tr46:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
	goto st66;
tr558:
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
	goto st66;
st66:
	if ( ++p == pe )
		goto _test_eof66;
case 66:
#line 17757 "scanner.c"
	switch( (*p) ) {
		case 51: goto st67;
		case 54: goto st69;
		case 79: goto st71;
		case 80: goto st73;
		case 111: goto st71;
		case 112: goto st73;
	}
	goto tr35;
st67:
	if ( ++p == pe )
		goto _test_eof67;
case 67:
	if ( (*p) == 50 )
		goto st68;
	goto tr35;
st68:
	if ( ++p == pe )
		goto _test_eof68;
case 68:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr300;
		case 32: goto tr300;
		case 40: goto tr301;
		case 41: goto tr302;
		case 2058: goto tr303;
		case 2107: goto tr304;
		case 2314: goto tr305;
		case 2363: goto tr305;
		case 2570: goto tr306;
		case 2619: goto tr307;
	}
	goto tr55;
st69:
	if ( ++p == pe )
		goto _test_eof69;
case 69:
	if ( (*p) == 52 )
		goto st70;
	goto tr35;
st70:
	if ( ++p == pe )
		goto _test_eof70;
case 70:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr309;
		case 32: goto tr309;
		case 40: goto tr310;
		case 41: goto tr311;
		case 2058: goto tr312;
		case 2107: goto tr313;
		case 2314: goto tr314;
		case 2363: goto tr314;
		case 2570: goto tr315;
		case 2619: goto tr316;
	}
	goto tr55;
st71:
	if ( ++p == pe )
		goto _test_eof71;
case 71:
	switch( (*p) ) {
		case 67: goto st72;
		case 99: goto st72;
	}
	goto tr35;
st72:
	if ( ++p == pe )
		goto _test_eof72;
case 72:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr318;
		case 32: goto tr318;
		case 40: goto tr319;
		case 41: goto tr320;
		case 2058: goto tr321;
		case 2107: goto tr322;
		case 2314: goto tr323;
		case 2363: goto tr323;
		case 2570: goto tr324;
		case 2619: goto tr325;
	}
	goto tr55;
st73:
	if ( ++p == pe )
		goto _test_eof73;
case 73:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr326;
		case 32: goto tr326;
		case 40: goto tr327;
		case 41: goto tr328;
		case 2058: goto tr329;
		case 2107: goto tr330;
		case 2314: goto tr331;
		case 2363: goto tr331;
		case 2570: goto tr332;
		case 2619: goto tr333;
	}
	goto tr55;
tr13:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
	goto st74;
tr47:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
	goto st74;
tr559:
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
	goto st74;
st74:
	if ( ++p == pe )
		goto _test_eof74;
case 74:
#line 17964 "scanner.c"
	switch( (*p) ) {
		case 73: goto st75;
		case 88: goto st79;
		case 105: goto st75;
		case 120: goto st79;
	}
	goto tr35;
st75:
	if ( ++p == pe )
		goto _test_eof75;
case 75:
	switch( (*p) ) {
		case 78: goto st76;
		case 110: goto st76;
	}
	goto tr35;
st76:
	if ( ++p == pe )
		goto _test_eof76;
case 76:
	switch( (*p) ) {
		case 70: goto st77;
		case 102: goto st77;
	}
	goto tr35;
st77:
	if ( ++p == pe )
		goto _test_eof77;
case 77:
	switch( (*p) ) {
		case 79: goto st78;
		case 111: goto st78;
	}
	goto tr35;
st78:
	if ( ++p == pe )
		goto _test_eof78;
case 78:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr339;
		case 32: goto tr339;
		case 40: goto tr340;
		case 41: goto tr341;
		case 2058: goto tr342;
		case 2107: goto tr343;
		case 2314: goto tr344;
		case 2363: goto tr344;
		case 2570: goto tr345;
		case 2619: goto tr346;
	}
	goto tr55;
st79:
	if ( ++p == pe )
		goto _test_eof79;
case 79:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr347;
		case 32: goto tr347;
		case 40: goto tr348;
		case 41: goto tr349;
		case 2058: goto tr350;
		case 2107: goto tr351;
		case 2314: goto tr352;
		case 2363: goto tr352;
		case 2570: goto tr353;
		case 2619: goto tr354;
	}
	goto tr55;
tr14:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
	goto st80;
tr48:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
	goto st80;
tr560:
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
	goto st80;
st80:
	if ( ++p == pe )
		goto _test_eof80;
case 80:
#line 18099 "scanner.c"
	switch( (*p) ) {
		case 65: goto st81;
		case 73: goto st85;
		case 83: goto st87;
		case 97: goto st81;
		case 105: goto st85;
		case 115: goto st87;
	}
	goto tr35;
st81:
	if ( ++p == pe )
		goto _test_eof81;
case 81:
	switch( (*p) ) {
		case 80: goto st82;
		case 112: goto st82;
	}
	goto tr35;
st82:
	if ( ++p == pe )
		goto _test_eof82;
case 82:
	switch( (*p) ) {
		case 84: goto st83;
		case 116: goto st83;
	}
	goto tr35;
st83:
	if ( ++p == pe )
		goto _test_eof83;
case 83:
	switch( (*p) ) {
		case 82: goto st84;
		case 114: goto st84;
	}
	goto tr35;
st84:
	if ( ++p == pe )
		goto _test_eof84;
case 84:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr361;
		case 32: goto tr361;
		case 40: goto tr362;
		case 41: goto tr363;
		case 2058: goto tr364;
		case 2107: goto tr365;
		case 2314: goto tr366;
		case 2363: goto tr366;
		case 2570: goto tr367;
		case 2619: goto tr368;
	}
	goto tr55;
st85:
	if ( ++p == pe )
		goto _test_eof85;
case 85:
	switch( (*p) ) {
		case 68: goto st86;
		case 100: goto st86;
	}
	goto tr35;
st86:
	if ( ++p == pe )
		goto _test_eof86;
case 86:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr370;
		case 32: goto tr370;
		case 40: goto tr371;
		case 41: goto tr372;
		case 2058: goto tr373;
		case 2107: goto tr374;
		case 2314: goto tr375;
		case 2363: goto tr375;
		case 2570: goto tr376;
		case 2619: goto tr377;
	}
	goto tr55;
st87:
	if ( ++p == pe )
		goto _test_eof87;
case 87:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr378;
		case 32: goto tr378;
		case 40: goto tr379;
		case 41: goto tr380;
		case 69: goto st88;
		case 101: goto st88;
		case 2058: goto tr382;
		case 2107: goto tr383;
		case 2314: goto tr384;
		case 2363: goto tr384;
		case 2570: goto tr385;
		case 2619: goto tr386;
	}
	goto tr55;
st88:
	if ( ++p == pe )
		goto _test_eof88;
case 88:
	switch( (*p) ) {
		case 67: goto st89;
		case 99: goto st89;
	}
	goto tr35;
st89:
	if ( ++p == pe )
		goto _test_eof89;
case 89:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr388;
		case 32: goto tr388;
		case 40: goto tr389;
		case 41: goto tr390;
		case 51: goto st90;
		case 2058: goto tr392;
		case 2107: goto tr393;
		case 2314: goto tr394;
		case 2363: goto tr394;
		case 2570: goto tr395;
		case 2619: goto tr396;
	}
	goto tr55;
st90:
	if ( ++p == pe )
		goto _test_eof90;
case 90:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr397;
		case 32: goto tr397;
		case 40: goto tr398;
		case 41: goto tr399;
		case 80: goto st91;
		case 112: goto st91;
		case 2058: goto tr401;
		case 2107: goto tr402;
		case 2314: goto tr403;
		case 2363: goto tr403;
		case 2570: goto tr404;
		case 2619: goto tr405;
	}
	goto tr55;
st91:
	if ( ++p == pe )
		goto _test_eof91;
case 91:
	switch( (*p) ) {
		case 65: goto st92;
		case 97: goto st92;
	}
	goto tr35;
st92:
	if ( ++p == pe )
		goto _test_eof92;
case 92:
	switch( (*p) ) {
		case 82: goto st93;
		case 114: goto st93;
	}
	goto tr35;
st93:
	if ( ++p == pe )
		goto _test_eof93;
case 93:
	switch( (*p) ) {
		case 65: goto st94;
		case 97: goto st94;
	}
	goto tr35;
st94:
	if ( ++p == pe )
		goto _test_eof94;
case 94:
	switch( (*p) ) {
		case 77: goto st95;
		case 109: goto st95;
	}
	goto tr35;
st95:
	if ( ++p == pe )
		goto _test_eof95;
case 95:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr410;
		case 32: goto tr410;
		case 40: goto tr411;
		case 41: goto tr412;
		case 2058: goto tr413;
		case 2107: goto tr414;
		case 2314: goto tr415;
		case 2363: goto tr415;
		case 2570: goto tr416;
		case 2619: goto tr417;
	}
	goto tr55;
tr15:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
	goto st96;
tr49:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
	goto st96;
tr561:
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
	goto st96;
st96:
	if ( ++p == pe )
		goto _test_eof96;
case 96:
#line 18443 "scanner.c"
	switch( (*p) ) {
		case 84: goto st97;
		case 116: goto st97;
	}
	goto tr35;
st97:
	if ( ++p == pe )
		goto _test_eof97;
case 97:
	switch( (*p) ) {
		case 82: goto st98;
		case 114: goto st98;
	}
	goto tr35;
st98:
	if ( ++p == pe )
		goto _test_eof98;
case 98:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr420;
		case 32: goto tr420;
		case 40: goto tr421;
		case 41: goto tr422;
		case 2058: goto tr423;
		case 2107: goto tr424;
		case 2314: goto tr425;
		case 2363: goto tr425;
		case 2570: goto tr426;
		case 2619: goto tr427;
	}
	goto tr55;
tr16:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
	goto st99;
tr50:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
	goto st99;
tr562:
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
	goto st99;
st99:
	if ( ++p == pe )
		goto _test_eof99;
case 99:
#line 18521 "scanner.c"
	switch( (*p) ) {
		case 80: goto st100;
		case 82: goto st101;
		case 84: goto st105;
		case 112: goto st100;
		case 114: goto st101;
		case 116: goto st105;
	}
	goto tr35;
st100:
	if ( ++p == pe )
		goto _test_eof100;
case 100:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr431;
		case 32: goto tr431;
		case 40: goto tr432;
		case 41: goto tr433;
		case 2058: goto tr434;
		case 2107: goto tr435;
		case 2314: goto tr436;
		case 2363: goto tr436;
		case 2570: goto tr437;
		case 2619: goto tr438;
	}
	goto tr55;
st101:
	if ( ++p == pe )
		goto _test_eof101;
case 101:
	switch( (*p) ) {
		case 83: goto st102;
		case 115: goto st102;
	}
	goto tr35;
st102:
	if ( ++p == pe )
		goto _test_eof102;
case 102:
	switch( (*p) ) {
		case 73: goto st103;
		case 105: goto st103;
	}
	goto tr35;
st103:
	if ( ++p == pe )
		goto _test_eof103;
case 103:
	switch( (*p) ) {
		case 71: goto st104;
		case 103: goto st104;
	}
	goto tr35;
st104:
	if ( ++p == pe )
		goto _test_eof104;
case 104:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr442;
		case 32: goto tr442;
		case 40: goto tr443;
		case 41: goto tr444;
		case 2058: goto tr445;
		case 2107: goto tr446;
		case 2314: goto tr447;
		case 2363: goto tr447;
		case 2570: goto tr448;
		case 2619: goto tr449;
	}
	goto tr55;
st105:
	if ( ++p == pe )
		goto _test_eof105;
case 105:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr450;
		case 32: goto tr450;
		case 40: goto tr451;
		case 41: goto tr452;
		case 2058: goto tr453;
		case 2107: goto tr454;
		case 2314: goto tr455;
		case 2363: goto tr455;
		case 2570: goto tr456;
		case 2619: goto tr457;
	}
	goto tr55;
tr17:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
	goto st106;
tr51:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
	goto st106;
tr563:
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
	goto st106;
st106:
	if ( ++p == pe )
		goto _test_eof106;
case 106:
#line 18695 "scanner.c"
	switch( (*p) ) {
		case 79: goto st107;
		case 80: goto st109;
		case 82: goto st111;
		case 83: goto st113;
		case 111: goto st107;
		case 112: goto st109;
		case 114: goto st111;
		case 115: goto st113;
	}
	goto tr35;
st107:
	if ( ++p == pe )
		goto _test_eof107;
case 107:
	switch( (*p) ) {
		case 65: goto st108;
		case 97: goto st108;
	}
	goto tr35;
st108:
	if ( ++p == pe )
		goto _test_eof108;
case 108:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr463;
		case 32: goto tr463;
		case 40: goto tr464;
		case 41: goto tr465;
		case 2058: goto tr466;
		case 2107: goto tr467;
		case 2314: goto tr468;
		case 2363: goto tr468;
		case 2570: goto tr469;
		case 2619: goto tr470;
	}
	goto tr55;
st109:
	if ( ++p == pe )
		goto _test_eof109;
case 109:
	switch( (*p) ) {
		case 70: goto st110;
		case 102: goto st110;
	}
	goto tr35;
st110:
	if ( ++p == pe )
		goto _test_eof110;
case 110:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr472;
		case 32: goto tr472;
		case 40: goto tr473;
		case 41: goto tr474;
		case 2058: goto tr475;
		case 2107: goto tr476;
		case 2314: goto tr477;
		case 2363: goto tr477;
		case 2570: goto tr478;
		case 2619: goto tr479;
	}
	goto tr55;
st111:
	if ( ++p == pe )
		goto _test_eof111;
case 111:
	switch( (*p) ) {
		case 86: goto st112;
		case 118: goto st112;
	}
	goto tr35;
st112:
	if ( ++p == pe )
		goto _test_eof112;
case 112:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr481;
		case 32: goto tr481;
		case 40: goto tr482;
		case 41: goto tr483;
		case 2058: goto tr484;
		case 2107: goto tr485;
		case 2314: goto tr486;
		case 2363: goto tr486;
		case 2570: goto tr487;
		case 2619: goto tr488;
	}
	goto tr55;
st113:
	if ( ++p == pe )
		goto _test_eof113;
case 113:
	switch( (*p) ) {
		case 72: goto st114;
		case 104: goto st114;
	}
	goto tr35;
st114:
	if ( ++p == pe )
		goto _test_eof114;
case 114:
	switch( (*p) ) {
		case 70: goto st115;
		case 102: goto st115;
	}
	goto tr35;
st115:
	if ( ++p == pe )
		goto _test_eof115;
case 115:
	switch( (*p) ) {
		case 80: goto st116;
		case 112: goto st116;
	}
	goto tr35;
st116:
	if ( ++p == pe )
		goto _test_eof116;
case 116:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr492;
		case 32: goto tr492;
		case 40: goto tr493;
		case 41: goto tr494;
		case 2058: goto tr495;
		case 2107: goto tr496;
		case 2314: goto tr497;
		case 2363: goto tr497;
		case 2570: goto tr498;
		case 2619: goto tr499;
	}
	goto tr55;
tr18:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
	goto st117;
tr52:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
	goto st117;
tr564:
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
	goto st117;
st117:
	if ( ++p == pe )
		goto _test_eof117;
case 117:
#line 18935 "scanner.c"
	switch( (*p) ) {
		case 76: goto st118;
		case 88: goto st121;
		case 89: goto st123;
		case 108: goto st118;
		case 120: goto st121;
		case 121: goto st123;
	}
	goto tr35;
st118:
	if ( ++p == pe )
		goto _test_eof118;
case 118:
	switch( (*p) ) {
		case 83: goto st119;
		case 115: goto st119;
	}
	goto tr35;
st119:
	if ( ++p == pe )
		goto _test_eof119;
case 119:
	switch( (*p) ) {
		case 65: goto st120;
		case 97: goto st120;
	}
	goto tr35;
st120:
	if ( ++p == pe )
		goto _test_eof120;
case 120:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr505;
		case 32: goto tr505;
		case 40: goto tr506;
		case 41: goto tr507;
		case 2058: goto tr508;
		case 2107: goto tr509;
		case 2314: goto tr510;
		case 2363: goto tr510;
		case 2570: goto tr511;
		case 2619: goto tr512;
	}
	goto tr55;
st121:
	if ( ++p == pe )
		goto _test_eof121;
case 121:
	switch( (*p) ) {
		case 84: goto st122;
		case 116: goto st122;
	}
	goto tr35;
st122:
	if ( ++p == pe )
		goto _test_eof122;
case 122:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr514;
		case 32: goto tr514;
		case 40: goto tr515;
		case 41: goto tr516;
		case 2058: goto tr517;
		case 2107: goto tr518;
		case 2314: goto tr519;
		case 2363: goto tr519;
		case 2570: goto tr520;
		case 2619: goto tr521;
	}
	goto tr55;
st123:
	if ( ++p == pe )
		goto _test_eof123;
case 123:
	switch( (*p) ) {
		case 80: goto st124;
		case 112: goto st124;
	}
	goto tr35;
st124:
	if ( ++p == pe )
		goto _test_eof124;
case 124:
	switch( (*p) ) {
		case 69: goto st125;
		case 101: goto st125;
	}
	goto tr35;
st125:
	if ( ++p == pe )
		goto _test_eof125;
case 125:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto tr525;
	goto tr524;
tr525:
#line 278 "./scanner_body.rl"
	{
		s->number64 = 0;
	}
#line 261 "./scanner_body.rl"
	{
		// Overflow check: 10*(s->number64) + fc - ASCII_0 <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + ASCII_0)
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ENUMBER64_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st126;
tr530:
#line 261 "./scanner_body.rl"
	{
		// Overflow check: 10*(s->number64) + fc - ASCII_0 <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + ASCII_0)
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ENUMBER64_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st126;
st126:
	if ( ++p == pe )
		goto _test_eof126;
case 126:
#line 19113 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr527;
		case 32: goto tr527;
		case 40: goto tr528;
		case 41: goto tr529;
		case 2058: goto tr531;
		case 2107: goto tr532;
		case 2314: goto tr533;
		case 2363: goto tr533;
		case 2570: goto tr534;
		case 2619: goto tr535;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr530;
	goto tr526;
tr577:
#line 731 "./scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st127;
tr242:
#line 727 "./scanner_body.rl"
	{
		s->r_class = KNOT_CLASS_IN;
	}
	goto st127;
tr590:
#line 423 "./scanner_body.rl"
	{
		if (s->number64 + s->number64_tmp < UINT32_MAX) {
			s->number64 += s->number64_tmp;
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 731 "./scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st127;
st127:
	if ( ++p == pe )
		goto _test_eof127;
case 127:
#line 19190 "scanner.c"
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	if ( _widec == 1034 )
		goto tr260;
	if ( 896 <= _widec && _widec <= 1151 )
		goto st127;
	goto tr35;
tr129:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1899 "./scanner_body.rl"
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZSCANNER_ERDATA_OVERFLOW);
			p--; {goto st246;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->process_record(s);
	}
	goto st1058;
tr123:
#line 731 "./scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1899 "./scanner_body.rl"
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZSCANNER_ERDATA_OVERFLOW);
			p--; {goto st246;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->process_record(s);
	}
	goto st1058;
st1058:
	if ( ++p == pe )
		goto _test_eof1058;
case 1058:
#line 19262 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr3153;
		case 32: goto tr3153;
		case 36: goto st138;
		case 40: goto tr3154;
		case 41: goto tr3155;
		case 42: goto tr3139;
		case 65: goto tr3156;
		case 67: goto tr3157;
		case 68: goto tr3158;
		case 69: goto tr3159;
		case 72: goto tr3160;
		case 73: goto tr3161;
		case 75: goto tr3162;
		case 76: goto tr3163;
		case 77: goto tr3164;
		case 78: goto tr3165;
		case 80: goto tr3166;
		case 82: goto tr3167;
		case 83: goto tr3168;
		case 84: goto tr3169;
		case 92: goto tr3139;
		case 95: goto tr3139;
		case 97: goto tr3156;
		case 99: goto tr3157;
		case 100: goto tr3158;
		case 101: goto tr3159;
		case 104: goto tr3160;
		case 105: goto tr3161;
		case 107: goto tr3162;
		case 108: goto tr3163;
		case 109: goto tr3164;
		case 110: goto tr3165;
		case 112: goto tr3166;
		case 114: goto tr3167;
		case 115: goto tr3168;
		case 116: goto tr3169;
		case 778: goto tr19;
		case 827: goto st137;
		case 1034: goto tr3170;
		case 1083: goto tr3171;
	}
	if ( _widec < 64 ) {
		if ( 45 <= _widec && _widec <= 57 )
			goto tr3139;
	} else if ( _widec > 90 ) {
		if ( 98 <= _widec && _widec <= 122 )
			goto tr3139;
	} else
		goto tr3139;
	goto tr3152;
tr537:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st128;
tr538:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st128;
tr3153:
#line 232 "./scanner_body.rl"
	{
		if (s->r_owner_length == 0) {
			WARN(ZSCANNER_EBAD_PREVIOUS_OWNER);
			p--; {goto st246;}
		}
	}
	goto st128;
tr3154:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
#line 232 "./scanner_body.rl"
	{
		if (s->r_owner_length == 0) {
			WARN(ZSCANNER_EBAD_PREVIOUS_OWNER);
			p--; {goto st246;}
		}
	}
	goto st128;
tr3155:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
#line 232 "./scanner_body.rl"
	{
		if (s->r_owner_length == 0) {
			WARN(ZSCANNER_EBAD_PREVIOUS_OWNER);
			p--; {goto st246;}
		}
	}
	goto st128;
st128:
	if ( ++p == pe )
		goto _test_eof128;
case 128:
#line 19395 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st128;
		case 32: goto st128;
		case 40: goto tr537;
		case 41: goto tr538;
		case 65: goto tr5;
		case 67: goto tr6;
		case 68: goto tr7;
		case 69: goto tr8;
		case 72: goto tr9;
		case 73: goto tr10;
		case 75: goto tr11;
		case 76: goto tr12;
		case 77: goto tr13;
		case 78: goto tr14;
		case 80: goto tr15;
		case 82: goto tr16;
		case 83: goto tr17;
		case 84: goto tr18;
		case 97: goto tr5;
		case 99: goto tr6;
		case 100: goto tr7;
		case 101: goto tr8;
		case 104: goto tr9;
		case 105: goto tr10;
		case 107: goto tr11;
		case 108: goto tr12;
		case 109: goto tr13;
		case 110: goto tr14;
		case 112: goto tr15;
		case 114: goto tr16;
		case 115: goto tr17;
		case 116: goto tr18;
		case 778: goto tr19;
		case 827: goto st137;
		case 1034: goto tr539;
		case 1083: goto st243;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr4;
	goto tr0;
tr10:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
	goto st129;
st129:
	if ( ++p == pe )
		goto _test_eof129;
case 129:
#line 19465 "scanner.c"
	switch( (*p) ) {
		case 78: goto st130;
		case 80: goto st55;
		case 110: goto st130;
		case 112: goto st55;
	}
	goto tr35;
st130:
	if ( ++p == pe )
		goto _test_eof130;
case 130:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr542;
		case 32: goto tr542;
		case 40: goto tr543;
		case 41: goto tr544;
		case 1034: goto tr545;
		case 1083: goto tr546;
	}
	goto tr35;
tr548:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st131;
tr549:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st131;
tr565:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st131;
tr542:
#line 727 "./scanner_body.rl"
	{
		s->r_class = KNOT_CLASS_IN;
	}
	goto st131;
tr543:
#line 727 "./scanner_body.rl"
	{
		s->r_class = KNOT_CLASS_IN;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st131;
tr544:
#line 727 "./scanner_body.rl"
	{
		s->r_class = KNOT_CLASS_IN;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st131;
tr545:
#line 727 "./scanner_body.rl"
	{
		s->r_class = KNOT_CLASS_IN;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st131;
st131:
	if ( ++p == pe )
		goto _test_eof131;
case 131:
#line 19574 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st131;
		case 32: goto st131;
		case 40: goto tr548;
		case 41: goto tr549;
		case 65: goto tr551;
		case 67: goto tr552;
		case 68: goto tr553;
		case 69: goto tr554;
		case 72: goto tr555;
		case 73: goto tr556;
		case 75: goto tr557;
		case 76: goto tr558;
		case 77: goto tr559;
		case 78: goto tr560;
		case 80: goto tr561;
		case 82: goto tr562;
		case 83: goto tr563;
		case 84: goto tr564;
		case 97: goto tr551;
		case 99: goto tr552;
		case 100: goto tr553;
		case 101: goto tr554;
		case 104: goto tr555;
		case 105: goto tr556;
		case 107: goto tr557;
		case 108: goto tr558;
		case 109: goto tr559;
		case 110: goto tr560;
		case 112: goto tr561;
		case 114: goto tr562;
		case 115: goto tr563;
		case 116: goto tr564;
		case 1034: goto tr565;
		case 1083: goto st136;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr550;
	goto tr524;
tr550:
#line 278 "./scanner_body.rl"
	{
		s->number64 = 0;
	}
#line 261 "./scanner_body.rl"
	{
		// Overflow check: 10*(s->number64) + fc - ASCII_0 <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + ASCII_0)
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ENUMBER64_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st132;
tr570:
#line 261 "./scanner_body.rl"
	{
		// Overflow check: 10*(s->number64) + fc - ASCII_0 <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + ASCII_0)
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ENUMBER64_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st132;
st132:
	if ( ++p == pe )
		goto _test_eof132;
case 132:
#line 19670 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr567;
		case 32: goto tr567;
		case 40: goto tr568;
		case 41: goto tr569;
		case 68: goto tr571;
		case 72: goto tr572;
		case 77: goto tr573;
		case 83: goto st133;
		case 87: goto tr575;
		case 100: goto tr571;
		case 104: goto tr572;
		case 109: goto tr573;
		case 115: goto st133;
		case 119: goto tr575;
		case 1034: goto tr576;
		case 1083: goto tr577;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr570;
	goto tr23;
tr571:
#line 403 "./scanner_body.rl"
	{ if (s->number64 <= (UINT32_MAX / 86400)) {
	                  s->number64 *= 86400;
	              } else {
	                  WARN(ZSCANNER_ENUMBER32_OVERFLOW);
	                  p--; {goto st246;}
	              }
	            }
	goto st133;
tr572:
#line 396 "./scanner_body.rl"
	{ if (s->number64 <= (UINT32_MAX / 3600)) {
	                  s->number64 *= 3600;
	              } else {
	                  WARN(ZSCANNER_ENUMBER32_OVERFLOW);
	                  p--; {goto st246;}
	              }
	            }
	goto st133;
tr573:
#line 389 "./scanner_body.rl"
	{ if (s->number64 <= (UINT32_MAX / 60)) {
	                  s->number64 *= 60;
	              } else {
	                  WARN(ZSCANNER_ENUMBER32_OVERFLOW);
	                  p--; {goto st246;}
	              }
	            }
	goto st133;
tr575:
#line 410 "./scanner_body.rl"
	{ if (s->number64 <= (UINT32_MAX / 604800)) {
	                  s->number64 *= 604800;
	              } else {
	                  WARN(ZSCANNER_ENUMBER32_OVERFLOW);
	                  p--; {goto st246;}
	              }
	            }
	goto st133;
st133:
	if ( ++p == pe )
		goto _test_eof133;
case 133:
#line 19750 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr567;
		case 32: goto tr567;
		case 40: goto tr568;
		case 41: goto tr569;
		case 1034: goto tr576;
		case 1083: goto tr577;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr578;
	goto tr23;
tr579:
#line 261 "./scanner_body.rl"
	{
		// Overflow check: 10*(s->number64) + fc - ASCII_0 <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + ASCII_0)
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ENUMBER64_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st134;
tr578:
#line 420 "./scanner_body.rl"
	{
		s->number64_tmp = s->number64;
	}
#line 278 "./scanner_body.rl"
	{
		s->number64 = 0;
	}
#line 261 "./scanner_body.rl"
	{
		// Overflow check: 10*(s->number64) + fc - ASCII_0 <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + ASCII_0)
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ENUMBER64_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st134;
tr588:
#line 423 "./scanner_body.rl"
	{
		if (s->number64 + s->number64_tmp < UINT32_MAX) {
			s->number64 += s->number64_tmp;
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 420 "./scanner_body.rl"
	{
		s->number64_tmp = s->number64;
	}
#line 278 "./scanner_body.rl"
	{
		s->number64 = 0;
	}
#line 261 "./scanner_body.rl"
	{
		// Overflow check: 10*(s->number64) + fc - ASCII_0 <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + ASCII_0)
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ENUMBER64_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st134;
st134:
	if ( ++p == pe )
		goto _test_eof134;
case 134:
#line 19856 "scanner.c"
	switch( (*p) ) {
		case 68: goto tr580;
		case 72: goto tr581;
		case 77: goto tr582;
		case 83: goto st135;
		case 87: goto tr584;
		case 100: goto tr580;
		case 104: goto tr581;
		case 109: goto tr582;
		case 115: goto st135;
		case 119: goto tr584;
	}
	if ( 48 <= (*p) && (*p) <= 57 )
		goto tr579;
	goto tr23;
tr580:
#line 403 "./scanner_body.rl"
	{ if (s->number64 <= (UINT32_MAX / 86400)) {
	                  s->number64 *= 86400;
	              } else {
	                  WARN(ZSCANNER_ENUMBER32_OVERFLOW);
	                  p--; {goto st246;}
	              }
	            }
	goto st135;
tr581:
#line 396 "./scanner_body.rl"
	{ if (s->number64 <= (UINT32_MAX / 3600)) {
	                  s->number64 *= 3600;
	              } else {
	                  WARN(ZSCANNER_ENUMBER32_OVERFLOW);
	                  p--; {goto st246;}
	              }
	            }
	goto st135;
tr582:
#line 389 "./scanner_body.rl"
	{ if (s->number64 <= (UINT32_MAX / 60)) {
	                  s->number64 *= 60;
	              } else {
	                  WARN(ZSCANNER_ENUMBER32_OVERFLOW);
	                  p--; {goto st246;}
	              }
	            }
	goto st135;
tr584:
#line 410 "./scanner_body.rl"
	{ if (s->number64 <= (UINT32_MAX / 604800)) {
	                  s->number64 *= 604800;
	              } else {
	                  WARN(ZSCANNER_ENUMBER32_OVERFLOW);
	                  p--; {goto st246;}
	              }
	            }
	goto st135;
st135:
	if ( ++p == pe )
		goto _test_eof135;
case 135:
#line 19916 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr585;
		case 32: goto tr585;
		case 40: goto tr586;
		case 41: goto tr587;
		case 1034: goto tr589;
		case 1083: goto tr590;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr588;
	goto tr23;
tr546:
#line 727 "./scanner_body.rl"
	{
		s->r_class = KNOT_CLASS_IN;
	}
	goto st136;
st136:
	if ( ++p == pe )
		goto _test_eof136;
case 136:
#line 19952 "scanner.c"
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	if ( _widec == 1034 )
		goto tr565;
	if ( 896 <= _widec && _widec <= 1151 )
		goto st136;
	goto tr35;
tr3195:
#line 704 "./scanner_body.rl"
	{
		s->stop = false;
	}
	goto st137;
st137:
	if ( ++p == pe )
		goto _test_eof137;
case 137:
#line 19989 "scanner.c"
	if ( (*p) == 10 )
		goto tr19;
	goto st137;
tr539:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st1059;
tr706:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1899 "./scanner_body.rl"
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZSCANNER_ERDATA_OVERFLOW);
			p--; {goto st246;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->process_record(s);
	}
	goto st1059;
tr701:
#line 229 "./scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 731 "./scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 1899 "./scanner_body.rl"
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZSCANNER_ERDATA_OVERFLOW);
			p--; {goto st246;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->process_record(s);
	}
	goto st1059;
tr3170:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 232 "./scanner_body.rl"
	{
		if (s->r_owner_length == 0) {
			WARN(ZSCANNER_EBAD_PREVIOUS_OWNER);
			p--; {goto st246;}
		}
	}
	goto st1059;
st1059:
	if ( ++p == pe )
		goto _test_eof1059;
case 1059:
#line 20061 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr3153;
		case 32: goto tr3153;
		case 36: goto st138;
		case 40: goto tr3154;
		case 41: goto tr3155;
		case 42: goto tr3139;
		case 65: goto tr3174;
		case 67: goto tr3175;
		case 68: goto tr3176;
		case 69: goto tr3177;
		case 72: goto tr3178;
		case 73: goto tr3179;
		case 75: goto tr3180;
		case 76: goto tr3181;
		case 77: goto tr3182;
		case 78: goto tr3183;
		case 80: goto tr3184;
		case 82: goto tr3185;
		case 83: goto tr3186;
		case 84: goto tr3187;
		case 92: goto tr3139;
		case 95: goto tr3139;
		case 97: goto tr3174;
		case 99: goto tr3175;
		case 100: goto tr3176;
		case 101: goto tr3177;
		case 104: goto tr3178;
		case 105: goto tr3179;
		case 107: goto tr3180;
		case 108: goto tr3181;
		case 109: goto tr3182;
		case 110: goto tr3183;
		case 112: goto tr3184;
		case 114: goto tr3185;
		case 115: goto tr3186;
		case 116: goto tr3187;
		case 778: goto tr19;
		case 827: goto st137;
		case 1034: goto tr3170;
		case 1083: goto tr3171;
	}
	if ( _widec < 48 ) {
		if ( 45 <= _widec && _widec <= 47 )
			goto tr3139;
	} else if ( _widec > 57 ) {
		if ( _widec > 90 ) {
			if ( 98 <= _widec && _widec <= 122 )
				goto tr3139;
		} else if ( _widec >= 64 )
			goto tr3139;
	} else
		goto tr3173;
	goto tr3172;
tr3190:
#line 704 "./scanner_body.rl"
	{
		s->stop = false;
	}
	goto st138;
st138:
	if ( ++p == pe )
		goto _test_eof138;
case 138:
#line 20140 "scanner.c"
	switch( (*p) ) {
		case 73: goto tr592;
		case 79: goto tr593;
		case 84: goto tr594;
		case 105: goto tr592;
		case 111: goto tr593;
		case 116: goto tr594;
	}
	goto tr591;
tr592:
#line 700 "./scanner_body.rl"
	{
		s->stop = true;
	}
	goto st139;
st139:
	if ( ++p == pe )
		goto _test_eof139;
case 139:
#line 20160 "scanner.c"
	switch( (*p) ) {
		case 78: goto st140;
		case 110: goto st140;
	}
	goto tr591;
st140:
	if ( ++p == pe )
		goto _test_eof140;
case 140:
	switch( (*p) ) {
		case 67: goto st141;
		case 99: goto st141;
	}
	goto tr591;
st141:
	if ( ++p == pe )
		goto _test_eof141;
case 141:
	switch( (*p) ) {
		case 76: goto st142;
		case 108: goto st142;
	}
	goto tr591;
st142:
	if ( ++p == pe )
		goto _test_eof142;
case 142:
	switch( (*p) ) {
		case 85: goto st143;
		case 117: goto st143;
	}
	goto tr591;
st143:
	if ( ++p == pe )
		goto _test_eof143;
case 143:
	switch( (*p) ) {
		case 68: goto st144;
		case 100: goto st144;
	}
	goto tr591;
st144:
	if ( ++p == pe )
		goto _test_eof144;
case 144:
	switch( (*p) ) {
		case 69: goto st145;
		case 101: goto st145;
	}
	goto tr591;
st145:
	if ( ++p == pe )
		goto _test_eof145;
case 145:
	switch( (*p) ) {
		case 32: goto tr601;
		case 59: goto tr601;
	}
	if ( (*p) > 10 ) {
		if ( 40 <= (*p) && (*p) <= 41 )
			goto tr601;
	} else if ( (*p) >= 9 )
		goto tr601;
	goto tr591;
tr601:
#line 694 "./scanner_body.rl"
	{ p--; {stack[top++] = 1060; goto st290;} }
	goto st1060;
tr687:
#line 603 "./scanner_body.rl"
	{ p--; {stack[top++] = 1060; goto st278;} }
	goto st1060;
tr690:
#line 586 "./scanner_body.rl"
	{ p--; {stack[top++] = 1060; goto st269;} }
	goto st1060;
st1060:
	if ( ++p == pe )
		goto _test_eof1060;
case 1060:
#line 20241 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr3189;
		case 32: goto tr3189;
		case 36: goto tr3190;
		case 40: goto tr3191;
		case 41: goto tr3192;
		case 42: goto tr3193;
		case 92: goto tr3193;
		case 95: goto tr3193;
		case 778: goto tr3194;
		case 827: goto tr3195;
		case 1034: goto tr3196;
		case 1083: goto tr3197;
	}
	if ( _widec < 64 ) {
		if ( 45 <= _widec && _widec <= 57 )
			goto tr3193;
	} else if ( _widec > 90 ) {
		if ( 97 <= _widec && _widec <= 122 )
			goto tr3193;
	} else
		goto tr3193;
	goto tr3188;
tr3139:
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 146; goto st248;} }
	goto st146;
tr3193:
#line 704 "./scanner_body.rl"
	{
		s->stop = false;
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 146; goto st248;} }
	goto st146;
st146:
	if ( ++p == pe )
		goto _test_eof146;
case 146:
#line 20305 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr603;
		case 32: goto tr603;
		case 40: goto tr604;
		case 41: goto tr605;
		case 1034: goto tr606;
		case 1083: goto tr607;
	}
	goto tr602;
tr609:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st147;
tr610:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st147;
tr611:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st147;
tr603:
#line 229 "./scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	goto st147;
tr604:
#line 229 "./scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st147;
tr605:
#line 229 "./scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st147;
tr606:
#line 229 "./scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st147;
st147:
	if ( ++p == pe )
		goto _test_eof147;
case 147:
#line 20403 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st147;
		case 32: goto st147;
		case 40: goto tr609;
		case 41: goto tr610;
		case 65: goto tr5;
		case 67: goto tr6;
		case 68: goto tr7;
		case 69: goto tr8;
		case 72: goto tr9;
		case 73: goto tr10;
		case 75: goto tr11;
		case 76: goto tr12;
		case 77: goto tr13;
		case 78: goto tr14;
		case 80: goto tr15;
		case 82: goto tr16;
		case 83: goto tr17;
		case 84: goto tr18;
		case 97: goto tr5;
		case 99: goto tr6;
		case 100: goto tr7;
		case 101: goto tr8;
		case 104: goto tr9;
		case 105: goto tr10;
		case 107: goto tr11;
		case 108: goto tr12;
		case 109: goto tr13;
		case 110: goto tr14;
		case 112: goto tr15;
		case 114: goto tr16;
		case 115: goto tr17;
		case 116: goto tr18;
		case 1034: goto tr611;
		case 1083: goto st148;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr4;
	goto tr524;
tr607:
#line 229 "./scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	goto st148;
st148:
	if ( ++p == pe )
		goto _test_eof148;
case 148:
#line 20467 "scanner.c"
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	if ( _widec == 1034 )
		goto tr611;
	if ( 896 <= _widec && _widec <= 1151 )
		goto st148;
	goto st0;
tr21:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st1061;
tr670:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1899 "./scanner_body.rl"
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZSCANNER_ERDATA_OVERFLOW);
			p--; {goto st246;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->process_record(s);
	}
	goto st1061;
tr695:
#line 229 "./scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1899 "./scanner_body.rl"
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZSCANNER_ERDATA_OVERFLOW);
			p--; {goto st246;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->process_record(s);
	}
	goto st1061;
tr3140:
#line 232 "./scanner_body.rl"
	{
		if (s->r_owner_length == 0) {
			WARN(ZSCANNER_EBAD_PREVIOUS_OWNER);
			p--; {goto st246;}
		}
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st1061;
tr3200:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 232 "./scanner_body.rl"
	{
		if (s->r_owner_length == 0) {
			WARN(ZSCANNER_EBAD_PREVIOUS_OWNER);
			p--; {goto st246;}
		}
	}
	goto st1061;
tr3196:
#line 704 "./scanner_body.rl"
	{
		s->stop = false;
	}
#line 232 "./scanner_body.rl"
	{
		if (s->r_owner_length == 0) {
			WARN(ZSCANNER_EBAD_PREVIOUS_OWNER);
			p--; {goto st246;}
		}
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st1061;
tr3204:
#line 1899 "./scanner_body.rl"
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZSCANNER_ERDATA_OVERFLOW);
			p--; {goto st246;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->process_record(s);
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 232 "./scanner_body.rl"
	{
		if (s->r_owner_length == 0) {
			WARN(ZSCANNER_EBAD_PREVIOUS_OWNER);
			p--; {goto st246;}
		}
	}
	goto st1061;
tr3227:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1899 "./scanner_body.rl"
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZSCANNER_ERDATA_OVERFLOW);
			p--; {goto st246;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->process_record(s);
	}
#line 232 "./scanner_body.rl"
	{
		if (s->r_owner_length == 0) {
			WARN(ZSCANNER_EBAD_PREVIOUS_OWNER);
			p--; {goto st246;}
		}
	}
	goto st1061;
st1061:
	if ( ++p == pe )
		goto _test_eof1061;
case 1061:
#line 20629 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr3135;
		case 32: goto tr3135;
		case 36: goto st138;
		case 40: goto tr3198;
		case 41: goto tr3199;
		case 42: goto tr3139;
		case 65: goto tr3174;
		case 67: goto tr3175;
		case 68: goto tr3176;
		case 69: goto tr3177;
		case 72: goto tr3178;
		case 73: goto tr3179;
		case 75: goto tr3180;
		case 76: goto tr3181;
		case 77: goto tr3182;
		case 78: goto tr3183;
		case 80: goto tr3184;
		case 82: goto tr3185;
		case 83: goto tr3186;
		case 84: goto tr3187;
		case 92: goto tr3139;
		case 95: goto tr3139;
		case 97: goto tr3174;
		case 99: goto tr3175;
		case 100: goto tr3176;
		case 101: goto tr3177;
		case 104: goto tr3178;
		case 105: goto tr3179;
		case 107: goto tr3180;
		case 108: goto tr3181;
		case 109: goto tr3182;
		case 110: goto tr3183;
		case 112: goto tr3184;
		case 114: goto tr3185;
		case 115: goto tr3186;
		case 116: goto tr3187;
		case 778: goto tr19;
		case 827: goto st137;
		case 1034: goto tr3200;
		case 1083: goto tr3141;
	}
	if ( _widec < 48 ) {
		if ( 45 <= _widec && _widec <= 47 )
			goto tr3139;
	} else if ( _widec > 57 ) {
		if ( _widec > 90 ) {
			if ( 98 <= _widec && _widec <= 122 )
				goto tr3139;
		} else if ( _widec >= 64 )
			goto tr3139;
	} else
		goto tr3173;
	goto tr3172;
tr3173:
#line 278 "./scanner_body.rl"
	{
		s->number64 = 0;
	}
#line 261 "./scanner_body.rl"
	{
		// Overflow check: 10*(s->number64) + fc - ASCII_0 <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + ASCII_0)
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ENUMBER64_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 149; goto st248;} }
	goto st149;
st149:
	if ( ++p == pe )
		goto _test_eof149;
case 149:
#line 20730 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr615;
		case 32: goto tr615;
		case 40: goto tr616;
		case 41: goto tr617;
		case 68: goto tr28;
		case 72: goto tr29;
		case 77: goto tr30;
		case 83: goto st152;
		case 87: goto tr32;
		case 100: goto tr28;
		case 104: goto tr29;
		case 109: goto tr30;
		case 115: goto st152;
		case 119: goto tr32;
		case 1034: goto tr618;
		case 1083: goto tr619;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr27;
	goto tr614;
tr621:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st150;
tr622:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st150;
tr623:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st150;
tr615:
#line 229 "./scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
#line 731 "./scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st150;
tr616:
#line 229 "./scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
#line 731 "./scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st150;
tr617:
#line 229 "./scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
#line 731 "./scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st150;
tr618:
#line 229 "./scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 731 "./scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st150;
st150:
	if ( ++p == pe )
		goto _test_eof150;
case 150:
#line 20876 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st150;
		case 32: goto st150;
		case 40: goto tr621;
		case 41: goto tr622;
		case 65: goto tr5;
		case 67: goto tr6;
		case 68: goto tr7;
		case 69: goto tr8;
		case 72: goto tr9;
		case 73: goto tr10;
		case 75: goto tr11;
		case 76: goto tr12;
		case 77: goto tr13;
		case 78: goto tr14;
		case 80: goto tr15;
		case 82: goto tr16;
		case 83: goto tr17;
		case 84: goto tr18;
		case 97: goto tr5;
		case 99: goto tr6;
		case 100: goto tr7;
		case 101: goto tr8;
		case 104: goto tr9;
		case 105: goto tr10;
		case 107: goto tr11;
		case 108: goto tr12;
		case 109: goto tr13;
		case 110: goto tr14;
		case 112: goto tr15;
		case 114: goto tr16;
		case 115: goto tr17;
		case 116: goto tr18;
		case 1034: goto tr623;
		case 1083: goto st151;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr4;
	goto tr524;
tr619:
#line 229 "./scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
#line 731 "./scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st151;
st151:
	if ( ++p == pe )
		goto _test_eof151;
case 151:
#line 20949 "scanner.c"
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	if ( _widec == 1034 )
		goto tr623;
	if ( 896 <= _widec && _widec <= 1151 )
		goto st151;
	goto tr35;
tr28:
#line 403 "./scanner_body.rl"
	{ if (s->number64 <= (UINT32_MAX / 86400)) {
	                  s->number64 *= 86400;
	              } else {
	                  WARN(ZSCANNER_ENUMBER32_OVERFLOW);
	                  p--; {goto st246;}
	              }
	            }
	goto st152;
tr29:
#line 396 "./scanner_body.rl"
	{ if (s->number64 <= (UINT32_MAX / 3600)) {
	                  s->number64 *= 3600;
	              } else {
	                  WARN(ZSCANNER_ENUMBER32_OVERFLOW);
	                  p--; {goto st246;}
	              }
	            }
	goto st152;
tr30:
#line 389 "./scanner_body.rl"
	{ if (s->number64 <= (UINT32_MAX / 60)) {
	                  s->number64 *= 60;
	              } else {
	                  WARN(ZSCANNER_ENUMBER32_OVERFLOW);
	                  p--; {goto st246;}
	              }
	            }
	goto st152;
tr32:
#line 410 "./scanner_body.rl"
	{ if (s->number64 <= (UINT32_MAX / 604800)) {
	                  s->number64 *= 604800;
	              } else {
	                  WARN(ZSCANNER_ENUMBER32_OVERFLOW);
	                  p--; {goto st246;}
	              }
	            }
	goto st152;
st152:
	if ( ++p == pe )
		goto _test_eof152;
case 152:
#line 21020 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr24;
		case 32: goto tr24;
		case 40: goto tr25;
		case 41: goto tr26;
		case 1034: goto tr33;
		case 1083: goto tr34;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr625;
	goto tr23;
tr626:
#line 261 "./scanner_body.rl"
	{
		// Overflow check: 10*(s->number64) + fc - ASCII_0 <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + ASCII_0)
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ENUMBER64_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st153;
tr625:
#line 420 "./scanner_body.rl"
	{
		s->number64_tmp = s->number64;
	}
#line 278 "./scanner_body.rl"
	{
		s->number64 = 0;
	}
#line 261 "./scanner_body.rl"
	{
		// Overflow check: 10*(s->number64) + fc - ASCII_0 <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + ASCII_0)
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ENUMBER64_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st153;
tr635:
#line 423 "./scanner_body.rl"
	{
		if (s->number64 + s->number64_tmp < UINT32_MAX) {
			s->number64 += s->number64_tmp;
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 420 "./scanner_body.rl"
	{
		s->number64_tmp = s->number64;
	}
#line 278 "./scanner_body.rl"
	{
		s->number64 = 0;
	}
#line 261 "./scanner_body.rl"
	{
		// Overflow check: 10*(s->number64) + fc - ASCII_0 <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + ASCII_0)
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ENUMBER64_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st153;
st153:
	if ( ++p == pe )
		goto _test_eof153;
case 153:
#line 21126 "scanner.c"
	switch( (*p) ) {
		case 68: goto tr627;
		case 72: goto tr628;
		case 77: goto tr629;
		case 83: goto st154;
		case 87: goto tr631;
		case 100: goto tr627;
		case 104: goto tr628;
		case 109: goto tr629;
		case 115: goto st154;
		case 119: goto tr631;
	}
	if ( 48 <= (*p) && (*p) <= 57 )
		goto tr626;
	goto tr23;
tr627:
#line 403 "./scanner_body.rl"
	{ if (s->number64 <= (UINT32_MAX / 86400)) {
	                  s->number64 *= 86400;
	              } else {
	                  WARN(ZSCANNER_ENUMBER32_OVERFLOW);
	                  p--; {goto st246;}
	              }
	            }
	goto st154;
tr628:
#line 396 "./scanner_body.rl"
	{ if (s->number64 <= (UINT32_MAX / 3600)) {
	                  s->number64 *= 3600;
	              } else {
	                  WARN(ZSCANNER_ENUMBER32_OVERFLOW);
	                  p--; {goto st246;}
	              }
	            }
	goto st154;
tr629:
#line 389 "./scanner_body.rl"
	{ if (s->number64 <= (UINT32_MAX / 60)) {
	                  s->number64 *= 60;
	              } else {
	                  WARN(ZSCANNER_ENUMBER32_OVERFLOW);
	                  p--; {goto st246;}
	              }
	            }
	goto st154;
tr631:
#line 410 "./scanner_body.rl"
	{ if (s->number64 <= (UINT32_MAX / 604800)) {
	                  s->number64 *= 604800;
	              } else {
	                  WARN(ZSCANNER_ENUMBER32_OVERFLOW);
	                  p--; {goto st246;}
	              }
	            }
	goto st154;
st154:
	if ( ++p == pe )
		goto _test_eof154;
case 154:
#line 21186 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr632;
		case 32: goto tr632;
		case 40: goto tr633;
		case 41: goto tr634;
		case 1034: goto tr636;
		case 1083: goto tr637;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr635;
	goto tr23;
tr34:
#line 731 "./scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st155;
tr637:
#line 423 "./scanner_body.rl"
	{
		if (s->number64 + s->number64_tmp < UINT32_MAX) {
			s->number64 += s->number64_tmp;
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 731 "./scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st155;
st155:
	if ( ++p == pe )
		goto _test_eof155;
case 155:
#line 21247 "scanner.c"
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	if ( _widec == 1034 )
		goto tr53;
	if ( 896 <= _widec && _widec <= 1151 )
		goto st155;
	goto tr35;
tr3156:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 156; goto st248;} }
	goto st156;
tr3174:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 156; goto st248;} }
	goto st156;
st156:
	if ( ++p == pe )
		goto _test_eof156;
case 156:
#line 21308 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr639;
		case 32: goto tr639;
		case 40: goto tr640;
		case 41: goto tr641;
		case 65: goto st213;
		case 70: goto st216;
		case 80: goto st220;
		case 97: goto st213;
		case 102: goto st216;
		case 112: goto st220;
		case 2058: goto tr642;
		case 2107: goto tr643;
		case 2314: goto tr64;
		case 2363: goto tr64;
		case 2570: goto tr644;
		case 2619: goto tr645;
	}
	goto tr638;
tr647:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st157;
tr648:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st157;
tr649:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st157;
tr639:
#line 229 "./scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
#line 1856 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_A; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st157;
tr640:
#line 229 "./scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
#line 1856 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_A; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st157;
tr641:
#line 229 "./scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
#line 1856 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_A; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st157;
tr642:
#line 229 "./scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1856 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_A; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st157;
st157:
	if ( ++p == pe )
		goto _test_eof157;
case 157:
#line 21446 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto st157;
		case 32: goto st157;
		case 40: goto tr647;
		case 41: goto tr648;
		case 58: goto tr67;
		case 65: goto tr99;
		case 67: goto tr100;
		case 68: goto tr101;
		case 69: goto tr102;
		case 72: goto tr103;
		case 73: goto tr104;
		case 75: goto tr105;
		case 76: goto tr106;
		case 77: goto tr107;
		case 78: goto tr108;
		case 80: goto tr109;
		case 82: goto tr110;
		case 83: goto tr111;
		case 84: goto tr112;
		case 92: goto st9;
		case 97: goto tr99;
		case 99: goto tr100;
		case 100: goto tr101;
		case 101: goto tr102;
		case 104: goto tr103;
		case 105: goto tr104;
		case 107: goto tr105;
		case 108: goto tr106;
		case 109: goto tr107;
		case 110: goto tr108;
		case 112: goto tr109;
		case 114: goto tr110;
		case 115: goto tr111;
		case 116: goto tr112;
		case 2058: goto tr649;
		case 2107: goto st227;
		case 2314: goto tr67;
		case 2363: goto tr67;
		case 2570: goto tr651;
		case 2619: goto tr652;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr67;
	} else if ( _widec > 47 ) {
		if ( _widec > 57 ) {
			if ( 60 <= _widec )
				goto tr67;
		} else if ( _widec >= 48 )
			goto tr98;
	} else
		goto tr67;
	goto tr526;
tr99:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 158; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 158; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 158; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 158; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 158; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 158; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 158; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 158; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 158; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 158; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 158; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 158; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 158; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 158; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 158; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 158; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 158; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 158; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 158; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 158; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 158; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 158; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 158; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 158; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 158; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 158; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 158; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st158;
st158:
	if ( ++p == pe )
		goto _test_eof158;
case 158:
#line 21609 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr654;
		case 32: goto tr654;
		case 40: goto tr655;
		case 41: goto tr656;
		case 65: goto st213;
		case 70: goto st216;
		case 80: goto st220;
		case 97: goto st213;
		case 102: goto st216;
		case 112: goto st220;
		case 1802: goto tr81;
		case 1851: goto st8;
		case 2058: goto tr657;
		case 2107: goto tr658;
		case 2314: goto tr659;
		case 2363: goto tr660;
		case 2570: goto tr661;
		case 2619: goto tr662;
	}
	goto tr653;
tr664:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st159;
tr665:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st159;
tr654:
#line 1856 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_A; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st159;
tr655:
#line 1856 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_A; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st159;
tr656:
#line 1856 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_A; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st159;
st159:
	if ( ++p == pe )
		goto _test_eof159;
case 159:
#line 21715 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto st159;
		case 32: goto st159;
		case 40: goto tr664;
		case 41: goto tr665;
		case 92: goto st9;
		case 1802: goto tr81;
		case 1851: goto st8;
		case 2058: goto tr93;
		case 2107: goto st12;
		case 2314: goto tr90;
		case 2363: goto tr91;
		case 2570: goto tr666;
		case 2619: goto tr92;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr67;
	} else if ( _widec > 58 ) {
		if ( 60 <= _widec )
			goto tr67;
	} else
		goto tr67;
	goto tr77;
tr90:
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 1062; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 1062; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 1062; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 1062; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 1062; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 1062; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 1062; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 1062; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 1062; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 1062; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 1062; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 1062; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 1062; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 1062; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 1062; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 1062; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 1062; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 1062; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 1062; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 1062; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 1062; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 1062; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 1062; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 1062; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 1062; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 1062; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 1062; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
#line 1899 "./scanner_body.rl"
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZSCANNER_ERDATA_OVERFLOW);
			p--; {goto st246;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->process_record(s);
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st1062;
tr115:
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 1062; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 1062; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 1062; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 1062; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 1062; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 1062; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 1062; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 1062; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 1062; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 1062; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 1062; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 1062; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 1062; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 1062; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 1062; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 1062; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 1062; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 1062; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 1062; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 1062; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 1062; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 1062; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 1062; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 1062; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 1062; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 1062; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 1062; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st1062;
tr659:
#line 1856 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_A; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 1062; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 1062; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 1062; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 1062; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 1062; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 1062; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 1062; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 1062; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 1062; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 1062; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 1062; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 1062; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 1062; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 1062; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 1062; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 1062; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 1062; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 1062; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 1062; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 1062; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 1062; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 1062; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 1062; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 1062; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 1062; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 1062; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 1062; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
#line 1899 "./scanner_body.rl"
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZSCANNER_ERDATA_OVERFLOW);
			p--; {goto st246;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->process_record(s);
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st1062;
st1062:
	if ( ++p == pe )
		goto _test_eof1062;
case 1062:
#line 22030 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr3201;
		case 32: goto tr3201;
		case 36: goto st138;
		case 40: goto tr3202;
		case 41: goto tr3203;
		case 42: goto tr3139;
		case 92: goto tr3139;
		case 95: goto tr3139;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr3204;
		case 1083: goto tr3205;
	}
	if ( _widec < 64 ) {
		if ( 45 <= _widec && _widec <= 57 )
			goto tr3139;
	} else if ( _widec > 90 ) {
		if ( 97 <= _widec && _widec <= 122 )
			goto tr3139;
	} else
		goto tr3139;
	goto tr691;
tr668:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st160;
tr669:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st160;
tr692:
#line 229 "./scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	goto st160;
tr693:
#line 229 "./scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st160;
tr694:
#line 229 "./scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st160;
tr3201:
#line 232 "./scanner_body.rl"
	{
		if (s->r_owner_length == 0) {
			WARN(ZSCANNER_EBAD_PREVIOUS_OWNER);
			p--; {goto st246;}
		}
	}
	goto st160;
tr3202:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
#line 232 "./scanner_body.rl"
	{
		if (s->r_owner_length == 0) {
			WARN(ZSCANNER_EBAD_PREVIOUS_OWNER);
			p--; {goto st246;}
		}
	}
	goto st160;
tr3203:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
#line 232 "./scanner_body.rl"
	{
		if (s->r_owner_length == 0) {
			WARN(ZSCANNER_EBAD_PREVIOUS_OWNER);
			p--; {goto st246;}
		}
	}
	goto st160;
st160:
	if ( ++p == pe )
		goto _test_eof160;
case 160:
#line 22169 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st160;
		case 32: goto st160;
		case 40: goto tr668;
		case 41: goto tr669;
		case 65: goto tr5;
		case 67: goto tr6;
		case 68: goto tr7;
		case 69: goto tr8;
		case 72: goto tr9;
		case 73: goto tr10;
		case 75: goto tr11;
		case 76: goto tr12;
		case 77: goto tr13;
		case 78: goto tr14;
		case 80: goto tr15;
		case 82: goto tr16;
		case 83: goto tr17;
		case 84: goto tr18;
		case 97: goto tr5;
		case 99: goto tr6;
		case 100: goto tr7;
		case 101: goto tr8;
		case 104: goto tr9;
		case 105: goto tr10;
		case 107: goto tr11;
		case 108: goto tr12;
		case 109: goto tr13;
		case 110: goto tr14;
		case 112: goto tr15;
		case 114: goto tr16;
		case 115: goto tr17;
		case 116: goto tr18;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr670;
		case 1083: goto st161;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr4;
	goto tr0;
tr696:
#line 229 "./scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	goto st161;
tr3205:
#line 232 "./scanner_body.rl"
	{
		if (s->r_owner_length == 0) {
			WARN(ZSCANNER_EBAD_PREVIOUS_OWNER);
			p--; {goto st246;}
		}
	}
	goto st161;
st161:
	if ( ++p == pe )
		goto _test_eof161;
case 161:
#line 22244 "scanner.c"
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 778: goto tr81;
		case 1034: goto tr670;
	}
	if ( _widec > 895 ) {
		if ( 896 <= _widec && _widec <= 1151 )
			goto st161;
	} else if ( _widec >= 640 )
		goto st8;
	goto tr83;
tr660:
#line 1856 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_A; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 162; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 162; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 162; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 162; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 162; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 162; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 162; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 162; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 162; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 162; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 162; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 162; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 162; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 162; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 162; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 162; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 162; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 162; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 162; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 162; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 162; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 162; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 162; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 162; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 162; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 162; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 162; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st162;
tr91:
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 162; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 162; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 162; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 162; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 162; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 162; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 162; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 162; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 162; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 162; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 162; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 162; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 162; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 162; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 162; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 162; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 162; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 162; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 162; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 162; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 162; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 162; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 162; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 162; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 162; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 162; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 162; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st162;
st162:
	if ( ++p == pe )
		goto _test_eof162;
case 162:
#line 22438 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st163;
		case 32: goto st163;
		case 40: goto tr673;
		case 41: goto tr674;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr81;
		case 1083: goto st8;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto st8;
	} else if ( _widec > 58 ) {
		if ( 60 <= _widec )
			goto st8;
	} else
		goto st8;
	goto tr77;
tr673:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st163;
tr674:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st163;
st163:
	if ( ++p == pe )
		goto _test_eof163;
case 163:
#line 22496 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st163;
		case 32: goto st163;
		case 40: goto tr673;
		case 41: goto tr674;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr81;
		case 1083: goto st8;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto st8;
	} else if ( _widec > 58 ) {
		if ( 60 <= _widec )
			goto st8;
	} else
		goto st8;
	goto tr83;
tr661:
#line 1856 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_A; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 1063; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 1063; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 1063; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 1063; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 1063; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 1063; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 1063; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 1063; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 1063; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 1063; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 1063; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 1063; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 1063; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 1063; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 1063; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 1063; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 1063; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 1063; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 1063; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 1063; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 1063; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 1063; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 1063; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 1063; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 1063; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 1063; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 1063; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
#line 1899 "./scanner_body.rl"
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZSCANNER_ERDATA_OVERFLOW);
			p--; {goto st246;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->process_record(s);
	}
	goto st1063;
tr666:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 1063; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 1063; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 1063; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 1063; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 1063; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 1063; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 1063; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 1063; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 1063; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 1063; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 1063; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 1063; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 1063; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 1063; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 1063; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 1063; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 1063; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 1063; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 1063; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 1063; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 1063; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 1063; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 1063; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 1063; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 1063; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 1063; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 1063; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
#line 1899 "./scanner_body.rl"
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZSCANNER_ERDATA_OVERFLOW);
			p--; {goto st246;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->process_record(s);
	}
	goto st1063;
st1063:
	if ( ++p == pe )
		goto _test_eof1063;
case 1063:
#line 22720 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr3206;
		case 32: goto tr3206;
		case 36: goto tr3143;
		case 40: goto tr3207;
		case 41: goto tr3208;
		case 42: goto tr3146;
		case 58: goto tr67;
		case 92: goto tr3147;
		case 95: goto tr3146;
		case 1802: goto tr81;
		case 1851: goto st8;
		case 2058: goto tr3204;
		case 2107: goto tr3209;
		case 2314: goto tr90;
		case 2363: goto tr91;
		case 2570: goto tr3204;
		case 2619: goto tr3210;
	}
	if ( _widec < 60 ) {
		if ( _widec < 11 ) {
			if ( _widec <= 8 )
				goto tr67;
		} else if ( _widec > 44 ) {
			if ( 45 <= _widec && _widec <= 57 )
				goto tr3146;
		} else
			goto tr67;
	} else if ( _widec > 63 ) {
		if ( _widec < 91 ) {
			if ( 64 <= _widec && _widec <= 90 )
				goto tr3146;
		} else if ( _widec > 96 ) {
			if ( _widec > 122 ) {
				if ( 123 <= _widec )
					goto tr67;
			} else if ( _widec >= 97 )
				goto tr3146;
		} else
			goto tr67;
	} else
		goto tr67;
	goto tr691;
tr676:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st164;
tr677:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st164;
tr651:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 164; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 164; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 164; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 164; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 164; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 164; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 164; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 164; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 164; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 164; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 164; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 164; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 164; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 164; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 164; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 164; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 164; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 164; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 164; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 164; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 164; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 164; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 164; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 164; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 164; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 164; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 164; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st164;
tr644:
#line 229 "./scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1856 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_A; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 164; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 164; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 164; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 164; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 164; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 164; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 164; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 164; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 164; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 164; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 164; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 164; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 164; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 164; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 164; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 164; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 164; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 164; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 164; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 164; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 164; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 164; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 164; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 164; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 164; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 164; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 164; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st164;
tr3206:
#line 232 "./scanner_body.rl"
	{
		if (s->r_owner_length == 0) {
			WARN(ZSCANNER_EBAD_PREVIOUS_OWNER);
			p--; {goto st246;}
		}
	}
	goto st164;
tr3207:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
#line 232 "./scanner_body.rl"
	{
		if (s->r_owner_length == 0) {
			WARN(ZSCANNER_EBAD_PREVIOUS_OWNER);
			p--; {goto st246;}
		}
	}
	goto st164;
tr3208:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
#line 232 "./scanner_body.rl"
	{
		if (s->r_owner_length == 0) {
			WARN(ZSCANNER_EBAD_PREVIOUS_OWNER);
			p--; {goto st246;}
		}
	}
	goto st164;
st164:
	if ( ++p == pe )
		goto _test_eof164;
case 164:
#line 23021 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto st164;
		case 32: goto st164;
		case 40: goto tr676;
		case 41: goto tr677;
		case 58: goto tr67;
		case 65: goto tr99;
		case 67: goto tr100;
		case 68: goto tr101;
		case 69: goto tr102;
		case 72: goto tr103;
		case 73: goto tr104;
		case 75: goto tr105;
		case 76: goto tr106;
		case 77: goto tr107;
		case 78: goto tr108;
		case 80: goto tr109;
		case 82: goto tr110;
		case 83: goto tr111;
		case 84: goto tr112;
		case 92: goto st9;
		case 97: goto tr99;
		case 99: goto tr100;
		case 100: goto tr101;
		case 101: goto tr102;
		case 104: goto tr103;
		case 105: goto tr104;
		case 107: goto tr105;
		case 108: goto tr106;
		case 109: goto tr107;
		case 110: goto tr108;
		case 112: goto tr109;
		case 114: goto tr110;
		case 115: goto tr111;
		case 116: goto tr112;
		case 1802: goto tr81;
		case 1851: goto st8;
		case 2058: goto tr670;
		case 2107: goto st178;
		case 2314: goto tr90;
		case 2363: goto tr91;
		case 2570: goto tr670;
		case 2619: goto tr679;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr67;
	} else if ( _widec > 47 ) {
		if ( _widec > 57 ) {
			if ( 60 <= _widec )
				goto tr67;
		} else if ( _widec >= 48 )
			goto tr98;
	} else
		goto tr67;
	goto tr95;
tr100:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 165; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 165; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 165; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 165; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 165; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 165; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 165; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 165; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 165; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 165; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 165; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 165; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 165; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 165; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 165; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 165; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 165; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 165; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 165; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 165; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 165; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 165; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 165; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 165; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 165; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 165; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 165; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st165;
st165:
	if ( ++p == pe )
		goto _test_eof165;
case 165:
#line 23186 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st7;
		case 32: goto st7;
		case 40: goto tr79;
		case 41: goto tr80;
		case 69: goto st17;
		case 78: goto st21;
		case 101: goto st17;
		case 110: goto st21;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr81;
		case 1083: goto st8;
	}
	goto tr653;
tr101:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 166; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 166; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 166; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 166; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 166; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 166; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 166; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 166; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 166; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 166; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 166; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 166; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 166; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 166; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 166; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 166; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 166; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 166; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 166; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 166; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 166; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 166; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 166; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 166; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 166; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 166; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 166; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st166;
st166:
	if ( ++p == pe )
		goto _test_eof166;
case 166:
#line 23304 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st7;
		case 32: goto st7;
		case 40: goto tr79;
		case 41: goto tr80;
		case 72: goto st26;
		case 78: goto st30;
		case 83: goto st38;
		case 104: goto st26;
		case 110: goto st30;
		case 115: goto st38;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr81;
		case 1083: goto st8;
	}
	goto tr653;
tr102:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 167; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 167; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 167; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 167; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 167; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 167; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 167; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 167; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 167; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 167; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 167; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 167; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 167; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 167; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 167; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 167; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 167; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 167; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 167; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 167; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 167; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 167; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 167; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 167; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 167; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 167; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 167; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st167;
st167:
	if ( ++p == pe )
		goto _test_eof167;
case 167:
#line 23424 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st7;
		case 32: goto st7;
		case 40: goto tr79;
		case 41: goto tr80;
		case 85: goto st40;
		case 117: goto st40;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr81;
		case 1083: goto st8;
	}
	goto tr653;
tr103:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 168; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 168; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 168; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 168; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 168; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 168; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 168; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 168; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 168; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 168; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 168; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 168; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 168; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 168; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 168; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 168; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 168; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 168; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 168; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 168; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 168; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 168; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 168; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 168; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 168; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 168; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 168; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st168;
st168:
	if ( ++p == pe )
		goto _test_eof168;
case 168:
#line 23540 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st7;
		case 32: goto st7;
		case 40: goto tr79;
		case 41: goto tr80;
		case 73: goto st47;
		case 105: goto st47;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr81;
		case 1083: goto st8;
	}
	goto tr653;
tr104:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 169; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 169; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 169; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 169; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 169; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 169; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 169; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 169; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 169; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 169; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 169; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 169; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 169; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 169; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 169; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 169; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 169; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 169; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 169; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 169; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 169; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 169; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 169; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 169; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 169; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 169; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 169; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st169;
st169:
	if ( ++p == pe )
		goto _test_eof169;
case 169:
#line 23656 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st7;
		case 32: goto st7;
		case 40: goto tr79;
		case 41: goto tr80;
		case 78: goto st130;
		case 80: goto st55;
		case 110: goto st130;
		case 112: goto st55;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr81;
		case 1083: goto st8;
	}
	goto tr653;
tr105:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 170; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 170; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 170; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 170; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 170; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 170; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 170; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 170; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 170; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 170; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 170; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 170; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 170; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 170; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 170; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 170; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 170; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 170; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 170; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 170; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 170; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 170; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 170; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 170; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 170; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 170; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 170; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st170;
st170:
	if ( ++p == pe )
		goto _test_eof170;
case 170:
#line 23774 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st7;
		case 32: goto st7;
		case 40: goto tr79;
		case 41: goto tr80;
		case 69: goto st63;
		case 88: goto st65;
		case 101: goto st63;
		case 120: goto st65;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr81;
		case 1083: goto st8;
	}
	goto tr653;
tr106:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 171; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 171; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 171; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 171; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 171; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 171; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 171; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 171; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 171; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 171; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 171; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 171; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 171; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 171; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 171; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 171; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 171; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 171; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 171; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 171; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 171; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 171; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 171; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 171; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 171; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 171; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 171; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st171;
st171:
	if ( ++p == pe )
		goto _test_eof171;
case 171:
#line 23892 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st7;
		case 32: goto st7;
		case 40: goto tr79;
		case 41: goto tr80;
		case 51: goto st67;
		case 54: goto st69;
		case 79: goto st71;
		case 80: goto st73;
		case 111: goto st71;
		case 112: goto st73;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr81;
		case 1083: goto st8;
	}
	goto tr653;
tr107:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 172; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 172; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 172; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 172; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 172; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 172; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 172; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 172; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 172; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 172; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 172; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 172; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 172; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 172; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 172; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 172; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 172; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 172; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 172; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 172; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 172; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 172; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 172; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 172; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 172; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 172; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 172; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st172;
st172:
	if ( ++p == pe )
		goto _test_eof172;
case 172:
#line 24012 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st7;
		case 32: goto st7;
		case 40: goto tr79;
		case 41: goto tr80;
		case 73: goto st75;
		case 88: goto st79;
		case 105: goto st75;
		case 120: goto st79;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr81;
		case 1083: goto st8;
	}
	goto tr653;
tr108:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 173; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 173; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 173; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 173; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 173; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 173; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 173; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 173; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 173; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 173; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 173; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 173; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 173; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 173; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 173; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 173; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 173; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 173; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 173; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 173; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 173; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 173; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 173; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 173; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 173; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 173; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 173; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st173;
st173:
	if ( ++p == pe )
		goto _test_eof173;
case 173:
#line 24130 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st7;
		case 32: goto st7;
		case 40: goto tr79;
		case 41: goto tr80;
		case 65: goto st81;
		case 73: goto st85;
		case 83: goto st87;
		case 97: goto st81;
		case 105: goto st85;
		case 115: goto st87;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr81;
		case 1083: goto st8;
	}
	goto tr653;
tr109:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 174; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 174; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 174; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 174; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 174; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 174; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 174; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 174; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 174; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 174; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 174; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 174; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 174; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 174; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 174; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 174; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 174; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 174; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 174; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 174; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 174; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 174; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 174; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 174; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 174; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 174; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 174; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st174;
st174:
	if ( ++p == pe )
		goto _test_eof174;
case 174:
#line 24250 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st7;
		case 32: goto st7;
		case 40: goto tr79;
		case 41: goto tr80;
		case 84: goto st97;
		case 116: goto st97;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr81;
		case 1083: goto st8;
	}
	goto tr653;
tr110:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 175; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 175; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 175; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 175; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 175; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 175; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 175; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 175; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 175; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 175; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 175; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 175; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 175; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 175; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 175; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 175; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 175; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 175; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 175; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 175; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 175; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 175; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 175; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 175; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 175; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 175; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 175; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st175;
st175:
	if ( ++p == pe )
		goto _test_eof175;
case 175:
#line 24366 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st7;
		case 32: goto st7;
		case 40: goto tr79;
		case 41: goto tr80;
		case 80: goto st100;
		case 82: goto st101;
		case 84: goto st105;
		case 112: goto st100;
		case 114: goto st101;
		case 116: goto st105;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr81;
		case 1083: goto st8;
	}
	goto tr653;
tr111:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 176; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 176; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 176; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 176; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 176; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 176; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 176; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 176; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 176; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 176; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 176; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 176; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 176; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 176; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 176; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 176; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 176; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 176; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 176; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 176; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 176; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 176; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 176; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 176; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 176; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 176; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 176; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st176;
st176:
	if ( ++p == pe )
		goto _test_eof176;
case 176:
#line 24486 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st7;
		case 32: goto st7;
		case 40: goto tr79;
		case 41: goto tr80;
		case 79: goto st107;
		case 80: goto st109;
		case 82: goto st111;
		case 83: goto st113;
		case 111: goto st107;
		case 112: goto st109;
		case 114: goto st111;
		case 115: goto st113;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr81;
		case 1083: goto st8;
	}
	goto tr653;
tr112:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 177; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 177; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 177; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 177; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 177; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 177; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 177; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 177; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 177; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 177; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 177; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 177; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 177; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 177; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 177; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 177; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 177; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 177; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 177; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 177; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 177; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 177; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 177; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 177; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 177; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 177; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 177; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st177;
st177:
	if ( ++p == pe )
		goto _test_eof177;
case 177:
#line 24608 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st7;
		case 32: goto st7;
		case 40: goto tr79;
		case 41: goto tr80;
		case 76: goto st118;
		case 88: goto st121;
		case 89: goto st123;
		case 108: goto st118;
		case 120: goto st121;
		case 121: goto st123;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr81;
		case 1083: goto st8;
	}
	goto tr653;
tr713:
#line 229 "./scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
#line 1856 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_A; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st178;
tr3209:
#line 232 "./scanner_body.rl"
	{
		if (s->r_owner_length == 0) {
			WARN(ZSCANNER_EBAD_PREVIOUS_OWNER);
			p--; {goto st246;}
		}
	}
	goto st178;
st178:
	if ( ++p == pe )
		goto _test_eof178;
case 178:
#line 24665 "scanner.c"
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 778: goto tr81;
		case 1034: goto tr680;
	}
	if ( _widec > 895 ) {
		if ( 896 <= _widec && _widec <= 1151 )
			goto st178;
	} else if ( _widec >= 640 )
		goto st8;
	goto tr77;
tr113:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st1064;
tr680:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1899 "./scanner_body.rl"
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZSCANNER_ERDATA_OVERFLOW);
			p--; {goto st246;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->process_record(s);
	}
	goto st1064;
tr712:
#line 229 "./scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1856 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_A; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1899 "./scanner_body.rl"
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZSCANNER_ERDATA_OVERFLOW);
			p--; {goto st246;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->process_record(s);
	}
	goto st1064;
tr3148:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 232 "./scanner_body.rl"
	{
		if (s->r_owner_length == 0) {
			WARN(ZSCANNER_EBAD_PREVIOUS_OWNER);
			p--; {goto st246;}
		}
	}
	goto st1064;
st1064:
	if ( ++p == pe )
		goto _test_eof1064;
case 1064:
#line 24762 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr3142;
		case 32: goto tr3142;
		case 36: goto tr3143;
		case 40: goto tr3144;
		case 41: goto tr3145;
		case 42: goto tr3146;
		case 58: goto tr67;
		case 65: goto tr3213;
		case 67: goto tr3214;
		case 68: goto tr3215;
		case 69: goto tr3216;
		case 72: goto tr3217;
		case 73: goto tr3218;
		case 75: goto tr3219;
		case 76: goto tr3220;
		case 77: goto tr3221;
		case 78: goto tr3222;
		case 80: goto tr3223;
		case 82: goto tr3224;
		case 83: goto tr3225;
		case 84: goto tr3226;
		case 92: goto tr3147;
		case 95: goto tr3146;
		case 97: goto tr3213;
		case 99: goto tr3214;
		case 100: goto tr3215;
		case 101: goto tr3216;
		case 104: goto tr3217;
		case 105: goto tr3218;
		case 107: goto tr3219;
		case 108: goto tr3220;
		case 109: goto tr3221;
		case 110: goto tr3222;
		case 112: goto tr3223;
		case 114: goto tr3224;
		case 115: goto tr3225;
		case 116: goto tr3226;
		case 1802: goto tr19;
		case 1851: goto st137;
		case 2058: goto tr3148;
		case 2107: goto tr3149;
		case 2314: goto tr115;
		case 2363: goto tr116;
		case 2570: goto tr3150;
		case 2619: goto tr3151;
	}
	if ( _widec < 60 ) {
		if ( _widec < 11 ) {
			if ( _widec <= 8 )
				goto tr67;
		} else if ( _widec > 44 ) {
			if ( _widec > 47 ) {
				if ( 48 <= _widec && _widec <= 57 )
					goto tr3212;
			} else if ( _widec >= 45 )
				goto tr3146;
		} else
			goto tr67;
	} else if ( _widec > 63 ) {
		if ( _widec < 91 ) {
			if ( 64 <= _widec && _widec <= 90 )
				goto tr3146;
		} else if ( _widec > 96 ) {
			if ( _widec > 122 ) {
				if ( 123 <= _widec )
					goto tr67;
			} else if ( _widec >= 98 )
				goto tr3146;
		} else
			goto tr67;
	} else
		goto tr67;
	goto tr3211;
tr3143:
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 179; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 179; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 179; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 179; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 179; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 179; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 179; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 179; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 179; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 179; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 179; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 179; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 179; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 179; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 179; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 179; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 179; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 179; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 179; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 179; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 179; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 179; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 179; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 179; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 179; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 179; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 179; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st179;
st179:
	if ( ++p == pe )
		goto _test_eof179;
case 179:
#line 24937 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st7;
		case 32: goto st7;
		case 40: goto tr79;
		case 41: goto tr80;
		case 73: goto tr592;
		case 79: goto tr593;
		case 84: goto tr594;
		case 105: goto tr592;
		case 111: goto tr593;
		case 116: goto tr594;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr81;
		case 1083: goto st8;
	}
	goto tr681;
tr593:
#line 700 "./scanner_body.rl"
	{
		s->stop = true;
	}
	goto st180;
st180:
	if ( ++p == pe )
		goto _test_eof180;
case 180:
#line 24979 "scanner.c"
	switch( (*p) ) {
		case 82: goto st181;
		case 114: goto st181;
	}
	goto tr591;
st181:
	if ( ++p == pe )
		goto _test_eof181;
case 181:
	switch( (*p) ) {
		case 73: goto st182;
		case 105: goto st182;
	}
	goto tr591;
st182:
	if ( ++p == pe )
		goto _test_eof182;
case 182:
	switch( (*p) ) {
		case 71: goto st183;
		case 103: goto st183;
	}
	goto tr591;
st183:
	if ( ++p == pe )
		goto _test_eof183;
case 183:
	switch( (*p) ) {
		case 73: goto st184;
		case 105: goto st184;
	}
	goto tr591;
st184:
	if ( ++p == pe )
		goto _test_eof184;
case 184:
	switch( (*p) ) {
		case 78: goto st185;
		case 110: goto st185;
	}
	goto tr591;
st185:
	if ( ++p == pe )
		goto _test_eof185;
case 185:
	switch( (*p) ) {
		case 32: goto tr687;
		case 59: goto tr687;
	}
	if ( (*p) > 10 ) {
		if ( 40 <= (*p) && (*p) <= 41 )
			goto tr687;
	} else if ( (*p) >= 9 )
		goto tr687;
	goto tr591;
tr594:
#line 700 "./scanner_body.rl"
	{
		s->stop = true;
	}
	goto st186;
st186:
	if ( ++p == pe )
		goto _test_eof186;
case 186:
#line 25045 "scanner.c"
	switch( (*p) ) {
		case 84: goto st187;
		case 116: goto st187;
	}
	goto tr591;
st187:
	if ( ++p == pe )
		goto _test_eof187;
case 187:
	switch( (*p) ) {
		case 76: goto st188;
		case 108: goto st188;
	}
	goto tr591;
st188:
	if ( ++p == pe )
		goto _test_eof188;
case 188:
	switch( (*p) ) {
		case 32: goto tr690;
		case 59: goto tr690;
	}
	if ( (*p) > 10 ) {
		if ( 40 <= (*p) && (*p) <= 41 )
			goto tr690;
	} else if ( (*p) >= 9 )
		goto tr690;
	goto tr591;
tr3146:
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 189; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 189; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 189; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 189; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 189; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 189; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 189; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 189; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 189; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 189; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 189; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 189; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 189; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 189; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 189; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 189; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 189; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 189; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 189; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 189; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 189; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 189; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 189; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 189; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 189; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 189; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 189; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 189; goto st248;} }
	goto st189;
st189:
	if ( ++p == pe )
		goto _test_eof189;
case 189:
#line 25161 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr692;
		case 32: goto tr692;
		case 40: goto tr693;
		case 41: goto tr694;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr695;
		case 1083: goto tr696;
	}
	goto tr691;
tr3212:
#line 278 "./scanner_body.rl"
	{
		s->number64 = 0;
	}
#line 261 "./scanner_body.rl"
	{
		// Overflow check: 10*(s->number64) + fc - ASCII_0 <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + ASCII_0)
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ENUMBER64_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 190; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 190; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 190; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 190; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 190; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 190; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 190; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 190; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 190; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 190; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 190; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 190; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 190; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 190; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 190; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 190; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 190; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 190; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 190; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 190; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 190; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 190; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 190; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 190; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 190; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 190; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 190; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 190; goto st248;} }
	goto st190;
st190:
	if ( ++p == pe )
		goto _test_eof190;
case 190:
#line 25293 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr698;
		case 32: goto tr698;
		case 40: goto tr699;
		case 41: goto tr700;
		case 68: goto tr28;
		case 72: goto tr29;
		case 77: goto tr30;
		case 83: goto st152;
		case 87: goto tr32;
		case 100: goto tr28;
		case 104: goto tr29;
		case 109: goto tr30;
		case 115: goto st152;
		case 119: goto tr32;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr701;
		case 1083: goto tr702;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr27;
	goto tr697;
tr704:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st191;
tr705:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st191;
tr698:
#line 229 "./scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
#line 731 "./scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st191;
tr699:
#line 229 "./scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
#line 731 "./scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st191;
tr700:
#line 229 "./scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
#line 731 "./scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st191;
st191:
	if ( ++p == pe )
		goto _test_eof191;
case 191:
#line 25416 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st191;
		case 32: goto st191;
		case 40: goto tr704;
		case 41: goto tr705;
		case 65: goto tr5;
		case 67: goto tr6;
		case 68: goto tr7;
		case 69: goto tr8;
		case 72: goto tr9;
		case 73: goto tr10;
		case 75: goto tr11;
		case 76: goto tr12;
		case 77: goto tr13;
		case 78: goto tr14;
		case 80: goto tr15;
		case 82: goto tr16;
		case 83: goto tr17;
		case 84: goto tr18;
		case 97: goto tr5;
		case 99: goto tr6;
		case 100: goto tr7;
		case 101: goto tr8;
		case 104: goto tr9;
		case 105: goto tr10;
		case 107: goto tr11;
		case 108: goto tr12;
		case 109: goto tr13;
		case 110: goto tr14;
		case 112: goto tr15;
		case 114: goto tr16;
		case 115: goto tr17;
		case 116: goto tr18;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr706;
		case 1083: goto st192;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr4;
	goto tr0;
tr702:
#line 229 "./scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
#line 731 "./scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st192;
st192:
	if ( ++p == pe )
		goto _test_eof192;
case 192:
#line 25491 "scanner.c"
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 778: goto tr81;
		case 1034: goto tr706;
	}
	if ( _widec > 895 ) {
		if ( 896 <= _widec && _widec <= 1151 )
			goto st192;
	} else if ( _widec >= 640 )
		goto st8;
	goto tr125;
tr3213:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 193; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 193; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 193; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 193; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 193; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 193; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 193; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 193; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 193; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 193; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 193; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 193; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 193; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 193; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 193; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 193; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 193; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 193; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 193; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 193; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 193; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 193; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 193; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 193; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 193; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 193; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 193; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 193; goto st248;} }
	goto st193;
st193:
	if ( ++p == pe )
		goto _test_eof193;
case 193:
#line 25618 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr709;
		case 32: goto tr709;
		case 40: goto tr710;
		case 41: goto tr711;
		case 65: goto st213;
		case 70: goto st216;
		case 80: goto st220;
		case 97: goto st213;
		case 102: goto st216;
		case 112: goto st220;
		case 1802: goto tr81;
		case 1851: goto st8;
		case 2058: goto tr712;
		case 2107: goto tr713;
		case 2314: goto tr659;
		case 2363: goto tr660;
		case 2570: goto tr714;
		case 2619: goto tr715;
	}
	goto tr708;
tr717:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st194;
tr718:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st194;
tr709:
#line 229 "./scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
#line 1856 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_A; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st194;
tr710:
#line 229 "./scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
#line 1856 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_A; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st194;
tr711:
#line 229 "./scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
#line 1856 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_A; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st194;
st194:
	if ( ++p == pe )
		goto _test_eof194;
case 194:
#line 25736 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto st194;
		case 32: goto st194;
		case 40: goto tr717;
		case 41: goto tr718;
		case 58: goto tr67;
		case 65: goto tr99;
		case 67: goto tr100;
		case 68: goto tr101;
		case 69: goto tr102;
		case 72: goto tr103;
		case 73: goto tr104;
		case 75: goto tr105;
		case 76: goto tr106;
		case 77: goto tr107;
		case 78: goto tr108;
		case 80: goto tr109;
		case 82: goto tr110;
		case 83: goto tr111;
		case 84: goto tr112;
		case 92: goto st9;
		case 97: goto tr99;
		case 99: goto tr100;
		case 100: goto tr101;
		case 101: goto tr102;
		case 104: goto tr103;
		case 105: goto tr104;
		case 107: goto tr105;
		case 108: goto tr106;
		case 109: goto tr107;
		case 110: goto tr108;
		case 112: goto tr109;
		case 114: goto tr110;
		case 115: goto tr111;
		case 116: goto tr112;
		case 1802: goto tr81;
		case 1851: goto st8;
		case 2058: goto tr680;
		case 2107: goto st178;
		case 2314: goto tr90;
		case 2363: goto tr91;
		case 2570: goto tr719;
		case 2619: goto tr679;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr67;
	} else if ( _widec > 47 ) {
		if ( _widec > 57 ) {
			if ( 60 <= _widec )
				goto tr67;
		} else if ( _widec >= 48 )
			goto tr98;
	} else
		goto tr67;
	goto tr95;
tr117:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 1065; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 1065; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 1065; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 1065; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 1065; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 1065; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 1065; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 1065; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 1065; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 1065; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 1065; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 1065; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 1065; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 1065; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 1065; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 1065; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 1065; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 1065; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 1065; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 1065; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 1065; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 1065; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 1065; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 1065; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 1065; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 1065; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 1065; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st1065;
tr719:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 1065; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 1065; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 1065; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 1065; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 1065; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 1065; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 1065; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 1065; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 1065; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 1065; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 1065; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 1065; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 1065; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 1065; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 1065; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 1065; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 1065; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 1065; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 1065; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 1065; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 1065; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 1065; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 1065; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 1065; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 1065; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 1065; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 1065; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
#line 1899 "./scanner_body.rl"
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZSCANNER_ERDATA_OVERFLOW);
			p--; {goto st246;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->process_record(s);
	}
	goto st1065;
tr714:
#line 229 "./scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1856 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_A; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 1065; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 1065; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 1065; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 1065; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 1065; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 1065; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 1065; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 1065; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 1065; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 1065; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 1065; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 1065; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 1065; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 1065; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 1065; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 1065; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 1065; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 1065; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 1065; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 1065; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 1065; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 1065; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 1065; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 1065; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 1065; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 1065; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 1065; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
#line 1899 "./scanner_body.rl"
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZSCANNER_ERDATA_OVERFLOW);
			p--; {goto st246;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->process_record(s);
	}
	goto st1065;
tr3150:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 1065; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 1065; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 1065; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 1065; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 1065; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 1065; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 1065; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 1065; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 1065; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 1065; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 1065; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 1065; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 1065; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 1065; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 1065; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 1065; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 1065; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 1065; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 1065; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 1065; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 1065; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 1065; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 1065; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 1065; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 1065; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 1065; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 1065; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
#line 232 "./scanner_body.rl"
	{
		if (s->r_owner_length == 0) {
			WARN(ZSCANNER_EBAD_PREVIOUS_OWNER);
			p--; {goto st246;}
		}
	}
	goto st1065;
st1065:
	if ( ++p == pe )
		goto _test_eof1065;
case 1065:
#line 26174 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr3206;
		case 32: goto tr3206;
		case 36: goto tr3143;
		case 40: goto tr3207;
		case 41: goto tr3208;
		case 42: goto tr3146;
		case 58: goto tr67;
		case 65: goto tr3213;
		case 67: goto tr3214;
		case 68: goto tr3215;
		case 69: goto tr3216;
		case 72: goto tr3217;
		case 73: goto tr3218;
		case 75: goto tr3219;
		case 76: goto tr3220;
		case 77: goto tr3221;
		case 78: goto tr3222;
		case 80: goto tr3223;
		case 82: goto tr3224;
		case 83: goto tr3225;
		case 84: goto tr3226;
		case 92: goto tr3147;
		case 95: goto tr3146;
		case 97: goto tr3213;
		case 99: goto tr3214;
		case 100: goto tr3215;
		case 101: goto tr3216;
		case 104: goto tr3217;
		case 105: goto tr3218;
		case 107: goto tr3219;
		case 108: goto tr3220;
		case 109: goto tr3221;
		case 110: goto tr3222;
		case 112: goto tr3223;
		case 114: goto tr3224;
		case 115: goto tr3225;
		case 116: goto tr3226;
		case 1802: goto tr81;
		case 1851: goto st8;
		case 2058: goto tr3227;
		case 2107: goto tr3209;
		case 2314: goto tr90;
		case 2363: goto tr91;
		case 2570: goto tr3227;
		case 2619: goto tr3210;
	}
	if ( _widec < 60 ) {
		if ( _widec < 11 ) {
			if ( _widec <= 8 )
				goto tr67;
		} else if ( _widec > 44 ) {
			if ( _widec > 47 ) {
				if ( 48 <= _widec && _widec <= 57 )
					goto tr3212;
			} else if ( _widec >= 45 )
				goto tr3146;
		} else
			goto tr67;
	} else if ( _widec > 63 ) {
		if ( _widec < 91 ) {
			if ( 64 <= _widec && _widec <= 90 )
				goto tr3146;
		} else if ( _widec > 96 ) {
			if ( _widec > 122 ) {
				if ( 123 <= _widec )
					goto tr67;
			} else if ( _widec >= 98 )
				goto tr3146;
		} else
			goto tr67;
	} else
		goto tr67;
	goto tr3211;
tr3214:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 195; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 195; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 195; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 195; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 195; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 195; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 195; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 195; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 195; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 195; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 195; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 195; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 195; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 195; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 195; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 195; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 195; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 195; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 195; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 195; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 195; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 195; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 195; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 195; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 195; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 195; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 195; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 195; goto st248;} }
	goto st195;
st195:
	if ( ++p == pe )
		goto _test_eof195;
case 195:
#line 26364 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr692;
		case 32: goto tr692;
		case 40: goto tr693;
		case 41: goto tr694;
		case 69: goto st17;
		case 78: goto st21;
		case 101: goto st17;
		case 110: goto st21;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr695;
		case 1083: goto tr696;
	}
	goto tr708;
tr3215:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 196; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 196; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 196; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 196; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 196; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 196; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 196; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 196; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 196; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 196; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 196; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 196; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 196; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 196; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 196; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 196; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 196; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 196; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 196; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 196; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 196; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 196; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 196; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 196; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 196; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 196; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 196; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 196; goto st248;} }
	goto st196;
st196:
	if ( ++p == pe )
		goto _test_eof196;
case 196:
#line 26489 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr692;
		case 32: goto tr692;
		case 40: goto tr693;
		case 41: goto tr694;
		case 72: goto st26;
		case 78: goto st30;
		case 83: goto st38;
		case 104: goto st26;
		case 110: goto st30;
		case 115: goto st38;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr695;
		case 1083: goto tr696;
	}
	goto tr708;
tr3216:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 197; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 197; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 197; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 197; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 197; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 197; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 197; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 197; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 197; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 197; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 197; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 197; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 197; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 197; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 197; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 197; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 197; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 197; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 197; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 197; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 197; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 197; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 197; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 197; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 197; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 197; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 197; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 197; goto st248;} }
	goto st197;
st197:
	if ( ++p == pe )
		goto _test_eof197;
case 197:
#line 26616 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr692;
		case 32: goto tr692;
		case 40: goto tr693;
		case 41: goto tr694;
		case 85: goto st40;
		case 117: goto st40;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr695;
		case 1083: goto tr696;
	}
	goto tr708;
tr3217:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 198; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 198; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 198; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 198; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 198; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 198; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 198; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 198; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 198; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 198; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 198; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 198; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 198; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 198; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 198; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 198; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 198; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 198; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 198; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 198; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 198; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 198; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 198; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 198; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 198; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 198; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 198; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 198; goto st248;} }
	goto st198;
st198:
	if ( ++p == pe )
		goto _test_eof198;
case 198:
#line 26739 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr692;
		case 32: goto tr692;
		case 40: goto tr693;
		case 41: goto tr694;
		case 73: goto st47;
		case 105: goto st47;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr695;
		case 1083: goto tr696;
	}
	goto tr708;
tr3218:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 199; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 199; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 199; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 199; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 199; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 199; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 199; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 199; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 199; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 199; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 199; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 199; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 199; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 199; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 199; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 199; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 199; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 199; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 199; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 199; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 199; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 199; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 199; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 199; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 199; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 199; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 199; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 199; goto st248;} }
	goto st199;
st199:
	if ( ++p == pe )
		goto _test_eof199;
case 199:
#line 26862 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr692;
		case 32: goto tr692;
		case 40: goto tr693;
		case 41: goto tr694;
		case 78: goto st130;
		case 80: goto st55;
		case 110: goto st130;
		case 112: goto st55;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr695;
		case 1083: goto tr696;
	}
	goto tr708;
tr3219:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 200; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 200; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 200; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 200; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 200; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 200; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 200; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 200; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 200; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 200; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 200; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 200; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 200; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 200; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 200; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 200; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 200; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 200; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 200; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 200; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 200; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 200; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 200; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 200; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 200; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 200; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 200; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 200; goto st248;} }
	goto st200;
st200:
	if ( ++p == pe )
		goto _test_eof200;
case 200:
#line 26987 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr692;
		case 32: goto tr692;
		case 40: goto tr693;
		case 41: goto tr694;
		case 69: goto st63;
		case 88: goto st65;
		case 101: goto st63;
		case 120: goto st65;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr695;
		case 1083: goto tr696;
	}
	goto tr708;
tr3220:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 201; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 201; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 201; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 201; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 201; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 201; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 201; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 201; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 201; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 201; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 201; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 201; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 201; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 201; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 201; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 201; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 201; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 201; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 201; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 201; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 201; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 201; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 201; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 201; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 201; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 201; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 201; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 201; goto st248;} }
	goto st201;
st201:
	if ( ++p == pe )
		goto _test_eof201;
case 201:
#line 27112 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr692;
		case 32: goto tr692;
		case 40: goto tr693;
		case 41: goto tr694;
		case 51: goto st67;
		case 54: goto st69;
		case 79: goto st71;
		case 80: goto st73;
		case 111: goto st71;
		case 112: goto st73;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr695;
		case 1083: goto tr696;
	}
	goto tr708;
tr3221:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 202; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 202; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 202; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 202; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 202; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 202; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 202; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 202; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 202; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 202; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 202; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 202; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 202; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 202; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 202; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 202; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 202; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 202; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 202; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 202; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 202; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 202; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 202; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 202; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 202; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 202; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 202; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 202; goto st248;} }
	goto st202;
st202:
	if ( ++p == pe )
		goto _test_eof202;
case 202:
#line 27239 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr692;
		case 32: goto tr692;
		case 40: goto tr693;
		case 41: goto tr694;
		case 73: goto st75;
		case 88: goto st79;
		case 105: goto st75;
		case 120: goto st79;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr695;
		case 1083: goto tr696;
	}
	goto tr708;
tr3222:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 203; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 203; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 203; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 203; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 203; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 203; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 203; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 203; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 203; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 203; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 203; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 203; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 203; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 203; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 203; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 203; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 203; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 203; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 203; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 203; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 203; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 203; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 203; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 203; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 203; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 203; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 203; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 203; goto st248;} }
	goto st203;
st203:
	if ( ++p == pe )
		goto _test_eof203;
case 203:
#line 27364 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr692;
		case 32: goto tr692;
		case 40: goto tr693;
		case 41: goto tr694;
		case 65: goto st81;
		case 73: goto st85;
		case 83: goto st87;
		case 97: goto st81;
		case 105: goto st85;
		case 115: goto st87;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr695;
		case 1083: goto tr696;
	}
	goto tr708;
tr3223:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 204; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 204; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 204; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 204; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 204; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 204; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 204; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 204; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 204; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 204; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 204; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 204; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 204; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 204; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 204; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 204; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 204; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 204; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 204; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 204; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 204; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 204; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 204; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 204; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 204; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 204; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 204; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 204; goto st248;} }
	goto st204;
st204:
	if ( ++p == pe )
		goto _test_eof204;
case 204:
#line 27491 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr692;
		case 32: goto tr692;
		case 40: goto tr693;
		case 41: goto tr694;
		case 84: goto st97;
		case 116: goto st97;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr695;
		case 1083: goto tr696;
	}
	goto tr708;
tr3224:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 205; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 205; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 205; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 205; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 205; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 205; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 205; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 205; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 205; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 205; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 205; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 205; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 205; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 205; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 205; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 205; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 205; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 205; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 205; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 205; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 205; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 205; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 205; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 205; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 205; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 205; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 205; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 205; goto st248;} }
	goto st205;
st205:
	if ( ++p == pe )
		goto _test_eof205;
case 205:
#line 27614 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr692;
		case 32: goto tr692;
		case 40: goto tr693;
		case 41: goto tr694;
		case 80: goto st100;
		case 82: goto st101;
		case 84: goto st105;
		case 112: goto st100;
		case 114: goto st101;
		case 116: goto st105;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr695;
		case 1083: goto tr696;
	}
	goto tr708;
tr3225:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 206; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 206; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 206; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 206; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 206; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 206; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 206; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 206; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 206; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 206; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 206; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 206; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 206; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 206; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 206; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 206; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 206; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 206; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 206; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 206; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 206; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 206; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 206; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 206; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 206; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 206; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 206; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 206; goto st248;} }
	goto st206;
st206:
	if ( ++p == pe )
		goto _test_eof206;
case 206:
#line 27741 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr692;
		case 32: goto tr692;
		case 40: goto tr693;
		case 41: goto tr694;
		case 79: goto st107;
		case 80: goto st109;
		case 82: goto st111;
		case 83: goto st113;
		case 111: goto st107;
		case 112: goto st109;
		case 114: goto st111;
		case 115: goto st113;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr695;
		case 1083: goto tr696;
	}
	goto tr708;
tr3226:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 207; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 207; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 207; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 207; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 207; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 207; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 207; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 207; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 207; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 207; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 207; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 207; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 207; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 207; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 207; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 207; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 207; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 207; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 207; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 207; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 207; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 207; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 207; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 207; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 207; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 207; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 207; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 207; goto st248;} }
	goto st207;
st207:
	if ( ++p == pe )
		goto _test_eof207;
case 207:
#line 27870 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr692;
		case 32: goto tr692;
		case 40: goto tr693;
		case 41: goto tr694;
		case 76: goto st118;
		case 88: goto st121;
		case 89: goto st123;
		case 108: goto st118;
		case 120: goto st121;
		case 121: goto st123;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr695;
		case 1083: goto tr696;
	}
	goto tr708;
tr3147:
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 208; goto st248;} }
	goto st208;
st208:
	if ( ++p == pe )
		goto _test_eof208;
case 208:
#line 27915 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr720;
		case 32: goto tr720;
		case 35: goto tr85;
		case 40: goto tr722;
		case 41: goto tr723;
		case 778: goto tr84;
		case 827: goto tr84;
		case 1034: goto tr724;
		case 1083: goto tr725;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr84;
	} else if ( _widec > 58 ) {
		if ( 60 <= _widec )
			goto tr84;
	} else
		goto tr84;
	goto tr721;
tr720:
#line 229 "./scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
#line 1843 "./scanner_body.rl"
	{ p--; }
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 209; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 209; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 209; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 209; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 209; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 209; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 209; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 209; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 209; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 209; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 209; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 209; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 209; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 209; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 209; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 209; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 209; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 209; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 209; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 209; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 209; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 209; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 209; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 209; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 209; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 209; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 209; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st209;
tr722:
#line 229 "./scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
#line 1843 "./scanner_body.rl"
	{ p--; }
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 209; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 209; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 209; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 209; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 209; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 209; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 209; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 209; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 209; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 209; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 209; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 209; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 209; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 209; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 209; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 209; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 209; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 209; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 209; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 209; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 209; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 209; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 209; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 209; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 209; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 209; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 209; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st209;
tr723:
#line 229 "./scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
#line 1843 "./scanner_body.rl"
	{ p--; }
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 209; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 209; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 209; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 209; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 209; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 209; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 209; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 209; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 209; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 209; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 209; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 209; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 209; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 209; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 209; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 209; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 209; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 209; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 209; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 209; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 209; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 209; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 209; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 209; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 209; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 209; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 209; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st209;
tr724:
#line 229 "./scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1843 "./scanner_body.rl"
	{ p--; }
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 209; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 209; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 209; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 209; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 209; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 209; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 209; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 209; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 209; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 209; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 209; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 209; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 209; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 209; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 209; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 209; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 209; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 209; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 209; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 209; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 209; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 209; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 209; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 209; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 209; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 209; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 209; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st209;
st209:
	if ( ++p == pe )
		goto _test_eof209;
case 209:
#line 28302 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st160;
		case 32: goto st160;
		case 40: goto tr668;
		case 41: goto tr669;
		case 65: goto tr5;
		case 67: goto tr6;
		case 68: goto tr7;
		case 69: goto tr8;
		case 72: goto tr9;
		case 73: goto tr10;
		case 75: goto tr11;
		case 76: goto tr12;
		case 77: goto tr13;
		case 78: goto tr14;
		case 80: goto tr15;
		case 82: goto tr16;
		case 83: goto tr17;
		case 84: goto tr18;
		case 97: goto tr5;
		case 99: goto tr6;
		case 100: goto tr7;
		case 101: goto tr8;
		case 104: goto tr9;
		case 105: goto tr10;
		case 107: goto tr11;
		case 108: goto tr12;
		case 109: goto tr13;
		case 110: goto tr14;
		case 112: goto tr15;
		case 114: goto tr16;
		case 115: goto tr17;
		case 116: goto tr18;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr670;
		case 1083: goto st161;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr4;
	goto tr95;
tr725:
#line 229 "./scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
#line 1843 "./scanner_body.rl"
	{ p--; }
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 210; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 210; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 210; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 210; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 210; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 210; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 210; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 210; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 210; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 210; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 210; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 210; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 210; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 210; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 210; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 210; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 210; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 210; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 210; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 210; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 210; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 210; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 210; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 210; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 210; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 210; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 210; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st210;
st210:
	if ( ++p == pe )
		goto _test_eof210;
case 210:
#line 28444 "scanner.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(640 + ((*p) - -128));
				if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			}
		} else {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 58 ) {
		if ( (*p) > 59 ) {
			if ( 60 <= (*p) )
 {				_widec = (short)(640 + ((*p) - -128));
				if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			}
		} else if ( (*p) >= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 777: goto st7;
		case 778: goto tr81;
		case 800: goto st7;
		case 808: goto tr79;
		case 809: goto tr80;
		case 827: goto st8;
		case 1033: goto st211;
		case 1034: goto tr670;
		case 1056: goto st211;
		case 1064: goto tr727;
		case 1065: goto tr728;
		case 1083: goto st161;
	}
	if ( 896 <= _widec && _widec <= 1151 )
		goto st148;
	goto tr77;
tr727:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st211;
tr728:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st211;
st211:
	if ( ++p == pe )
		goto _test_eof211;
case 211:
#line 28521 "scanner.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(640 + ((*p) - -128));
				if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			}
		} else {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 58 ) {
		if ( (*p) > 59 ) {
			if ( 60 <= (*p) )
 {				_widec = (short)(640 + ((*p) - -128));
				if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			}
		} else if ( (*p) >= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 777: goto st7;
		case 778: goto tr81;
		case 800: goto st7;
		case 808: goto tr79;
		case 809: goto tr80;
		case 827: goto st8;
		case 1033: goto st211;
		case 1034: goto tr670;
		case 1056: goto st211;
		case 1064: goto tr727;
		case 1065: goto tr728;
		case 1083: goto st161;
	}
	if ( 896 <= _widec && _widec <= 1151 )
		goto st148;
	goto tr83;
tr730:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st212;
tr731:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st212;
tr679:
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 212; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 212; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 212; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 212; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 212; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 212; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 212; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 212; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 212; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 212; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 212; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 212; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 212; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 212; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 212; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 212; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 212; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 212; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 212; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 212; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 212; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 212; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 212; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 212; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 212; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 212; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 212; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st212;
tr715:
#line 229 "./scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
#line 1856 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_A; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 212; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 212; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 212; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 212; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 212; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 212; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 212; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 212; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 212; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 212; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 212; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 212; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 212; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 212; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 212; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 212; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 212; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 212; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 212; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 212; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 212; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 212; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 212; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 212; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 212; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 212; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 212; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st212;
tr3210:
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 212; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 212; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 212; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 212; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 212; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 212; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 212; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 212; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 212; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 212; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 212; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 212; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 212; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 212; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 212; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 212; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 212; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 212; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 212; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 212; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 212; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 212; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 212; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 212; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 212; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 212; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 212; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
#line 232 "./scanner_body.rl"
	{
		if (s->r_owner_length == 0) {
			WARN(ZSCANNER_EBAD_PREVIOUS_OWNER);
			p--; {goto st246;}
		}
	}
	goto st212;
st212:
	if ( ++p == pe )
		goto _test_eof212;
case 212:
#line 28843 "scanner.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(640 + ((*p) - -128));
				if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			}
		} else {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 58 ) {
		if ( (*p) > 59 ) {
			if ( 60 <= (*p) )
 {				_widec = (short)(640 + ((*p) - -128));
				if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			}
		} else if ( (*p) >= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 777: goto st163;
		case 778: goto tr81;
		case 800: goto st163;
		case 808: goto tr673;
		case 809: goto tr674;
		case 1033: goto st212;
		case 1034: goto tr670;
		case 1056: goto st212;
		case 1064: goto tr730;
		case 1065: goto tr731;
	}
	if ( _widec > 895 ) {
		if ( 896 <= _widec && _widec <= 1151 )
			goto st178;
	} else if ( _widec >= 640 )
		goto st8;
	goto tr77;
st213:
	if ( ++p == pe )
		goto _test_eof213;
case 213:
	switch( (*p) ) {
		case 65: goto st214;
		case 97: goto st214;
	}
	goto tr35;
st214:
	if ( ++p == pe )
		goto _test_eof214;
case 214:
	switch( (*p) ) {
		case 65: goto st215;
		case 97: goto st215;
	}
	goto tr35;
st215:
	if ( ++p == pe )
		goto _test_eof215;
case 215:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr734;
		case 32: goto tr734;
		case 40: goto tr735;
		case 41: goto tr736;
		case 2058: goto tr737;
		case 2107: goto tr738;
		case 2314: goto tr739;
		case 2363: goto tr739;
		case 2570: goto tr740;
		case 2619: goto tr741;
	}
	goto tr55;
st216:
	if ( ++p == pe )
		goto _test_eof216;
case 216:
	switch( (*p) ) {
		case 83: goto st217;
		case 115: goto st217;
	}
	goto tr35;
st217:
	if ( ++p == pe )
		goto _test_eof217;
case 217:
	switch( (*p) ) {
		case 68: goto st218;
		case 100: goto st218;
	}
	goto tr35;
st218:
	if ( ++p == pe )
		goto _test_eof218;
case 218:
	switch( (*p) ) {
		case 66: goto st219;
		case 98: goto st219;
	}
	goto tr35;
st219:
	if ( ++p == pe )
		goto _test_eof219;
case 219:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr745;
		case 32: goto tr745;
		case 40: goto tr746;
		case 41: goto tr747;
		case 2058: goto tr748;
		case 2107: goto tr749;
		case 2314: goto tr750;
		case 2363: goto tr750;
		case 2570: goto tr751;
		case 2619: goto tr752;
	}
	goto tr55;
st220:
	if ( ++p == pe )
		goto _test_eof220;
case 220:
	switch( (*p) ) {
		case 76: goto st221;
		case 108: goto st221;
	}
	goto tr35;
st221:
	if ( ++p == pe )
		goto _test_eof221;
case 221:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr754;
		case 32: goto tr754;
		case 40: goto tr755;
		case 41: goto tr756;
		case 2058: goto tr757;
		case 2107: goto tr758;
		case 2314: goto tr759;
		case 2363: goto tr759;
		case 2570: goto tr760;
		case 2619: goto tr761;
	}
	goto tr55;
tr3149:
#line 232 "./scanner_body.rl"
	{
		if (s->r_owner_length == 0) {
			WARN(ZSCANNER_EBAD_PREVIOUS_OWNER);
			p--; {goto st246;}
		}
	}
	goto st222;
st222:
	if ( ++p == pe )
		goto _test_eof222;
case 222:
#line 29075 "scanner.c"
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 778: goto tr19;
		case 1034: goto tr113;
	}
	if ( _widec > 895 ) {
		if ( 896 <= _widec && _widec <= 1151 )
			goto st222;
	} else if ( _widec >= 640 )
		goto st137;
	goto tr77;
tr116:
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 223; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 223; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 223; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 223; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 223; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 223; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 223; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 223; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 223; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 223; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 223; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 223; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 223; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 223; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 223; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 223; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 223; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 223; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 223; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 223; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 223; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 223; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 223; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 223; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 223; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 223; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 223; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st223;
st223:
	if ( ++p == pe )
		goto _test_eof223;
case 223:
#line 29187 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st224;
		case 32: goto st224;
		case 40: goto tr763;
		case 41: goto tr764;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr81;
		case 1083: goto st8;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto st137;
	} else if ( _widec > 58 ) {
		if ( 60 <= _widec )
			goto st137;
	} else
		goto st137;
	goto tr77;
tr763:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st224;
tr764:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st224;
st224:
	if ( ++p == pe )
		goto _test_eof224;
case 224:
#line 29245 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st224;
		case 32: goto st224;
		case 40: goto tr763;
		case 41: goto tr764;
		case 778: goto tr81;
		case 827: goto st8;
		case 1034: goto tr81;
		case 1083: goto st8;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto st137;
	} else if ( _widec > 58 ) {
		if ( 60 <= _widec )
			goto st137;
	} else
		goto st137;
	goto tr83;
tr766:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st225;
tr767:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st225;
tr118:
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 225; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 225; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 225; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 225; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 225; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 225; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 225; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 225; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 225; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 225; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 225; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 225; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 225; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 225; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 225; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 225; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 225; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 225; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 225; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 225; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 225; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 225; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 225; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 225; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 225; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 225; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 225; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st225;
tr3151:
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 225; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 225; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 225; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 225; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 225; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 225; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 225; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 225; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 225; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 225; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 225; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 225; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 225; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 225; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 225; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 225; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 225; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 225; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 225; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 225; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 225; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 225; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 225; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 225; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 225; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 225; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 225; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
#line 232 "./scanner_body.rl"
	{
		if (s->r_owner_length == 0) {
			WARN(ZSCANNER_EBAD_PREVIOUS_OWNER);
			p--; {goto st246;}
		}
	}
	goto st225;
st225:
	if ( ++p == pe )
		goto _test_eof225;
case 225:
#line 29462 "scanner.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(640 + ((*p) - -128));
				if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			}
		} else {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 58 ) {
		if ( (*p) > 59 ) {
			if ( 60 <= (*p) )
 {				_widec = (short)(640 + ((*p) - -128));
				if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			}
		} else if ( (*p) >= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 777: goto st224;
		case 778: goto tr81;
		case 800: goto st224;
		case 808: goto tr763;
		case 809: goto tr764;
		case 827: goto st8;
		case 1033: goto st225;
		case 1034: goto tr670;
		case 1056: goto st225;
		case 1064: goto tr766;
		case 1065: goto tr767;
		case 1083: goto st178;
	}
	if ( _widec > 895 ) {
		if ( 896 <= _widec && _widec <= 1151 )
			goto st222;
	} else if ( _widec >= 640 )
		goto st137;
	goto tr77;
tr769:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st226;
tr770:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st226;
tr662:
#line 1856 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_A; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 226; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 226; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 226; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 226; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 226; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 226; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 226; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 226; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 226; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 226; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 226; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 226; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 226; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 226; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 226; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 226; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 226; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 226; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 226; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 226; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 226; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 226; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 226; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 226; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 226; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 226; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 226; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st226;
tr92:
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 226; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 226; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 226; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 226; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 226; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 226; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 226; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 226; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 226; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 226; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 226; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 226; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 226; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 226; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 226; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 226; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 226; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 226; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 226; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 226; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 226; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 226; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 226; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 226; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 226; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 226; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 226; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st226;
st226:
	if ( ++p == pe )
		goto _test_eof226;
case 226:
#line 29700 "scanner.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(640 + ((*p) - -128));
				if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			}
		} else {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 58 ) {
		if ( (*p) > 59 ) {
			if ( 60 <= (*p) )
 {				_widec = (short)(640 + ((*p) - -128));
				if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			}
		} else if ( (*p) >= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 777: goto st163;
		case 778: goto tr81;
		case 800: goto st163;
		case 808: goto tr673;
		case 809: goto tr674;
		case 1033: goto st226;
		case 1034: goto tr81;
		case 1056: goto st226;
		case 1064: goto tr769;
		case 1065: goto tr770;
	}
	if ( _widec > 895 ) {
		if ( 896 <= _widec && _widec <= 1151 )
			goto st12;
	} else if ( _widec >= 640 )
		goto st8;
	goto tr77;
tr643:
#line 229 "./scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
#line 1856 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_A; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	goto st227;
st227:
	if ( ++p == pe )
		goto _test_eof227;
case 227:
#line 29770 "scanner.c"
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	if ( _widec == 1034 )
		goto tr649;
	if ( 896 <= _widec && _widec <= 1151 )
		goto st227;
	goto tr69;
tr772:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st228;
tr773:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st228;
tr652:
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 228; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 228; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 228; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 228; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 228; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 228; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 228; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 228; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 228; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 228; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 228; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 228; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 228; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 228; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 228; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 228; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 228; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 228; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 228; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 228; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 228; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 228; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 228; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 228; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 228; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 228; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 228; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st228;
tr645:
#line 229 "./scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
#line 1856 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_A; }
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = 228; goto st589;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = 228; goto st591;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = 228; goto st593;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = 228; goto st625;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = 228; goto st630;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = 228; goto st635;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = 228; goto st640;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = 228; goto st644;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = 228; goto st646;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = 228; goto st701;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = 228; goto st712;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = 228; goto st729;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = 228; goto st740;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = 228; goto st751;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = 228; goto st764;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = 228; goto st774;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = 228; goto st813;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = 228; goto st955;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = 228; goto st958;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = 228; goto st969;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = 228; goto st971;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = 228; goto st1000;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = 228; goto st1013;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = 228; goto st1031;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = 228; goto st1026;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = 228; goto st1044;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = 228; goto st1050;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{goto st246;}
		}
	}
	goto st228;
st228:
	if ( ++p == pe )
		goto _test_eof228;
case 228:
#line 29983 "scanner.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(640 + ((*p) - -128));
				if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			}
		} else {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 58 ) {
		if ( (*p) > 59 ) {
			if ( 60 <= (*p) )
 {				_widec = (short)(640 + ((*p) - -128));
				if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			}
		} else if ( (*p) >= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 777: goto st7;
		case 778: goto tr81;
		case 800: goto st7;
		case 808: goto tr79;
		case 809: goto tr80;
		case 827: goto st8;
		case 1033: goto st228;
		case 1034: goto tr670;
		case 1056: goto st228;
		case 1064: goto tr772;
		case 1065: goto tr773;
		case 1083: goto st178;
	}
	if ( 896 <= _widec && _widec <= 1151 )
		goto st227;
	goto tr77;
tr3157:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 229; goto st248;} }
	goto st229;
tr3175:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 229; goto st248;} }
	goto st229;
st229:
	if ( ++p == pe )
		goto _test_eof229;
case 229:
#line 30070 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr603;
		case 32: goto tr603;
		case 40: goto tr604;
		case 41: goto tr605;
		case 69: goto st17;
		case 78: goto st21;
		case 101: goto st17;
		case 110: goto st21;
		case 1034: goto tr606;
		case 1083: goto tr607;
	}
	goto tr774;
tr3158:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 230; goto st248;} }
	goto st230;
tr3176:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 230; goto st248;} }
	goto st230;
st230:
	if ( ++p == pe )
		goto _test_eof230;
case 230:
#line 30132 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr603;
		case 32: goto tr603;
		case 40: goto tr604;
		case 41: goto tr605;
		case 72: goto st26;
		case 78: goto st30;
		case 83: goto st38;
		case 104: goto st26;
		case 110: goto st30;
		case 115: goto st38;
		case 1034: goto tr606;
		case 1083: goto tr607;
	}
	goto tr774;
tr3159:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 231; goto st248;} }
	goto st231;
tr3177:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 231; goto st248;} }
	goto st231;
st231:
	if ( ++p == pe )
		goto _test_eof231;
case 231:
#line 30196 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr603;
		case 32: goto tr603;
		case 40: goto tr604;
		case 41: goto tr605;
		case 85: goto st40;
		case 117: goto st40;
		case 1034: goto tr606;
		case 1083: goto tr607;
	}
	goto tr774;
tr3160:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 232; goto st248;} }
	goto st232;
tr3178:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 232; goto st248;} }
	goto st232;
st232:
	if ( ++p == pe )
		goto _test_eof232;
case 232:
#line 30256 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr603;
		case 32: goto tr603;
		case 40: goto tr604;
		case 41: goto tr605;
		case 73: goto st47;
		case 105: goto st47;
		case 1034: goto tr606;
		case 1083: goto tr607;
	}
	goto tr774;
tr3179:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 233; goto st248;} }
	goto st233;
st233:
	if ( ++p == pe )
		goto _test_eof233;
case 233:
#line 30303 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr603;
		case 32: goto tr603;
		case 40: goto tr604;
		case 41: goto tr605;
		case 78: goto st130;
		case 80: goto st55;
		case 110: goto st130;
		case 112: goto st55;
		case 1034: goto tr606;
		case 1083: goto tr607;
	}
	goto tr774;
tr3162:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 234; goto st248;} }
	goto st234;
tr3180:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 234; goto st248;} }
	goto st234;
st234:
	if ( ++p == pe )
		goto _test_eof234;
case 234:
#line 30365 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr603;
		case 32: goto tr603;
		case 40: goto tr604;
		case 41: goto tr605;
		case 69: goto st63;
		case 88: goto st65;
		case 101: goto st63;
		case 120: goto st65;
		case 1034: goto tr606;
		case 1083: goto tr607;
	}
	goto tr774;
tr3163:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 235; goto st248;} }
	goto st235;
tr3181:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 235; goto st248;} }
	goto st235;
st235:
	if ( ++p == pe )
		goto _test_eof235;
case 235:
#line 30427 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr603;
		case 32: goto tr603;
		case 40: goto tr604;
		case 41: goto tr605;
		case 51: goto st67;
		case 54: goto st69;
		case 79: goto st71;
		case 80: goto st73;
		case 111: goto st71;
		case 112: goto st73;
		case 1034: goto tr606;
		case 1083: goto tr607;
	}
	goto tr774;
tr3164:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 236; goto st248;} }
	goto st236;
tr3182:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 236; goto st248;} }
	goto st236;
st236:
	if ( ++p == pe )
		goto _test_eof236;
case 236:
#line 30491 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr603;
		case 32: goto tr603;
		case 40: goto tr604;
		case 41: goto tr605;
		case 73: goto st75;
		case 88: goto st79;
		case 105: goto st75;
		case 120: goto st79;
		case 1034: goto tr606;
		case 1083: goto tr607;
	}
	goto tr774;
tr3165:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 237; goto st248;} }
	goto st237;
tr3183:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 237; goto st248;} }
	goto st237;
st237:
	if ( ++p == pe )
		goto _test_eof237;
case 237:
#line 30553 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr603;
		case 32: goto tr603;
		case 40: goto tr604;
		case 41: goto tr605;
		case 65: goto st81;
		case 73: goto st85;
		case 83: goto st87;
		case 97: goto st81;
		case 105: goto st85;
		case 115: goto st87;
		case 1034: goto tr606;
		case 1083: goto tr607;
	}
	goto tr774;
tr3166:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 238; goto st248;} }
	goto st238;
tr3184:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 238; goto st248;} }
	goto st238;
st238:
	if ( ++p == pe )
		goto _test_eof238;
case 238:
#line 30617 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr603;
		case 32: goto tr603;
		case 40: goto tr604;
		case 41: goto tr605;
		case 84: goto st97;
		case 116: goto st97;
		case 1034: goto tr606;
		case 1083: goto tr607;
	}
	goto tr774;
tr3167:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 239; goto st248;} }
	goto st239;
tr3185:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 239; goto st248;} }
	goto st239;
st239:
	if ( ++p == pe )
		goto _test_eof239;
case 239:
#line 30677 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr603;
		case 32: goto tr603;
		case 40: goto tr604;
		case 41: goto tr605;
		case 80: goto st100;
		case 82: goto st101;
		case 84: goto st105;
		case 112: goto st100;
		case 114: goto st101;
		case 116: goto st105;
		case 1034: goto tr606;
		case 1083: goto tr607;
	}
	goto tr774;
tr3168:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 240; goto st248;} }
	goto st240;
tr3186:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 240; goto st248;} }
	goto st240;
st240:
	if ( ++p == pe )
		goto _test_eof240;
case 240:
#line 30741 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr603;
		case 32: goto tr603;
		case 40: goto tr604;
		case 41: goto tr605;
		case 79: goto st107;
		case 80: goto st109;
		case 82: goto st111;
		case 83: goto st113;
		case 111: goto st107;
		case 112: goto st109;
		case 114: goto st111;
		case 115: goto st113;
		case 1034: goto tr606;
		case 1083: goto tr607;
	}
	goto tr774;
tr3169:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 241; goto st248;} }
	goto st241;
tr3187:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 241; goto st248;} }
	goto st241;
st241:
	if ( ++p == pe )
		goto _test_eof241;
case 241:
#line 30807 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr603;
		case 32: goto tr603;
		case 40: goto tr604;
		case 41: goto tr605;
		case 76: goto st118;
		case 88: goto st121;
		case 89: goto st123;
		case 108: goto st118;
		case 120: goto st121;
		case 121: goto st123;
		case 1034: goto tr606;
		case 1083: goto tr607;
	}
	goto tr774;
tr3141:
#line 232 "./scanner_body.rl"
	{
		if (s->r_owner_length == 0) {
			WARN(ZSCANNER_EBAD_PREVIOUS_OWNER);
			p--; {goto st246;}
		}
	}
	goto st242;
tr3197:
#line 704 "./scanner_body.rl"
	{
		s->stop = false;
	}
#line 232 "./scanner_body.rl"
	{
		if (s->r_owner_length == 0) {
			WARN(ZSCANNER_EBAD_PREVIOUS_OWNER);
			p--; {goto st246;}
		}
	}
	goto st242;
st242:
	if ( ++p == pe )
		goto _test_eof242;
case 242:
#line 30863 "scanner.c"
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 778: goto tr19;
		case 1034: goto tr21;
	}
	if ( _widec > 895 ) {
		if ( 896 <= _widec && _widec <= 1151 )
			goto st242;
	} else if ( _widec >= 640 )
		goto st137;
	goto tr83;
tr3171:
#line 232 "./scanner_body.rl"
	{
		if (s->r_owner_length == 0) {
			WARN(ZSCANNER_EBAD_PREVIOUS_OWNER);
			p--; {goto st246;}
		}
	}
	goto st243;
st243:
	if ( ++p == pe )
		goto _test_eof243;
case 243:
#line 30908 "scanner.c"
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 778: goto tr19;
		case 1034: goto tr539;
	}
	if ( _widec > 895 ) {
		if ( 896 <= _widec && _widec <= 1151 )
			goto st243;
	} else if ( _widec >= 640 )
		goto st137;
	goto tr125;
tr3161:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = 244; goto st248;} }
	goto st244;
st244:
	if ( ++p == pe )
		goto _test_eof244;
case 244:
#line 30957 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr603;
		case 32: goto tr603;
		case 40: goto tr604;
		case 41: goto tr605;
		case 78: goto st52;
		case 80: goto st55;
		case 110: goto st52;
		case 112: goto st55;
		case 1034: goto tr606;
		case 1083: goto tr607;
	}
	goto tr774;
tr124:
#line 731 "./scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st245;
st245:
	if ( ++p == pe )
		goto _test_eof245;
case 245:
#line 31000 "scanner.c"
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 778: goto tr81;
		case 1034: goto tr129;
	}
	if ( _widec > 895 ) {
		if ( 896 <= _widec && _widec <= 1151 )
			goto st245;
	} else if ( _widec >= 640 )
		goto st8;
	goto tr125;
st246:
	if ( ++p == pe )
		goto _test_eof246;
case 246:
	if ( (*p) == 10 )
		goto tr776;
	goto tr775;
tr775:
#line 67 "./scanner_body.rl"
	{
		s->buffer_length = 0;
	}
#line 70 "./scanner_body.rl"
	{
		if ((*p) == '\r') {
			ERR(ZSCANNER_DOS_NEWLINE);
		}

		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st247;
tr777:
#line 70 "./scanner_body.rl"
	{
		if ((*p) == '\r') {
			ERR(ZSCANNER_DOS_NEWLINE);
		}

		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	goto st247;
st247:
	if ( ++p == pe )
		goto _test_eof247;
case 247:
#line 31071 "scanner.c"
	if ( (*p) == 10 )
		goto tr778;
	goto tr777;
tr776:
#line 67 "./scanner_body.rl"
	{
		s->buffer_length = 0;
	}
#line 79 "./scanner_body.rl"
	{
		// Ending string in buffer.
		s->buffer[s->buffer_length++] = 0;

		// Error counter incrementation.
		s->error_counter++;

		// Initialization of fcall stack.
		top = 0;

		// Process error message.
		s->process_error(s);

		// Reset.
		s->error_code = ZSCANNER_OK;
		s->multiline = false;

		// In case of serious error, stop scanner.
		if (s->stop == true) {
			return -1;
		}
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 104 "./scanner_body.rl"
	{ {goto st1056;} }
	goto st1066;
tr778:
#line 79 "./scanner_body.rl"
	{
		// Ending string in buffer.
		s->buffer[s->buffer_length++] = 0;

		// Error counter incrementation.
		s->error_counter++;

		// Initialization of fcall stack.
		top = 0;

		// Process error message.
		s->process_error(s);

		// Reset.
		s->error_code = ZSCANNER_OK;
		s->multiline = false;

		// In case of serious error, stop scanner.
		if (s->stop == true) {
			return -1;
		}
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 104 "./scanner_body.rl"
	{ {goto st1056;} }
	goto st1066;
st1066:
	if ( ++p == pe )
		goto _test_eof1066;
case 1066:
#line 31145 "scanner.c"
	goto st0;
st248:
	if ( ++p == pe )
		goto _test_eof248;
case 248:
	switch( (*p) ) {
		case 42: goto tr780;
		case 46: goto tr781;
		case 64: goto st256;
		case 92: goto tr783;
		case 95: goto tr780;
	}
	if ( (*p) < 65 ) {
		if ( 45 <= (*p) && (*p) <= 57 )
			goto tr780;
	} else if ( (*p) > 90 ) {
		if ( 97 <= (*p) && (*p) <= 122 )
			goto tr780;
	} else
		goto tr780;
	goto tr779;
tr780:
#line 189 "./scanner_body.rl"
	{
		s->item_length_position = 0;
		s->dname_tmp_length = 0;
	}
#line 108 "./scanner_body.rl"
	{
		s->item_length = 0;
		s->item_length_position = s->dname_tmp_length++;
	}
#line 112 "./scanner_body.rl"
	{
		if (s->item_length < MAX_LABEL_LENGTH) {
			(s->dname)[s->dname_tmp_length++] = (*p);
			s->item_length++;
		} else {
			WARN(ZSCANNER_ELABEL_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st249;
tr785:
#line 112 "./scanner_body.rl"
	{
		if (s->item_length < MAX_LABEL_LENGTH) {
			(s->dname)[s->dname_tmp_length++] = (*p);
			s->item_length++;
		} else {
			WARN(ZSCANNER_ELABEL_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st249;
tr789:
#line 108 "./scanner_body.rl"
	{
		s->item_length = 0;
		s->item_length_position = s->dname_tmp_length++;
	}
#line 112 "./scanner_body.rl"
	{
		if (s->item_length < MAX_LABEL_LENGTH) {
			(s->dname)[s->dname_tmp_length++] = (*p);
			s->item_length++;
		} else {
			WARN(ZSCANNER_ELABEL_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st249;
tr796:
#line 144 "./scanner_body.rl"
	{
		s->dname_tmp_length++;
	}
#line 112 "./scanner_body.rl"
	{
		if (s->item_length < MAX_LABEL_LENGTH) {
			(s->dname)[s->dname_tmp_length++] = (*p);
			s->item_length++;
		} else {
			WARN(ZSCANNER_ELABEL_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st249;
st249:
	if ( ++p == pe )
		goto _test_eof249;
case 249:
#line 31238 "scanner.c"
	switch( (*p) ) {
		case 32: goto tr784;
		case 42: goto tr785;
		case 46: goto tr786;
		case 59: goto tr784;
		case 92: goto st251;
		case 95: goto tr785;
	}
	if ( (*p) < 45 ) {
		if ( (*p) > 10 ) {
			if ( 40 <= (*p) && (*p) <= 41 )
				goto tr784;
		} else if ( (*p) >= 9 )
			goto tr784;
	} else if ( (*p) > 57 ) {
		if ( (*p) > 90 ) {
			if ( 97 <= (*p) && (*p) <= 122 )
				goto tr785;
		} else if ( (*p) >= 65 )
			goto tr785;
	} else
		goto tr785;
	goto tr779;
tr784:
#line 121 "./scanner_body.rl"
	{
		if (s->dname_tmp_length < MAX_DNAME_LENGTH) {
			(s->dname)[s->item_length_position] =
				(uint8_t)(s->item_length);
		} else {
			WARN(ZSCANNER_EDNAME_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 169 "./scanner_body.rl"
	{
		memcpy(s->dname + s->dname_tmp_length,
		       s->zone_origin,
		       s->zone_origin_length);

		s->dname_tmp_length += s->zone_origin_length;

		if (s->dname_tmp_length > MAX_DNAME_LENGTH) {
			WARN(ZSCANNER_EDNAME_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1067;
tr788:
#line 166 "./scanner_body.rl"
	{
		(s->dname)[s->dname_tmp_length++] = 0;
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1067;
tr795:
#line 144 "./scanner_body.rl"
	{
		s->dname_tmp_length++;
	}
#line 121 "./scanner_body.rl"
	{
		if (s->dname_tmp_length < MAX_DNAME_LENGTH) {
			(s->dname)[s->item_length_position] =
				(uint8_t)(s->item_length);
		} else {
			WARN(ZSCANNER_EDNAME_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 169 "./scanner_body.rl"
	{
		memcpy(s->dname + s->dname_tmp_length,
		       s->zone_origin,
		       s->zone_origin_length);

		s->dname_tmp_length += s->zone_origin_length;

		if (s->dname_tmp_length > MAX_DNAME_LENGTH) {
			WARN(ZSCANNER_EDNAME_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1067;
tr799:
#line 181 "./scanner_body.rl"
	{
		memcpy(s->dname,
		       s->zone_origin,
		       s->zone_origin_length);

		s->dname_tmp_length = s->zone_origin_length;
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1067;
st1067:
	if ( ++p == pe )
		goto _test_eof1067;
case 1067:
#line 31352 "scanner.c"
	goto st0;
tr786:
#line 121 "./scanner_body.rl"
	{
		if (s->dname_tmp_length < MAX_DNAME_LENGTH) {
			(s->dname)[s->item_length_position] =
				(uint8_t)(s->item_length);
		} else {
			WARN(ZSCANNER_EDNAME_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st250;
tr797:
#line 144 "./scanner_body.rl"
	{
		s->dname_tmp_length++;
	}
#line 121 "./scanner_body.rl"
	{
		if (s->dname_tmp_length < MAX_DNAME_LENGTH) {
			(s->dname)[s->item_length_position] =
				(uint8_t)(s->item_length);
		} else {
			WARN(ZSCANNER_EDNAME_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st250;
st250:
	if ( ++p == pe )
		goto _test_eof250;
case 250:
#line 31386 "scanner.c"
	switch( (*p) ) {
		case 32: goto tr788;
		case 42: goto tr789;
		case 45: goto tr789;
		case 59: goto tr788;
		case 92: goto tr790;
		case 95: goto tr789;
	}
	if ( (*p) < 47 ) {
		if ( (*p) > 10 ) {
			if ( 40 <= (*p) && (*p) <= 41 )
				goto tr788;
		} else if ( (*p) >= 9 )
			goto tr788;
	} else if ( (*p) > 57 ) {
		if ( (*p) > 90 ) {
			if ( 97 <= (*p) && (*p) <= 122 )
				goto tr789;
		} else if ( (*p) >= 65 )
			goto tr789;
	} else
		goto tr789;
	goto tr779;
tr783:
#line 189 "./scanner_body.rl"
	{
		s->item_length_position = 0;
		s->dname_tmp_length = 0;
	}
#line 108 "./scanner_body.rl"
	{
		s->item_length = 0;
		s->item_length_position = s->dname_tmp_length++;
	}
	goto st251;
tr790:
#line 108 "./scanner_body.rl"
	{
		s->item_length = 0;
		s->item_length_position = s->dname_tmp_length++;
	}
	goto st251;
tr798:
#line 144 "./scanner_body.rl"
	{
		s->dname_tmp_length++;
	}
	goto st251;
st251:
	if ( ++p == pe )
		goto _test_eof251;
case 251:
#line 31439 "scanner.c"
	if ( 48 <= (*p) && (*p) <= 57 )
		goto tr791;
	goto tr785;
tr791:
#line 131 "./scanner_body.rl"
	{
		if (s->item_length < MAX_LABEL_LENGTH) {
			(s->dname)[s->dname_tmp_length] = 0;
			s->item_length++;
		} else {
			WARN(ZSCANNER_ELABEL_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 140 "./scanner_body.rl"
	{
		(s->dname)[s->dname_tmp_length] *= 10;
		(s->dname)[s->dname_tmp_length] += digit_to_num[(uint8_t)(*p)];
	}
	goto st252;
st252:
	if ( ++p == pe )
		goto _test_eof252;
case 252:
#line 31464 "scanner.c"
	if ( 48 <= (*p) && (*p) <= 57 )
		goto tr793;
	goto tr792;
tr793:
#line 140 "./scanner_body.rl"
	{
		(s->dname)[s->dname_tmp_length] *= 10;
		(s->dname)[s->dname_tmp_length] += digit_to_num[(uint8_t)(*p)];
	}
	goto st253;
st253:
	if ( ++p == pe )
		goto _test_eof253;
case 253:
#line 31479 "scanner.c"
	if ( 48 <= (*p) && (*p) <= 57 )
		goto tr794;
	goto tr792;
tr794:
#line 140 "./scanner_body.rl"
	{
		(s->dname)[s->dname_tmp_length] *= 10;
		(s->dname)[s->dname_tmp_length] += digit_to_num[(uint8_t)(*p)];
	}
	goto st254;
st254:
	if ( ++p == pe )
		goto _test_eof254;
case 254:
#line 31494 "scanner.c"
	switch( (*p) ) {
		case 32: goto tr795;
		case 42: goto tr796;
		case 46: goto tr797;
		case 59: goto tr795;
		case 92: goto tr798;
		case 95: goto tr796;
	}
	if ( (*p) < 45 ) {
		if ( (*p) > 10 ) {
			if ( 40 <= (*p) && (*p) <= 41 )
				goto tr795;
		} else if ( (*p) >= 9 )
			goto tr795;
	} else if ( (*p) > 57 ) {
		if ( (*p) > 90 ) {
			if ( 97 <= (*p) && (*p) <= 122 )
				goto tr796;
		} else if ( (*p) >= 65 )
			goto tr796;
	} else
		goto tr796;
	goto tr792;
tr781:
#line 189 "./scanner_body.rl"
	{
		s->item_length_position = 0;
		s->dname_tmp_length = 0;
	}
	goto st255;
st255:
	if ( ++p == pe )
		goto _test_eof255;
case 255:
#line 31529 "scanner.c"
	switch( (*p) ) {
		case 32: goto tr788;
		case 59: goto tr788;
	}
	if ( (*p) > 10 ) {
		if ( 40 <= (*p) && (*p) <= 41 )
			goto tr788;
	} else if ( (*p) >= 9 )
		goto tr788;
	goto tr779;
st256:
	if ( ++p == pe )
		goto _test_eof256;
case 256:
	switch( (*p) ) {
		case 32: goto tr799;
		case 59: goto tr799;
	}
	if ( (*p) > 10 ) {
		if ( 40 <= (*p) && (*p) <= 41 )
			goto tr799;
	} else if ( (*p) >= 9 )
		goto tr799;
	goto tr779;
st257:
	if ( ++p == pe )
		goto _test_eof257;
case 257:
	switch( (*p) ) {
		case 34: goto st263;
		case 92: goto st259;
	}
	if ( (*p) > 58 ) {
		if ( 60 <= (*p) && (*p) <= 126 )
			goto tr801;
	} else if ( (*p) >= 33 )
		goto tr801;
	goto tr800;
tr801:
#line 497 "./scanner_body.rl"
	{
		if (rdata_tail <= rdata_stop) {
			*(rdata_tail++) = (*p);
		} else {
			WARN(ZSCANNER_ETEXT_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st258;
tr811:
#line 536 "./scanner_body.rl"
	{
		rdata_tail++;
	}
#line 497 "./scanner_body.rl"
	{
		if (rdata_tail <= rdata_stop) {
			*(rdata_tail++) = (*p);
		} else {
			WARN(ZSCANNER_ETEXT_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st258;
st258:
	if ( ++p == pe )
		goto _test_eof258;
case 258:
#line 31598 "scanner.c"
	switch( (*p) ) {
		case 32: goto tr804;
		case 33: goto tr801;
		case 59: goto tr804;
		case 92: goto st259;
	}
	if ( (*p) < 35 ) {
		if ( 9 <= (*p) && (*p) <= 10 )
			goto tr804;
	} else if ( (*p) > 39 ) {
		if ( (*p) > 41 ) {
			if ( 42 <= (*p) && (*p) <= 126 )
				goto tr801;
		} else if ( (*p) >= 40 )
			goto tr805;
	} else
		goto tr801;
	goto tr800;
tr804:
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1068;
tr810:
#line 536 "./scanner_body.rl"
	{
		rdata_tail++;
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1068;
st1068:
	if ( ++p == pe )
		goto _test_eof1068;
case 1068:
#line 31637 "scanner.c"
	goto st0;
tr805:
#line 497 "./scanner_body.rl"
	{
		if (rdata_tail <= rdata_stop) {
			*(rdata_tail++) = (*p);
		} else {
			WARN(ZSCANNER_ETEXT_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1069;
tr812:
#line 536 "./scanner_body.rl"
	{
		rdata_tail++;
	}
#line 497 "./scanner_body.rl"
	{
		if (rdata_tail <= rdata_stop) {
			*(rdata_tail++) = (*p);
		} else {
			WARN(ZSCANNER_ETEXT_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1069;
st1069:
	if ( ++p == pe )
		goto _test_eof1069;
case 1069:
#line 31677 "scanner.c"
	switch( (*p) ) {
		case 32: goto tr804;
		case 33: goto tr801;
		case 59: goto tr804;
		case 92: goto st259;
	}
	if ( (*p) < 35 ) {
		if ( 9 <= (*p) && (*p) <= 10 )
			goto tr804;
	} else if ( (*p) > 39 ) {
		if ( (*p) > 41 ) {
			if ( 42 <= (*p) && (*p) <= 126 )
				goto tr801;
		} else if ( (*p) >= 40 )
			goto tr805;
	} else
		goto tr801;
	goto tr800;
tr813:
#line 536 "./scanner_body.rl"
	{
		rdata_tail++;
	}
	goto st259;
st259:
	if ( ++p == pe )
		goto _test_eof259;
case 259:
#line 31706 "scanner.c"
	if ( 48 <= (*p) && (*p) <= 57 )
		goto tr806;
	goto tr801;
tr806:
#line 514 "./scanner_body.rl"
	{
		if (rdata_tail <= rdata_stop) {
			*rdata_tail = 0;
			s->item_length++;
		} else {
			WARN(ZSCANNER_ETEXT_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 523 "./scanner_body.rl"
	{
		if ((*rdata_tail < (UINT8_MAX / 10)) ||   // Dominant fast check.
			((*rdata_tail == (UINT8_MAX / 10)) && // Marginal case.
			 ((*p) <= (UINT8_MAX % 10) + ASCII_0)
			)
		   ) {
			*rdata_tail *= 10;
			*rdata_tail += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ENUMBER8_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st260;
st260:
	if ( ++p == pe )
		goto _test_eof260;
case 260:
#line 31740 "scanner.c"
	if ( 48 <= (*p) && (*p) <= 57 )
		goto tr808;
	goto tr807;
tr808:
#line 523 "./scanner_body.rl"
	{
		if ((*rdata_tail < (UINT8_MAX / 10)) ||   // Dominant fast check.
			((*rdata_tail == (UINT8_MAX / 10)) && // Marginal case.
			 ((*p) <= (UINT8_MAX % 10) + ASCII_0)
			)
		   ) {
			*rdata_tail *= 10;
			*rdata_tail += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ENUMBER8_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st261;
st261:
	if ( ++p == pe )
		goto _test_eof261;
case 261:
#line 31764 "scanner.c"
	if ( 48 <= (*p) && (*p) <= 57 )
		goto tr809;
	goto tr807;
tr809:
#line 523 "./scanner_body.rl"
	{
		if ((*rdata_tail < (UINT8_MAX / 10)) ||   // Dominant fast check.
			((*rdata_tail == (UINT8_MAX / 10)) && // Marginal case.
			 ((*p) <= (UINT8_MAX % 10) + ASCII_0)
			)
		   ) {
			*rdata_tail *= 10;
			*rdata_tail += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ENUMBER8_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st262;
st262:
	if ( ++p == pe )
		goto _test_eof262;
case 262:
#line 31788 "scanner.c"
	switch( (*p) ) {
		case 32: goto tr810;
		case 33: goto tr811;
		case 59: goto tr810;
		case 92: goto tr813;
	}
	if ( (*p) < 35 ) {
		if ( 9 <= (*p) && (*p) <= 10 )
			goto tr810;
	} else if ( (*p) > 39 ) {
		if ( (*p) > 41 ) {
			if ( 42 <= (*p) && (*p) <= 126 )
				goto tr811;
		} else if ( (*p) >= 40 )
			goto tr812;
	} else
		goto tr811;
	goto tr807;
tr814:
#line 497 "./scanner_body.rl"
	{
		if (rdata_tail <= rdata_stop) {
			*(rdata_tail++) = (*p);
		} else {
			WARN(ZSCANNER_ETEXT_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st263;
tr821:
#line 536 "./scanner_body.rl"
	{
		rdata_tail++;
	}
#line 497 "./scanner_body.rl"
	{
		if (rdata_tail <= rdata_stop) {
			*(rdata_tail++) = (*p);
		} else {
			WARN(ZSCANNER_ETEXT_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st263;
st263:
	if ( ++p == pe )
		goto _test_eof263;
case 263:
#line 31837 "scanner.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 555 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr814;
		case 34: goto st264;
		case 92: goto st265;
		case 522: goto tr814;
	}
	if ( 32 <= _widec && _widec <= 126 )
		goto tr814;
	goto tr800;
tr822:
#line 536 "./scanner_body.rl"
	{
		rdata_tail++;
	}
	goto st264;
st264:
	if ( ++p == pe )
		goto _test_eof264;
case 264:
#line 31864 "scanner.c"
	switch( (*p) ) {
		case 32: goto tr804;
		case 59: goto tr804;
	}
	if ( (*p) > 10 ) {
		if ( 40 <= (*p) && (*p) <= 41 )
			goto tr804;
	} else if ( (*p) >= 9 )
		goto tr804;
	goto tr817;
tr823:
#line 536 "./scanner_body.rl"
	{
		rdata_tail++;
	}
	goto st265;
st265:
	if ( ++p == pe )
		goto _test_eof265;
case 265:
#line 31885 "scanner.c"
	if ( 48 <= (*p) && (*p) <= 57 )
		goto tr818;
	goto tr814;
tr818:
#line 514 "./scanner_body.rl"
	{
		if (rdata_tail <= rdata_stop) {
			*rdata_tail = 0;
			s->item_length++;
		} else {
			WARN(ZSCANNER_ETEXT_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 523 "./scanner_body.rl"
	{
		if ((*rdata_tail < (UINT8_MAX / 10)) ||   // Dominant fast check.
			((*rdata_tail == (UINT8_MAX / 10)) && // Marginal case.
			 ((*p) <= (UINT8_MAX % 10) + ASCII_0)
			)
		   ) {
			*rdata_tail *= 10;
			*rdata_tail += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ENUMBER8_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st266;
st266:
	if ( ++p == pe )
		goto _test_eof266;
case 266:
#line 31919 "scanner.c"
	if ( 48 <= (*p) && (*p) <= 57 )
		goto tr819;
	goto tr807;
tr819:
#line 523 "./scanner_body.rl"
	{
		if ((*rdata_tail < (UINT8_MAX / 10)) ||   // Dominant fast check.
			((*rdata_tail == (UINT8_MAX / 10)) && // Marginal case.
			 ((*p) <= (UINT8_MAX % 10) + ASCII_0)
			)
		   ) {
			*rdata_tail *= 10;
			*rdata_tail += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ENUMBER8_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st267;
st267:
	if ( ++p == pe )
		goto _test_eof267;
case 267:
#line 31943 "scanner.c"
	if ( 48 <= (*p) && (*p) <= 57 )
		goto tr820;
	goto tr807;
tr820:
#line 523 "./scanner_body.rl"
	{
		if ((*rdata_tail < (UINT8_MAX / 10)) ||   // Dominant fast check.
			((*rdata_tail == (UINT8_MAX / 10)) && // Marginal case.
			 ((*p) <= (UINT8_MAX % 10) + ASCII_0)
			)
		   ) {
			*rdata_tail *= 10;
			*rdata_tail += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ENUMBER8_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st268;
st268:
	if ( ++p == pe )
		goto _test_eof268;
case 268:
#line 31967 "scanner.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 555 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr821;
		case 34: goto tr822;
		case 92: goto tr823;
		case 522: goto tr821;
	}
	if ( 32 <= _widec && _widec <= 126 )
		goto tr821;
	goto tr807;
st269:
	if ( ++p == pe )
		goto _test_eof269;
case 269:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st270;
		case 32: goto st270;
		case 40: goto tr826;
		case 41: goto tr827;
		case 1034: goto tr828;
		case 1083: goto st277;
	}
	goto tr824;
tr826:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st270;
tr827:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st270;
tr828:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st270;
st270:
	if ( ++p == pe )
		goto _test_eof270;
case 270:
#line 32041 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st270;
		case 32: goto st270;
		case 40: goto tr826;
		case 41: goto tr827;
		case 1034: goto tr828;
		case 1083: goto st277;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr831;
	goto tr830;
tr831:
#line 278 "./scanner_body.rl"
	{
		s->number64 = 0;
	}
#line 261 "./scanner_body.rl"
	{
		// Overflow check: 10*(s->number64) + fc - ASCII_0 <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + ASCII_0)
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ENUMBER64_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st271;
tr836:
#line 261 "./scanner_body.rl"
	{
		// Overflow check: 10*(s->number64) + fc - ASCII_0 <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + ASCII_0)
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ENUMBER64_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st271;
st271:
	if ( ++p == pe )
		goto _test_eof271;
case 271:
#line 32109 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st272;
		case 32: goto st272;
		case 40: goto tr834;
		case 41: goto tr835;
		case 68: goto tr837;
		case 72: goto tr838;
		case 77: goto tr839;
		case 83: goto st274;
		case 87: goto tr841;
		case 100: goto tr837;
		case 104: goto tr838;
		case 109: goto tr839;
		case 115: goto st274;
		case 119: goto tr841;
		case 778: goto tr842;
		case 827: goto st273;
		case 1034: goto tr842;
		case 1083: goto st273;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr836;
	goto tr832;
tr834:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st272;
tr835:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st272;
tr853:
#line 423 "./scanner_body.rl"
	{
		if (s->number64 + s->number64_tmp < UINT32_MAX) {
			s->number64 += s->number64_tmp;
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st272;
tr854:
#line 423 "./scanner_body.rl"
	{
		if (s->number64 + s->number64_tmp < UINT32_MAX) {
			s->number64 += s->number64_tmp;
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st272;
tr855:
#line 423 "./scanner_body.rl"
	{
		if (s->number64 + s->number64_tmp < UINT32_MAX) {
			s->number64 += s->number64_tmp;
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st272;
st272:
	if ( ++p == pe )
		goto _test_eof272;
case 272:
#line 32220 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st272;
		case 32: goto st272;
		case 40: goto tr834;
		case 41: goto tr835;
		case 778: goto tr842;
		case 827: goto st273;
		case 1034: goto tr842;
		case 1083: goto st273;
	}
	goto tr844;
tr842:
#line 571 "./scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->default_ttl = (uint32_t)(s->number64);
		} else {
			ERR(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st1070;
tr857:
#line 423 "./scanner_body.rl"
	{
		if (s->number64 + s->number64_tmp < UINT32_MAX) {
			s->number64 += s->number64_tmp;
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 571 "./scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->default_ttl = (uint32_t)(s->number64);
		} else {
			ERR(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st1070;
st1070:
	if ( ++p == pe )
		goto _test_eof1070;
case 1070:
#line 32297 "scanner.c"
	goto st0;
tr858:
#line 423 "./scanner_body.rl"
	{
		if (s->number64 + s->number64_tmp < UINT32_MAX) {
			s->number64 += s->number64_tmp;
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st273;
st273:
	if ( ++p == pe )
		goto _test_eof273;
case 273:
#line 32314 "scanner.c"
	if ( (*p) == 10 )
		goto tr842;
	goto st273;
tr837:
#line 403 "./scanner_body.rl"
	{ if (s->number64 <= (UINT32_MAX / 86400)) {
	                  s->number64 *= 86400;
	              } else {
	                  WARN(ZSCANNER_ENUMBER32_OVERFLOW);
	                  p--; {goto st246;}
	              }
	            }
	goto st274;
tr838:
#line 396 "./scanner_body.rl"
	{ if (s->number64 <= (UINT32_MAX / 3600)) {
	                  s->number64 *= 3600;
	              } else {
	                  WARN(ZSCANNER_ENUMBER32_OVERFLOW);
	                  p--; {goto st246;}
	              }
	            }
	goto st274;
tr839:
#line 389 "./scanner_body.rl"
	{ if (s->number64 <= (UINT32_MAX / 60)) {
	                  s->number64 *= 60;
	              } else {
	                  WARN(ZSCANNER_ENUMBER32_OVERFLOW);
	                  p--; {goto st246;}
	              }
	            }
	goto st274;
tr841:
#line 410 "./scanner_body.rl"
	{ if (s->number64 <= (UINT32_MAX / 604800)) {
	                  s->number64 *= 604800;
	              } else {
	                  WARN(ZSCANNER_ENUMBER32_OVERFLOW);
	                  p--; {goto st246;}
	              }
	            }
	goto st274;
st274:
	if ( ++p == pe )
		goto _test_eof274;
case 274:
#line 32362 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st272;
		case 32: goto st272;
		case 40: goto tr834;
		case 41: goto tr835;
		case 778: goto tr842;
		case 827: goto st273;
		case 1034: goto tr842;
		case 1083: goto st273;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr845;
	goto tr832;
tr847:
#line 261 "./scanner_body.rl"
	{
		// Overflow check: 10*(s->number64) + fc - ASCII_0 <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + ASCII_0)
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ENUMBER64_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st275;
tr845:
#line 420 "./scanner_body.rl"
	{
		s->number64_tmp = s->number64;
	}
#line 278 "./scanner_body.rl"
	{
		s->number64 = 0;
	}
#line 261 "./scanner_body.rl"
	{
		// Overflow check: 10*(s->number64) + fc - ASCII_0 <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + ASCII_0)
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ENUMBER64_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st275;
tr856:
#line 423 "./scanner_body.rl"
	{
		if (s->number64 + s->number64_tmp < UINT32_MAX) {
			s->number64 += s->number64_tmp;
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 420 "./scanner_body.rl"
	{
		s->number64_tmp = s->number64;
	}
#line 278 "./scanner_body.rl"
	{
		s->number64 = 0;
	}
#line 261 "./scanner_body.rl"
	{
		// Overflow check: 10*(s->number64) + fc - ASCII_0 <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + ASCII_0)
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ENUMBER64_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st275;
st275:
	if ( ++p == pe )
		goto _test_eof275;
case 275:
#line 32470 "scanner.c"
	switch( (*p) ) {
		case 68: goto tr848;
		case 72: goto tr849;
		case 77: goto tr850;
		case 83: goto st276;
		case 87: goto tr852;
		case 100: goto tr848;
		case 104: goto tr849;
		case 109: goto tr850;
		case 115: goto st276;
		case 119: goto tr852;
	}
	if ( 48 <= (*p) && (*p) <= 57 )
		goto tr847;
	goto tr846;
tr848:
#line 403 "./scanner_body.rl"
	{ if (s->number64 <= (UINT32_MAX / 86400)) {
	                  s->number64 *= 86400;
	              } else {
	                  WARN(ZSCANNER_ENUMBER32_OVERFLOW);
	                  p--; {goto st246;}
	              }
	            }
	goto st276;
tr849:
#line 396 "./scanner_body.rl"
	{ if (s->number64 <= (UINT32_MAX / 3600)) {
	                  s->number64 *= 3600;
	              } else {
	                  WARN(ZSCANNER_ENUMBER32_OVERFLOW);
	                  p--; {goto st246;}
	              }
	            }
	goto st276;
tr850:
#line 389 "./scanner_body.rl"
	{ if (s->number64 <= (UINT32_MAX / 60)) {
	                  s->number64 *= 60;
	              } else {
	                  WARN(ZSCANNER_ENUMBER32_OVERFLOW);
	                  p--; {goto st246;}
	              }
	            }
	goto st276;
tr852:
#line 410 "./scanner_body.rl"
	{ if (s->number64 <= (UINT32_MAX / 604800)) {
	                  s->number64 *= 604800;
	              } else {
	                  WARN(ZSCANNER_ENUMBER32_OVERFLOW);
	                  p--; {goto st246;}
	              }
	            }
	goto st276;
st276:
	if ( ++p == pe )
		goto _test_eof276;
case 276:
#line 32530 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr853;
		case 32: goto tr853;
		case 40: goto tr854;
		case 41: goto tr855;
		case 778: goto tr857;
		case 827: goto tr858;
		case 1034: goto tr857;
		case 1083: goto tr858;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr856;
	goto tr832;
st277:
	if ( ++p == pe )
		goto _test_eof277;
case 277:
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	if ( _widec == 1034 )
		goto tr828;
	if ( 896 <= _widec && _widec <= 1151 )
		goto st277;
	goto tr824;
st278:
	if ( ++p == pe )
		goto _test_eof278;
case 278:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st279;
		case 32: goto st279;
		case 40: goto tr861;
		case 41: goto tr862;
		case 1034: goto tr863;
		case 1083: goto st289;
	}
	goto tr859;
tr861:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st279;
tr862:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st279;
tr863:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st279;
st279:
	if ( ++p == pe )
		goto _test_eof279;
case 279:
#line 32645 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st279;
		case 32: goto st279;
		case 40: goto tr861;
		case 41: goto tr862;
		case 42: goto tr865;
		case 46: goto tr866;
		case 92: goto tr867;
		case 95: goto tr865;
		case 1034: goto tr863;
		case 1083: goto st289;
	}
	if ( _widec < 65 ) {
		if ( 45 <= _widec && _widec <= 57 )
			goto tr865;
	} else if ( _widec > 90 ) {
		if ( 97 <= _widec && _widec <= 122 )
			goto tr865;
	} else
		goto tr865;
	goto tr859;
tr868:
#line 112 "./scanner_body.rl"
	{
		if (s->item_length < MAX_LABEL_LENGTH) {
			(s->dname)[s->dname_tmp_length++] = (*p);
			s->item_length++;
		} else {
			WARN(ZSCANNER_ELABEL_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st280;
tr875:
#line 108 "./scanner_body.rl"
	{
		s->item_length = 0;
		s->item_length_position = s->dname_tmp_length++;
	}
#line 112 "./scanner_body.rl"
	{
		if (s->item_length < MAX_LABEL_LENGTH) {
			(s->dname)[s->dname_tmp_length++] = (*p);
			s->item_length++;
		} else {
			WARN(ZSCANNER_ELABEL_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st280;
tr888:
#line 144 "./scanner_body.rl"
	{
		s->dname_tmp_length++;
	}
#line 112 "./scanner_body.rl"
	{
		if (s->item_length < MAX_LABEL_LENGTH) {
			(s->dname)[s->dname_tmp_length++] = (*p);
			s->item_length++;
		} else {
			WARN(ZSCANNER_ELABEL_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st280;
tr865:
#line 590 "./scanner_body.rl"
	{
		s->dname = s->zone_origin;
	}
#line 189 "./scanner_body.rl"
	{
		s->item_length_position = 0;
		s->dname_tmp_length = 0;
	}
#line 108 "./scanner_body.rl"
	{
		s->item_length = 0;
		s->item_length_position = s->dname_tmp_length++;
	}
#line 112 "./scanner_body.rl"
	{
		if (s->item_length < MAX_LABEL_LENGTH) {
			(s->dname)[s->dname_tmp_length++] = (*p);
			s->item_length++;
		} else {
			WARN(ZSCANNER_ELABEL_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st280;
st280:
	if ( ++p == pe )
		goto _test_eof280;
case 280:
#line 32756 "scanner.c"
	switch( (*p) ) {
		case 42: goto tr868;
		case 46: goto tr869;
		case 92: goto st284;
		case 95: goto tr868;
	}
	if ( (*p) < 65 ) {
		if ( 45 <= (*p) && (*p) <= 57 )
			goto tr868;
	} else if ( (*p) > 90 ) {
		if ( 97 <= (*p) && (*p) <= 122 )
			goto tr868;
	} else
		goto tr868;
	goto tr859;
tr869:
#line 121 "./scanner_body.rl"
	{
		if (s->dname_tmp_length < MAX_DNAME_LENGTH) {
			(s->dname)[s->item_length_position] =
				(uint8_t)(s->item_length);
		} else {
			WARN(ZSCANNER_EDNAME_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st281;
tr889:
#line 144 "./scanner_body.rl"
	{
		s->dname_tmp_length++;
	}
#line 121 "./scanner_body.rl"
	{
		if (s->dname_tmp_length < MAX_DNAME_LENGTH) {
			(s->dname)[s->item_length_position] =
				(uint8_t)(s->item_length);
		} else {
			WARN(ZSCANNER_EDNAME_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st281;
st281:
	if ( ++p == pe )
		goto _test_eof281;
case 281:
#line 32804 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr872;
		case 32: goto tr872;
		case 40: goto tr873;
		case 41: goto tr874;
		case 42: goto tr875;
		case 45: goto tr875;
		case 92: goto tr876;
		case 95: goto tr875;
		case 778: goto tr877;
		case 827: goto tr878;
		case 1034: goto tr877;
		case 1083: goto tr878;
	}
	if ( _widec < 65 ) {
		if ( 47 <= _widec && _widec <= 57 )
			goto tr875;
	} else if ( _widec > 90 ) {
		if ( 97 <= _widec && _widec <= 122 )
			goto tr875;
	} else
		goto tr875;
	goto tr871;
tr880:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st282;
tr881:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st282;
tr872:
#line 166 "./scanner_body.rl"
	{
		(s->dname)[s->dname_tmp_length++] = 0;
	}
	goto st282;
tr873:
#line 166 "./scanner_body.rl"
	{
		(s->dname)[s->dname_tmp_length++] = 0;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st282;
tr874:
#line 166 "./scanner_body.rl"
	{
		(s->dname)[s->dname_tmp_length++] = 0;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st282;
st282:
	if ( ++p == pe )
		goto _test_eof282;
case 282:
#line 32900 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st282;
		case 32: goto st282;
		case 40: goto tr880;
		case 41: goto tr881;
		case 778: goto tr882;
		case 827: goto st283;
		case 1034: goto tr882;
		case 1083: goto st283;
	}
	goto tr871;
tr877:
#line 166 "./scanner_body.rl"
	{
		(s->dname)[s->dname_tmp_length++] = 0;
	}
#line 593 "./scanner_body.rl"
	{
		s->zone_origin_length = s->dname_tmp_length;
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st1071;
tr882:
#line 593 "./scanner_body.rl"
	{
		s->zone_origin_length = s->dname_tmp_length;
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st1071;
st1071:
	if ( ++p == pe )
		goto _test_eof1071;
case 1071:
#line 32962 "scanner.c"
	goto st0;
tr878:
#line 166 "./scanner_body.rl"
	{
		(s->dname)[s->dname_tmp_length++] = 0;
	}
	goto st283;
st283:
	if ( ++p == pe )
		goto _test_eof283;
case 283:
#line 32974 "scanner.c"
	if ( (*p) == 10 )
		goto tr882;
	goto st283;
tr876:
#line 108 "./scanner_body.rl"
	{
		s->item_length = 0;
		s->item_length_position = s->dname_tmp_length++;
	}
	goto st284;
tr890:
#line 144 "./scanner_body.rl"
	{
		s->dname_tmp_length++;
	}
	goto st284;
tr867:
#line 590 "./scanner_body.rl"
	{
		s->dname = s->zone_origin;
	}
#line 189 "./scanner_body.rl"
	{
		s->item_length_position = 0;
		s->dname_tmp_length = 0;
	}
#line 108 "./scanner_body.rl"
	{
		s->item_length = 0;
		s->item_length_position = s->dname_tmp_length++;
	}
	goto st284;
st284:
	if ( ++p == pe )
		goto _test_eof284;
case 284:
#line 33011 "scanner.c"
	if ( 48 <= (*p) && (*p) <= 57 )
		goto tr884;
	goto tr868;
tr884:
#line 131 "./scanner_body.rl"
	{
		if (s->item_length < MAX_LABEL_LENGTH) {
			(s->dname)[s->dname_tmp_length] = 0;
			s->item_length++;
		} else {
			WARN(ZSCANNER_ELABEL_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 140 "./scanner_body.rl"
	{
		(s->dname)[s->dname_tmp_length] *= 10;
		(s->dname)[s->dname_tmp_length] += digit_to_num[(uint8_t)(*p)];
	}
	goto st285;
st285:
	if ( ++p == pe )
		goto _test_eof285;
case 285:
#line 33036 "scanner.c"
	if ( 48 <= (*p) && (*p) <= 57 )
		goto tr886;
	goto tr885;
tr886:
#line 140 "./scanner_body.rl"
	{
		(s->dname)[s->dname_tmp_length] *= 10;
		(s->dname)[s->dname_tmp_length] += digit_to_num[(uint8_t)(*p)];
	}
	goto st286;
st286:
	if ( ++p == pe )
		goto _test_eof286;
case 286:
#line 33051 "scanner.c"
	if ( 48 <= (*p) && (*p) <= 57 )
		goto tr887;
	goto tr885;
tr887:
#line 140 "./scanner_body.rl"
	{
		(s->dname)[s->dname_tmp_length] *= 10;
		(s->dname)[s->dname_tmp_length] += digit_to_num[(uint8_t)(*p)];
	}
	goto st287;
st287:
	if ( ++p == pe )
		goto _test_eof287;
case 287:
#line 33066 "scanner.c"
	switch( (*p) ) {
		case 42: goto tr888;
		case 46: goto tr889;
		case 92: goto tr890;
		case 95: goto tr888;
	}
	if ( (*p) < 65 ) {
		if ( 45 <= (*p) && (*p) <= 57 )
			goto tr888;
	} else if ( (*p) > 90 ) {
		if ( 97 <= (*p) && (*p) <= 122 )
			goto tr888;
	} else
		goto tr888;
	goto tr885;
tr866:
#line 590 "./scanner_body.rl"
	{
		s->dname = s->zone_origin;
	}
#line 189 "./scanner_body.rl"
	{
		s->item_length_position = 0;
		s->dname_tmp_length = 0;
	}
	goto st288;
st288:
	if ( ++p == pe )
		goto _test_eof288;
case 288:
#line 33097 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr872;
		case 32: goto tr872;
		case 40: goto tr873;
		case 41: goto tr874;
		case 778: goto tr877;
		case 827: goto tr878;
		case 1034: goto tr877;
		case 1083: goto tr878;
	}
	goto tr871;
st289:
	if ( ++p == pe )
		goto _test_eof289;
case 289:
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	if ( _widec == 1034 )
		goto tr863;
	if ( 896 <= _widec && _widec <= 1151 )
		goto st289;
	goto tr859;
st290:
	if ( ++p == pe )
		goto _test_eof290;
case 290:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st291;
		case 32: goto st291;
		case 40: goto tr892;
		case 41: goto tr893;
		case 1034: goto tr894;
		case 1083: goto st306;
	}
	goto st0;
tr892:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st291;
tr893:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st291;
tr894:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st291;
st291:
	if ( ++p == pe )
		goto _test_eof291;
case 291:
#line 33210 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st291;
		case 32: goto st291;
		case 40: goto tr892;
		case 41: goto tr893;
		case 1034: goto tr894;
		case 1083: goto st306;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr896;
	} else if ( _widec > 58 ) {
		if ( 60 <= _widec )
			goto tr896;
	} else
		goto tr896;
	goto tr897;
tr896:
#line 607 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
#line 561 "./scanner_body.rl"
	{ p--; {stack[top++] = 292; goto st257;} }
	goto st292;
st292:
	if ( ++p == pe )
		goto _test_eof292;
case 292:
#line 33254 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr899;
		case 32: goto tr899;
		case 40: goto tr900;
		case 41: goto tr901;
		case 778: goto tr902;
		case 827: goto tr903;
		case 1034: goto tr902;
		case 1083: goto tr904;
	}
	goto tr898;
tr907:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st293;
tr908:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st293;
tr899:
#line 610 "./scanner_body.rl"
	{
		*rdata_tail = 0; // Ending filename string.
		strncpy((char*)(s->include_filename), (char*)(s->r_data),
		        sizeof(s->include_filename));

		// Check for correct string copy.
		if (strlen(s->include_filename) !=
		    (size_t)(rdata_tail - s->r_data)) {
			ERR(ZSCANNER_EBAD_INCLUDE_FILENAME);
			p--; {goto st246;}
		}

		// For detection whether origin is not present.
		s->dname = NULL;
	}
	goto st293;
tr900:
#line 610 "./scanner_body.rl"
	{
		*rdata_tail = 0; // Ending filename string.
		strncpy((char*)(s->include_filename), (char*)(s->r_data),
		        sizeof(s->include_filename));

		// Check for correct string copy.
		if (strlen(s->include_filename) !=
		    (size_t)(rdata_tail - s->r_data)) {
			ERR(ZSCANNER_EBAD_INCLUDE_FILENAME);
			p--; {goto st246;}
		}

		// For detection whether origin is not present.
		s->dname = NULL;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st293;
tr901:
#line 610 "./scanner_body.rl"
	{
		*rdata_tail = 0; // Ending filename string.
		strncpy((char*)(s->include_filename), (char*)(s->r_data),
		        sizeof(s->include_filename));

		// Check for correct string copy.
		if (strlen(s->include_filename) !=
		    (size_t)(rdata_tail - s->r_data)) {
			ERR(ZSCANNER_EBAD_INCLUDE_FILENAME);
			p--; {goto st246;}
		}

		// For detection whether origin is not present.
		s->dname = NULL;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st293;
st293:
	if ( ++p == pe )
		goto _test_eof293;
case 293:
#line 33374 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st293;
		case 32: goto st293;
		case 40: goto tr907;
		case 41: goto tr908;
		case 42: goto tr909;
		case 46: goto tr910;
		case 92: goto tr911;
		case 95: goto tr909;
		case 778: goto tr912;
		case 827: goto st297;
		case 1034: goto tr912;
		case 1083: goto st303;
	}
	if ( _widec < 65 ) {
		if ( 45 <= _widec && _widec <= 57 )
			goto tr909;
	} else if ( _widec > 90 ) {
		if ( 97 <= _widec && _widec <= 122 )
			goto tr909;
	} else
		goto tr909;
	goto tr905;
tr916:
#line 112 "./scanner_body.rl"
	{
		if (s->item_length < MAX_LABEL_LENGTH) {
			(s->dname)[s->dname_tmp_length++] = (*p);
			s->item_length++;
		} else {
			WARN(ZSCANNER_ELABEL_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st294;
tr922:
#line 108 "./scanner_body.rl"
	{
		s->item_length = 0;
		s->item_length_position = s->dname_tmp_length++;
	}
#line 112 "./scanner_body.rl"
	{
		if (s->item_length < MAX_LABEL_LENGTH) {
			(s->dname)[s->dname_tmp_length++] = (*p);
			s->item_length++;
		} else {
			WARN(ZSCANNER_ELABEL_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st294;
tr933:
#line 144 "./scanner_body.rl"
	{
		s->dname_tmp_length++;
	}
#line 112 "./scanner_body.rl"
	{
		if (s->item_length < MAX_LABEL_LENGTH) {
			(s->dname)[s->dname_tmp_length++] = (*p);
			s->item_length++;
		} else {
			WARN(ZSCANNER_ELABEL_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st294;
tr909:
#line 630 "./scanner_body.rl"
	{
		s->dname = s->r_data;
	}
#line 189 "./scanner_body.rl"
	{
		s->item_length_position = 0;
		s->dname_tmp_length = 0;
	}
#line 108 "./scanner_body.rl"
	{
		s->item_length = 0;
		s->item_length_position = s->dname_tmp_length++;
	}
#line 112 "./scanner_body.rl"
	{
		if (s->item_length < MAX_LABEL_LENGTH) {
			(s->dname)[s->dname_tmp_length++] = (*p);
			s->item_length++;
		} else {
			WARN(ZSCANNER_ELABEL_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st294;
st294:
	if ( ++p == pe )
		goto _test_eof294;
case 294:
#line 33487 "scanner.c"
	switch( (*p) ) {
		case 42: goto tr916;
		case 46: goto tr917;
		case 92: goto st298;
		case 95: goto tr916;
	}
	if ( (*p) < 65 ) {
		if ( 45 <= (*p) && (*p) <= 57 )
			goto tr916;
	} else if ( (*p) > 90 ) {
		if ( 97 <= (*p) && (*p) <= 122 )
			goto tr916;
	} else
		goto tr916;
	goto tr915;
tr917:
#line 121 "./scanner_body.rl"
	{
		if (s->dname_tmp_length < MAX_DNAME_LENGTH) {
			(s->dname)[s->item_length_position] =
				(uint8_t)(s->item_length);
		} else {
			WARN(ZSCANNER_EDNAME_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st295;
tr934:
#line 144 "./scanner_body.rl"
	{
		s->dname_tmp_length++;
	}
#line 121 "./scanner_body.rl"
	{
		if (s->dname_tmp_length < MAX_DNAME_LENGTH) {
			(s->dname)[s->item_length_position] =
				(uint8_t)(s->item_length);
		} else {
			WARN(ZSCANNER_EDNAME_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st295;
st295:
	if ( ++p == pe )
		goto _test_eof295;
case 295:
#line 33535 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr919;
		case 32: goto tr919;
		case 40: goto tr920;
		case 41: goto tr921;
		case 42: goto tr922;
		case 45: goto tr922;
		case 92: goto tr923;
		case 95: goto tr922;
		case 778: goto tr924;
		case 827: goto tr925;
		case 1034: goto tr924;
		case 1083: goto tr925;
	}
	if ( _widec < 65 ) {
		if ( 47 <= _widec && _widec <= 57 )
			goto tr922;
	} else if ( _widec > 90 ) {
		if ( 97 <= _widec && _widec <= 122 )
			goto tr922;
	} else
		goto tr922;
	goto tr905;
tr927:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st296;
tr928:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st296;
tr919:
#line 166 "./scanner_body.rl"
	{
		(s->dname)[s->dname_tmp_length++] = 0;
	}
#line 633 "./scanner_body.rl"
	{
		s->r_data_length = s->dname_tmp_length;
	}
	goto st296;
tr920:
#line 166 "./scanner_body.rl"
	{
		(s->dname)[s->dname_tmp_length++] = 0;
	}
#line 633 "./scanner_body.rl"
	{
		s->r_data_length = s->dname_tmp_length;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st296;
tr921:
#line 166 "./scanner_body.rl"
	{
		(s->dname)[s->dname_tmp_length++] = 0;
	}
#line 633 "./scanner_body.rl"
	{
		s->r_data_length = s->dname_tmp_length;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st296;
st296:
	if ( ++p == pe )
		goto _test_eof296;
case 296:
#line 33643 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st296;
		case 32: goto st296;
		case 40: goto tr927;
		case 41: goto tr928;
		case 778: goto tr912;
		case 827: goto st297;
		case 1034: goto tr912;
		case 1083: goto st297;
	}
	goto tr83;
tr902:
#line 610 "./scanner_body.rl"
	{
		*rdata_tail = 0; // Ending filename string.
		strncpy((char*)(s->include_filename), (char*)(s->r_data),
		        sizeof(s->include_filename));

		// Check for correct string copy.
		if (strlen(s->include_filename) !=
		    (size_t)(rdata_tail - s->r_data)) {
			ERR(ZSCANNER_EBAD_INCLUDE_FILENAME);
			p--; {goto st246;}
		}

		// For detection whether origin is not present.
		s->dname = NULL;
	}
#line 641 "./scanner_body.rl"
	{
		char text_origin[4 * MAX_DNAME_LENGTH]; // Each char as \DDD.

		// Origin conversion from wire to text form.
		if (s->dname == NULL) { // Use current origin.
			wire_dname_to_str(s->zone_origin,
			                  s->zone_origin_length,
			                  text_origin);
		} else { // Use specified origin.
			wire_dname_to_str(s->r_data,
			                  s->r_data_length,
			                  text_origin);
		}

		// Relative file path.
		if (s->include_filename[0] != '/') {
			snprintf((char*)(s->buffer), sizeof(s->buffer),
			         "%s/%s", s->path, s->include_filename);
		} else {
			strncpy((char*)(s->buffer), (char*)(s->include_filename),
			        sizeof(s->buffer));
		}

		// Create new file loader for included zone file.
		file_loader_t *fl = file_loader_create((char*)(s->buffer),
		                                       text_origin,
		                                       s->default_class,
		                                       s->default_ttl,
		                                       s->process_record,
		                                       s->process_error,
		                                       s->data);
		if (fl != NULL) {
			// Process included zone file.
			ret = file_loader_process(fl);
			file_loader_free(fl);

			if (ret != 0) {
				ERR(ZSCANNER_EUNPROCESSED_INCLUDE);
				p--; {goto st246;}
			}
		} else {
			ERR(ZSCANNER_EUNOPENED_INCLUDE);
			p--; {goto st246;}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st1072;
tr912:
#line 641 "./scanner_body.rl"
	{
		char text_origin[4 * MAX_DNAME_LENGTH]; // Each char as \DDD.

		// Origin conversion from wire to text form.
		if (s->dname == NULL) { // Use current origin.
			wire_dname_to_str(s->zone_origin,
			                  s->zone_origin_length,
			                  text_origin);
		} else { // Use specified origin.
			wire_dname_to_str(s->r_data,
			                  s->r_data_length,
			                  text_origin);
		}

		// Relative file path.
		if (s->include_filename[0] != '/') {
			snprintf((char*)(s->buffer), sizeof(s->buffer),
			         "%s/%s", s->path, s->include_filename);
		} else {
			strncpy((char*)(s->buffer), (char*)(s->include_filename),
			        sizeof(s->buffer));
		}

		// Create new file loader for included zone file.
		file_loader_t *fl = file_loader_create((char*)(s->buffer),
		                                       text_origin,
		                                       s->default_class,
		                                       s->default_ttl,
		                                       s->process_record,
		                                       s->process_error,
		                                       s->data);
		if (fl != NULL) {
			// Process included zone file.
			ret = file_loader_process(fl);
			file_loader_free(fl);

			if (ret != 0) {
				ERR(ZSCANNER_EUNPROCESSED_INCLUDE);
				p--; {goto st246;}
			}
		} else {
			ERR(ZSCANNER_EUNOPENED_INCLUDE);
			p--; {goto st246;}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st1072;
tr924:
#line 166 "./scanner_body.rl"
	{
		(s->dname)[s->dname_tmp_length++] = 0;
	}
#line 633 "./scanner_body.rl"
	{
		s->r_data_length = s->dname_tmp_length;
	}
#line 641 "./scanner_body.rl"
	{
		char text_origin[4 * MAX_DNAME_LENGTH]; // Each char as \DDD.

		// Origin conversion from wire to text form.
		if (s->dname == NULL) { // Use current origin.
			wire_dname_to_str(s->zone_origin,
			                  s->zone_origin_length,
			                  text_origin);
		} else { // Use specified origin.
			wire_dname_to_str(s->r_data,
			                  s->r_data_length,
			                  text_origin);
		}

		// Relative file path.
		if (s->include_filename[0] != '/') {
			snprintf((char*)(s->buffer), sizeof(s->buffer),
			         "%s/%s", s->path, s->include_filename);
		} else {
			strncpy((char*)(s->buffer), (char*)(s->include_filename),
			        sizeof(s->buffer));
		}

		// Create new file loader for included zone file.
		file_loader_t *fl = file_loader_create((char*)(s->buffer),
		                                       text_origin,
		                                       s->default_class,
		                                       s->default_ttl,
		                                       s->process_record,
		                                       s->process_error,
		                                       s->data);
		if (fl != NULL) {
			// Process included zone file.
			ret = file_loader_process(fl);
			file_loader_free(fl);

			if (ret != 0) {
				ERR(ZSCANNER_EUNPROCESSED_INCLUDE);
				p--; {goto st246;}
			}
		} else {
			ERR(ZSCANNER_EUNOPENED_INCLUDE);
			p--; {goto st246;}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st1072;
st1072:
	if ( ++p == pe )
		goto _test_eof1072;
case 1072:
#line 33865 "scanner.c"
	goto st0;
tr903:
#line 610 "./scanner_body.rl"
	{
		*rdata_tail = 0; // Ending filename string.
		strncpy((char*)(s->include_filename), (char*)(s->r_data),
		        sizeof(s->include_filename));

		// Check for correct string copy.
		if (strlen(s->include_filename) !=
		    (size_t)(rdata_tail - s->r_data)) {
			ERR(ZSCANNER_EBAD_INCLUDE_FILENAME);
			p--; {goto st246;}
		}

		// For detection whether origin is not present.
		s->dname = NULL;
	}
	goto st297;
tr925:
#line 166 "./scanner_body.rl"
	{
		(s->dname)[s->dname_tmp_length++] = 0;
	}
#line 633 "./scanner_body.rl"
	{
		s->r_data_length = s->dname_tmp_length;
	}
	goto st297;
st297:
	if ( ++p == pe )
		goto _test_eof297;
case 297:
#line 33899 "scanner.c"
	if ( (*p) == 10 )
		goto tr912;
	goto st297;
tr923:
#line 108 "./scanner_body.rl"
	{
		s->item_length = 0;
		s->item_length_position = s->dname_tmp_length++;
	}
	goto st298;
tr935:
#line 144 "./scanner_body.rl"
	{
		s->dname_tmp_length++;
	}
	goto st298;
tr911:
#line 630 "./scanner_body.rl"
	{
		s->dname = s->r_data;
	}
#line 189 "./scanner_body.rl"
	{
		s->item_length_position = 0;
		s->dname_tmp_length = 0;
	}
#line 108 "./scanner_body.rl"
	{
		s->item_length = 0;
		s->item_length_position = s->dname_tmp_length++;
	}
	goto st298;
st298:
	if ( ++p == pe )
		goto _test_eof298;
case 298:
#line 33936 "scanner.c"
	if ( 48 <= (*p) && (*p) <= 57 )
		goto tr929;
	goto tr916;
tr929:
#line 131 "./scanner_body.rl"
	{
		if (s->item_length < MAX_LABEL_LENGTH) {
			(s->dname)[s->dname_tmp_length] = 0;
			s->item_length++;
		} else {
			WARN(ZSCANNER_ELABEL_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 140 "./scanner_body.rl"
	{
		(s->dname)[s->dname_tmp_length] *= 10;
		(s->dname)[s->dname_tmp_length] += digit_to_num[(uint8_t)(*p)];
	}
	goto st299;
st299:
	if ( ++p == pe )
		goto _test_eof299;
case 299:
#line 33961 "scanner.c"
	if ( 48 <= (*p) && (*p) <= 57 )
		goto tr931;
	goto tr930;
tr931:
#line 140 "./scanner_body.rl"
	{
		(s->dname)[s->dname_tmp_length] *= 10;
		(s->dname)[s->dname_tmp_length] += digit_to_num[(uint8_t)(*p)];
	}
	goto st300;
st300:
	if ( ++p == pe )
		goto _test_eof300;
case 300:
#line 33976 "scanner.c"
	if ( 48 <= (*p) && (*p) <= 57 )
		goto tr932;
	goto tr930;
tr932:
#line 140 "./scanner_body.rl"
	{
		(s->dname)[s->dname_tmp_length] *= 10;
		(s->dname)[s->dname_tmp_length] += digit_to_num[(uint8_t)(*p)];
	}
	goto st301;
st301:
	if ( ++p == pe )
		goto _test_eof301;
case 301:
#line 33991 "scanner.c"
	switch( (*p) ) {
		case 42: goto tr933;
		case 46: goto tr934;
		case 92: goto tr935;
		case 95: goto tr933;
	}
	if ( (*p) < 65 ) {
		if ( 45 <= (*p) && (*p) <= 57 )
			goto tr933;
	} else if ( (*p) > 90 ) {
		if ( 97 <= (*p) && (*p) <= 122 )
			goto tr933;
	} else
		goto tr933;
	goto tr930;
tr910:
#line 630 "./scanner_body.rl"
	{
		s->dname = s->r_data;
	}
#line 189 "./scanner_body.rl"
	{
		s->item_length_position = 0;
		s->dname_tmp_length = 0;
	}
	goto st302;
st302:
	if ( ++p == pe )
		goto _test_eof302;
case 302:
#line 34022 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr919;
		case 32: goto tr919;
		case 40: goto tr920;
		case 41: goto tr921;
		case 778: goto tr924;
		case 827: goto tr925;
		case 1034: goto tr924;
		case 1083: goto tr925;
	}
	goto tr905;
tr904:
#line 610 "./scanner_body.rl"
	{
		*rdata_tail = 0; // Ending filename string.
		strncpy((char*)(s->include_filename), (char*)(s->r_data),
		        sizeof(s->include_filename));

		// Check for correct string copy.
		if (strlen(s->include_filename) !=
		    (size_t)(rdata_tail - s->r_data)) {
			ERR(ZSCANNER_EBAD_INCLUDE_FILENAME);
			p--; {goto st246;}
		}

		// For detection whether origin is not present.
		s->dname = NULL;
	}
	goto st303;
st303:
	if ( ++p == pe )
		goto _test_eof303;
case 303:
#line 34070 "scanner.c"
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 778: goto tr912;
		case 1034: goto tr936;
	}
	if ( _widec > 895 ) {
		if ( 896 <= _widec && _widec <= 1151 )
			goto st303;
	} else if ( _widec >= 640 )
		goto st297;
	goto tr83;
tr936:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 641 "./scanner_body.rl"
	{
		char text_origin[4 * MAX_DNAME_LENGTH]; // Each char as \DDD.

		// Origin conversion from wire to text form.
		if (s->dname == NULL) { // Use current origin.
			wire_dname_to_str(s->zone_origin,
			                  s->zone_origin_length,
			                  text_origin);
		} else { // Use specified origin.
			wire_dname_to_str(s->r_data,
			                  s->r_data_length,
			                  text_origin);
		}

		// Relative file path.
		if (s->include_filename[0] != '/') {
			snprintf((char*)(s->buffer), sizeof(s->buffer),
			         "%s/%s", s->path, s->include_filename);
		} else {
			strncpy((char*)(s->buffer), (char*)(s->include_filename),
			        sizeof(s->buffer));
		}

		// Create new file loader for included zone file.
		file_loader_t *fl = file_loader_create((char*)(s->buffer),
		                                       text_origin,
		                                       s->default_class,
		                                       s->default_ttl,
		                                       s->process_record,
		                                       s->process_error,
		                                       s->data);
		if (fl != NULL) {
			// Process included zone file.
			ret = file_loader_process(fl);
			file_loader_free(fl);

			if (ret != 0) {
				ERR(ZSCANNER_EUNPROCESSED_INCLUDE);
				p--; {goto st246;}
			}
		} else {
			ERR(ZSCANNER_EUNOPENED_INCLUDE);
			p--; {goto st246;}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1073;
st1073:
	if ( ++p == pe )
		goto _test_eof1073;
case 1073:
#line 34162 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st304;
		case 32: goto st304;
		case 40: goto tr938;
		case 41: goto tr939;
		case 42: goto tr909;
		case 46: goto tr910;
		case 92: goto tr911;
		case 95: goto tr909;
		case 1034: goto tr940;
		case 1083: goto st305;
	}
	if ( _widec < 65 ) {
		if ( 45 <= _widec && _widec <= 57 )
			goto tr909;
	} else if ( _widec > 90 ) {
		if ( 97 <= _widec && _widec <= 122 )
			goto tr909;
	} else
		goto tr909;
	goto tr915;
tr938:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st304;
tr939:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st304;
tr940:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st304;
st304:
	if ( ++p == pe )
		goto _test_eof304;
case 304:
#line 34228 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st304;
		case 32: goto st304;
		case 40: goto tr938;
		case 41: goto tr939;
		case 42: goto tr909;
		case 46: goto tr910;
		case 92: goto tr911;
		case 95: goto tr909;
		case 1034: goto tr940;
		case 1083: goto st305;
	}
	if ( _widec < 65 ) {
		if ( 45 <= _widec && _widec <= 57 )
			goto tr909;
	} else if ( _widec > 90 ) {
		if ( 97 <= _widec && _widec <= 122 )
			goto tr909;
	} else
		goto tr909;
	goto tr915;
st305:
	if ( ++p == pe )
		goto _test_eof305;
case 305:
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	if ( _widec == 1034 )
		goto tr940;
	if ( 896 <= _widec && _widec <= 1151 )
		goto st305;
	goto st0;
st306:
	if ( ++p == pe )
		goto _test_eof306;
case 306:
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	if ( _widec == 1034 )
		goto tr894;
	if ( 896 <= _widec && _widec <= 1151 )
		goto st306;
	goto st0;
st307:
	if ( ++p == pe )
		goto _test_eof307;
case 307:
	if ( (*p) == 43 )
		goto tr943;
	if ( (*p) < 65 ) {
		if ( 47 <= (*p) && (*p) <= 57 )
			goto tr943;
	} else if ( (*p) > 90 ) {
		if ( 97 <= (*p) && (*p) <= 122 )
			goto tr943;
	} else
		goto tr943;
	goto tr942;
tr943:
#line 919 "./scanner_body.rl"
	{
		if (rdata_tail <= rdata_stop) {
			*rdata_tail = first_base64_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ERDATA_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st308;
st308:
	if ( ++p == pe )
		goto _test_eof308;
case 308:
#line 34354 "scanner.c"
	if ( (*p) == 43 )
		goto tr944;
	if ( (*p) < 65 ) {
		if ( 47 <= (*p) && (*p) <= 57 )
			goto tr944;
	} else if ( (*p) > 90 ) {
		if ( 97 <= (*p) && (*p) <= 122 )
			goto tr944;
	} else
		goto tr944;
	goto tr942;
tr944:
#line 927 "./scanner_body.rl"
	{
		*(rdata_tail++) += second_left_base64_to_num[(uint8_t)(*p)];

		if (rdata_tail <= rdata_stop) {
			*rdata_tail = second_right_base64_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ERDATA_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st309;
st309:
	if ( ++p == pe )
		goto _test_eof309;
case 309:
#line 34383 "scanner.c"
	switch( (*p) ) {
		case 43: goto tr945;
		case 61: goto st313;
	}
	if ( (*p) < 65 ) {
		if ( 47 <= (*p) && (*p) <= 57 )
			goto tr945;
	} else if ( (*p) > 90 ) {
		if ( 97 <= (*p) && (*p) <= 122 )
			goto tr945;
	} else
		goto tr945;
	goto tr942;
tr945:
#line 937 "./scanner_body.rl"
	{
		*(rdata_tail++) += third_left_base64_to_num[(uint8_t)(*p)];

		if (rdata_tail <= rdata_stop) {
			*rdata_tail = third_right_base64_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ERDATA_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st310;
st310:
	if ( ++p == pe )
		goto _test_eof310;
case 310:
#line 34414 "scanner.c"
	switch( (*p) ) {
		case 43: goto tr947;
		case 61: goto st311;
	}
	if ( (*p) < 65 ) {
		if ( 47 <= (*p) && (*p) <= 57 )
			goto tr947;
	} else if ( (*p) > 90 ) {
		if ( 97 <= (*p) && (*p) <= 122 )
			goto tr947;
	} else
		goto tr947;
	goto tr942;
tr949:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st311;
tr950:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st311;
tr951:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st311;
tr947:
#line 947 "./scanner_body.rl"
	{
		*(rdata_tail++) += fourth_base64_to_num[(uint8_t)(*p)];
	}
	goto st311;
st311:
	if ( ++p == pe )
		goto _test_eof311;
case 311:
#line 34464 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto st311;
		case 32: goto st311;
		case 40: goto tr949;
		case 41: goto tr950;
		case 43: goto tr943;
		case 2058: goto tr951;
		case 2107: goto st312;
		case 2314: goto tr953;
		case 2363: goto tr953;
		case 2570: goto tr954;
		case 2619: goto tr955;
	}
	if ( _widec < 65 ) {
		if ( 47 <= _widec && _widec <= 57 )
			goto tr943;
	} else if ( _widec > 90 ) {
		if ( 97 <= _widec && _widec <= 122 )
			goto tr943;
	} else
		goto tr943;
	goto tr942;
st312:
	if ( ++p == pe )
		goto _test_eof312;
case 312:
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	if ( _widec == 1034 )
		goto tr951;
	if ( 896 <= _widec && _widec <= 1151 )
		goto st312;
	goto tr942;
tr953:
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1074;
st1074:
	if ( ++p == pe )
		goto _test_eof1074;
case 1074:
#line 34547 "scanner.c"
	goto st0;
tr954:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1075;
st1075:
	if ( ++p == pe )
		goto _test_eof1075;
case 1075:
#line 34563 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto st311;
		case 32: goto st311;
		case 40: goto tr949;
		case 41: goto tr950;
		case 43: goto tr943;
		case 2058: goto tr951;
		case 2107: goto st312;
		case 2314: goto tr953;
		case 2363: goto tr953;
		case 2570: goto tr954;
		case 2619: goto tr955;
	}
	if ( _widec < 65 ) {
		if ( 47 <= _widec && _widec <= 57 )
			goto tr943;
	} else if ( _widec > 90 ) {
		if ( 97 <= _widec && _widec <= 122 )
			goto tr943;
	} else
		goto tr943;
	goto tr942;
tr955:
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1076;
st1076:
	if ( ++p == pe )
		goto _test_eof1076;
case 1076:
#line 34616 "scanner.c"
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	if ( _widec == 1034 )
		goto tr951;
	if ( 896 <= _widec && _widec <= 1151 )
		goto st312;
	goto tr942;
st313:
	if ( ++p == pe )
		goto _test_eof313;
case 313:
	if ( (*p) == 61 )
		goto st311;
	goto tr942;
st314:
	if ( ++p == pe )
		goto _test_eof314;
case 314:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr957;
		case 32: goto tr957;
		case 40: goto tr958;
		case 41: goto tr959;
		case 2058: goto tr960;
		case 2107: goto tr961;
		case 2314: goto tr962;
		case 2363: goto tr962;
		case 2570: goto tr963;
		case 2619: goto tr964;
	}
	goto tr956;
tr966:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr967:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr982:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
tr957:
#line 1284 "./scanner_body.rl"
	{
		memset(s->windows, 0, sizeof(s->windows));
		s->last_window = -1;
	}
	goto st315;
tr958:
#line 1284 "./scanner_body.rl"
	{
		memset(s->windows, 0, sizeof(s->windows));
		s->last_window = -1;
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr959:
#line 1284 "./scanner_body.rl"
	{
		memset(s->windows, 0, sizeof(s->windows));
		s->last_window = -1;
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr960:
#line 1284 "./scanner_body.rl"
	{
		memset(s->windows, 0, sizeof(s->windows));
		s->last_window = -1;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
tr987:
#line 1243 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_A, s); }
	goto st315;
tr988:
#line 1243 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_A, s); }
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr989:
#line 1243 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_A, s); }
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr993:
#line 1243 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_A, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
tr1000:
#line 1256 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_AAAA, s); }
	goto st315;
tr1001:
#line 1256 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_AAAA, s); }
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr1002:
#line 1256 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_AAAA, s); }
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr1003:
#line 1256 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_AAAA, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
tr1012:
#line 1261 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_CERT, s); }
	goto st315;
tr1013:
#line 1261 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_CERT, s); }
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr1014:
#line 1261 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_CERT, s); }
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr1015:
#line 1261 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_CERT, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
tr1023:
#line 1245 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_CNAME, s); }
	goto st315;
tr1024:
#line 1245 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_CNAME, s); }
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr1025:
#line 1245 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_CNAME, s); }
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr1026:
#line 1245 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_CNAME, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
tr1037:
#line 1270 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_DHCID, s); }
	goto st315;
tr1038:
#line 1270 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_DHCID, s); }
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr1039:
#line 1270 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_DHCID, s); }
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr1040:
#line 1270 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_DHCID, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
tr1049:
#line 1262 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_DNAME, s); }
	goto st315;
tr1050:
#line 1262 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_DNAME, s); }
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr1051:
#line 1262 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_DNAME, s); }
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr1052:
#line 1262 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_DNAME, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
tr1060:
#line 1269 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_DNSKEY, s); }
	goto st315;
tr1061:
#line 1269 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_DNSKEY, s); }
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr1062:
#line 1269 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_DNSKEY, s); }
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr1063:
#line 1269 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_DNSKEY, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
tr1068:
#line 1264 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_DS, s); }
	goto st315;
tr1069:
#line 1264 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_DS, s); }
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr1070:
#line 1264 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_DS, s); }
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr1071:
#line 1264 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_DS, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
tr1081:
#line 1279 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_EUI48, s); }
	goto st315;
tr1082:
#line 1279 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_EUI48, s); }
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr1083:
#line 1279 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_EUI48, s); }
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr1084:
#line 1279 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_EUI48, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
tr1090:
#line 1280 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_EUI64, s); }
	goto st315;
tr1091:
#line 1280 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_EUI64, s); }
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr1092:
#line 1280 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_EUI64, s); }
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr1093:
#line 1280 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_EUI64, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
tr1102:
#line 1248 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_HINFO, s); }
	goto st315;
tr1103:
#line 1248 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_HINFO, s); }
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr1104:
#line 1248 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_HINFO, s); }
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr1105:
#line 1248 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_HINFO, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
tr1117:
#line 1266 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_IPSECKEY, s); }
	goto st315;
tr1118:
#line 1266 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_IPSECKEY, s); }
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr1119:
#line 1266 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_IPSECKEY, s); }
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr1120:
#line 1266 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_IPSECKEY, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
tr1128:
#line 1255 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_KEY, s); }
	goto st315;
tr1129:
#line 1255 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_KEY, s); }
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr1130:
#line 1255 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_KEY, s); }
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr1131:
#line 1255 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_KEY, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
tr1136:
#line 1260 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_KX, s); }
	goto st315;
tr1137:
#line 1260 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_KX, s); }
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr1138:
#line 1260 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_KX, s); }
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr1139:
#line 1260 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_KX, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
tr1149:
#line 1276 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_L32, s); }
	goto st315;
tr1150:
#line 1276 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_L32, s); }
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr1151:
#line 1276 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_L32, s); }
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr1152:
#line 1276 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_L32, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
tr1158:
#line 1277 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_L64, s); }
	goto st315;
tr1159:
#line 1277 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_L64, s); }
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr1160:
#line 1277 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_L64, s); }
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr1161:
#line 1277 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_L64, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
tr1167:
#line 1257 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_LOC, s); }
	goto st315;
tr1168:
#line 1257 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_LOC, s); }
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr1169:
#line 1257 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_LOC, s); }
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr1170:
#line 1257 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_LOC, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
tr1175:
#line 1278 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_LP, s); }
	goto st315;
tr1176:
#line 1278 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_LP, s); }
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr1177:
#line 1278 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_LP, s); }
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr1178:
#line 1278 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_LP, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
tr1188:
#line 1249 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_MINFO, s); }
	goto st315;
tr1189:
#line 1249 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_MINFO, s); }
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr1190:
#line 1249 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_MINFO, s); }
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr1191:
#line 1249 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_MINFO, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
tr1196:
#line 1250 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_MX, s); }
	goto st315;
tr1197:
#line 1250 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_MX, s); }
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr1198:
#line 1250 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_MX, s); }
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr1199:
#line 1250 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_MX, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
tr1210:
#line 1259 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NAPTR, s); }
	goto st315;
tr1211:
#line 1259 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NAPTR, s); }
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr1212:
#line 1259 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NAPTR, s); }
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr1213:
#line 1259 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NAPTR, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
tr1219:
#line 1275 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NID, s); }
	goto st315;
tr1220:
#line 1275 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NID, s); }
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr1221:
#line 1275 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NID, s); }
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr1222:
#line 1275 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NID, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
tr1227:
#line 1244 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NS, s); }
	goto st315;
tr1228:
#line 1244 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NS, s); }
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr1229:
#line 1244 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NS, s); }
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr1231:
#line 1244 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NS, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
tr1237:
#line 1268 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NSEC, s); }
	goto st315;
tr1238:
#line 1268 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NSEC, s); }
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr1239:
#line 1268 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NSEC, s); }
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr1241:
#line 1268 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NSEC, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
tr1246:
#line 1271 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NSEC3, s); }
	goto st315;
tr1247:
#line 1271 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NSEC3, s); }
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr1248:
#line 1271 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NSEC3, s); }
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr1250:
#line 1271 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NSEC3, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
tr1259:
#line 1272 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NSEC3PARAM, s); }
	goto st315;
tr1260:
#line 1272 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NSEC3PARAM, s); }
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr1261:
#line 1272 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NSEC3PARAM, s); }
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr1262:
#line 1272 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NSEC3PARAM, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
tr1269:
#line 1247 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_PTR, s); }
	goto st315;
tr1270:
#line 1247 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_PTR, s); }
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr1271:
#line 1247 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_PTR, s); }
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr1272:
#line 1247 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_PTR, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
tr1280:
#line 1252 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_RP, s); }
	goto st315;
tr1281:
#line 1252 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_RP, s); }
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr1282:
#line 1252 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_RP, s); }
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr1283:
#line 1252 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_RP, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
tr1291:
#line 1267 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_RRSIG, s); }
	goto st315;
tr1292:
#line 1267 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_RRSIG, s); }
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr1293:
#line 1267 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_RRSIG, s); }
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr1294:
#line 1267 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_RRSIG, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
tr1299:
#line 1254 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_RT, s); }
	goto st315;
tr1300:
#line 1254 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_RT, s); }
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr1301:
#line 1254 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_RT, s); }
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr1302:
#line 1254 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_RT, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
tr1312:
#line 1246 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_SOA, s); }
	goto st315;
tr1313:
#line 1246 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_SOA, s); }
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr1314:
#line 1246 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_SOA, s); }
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr1315:
#line 1246 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_SOA, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
tr1321:
#line 1274 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_SPF, s); }
	goto st315;
tr1322:
#line 1274 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_SPF, s); }
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr1323:
#line 1274 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_SPF, s); }
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr1324:
#line 1274 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_SPF, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
tr1330:
#line 1258 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_SRV, s); }
	goto st315;
tr1331:
#line 1258 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_SRV, s); }
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr1332:
#line 1258 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_SRV, s); }
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr1333:
#line 1258 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_SRV, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
tr1341:
#line 1265 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_SSHFP, s); }
	goto st315;
tr1342:
#line 1265 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_SSHFP, s); }
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr1343:
#line 1265 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_SSHFP, s); }
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr1344:
#line 1265 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_SSHFP, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
tr1354:
#line 1273 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_TLSA, s); }
	goto st315;
tr1355:
#line 1273 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_TLSA, s); }
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr1356:
#line 1273 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_TLSA, s); }
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr1357:
#line 1273 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_TLSA, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
tr1363:
#line 1251 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_TXT, s); }
	goto st315;
tr1364:
#line 1251 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_TXT, s); }
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr1365:
#line 1251 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_TXT, s); }
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr1366:
#line 1251 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_TXT, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
tr1374:
#line 1230 "./scanner_body.rl"
	{
		if (s->number64 <= UINT16_MAX) {
			window_add_bit(s->number64, s);
		} else {
			WARN(ZSCANNER_ENUMBER16_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st315;
tr1375:
#line 1230 "./scanner_body.rl"
	{
		if (s->number64 <= UINT16_MAX) {
			window_add_bit(s->number64, s);
		} else {
			WARN(ZSCANNER_ENUMBER16_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr1376:
#line 1230 "./scanner_body.rl"
	{
		if (s->number64 <= UINT16_MAX) {
			window_add_bit(s->number64, s);
		} else {
			WARN(ZSCANNER_ENUMBER16_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr1378:
#line 1230 "./scanner_body.rl"
	{
		if (s->number64 <= UINT16_MAX) {
			window_add_bit(s->number64, s);
		} else {
			WARN(ZSCANNER_ENUMBER16_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
tr1386:
#line 1253 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_AFSDB, s); }
	goto st315;
tr1387:
#line 1253 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_AFSDB, s); }
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr1388:
#line 1253 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_AFSDB, s); }
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr1389:
#line 1253 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_AFSDB, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
tr1395:
#line 1263 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_APL, s); }
	goto st315;
tr1396:
#line 1263 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_APL, s); }
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st315;
tr1397:
#line 1263 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_APL, s); }
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st315;
tr1398:
#line 1263 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_APL, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st315;
st315:
	if ( ++p == pe )
		goto _test_eof315;
case 315:
#line 36197 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto st315;
		case 32: goto st315;
		case 40: goto tr966;
		case 41: goto tr967;
		case 65: goto st316;
		case 67: goto st321;
		case 68: goto st329;
		case 69: goto st343;
		case 72: goto st350;
		case 73: goto st355;
		case 75: goto st363;
		case 76: goto st367;
		case 77: goto st375;
		case 78: goto st381;
		case 80: goto st397;
		case 82: goto st400;
		case 83: goto st407;
		case 84: goto st418;
		case 97: goto st316;
		case 99: goto st321;
		case 100: goto st329;
		case 101: goto st343;
		case 104: goto st350;
		case 105: goto st355;
		case 107: goto st363;
		case 108: goto st367;
		case 109: goto st375;
		case 110: goto st381;
		case 112: goto st397;
		case 114: goto st400;
		case 115: goto st407;
		case 116: goto st418;
		case 2058: goto tr982;
		case 2107: goto st320;
		case 2314: goto tr984;
		case 2363: goto tr984;
		case 2570: goto tr985;
		case 2619: goto tr986;
	}
	goto tr956;
st316:
	if ( ++p == pe )
		goto _test_eof316;
case 316:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr987;
		case 32: goto tr987;
		case 40: goto tr988;
		case 41: goto tr989;
		case 65: goto st317;
		case 70: goto st428;
		case 80: goto st432;
		case 97: goto st317;
		case 102: goto st428;
		case 112: goto st432;
		case 2058: goto tr993;
		case 2107: goto tr994;
		case 2314: goto tr995;
		case 2363: goto tr995;
		case 2570: goto tr996;
		case 2619: goto tr997;
	}
	goto tr956;
st317:
	if ( ++p == pe )
		goto _test_eof317;
case 317:
	switch( (*p) ) {
		case 65: goto st318;
		case 97: goto st318;
	}
	goto tr956;
st318:
	if ( ++p == pe )
		goto _test_eof318;
case 318:
	switch( (*p) ) {
		case 65: goto st319;
		case 97: goto st319;
	}
	goto tr956;
st319:
	if ( ++p == pe )
		goto _test_eof319;
case 319:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr1000;
		case 32: goto tr1000;
		case 40: goto tr1001;
		case 41: goto tr1002;
		case 2058: goto tr1003;
		case 2107: goto tr1004;
		case 2314: goto tr1005;
		case 2363: goto tr1005;
		case 2570: goto tr1006;
		case 2619: goto tr1007;
	}
	goto tr956;
tr961:
#line 1284 "./scanner_body.rl"
	{
		memset(s->windows, 0, sizeof(s->windows));
		s->last_window = -1;
	}
	goto st320;
tr994:
#line 1243 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_A, s); }
	goto st320;
tr1004:
#line 1256 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_AAAA, s); }
	goto st320;
tr1016:
#line 1261 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_CERT, s); }
	goto st320;
tr1027:
#line 1245 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_CNAME, s); }
	goto st320;
tr1041:
#line 1270 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_DHCID, s); }
	goto st320;
tr1053:
#line 1262 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_DNAME, s); }
	goto st320;
tr1064:
#line 1269 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_DNSKEY, s); }
	goto st320;
tr1072:
#line 1264 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_DS, s); }
	goto st320;
tr1085:
#line 1279 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_EUI48, s); }
	goto st320;
tr1094:
#line 1280 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_EUI64, s); }
	goto st320;
tr1106:
#line 1248 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_HINFO, s); }
	goto st320;
tr1121:
#line 1266 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_IPSECKEY, s); }
	goto st320;
tr1132:
#line 1255 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_KEY, s); }
	goto st320;
tr1140:
#line 1260 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_KX, s); }
	goto st320;
tr1153:
#line 1276 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_L32, s); }
	goto st320;
tr1162:
#line 1277 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_L64, s); }
	goto st320;
tr1171:
#line 1257 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_LOC, s); }
	goto st320;
tr1179:
#line 1278 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_LP, s); }
	goto st320;
tr1192:
#line 1249 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_MINFO, s); }
	goto st320;
tr1200:
#line 1250 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_MX, s); }
	goto st320;
tr1214:
#line 1259 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NAPTR, s); }
	goto st320;
tr1223:
#line 1275 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NID, s); }
	goto st320;
tr1232:
#line 1244 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NS, s); }
	goto st320;
tr1242:
#line 1268 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NSEC, s); }
	goto st320;
tr1251:
#line 1271 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NSEC3, s); }
	goto st320;
tr1263:
#line 1272 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NSEC3PARAM, s); }
	goto st320;
tr1273:
#line 1247 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_PTR, s); }
	goto st320;
tr1284:
#line 1252 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_RP, s); }
	goto st320;
tr1295:
#line 1267 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_RRSIG, s); }
	goto st320;
tr1303:
#line 1254 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_RT, s); }
	goto st320;
tr1316:
#line 1246 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_SOA, s); }
	goto st320;
tr1325:
#line 1274 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_SPF, s); }
	goto st320;
tr1334:
#line 1258 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_SRV, s); }
	goto st320;
tr1345:
#line 1265 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_SSHFP, s); }
	goto st320;
tr1358:
#line 1273 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_TLSA, s); }
	goto st320;
tr1367:
#line 1251 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_TXT, s); }
	goto st320;
tr1379:
#line 1230 "./scanner_body.rl"
	{
		if (s->number64 <= UINT16_MAX) {
			window_add_bit(s->number64, s);
		} else {
			WARN(ZSCANNER_ENUMBER16_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st320;
tr1390:
#line 1253 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_AFSDB, s); }
	goto st320;
tr1399:
#line 1263 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_APL, s); }
	goto st320;
st320:
	if ( ++p == pe )
		goto _test_eof320;
case 320:
#line 36531 "scanner.c"
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	if ( _widec == 1034 )
		goto tr982;
	if ( 896 <= _widec && _widec <= 1151 )
		goto st320;
	goto tr956;
tr962:
#line 1284 "./scanner_body.rl"
	{
		memset(s->windows, 0, sizeof(s->windows));
		s->last_window = -1;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
tr984:
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
tr995:
#line 1243 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_A, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
tr1005:
#line 1256 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_AAAA, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
tr1017:
#line 1261 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_CERT, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
tr1028:
#line 1245 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_CNAME, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
tr1042:
#line 1270 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_DHCID, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
tr1054:
#line 1262 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_DNAME, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
tr1065:
#line 1269 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_DNSKEY, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
tr1073:
#line 1264 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_DS, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
tr1086:
#line 1279 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_EUI48, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
tr1095:
#line 1280 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_EUI64, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
tr1107:
#line 1248 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_HINFO, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
tr1122:
#line 1266 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_IPSECKEY, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
tr1133:
#line 1255 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_KEY, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
tr1141:
#line 1260 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_KX, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
tr1154:
#line 1276 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_L32, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
tr1163:
#line 1277 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_L64, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
tr1172:
#line 1257 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_LOC, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
tr1180:
#line 1278 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_LP, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
tr1193:
#line 1249 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_MINFO, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
tr1201:
#line 1250 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_MX, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
tr1215:
#line 1259 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NAPTR, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
tr1224:
#line 1275 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NID, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
tr1233:
#line 1244 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NS, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
tr1243:
#line 1268 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NSEC, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
tr1252:
#line 1271 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NSEC3, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
tr1264:
#line 1272 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NSEC3PARAM, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
tr1274:
#line 1247 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_PTR, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
tr1285:
#line 1252 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_RP, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
tr1296:
#line 1267 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_RRSIG, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
tr1304:
#line 1254 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_RT, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
tr1317:
#line 1246 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_SOA, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
tr1326:
#line 1274 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_SPF, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
tr1335:
#line 1258 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_SRV, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
tr1346:
#line 1265 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_SSHFP, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
tr1359:
#line 1273 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_TLSA, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
tr1368:
#line 1251 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_TXT, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
tr1380:
#line 1230 "./scanner_body.rl"
	{
		if (s->number64 <= UINT16_MAX) {
			window_add_bit(s->number64, s);
		} else {
			WARN(ZSCANNER_ENUMBER16_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
tr1391:
#line 1253 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_AFSDB, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
tr1400:
#line 1263 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_APL, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1077;
st1077:
	if ( ++p == pe )
		goto _test_eof1077;
case 1077:
#line 37882 "scanner.c"
	goto st0;
tr963:
#line 1284 "./scanner_body.rl"
	{
		memset(s->windows, 0, sizeof(s->windows));
		s->last_window = -1;
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
tr985:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
tr996:
#line 1243 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_A, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
tr1006:
#line 1256 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_AAAA, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
tr1018:
#line 1261 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_CERT, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
tr1029:
#line 1245 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_CNAME, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
tr1043:
#line 1270 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_DHCID, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
tr1055:
#line 1262 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_DNAME, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
tr1066:
#line 1269 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_DNSKEY, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
tr1074:
#line 1264 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_DS, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
tr1087:
#line 1279 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_EUI48, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
tr1096:
#line 1280 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_EUI64, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
tr1108:
#line 1248 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_HINFO, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
tr1123:
#line 1266 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_IPSECKEY, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
tr1134:
#line 1255 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_KEY, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
tr1142:
#line 1260 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_KX, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
tr1155:
#line 1276 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_L32, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
tr1164:
#line 1277 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_L64, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
tr1173:
#line 1257 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_LOC, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
tr1181:
#line 1278 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_LP, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
tr1194:
#line 1249 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_MINFO, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
tr1202:
#line 1250 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_MX, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
tr1216:
#line 1259 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NAPTR, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
tr1225:
#line 1275 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NID, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
tr1234:
#line 1244 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NS, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
tr1244:
#line 1268 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NSEC, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
tr1253:
#line 1271 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NSEC3, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
tr1265:
#line 1272 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NSEC3PARAM, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
tr1275:
#line 1247 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_PTR, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
tr1286:
#line 1252 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_RP, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
tr1297:
#line 1267 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_RRSIG, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
tr1305:
#line 1254 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_RT, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
tr1318:
#line 1246 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_SOA, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
tr1327:
#line 1274 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_SPF, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
tr1336:
#line 1258 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_SRV, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
tr1347:
#line 1265 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_SSHFP, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
tr1360:
#line 1273 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_TLSA, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
tr1369:
#line 1251 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_TXT, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
tr1381:
#line 1230 "./scanner_body.rl"
	{
		if (s->number64 <= UINT16_MAX) {
			window_add_bit(s->number64, s);
		} else {
			WARN(ZSCANNER_ENUMBER16_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
tr1392:
#line 1253 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_AFSDB, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
tr1401:
#line 1263 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_APL, s); }
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1078;
st1078:
	if ( ++p == pe )
		goto _test_eof1078;
case 1078:
#line 39372 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto st315;
		case 32: goto st315;
		case 40: goto tr966;
		case 41: goto tr967;
		case 65: goto st316;
		case 67: goto st321;
		case 68: goto st329;
		case 69: goto st343;
		case 72: goto st350;
		case 73: goto st355;
		case 75: goto st363;
		case 76: goto st367;
		case 77: goto st375;
		case 78: goto st381;
		case 80: goto st397;
		case 82: goto st400;
		case 83: goto st407;
		case 84: goto st418;
		case 97: goto st316;
		case 99: goto st321;
		case 100: goto st329;
		case 101: goto st343;
		case 104: goto st350;
		case 105: goto st355;
		case 107: goto st363;
		case 108: goto st367;
		case 109: goto st375;
		case 110: goto st381;
		case 112: goto st397;
		case 114: goto st400;
		case 115: goto st407;
		case 116: goto st418;
		case 2058: goto tr982;
		case 2107: goto st320;
		case 2314: goto tr984;
		case 2363: goto tr984;
		case 2570: goto tr985;
		case 2619: goto tr986;
	}
	goto tr956;
st321:
	if ( ++p == pe )
		goto _test_eof321;
case 321:
	switch( (*p) ) {
		case 69: goto st322;
		case 78: goto st325;
		case 101: goto st322;
		case 110: goto st325;
	}
	goto tr956;
st322:
	if ( ++p == pe )
		goto _test_eof322;
case 322:
	switch( (*p) ) {
		case 82: goto st323;
		case 114: goto st323;
	}
	goto tr956;
st323:
	if ( ++p == pe )
		goto _test_eof323;
case 323:
	switch( (*p) ) {
		case 84: goto st324;
		case 116: goto st324;
	}
	goto tr956;
st324:
	if ( ++p == pe )
		goto _test_eof324;
case 324:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr1012;
		case 32: goto tr1012;
		case 40: goto tr1013;
		case 41: goto tr1014;
		case 2058: goto tr1015;
		case 2107: goto tr1016;
		case 2314: goto tr1017;
		case 2363: goto tr1017;
		case 2570: goto tr1018;
		case 2619: goto tr1019;
	}
	goto tr956;
tr964:
#line 1284 "./scanner_body.rl"
	{
		memset(s->windows, 0, sizeof(s->windows));
		s->last_window = -1;
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
tr986:
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
tr997:
#line 1243 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_A, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
tr1007:
#line 1256 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_AAAA, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
tr1019:
#line 1261 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_CERT, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
tr1030:
#line 1245 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_CNAME, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
tr1044:
#line 1270 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_DHCID, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
tr1056:
#line 1262 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_DNAME, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
tr1067:
#line 1269 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_DNSKEY, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
tr1075:
#line 1264 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_DS, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
tr1088:
#line 1279 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_EUI48, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
tr1097:
#line 1280 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_EUI64, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
tr1109:
#line 1248 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_HINFO, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
tr1124:
#line 1266 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_IPSECKEY, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
tr1135:
#line 1255 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_KEY, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
tr1143:
#line 1260 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_KX, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
tr1156:
#line 1276 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_L32, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
tr1165:
#line 1277 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_L64, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
tr1174:
#line 1257 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_LOC, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
tr1182:
#line 1278 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_LP, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
tr1195:
#line 1249 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_MINFO, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
tr1203:
#line 1250 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_MX, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
tr1217:
#line 1259 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NAPTR, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
tr1226:
#line 1275 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NID, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
tr1235:
#line 1244 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NS, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
tr1245:
#line 1268 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NSEC, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
tr1254:
#line 1271 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NSEC3, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
tr1266:
#line 1272 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NSEC3PARAM, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
tr1276:
#line 1247 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_PTR, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
tr1287:
#line 1252 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_RP, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
tr1298:
#line 1267 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_RRSIG, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
tr1306:
#line 1254 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_RT, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
tr1319:
#line 1246 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_SOA, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
tr1328:
#line 1274 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_SPF, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
tr1337:
#line 1258 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_SRV, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
tr1348:
#line 1265 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_SSHFP, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
tr1361:
#line 1273 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_TLSA, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
tr1370:
#line 1251 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_TXT, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
tr1382:
#line 1230 "./scanner_body.rl"
	{
		if (s->number64 <= UINT16_MAX) {
			window_add_bit(s->number64, s);
		} else {
			WARN(ZSCANNER_ENUMBER16_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
tr1393:
#line 1253 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_AFSDB, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
tr1402:
#line 1263 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_APL, s); }
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {goto st246;}
				}
			}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1079;
st1079:
	if ( ++p == pe )
		goto _test_eof1079;
case 1079:
#line 40824 "scanner.c"
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	if ( _widec == 1034 )
		goto tr982;
	if ( 896 <= _widec && _widec <= 1151 )
		goto st320;
	goto tr956;
st325:
	if ( ++p == pe )
		goto _test_eof325;
case 325:
	switch( (*p) ) {
		case 65: goto st326;
		case 97: goto st326;
	}
	goto tr956;
st326:
	if ( ++p == pe )
		goto _test_eof326;
case 326:
	switch( (*p) ) {
		case 77: goto st327;
		case 109: goto st327;
	}
	goto tr956;
st327:
	if ( ++p == pe )
		goto _test_eof327;
case 327:
	switch( (*p) ) {
		case 69: goto st328;
		case 101: goto st328;
	}
	goto tr956;
st328:
	if ( ++p == pe )
		goto _test_eof328;
case 328:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr1023;
		case 32: goto tr1023;
		case 40: goto tr1024;
		case 41: goto tr1025;
		case 2058: goto tr1026;
		case 2107: goto tr1027;
		case 2314: goto tr1028;
		case 2363: goto tr1028;
		case 2570: goto tr1029;
		case 2619: goto tr1030;
	}
	goto tr956;
st329:
	if ( ++p == pe )
		goto _test_eof329;
case 329:
	switch( (*p) ) {
		case 72: goto st330;
		case 78: goto st334;
		case 83: goto st342;
		case 104: goto st330;
		case 110: goto st334;
		case 115: goto st342;
	}
	goto tr956;
st330:
	if ( ++p == pe )
		goto _test_eof330;
case 330:
	switch( (*p) ) {
		case 67: goto st331;
		case 99: goto st331;
	}
	goto tr956;
st331:
	if ( ++p == pe )
		goto _test_eof331;
case 331:
	switch( (*p) ) {
		case 73: goto st332;
		case 105: goto st332;
	}
	goto tr956;
st332:
	if ( ++p == pe )
		goto _test_eof332;
case 332:
	switch( (*p) ) {
		case 68: goto st333;
		case 100: goto st333;
	}
	goto tr956;
st333:
	if ( ++p == pe )
		goto _test_eof333;
case 333:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr1037;
		case 32: goto tr1037;
		case 40: goto tr1038;
		case 41: goto tr1039;
		case 2058: goto tr1040;
		case 2107: goto tr1041;
		case 2314: goto tr1042;
		case 2363: goto tr1042;
		case 2570: goto tr1043;
		case 2619: goto tr1044;
	}
	goto tr956;
st334:
	if ( ++p == pe )
		goto _test_eof334;
case 334:
	switch( (*p) ) {
		case 65: goto st335;
		case 83: goto st338;
		case 97: goto st335;
		case 115: goto st338;
	}
	goto tr956;
st335:
	if ( ++p == pe )
		goto _test_eof335;
case 335:
	switch( (*p) ) {
		case 77: goto st336;
		case 109: goto st336;
	}
	goto tr956;
st336:
	if ( ++p == pe )
		goto _test_eof336;
case 336:
	switch( (*p) ) {
		case 69: goto st337;
		case 101: goto st337;
	}
	goto tr956;
st337:
	if ( ++p == pe )
		goto _test_eof337;
case 337:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr1049;
		case 32: goto tr1049;
		case 40: goto tr1050;
		case 41: goto tr1051;
		case 2058: goto tr1052;
		case 2107: goto tr1053;
		case 2314: goto tr1054;
		case 2363: goto tr1054;
		case 2570: goto tr1055;
		case 2619: goto tr1056;
	}
	goto tr956;
st338:
	if ( ++p == pe )
		goto _test_eof338;
case 338:
	switch( (*p) ) {
		case 75: goto st339;
		case 107: goto st339;
	}
	goto tr956;
st339:
	if ( ++p == pe )
		goto _test_eof339;
case 339:
	switch( (*p) ) {
		case 69: goto st340;
		case 101: goto st340;
	}
	goto tr956;
st340:
	if ( ++p == pe )
		goto _test_eof340;
case 340:
	switch( (*p) ) {
		case 89: goto st341;
		case 121: goto st341;
	}
	goto tr956;
st341:
	if ( ++p == pe )
		goto _test_eof341;
case 341:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr1060;
		case 32: goto tr1060;
		case 40: goto tr1061;
		case 41: goto tr1062;
		case 2058: goto tr1063;
		case 2107: goto tr1064;
		case 2314: goto tr1065;
		case 2363: goto tr1065;
		case 2570: goto tr1066;
		case 2619: goto tr1067;
	}
	goto tr956;
st342:
	if ( ++p == pe )
		goto _test_eof342;
case 342:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr1068;
		case 32: goto tr1068;
		case 40: goto tr1069;
		case 41: goto tr1070;
		case 2058: goto tr1071;
		case 2107: goto tr1072;
		case 2314: goto tr1073;
		case 2363: goto tr1073;
		case 2570: goto tr1074;
		case 2619: goto tr1075;
	}
	goto tr956;
st343:
	if ( ++p == pe )
		goto _test_eof343;
case 343:
	switch( (*p) ) {
		case 85: goto st344;
		case 117: goto st344;
	}
	goto tr956;
st344:
	if ( ++p == pe )
		goto _test_eof344;
case 344:
	switch( (*p) ) {
		case 73: goto st345;
		case 105: goto st345;
	}
	goto tr956;
st345:
	if ( ++p == pe )
		goto _test_eof345;
case 345:
	switch( (*p) ) {
		case 52: goto st346;
		case 54: goto st348;
	}
	goto tr956;
st346:
	if ( ++p == pe )
		goto _test_eof346;
case 346:
	if ( (*p) == 56 )
		goto st347;
	goto tr956;
st347:
	if ( ++p == pe )
		goto _test_eof347;
case 347:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr1081;
		case 32: goto tr1081;
		case 40: goto tr1082;
		case 41: goto tr1083;
		case 2058: goto tr1084;
		case 2107: goto tr1085;
		case 2314: goto tr1086;
		case 2363: goto tr1086;
		case 2570: goto tr1087;
		case 2619: goto tr1088;
	}
	goto tr956;
st348:
	if ( ++p == pe )
		goto _test_eof348;
case 348:
	if ( (*p) == 52 )
		goto st349;
	goto tr956;
st349:
	if ( ++p == pe )
		goto _test_eof349;
case 349:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr1090;
		case 32: goto tr1090;
		case 40: goto tr1091;
		case 41: goto tr1092;
		case 2058: goto tr1093;
		case 2107: goto tr1094;
		case 2314: goto tr1095;
		case 2363: goto tr1095;
		case 2570: goto tr1096;
		case 2619: goto tr1097;
	}
	goto tr956;
st350:
	if ( ++p == pe )
		goto _test_eof350;
case 350:
	switch( (*p) ) {
		case 73: goto st351;
		case 105: goto st351;
	}
	goto tr956;
st351:
	if ( ++p == pe )
		goto _test_eof351;
case 351:
	switch( (*p) ) {
		case 78: goto st352;
		case 110: goto st352;
	}
	goto tr956;
st352:
	if ( ++p == pe )
		goto _test_eof352;
case 352:
	switch( (*p) ) {
		case 70: goto st353;
		case 102: goto st353;
	}
	goto tr956;
st353:
	if ( ++p == pe )
		goto _test_eof353;
case 353:
	switch( (*p) ) {
		case 79: goto st354;
		case 111: goto st354;
	}
	goto tr956;
st354:
	if ( ++p == pe )
		goto _test_eof354;
case 354:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr1102;
		case 32: goto tr1102;
		case 40: goto tr1103;
		case 41: goto tr1104;
		case 2058: goto tr1105;
		case 2107: goto tr1106;
		case 2314: goto tr1107;
		case 2363: goto tr1107;
		case 2570: goto tr1108;
		case 2619: goto tr1109;
	}
	goto tr956;
st355:
	if ( ++p == pe )
		goto _test_eof355;
case 355:
	switch( (*p) ) {
		case 80: goto st356;
		case 112: goto st356;
	}
	goto tr956;
st356:
	if ( ++p == pe )
		goto _test_eof356;
case 356:
	switch( (*p) ) {
		case 83: goto st357;
		case 115: goto st357;
	}
	goto tr956;
st357:
	if ( ++p == pe )
		goto _test_eof357;
case 357:
	switch( (*p) ) {
		case 69: goto st358;
		case 101: goto st358;
	}
	goto tr956;
st358:
	if ( ++p == pe )
		goto _test_eof358;
case 358:
	switch( (*p) ) {
		case 67: goto st359;
		case 99: goto st359;
	}
	goto tr956;
st359:
	if ( ++p == pe )
		goto _test_eof359;
case 359:
	switch( (*p) ) {
		case 75: goto st360;
		case 107: goto st360;
	}
	goto tr956;
st360:
	if ( ++p == pe )
		goto _test_eof360;
case 360:
	switch( (*p) ) {
		case 69: goto st361;
		case 101: goto st361;
	}
	goto tr956;
st361:
	if ( ++p == pe )
		goto _test_eof361;
case 361:
	switch( (*p) ) {
		case 89: goto st362;
		case 121: goto st362;
	}
	goto tr956;
st362:
	if ( ++p == pe )
		goto _test_eof362;
case 362:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr1117;
		case 32: goto tr1117;
		case 40: goto tr1118;
		case 41: goto tr1119;
		case 2058: goto tr1120;
		case 2107: goto tr1121;
		case 2314: goto tr1122;
		case 2363: goto tr1122;
		case 2570: goto tr1123;
		case 2619: goto tr1124;
	}
	goto tr956;
st363:
	if ( ++p == pe )
		goto _test_eof363;
case 363:
	switch( (*p) ) {
		case 69: goto st364;
		case 88: goto st366;
		case 101: goto st364;
		case 120: goto st366;
	}
	goto tr956;
st364:
	if ( ++p == pe )
		goto _test_eof364;
case 364:
	switch( (*p) ) {
		case 89: goto st365;
		case 121: goto st365;
	}
	goto tr956;
st365:
	if ( ++p == pe )
		goto _test_eof365;
case 365:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr1128;
		case 32: goto tr1128;
		case 40: goto tr1129;
		case 41: goto tr1130;
		case 2058: goto tr1131;
		case 2107: goto tr1132;
		case 2314: goto tr1133;
		case 2363: goto tr1133;
		case 2570: goto tr1134;
		case 2619: goto tr1135;
	}
	goto tr956;
st366:
	if ( ++p == pe )
		goto _test_eof366;
case 366:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr1136;
		case 32: goto tr1136;
		case 40: goto tr1137;
		case 41: goto tr1138;
		case 2058: goto tr1139;
		case 2107: goto tr1140;
		case 2314: goto tr1141;
		case 2363: goto tr1141;
		case 2570: goto tr1142;
		case 2619: goto tr1143;
	}
	goto tr956;
st367:
	if ( ++p == pe )
		goto _test_eof367;
case 367:
	switch( (*p) ) {
		case 51: goto st368;
		case 54: goto st370;
		case 79: goto st372;
		case 80: goto st374;
		case 111: goto st372;
		case 112: goto st374;
	}
	goto tr956;
st368:
	if ( ++p == pe )
		goto _test_eof368;
case 368:
	if ( (*p) == 50 )
		goto st369;
	goto tr956;
st369:
	if ( ++p == pe )
		goto _test_eof369;
case 369:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr1149;
		case 32: goto tr1149;
		case 40: goto tr1150;
		case 41: goto tr1151;
		case 2058: goto tr1152;
		case 2107: goto tr1153;
		case 2314: goto tr1154;
		case 2363: goto tr1154;
		case 2570: goto tr1155;
		case 2619: goto tr1156;
	}
	goto tr956;
st370:
	if ( ++p == pe )
		goto _test_eof370;
case 370:
	if ( (*p) == 52 )
		goto st371;
	goto tr956;
st371:
	if ( ++p == pe )
		goto _test_eof371;
case 371:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr1158;
		case 32: goto tr1158;
		case 40: goto tr1159;
		case 41: goto tr1160;
		case 2058: goto tr1161;
		case 2107: goto tr1162;
		case 2314: goto tr1163;
		case 2363: goto tr1163;
		case 2570: goto tr1164;
		case 2619: goto tr1165;
	}
	goto tr956;
st372:
	if ( ++p == pe )
		goto _test_eof372;
case 372:
	switch( (*p) ) {
		case 67: goto st373;
		case 99: goto st373;
	}
	goto tr956;
st373:
	if ( ++p == pe )
		goto _test_eof373;
case 373:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr1167;
		case 32: goto tr1167;
		case 40: goto tr1168;
		case 41: goto tr1169;
		case 2058: goto tr1170;
		case 2107: goto tr1171;
		case 2314: goto tr1172;
		case 2363: goto tr1172;
		case 2570: goto tr1173;
		case 2619: goto tr1174;
	}
	goto tr956;
st374:
	if ( ++p == pe )
		goto _test_eof374;
case 374:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr1175;
		case 32: goto tr1175;
		case 40: goto tr1176;
		case 41: goto tr1177;
		case 2058: goto tr1178;
		case 2107: goto tr1179;
		case 2314: goto tr1180;
		case 2363: goto tr1180;
		case 2570: goto tr1181;
		case 2619: goto tr1182;
	}
	goto tr956;
st375:
	if ( ++p == pe )
		goto _test_eof375;
case 375:
	switch( (*p) ) {
		case 73: goto st376;
		case 88: goto st380;
		case 105: goto st376;
		case 120: goto st380;
	}
	goto tr956;
st376:
	if ( ++p == pe )
		goto _test_eof376;
case 376:
	switch( (*p) ) {
		case 78: goto st377;
		case 110: goto st377;
	}
	goto tr956;
st377:
	if ( ++p == pe )
		goto _test_eof377;
case 377:
	switch( (*p) ) {
		case 70: goto st378;
		case 102: goto st378;
	}
	goto tr956;
st378:
	if ( ++p == pe )
		goto _test_eof378;
case 378:
	switch( (*p) ) {
		case 79: goto st379;
		case 111: goto st379;
	}
	goto tr956;
st379:
	if ( ++p == pe )
		goto _test_eof379;
case 379:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr1188;
		case 32: goto tr1188;
		case 40: goto tr1189;
		case 41: goto tr1190;
		case 2058: goto tr1191;
		case 2107: goto tr1192;
		case 2314: goto tr1193;
		case 2363: goto tr1193;
		case 2570: goto tr1194;
		case 2619: goto tr1195;
	}
	goto tr956;
st380:
	if ( ++p == pe )
		goto _test_eof380;
case 380:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr1196;
		case 32: goto tr1196;
		case 40: goto tr1197;
		case 41: goto tr1198;
		case 2058: goto tr1199;
		case 2107: goto tr1200;
		case 2314: goto tr1201;
		case 2363: goto tr1201;
		case 2570: goto tr1202;
		case 2619: goto tr1203;
	}
	goto tr956;
st381:
	if ( ++p == pe )
		goto _test_eof381;
case 381:
	switch( (*p) ) {
		case 65: goto st382;
		case 73: goto st386;
		case 83: goto st388;
		case 97: goto st382;
		case 105: goto st386;
		case 115: goto st388;
	}
	goto tr956;
st382:
	if ( ++p == pe )
		goto _test_eof382;
case 382:
	switch( (*p) ) {
		case 80: goto st383;
		case 112: goto st383;
	}
	goto tr956;
st383:
	if ( ++p == pe )
		goto _test_eof383;
case 383:
	switch( (*p) ) {
		case 84: goto st384;
		case 116: goto st384;
	}
	goto tr956;
st384:
	if ( ++p == pe )
		goto _test_eof384;
case 384:
	switch( (*p) ) {
		case 82: goto st385;
		case 114: goto st385;
	}
	goto tr956;
st385:
	if ( ++p == pe )
		goto _test_eof385;
case 385:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr1210;
		case 32: goto tr1210;
		case 40: goto tr1211;
		case 41: goto tr1212;
		case 2058: goto tr1213;
		case 2107: goto tr1214;
		case 2314: goto tr1215;
		case 2363: goto tr1215;
		case 2570: goto tr1216;
		case 2619: goto tr1217;
	}
	goto tr956;
st386:
	if ( ++p == pe )
		goto _test_eof386;
case 386:
	switch( (*p) ) {
		case 68: goto st387;
		case 100: goto st387;
	}
	goto tr956;
st387:
	if ( ++p == pe )
		goto _test_eof387;
case 387:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr1219;
		case 32: goto tr1219;
		case 40: goto tr1220;
		case 41: goto tr1221;
		case 2058: goto tr1222;
		case 2107: goto tr1223;
		case 2314: goto tr1224;
		case 2363: goto tr1224;
		case 2570: goto tr1225;
		case 2619: goto tr1226;
	}
	goto tr956;
st388:
	if ( ++p == pe )
		goto _test_eof388;
case 388:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr1227;
		case 32: goto tr1227;
		case 40: goto tr1228;
		case 41: goto tr1229;
		case 69: goto st389;
		case 101: goto st389;
		case 2058: goto tr1231;
		case 2107: goto tr1232;
		case 2314: goto tr1233;
		case 2363: goto tr1233;
		case 2570: goto tr1234;
		case 2619: goto tr1235;
	}
	goto tr956;
st389:
	if ( ++p == pe )
		goto _test_eof389;
case 389:
	switch( (*p) ) {
		case 67: goto st390;
		case 99: goto st390;
	}
	goto tr956;
st390:
	if ( ++p == pe )
		goto _test_eof390;
case 390:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr1237;
		case 32: goto tr1237;
		case 40: goto tr1238;
		case 41: goto tr1239;
		case 51: goto st391;
		case 2058: goto tr1241;
		case 2107: goto tr1242;
		case 2314: goto tr1243;
		case 2363: goto tr1243;
		case 2570: goto tr1244;
		case 2619: goto tr1245;
	}
	goto tr956;
st391:
	if ( ++p == pe )
		goto _test_eof391;
case 391:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr1246;
		case 32: goto tr1246;
		case 40: goto tr1247;
		case 41: goto tr1248;
		case 80: goto st392;
		case 112: goto st392;
		case 2058: goto tr1250;
		case 2107: goto tr1251;
		case 2314: goto tr1252;
		case 2363: goto tr1252;
		case 2570: goto tr1253;
		case 2619: goto tr1254;
	}
	goto tr956;
st392:
	if ( ++p == pe )
		goto _test_eof392;
case 392:
	switch( (*p) ) {
		case 65: goto st393;
		case 97: goto st393;
	}
	goto tr956;
st393:
	if ( ++p == pe )
		goto _test_eof393;
case 393:
	switch( (*p) ) {
		case 82: goto st394;
		case 114: goto st394;
	}
	goto tr956;
st394:
	if ( ++p == pe )
		goto _test_eof394;
case 394:
	switch( (*p) ) {
		case 65: goto st395;
		case 97: goto st395;
	}
	goto tr956;
st395:
	if ( ++p == pe )
		goto _test_eof395;
case 395:
	switch( (*p) ) {
		case 77: goto st396;
		case 109: goto st396;
	}
	goto tr956;
st396:
	if ( ++p == pe )
		goto _test_eof396;
case 396:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr1259;
		case 32: goto tr1259;
		case 40: goto tr1260;
		case 41: goto tr1261;
		case 2058: goto tr1262;
		case 2107: goto tr1263;
		case 2314: goto tr1264;
		case 2363: goto tr1264;
		case 2570: goto tr1265;
		case 2619: goto tr1266;
	}
	goto tr956;
st397:
	if ( ++p == pe )
		goto _test_eof397;
case 397:
	switch( (*p) ) {
		case 84: goto st398;
		case 116: goto st398;
	}
	goto tr956;
st398:
	if ( ++p == pe )
		goto _test_eof398;
case 398:
	switch( (*p) ) {
		case 82: goto st399;
		case 114: goto st399;
	}
	goto tr956;
st399:
	if ( ++p == pe )
		goto _test_eof399;
case 399:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr1269;
		case 32: goto tr1269;
		case 40: goto tr1270;
		case 41: goto tr1271;
		case 2058: goto tr1272;
		case 2107: goto tr1273;
		case 2314: goto tr1274;
		case 2363: goto tr1274;
		case 2570: goto tr1275;
		case 2619: goto tr1276;
	}
	goto tr956;
st400:
	if ( ++p == pe )
		goto _test_eof400;
case 400:
	switch( (*p) ) {
		case 80: goto st401;
		case 82: goto st402;
		case 84: goto st406;
		case 112: goto st401;
		case 114: goto st402;
		case 116: goto st406;
	}
	goto tr956;
st401:
	if ( ++p == pe )
		goto _test_eof401;
case 401:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr1280;
		case 32: goto tr1280;
		case 40: goto tr1281;
		case 41: goto tr1282;
		case 2058: goto tr1283;
		case 2107: goto tr1284;
		case 2314: goto tr1285;
		case 2363: goto tr1285;
		case 2570: goto tr1286;
		case 2619: goto tr1287;
	}
	goto tr956;
st402:
	if ( ++p == pe )
		goto _test_eof402;
case 402:
	switch( (*p) ) {
		case 83: goto st403;
		case 115: goto st403;
	}
	goto tr956;
st403:
	if ( ++p == pe )
		goto _test_eof403;
case 403:
	switch( (*p) ) {
		case 73: goto st404;
		case 105: goto st404;
	}
	goto tr956;
st404:
	if ( ++p == pe )
		goto _test_eof404;
case 404:
	switch( (*p) ) {
		case 71: goto st405;
		case 103: goto st405;
	}
	goto tr956;
st405:
	if ( ++p == pe )
		goto _test_eof405;
case 405:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr1291;
		case 32: goto tr1291;
		case 40: goto tr1292;
		case 41: goto tr1293;
		case 2058: goto tr1294;
		case 2107: goto tr1295;
		case 2314: goto tr1296;
		case 2363: goto tr1296;
		case 2570: goto tr1297;
		case 2619: goto tr1298;
	}
	goto tr956;
st406:
	if ( ++p == pe )
		goto _test_eof406;
case 406:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr1299;
		case 32: goto tr1299;
		case 40: goto tr1300;
		case 41: goto tr1301;
		case 2058: goto tr1302;
		case 2107: goto tr1303;
		case 2314: goto tr1304;
		case 2363: goto tr1304;
		case 2570: goto tr1305;
		case 2619: goto tr1306;
	}
	goto tr956;
st407:
	if ( ++p == pe )
		goto _test_eof407;
case 407:
	switch( (*p) ) {
		case 79: goto st408;
		case 80: goto st410;
		case 82: goto st412;
		case 83: goto st414;
		case 111: goto st408;
		case 112: goto st410;
		case 114: goto st412;
		case 115: goto st414;
	}
	goto tr956;
st408:
	if ( ++p == pe )
		goto _test_eof408;
case 408:
	switch( (*p) ) {
		case 65: goto st409;
		case 97: goto st409;
	}
	goto tr956;
st409:
	if ( ++p == pe )
		goto _test_eof409;
case 409:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr1312;
		case 32: goto tr1312;
		case 40: goto tr1313;
		case 41: goto tr1314;
		case 2058: goto tr1315;
		case 2107: goto tr1316;
		case 2314: goto tr1317;
		case 2363: goto tr1317;
		case 2570: goto tr1318;
		case 2619: goto tr1319;
	}
	goto tr956;
st410:
	if ( ++p == pe )
		goto _test_eof410;
case 410:
	switch( (*p) ) {
		case 70: goto st411;
		case 102: goto st411;
	}
	goto tr956;
st411:
	if ( ++p == pe )
		goto _test_eof411;
case 411:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr1321;
		case 32: goto tr1321;
		case 40: goto tr1322;
		case 41: goto tr1323;
		case 2058: goto tr1324;
		case 2107: goto tr1325;
		case 2314: goto tr1326;
		case 2363: goto tr1326;
		case 2570: goto tr1327;
		case 2619: goto tr1328;
	}
	goto tr956;
st412:
	if ( ++p == pe )
		goto _test_eof412;
case 412:
	switch( (*p) ) {
		case 86: goto st413;
		case 118: goto st413;
	}
	goto tr956;
st413:
	if ( ++p == pe )
		goto _test_eof413;
case 413:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr1330;
		case 32: goto tr1330;
		case 40: goto tr1331;
		case 41: goto tr1332;
		case 2058: goto tr1333;
		case 2107: goto tr1334;
		case 2314: goto tr1335;
		case 2363: goto tr1335;
		case 2570: goto tr1336;
		case 2619: goto tr1337;
	}
	goto tr956;
st414:
	if ( ++p == pe )
		goto _test_eof414;
case 414:
	switch( (*p) ) {
		case 72: goto st415;
		case 104: goto st415;
	}
	goto tr956;
st415:
	if ( ++p == pe )
		goto _test_eof415;
case 415:
	switch( (*p) ) {
		case 70: goto st416;
		case 102: goto st416;
	}
	goto tr956;
st416:
	if ( ++p == pe )
		goto _test_eof416;
case 416:
	switch( (*p) ) {
		case 80: goto st417;
		case 112: goto st417;
	}
	goto tr956;
st417:
	if ( ++p == pe )
		goto _test_eof417;
case 417:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr1341;
		case 32: goto tr1341;
		case 40: goto tr1342;
		case 41: goto tr1343;
		case 2058: goto tr1344;
		case 2107: goto tr1345;
		case 2314: goto tr1346;
		case 2363: goto tr1346;
		case 2570: goto tr1347;
		case 2619: goto tr1348;
	}
	goto tr956;
st418:
	if ( ++p == pe )
		goto _test_eof418;
case 418:
	switch( (*p) ) {
		case 76: goto st419;
		case 88: goto st422;
		case 89: goto st424;
		case 108: goto st419;
		case 120: goto st422;
		case 121: goto st424;
	}
	goto tr956;
st419:
	if ( ++p == pe )
		goto _test_eof419;
case 419:
	switch( (*p) ) {
		case 83: goto st420;
		case 115: goto st420;
	}
	goto tr956;
st420:
	if ( ++p == pe )
		goto _test_eof420;
case 420:
	switch( (*p) ) {
		case 65: goto st421;
		case 97: goto st421;
	}
	goto tr956;
st421:
	if ( ++p == pe )
		goto _test_eof421;
case 421:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr1354;
		case 32: goto tr1354;
		case 40: goto tr1355;
		case 41: goto tr1356;
		case 2058: goto tr1357;
		case 2107: goto tr1358;
		case 2314: goto tr1359;
		case 2363: goto tr1359;
		case 2570: goto tr1360;
		case 2619: goto tr1361;
	}
	goto tr956;
st422:
	if ( ++p == pe )
		goto _test_eof422;
case 422:
	switch( (*p) ) {
		case 84: goto st423;
		case 116: goto st423;
	}
	goto tr956;
st423:
	if ( ++p == pe )
		goto _test_eof423;
case 423:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr1363;
		case 32: goto tr1363;
		case 40: goto tr1364;
		case 41: goto tr1365;
		case 2058: goto tr1366;
		case 2107: goto tr1367;
		case 2314: goto tr1368;
		case 2363: goto tr1368;
		case 2570: goto tr1369;
		case 2619: goto tr1370;
	}
	goto tr956;
st424:
	if ( ++p == pe )
		goto _test_eof424;
case 424:
	switch( (*p) ) {
		case 80: goto st425;
		case 112: goto st425;
	}
	goto tr956;
st425:
	if ( ++p == pe )
		goto _test_eof425;
case 425:
	switch( (*p) ) {
		case 69: goto st426;
		case 101: goto st426;
	}
	goto tr956;
st426:
	if ( ++p == pe )
		goto _test_eof426;
case 426:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto tr1373;
	goto tr956;
tr1373:
#line 278 "./scanner_body.rl"
	{
		s->number64 = 0;
	}
#line 261 "./scanner_body.rl"
	{
		// Overflow check: 10*(s->number64) + fc - ASCII_0 <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + ASCII_0)
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ENUMBER64_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st427;
tr1377:
#line 261 "./scanner_body.rl"
	{
		// Overflow check: 10*(s->number64) + fc - ASCII_0 <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + ASCII_0)
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ENUMBER64_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st427;
st427:
	if ( ++p == pe )
		goto _test_eof427;
case 427:
#line 42762 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr1374;
		case 32: goto tr1374;
		case 40: goto tr1375;
		case 41: goto tr1376;
		case 2058: goto tr1378;
		case 2107: goto tr1379;
		case 2314: goto tr1380;
		case 2363: goto tr1380;
		case 2570: goto tr1381;
		case 2619: goto tr1382;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1377;
	goto tr956;
st428:
	if ( ++p == pe )
		goto _test_eof428;
case 428:
	switch( (*p) ) {
		case 83: goto st429;
		case 115: goto st429;
	}
	goto tr956;
st429:
	if ( ++p == pe )
		goto _test_eof429;
case 429:
	switch( (*p) ) {
		case 68: goto st430;
		case 100: goto st430;
	}
	goto tr956;
st430:
	if ( ++p == pe )
		goto _test_eof430;
case 430:
	switch( (*p) ) {
		case 66: goto st431;
		case 98: goto st431;
	}
	goto tr956;
st431:
	if ( ++p == pe )
		goto _test_eof431;
case 431:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr1386;
		case 32: goto tr1386;
		case 40: goto tr1387;
		case 41: goto tr1388;
		case 2058: goto tr1389;
		case 2107: goto tr1390;
		case 2314: goto tr1391;
		case 2363: goto tr1391;
		case 2570: goto tr1392;
		case 2619: goto tr1393;
	}
	goto tr956;
st432:
	if ( ++p == pe )
		goto _test_eof432;
case 432:
	switch( (*p) ) {
		case 76: goto st433;
		case 108: goto st433;
	}
	goto tr956;
st433:
	if ( ++p == pe )
		goto _test_eof433;
case 433:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto tr1395;
		case 32: goto tr1395;
		case 40: goto tr1396;
		case 41: goto tr1397;
		case 2058: goto tr1398;
		case 2107: goto tr1399;
		case 2314: goto tr1400;
		case 2363: goto tr1400;
		case 2570: goto tr1401;
		case 2619: goto tr1402;
	}
	goto tr956;
st434:
	if ( ++p == pe )
		goto _test_eof434;
case 434:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st435;
		case 32: goto st435;
		case 40: goto tr1405;
		case 41: goto tr1406;
		case 1034: goto tr1407;
		case 1083: goto st442;
	}
	goto tr1403;
tr1405:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st435;
tr1406:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st435;
tr1407:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st435;
st435:
	if ( ++p == pe )
		goto _test_eof435;
case 435:
#line 42965 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st435;
		case 32: goto st435;
		case 40: goto tr1405;
		case 41: goto tr1406;
		case 1034: goto tr1407;
		case 1083: goto st442;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1410;
	goto tr1409;
tr1410:
#line 278 "./scanner_body.rl"
	{
		s->number64 = 0;
	}
#line 261 "./scanner_body.rl"
	{
		// Overflow check: 10*(s->number64) + fc - ASCII_0 <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + ASCII_0)
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ENUMBER64_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st436;
tr1414:
#line 261 "./scanner_body.rl"
	{
		// Overflow check: 10*(s->number64) + fc - ASCII_0 <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + ASCII_0)
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ENUMBER64_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st436;
st436:
	if ( ++p == pe )
		goto _test_eof436;
case 436:
#line 43033 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1411;
		case 32: goto tr1411;
		case 40: goto tr1412;
		case 41: goto tr1413;
		case 1034: goto tr1415;
		case 1083: goto tr1416;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1414;
	goto tr1409;
tr1419:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st437;
tr1420:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st437;
tr1422:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st437;
tr1411:
#line 365 "./scanner_body.rl"
	{
		if (s->number64 <= UINT16_MAX) {
			s->r_data_length = (uint16_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER16_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st437;
tr1412:
#line 365 "./scanner_body.rl"
	{
		if (s->number64 <= UINT16_MAX) {
			s->r_data_length = (uint16_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER16_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st437;
tr1413:
#line 365 "./scanner_body.rl"
	{
		if (s->number64 <= UINT16_MAX) {
			s->r_data_length = (uint16_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER16_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st437;
tr1415:
#line 365 "./scanner_body.rl"
	{
		if (s->number64 <= UINT16_MAX) {
			s->r_data_length = (uint16_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER16_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st437;
st437:
	if ( ++p == pe )
		goto _test_eof437;
case 437:
#line 43153 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st437;
		case 32: goto st437;
		case 40: goto tr1419;
		case 41: goto tr1420;
		case 1034: goto tr1422;
		case 1083: goto st441;
	}
	if ( _widec < 65 ) {
		if ( 48 <= _widec && _widec <= 57 )
			goto tr1421;
	} else if ( _widec > 70 ) {
		if ( 97 <= _widec && _widec <= 102 )
			goto tr1421;
	} else
		goto tr1421;
	goto tr1417;
tr1421:
#line 876 "./scanner_body.rl"
	{
		if (rdata_tail <= rdata_stop) {
			*rdata_tail = first_hex_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ERDATA_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st438;
st438:
	if ( ++p == pe )
		goto _test_eof438;
case 438:
#line 43200 "scanner.c"
	if ( (*p) < 65 ) {
		if ( 48 <= (*p) && (*p) <= 57 )
			goto tr1424;
	} else if ( (*p) > 70 ) {
		if ( 97 <= (*p) && (*p) <= 102 )
			goto tr1424;
	} else
		goto tr1424;
	goto tr1417;
tr1426:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st439;
tr1427:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st439;
tr1428:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st439;
tr1424:
#line 884 "./scanner_body.rl"
	{
		*rdata_tail += second_hex_to_num[(uint8_t)(*p)];
		rdata_tail++;
	}
	goto st439;
st439:
	if ( ++p == pe )
		goto _test_eof439;
case 439:
#line 43247 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto st439;
		case 32: goto st439;
		case 40: goto tr1426;
		case 41: goto tr1427;
		case 2058: goto tr1428;
		case 2107: goto st440;
		case 2314: goto tr1430;
		case 2363: goto tr1430;
		case 2570: goto tr1431;
		case 2619: goto tr1432;
	}
	if ( _widec < 65 ) {
		if ( 48 <= _widec && _widec <= 57 )
			goto tr1421;
	} else if ( _widec > 70 ) {
		if ( 97 <= _widec && _widec <= 102 )
			goto tr1421;
	} else
		goto tr1421;
	goto tr1417;
st440:
	if ( ++p == pe )
		goto _test_eof440;
case 440:
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	if ( _widec == 1034 )
		goto tr1428;
	if ( 896 <= _widec && _widec <= 1151 )
		goto st440;
	goto tr1417;
tr1430:
#line 902 "./scanner_body.rl"
	{
		if ((rdata_tail - s->r_data) != s->r_data_length) {
			WARN(ZSCANNER_EBAD_RDATA_LENGTH);
			p--; {goto st246;}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1080;
st1080:
	if ( ++p == pe )
		goto _test_eof1080;
case 1080:
#line 43336 "scanner.c"
	goto st0;
tr1431:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
#line 902 "./scanner_body.rl"
	{
		if ((rdata_tail - s->r_data) != s->r_data_length) {
			WARN(ZSCANNER_EBAD_RDATA_LENGTH);
			p--; {goto st246;}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1081;
st1081:
	if ( ++p == pe )
		goto _test_eof1081;
case 1081:
#line 43359 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(1664 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
			if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
	}
	switch( _widec ) {
		case 9: goto st439;
		case 32: goto st439;
		case 40: goto tr1426;
		case 41: goto tr1427;
		case 2058: goto tr1428;
		case 2107: goto st440;
		case 2314: goto tr1430;
		case 2363: goto tr1430;
		case 2570: goto tr1431;
		case 2619: goto tr1432;
	}
	if ( _widec < 65 ) {
		if ( 48 <= _widec && _widec <= 57 )
			goto tr1421;
	} else if ( _widec > 70 ) {
		if ( 97 <= _widec && _widec <= 102 )
			goto tr1421;
	} else
		goto tr1421;
	goto tr1417;
tr1432:
#line 902 "./scanner_body.rl"
	{
		if ((rdata_tail - s->r_data) != s->r_data_length) {
			WARN(ZSCANNER_EBAD_RDATA_LENGTH);
			p--; {goto st246;}
		}
	}
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
	goto st1082;
st1082:
	if ( ++p == pe )
		goto _test_eof1082;
case 1082:
#line 43418 "scanner.c"
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	if ( _widec == 1034 )
		goto tr1428;
	if ( 896 <= _widec && _widec <= 1151 )
		goto st440;
	goto tr1417;
tr1416:
#line 365 "./scanner_body.rl"
	{
		if (s->number64 <= UINT16_MAX) {
			s->r_data_length = (uint16_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER16_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st441;
st441:
	if ( ++p == pe )
		goto _test_eof441;
case 441:
#line 43460 "scanner.c"
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	if ( _widec == 1034 )
		goto tr1422;
	if ( 896 <= _widec && _widec <= 1151 )
		goto st441;
	goto tr1403;
st442:
	if ( ++p == pe )
		goto _test_eof442;
case 442:
	_widec = (*p);
	if ( (*p) < 10 ) {
		if ( (*p) <= 9 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) > 10 ) {
		if ( 11 <= (*p) )
 {			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	if ( _widec == 1034 )
		goto tr1407;
	if ( 896 <= _widec && _widec <= 1151 )
		goto st442;
	goto tr1403;
st443:
	if ( ++p == pe )
		goto _test_eof443;
case 443:
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st444;
		case 32: goto st444;
		case 40: goto tr1434;
		case 41: goto tr1435;
		case 1034: goto tr1436;
		case 1083: goto st452;
	}
	goto tr1403;
tr1434:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st444;
tr1435:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = false;
	}
	goto st444;
tr1436:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	goto st444;
st444:
	if ( ++p == pe )
		goto _test_eof444;
case 444:
#line 43574 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st444;
		case 32: goto st444;
		case 40: goto tr1434;
		case 41: goto tr1435;
		case 48: goto tr1438;
		case 1034: goto tr1436;
		case 1083: goto st452;
	}
	if ( 49 <= _widec && _widec <= 57 )
		goto tr1439;
	goto tr1409;
tr1438:
#line 278 "./scanner_body.rl"
	{
		s->number64 = 0;
	}
#line 261 "./scanner_body.rl"
	{
		// Overflow check: 10*(s->number64) + fc - ASCII_0 <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + ASCII_0)
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ENUMBER64_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st445;
st445:
	if ( ++p == pe )
		goto _test_eof445;
case 445:
#line 43626 "scanner.c"
	_widec = (*p);
	if ( (*p) > 10 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(640 + ((*p) - -128));
			if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		}
	} else if ( (*p) >= 10 ) {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1440;
		case 32: goto tr1440;
		case 40: goto tr1441;
		case 41: goto tr1442;
		case 778: goto tr1444;
		case 827: goto tr1444;
		case 1034: goto tr1445;
		case 1083: goto tr1446;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1443;
	goto tr1409;
tr1440:
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
#line 365 "./scanner_body.rl"
	{
		if (s->number64 <= UINT16_MAX) {
			s->r_data_length = (uint16_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER16_OVERFLOW);
			p--; {goto st246;}
		}
	}
	goto st1083;
tr1441:
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
#line 365 "./scanner_body.rl"
	{
		if (s->number64 <= UINT16_MAX) {
			s->r_data_length = (uint16_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER16_OVERFLOW);
			p--; {goto st246;}
		}
	}
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {goto st246;}
		}
		s->multiline = true;
	}
	goto st1083;
tr1442:
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top];goto _again;}
	}
#line 365 "./scanner_body.rl"
	{
		if (s->number64 <= UINT16_MAX) {
			s->